/* Copyright (c) 1992 The Geometry Center; University of Minnesota
   1300 South Second Street;  Minneapolis, MN  55454, USA;
   
This file is part of geomview/OOGL. geomview/OOGL is free software;
you can redistribute it and/or modify it only under the terms given in
the file COPYING, which you should have received along with this file.
This and other related software may be obtained via anonymous ftp from
geom.umn.edu; email: software@geom.umn.edu. */

/* Authors: Charlie Gunn, Stuart Levy, Tamara Munzner, Mark Phillips */

#include "tlistP.h"
#include "inst.h"

/*
 * Load a Tlist from a file.
 * Note we transpose the matrix while loading, to suit
 * the peculiar internal form of a Transform.
 */
Geom *
TlistFLoad( file, fname )
    FILE *file;
    char *fname;
{
    Tlist *g;
    Inst *inst = NULL;
    Transform *t;
    int maxel, nel;
    register int i;
    char ch;
    int binary = 0;
    int sawheader = 1;

    ch = fnextc (file, 0);
    switch(ch) {
    case 'G':
	if(fexpectstr (file, "GROUP")) return NULL;
	break;
    case 'T':
        if(fexpectstr (file, "TLIST")) return NULL;
	break;
    default:
	sawheader = 0;
    }
    if (sawheader && (fnextc(file, 1) == 'B')) {
	if(fexpectstr(file, "BINARY")) return NULL;
	binary = 1;
	if(fnextc(file, 1) == '\n') (void) fgetc(file);
    }

    if(binary) {
	/*
	 * TLIST BINARY ... \n
	 * <number of tlist elements as a 32-bit int>
	 * <that many 4x4 matrices>
	 */
	if(fgetni(file, 1, &nel, binary) != 1 || maxel < 0 || maxel > 9999999)
	    return NULL;
	t = OOGLNewN(Transform, nel);
	if(t == NULL)
	    return NULL;
	if(fgettransform(file, nel, (float *)t, 1) < nel) {
	    OOGLFree(t);
	    return NULL;
	}
    } else {
	maxel = 7;
	nel = 0;
	t = OOGLNewN(Transform, maxel);
	for(;;) {
	    i = fgettransform(file, maxel-nel, (float *)(t + nel), binary);
	    if(i <= 0)
		break;
	    nel += i;
	    if(nel >= maxel) {
		maxel *= 3;
		t = OOGLRenewN(Transform, t, maxel);
		if(t == NULL) {
		     OOGLError(1,"TlistFLoad: couldn't malloc %d Transforms for %s",
			    maxel, fname);
		     return NULL;
		}
	    }
	}

	if(i != 0 || ferror(file)) {
	    if(sawheader)
		OOGLError(1,
		    "TlistFLoad: bad tlist file format or read error in %s",
		fname);
	    OOGLFree(t);
	    return NULL;
	}
    }


    g = nel>0 ? (Tlist *) GeomCreate("tlist", CR_NELEM, nel, CR_ELEM, t, NULL) : (Tlist *) NULL;
    OOGLFree(t);

    if(g != NULL && fexpecttoken(file, "tlist") == 0) {
	if(GeomFLoadEmbedded(&g->tlist, &g->tlisthandle, file, fname))
	    OOGLError(1/*Warn*/, "TlistFLoad: couldn't load 'tlist'");
    }
    if(g != NULL && (fexpecttoken(file, "unit") == 0
		  || fexpecttoken(file, "geom") == 0)) {
	/*
	 * Compatibility mode -- create an Inst comprising our Tlist and
	 * the given object.
	 */
	Geom *geom = NULL;
	Handle *geomhandle = NULL;

	if(GeomFLoadEmbedded(&geom, &geomhandle, file, fname))
	    OOGLError(1/*Warn*/, "TlistFLoad: couldn't load 'unit' from %s", fname);
	else
	    inst = (Inst *)GeomCCreate(NULL, InstMethods(),
		CR_TLIST, g, CR_GEOM, geom, CR_GEOMHANDLE, geomhandle, NULL);
    }
    return inst ? (Geom *)inst : (Geom *)g;
}

Geom *
TlistLoad( name )
    char *name;
{
    FILE *file;
    Geom *g;

    file = fopen( name, "r" );
    if( file == NULL ) {
	OOGLError(1, "TlistLoad: Can't open %s: %s", name, sperror());
	return NULL;
    }
    g = TlistFLoad( file, name );
    fclose(file);

    return g;
}
