/* Copyright (c) 1992 The Geometry Center; University of Minnesota
   1300 South Second Street;  Minneapolis, MN  55454, USA;
   
This file is part of geomview/OOGL. geomview/OOGL is free software;
you can redistribute it and/or modify it only under the terms given in
the file COPYING, which you should have received along with this file.
This and other related software may be obtained via anonymous ftp from
geom.umn.edu; email: software@geom.umn.edu. */

/* Authors: Charlie Gunn, Stuart Levy, Tamara Munzner, Mark Phillips */

#include <math.h>
#include "polylistP.h"
#include "pickP.h"

#define	VSUB3(a, b, dst)  { (dst).x = (a).x - (b).x; \
			    (dst).y = (a).y - (b).y; \
			    (dst).z = (a).z - (b).z; }
#define	VSUB2(a, b, dst)  { (dst).x = (a).x - (b).x; \
			    (dst).y = (a).y - (b).y; }
#define VNORM2(a)	sqrt( (a).x*(a).x + (a).y*(a).y )
#define	VDOT2(v1, v2)	( (v1).x*(v2).x + (v1).y*(v2).y )
#define SIGN(x)		( ((x) >= 0) ? 1 : -1 )

#include "polylistP.h"

/*static*/ int PolyPick(Poly *poly, Pick *p, Appearance *ap, Transform T);
static float angle(Point3 *a, Point3 *b);
static void poly_origin(Poly *poly, Point3 *pt, Transform T);
static int poly_bounds(Poly *poly, Transform T, float thresh);
static void edge_compute(Point3 *a, Point3 *b, Point3 *pt,
	     float *d, float *t, float *len );
static void nearest_to_origin(Point3 *pt, Point3 *a, Point3 *b, float *t, float *len);

PolyList *
PolyListPick(PolyList *pl, Pick *p, Appearance *ap, Transform T)
{
  int fi, found;
  Poly *poly;

  found = 0;
  for (fi=0, poly=pl->p; fi<pl->n_polys; ++fi, ++poly) {
    found |= PolyPick(poly, p, ap, T);
  }
  if (found) return pl;
  else return NULL;
}

int
PolyPick(Poly *poly, Pick *p, Appearance *ap, Transform T)
{
  int i,j, found;
  float angsum, d, t, len;
  Point3 pi, pj, pt;

  if (! poly_bounds(poly, T, 0.) ) return 0;

  angsum = 0;
  found = 0;
  for (i=0; i<poly->n_vertices; ++i) {
    j = (i + 1) % poly->n_vertices;
    HPt3TransPt3(T, &(poly->v[i]->pt), &pi);
    HPt3TransPt3(T, &(poly->v[j]->pt), &pj);
    edge_compute( &pi, &pj, &pt, &d, &t, &len );
    if (d < p->thresh) {
      if (t > -p->thresh) {
	if (t < p->thresh) {
	  /* we're over v[i] */
	  if (pt.z < p->got.z) {
	    p->got = pt;
	    if (p->want & PW_VERT) {
	      found |= PW_VERT;
	      p->v = pi;
	    }
	    if (p->want & PW_EDGE) {
	      found |= PW_EDGE;
	      p->e[0] = pi;
	      p->e[1] = pj;
	    }
	    break;
	  }
	} else if (t < len - p->thresh) {
	    /* we're over edge [pi,pj]  */
	    if (pt.z < p->got.z) {
	      p->got = pt;
	      if (p->want & PW_EDGE) {
		found |= PW_EDGE;
		p->e[0] = pi;
		p->e[1] = pj;
	      }
	      break;
	    }
	  } else if (t < len + p->thresh) {
	    /* we're over v[j] */
	    if (pt.z < p->got.z) {
	      p->got = pt;
	      if (p->want & PW_VERT) {
		found |= PW_VERT;
		p->v = pj;
	      }
	      if (p->want & PW_EDGE) {
		found |= PW_EDGE;
		p->e[0] = pi;
		p->e[1] = pj;
	      }
	      break;
	    }
	  }
      }
    }
    angsum += angle(&pi, &pj);
  }
  if (poly->n_vertices > 2) {
    if (fabs(angsum/M_2_PI) > 0.5) {
      poly_origin(poly, &pt, T);
      if (pt.z < p->got.z) {
	p->got = pt;
	found |= PW_FACE;
      }
    }
  }
  if (found) {
    p->found = found;
    TmCopy(T, p->Tprim);
    return 1;
  }
  return 0;
}

static float
angle(Point3 *a, Point3 *b)
{
  float ans;

  ans = SIGN( a->x*b->y - a->y*b->x )
        * acos( VDOT2(*a,*b) / sqrt((double)(VDOT2(*a,*a) * VDOT2(*b,*b))) );
  return ans;
}

/*-----------------------------------------------------------------------
 * Function:	edge_compute
 * Description:	do computations associated with the edge of a polygon
 * Args:	a,b: INPUT (see below)
 *		pt,d,t,len: OUTPUT (see below)
 * Returns:	nothing
 * Author:	mbp
 * Date:	Fri Mar 13 20:11:33 1992
 * Notes:	All points are considered to be in the (x,y) plane; 
 *		z coord is ignored.
 *
 *		Sets
 *		  pt to the point of the line [a,b] closest to the
 *		    origin
 *		  t to the signed distance from a to pt, where + is
 *		    in the direction of b
 *		  d to the distance from pt to the origin
 *		  len to the distance from a to b
 */
static void
edge_compute(Point3 *a, Point3 *b, Point3 *pt,
	     float *d, float *t, float *len )
{
  nearest_to_origin(pt, a, b, t, len);
  *d = VNORM2(*pt);
}

/*-----------------------------------------------------------------------
 * Function:	nearest_to_origin
 * Description:	computations related to the nearest point of a line
 *		  to the origin
 * Args:	pt, t, len: OUTPUT (see below)
 *		*a, *b: INPUT (see below)
 * Returns:	nothing
 * Author:	mbp
 * Date:	Fri Mar 13 20:00:26 1992
 * Notes:	All points are considered to be in the (x,y) plane; 
 *		z coord is ignored.
 *	
 *		Sets
 *		  pt to the point of the line [a,b] closest to the
 *		    origin
 *		  t to the signed distance from a to pt, where + is
 *		    in the direction of b
 *		  len to the distance from a to b
 */
static void
nearest_to_origin(Point3 *pt, Point3 *a, Point3 *b, float *t, float *len)
{
  Point3 bma;
  register float f;

  VSUB3(*b, *a, bma);
  *len = VNORM2(bma);
  *t = - VDOT2(*a, bma) / *len;

  f = *t / (*len);
  pt->x = a->x + f * bma.x;
  pt->y = a->y + f * bma.y;
  pt->z = a->z + f * bma.z;
}

/*-----------------------------------------------------------------------
 * Function:	poly_origin
 * Description:	compute the point on a polygon under the screen origin
 * Args:	*poly: the polygon
 *		*pt: the answer point
 *		T: world to screen xform
 * Author:	mbp
 * Date:	Sun Mar 15 13:34:15 1992
 * Notes:	Always sets pt.x and pt.y are always 0; sets pt.z
 *		  to the z screen coord of the point on poly under
 *		  (0,0).
 *
 *		Hunts for the first 3 verts of poly which are non-collinear
 *		and uses them to determine the plane of the poly.
 */
static void
poly_origin(Poly *poly, Point3 *pt, Transform T)
{
  int bi=0, ci;
  Point3 a,b,c;
  float pz,pw;
  static Point3 default_pt = {0,0,2};

  /* a = 1st vertex, in screen coords */
  HPt3TransPt3(T, &(poly->v[0]->pt), &a);

  /* b = next vertex different from a, in screen coords */
  while (bi<poly->n_vertices) {
    if (bcmp(&(poly->v[0]->pt), &(poly->v[bi]->pt), sizeof(HPoint3))) {
      HPt3TransPt3(T, &(poly->v[bi]->pt), &b);
      break;
    } else {
      ++bi;
    }
  }
  if (bi>=poly->n_vertices) {
    OOGLError("interior hit on polygon whose %1d verts are all (%f,%f,%f,%f)",
	      poly->n_vertices,
	      poly->v[0]->pt.x, poly->v[0]->pt.y, poly->v[0]->pt.z, poly->v[0]->pt.w);
    *pt = default_pt;
    return;
  }

  /* c = next vertex after b non-collinear with a & b, in screen coords */
  ci = bi+1;
  while (ci<poly->n_vertices) {
    HPt3TransPt3(T, &(poly->v[ci]->pt), &c);
    pz = a.x * (b.y - c.y) - a.y * (b.x - c.x) + (b.x * c.y - b.y * c.x);
    if (pz != 0) break;
    ++ci;
  }
  if (ci>=poly->n_vertices) {
    OOGLError("interior hit on 0-area polygon with vertices\n\
               (%f, %f, %f, %f)\n\
               (%f, %f, %f, %f)...",
	      poly->v[0]->pt.x,poly->v[0]->pt.y,poly->v[0]->pt.z,poly->v[0]->pt.w,
	      poly->v[1]->pt.x,poly->v[1]->pt.y,poly->v[1]->pt.z,poly->v[1]->pt.w);
    *pt = default_pt;
    return;
  }

  pw =   a.x * ( b.z * c.y - b.y * c.z )
       - a.y * ( b.z * c.x - b.x * c.z )
       + a.z * ( b.y * c.x - b.x * c.y );

  pt->x = pt->y = 0;
  pt->z = - pw / pz;
}

/*-----------------------------------------------------------------------
 * Function:	poly_bounds
 * Description:	determine whether the screen origin lies within a polygon's
 *		  screen projection
 * Args:	*poly: the polygon
 *		T: object->world projection
 *		thresh: tolerance
 * Returns:	1 if yes, 0 if no
 * Author:	mbp
 * Date:	Sun Mar 15 18:06:14 1992
 */
static int
poly_bounds(Poly *poly, Transform T, float thresh)
{
  Point3 pt;
  register Vertex **v;
  register float xmin,xmax,ymin,ymax;
  register int i,n;

  n = poly->n_vertices;
  v = poly->v;
  xmin = ymin = 2;
  xmax = ymax = -2;
  for (i=0; i<n; ++i) {
    HPt3TransPt3(T, &(v[i]->pt), &pt);
    if (pt.x < xmin) xmin = pt.x;
    if (pt.x > xmax) xmax = pt.x;
    if (pt.y < ymin) ymin = pt.y;
    if (pt.y > ymax) ymax = pt.y;
  }
  if (xmin > thresh) return 0;
  if (xmax < -thresh) return 0;
  if (ymin > thresh) return 0;
  if (ymax < -thresh) return 0;
  return 1;
}
