/* Copyright (c) 1992 The Geometry Center; University of Minnesota
   1300 South Second Street;  Minneapolis, MN  55454, USA;
   
This file is part of geomview/OOGL. geomview/OOGL is free software;
you can redistribute it and/or modify it only under the terms given in
the file COPYING, which you should have received along with this file.
This and other related software may be obtained via anonymous ftp from
geom.umn.edu; email: software@geom.umn.edu. */

/* Authors: Charlie Gunn, Stuart Levy, Tamara Munzner, Mark Phillips */

 /*
  * Load a PolyList object from an "off"-format file.
  * 
  */

#include <string.h>
#include "polylistP.h"

#define	USE_CMAP	1
#define	GOT_CMAP	2
#define	SIZEOF_CMAP	256
static struct 	{short r,g,b,a; }	colormap[SIZEOF_CMAP];

static int
LoadCmap(char *file)
{
	int cnt;
	char *index;
	float r,g,b,a;
	FILE *fp;
	fp = fopen(file,"r");
	if (fp == NULL)	return(-1);
	
	index = (char *) rindex(file, '.');	/* look at last suffix */
	if (index  == 0 )	return (-1);
	
	cnt = 0;
	/* check for float cmap (".fmap") or int cmap (".cmap" or ".imap") */
	if (strcmp(index,".fmap")== 0)	
		while (fscanf(fp,"%f%f%f%f",&r,&g,&b,&a) == 4 && cnt < SIZEOF_CMAP)
			{
			colormap[cnt].r = 255*r;
			colormap[cnt].g = 255*g;
			colormap[cnt].b = 255*b;
			colormap[cnt].a = 255*a;
			cnt++;
			}
			
	else if (strcmp(index,".cmap") == 0  || strcmp(index,".imap") == 0)
		while (fscanf(fp,"%d%d%d%d",&colormap[cnt].r,&colormap[cnt].g,
			&colormap[cnt].b,&colormap[cnt].a) == 4 && cnt < SIZEOF_CMAP)  cnt++;
	fclose(fp);
}

PolyList *
PolyListFLoad(FILE *file, char *fname)
{
    register PolyList *pl;
    int edges;
    register int i;
    register Vertex *v;
    int cmap_flag = 0;
    int binary = 0;
    int flags = 0;
    int makenorm = 0;
    int dimn = 3;

    if (file == NULL)
	return NULL;

    (void) fnextc(file, 0);
    i = fgetc(file);
    if(i == 'C') {
	flags = PL_HASVCOL;	/* Per-vertex colors */
	i = fgetc(file);
    }
    if(i == 'N') {
	flags |= PL_HASVN;	/* Per-vertex normals */
	i = fgetc(file);
    }
    if(i == '4') {
	flags |= PL_4D;		/* true 4D points */
	dimn = 4;
	i = fgetc(file);
    }
    if(i == 'S') {		/* "smooth surface": we compute vtx normals */
	makenorm = 1;
	i = fgetc(file);
    }
    if(i == 'O') {		/* Skip blanks & comments */
	(void) fexpectstr(file, "FF");	/* Ignore optional "OFF" header */

	if(fnextc(file, 1) == 'B' && fexpectstr(file, "BINARY") == 0) {
	    binary = 1;
	    if(fnextc(file, 1) != '\n')	/* Expect \n after BINARY */
		return NULL;
	    (void) getc(file);		/* Toss \n */
	}
    } else {
	ungetc(i, file);
    }


    pl = OOGLNewE(PolyList, "PolyListFLoad PolyList");
    GGeomInit(pl, PolyListMethods(), PLMAGIC, NULL);
    pl->p = NULL;
    pl->vl = NULL;

    pl->flags = flags;

    if(fgetni(file, 1, &pl->n_verts, binary) <= 0 ||
       fgetni(file, 1, &pl->n_polys, binary) <= 0 ||
       fgetni(file, 1, &edges, binary) <= 0)
		goto bogus;

    pl->vl = OOGLNewNE(Vertex, pl->n_verts, "PolyListFLoad vertices");

    for(v = pl->vl, i = 0; i < pl->n_verts; v++, i++) {
	if(fgetnf(file, dimn, (float *)&v->pt, binary) < dimn ||
	   flags & PL_HASVN && fgetnf(file, 3, (float *)&v->vn, binary) < 3 ||
	   flags & PL_HASVCOL && fgetnf(file, 4, (float *)&v->vcol, binary) < 4) {
		OOGLError(0/*Syntax*/, "PolyListFLoad: %s: Error reading vertex %d/%d", fname, i, pl->n_verts);
		goto bogus;
	}
	if (dimn == 3)  v->pt.w = 1.0;
    }

    pl->p = OOGLNewNE(Poly, pl->n_polys, "PolyListFLoad polygons");
    for(i = 0; i < pl->n_polys; i++) {
	register Poly *p;
	register int k,index,isfloat,m;
	float rgba[4];

	p = &pl->p[i];
	if(fgetni(file, 1, &p->n_vertices, binary) <= 0 || p->n_vertices <= 0) {
	   OOGLError(1/*Syntax*/, "PolyListFLoad: %s: bad %d/%d'th polygon",
		fname, i, pl->n_polys);
	   goto bogus;
	}
	p->v = OOGLNewNE(Vertex *, p->n_vertices, "PolyListFLoad polygon");

	for(k = 0; k < p->n_vertices; k++) {
	    int index;

	    if(fgetni(file, 1, &index, binary) <= 0 ||
	       index < 0 || index >= pl->n_verts) {
		    OOGLError(1/*Syntax*/,
			"PolyListFLoad: %s: bad index %d on %d/%d'th polygon", 
			fname, index, k, p->n_vertices);
		    pl->n_polys = i+1;
		    goto bogus;
	    }
	    p->v[k] = &pl->vl[index];
	}

	/* Pick up the color, if any.
	 * In ASCII format, take whatever's left before end-of-line
	 */
	if(binary) {
	    int ncol;

	    if(fgetni(file, 1, &ncol, 1) <= 0
	       || fgetnf(file, ncol, &rgba[0], 1) < ncol)
		goto bogus;
	    k = ncol;
	} else {
	    for(k = 0; k < 4 && fnextc(file, 1) != '\n'; k++) {
		if(fgetnf(file, 1, &rgba[k], 0) < 1)
		    goto bogus;
	    }
	}

	if((flags & PL_HASVCOL) == 0) {
	    if(k > 0)
		pl->flags |= PL_HASPCOL;

	    for (isfloat = 1, m=0; m<k; ++m)
		if (rgba[m] > 1.0)	isfloat = 0;

	    switch(k) {
	    case 0:
		rgba[0] = isfloat ? 170/255.0 : 170;		/* Gray */
		rgba[1] = rgba[0];
	    case 2:
		rgba[2] = rgba[1];
	    case 3:
		rgba[3] = isfloat ? 170/255.0 : 170;	/* Semitransparent */
		break;
	    case 1:				/* use colormap */
		if ( !(cmap_flag & GOT_CMAP))	
			{
			LoadCmap("/u/gcg/grap/data/cmap.fmap");
			cmap_flag |= GOT_CMAP;
			}
		if (rgba[0] >= SIZEOF_CMAP)	index = SIZEOF_CMAP - 1;
		else	index = rgba[0];
		rgba[0] = colormap[index].r;
		rgba[1] = colormap[index].g;
		rgba[2] = colormap[index].b;
		rgba[3] = colormap[index].a;
		isfloat = 0;
	    }				/* case 4, all components supplied */
	}
	/* here should check to see if numbers are small */
	/* this only can happen if file contains 4 small numbers */
	/* ( examine above switch) */ 
	if (isfloat)
		{
		p->pcol.r = rgba[0] ;
		p->pcol.g = rgba[1] ;
		p->pcol.b = rgba[2] ;
		p->pcol.a = rgba[3] ;
		}
	else
		{
		p->pcol.r = rgba[0] / 255.0;
		p->pcol.g = rgba[1] / 255.0;
		p->pcol.b = rgba[2] / 255.0;
		p->pcol.a = rgba[3] / 255.0;
		}
    }

    if(makenorm && !(flags & PL_HASVN)) {
	pl->flags |= PL_HASVN;
	pl->flags &= ~PL_HASPN;		/* Leave vertex-normals only */
    }
    return pl;

  bogus:
    GeomDelete((Geom *)pl);
    return NULL;
}


PolyList *
PolyListLoad(name)
    char *name;
{
    PolyList *pl;
    FILE *inf;

    if((inf = fopen(name, "r")) == NULL) {
	perror(name);
	return NULL;
    }
    pl = PolyListFLoad(inf, name);
    fclose(inf);

    return (pl);
}
