/* Copyright (c) 1992 The Geometry Center; University of Minnesota
   1300 South Second Street;  Minneapolis, MN  55454, USA;
   
This file is part of geomview/OOGL. geomview/OOGL is free software;
you can redistribute it and/or modify it only under the terms given in
the file COPYING, which you should have received along with this file.
This and other related software may be obtained via anonymous ftp from
geom.umn.edu; email: software@geom.umn.edu. */

/* Authors: Charlie Gunn, Stuart Levy, Tamara Munzner, Mark Phillips */

#include <ctype.h>
#include "meshP.h"

static int
getmeshvert(file, flag, u, v, p, n, c, t)
	FILE	*file;
	register int	flag;
	int	u;
	int	v;
	register HPoint3	*p;
	Point3	*n;
	ColorA	*c;
	Point3	*t;		/* actually u, the texture parameter */
{
	float	inputs[4];
	Point3  p3;

	if (flag & MESH_Z) {
		p->x = (float) u;
		p->y = (float) v;
		p->w = 1.0;
		if (fgetnf(file, 1, &p->z, flag&MESH_BINARY) <= 0)
			return 0;
	} else if (flag & MESH_4D) {
		if (fgetnf(file, 4, (float *)p, flag&MESH_BINARY) < 4)
			return 0;
	} else 			{
		if (fgetnf(file, 3, (float *)p, flag&MESH_BINARY) < 3)
			return 0;
		p->w = 1.0;
	}

	if (flag & MESH_N && fgetnf(file, 3, (float *)n, 0) < 3)
		return 0;

	if (flag & MESH_C && fgetnf(file, 4, (float *)c, 0) < 4)
		return 0;

	if (flag & MESH_U && fgetnf(file, 3, (float *)t, 0) < 3)
		return 0;

	return 1;
}

static char oldbinary;	/* Old binary format -- has 3-component colors */

static int
getheader(file)
	FILE	*file;
{
	register int c, i, flag;
	static char keys[] = "CNZ4Uuv";
	static short bit[] =
		{ MESH_C, MESH_N, MESH_Z, MESH_4D, MESH_U, MESH_UWRAP, MESH_VWRAP };

	    /* Parse [C][N][Z][4][U][u][v]MESH[ BINARY]\n */
	if(fnextc(file, 0) == '\0') {
	    /* Looks like first byte of magic number of old binary mesh */
	    oldbinary = 1;
	    return(MESH_BINARY);
	}

	oldbinary = 0;
	flag = 0;
	c = fgetc(file);
	for(i = 0; keys[i] != '\0'; i++) {
	    if(c == keys[i]) {
		flag |= bit[i];
		c = fgetc(file);
	    }
	}
	ungetc(c, file);
	if(fexpectstr(file, "MESH"))
	    return(-1);

	if(fnextc(file, 1) == 'B') {
	    if(fexpectstr(file, "BINARY"))
		return(-1);
	    flag |= MESH_BINARY;
	    if(fnextc(file, 1) == '\n')
		(void) fgetc(file);	/* Toss \n */
	}
	return(flag);
}


Mesh *
MeshFLoad(file, fname)
	FILE *file;
	char *fname;
{
	Mesh	m;
	int	n;
	Point3	*p3;
	register int i, u, v;

	if (!file)
		return NULL;

	if((m.flag = getheader(file)) == -1)
		return NULL;

	if (m.flag & MESH_BINARY)
	{
	    int magic;
	    char bogus[6*4];

	    /* This is a hack, for backward compatibility:
	     * Old implementations just wrote out the raw Mesh header,
	     * i.e. int magic, flag, nu, nv;
	     * float umin, umax, vmin, vmax;
	     * and six 32-bit pointers which were ignored.
	     * We emulate this.
	     * NOTE this code assumes the
	     * layout of the above values in Mesh isn't changed!
	     */
	    if(	    fgetni(file, 1, &magic, 1) <= 0 || magic == 0 ||
		    fgetni(file, 4, &m.flag, 1) < 4 ||
		    fgetnf(file, 4, (float *)&m.umin, 1) < 4 ||
		    fread(bogus, sizeof(bogus), 1, file) < 1 ||
		    m.nu <= 0 || m.nu > 100000 || m.nv <= 0 || m.nv > 100000) {
		return NULL;
	    }

	    m.p = OOGLNewNE(HPoint3, m.nu * m.nv, "MeshFLoad: vertices");
	    m.n = NULL;
	    m.u = NULL;
	    m.c = NULL;
	    m.d = NULL;
	    m.nd = NULL;

	    n = m.nu * m.nv;

	    if (m.flag & MESH_4D)	{
	        if(fgetnf(file, 4*n, (float *)m.p, 1) < 4*n)
		    return NULL;
	 	}
	    else	{
		p3 = OOGLNewNE(Point3, n, "3D mesh points");
	        if(fgetnf(file, 3*n, (float *)p3, 1) < 3*n)
		    return NULL;
		Pt3ToPt4(p3, m.p, n);
		OOGLFree(p3);
	 	}

	    if (m.flag & MESH_N) {
		m.n = OOGLNewNE(Point3, n, "MeshFLoad: normals");
		if(fgetnf(file, 3*n, (float *)m.n, 1) < 3*n)
		    return NULL;
	    }
	    if (m.flag & MESH_C) {
		m.c = OOGLNewNE(ColorA, n, "MeshFLoad: colors");
		if(oldbinary) {
		    for(i = 0; i < n; i++) {
			if(fgetnf(file, 3, (float *)&m.c[i], 1) < 3)
			    return NULL;
			m.c[i].a = 1.0;
		    }
		} else {
		    if(fgetnf(file, 4*n, (float *)m.c, 1) < 4*n)
			return NULL;
		}
	    }
	    if (m.flag & MESH_U) {
		m.u = OOGLNewNE(Point3, n, "MeshFLoad: texture coords");
		if(fgetnf(file, 3*n, (float *)m.u, 1) <= 0)
		    return NULL;
	    }
	} else {
	    if (fgetni(file, 1, &m.nu, 0) <= 0 ||
		fgetni(file, 1, &m.nv, 0) <= 0)
		return NULL;

	    n = m.nu * m.nv;

	    m.p = OOGLNewNE(HPoint3, n, "MeshFLoad: vertices");
	    m.n = NULL;
	    m.u = NULL;
	    m.c = NULL;
	    m.d = NULL;
	    m.nd = NULL;

	    if (m.flag & MESH_N) {
		    if((m.n = OOGLNewNE(Point3, n, "MeshFLoad: normals")) == NULL)
			return NULL;
	    }
	    if (m.flag & MESH_C) {
		    if((m.c = OOGLNewNE(ColorA, n, "MeshFLoad: colors")) == NULL)
			return NULL;
	    }
	    if (m.flag & MESH_U) {
		    if((m.u = OOGLNewNE(Point3, n, "MeshFLoad: texture coords")) == NULL)
			return NULL;
	    }
	    for (i = 0, u = 0; u < m.nu; u++) {
		for (v = 0; v < m.nv; v++, i++) {
		    if(getmeshvert(file, m.flag, u, v,
			    &m.p[i], &m.n[i], &m.c[i], &m.u[i]) == 0) {
			OOGLError(1,
		    "MeshFLoad: syntax error in %s element (%d,%d) of (%d,%d)",
				fname,u,v,m.nu,m.nv);
			return NULL;
		    }
		}
	    }
	}
	return (Mesh *) GeomCreate ("mesh", CR_NOCOPY,
		CR_FLAG, m.flag, CR_NU, m.nu,
		CR_NV, m.nv, CR_POINT4, m.p, CR_COLOR, m.c, CR_NORMAL, m.n,
		CR_U, m.u, NULL);
}


Mesh *
MeshLoad(name)
	char	*name;
{
	Mesh	*m;
	FILE	*inf;
	int	n;

	inf = fopen(name, "r");
	if (inf == NULL)
	{
		OOGLError(1,"MeshLoad: can't open %s: %s", name, sperror());
		return NULL;
	}
	m = MeshFLoad(inf, name);
	fclose(inf);

	return m;
}
