/* Copyright (c) 1992 The Geometry Center; University of Minnesota
   1300 South Second Street;  Minneapolis, MN  55454, USA;
   
This file is part of geomview/OOGL. geomview/OOGL is free software;
you can redistribute it and/or modify it only under the terms given in
the file COPYING, which you should have received along with this file.
This and other related software may be obtained via anonymous ftp from
geom.umn.edu; email: software@geom.umn.edu. */
static char *copyright = "Copyright (C) 1992 The Geometry Center";

/* Authors: Charlie Gunn, Stuart Levy, Tamara Munzner, Mark Phillips */

#include "listP.h"

/*
 * List editing, deletion and creation.
 */
Geom *
ListReplace( list, geom )
    List *list;
    Geom *geom;
{
    register List *l;
    Geom *g;

    if(list == NULL)
	return NULL;
    g = list->car;
    list->car = geom;
    return g;
}

Geom *
ListRemove( Geom *list, Geom *g )
{
    register List *l;
    register List **prev;

    if(list == NULL)
	return NULL;

    if(list->class != ListClass) {
	OOGLError(1, "ListRemove: %x is a %s not a List!", list, GeomName(list));
	return NULL;
    }
    for(prev = (List **)&list; (l = *prev) != NULL; prev = &l->cdr) {
	if(l->car == g) {
	    *prev = l->cdr;
	    l->cdr = NULL;
	    GeomDelete( (Geom *)l );
	    break;
	}
    }
    return list;
}

/*
 * Delete a List.
 * Don't use a for-loop over all the list elements;
 * someone else might have pointers to them.
 */
void
ListDelete(register List *l)
{
    if(l->cdr) GeomDelete((Geom *)l->cdr);
    if(l->car) GeomDelete(l->car);
    if(l->carhandle) HandlePDelete(&l->carhandle);
}

int
ListGet(register List *l, int attr, register void *attrp)
{
    switch(attr) {
    case CR_GEOM: *(Geom **)attrp = l->car; break;
    case CR_GEOMHANDLE: *(Handle **)attrp = l->carhandle; break;
    case CR_CDR: *(Geom **)attrp = (Geom *)l->cdr; break;
    default: return -1;
    }
    return 1;
}

Geom *
ListAppend(Geom *lg, Geom *g)
{
  List *new = OOGLNewE(List, "ListAppend: List");
  register List *l = (List*)lg;

  if(l && l->class->delete != (GeomDeleteFunc *)ListDelete) {
    OOGLError(0, "ListAppend: attempt to append to a %s, not a List",
	GeomName((Geom *)l));
    return NULL;
  }

  new->car = g;
  new->cdr = NULL;
  if (l) {
    while (l->cdr) l = l->cdr;
    l->cdr = new;
    GGeomInit(new, lg->class, lg->magic, NULL);
  }
  else {
    l = new;
    GGeomInit(new, ListClass, LISTMAGIC, NULL);
  }
  new->carhandle = NULL;
  return lg ? lg : (Geom *)new;
}

List *
ListCreate (List *exist, GeomClass *classp, va_list a_list )
{
    register List *list, *l;
    int attr, copy = 1;
    Handle *h;
    Geom *g;

    if (exist == NULL) {
	list = OOGLNewE( List, "ListCreate: new List" );
	GGeomInit(list, classp, LISTMAGIC, NULL);
	list->cdr = NULL;
	list->carhandle = NULL;
	list->car = NULL;
    } else {
	if(exist->class != classp) {
	    OOGLError(0, "ListCreate: existing_value %x (magic %x) not a List",
		exist, exist->magic);
	    return NULL;
	}
	list = exist;
    }

    while (attr = va_arg (a_list, int)) switch (attr) {
	case CR_HANDLE_GEOM:
	    /*
	     * Like GeomReplace, but takes a Handle too.
	     */
	    h = va_arg(a_list, Handle *);
	    g = va_arg(a_list, Geom *);
	    if(copy) {
		RefIncr((Ref *)h);
		RefIncr((Ref *)g);
	    }
	    if(g == NULL && h != NULL)
		g = (Geom *)HandleObject(h);
	    GeomDelete(list->car);
	    HandlePDelete(&list->carhandle);
	    list->car = g;
	    list->carhandle = h;
	    if(h)
		HandleRegister(&list->carhandle, (Ref *)list, &list->car, HandleUpdRef);
	    break;
	    
	case CR_GEOM:	/* == CR_CAR */
	    if(list->car != NULL || list->carhandle != NULL) {
		l = OOGLNewE(List, "ListCreate: List");
		l->car = list->car;
		l->carhandle = list->carhandle;
		RefIncr((Ref *)list->carhandle);
		l->cdr = list->cdr;
		list->cdr = l;
	    }
	    list->car = va_arg (a_list, Geom *);
	    if(copy) RefIncr((Ref *)list->car);
	    break;

	case CR_GEOMHANDLE:
	    if(list->car != NULL || list->carhandle != NULL) {
		l = OOGLNewE(List, "ListCreate: List");
		l->car = list->car;
		l->carhandle = list->carhandle;
		l->cdr = list->cdr;
		list->car = NULL;
	    }
	    h = va_arg(a_list, Handle *);
	    if(copy) RefIncr((Ref *)h);
	    HandlePDelete(&list->carhandle);
	    list->carhandle = h;
	    HandleRegister(&list->carhandle, (Ref *)list, &list->car, HandleUpdRef);
	    break;

	case CR_CDR:
	    l = va_arg (a_list, List *);
	    if(l && l->class != classp) {
		OOGLError(0, "ListCreate: CDR %x (magic %x) not a List",
			l, l->magic);
		goto fail;
	    }
	    if(list->cdr)
		GeomDelete((Geom *)list->cdr);
	    if(copy) RefIncr((Ref *)l);
	    list->cdr = l;
	    break;
	default:
	    if (GeomDecorate (list, &copy, attr, &a_list)) {
		OOGLError (0, "ListCreate: Undefined attribute: %d", attr);

	      fail:
		if(exist == NULL)
		    GeomDelete ((Geom *)list);
		return NULL;
	    }
    }

    return list;
}
