/* Copyright (c) 1992 The Geometry Center; University of Minnesota
   1300 South Second Street;  Minneapolis, MN  55454, USA;
   
This file is part of geomview/OOGL. geomview/OOGL is free software;
you can redistribute it and/or modify it only under the terms given in
the file COPYING, which you should have received along with this file.
This and other related software may be obtained via anonymous ftp from
geom.umn.edu; email: software@geom.umn.edu. */

/* Authors: Charlie Gunn, Stuart Levy, Tamara Munzner, Mark Phillips */

#include "bezierP.h"

Bezier *
BezierDice( Bezier *bezier, int nu, int nv )
{
    if(nu < 2) nu = BEZ_DEFAULT_MESH_SIZE;
    if(nv < 2) nv = BEZ_DEFAULT_MESH_SIZE;
    if(nu != bezier->nu || nv != bezier->nv) {
	bezier->flag |= BEZ_REMESH;
	bezier->nu = nu;
	bezier->nv = nv;
    }
    return bezier;
}

/* this gets called when it's time to remesh, so the current mesh is destroyed
   and a new one is allocated */
Mesh *
BezierReDice(bezier)
    Bezier *bezier;
{
    Mesh 	mesh;
    register int i,j, k, m, n;
    int		index, offset;
    int		dimn;
    register float 	*fptr, *med;
    register HPoint3	*bmp;
    float 	p[(MAX_BEZ_DEGREE+1)*MAX_BEZ_DIMN];
    float	delta;

    if (bezier->nu <= 1) 	bezier->nu = BEZ_DEFAULT_MESH_SIZE;
    if (bezier->nv <= 1) 	bezier->nv = BEZ_DEFAULT_MESH_SIZE;

    /* clean out what's here if the size is wrong */
    if (bezier->mesh)
      /*if (bezier->nu != bezier->mesh->nu || bezier->nv != bezier->mesh->nv)*/
	{
	GeomDelete((Geom *)bezier->mesh);
	bezier->mesh = NULL;
	}

    dimn = bezier->dimn;
    /* create a "dummy" mesh to pass to MeshCreate */
    mesh.nu = bezier->nu;
    mesh.nv = bezier->nv;
    mesh.flag = 0;
    mesh.p = NULL;
    mesh.c = NULL;
    if (bezier->flag & BEZ_C)	mesh.flag |= MESH_C;
    if (bezier->dimn == 4)	mesh.flag |= MESH_4D;

    mesh.p = OOGLNewNE(HPoint3, mesh.nu*mesh.nv, "BezierReDice: mesh points");

    if (mesh.flag & MESH_C)
      mesh.c = OOGLNewNE(ColorA, mesh.nu*mesh.nv, "BezierReDice: mesh colors");

    /* compute first pass: interpolate mesh points in u direction */
    med = OOGLNewNE(float, dimn*(bezier->degree_v+1)*bezier->nu, "BezierReDice: working mesh");
    /* for each row ... */
    for (fptr=bezier->CtrlPnts,index = 0, i=0; 
	i<bezier->degree_v+1; 
	++i, fptr+=dimn*(bezier->degree_u+1)) {	/* jump over one row */

        /* compute interpolating points on rows  */
	for (j=0; j<bezier->nu; ++j) {
	    delta = ((float) j) / (bezier->nu - 1);
	    /* initialize the interpolation array: copy this row */
	    bcopy(fptr, p, dimn*sizeof(float)*(bezier->degree_u+1));
	    for (k=0; k<bezier->degree_u; ++k)
		for (offset=0, m=0; m<bezier->degree_u-k; ++m, offset += dimn) {
		    p[offset] += delta * (p[offset+dimn] - p[offset]);
		    p[offset+1] += delta * (p[offset+dimn+1] - p[offset+1]);
		    p[offset+2] += delta * (p[offset+dimn+2] - p[offset+2]);
		    if (dimn == 4)
			p[offset+3] += delta * (p[offset+dimn+3] - p[offset+3]);
		}
	    /* copy over  this interpolated point into intermediate mesh */
	    bcopy(p, &med[index], sizeof(float) * dimn);
	    index += dimn;
	}
    }
    /* now compute second pass, filling in columns */
    bmp = mesh.p;
    for (fptr = med, index = 0, i=0; 
	i<bezier->nu; 			/* for each column... */
	++i, fptr += dimn)	/* move to next column */
	{
        /* compute interpolating points on columns  */
	for (j=0; j<bezier->nv; ++j) 
 	    {
	    delta = ((float) j) / (bezier->nv - 1);
	    /* pull out this column into the temporary array */
	    for (offset = 0, k=0; 
		k<bezier->degree_v+1; 
		++k, offset += bezier->nu*dimn)
	            bcopy(fptr+offset, p+dimn*k, dimn*sizeof(float));
	    
	    for (k=0; k<bezier->degree_v; ++k) {
		for (offset = 0, m=0; m<bezier->degree_v-k; ++m, offset += dimn) {
		    p[offset] += delta * (p[offset+dimn] - p[offset]);
		    p[offset+1] += delta * (p[offset+dimn+1] - p[offset+1]);
		    p[offset+2] += delta * (p[offset+dimn+2] - p[offset+2]);
		    if (dimn == 4)
			p[offset+3] += delta * (p[offset+dimn+3] - p[offset+3]);
		}
	    }
	    /* copy over  this interpolated point into final mesh */
	    *bmp = *(HPoint3 *)p;
	    if (dimn == 3) bmp->w = 1.0;
	    /* this should no longer be necessary but I'm leaving it to check 
	    if(dimn == 4 && p[3] != 1.0) {
		bmp->x /= p[3];
		bmp->y /= p[3];
		bmp->z /= p[3];
	    }
	    */
	    bmp++;
	}
    }
    OOGLFree(med);
    bezier->flag &= ~BEZ_REMESH;	/* turn off this bit */

    if(bezier->flag & BEZ_C) {
	float u, unu, v, unv;
	ColorA u0, u1;
	ColorA *cp;

#define INTC(c0, c1, t, unt, dest)	/* Interpolate color */ \
		dest.r = c0.r * unt + c1.r * t; \
		dest.g = c0.g * unt + c1.g * t; \
		dest.b = c0.b * unt + c1.b * t; \
		dest.a = c0.a * unt + c1.a * t; \

	cp = mesh.c;
	for(i = 0; i < mesh.nv; i++) {
	   v = (float) i / (mesh.nv - 1);
	   unv = 1 - v;
	   INTC(bezier->c[0], bezier->c[1], v, unv, u0);
	   INTC(bezier->c[2], bezier->c[3], v, unv, u1);
	   for(j = 0; j < mesh.nu; j++) {
		u = (float) j / (mesh.nu - 1);
		unu = 1 - u;
		INTC(u0, u1, u, unu, (*cp));
		cp++;
	   }
	}
    }
    GeomDelete((Geom *)bezier->mesh);
    if((bezier->mesh = (Mesh *) GeomCCreate(NULL, MeshMethods(),
		CR_NOCOPY,
		CR_FLAG,mesh.flag,
		CR_NU,mesh.nu,
		CR_NV,mesh.nv, 
		CR_POINT4,mesh.p,
		CR_COLOR,mesh.c,
		CR_END)) == NULL) {
	OOGLError(1, "BezierReDice: can't create Mesh");
	return NULL;
    }

    /* this should be replacd by an exact computation */
    MeshComputeNormals(bezier->mesh);
    return(bezier->mesh);
}
