/* Copyright (c) 1992 The Geometry Center; University of Minnesota
   1300 South Second Street;  Minneapolis, MN  55454, USA;
   
This file is part of geomview/OOGL. geomview/OOGL is free software;
you can redistribute it and/or modify it only under the terms given in
the file COPYING, which you should have received along with this file.
This and other related software may be obtained via anonymous ftp from
geom.umn.edu; email: software@geom.umn.edu. */
static char *copyright = "Copyright (C) 1992 The Geometry Center";

/* Authors: Charlie Gunn, Stuart Levy, Tamara Munzner, Mark Phillips */

#include "bezierP.h"

/*
 * Bezier creation, deletion and editing.
 */
Bezier *
BezierCopy( register Bezier *ob )
{
	register Bezier *b;
	register int n;

	if(ob == NULL) return NULL;

	b = OOGLNewE(Bezier, "new Bezier");

	*b = *ob;	/* Copy all fields */
	GGeomInit(b, BezierMethods(), BEZIERMAGIC, NULL);

	if(b->flag & BEZ_ST) {
	    if(ob->STCords == NULL) {
               OOGLError(0,"Inconsistency in BEZ_ST field of flag");
	       b->flag &= ~BEZ_ST;
            } else {
		b->STCords = OOGLNewNE(float, 4*2, "Bezier ST coords");
		bcopy(ob->STCords, b->STCords, 4*2*sizeof(float));
	    }
	} else
	    b->STCords = NULL;
	
	if(ob->CtrlPnts != NULL) {
	    n = (b->degree_u + 1) * (b->degree_v + 1) * b->dimn;
	    b->CtrlPnts = OOGLNewNE(float, n, "Bezier control points");
	    bcopy(ob->CtrlPnts, b->CtrlPnts, n * sizeof(float));
	}

	if(b->flag & BEZ_REMESH)
	    b->mesh = (Mesh *) GeomCCreate (NULL, MeshMethods(), CR_END);
	else if(ob->mesh != NULL)
	    b->mesh = (Mesh *) GeomCopy((Geom *)ob->mesh);

	return(b);
}

void
BezierDelete( Bezier *bezier )
{
	if(bezier) {
	    if(bezier->CtrlPnts != NULL)
		OOGLFree(bezier->CtrlPnts);
	    if(bezier->STCords != NULL)
		OOGLFree(bezier->STCords);
	    if(bezier->mesh != NULL)
		GeomDelete((Geom *)bezier->mesh);
	}
}

Bezier *
BezierCreate ( Bezier *exist, GeomClass *classp, va_list a_list )
{
    register Bezier *bezier;
    int attr, copy = 1;
    ColorA *color = NULL;
    int i;

    if (exist == NULL) {
	bezier = OOGLNewE(Bezier, "BezierCreate Bezier");
	bzero(bezier,sizeof(Bezier));
	bezier->flag = BEZ_P;
	bezier->mesh = NULL;
	bezier->CtrlPnts = NULL;
	bezier->STCords = NULL;
	bezier->nu = bezier->nv = 0;	/* no mesh yet */
    } else {
	/* Check that exist is a Bezier. */
	bezier = exist;
    }

    while (attr = va_arg (a_list, int)) switch (attr) {
	case CR_FLAG:
	    bezier->flag = va_arg (a_list, int);
	    break;
	case CR_DEGU:
	    bezier->degree_u = va_arg (a_list, int);
	    break;
	case CR_DEGV:
	    bezier->degree_v = va_arg (a_list, int);
	    break;
	case CR_DIM:
	    bezier->dimn = va_arg (a_list, int);
	    break;
	case CR_NU:
	    bezier->nu = va_arg (a_list, int);
	    break;
	case CR_NV:
	    bezier->nv = va_arg (a_list, int);
	    break;
	case CR_POINT:
	    bezier->CtrlPnts = va_arg (a_list, float *);
	    break;
	case CR_MESH:
	    bezier->mesh = va_arg (a_list, Mesh *);
	    break;
	case CR_COLOR:
	    color = va_arg (a_list, ColorA *);
	    if (color != NULL) for (i=0; i<4; i++) bezier->c[i] = color[i];
	    break;
	default:
	    if (GeomDecorate (bezier, &copy, attr, &a_list)) {
		OOGLError (0, "BezierCreate: undefined option: %d", attr);
		OOGLFree (bezier);
		return NULL;
	    }
    }

    if (bezier->dimn > MAX_BEZ_DIMN) {
	OOGLError (0, "BezierCreate: dimension (%d) too high.", bezier->dimn);
	OOGLFree (bezier);
	return NULL;
    }

    if (exist != NULL) return exist;

    GGeomInit (bezier, classp, BEZIERMAGIC, NULL);
    return bezier;
}
