/* Copyright (c) 1992 The Geometry Center; University of Minnesota
   1300 South Second Street;  Minneapolis, MN  55454, USA;
   
This file is part of geomview/OOGL. geomview/OOGL is free software;
you can redistribute it and/or modify it only under the terms given in
the file COPYING, which you should have received along with this file.
This and other related software may be obtained via anonymous ftp from
geom.umn.edu; email: software@geom.umn.edu. */

/* Authors: Charlie Gunn, Pat Hanrahan, Stuart Levy, Tamara Munzner, Mark Phillips */

#include "transform.h"
#include <math.h>
#include <stdio.h>

/*
 * h31Matrix = DiagonalMatrix[ {1,1,1,-1} ];
 * 
 * h31Dot[x_, y_] := x . (y * {1,1,1,-1});
 * 
 * reflection[v_] :=
 *   IdentityMatrix[n+1] - 2 Outer[ Times, v, v ] . h31Matrix / h31Dot[v,v];
 * 
 * midpoint[x_, y_] :=
 *   Sqrt[ h31Dot[y,y] h31Dot[y,x] ] x + Sqrt[ h31Dot[x,x] h31Dot[x,y] ] y;
 *
 * translation[a,b] :=  (* translation from a to b *)
 *   reflection[ midpoint[a,b] ] . reflection[a]
 */

static Transform I31Matrix =
  {1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, -1};
static halfpoint( HPoint3 *a, HPoint3 *half );

/*-----------------------------------------------------------------------
 * Function:	Tm3HypTranslate
 * Description:	compute a hyperbolic translation transform
 * Args:	T: the transform (output)
 *		base: base point of translation
 *		x,y,z: direction of translation
 *		dist: distance of translation
 * Author:	mbp
 * Date:	Tue Mar  3 23:02:15 1992
 * Notes:	base can be any point on the translation axis.
 *		distance of translation is dist rather than the length
 *		  of (x,y,z) so we don't have to compute hyperbolic length
 *		  of (x,y,z), and also because it seems that applications
 *		  will want to explicity specify the distance rather than
 *		  having to normalize (x,y,z) in advance  to have  certain
 *		  hyperbolic length.
 */
void
Tm3HypTranslate(T,base,x,y,z,dist)
     Transform T;
     HPoint3 *base;
     float x, y, z, dist;
{
  HPoint3 l0, v0, dir;
  float ax, ay, az, avvv, f;
  Transform P,Pinv,T0, tmp;

  dir.x = x;
  dir.y = y;
  dir.z = z;
  Tm3HypNearestToOrigin( base, &dir, &l0 );
  Tm3HypTranslateOrigin(&l0, P, Pinv);
  f = tanh( dist ) / sqrt( x*x  + y*y + z*z );
  v0.x = x * f;
  v0.y = y * f;
  v0.z = z * f;
  v0.w = 1;
  Tm3HypTranslateOrigin(&v0, T0, NULL);

  Tm3Concat(Pinv, T0, tmp);
  Tm3Concat(tmp, P, T);  
}

Tm3HypNearestToOrigin( base, dir, ans )
     HPoint3 *base, *ans, *dir;
{
  float ax, ay, az, avvv, vv;
  ax = base->x / base->w;
  ay = base->y / base->w;
  az = base->z / base->w;
  avvv = ( ax * dir->x + ay * dir->y + az * dir->z ) /
    ( dir->x*dir->x  + dir->y*dir->y + dir->z*dir->z );
  ans->x = ax - avvv * dir->x;
  ans->y = ay - avvv * dir->y;
  ans->z = az - avvv * dir->z;
  ans->w = 1;
}


/* hyp translation from 0 to a: */
void
Tm3HypTranslateOrigin( a, T, Tinv )
     HPoint3 *a;
     Transform T, Tinv;
{
  HPoint3 half;
  Transform r;

  halfpoint( a, &half );
  Tm3HypReflect( r, &half );
  Tm3Concat( I31Matrix, r, T );
  if (Tinv != NULL)
    Tm3Concat( r, I31Matrix, Tinv );
}

/* compute the point half-way between 0 and a: */
static halfpoint( HPoint3 *a, HPoint3 *half )
{
#define SQR(x) ((x)*(x))
#define SIGN(x) (((x)>=0)?1:-1)
  half->x = a->x;
  half->y = a->y;
  half->z = a->z;
  half->w =
    a->w + SIGN(a->w) * sqrt(SQR(a->w) - SQR(a->x) - SQR(a->y) - SQR(a->z));
#undef SQR
#undef SIGN
}

/*-----------------------------------------------------------------------
 * Function:	Tm3HypReflect
 * Description:	hyperbolic reflection in a point or a plane
 * Args:	r: the answer
 *		a: homog coords of a point or a plane
 * Author:	mbp
 * Date:	Wed Mar 11 14:51:41 1992
 * Notes:	
 */
void
Tm3HypReflect( r, a )
     Transform r;
     HPoint3 *a;
{
  register float ax,ay,az,aw, dot;
  ax = a->x;  ay = a->y;  az = a->z;  aw = a->w;
  dot = ax*ax + ay*ay + az*az - aw*aw;

  r[0][0] = 1 - ( 2 * ax * ax ) / dot;
  r[1][0] =   - ( 2 * ax * ay ) / dot;
  r[2][0] =   - ( 2 * ax * az ) / dot;
  r[3][0] =     ( 2 * aw * ax ) / dot;
  
  r[0][1] =   - ( 2 * ax * ay ) / dot;
  r[1][1] = 1 - ( 2 * ay * ay ) / dot;
  r[2][1] =   - ( 2 * ay * az ) / dot;
  r[3][1] =     ( 2 * aw * ay ) / dot;
  
  r[0][2] =   - ( 2 * ax * az ) / dot;
  r[1][2] =   - ( 2 * ay * az ) / dot;
  r[2][2] = 1 - ( 2 * az * az ) / dot;
  r[3][2] =     ( 2 * aw * az ) / dot;
  
  r[0][3] =   - ( 2 * aw * ax ) / dot;
  r[1][3] =   - ( 2 * aw * ay ) / dot;
  r[2][3] =   - ( 2 * aw * az ) / dot;
  r[3][3] = 1 + ( 2 * aw * aw ) / dot;
}

/*-----------------------------------------------------------------------
 * Function:	Tm3HypRotate
 * Description:	compute a hyperbolic rotation matrix
 * Args:	T: the answer matrix
 *		base: base point on axis of rotation
 *		axis: axis of rotation (only (x,y,z) components are used)
 *		angle: angle of rotation, in radians, as seen from
 *		  tip of axis looking towards base point
 * Author:	mbp
 * Date:	Wed Mar  4 17:53:10 1992
 * Notes:	
 */
void
Tm3HypRotate( T, base, axis, angle )
     Transform T;
     HPoint3 *base, *axis;
     float angle;
{
  HPoint3 l0;
  Transform P,Pinv,R0, tmp;

  Tm3HypNearestToOrigin( base, axis, &l0 );
  Tm3HypTranslateOrigin(&l0, P, Pinv);
  Tm3Rotate( R0, angle, axis );
  Tm3Concat(Pinv, R0, tmp);
  Tm3Concat(tmp, P, T);  
}

/*
 * Pre-multiply a matrix by a hyperbolic translation
 */
void
Ctm3HypTranslate(T,base,x,y,z,dist)
     Transform3 T;
     HPoint3 *base;
     float x, y, z, dist;
{
    register Transform3 HypT;

    Tm3HypTranslate(HypT,base,x,y,z,dist);
    Tm3Concat(HypT,T,T);
}

void
Ctm3HypRotate( T, base, axis, angle )
     Transform3 T;
     HPoint3 *base, *axis;
     float angle;
{
    register Transform3 HypT;

    Tm3HypRotate( HypT, base, axis, angle );
    Tm3Concat(HypT,T,T);
}
