/* Copyright (c) 1992 The Geometry Center; University of Minnesota
   1300 South Second Street;  Minneapolis, MN  55454, USA;
   
This file is part of geomview/OOGL. geomview/OOGL is free software;
you can redistribute it and/or modify it only under the terms given in
the file COPYING, which you should have received along with this file.
This and other related software may be obtained via anonymous ftp from
geom.umn.edu; email: software@geom.umn.edu. */
static char *copyright = "Copyright (C) 1992 The Geometry Center";

/* Authors: Charlie Gunn, Pat Hanrahan, Stuart Levy, Tamara Munzner, Mark Phillips */

/*
**	point3.c - 3D vector arithmetic module.
**
**	pat hanrahan
**
*/
# include <math.h>
# include "tolerance.h"
# include "point3.h"
# include "transform3.h"

Point3 Pt3Origin = { 0., 0., 0. };

/* print vector */
void
Pt3Print( v )
    Point3 *v;
{
    /* Only works if Pt3Coord is a float */
    printf("[%g %g %g]\n", v->x, v->y, v->z);
}

void
Pt3From(v,x,y,z)
    Point3 *v;
    Pt3Coord x, y, z;
{
    v->x = x;
    v->y = y;
    v->z = z;
}

/* v2 = v1 */
void
Pt3Copy( v1, v2 )
    Point3 *v1, *v2;
{
    v2->x = v1->x;
    v2->y = v1->y;
    v2->z = v1->z;
}

/* v3 = v1 + v2 */
void
Pt3Add(v1,v2,v3)
    Point3 *v1,*v2,*v3;
{
    v3->x = v1->x + v2->x;
    v3->y = v1->y + v2->y;
    v3->z = v1->z + v2->z;
}

/* v3 = v1 - v2 */
void
Pt3Sub(v1,v2,v3)
    Point3 *v1,*v2,*v3;
{
    v3->x = v1->x - v2->x;
    v3->y = v1->y - v2->y;
    v3->z = v1->z - v2->z;
}

/* v2 = s * v1  */
void
Pt3Mul(s,v1,v2)
    Pt3Coord s;
    Point3 *v1,*v2;
{
    v2->x = s*v1->x;
    v2->y = s*v1->y;
    v2->z = s*v1->z;
}


/* v1 . v2 */
Pt3Coord
Pt3Dot( v1, v2 )
    Point3 *v1, *v2;
{
    return v1->x*v2->x+v1->y*v2->y+v1->z*v2->z;
}

/* v3 = v1 x v2 */
void
Pt3Cross( v1,v2,v3)
    Point3 *v1,*v2,*v3;
{
    v3->x = v1->y*v2->z-v1->z*v2->y; 
    v3->y = v1->z*v2->x-v1->x*v2->z; 
    v3->z = v1->x*v2->y-v1->y*v2->x;
}

/* v1 . (v2 x v3) */
Pt3Coord 
Pt3TripleDot( v1, v2, v3 )
    Point3 *v1, *v2, *v3;
{
    return v1->x*(v2->y*v3->z-v2->z*v3->y) 
         + v1->y*(v2->z*v3->x-v2->x*v3->z) 
         + v1->z*(v2->x*v3->y-v2->y*v3->x);
}

/* v4 = (v1 x v2) x v3 */
void
Pt3TripleCross( v1, v2, v3, v4 )
    Point3 *v1, *v2, *v3, *v4;
{
    Point3 v;
 
    Pt3Cross( v1, v2, &v );
    Pt3Cross( &v, v3, v4 );
}

Pt3Coord
Pt3Length( v )
    Point3 *v;
{
    return sqrt( v->x*v->x + v->y*v->y + v->z*v->z );
}

Pt3Coord 
Pt3Distance( v1, v2 )
    Point3 *v1, *v2;
{
    Point3 v12;
	
    Pt3Sub(v1,v2,&v12);

    return Pt3Length(&v12);
}

Pt3Coord
Pt3Angle( v1, v2 )
    Point3 *v1, *v2;
{
}

void
Pt3Unit(v)
    Point3 *v;
{
    Pt3Coord len;

    len = Pt3Length(v);
    if( len != 0. )
	Pt3Mul( 1./len, v, v );
}


/*
 * lerp - linear interpolation
 *
 * v3 = (1-t)*v1 + t*v2
 */
void
Pt3Lerp( t, v1, v2, v3 )
    Pt3Coord t;
    Point3 *v1, *v2, *v3;
{
    v3->x = (1.-t)*v1->x + t*v2->x;
    v3->y = (1.-t)*v1->y + t*v2->y;
    v3->z = (1.-t)*v1->z + t*v2->z;
}

void
Pt3Comb( t1, v1, t2, v2, v3 )
    Pt3Coord t1, t2;
    Point3 *v1, *v2, *v3;
{
    v3->x = t1*v1->x + t2*v2->x;
    v3->y = t1*v1->y + t2*v2->y;
    v3->z = t1*v1->z + t2*v2->z;
}


/* vectors equal */
int 
Pt3Equal(v1,v2)
    Point3 *v1, v2;
{
    Pt3Coord v;
	
    v = Pt3Distance(v1, v2);

    return fz(v);
}

/* parallel vectors */
int 
Pt3Parallel(v1,v2)
    Point3 *v1, *v2;
{
    Point3 v3;
    Pt3Coord v;
	
    Pt3Cross(v1,v2,&v3);
    v=Pt3Length(&v3);

    return fz(v);
}

/* perpendicular vectors */
int 
Pt3Perpendicular(v1,v2)
    Point3 v1,v2;
{
    Pt3Coord v;

    v=Pt3Dot(v1,v2);

    return fz(v);
}

void
Pt3Transform( T, p1, p2 )
    Transform3 T;
    Point3 *p1, *p2;
{
    register Pt3Coord x, y, z, w;

    x = p1->x;
    y = p1->y;
    z = p1->z;
    w = (T[X][W]*x + T[Y][W]*y + T[Z][W]*z + T[W][W]);
    if(w != 1.0) {
	w = 1.0 / w;
	p2->x = w * (x*T[X][X] + y*T[Y][X] + z*T[Z][X] + T[W][X]);
	p2->y = w * (x*T[X][Y] + y*T[Y][Y] + z*T[Z][Y] + T[W][Y]);
	p2->z = w * (x*T[X][Z] + y*T[Y][Z] + z*T[Z][Z] + T[W][Z]);
    } else {
	p2->x = x*T[X][X] + y*T[Y][X] + z*T[Z][X] + T[W][X];
	p2->y = x*T[X][Y] + y*T[Y][Y] + z*T[Z][Y] + T[W][Y];
	p2->z = x*T[X][Z] + y*T[Y][Z] + z*T[Z][Z] + T[W][Z];
    }
}

void
Pt3TransformN( T, p1, p2, n )
    Transform3 T;
    Point3 *p1, *p2;
    int n;
{
    while( n-- ) 
	Pt3Transform( T, p1++, p2++ );
}

#ifdef NEW
void
Pt3Project( v1, v2, v3 )
    Point3 v1, v2, v3;
{
    Pt3Scalar( Pt3Dot( v1, v2 ), v2, v3 );
}

void
Pt3Reflect( v1, v2, v3 )
    Point3 v1, v2, v3;
{
}

Pt3Rotate( v1, v2, angle )
    Point3 v1, v2;
    Pt3Coord angle;
{
    Pt3Coord ca, sa;
    Pt3Coord tv1, ttv1, tv2, ttv2;

    sa = sin( angle );
    ca = cos( angle );

    Pt3Copy( v1, tv1 );
    Pt3Copy( v2, tv2 );

    Pt3Scalar( ca, tv1, ttv1 );
    Pt3Scalar( sa, tv2, ttv2 );
    Pt3Add( ttv1, ttv2 );
    Pt3Scalar( ca, tv1, ttv1 );
    Pt3Scalar( sa, tv2, ttv2 );
    Pt3Add( ttv1, ttv2, v1 );
    Pt3Scalar( -sa, tv1, ttv1 );
    Pt3Scalar(  ca, tv2, ttv2 );
    Pt3Add( ttv1, ttv2, v2 );
}
# define min(a,b) ((a)<(b)?(a):(b))
# define max(a,b) ((a)>(b)?(a):(b))

static int
son( p, p1, p2 )
    Pt3Coord p;
    Pt3Coord p1;
    Pt3Coord p2;
{
    return (p >= min( p1, p2 )) && (p <= max( p1, p2 ));
}

static int
von( P, P1, P2 )
    Point3 P;
    Point3 P1;
    Point3 P2;
{
    return 
	son( P->x, P1->x, P2->x ) && 
	son( P->y, P1->y, P2->y ) &&
	son( P->z, P1->z, P2->z );
}

int
Pt3Skewp( P1, P2, Q1, Q2 )
    Point3 P1, P2;
    Point3 Q1, Q2;
{
    Pt3Coord R;

    return Pt3Skew( P1, P2, Q1, Q2, R );
}

int
Pt3Skew( P1, P2, Q1, Q2, R )
    Pt3Coord P1, P2;
    Pt3Coord Q1, Q2;
    Pt3Coord R;
{
    Point3 PQ, P, Q;
    Point3 DP, DQ;
    Pt3Coord ap, aq, a, bp, bq, b, w;

    Pt3Sub( P1, Q1, PQ );
    Pt3Sub( P2, P1, DP );
    Pt3Sub( Q2, Q1, DQ );

    ap =  Pt3Dot( DP, DP );
    aq = -Pt3Dot( DP, DQ );
    a  =  Pt3Dot( DP, PQ );

    bp =  aq;
    bq =  Pt3Dot( DQ, DQ );
    b  = -Pt3Dot( DQ, PQ );

    w = ap*bq-aq*bp;
    if( w != 0. ) {
	Pt3Scalar( (aq*b-a*bq)/w, DP, DP );
	Pt3Scalar( (a*bp-ap*b)/w, DQ, DQ );
	Pt3Add( P1, DP, P );
	Pt3Add( Q1, DQ, Q );
	Pt3Add( P, Q, R );
	Pt3Scalar( 0.5, R, R );
	return Pt3Equal( P, Q ) && von( R, P1, P2 ) && von( R, Q1, Q2 );
  }
  return 
	von( P1, Q1, Q2 ) || von( P2, Q1, Q2 ) ||
	von( Q1, P1, P2 ) || von( Q2, P1, P2 );
}
#endif

# ifdef MAIN
main()
{
    static char Point[] = "%F %F %F";
    int i;
    Pt3Coord P1[3], P2[3], Q1[3], Q2[3], R[3];

    printf("P1 = "); scanf( Point, &P1->x, &P1->y, &P1->z );
    printf("P2 = "); scanf( Point, &P2->x, &P2->y, &P2->z );
    printf("Q1 = "); scanf( Point, &Q1->x, &Q1->y, &Q1->z );
    printf("Q2 = "); scanf( Point, &Q2->x, &Q2->y, &Q2->z );
    i = Skew( P1, P2, Q1, Q2, R );
    printf("Skew(%d) returns %g %g %g \n", i, R->x, R->y, R->z );
}
# endif
