/* Copyright (c) 1992 The Geometry Center; University of Minnesota
   1300 South Second Street;  Minneapolis, MN  55454, USA;
   
This file is part of geomview/OOGL. geomview/OOGL is free software;
you can redistribute it and/or modify it only under the terms given in
the file COPYING, which you should have received along with this file.
This and other related software may be obtained via anonymous ftp from
geom.umn.edu; email: software@geom.umn.edu. */

/* Authors: Charlie Gunn, Pat Hanrahan, Stuart Levy, Tamara Munzner, Mark Phillips */

#
/*
**	hpoint3.c - procedural interface to 3D point geometry
**
**	pat hanrahan
*/

#include <math.h>
#include "hg4.h"
#include "point3.h"
#include "hpoint3.h"
#include "hplane3.h"
#include "hline3.h"
#include "transform3.h"
#include "tolerance.h"

HPoint3 *
HPt3Create()
{
    return (HPoint3 *) Hg4Create();
}

void
HPt3Delete( pt )
    HPoint3 *pt;
{
    Hg4Delete( (Hg4Tensor1 *) pt );
}

void
HPt3Print( pt )
    HPoint3 *pt;
{
    Hg4Print( (Hg4Tensor1 *) pt );
}

void
HPt3Copy( pt1, pt2 )
    HPoint3 *pt1, *pt2;
{
    Hg4Copy( (Hg4Tensor1 *) pt1, (Hg4Tensor1 *) pt2 );
}

void
HPt3Add( pt1, pt2, pt3 )
    HPoint3 *pt1, *pt2, *pt3;
{
    Hg4Add( (Hg4Tensor1 *) pt1, (Hg4Tensor1 *) pt2 , (Hg4Tensor1 *) pt3);
}


void
HPt3From( pt, x, y, z, w )
    HPoint3 *pt;
    HPt3Coord x, y, z, w;
{
    Hg4From( (Hg4Tensor1 *)pt, x, y, z, w );
}

int
HPt3From3HPl3s( pt, pl1, pl2, pl3 )
    HPoint3 *pt;
    HPlane3 *pl1, *pl2, *pl3;
{
    return Hg4Intersect3( 
	(Hg4Tensor1 *)pl1, (Hg4Tensor1 *)pl2, (Hg4Tensor1 *)pl3,
	(Hg4Tensor1 *)pt, 1 );
}

int
HPt3From2HLn3s( pt, ln1, ln2 )
    HPoint3 *pt;
    HLine3 *ln1, *ln2;
{
    HPlane3 pl;

    return HLn3IntersectHLn3( ln1, ln2, &pl, pt );
}

int
HPt3IntersectHPt3( pt1, pt2, ln )
    HPoint3 *pt1, *pt2;
    HLine3 *ln;
{
    return HLn3From2HPt3s( ln, pt1, pt2 );
}

void
HPt3Pencil( t1, pt1, t2, pt2, pt3 )
    HPt3Coord t1, t2;
    HPoint3 *pt1, *pt2, *pt3;
{
    Hg4Pencil( t1, (Hg4Tensor1 *)pt1, t2, (Hg4Tensor2 *)pt2, (Hg4Tensor1 *)pt3);
}

float
HPt3DotHPl3( pt, pl )
    HPoint3 *pt;
    HPlane3 *pl;
{
    return Hg4ContractPiQi( (Hg4Tensor1 *)pt, (Hg4Tensor1 *)pl );
}

void
HPt3Normalize( pt1, pt2 )
    HPoint3 *pt1, *pt2;
{
    Hg4Normalize( (Hg4Tensor1 *)pt1, (Hg4Tensor1 *)pt2 );
}

int
HPt3Compare( pt1, pt2 )
    HPoint3 *pt1, *pt2;
{
    return Hg4Compare( (Hg4Tensor1 *)pt1, (Hg4Tensor1 *)pt2 );
}

int
HPt3Undefined( pt )
    HPoint3 *pt;
{
    return Hg4Undefined( (Hg4Tensor1 *)pt );
}

int
HPt3Infinity( pt )
    HPoint3 *pt;
{
    return Hg4Infinity( (Hg4Tensor1 *)pt, 0 );
}

int
HPt3CoincidentHPl3( pt, pl )
    HPoint3 *pt;
    HPlane3 *pl;
{
    return fzero(HPt3DotHPl3(pt,pl));
}

int
HPt3CoincidentHLn3( pt, ln )
    HPoint3 *pt;
    HLine3 *ln;
{
    HPlane3 pl;

    return HLn3IntersectHPt3( ln, pt, &pl );
}

int
HPt3CoincidentHPt3( pt1, pt2 )
    HPoint3 *pt1, *pt2;
{
    return Hg4Coincident( (Hg4Tensor1 *)pt1, (Hg4Tensor1 *)pt2 );
}

void
HPt3Transform( T, pt1, pt2 )
    HPoint3 *pt1, *pt2;
{
    Hg4Transform( T, (Hg4Tensor1 *)pt1, (Hg4Tensor1 *)pt2 );
}

void
HPt3TransformN( T, pt1, pt2, n )
    HPoint3 *pt1, *pt2;
    int n;
{
    while( n-- )
	Hg4Transform( T, (Hg4Tensor1 *)pt1++, (Hg4Tensor1 *)pt2++ );
}

/*
 * Transform and project an HPoint3 onto a plain Point3.
 * Transforms pin . T -> pout,
 * then projects pout.{x,y,z} /= pout.w.
 * Returns pout.w.
 */
HPt3Coord
HPt3TransPt3( Transform3 T, HPoint3 *pin, Point3 *pout )
{
    Hg4Tensor1 tp;
    register HPt3Coord w;

    Hg4Transform( T, (Hg4Tensor1 *)pin, tp );
    pout->x = tp[X] / tp[W];
    pout->y = tp[Y] / tp[W];
    pout->z = tp[Z] / tp[W];
    return tp[W];
}
    

HPt3Coord
HPt3Distance( pt1, pt2 )
    HPoint3 *pt1, *pt2;
{
    float dx, dy, dz;
    float w1w2;

    w1w2 = pt1->w * pt2->w;
    if( w1w2 == 0. )
	return 0.;

    dx = pt2->w * pt1->x - pt2->x * pt1->w;
    dy = pt2->w * pt1->y - pt2->y * pt1->w;
    dz = pt2->w * pt1->z - pt2->z * pt1->w;

    return (sqrt( dx*dx + dy*dy + dz*dz )) / w1w2;
}

/* Fishy procedure */
void
HPt3Dual( pt, pl )
    HPoint3 *pt;
    HPlane3 *pl;
{
    pl->a = pt->x;
    pl->b = pt->y;
    pl->c = pt->z;
    pl->d = pt->w;
}

void
HPt3LinSum (scale1, in1, scale2, in2, out)
    HPt3Coord scale1, scale2;
    HPoint3 *in1, *in2, *out;
{
    if ((in1->w == 0) || (in2->w == 0)) {
	out->w = 0;
	out->x = scale1 * in1->x + scale2 * in2->x;
	out->y = scale1 * in1->y + scale2 * in2->y;
	out->z = scale1 * in1->z + scale2 * in2->z;
	return;
    }
    out->w = 1;
    out->x = scale1 * (in1->x/in1->w) + scale2 * (in2->x/in2->w);
    out->y = scale1 * (in1->y/in1->w) + scale2 * (in2->y/in2->w);
    out->z = scale1 * (in1->z/in1->w) + scale2 * (in2->z/in2->w);
}

void
HPt3SizeOne (pt, out)
HPoint3 *pt;
HPoint3 *out;
{
    HPt3Coord size;

    size = sqrt (pt->x * pt->x + pt->y * pt->y + pt->z * pt->z);
    if (size == 0) return;
    out->x = pt->x / size;
    out->y = pt->y / size;
    out->z = pt->z / size;
    out->w = 1.;
}

/* 
 *
 * Pt3ToPt4: convert 3-vectors to 4-vectors by padding with 1.0 's.
 *
 * Charlie Gunn
 * Nov 26, 1991: originally written
 */

Pt3ToPt4(v3, v4, n)
Point3 *v3;
HPoint3 *v4;
int n;
{
	register int i;
        for (i = 0; i<n; ++i)	{
	  v4[i].x = v3[i].x;
	  v4[i].y = v3[i].y;
	  v4[i].z = v3[i].z;
	  v4[i].w = 1.0;
	  }
}

HPt3ToPt3(hp, p)
     HPoint3 *hp;
     Point3 *p;
{
  p->x = hp->x / hp->w;
  p->y = hp->y / hp->w;
  p->z = hp->z / hp->w;
}
