/* Copyright (c) 1992 The Geometry Center; University of Minnesota
   1300 South Second Street;  Minneapolis, MN  55454, USA;
   
This file is part of geomview/OOGL. geomview/OOGL is free software;
you can redistribute it and/or modify it only under the terms given in
the file COPYING, which you should have received along with this file.
This and other related software may be obtained via anonymous ftp from
geom.umn.edu; email: software@geom.umn.edu. */

/* Authors: Stuart Levy, Tamara Munzner, Mark Phillips */

#include "mg.h"
#include "../common/drawer.h"
#include "../common/comm.h"
#include "../common/ui.h"
#include "../common/event.h"
#include "../common/motion.h"
#include "../common/worldio.h"
#include "../common/config.h"
#include "../common/version.h"
#include "streampool.h"
#include "stopsign.h"
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/dir.h>
#include <sys/signal.h>


#define FILE_GENERAL	0 /* These are indices into FileTitles[] array */
#define FILE_TRANSFORM	1 /* and used by usefilename() when disposing of */
#define FILE_SAVE	2 /* file names supplied in the popup/browser */

#include "panel.c"

#define	BLACKINDEX	0
#define	WHITEINDEX	7

#define FACEINDEX	250
#define EDGEINDEX 	251
#define NORMALINDEX 	252
#define BBOXINDEX 	253
#define BACKINDEX 	254
#define LIGHTINDEX	255


#define	P_MAIN		1
#define	P_APPEARANCE	2
#define	P_LIGHTS	3
#define	P_OBSCURE	4
#define	P_COMMANDS	5
#define	P_CREDITS	6
#define	P_INPUT		7
#define	P_FILEBROWSER	8
#define	P_COLORPICKER	9
#define	P_MAX	10

static struct panel {
    FL_FORM **objp;
    char *name;
    char shown;
    char browse;
    char placement;
    short winid;
    int x0, y0;
} panels[] = {
    { NULL,		0,		0,0, 0,			0, 0,0 },
    { &MotionPanel,	"geomview", 	1,1, FL_PLACE_POSITION,	0, 0,0 },
    { &AppearancePanel,	"Appearance", 	0,1, FL_PLACE_SIZE,	0, 0,0 },
    { &LightingPanel,	"Lighting", 	0,1, FL_PLACE_SIZE,	0, 0,0 },
    { &ObscurePanel,	"Obscure", 	0,1, FL_PLACE_SIZE,	0, 0,0 },
    { &CommandPanel,	"Commands", 	0,1, FL_PLACE_SIZE,	0, 0,0 },
    { &CreditsPanel,	"Credits", 	0,1, FL_PLACE_SIZE,	0, 0,0 },
    { &InputPanel,	"Input",	0,0, FL_PLACE_MOUSE,	0, 0,0 },
    { &BrowserPanel,	"Files",	0,0, FL_PLACE_POSITION,	0, 0,0 },
    { &ColorPanel,	"Color",	0,0, FL_PLACE_MOUSE,	0, 0,0 },
};


UIState uistate;

char   *lights[MAXLIGHTS+1];

static int colindex = 249;
static Color oldcol = {0, 0, 0};
static Color curcol = {0, 0, 0};
static int curval = 0;
static FL_OBJECT *curobj = NULL;
static int *menuindex = NULL;
static int menucount = 0;
static int norefresh = 0;
static char *browsing_dir = NULL;

#define QUIETLY(cmds)	\
  do {			\
    int old;		\
    old=norefresh;	\
    norefresh=1;	\
    cmds;		\
    norefresh=old; } while (0)

#define	COUNT(array)  (sizeof(array) / sizeof(array[0]))

static int id2menuindex(int id);
static void BuildBrowserMenu(FL_OBJECT *obj);
static void BuildBrowser(FL_OBJECT *browser, int count, char *names[]);
void may_colorbutton( FL_OBJECT *btn, int index, Color *c );
void ui_keyboard(int ch);
void ui_showpanel(int index, int show);
int  ui_panelshown(int index);
void ui_panelwindow(int index, WnWindow *window);
int  ui_name2panel(char *name);
static void ui_color(int index, Color *color);
static int gv_fl_show_form(FL_FORM *, int position, int, char *name);
static void glui_float(FL_OBJECT *, int key, int id, float min, float max);
static void glui_int(FL_OBJECT *, int key, int id, int min, int max);
static int getfloat(FL_OBJECT *, float *);
static int getint(FL_OBJECT *, int *);
static void may_set_slider( FL_OBJECT *obj, float value );
static void may_set_button( FL_OBJECT *btn, int state );
static void may_set_iinput( FL_OBJECT *input, int val);
static void may_set_finput( FL_OBJECT *input, float val);
static void may_set_sinput( FL_OBJECT *input, char *val);
static void may_set_browser( FL_OBJECT *browser, int line, int size);
static Handle *emotion_target();
static void adjust_browser( FL_OBJECT *brow, int is, int visible );
static void usefilename(char *name);
static void show_browsing_dir();
void ShowColor(char *name, Color *old, int index, int val, FL_OBJECT *obj);
static Handle *has_extern_motion(int id);

#define streq(s1,s2)  (!strcmp(s1,s2))

/*
 * This wrapper for fl_show_form() resets SIGCHLD handling
 * to ensure we don't sleep forever in the 4.0 fmfont libraries.
 */
static int
gv_fl_show_form(FL_FORM *form, int posn, int bool, char *name)
{
  void *oldchld = signal(SIGCHLD, SIG_DFL);
  int win = fl_show_form(form, posn, bool, name);
  signal(SIGCHLD, oldchld);
  return win;
}

ui_init()
{
  cui_init();
  clight_init();
  panel_init();

  event_mode( OBJROTATE , 0);
  uistate.cursor_on = 1;
}

static char *credits[] = {
  "By Stuart Levy, Tamara Munzner, Mark Phillips",
  "Copyright (c) 1992",
  "The Geometry Center",
  "anonymous ftp: geom.umn.edu",
  "email: software@geom.umn.edu",
  "",
  "geomview/OOGL is free software which may be.",
  "obtained via anonymous ftp from the above site.",
  "You can redistribute and/or modify it only",
  "under the terms given in the file COPYING,",
  "which is included in the geomview distribution.",
  "",
  "The National Science and Technology Research",
  "Center for the Computation and Visualization of",
  "Geometric Structures",
  "",
  "University of Minnesota",
  "1300 South Second Street",
  "Minneapolis, MN  55454  USA",
};

panel_init()
{
  register int i, x = 0;
  char buf[80];
  WnPosition wp;

  create_the_forms();

  if(WnGet(drawerstate.defview.win, WN_PREFPOS, &wp) > 0) {
    x = wp.xmax + 5;
    if (x > getgdesc(GD_XPMAX)) x = getgdesc(GD_XPMAX)-300;
  }
  fl_set_form_position(MotionPanel, x, -33);

  for(i = 2; i < P_MAX; i++) /* don't include main panel itself */
    if(panels[i].browse)
	fl_add_browser_line( MoreBrowser, panels[i].name );

  lights[0] = "ambient";
  for (i=1; i<=MAXLIGHTS; ++i) {
    sprintf(buf, "light %1d", i);
    lights[i] = strdup(buf);
  }

  BuildBrowser( ModeBrowser, uistate.mode_count, uistate.modenames );
  BuildBrowser( TitleBrowser, uistate.title_count, uistate.titles );
  BuildBrowser( ShadingBrowser, COUNT(shades), shades);
  BuildBrowser( ProjectionBrowser, COUNT(proj), proj);
  BuildBrowser( NormalizationBrowser, COUNT(norm), norm);
  BuildBrowser( SpaceBrowser, COUNT(spc), spc);
  BuildBrowser( SaveBrowser, COUNT(save), save);
  BuildBrowser( LightingBrowser, light_count(), lights);
  BuildBrowser( CreditsBrowser, COUNT(credits), credits);
  fl_set_browser_topline(CreditsBrowser, 1);

  set_light( light_count() > 0 ? 1 : 0 );
  fl_select_browser_line( TitleBrowser, 1 );

  fl_set_object_color(FaceColorButton,FACEINDEX,FACEINDEX);
  fl_set_object_color(EdgeColorButton,EDGEINDEX,EDGEINDEX);
  fl_set_object_color(NormalColorButton,NORMALINDEX,NORMALINDEX);
  fl_set_object_color(BBoxColorButton,BBOXINDEX,BBOXINDEX);
  fl_set_object_color(BackColorButton,BACKINDEX,BACKINDEX);
  fl_set_object_color(LightColorButton,LIGHTINDEX,LIGHTINDEX);
  fl_set_object_color(StopBitmap, 1, 7); 

  fl_set_bitmap(StopBitmap,stopsign_width,stopsign_height,stopsign_bits);  

  fl_set_input_return(NearClippingInput,FALSE);
  fl_set_input_return(FarClippingInput,FALSE);
  fl_set_input_return(NormalScaleInput,FALSE);
  fl_set_input_return(LinewidthInput,FALSE);
  fl_set_input_return(FOVInput,FALSE);
  fl_set_input_return(LinesCloserInput,FALSE);
  fl_set_input_return(NormalScaleInput,FALSE);
  fl_set_object_focus(ObscurePanel, ObscureHiddenInput1);
  fl_set_input_return(AcceptInput,FALSE);

  fl_set_input_return(AppearanceHiddenInput, TRUE);
  fl_set_input_return(ColorHiddenInput, TRUE);
  fl_set_input_return(MotionHiddenInput, TRUE);
  fl_set_input_return(LightingHiddenInput, TRUE);
  fl_set_input_return(CreditsHiddenInput, TRUE);
  fl_set_input_return(ObscureHiddenInput1, TRUE);
  fl_set_input_return(ObscureHiddenInput2, TRUE);
  fl_set_input_return(ObscureHiddenInput3, TRUE);
  fl_set_input_return(ObscureHiddenInput4, TRUE);
  fl_set_input_return(ObscureHiddenInput5, TRUE);
  fl_set_input_return(ObscureHiddenInput6, TRUE);

  fl_set_slider_bounds(IntensitySlider, 0.0, 1.0);
  fl_set_slider_bounds(ShininessSlider, 1.0, 128.0);
  fl_set_slider_bounds(KsSlider, 0.0, 1.0);
  fl_set_slider_bounds(KdSlider, 0.0, 1.0);
  fl_set_slider_bounds(KaSlider, 0.0, 1.0);

  sprintf(buf, "geomview %s", GEOMVIEW_VERSION);
  fl_set_object_label(MotionPanelTitle, buf);
  fl_set_object_label(CreditsPanelTitle, buf);

  fl_hide_object(DrawSphereButton);

  ui_select(WORLDGEOM);
}

/*
 * This is called after all other initialization (command-line, &c) is done.
 * It makes the main control window visible, if appropriate.
 */
void ui_final_init()
{
  int i;
  ui_select(CAMID(uistate.targetcam));
  ui_select(GEOMID(uistate.targetgeom));
  lights_changed_check();
  for(i=1; i<P_MAX; i++)
    if(panels[i].shown) ui_showpanel(i, 1);
  ui_objectchange();
}

int ui_panelshown(int index) { return (unsigned)index >= P_MAX ? 0 : panels[index].shown; }

void ui_panelwindow(int index, WnWindow *win)
{
  struct panel *p = &panels[index];
  WnPosition wp;
  if(p->objp == NULL) return;
  p->placement = FL_PLACE_SIZE;
  if(WnGet(win, WN_PREFPOS, &wp) > 0) {
    p->placement = FL_PLACE_POSITION;
    if(*p->objp)
	fl_set_form_position(*p->objp, p->x0=wp.xmin, p->y0=wp.ymin);
  }
}

void ui_showpanel(int index, int show)
{
  struct panel *p = &panels[index];
  if((unsigned) index >= P_MAX || p->objp == NULL || *p->objp == NULL) return;
  if(show < 0) show = !p->shown;
  if(show) {
    int owin = winget();
    winset(p->winid = gv_fl_show_form(*p->objp, p->placement, TRUE, p->name));
    winpop();
    if(owin>0) winset(owin); /* prevent NOSUCHWINDOW error if this is a popup */
  } else if(p->winid != 0) {
    fl_hide_form(*p->objp);
    p->winid = 0;
  }
  p->shown = show;
}

int ui_name2panel(char *name)
{
  register int i;
  if(strcasecmp(name, "main") == 0) return P_MAIN;
  for(i=P_MAX; --i > 0 && strncasecmp(name, panels[i].name, 5) != 0; )   ;
  return i;
}

int ui_winid2panel(int winid)
{
  register int i;
  for(i = P_MAX; --i > 0 && panels[i].winid != winid; )   ;
  return i;
}

/*-----------------------------------------------------------------------
 * Function:	panel_check
 * Description:	check for and process events in the UI panel
 * Returns:	nothing
 * Author:	mbp
 * Date:	Sat Jan 18 14:51:17 1992
 */
void panel_check()
{
  fl_check_forms();
}



/*-----------------------------------------------------------------------
 * Function:	ui_select_event_mode
 * Description:	
 * Args:	index: index into uistate.mode* arrays
 * Returns:	
 * Author:	munzner,slevy
 * Date:	Fri Jan 31 20:58:34 1992
 * Notes:	separates case where motion change explicitly picked from case
 *              where motion changes because of target choice. 
 */
void ui_select_event_mode(int index)
{
  if (index == uistate.extern_index) {
    /* turn on extern motion */
    ShowInput(NULL, FILE_TRANSFORM);
  }
  event_mode(uistate.modenames[index], 0);
}


/*-----------------------------------------------------------------------
 * Function:	ui_event_mode
 * Description:	Handle UI aspects of event mode change.
 * Args:	*mode: new motion mode string
 *		preserve: if 1 don't change target
 * Returns:	
 * Author:	munzner,slevy
 * Date:	Fri Jan 31 20:56:46 1992
 * Notes:	Check preserve and uistate.mode_current to stop infinite loop
 *		of ui_target, event_mode and ui_event_mode.
 */

void ui_event_mode(char *mode, int preserve)
{
  int index = ui_mode_index(mode);
  int newmodetype = uistate.modetype[index];
  Handle *h;
  int id;

  if (index != uistate.mode_current && index != uistate.extern_index) {
    
    if (uistate.mode_current == uistate.extern_index &&
	uistate.modetype[index] == TYPEOF(uistate.targetid)) {
      /* turn off extern motion */
      drawer_xform_set(uistate.targetid, NULL, TMNULL);
      emotion_target();
    }

    uistate.prevmode[newmodetype] = mode;

    if (newmodetype != TYPEOF(uistate.targetid) && !preserve) {
      /* may switch target based on new motion mode */
      switch (newmodetype) {
      case T_GEOM: id = GEOMID(uistate.targetgeom); goto twerp;
      case T_CAM: id = CAMID(uistate.targetcam); goto twerp;
      twerp:
	ui_target( has_extern_motion(id) ? NOID : id, IMMEDIATE); break;
      case T_NONE: emotion_target(); break;
      }
    }
  }
  uistate.mode_current = index;
  adjust_browser( ModeBrowser, index+1, 6);
}

/*-----------------------------------------------------------------------
 * Function:	ui_target
 * Description:	set the target of UI actions
 * Args:	id: object to become target
 *		immediate: 1 or 0
 * Returns:	
 * Author:	mbp, slevy, munzner
 * Date:	Fri Jan 31 20:54:06 1992
 * Notes:	immediate=1 means make this the current object
 *		immediately, switching target types if necessary.
 *		immediate=0 means retain current target type,
 *		making id the current object of its type.
 * 		The bbox of the picked object becomes white, 
 * 		bbox of unpicked object reset to bboxcolor.
 *		May switch motion modes based on target choice.
 */
void
ui_target(int id, int immediate)
{
  int oldtype = TYPEOF(uistate.targetid);
  int newtype = TYPEOF(id);
  static Color oldcol;
  Handle *h;
  DView *v;
  if (id == uistate.targetid) return;

  if(id == NOID) {
    /* the new motion mode is for a different type than the last one
     * and the target for that type is hooked up to external motion. 
     */
    set_ui_target_id( id );
    fl_deselect_browser_line(PickBrowser, fl_get_browser(PickBrowser));

  } else if (immediate || newtype==oldtype) {
    {
      /* change bbox colors to highlight new pick, unhighlight old */
      DGeom *dg;
      if ( newtype == T_GEOM &&
	  (dg = dgeom[uistate.targetgeom]) && dg->bboxap ) {
	static Color black = {0.0, 0.0, 0.0};
	Color c;
	c = uistate.pickedbboxcolor;
	drawer_color( GEOMID(uistate.targetgeom), DRAWER_BBOXCOLOR, &black );
	drawer_color( id, DRAWER_BBOXCOLOR, &c );
      } else if (newtype == T_CAM) { /* pop cam window in case it's obscured */
	v = (DView *) drawer_get_object(id);
	if (v) {
	  mggl_ctxselect(v->mgctx); /* winset happens in here */
	  winpop();
	}
      }
    }
      
    /* the main point of the procedure */
    set_ui_target_id( id );
    adjust_browser(PickBrowser, id2menuindex(id)+1, 6);
    h = emotion_target(); /* may rebuild extern motion line of mode browser */
    ui_select( id );


    /* may switch motion modes based on new target choice.
     * involves extern motion or different types for new/old targets.
     *
     * uistate.mode_current: index of old mode    
     * h: handle of new obj. NULL if new obj does not have external motion 
     */
    if ((uistate.mode_current == uistate.extern_index) && (!h)) {
      /* old: extern. new: intern */
      event_mode(uistate.prevmode[newtype], 1); 
    } else if((uistate.mode_current != uistate.extern_index) && (h)) {
      /* old: intern. new: extern */
      uistate.prevmode[oldtype] = uistate.modenames[uistate.mode_current];
      event_mode(uistate.modenames[uistate.extern_index], 1);
    } else if(newtype != oldtype && 
	      uistate.modetype[uistate.mode_current] != T_NONE) {
      uistate.prevmode[oldtype] = uistate.modenames[uistate.mode_current];
      event_mode(uistate.prevmode[newtype], 1);
    }

  } else {
    /* immediate == NOIMMEDIATE: for cases like deleting geometry
     * with the command language when current target is a camera.
     * update targettype but don't change browser.
     */
    switch (newtype) {
    case T_GEOM: uistate.targetgeom = INDEXOF(id); break;
    case T_CAM: uistate.targetcam = INDEXOF(id); break;
    }
  }
}

void ui_mousefocus(int index)
{
  uistate.mousefocus = index;
  if (uistate.targetid == FOCUSID)
    ui_select(CAMID(uistate.mousefocus));
}

void
ui_newwindow(DView *dv)
{	/* Ignored on SGI machines -- no need to know about new windows */
}

int
ui_snapshot(int id, char *fname)
{
    char cmd[512];
    DView *dv;
    WnWindow *wn = NULL;
    WnPosition wp;
    
    if(!ISCAM(id) || (dv = (DView *)drawer_get_object(id)) == NULL || dv->mgctx == NULL) {
	fprintf(stderr, "snapshot: id %d: no such camera\n", id);
	return 1;
    }
    mgctxselect(dv->mgctx);
    mgctxget(MG_WINDOW, &wn);
    if(WnGet(wn, WN_CURPOS, &wp) <= 0) {
	fprintf(stderr, "snapshot: can't find window for camera %s\n", dv->name[1]);
	return 1;
    }
    sprintf(cmd, "scrsave '%s' %d %d %d %d", fname,
	wp.xmin, wp.xmax, wp.ymin, wp.ymax);
    return system(cmd);
}

void ui_objectchange()
{
  if(PickBrowser == NULL) return;	/* Initialized? */
  BuildBrowserMenu(PickBrowser);
  adjust_browser( PickBrowser, id2menuindex(uistate.targetid)+1, 6 );
}

void ui_emotion_program(char *title, char *path)
{
  int current = fl_get_browser( TitleBrowser );
  ExternProgramStruct *eps = OOGLNewE(ExternProgramStruct, "ExternProgramStruct");
  eps->running = 0;
  eps->path = strdup(path);
  ui_install_title(strdup(title), (PFI)start_emotion_module, (void*)eps);
  BuildBrowser( TitleBrowser, uistate.title_count, uistate.titles );
  adjust_browser( TitleBrowser, current, 3);
}

ui_title_mode(char *title)
{
  int i = ui_title_index(title);
  if (i>=0 && uistate.titlefuncs[i]) {
    (*uistate.titlefuncs[i])(uistate.titledata[i]);
    may_set_browser( TitleBrowser, i+1, 2 );
  }
}
    

void ui_emotion_modename(char *modename)
{
  ui_add_mode(strdup(modename), emotion, T_NONE);
}  

void ui_replace_mode(char *name, PFI proc, int type, int index)
{
  int val = fl_get_browser(ModeBrowser);
  int id = (uistate.targetid == FOCUSID) ? CAMID(uistate.targetcam) : 
    uistate.targetid;

  ui_reinstall_mode(name, proc, type, index);
  fl_freeze_form(MotionPanel);
  BuildBrowser( ModeBrowser,uistate.mode_count,uistate.modenames);
  adjust_browser(ModeBrowser, val, 6);
  fl_unfreeze_form(MotionPanel);
}

void ui_add_mode(char *name, PFI proc, int type)
{
  int val = fl_get_browser(ModeBrowser);

  ui_install_mode(name, proc, type);
  BuildBrowser( ModeBrowser,uistate.mode_count,uistate.modenames);
  adjust_browser(ModeBrowser, val, 6);
}

void ui_remove_mode(char *name)
{
  int val = fl_get_browser(ModeBrowser);
  int first;
  
  first = (val == ui_mode_index(name)+1);
  ui_uninstall_mode(name);
  BuildBrowser( ModeBrowser,uistate.mode_count,uistate.modenames);
  if (first) {
    event_mode( uistate.modenames[0] , 0);
  }
  else
    adjust_browser(ModeBrowser, val, 6);
}

void ui_maybe_refresh(int id)
{
  if (norefresh) return;
  if ( id==WORLDGEOM || id==GEOMID(uistate.targetgeom) || id==ALLGEOMS ) {
    ui_select(GEOMID(uistate.targetgeom));
  } else if( id==ALLCAMS || id==FOCUSID
	|| id==CAMID(uistate.targetcam) || id==CAMID(uistate.mousefocus) ) {
    ui_select(CAMID(uistate.targetcam));
  } else {
    ui_select(NOID);
  }
}

void ui_select(int id)
{
  if(!AppearancePanel) return;	/* Initialized? */
  if(ISGEOM(id)) {
    register DGeom *dg;
    register Appearance *ap;
    
    if (dg=((id == GEOMID(ALLINDEX)) ? dgeom[0] : dgeom[INDEXOF(id)])) {
      ap = drawer_get_ap(id);
      fl_freeze_form( AppearancePanel );
      may_set_button( FaceDrawButton, (ap->flag & APF_FACEDRAW) ? 1 : 0);
      may_set_button( EdgeDrawButton, (ap->flag & APF_EDGEDRAW) ? 1 : 0);
      may_set_button( NormalDrawButton, (ap->flag & APF_NORMALDRAW) ? 1 : 0);
      may_set_button( EvertButton, (ap->flag & APF_EVERT) ? 1 : 0);
      may_set_button( BBoxDrawButton, (dg->bboxdraw) ? 1 : 0);
      if (ap->mat) {
	if(ap->mat->valid & MTF_DIFFUSE)
	  may_colorbutton( FaceColorButton, FACEINDEX, &ap->mat->diffuse);
	if(ap->mat->valid & MTF_EDGECOLOR)
	  may_colorbutton( EdgeColorButton, EDGEINDEX, &ap->mat->edgecolor);
	if(ap->mat->valid & MTF_NORMALCOLOR)
	  may_colorbutton( NormalColorButton, NORMALINDEX,&ap->mat->normalcolor);
	if(dg->bboxap && dg->bboxap->mat)
	  may_colorbutton( BBoxColorButton, BBOXINDEX,
			  &dg->bboxap->mat->edgecolor);
	if(ap->mat->valid & MTF_Ka)
	  may_set_slider( KaSlider, ap->mat->ka);
	if(ap->mat->valid & MTF_Kd)
	  may_set_slider( KdSlider, ap->mat->kd);
	if(ap->mat->valid & MTF_Ks)
	  may_set_slider( KsSlider, ap->mat->ks);
	if(ap->mat->valid & MTF_SHININESS)
	  may_set_slider( ShininessSlider, ap->mat->shininess);
      }
      may_set_browser( ShadingBrowser, ap->shading - CONSTANTSHADE +1, 3);
      fl_unfreeze_form( AppearancePanel );
      
      fl_freeze_form( ObscurePanel );
      may_set_finput( NormalScaleInput, ap->nscale);
      may_set_iinput( LinewidthInput, ap->linewidth);
      may_set_browser( NormalizationBrowser, dg->normalization+1, 3 );
      may_set_browser( SaveBrowser, uistate.savehow+1, 2 );
      fl_unfreeze_form( ObscurePanel );
      ApDelete(ap);
    }
    
  } else if(ISCAM(id)) {
    register DView *dv;
    int i;
    float f;
    
    if(id == FOCUSID)
      id = CAMID(uistate.mousefocus);
    if(INDEXOF(id) >= 0 && INDEXOF(id) < dview_max && 
       (dv = dview[INDEXOF(id)])) {
      
      fl_freeze_form( ObscurePanel );
      may_colorbutton( BackColorButton, BACKINDEX, &(dv->backcolor));
      CamGet(dv->cam, CAM_NEAR, &f);
      may_set_finput( NearClippingInput, f);
      CamGet(dv->cam, CAM_FAR, &f);
      may_set_finput( FarClippingInput, f);
      CamGet(dv->cam, CAM_PERSPECTIVE, &i);
      may_set_browser( ProjectionBrowser, i+1, 2);
      CamGet(dv->cam, CAM_FOV, &f);	/* perspective -> degrees, */
      may_set_finput( FOVInput, f );	/*  orthographic -> abs size */
      may_set_iinput( LinesCloserInput, (int)dv->lineznudge );
      may_set_button( DrawCameraButton, dv->cameradraw );
      fl_unfreeze_form( ObscurePanel );
    }
  }
  /* neither camera nor geometry specific */
  may_set_browser( SpaceBrowser, drawerstate.space+1, 2 );
  may_set_browser( SaveBrowser, uistate.savehow+1, 2 );
}

set_light_display(int lightno)
{
  extern Color *light_color();
  extern float light_intensity();
  Color *color;

  adjust_browser( LightingBrowser, lightno+1, 5);
  color = light_color();
  fl_mapcolor(LIGHTINDEX,(int) (color->r*255.0),(int) (color->g*255.0) ,
	      (int) (color->b*255.0));
  fl_redraw_object( LightColorButton );

  fl_set_slider_value( IntensitySlider, light_intensity() );
}

void ui_light_button()
{
  may_set_button( LightEditButton, uistate.lights_shown);
}

/*
 * Show what's happening on the keyboard.
 * Commands are enclosed in [brackets].  In case of error, a '?' appears
 * following the close-bracket.
 */
void
ui_keyboard(int ch)
{
  static char kybd[9];
  static int nextc = 0;
  if(ch <= 0) {
    ui_keyboard(']');
    if(ch < 0) ui_keyboard('?');
    nextc = 0;
  } else {
    if(nextc >= sizeof(kybd)-1) {
	bcopy(kybd+1, kybd, sizeof(kybd)-2);
	nextc = sizeof(kybd)-2;
    } else if(nextc == 0) {
	kybd[nextc++] = '[';
    }
    kybd[nextc++] = ch;
    kybd[nextc] = '\0';
    fl_set_object_label(KeyboardText, kybd);
  }
}

void ui_lights_changed()
{
  if (LightingBrowser) {
    fl_freeze_form(LightingPanel);
    BuildBrowser( LightingBrowser, light_count(), lights);
    set_light( uistate.current_light );
    fl_unfreeze_form(LightingPanel);
  }
}

/*
 * Invoke file input/output popup
 */
void ui_fileio( int dosave )
{
  ShowInput(NULL, dosave ? FILE_SAVE : FILE_GENERAL );
}

/*
 * Invoke color popup
 */
void ui_pickcolor(int val)
{
  Color old;
  char *name;
  int index;
  FL_OBJECT *obj;
  DView *dv;

  switch (val) {
  case DRAWER_BACKCOLOR:
	dv = (DView*)drawer_get_object( CAMID(uistate.targetcam) );
	ShowColor("Background", &(dv->backcolor), BACKINDEX,
		val, BackColorButton);
	return;

  case DRAWER_DIFFUSE:
	name = "Faces"; index = FACEINDEX; obj = FaceColorButton; break;
  case DRAWER_EDGECOLOR:
	name = "Edges"; index = EDGEINDEX; obj = EdgeColorButton; break;
  case DRAWER_NORMALCOLOR:
	name = "Normals"; index = NORMALINDEX; obj = NormalColorButton; break;
  case DRAWER_BBOXCOLOR:
	name = "BBoxes"; index = BBOXINDEX; obj = BBoxColorButton;
	break;
  }
  ui_color(index, &old);
  ShowColor(name, &old, index, val, obj);
}

/*
 * Toggle cursor
 */
void ui_curson(int on)
{
  if( (uistate.cursor_on = (on<0) ? !uistate.cursor_on : on) )
    curson();
  else cursoff();
}


/* 
 * Callback Procedures
 */

void CloseThisPanel(FL_OBJECT *obj, long val)
{
  fl_hide_form(obj->form);
}

static void ui_color(int index, Color *color)
{
  short r, g, b;

  fl_getmcolor(index, &r, &g, &b);
  color->r = ((float)r)/255.0;
  color->g = ((float)g)/255.0;
  color->b = ((float)b)/255.0;
}

void ColorSliderProc(FL_OBJECT *obj, long val)
{
  if (obj == Red)
    curcol.r = fl_get_slider_value(obj);
  else if (obj == Green)
    curcol.g = fl_get_slider_value(obj);
  else if (obj == Blue) 
    curcol.b = fl_get_slider_value(obj);

  fl_mapcolor(colindex,(int) (curcol.r*255.0),(int) (curcol.g*255.0) ,
	      (int) (curcol.b*255.0));
  fl_redraw_object(ColorPatch);

  if (curval == DRAWER_BACKCOLOR) {
    drawer_color( CAMID(uistate.targetcam), curval, &curcol );
  } else if (curval == DRAWER_LIGHTCOLOR) {
    set_light_color(&curcol);
  } else if (curval == DRAWER_BBOXCOLOR) {
    drawer_color( GEOMID(uistate.targetgeom), curval, &curcol);
  } else {
    drawer_color(uistate.targetid, curval, &curcol);
  }
  fl_redraw_object(curobj);

}

void OKColorProc(FL_OBJECT *obj, long val)
{
  ui_showpanel(P_COLORPICKER, 0);
}

void CancelColorProc(FL_OBJECT *obj, long val)
{
  ui_showpanel(P_COLORPICKER, 0);
  fl_mapcolor(colindex,(int) (oldcol.r*255.0),(int) (oldcol.g*255.0) ,
		(int) (oldcol.b*255.0));
  fl_redraw_object(curobj);

  if (curval == DRAWER_BACKCOLOR) {
    drawer_color( CAMID(uistate.targetcam), curval, &oldcol );
  } else if (curval == DRAWER_LIGHTCOLOR) {
    set_light_color(&oldcol);
  } else {
    drawer_color(uistate.targetid, curval, &oldcol);
  }

}
void HiddenReturnProc(FL_OBJECT *obj, long val)
{
  if (obj->form == ObscurePanel) {
    fl_set_object_focus(obj->form, ObscureHiddenInput1);
  } else if (obj->form == AppearancePanel)
    fl_set_object_focus(obj->form, AppearanceHiddenInput);
  else if (obj->form == ColorPanel)
    fl_set_object_focus(obj->form, ColorHiddenInput);
  else if (obj->form == LightingPanel)
    fl_set_object_focus(obj->form, LightingHiddenInput);
  else if (obj->form == CreditsPanel)
    fl_set_object_focus(obj->form, CreditsHiddenInput);
  else /*  if (obj->form == MotionPanel) */
    fl_set_object_focus(obj->form, MotionHiddenInput);
}

void HiddenInputProc(FL_OBJECT *obj, long val)
{
  Event event;
  char *str = fl_get_input(obj);
  char c = str[strlen(str)-1];

  if (c) {
    event.dev = c; 
    event.val = 1;
    fl_set_input(obj, "");
    dispatch_event(&event);
  }
}

/*
 * Input Panel
 */

void OKInputProc(FL_OBJECT *obj, long val)
{
  char *str = fl_get_input(PopupInput);
  if (str && *str) {
    usefilename(str);
    ui_showpanel(P_INPUT, 0);
  }
}

static char *FileTitles[] = {
	"Load Geometry/Camera/Command",
	"Load Transform",
	"Save %s %.20s Into ..."
};

void ShowInput(FL_OBJECT *obj, long val)
{
  int winid, oldid;
  char whatfor[64];
  DObject *dobj;

  uistate.fileuse = val;
  if (uistate.fileuse == FILE_SAVE) {
    fl_hide_object(FileBrowserButton);
  } else {
    fl_show_object(FileBrowserButton);
  }
  fl_set_input(PopupInput,"");
  sprintf(whatfor, FileTitles[uistate.fileuse],
	uistate.savehow ? "Commands for" : "Geometry of",
	uistate.savewhat == NOID ? "everything" :
	  (dobj = drawer_get_object(uistate.savewhat)) ? dobj->name[0] : NULL);
  panels[P_INPUT].name = whatfor;
  ui_showpanel(P_INPUT, 1);
}

void FileBrowserButtonProc(FL_OBJECT *obj, long val)
{
  static int winid;
  char **dirp = getfiledirs();
  char *dirpath, path[1024], *getwd(), *str, *line, *current;
  int i, changed;
  static char *array[64];

  uistate.fileuse = val;
  changed = 0;
  for (i=0; dirp[i] != NULL; i++) {
    if (!strcmp(dirp[i],".")) { /* dirp[i] == "." */
      current = getwd(path);
      if (!(array[i]) || (strcmp(array[i],current))) { 
	if (array[i]) OOGLFree(array[i]);
	array[i] = strdup(current);
	changed = 1;
      }
    } else {
      if (!(array[i]) || (strcmp(array[i],dirp[i]))) { 
	if (array[i]) OOGLFree(array[i]);
	array[i] = strdup(dirp[i]);
	changed = 1;
      }
    }
  }
  if (changed) BuildBrowser(DirectoryBrowser, i, array);
  show_browsing_dir();
  ui_showpanel(P_FILEBROWSER, 1);
}


/*
 * Motion Panel
 */

void
QuitProc(FL_OBJECT *obj, long val)
{
  drawer_exit_program();
}

void
DeleteProc(FL_OBJECT *obj, long val)
{
  drawer_delete( uistate.targetid );
}

void RecenterProc(FL_OBJECT *obj, long val)
{
  Transform T;

  TmIdentity( T );

  drawer_xform_set(ALLGEOMS, NULL, T);
  drawer_xform_set(WORLDGEOM, NULL, T);
  drawer_camera_reset(ALLCAMS);
}

void StopAllProc(FL_OBJECT *obj, long val)
{
  drawer_xform_incr(ALLCAMS,	NULL,  TM_IDENTITY);
  drawer_xform_incr(ALLGEOMS,	NULL,  TM_IDENTITY);
  drawer_xform_incr(WORLDGEOM,	NULL,  TM_IDENTITY);
}

void
PickBrowserProc(FL_OBJECT *obj, long type)
{
  ui_target( menuindex[fl_get_browser(obj)-1], IMMEDIATE );
}

void MoreBrowserProc(FL_OBJECT *obj, long val)
{
  ui_showpanel(fl_get_browser(obj)-1 + (P_MAIN+1), 1);
}

void
ModeBrowserProc(FL_OBJECT *obj, long val)
{
  int index = fl_get_browser( obj ) - 1;
  if (index < 0) return;
  ui_select_event_mode(index);
}

void TitleBrowserProc(FL_OBJECT *obj, long val)
{
  int i = fl_get_browser(TitleBrowser)-1;
  ui_title_mode( uistate.titles[i] );
}

void
AddCameraProc(FL_OBJECT *obj, long val)
{
  drawer_new_camera(NULL, NULL, NULL);
}

/* 
 * Appearance Panel
 */

void ToggleButtonProc(FL_OBJECT *obj, long val)
{
  int onoff = fl_get_button( obj );
  drawer_int(uistate.targetid, val, onoff);
}

void
ShadingBrowserProc(FL_OBJECT *obj, long val)
{
  drawer_int(uistate.targetid, DRAWER_SHADING,
	     fl_get_browser( obj ) - 1 + CONSTANTSHADE);
}


void ColorChange(FL_OBJECT *obj, long val)
{
  ui_pickcolor(val);
}
void SliderProc(FL_OBJECT *obj, long val)
{
  float f = fl_get_slider_value(obj);
  QUIETLY( drawer_float(uistate.targetid, val, f) );
}

/* 
 * Obscure Panel
 */

void ClippingProc(FL_OBJECT *obj, long val)
{ 
  glui_float(obj, val, CAMID(uistate.targetcam), 1e-6, 1e20); 
}

void FOVProc(FL_OBJECT *obj, long val)
{ 
  glui_float(obj, val, CAMID(uistate.targetcam), 1e-6, 179.999); 
}

void LinesCloserProc(FL_OBJECT *obj, long val)
{ 
  glui_float(obj, val, CAMID(uistate.targetcam), -10000., 10000.); 
  HiddenReturnProc(obj,val);
}

void NormalScaleProc(FL_OBJECT *obj, long val)
{ 
  glui_float(obj, val, GEOMID(uistate.targetgeom), 0., 999.); 
}

void LinewidthProc(FL_OBJECT *obj, long val)
{ 
  glui_int(obj, val, GEOMID(uistate.targetgeom), 1, 256); 
}

void BezDiceProc(FL_OBJECT *obj, long val)
{ 
  glui_int(obj, val, GEOMID(uistate.targetgeom), 0, 999); 
}


void
ObscureBrowserProc(FL_OBJECT *obj, long val)
{
  int i = fl_get_browser( obj ) - 1;
  int id;
  if (i >= 0) {
    switch (val) {
    case DRAWER_SPACE: id = NOID; break;
    case DRAWER_SAVE: id = NOID; break;
    case DRAWER_PROJECTION: id = CAMID( uistate.targetcam ); break;
    case DRAWER_NORMALIZATION: id = GEOMID( uistate.targetgeom ); break;
    }
    drawer_int( id, val, i );
  }
}

void
DrawCameraProc(FL_OBJECT *obj, long val)
{
  int button = fl_get_button(obj);
  drawer_int(CAMID(uistate.targetcam), DRAWER_CAMERADRAW, button );
}

/*
 * Lighting Panel
 */

void LightingBrowserProc(FL_OBJECT *obj, long val)
{
  int lightno = fl_get_browser(obj) - 1;
  set_light( lightno );
}

void LightColorButtonProc(FL_OBJECT *obj, long val)
{
  extern Color *light_color(); /* in clights.c */

  ShowColor("Light Color", light_color(), LIGHTINDEX, val, obj);
}

void AddLightButtonProc(FL_OBJECT *obj, long val)
{
  if (light_count() > 8) {
    OOGLError(0, "Can't have more than 8 lights in GL\n");
    return;
  }
  add_light();
}

void DelLightButtonProc(FL_OBJECT *obj, long val)
{
  /* don't delete ambient light ! */
  if (uistate.current_light == 0) return;
  delete_light();
}


void LightEditButtonProc(FL_OBJECT *obj, long val)
{
  int editing = fl_get_button(obj);

  light_edit_mode(editing);
}

/* 
 * Browser Panel
 */

static void show_browsing_dir()
{
    char hint[512];
    if(browsing_dir) {
	sprintf(hint, "%s/", browsing_dir);
	fl_set_input(AcceptInput, hint);
    }
}

static int maybe_rebuild_file_browser(char *str, char *path)
{
  char *array[512];
  DIR *dirp;
  struct direct *dp;
  register int i, j;
  char fullpath[512];
  char *s = str;

  if (path && str[0] != '/') {
    sprintf(fullpath, "%s/%s", path, str);
    s = fullpath;
  }
  dirp = opendir(s);
  if (dirp != NULL) {
    if(browsing_dir) free(browsing_dir);
    browsing_dir = strdup(s);
    show_browsing_dir();
    for(i = 0; (dp = readdir(dirp)) != NULL; i++) {
	/* Straight insert sort.  Could qsort(), but let's save space. */
      for(j = i; --j >= 0 && strcmp(dp->d_name, array[j]) < 0; )
	array[j+1] = array[j];
      array[j+1] = strdup(dp->d_name);
      if(i == COUNT(array)-2) {
	array[i++] = strdup("<more entries omitted>");
	break;
      }
    }
    closedir(dirp);
    BuildBrowser(FileBrowser, i, array);
    while(--i >= 0) free(array[i]);
    return 1;
  }
  return 0;
}

void DirectoryBrowserProc(FL_OBJECT *obj, long val)
{
  if (maybe_rebuild_file_browser(fl_get_browser_line(obj,fl_get_browser(obj)), NULL))
    show_browsing_dir();
}

void FileBrowserProc(FL_OBJECT *obj, long val)
{
  int i = fl_get_browser( obj ) - 1;
  char *str, fullpath[256];

  if ( (i >= 0) && (str = fl_get_browser_line(obj, i+1)) ) {
    if (!maybe_rebuild_file_browser(str,browsing_dir)) {
	if(browsing_dir && str[0] != '/') {
	    sprintf(fullpath, "%s/%s", browsing_dir, str);
	    str = fullpath;
	}
	ui_showpanel(P_INPUT, 0);
	usefilename(str);
	return;
    }
  }
}

void AcceptProc(FL_OBJECT *obj, long val)
{
  char *str = fl_get_input(obj);
  if (str && *str) {
    fl_deselect_browser_line(DirectoryBrowser, 
			     fl_get_browser(DirectoryBrowser));
    fl_deselect_browser_line(FileBrowser, fl_get_browser(FileBrowser));
    if (!maybe_rebuild_file_browser(str, NULL)) {
      usefilename(str);
    }
  }
}

/*
 * Command Panel
 */

/* hack: this is actually the callback of a hidden return button
 * "CommandReturnButton" instead of the CommandInput so that
 * when you hit return you re-execute the previous command
 * (Forms will not trigger an input's callback if nothing has changed.)
 */

void CommandInputProc(FL_OBJECT *obj, long val)
{
  char *str = fl_get_input(CommandInput);
  if (str && *str)
    comm_object(str, &CommandOps, NULL, NULL, COMM_NOW);
}

/*
 * Private procedures: make these static soon? 
 */

static int id2menuindex(int id)
{
  int i;

  for (i=0; i<menucount; ++i)
    if (menuindex[i] == id) return i;
  return -1;
}


void may_colorbutton( FL_OBJECT *btn, int index, Color *c )
{
  short cwas[3], cis[3];
  cis[0] = 255*c->r;  cis[1] = 255*c->g; cis[2] = 255*c->b;
  fl_getmcolor( index, &cwas[0],&cwas[1],&cwas[2] );
  if(memcmp(cwas, cis, sizeof(cwas))) {
      fl_mapcolor( index, cis[0],cis[1],cis[2] );
      fl_set_object_color( btn, index, index );
  }
}

static void may_set_button( FL_OBJECT *btn, int state )
{
  if(fl_get_button(btn) != state)
    fl_set_button(btn, state);
}

/* note inputs deal with strings, not numbers */
static void may_set_iinput( FL_OBJECT *input, int val)
{
  char buf[10];
  int i;
  if (!getint(input, &i) || i != val) {
    sprintf(buf,"%3d",val);
    fl_set_input(input, buf);
  }
}

/* note inputs deal with strings */
static void may_set_sinput( FL_OBJECT *input, char *val)
{
  char *str = fl_get_input(input);
  if ((strcmp(str,val))) { /* if str != val */
    fl_set_input(input, val);
  }
}

/* note inputs deal with strings, not numbers */
static void may_set_finput( FL_OBJECT *input, float val)
{
  char buf[80];
  float f;
  if (!getfloat(input, &f) || f != val) {
    sprintf(buf,"%5g",val);
    fl_set_input(input, buf);
  }
}

static void may_set_browser( FL_OBJECT *browser, int line, int size)
{
  if(fl_get_browser(browser) != line) 
    adjust_browser(browser, line, size);
}

static Handle *emotion_target()
{
  char newname[32], *oldname, *str, *tail;
  Handle *h;

  h = has_extern_motion(uistate.targetid);
  str = HandleName( h );
  if (h) {
    /* only use tail of pathname */
    tail = strrchr(str, '/');
    if (tail) tail++;
    else tail = str;
    sprintf(newname,"Extern: %s", tail);
  } else  {
    sprintf(newname,"%s","External Motion");
  }
  if (!streq( uistate.modenames[uistate.extern_index], newname)) 
    ui_replace_mode(strdup(newname), NULL, T_NONE, uistate.extern_index);
  return h;
}


/*
 * Adjust a browser to make its new choice visible.
 * If possible, leave the old choice visible too.
 */
static void adjust_browser( FL_OBJECT *brow, int is, int visible )
{
  int was, max;

  was = fl_get_browser(brow);
  if(was == is)
    return;
  max = fl_get_browser_maxline( brow );

  /*
   * If we can't place both old & new values in window, don't try.
   */
  if(was - is < -visible || was - is > visible)
    was = is;

  fl_select_browser_line( brow, is );

  if(was <= visible && is <= visible) {
    fl_set_browser_topline( brow, 1 );	/* First preference: hop to top.  */
  } else if(was >= max - visible && is >= max - visible) {
    fl_set_browser_topline( brow, max - visible + 1 ); /* Second: hop to bottom */
  } else {
    /* Last: center both in window */
    fl_set_browser_topline( brow, (was+is-visible) / 2 + 1 );
  }
}

static void
BuildBrowser(FL_OBJECT *browser, int count, char *names[])
{
  int i;

  fl_freeze_object( browser );
  fl_clear_browser( browser );
  for (i=0; i<count; ++i)
    fl_add_browser_line(browser, names[i]);
  fl_unfreeze_object( browser );
}

BuildObjectMenuList( char ***menulist, int **menuindex, int *count )
{
  register int i;
  int n, max, camcount;
  register DObject *obj, **objs;
  char buf[64];

  if ((camcount = drawer_cam_count()) > 1) {
    /* + 1 for "Current Camera" entry */
    *count = camcount + drawer_geom_count() + 1;     

    /* an obscure place to put code for the obscure panel... */
    fl_show_object(DrawCameraButton);
  } else {
    /* only have "Current Camera" */
    fl_hide_object(DrawCameraButton);
    *count = drawer_geom_count() + 1;
  }

  *menulist = OOGLNewNE(char *, *count, "no space for menulist array");
  *menuindex = OOGLNewNE(int, *count, "no space for menuindex array");

  n = 0;
  objs = (DObject**)dgeom;
  for (i=0; i<dgeom_max; ++i) {
    if ( (obj=objs[i]) != NULL && ((DGeom*)obj)->citizenship != ALIEN) {
      if (obj->name[1] != NULL) 
	sprintf(buf, "[%s] %s", obj->name[0], obj->name[1]);
      else
	strcpy(buf, obj->name[0]);
      (*menuindex)[n] = obj->id;
      (*menulist)[n] = strdup(buf);
      ++n;
    }
  }
  (*menuindex)[n] = FOCUSID;
  (*menulist)[n] = strdup("[c] Current Camera");
  ++n;
  if (camcount > 1) {
    objs = (DObject **)dview;
    for (i=0; i<dview_max; ++i) {
      if ( (obj=objs[i]) != NULL ) {
	if (obj->name[1] == NULL) obj->name[1] = "Default Camera";
	sprintf(buf, "[%s] %s", obj->name[0], obj->name[1]);
	(*menuindex)[n] = obj->id;
	(*menulist)[n] = strdup(buf);
	++n;
      }
    }
  }
}

DestroyObjectMenuList( char **menulist, int count )
{
  while ( --count >= 0 )
    OOGLFree(menulist[count]);
  OOGLFree(menulist);
}

static void
BuildBrowserMenu(FL_OBJECT *obj)
{
  register int i;
  char **menulist;

  fl_freeze_object( obj );
  fl_clear_browser( obj );
  if (menuindex) OOGLFree(menuindex);
  BuildObjectMenuList(&menulist, &menuindex, &menucount);
  for (i=0; i<menucount; ++i)
    fl_add_browser_line(obj, menulist[i]);
  fl_unfreeze_object( obj );
  DestroyObjectMenuList(menulist, menucount);
}



static void
may_set_slider( FL_OBJECT *obj, float value )
{
  if (fl_get_slider_value(obj) != value)
    fl_set_slider_value(obj, value);
}

/*
 * Utility routines for interpreting input values
 */

static void
glui_int(FL_OBJECT *obj, int key, int id, int min, int max)
{
    int i;
    if(getint(obj, &i)) {
	if(i < min) i = min;
	else if(i > max) i = max;
	drawer_int(id, key, i);
    } else {
	ui_select(id);
    }
}

static void
glui_float(FL_OBJECT *obj, int key, int id, float min, float max)
{
    float f;
    if(getfloat(obj, &f)) {
	if(f < min) f = min;
	else if(f > max) f = max;
	drawer_float(id, key, f);
    } else {
	ui_select(id);
    }
}
static int
getfloat(FL_OBJECT *input, float *fp)
{
   char *after, *str = fl_get_input(input);
   *fp = strtod(str, &after);
   return str != after;
}

static int
getint(FL_OBJECT *input, int *ip)
{
   char *after, *str = fl_get_input(input);
   *ip = strtol(str, &after, 0);
   return str != after;
}

/* str is full pathname when this is called FileSelectorProc and
 * possibly just a tail when called from InputProc.
 */
static void usefilename(char *str)
{
  Geom *geom = NULL;
  Camera *cam = NULL;
  Handle *h = NULL;
  Transform *xform = NULL;
  extern HandleOps GeomOps, CamOps, TransOps, CommandOps;
  int index, id;
  char *pathname;
  Pool *p;
  
  switch (uistate.fileuse) {
  case FILE_GENERAL:
    loadfile(str, &CommandOps);
    break;
  case FILE_TRANSFORM:
    if (pathname=findfile(NULL, str))
      str = pathname;
    if (comm_object(str, &TransOps, &h, NULL, COMM_LATER)) {
	drawer_xform_set(uistate.targetid, h, TMNULL);
	emotion_target();
    } else {
      OOGLError(0,"Can't find or interpret transform %s",str);
    }
    break;
  case FILE_SAVE:
    p = PoolStreamTemp(str, streq(str,"-") ? stdout : NULL, 1, &CommandOps);
    if(p == NULL) {
	OOGLError(1, "Can't write to \"%s\": %s", str, sperror());
    } else {
	save_world(p, uistate.savewhat, uistate.savehow, 1);
	if(PoolOutputFile(p) != stdout) PoolClose(p);
	PoolDelete(p);
    }
    break;
  }
}      	

void
ShowColor(char *name, Color *old, int index, int val, FL_OBJECT *obj)
{
  int winid, key = 0;

  curval = val;
  colindex = index;
  CoCopy(old,&oldcol);	    
  CoCopy(old,&curcol);	    
  curobj = obj;
  fl_set_slider_value(Red,oldcol.r);
  fl_set_slider_value(Green,oldcol.g);
  fl_set_slider_value(Blue,oldcol.b);
  fl_set_object_color(ColorPatch,index,index);
  fl_mapcolor(index,(int) (oldcol.r*255.0),(int) (oldcol.g*255.0) ,
	      (int) (oldcol.b*255.0));
  panels[P_COLORPICKER].name = name;
  ui_showpanel(P_COLORPICKER, 1);
  /* turn on drawing of relevant thing automatically if you change the color */
  switch(val) {
  case DRAWER_DIFFUSE:	key = DRAWER_FACEDRAW; break;
  case DRAWER_EDGECOLOR: key = DRAWER_EDGEDRAW; break;
  case DRAWER_NORMALCOLOR: key = DRAWER_NORMALDRAW; break;
  case DRAWER_BBOXCOLOR: key = DRAWER_BBOXDRAW; break;
  default: return;
  }
  drawer_int(uistate.targetid, key, 1);
}

static Handle *has_extern_motion(int id)
{
  Handle *h = NULL;
  DObject *obj;

  if (obj = drawer_get_object(id)) {
    if (TYPEOF(id) == T_GEOM) {
      GeomGet(((DGeom*)obj)->Item, CR_AXISHANDLE, &h);
    } else if (TYPEOF(id) == T_CAM) {
      CamGet(((DView*)obj)->cam, CAM_C2WHANDLE, &h);
    }
  }
  return h;
}

void
ui_hyperbolic_setup(int enter)
{
  if (enter) {
    fl_show_object(DrawSphereButton);
  } else {
    fl_hide_object(DrawSphereButton);
  }
}

void
ui_hsphere_draw(int draw)
{
  may_set_button( DrawSphereButton, draw );
}
