/*
 *  latdiffs.c:    does the dirty work with respect to lattices
 *		to get more information on lattices and what freedoms they
 * 		possess, see "Geometric Symmetry", by Lockwood and MacMillan.
 *
 *	Routines:  ouax, and its minions.
 *
 *  July 1990, by Olaf Holt
 */

#include <math.h>
#include "geom.h"
#include "linalg.h"
#include "forms.h"
#include "controlpanel.h"

#define PRINT 3

extern float parama,paramb,paramc;
extern int lattype;
extern Transform zszhrp;
extern int usedsliders;
extern short VERBOSEFLAG;
char labelstring[40];

/*
 *  Function:   triclinic
 *  Description:        prints information for a triclinic lattice
 * 					and updates parameters appropriately when the user
 *					wants to change the shape of the lattice
 *  Args IN:    lflag:  flag telling if routine should print (3) or not (0)
 *  Returns:  nothing
 */
triclinic(lflag)
short lflag;
{
    if(lflag == PRINT) {
        if (VERBOSEFLAG) fprintf(stderr,"This group uses triclinic co-ordinates:\n\t'X' = (1,0,0), 'Y' =  (a,1,0), and 'Z' = (0,b,c).\n");
    }
		fl_set_object_label(xposition,"'x' = (1,0,0)");
		fl_set_object_label(wyeposition,"'wye' = (a,1,0)");
		fl_set_object_label(zedposition,"'zed' = (0,b,c)");
		fl_set_object_label(xdir,"(1,0,0)");
		sprintf(labelstring,"(%.1f,1,0)",parama);
		fl_set_object_label(wyedir,labelstring);
		sprintf(labelstring,"(0,%.1f,%.1f)",paramb,paramc);
		fl_set_object_label(zeddir,labelstring);
        MATEQUAL4(zszhrp,IDENTITYMATRIX4);
	usedsliders = 3;
    zszhrp[0][1] = parama; zszhrp[1][2] = paramb; zszhrp[2][2] = paramc;
}

/*
 *  Function:   monoclinic
 *  Description:        prints information for a monoclinic lattice
 * 					and updates parameters appropriately when the user
 *					wants to change the shape of the lattice
 *  Args IN:    lflag:  flag telling if routine should print (3) or not (0)
 *  Returns:  nothing
 */
monoclinic(lflag)
short lflag;
{
    if(lflag == PRINT) {
        if (VERBOSEFLAG) fprintf(stderr,"This group uses monoclinic co-ordinates:\n\t'X' = (1,0,0), 'Y' = (a,b,0), and 'Z' = (0,0,c).\n");
    }
		fl_set_object_label(xposition,"'x' = (1,0,0)");
		fl_set_object_label(wyeposition,"'wye' = (a,b,0)");
		fl_set_object_label(zedposition,"'zed' = (0,0,c)");
		fl_set_object_label(xdir,"(1,0,0)");
		sprintf(labelstring,"(%.1f,%.1f,0)",parama,paramb);
		fl_set_object_label(wyedir,labelstring);
		sprintf(labelstring,"(0,0,%.1f)",paramc);
		fl_set_object_label(zeddir,labelstring);
        MATEQUAL4(zszhrp,IDENTITYMATRIX4);
	usedsliders = 3;
    zszhrp[0][1] = parama; zszhrp[1][1] = paramb; zszhrp[2][2] = paramc;
}

/*
 *  Function:   orthorhombic
 *  Description:        prints information for a orthorhombic lattice
 * 					and updates parameters appropriately when the user
 *					wants to change the shape of the lattice
 *  Args IN:    lflag:  flag telling if routine should print (3) or not (0)
 *  Returns:  nothing
 */
orthorhombic(lflag)
short lflag;
{
    if(lflag == PRINT) {
        if (VERBOSEFLAG) fprintf(stderr,"This group uses orthorhombic co-ordinates:\n\t'X' = (1,0,0), 'Y' = (0,a,0), and 'Z' = (0,0,b).\n");
    }
		fl_set_object_label(xposition,"'x' = (1,0,0)");
		fl_set_object_label(wyeposition,"'wye' = (0,a,0)");
		fl_set_object_label(zedposition,"'zed' = (0,0,b)");
		fl_set_object_label(xdir,"(1,0,0)");
		sprintf(labelstring,"(0,%.1f,0)",parama);
		fl_set_object_label(wyedir,labelstring);
		sprintf(labelstring,"(0,0,%.1f)",paramb);
		fl_set_object_label(zeddir,labelstring);
        MATEQUAL4(zszhrp,IDENTITYMATRIX4);
	usedsliders = 2;
    zszhrp[1][1] = parama; zszhrp[2][2] = paramb;
}

/*
 *  Function:   tetragonal
 *  Description:        prints information for a tetragonal lattice
 * 					and updates parameters appropriately when the user
 *					wants to change the shape of the lattice
 *  Args IN:    lflag:  flag telling if routine should print (3) or not (0)
 *  Returns:  nothing
 */
tetragonal(lflag)
short lflag;
{
    if(lflag == PRINT) {
        if (VERBOSEFLAG) fprintf(stderr,"This group uses tetragonal co-ordinates:\n\t'X' = (1,0,0), 'Y' =(0,1,0), and 'Z' = (0,0,b).\n");
    }
		fl_set_object_label(xposition,"'x' = (1,0,0)");
		fl_set_object_label(wyeposition,"'wye' = (0,1,0)");
		fl_set_object_label(zedposition,"'zed' = (0,0,b)");
		fl_set_object_label(xdir,"(1,0,0)");
		fl_set_object_label(wyedir,"(0,1,0)");
		sprintf(labelstring,"(0,0,%.1f)",paramb);
		fl_set_object_label(zeddir,labelstring);
        MATEQUAL4(zszhrp,IDENTITYMATRIX4);
	usedsliders = 1;
    zszhrp[2][2] = paramb;
}

/*
 *  Function:   cubic
 *  Description:        prints information for a cubic lattice
 *  Args IN:    lflag:  flag telling if routine should print (3) or not (0)
 *  Returns:  nothing
 */
cubic(lflag)
short lflag;
{
    if(lflag == PRINT) {
        if (VERBOSEFLAG) fprintf(stderr,"This group uses cubic co-ordinates:\n\t'X' = (1,0,0), 'Y' =(0,1,0), and 'Z' = (0,0,1).\n");
    }
		fl_set_object_label(xposition,"'x' = (1,0,0)");
		fl_set_object_label(wyeposition,"'wye' = (0,1,0)");
		fl_set_object_label(zedposition,"'zed' = (0,0,1)");
		fl_set_object_label(xdir,"(1,0,0)");
		fl_set_object_label(wyedir,"(0,1,0)");
		fl_set_object_label(zeddir,"(0,0,1)");
        MATEQUAL4(zszhrp,IDENTITYMATRIX4);
	usedsliders = 0;
}

/*
 *  Function:   rhombohedral
 *  Description:        prints information for a rhombohedral lattice
 * 					and updates parameters appropriately when the user
 *					wants to change the shape of the lattice
 *  Args IN:    lflag:  flag telling if routine should print (3) or not (0)
 *  Returns:  nothing
 */
rhombohedral(lflag)
short lflag;
{
	float syne, cyne;
	syne = sin(paramb);  cyne = cos(paramb);

    if(lflag == PRINT) {
        if (VERBOSEFLAG) fprintf(stderr,"This group uses rhombohedral co-ordinates:\n\tThree axially symmetric translations determined by an angle (b).\n");
    }
		fl_set_object_label(xposition,"'x','wye','zed'");
		fl_set_object_label(wyeposition,"are axially sym-");
		fl_set_object_label(zedposition,"metric! (use b!)");
        MATEQUAL4(zszhrp,IDENTITYMATRIX4);
	usedsliders = 1;
zszhrp[0][1] = cyne*cyne-0.5*syne*syne; zszhrp[1][1] = ROOTTOT*syne;
        zszhrp[2][1] = syne*cyne+0.5*cyne*syne;
zszhrp[0][2] = zszhrp[0][1]; zszhrp[1][2] = -(zszhrp[1][1]); zszhrp[2][2] = zszhrp[2][1];
		fl_set_object_label(xdir,"(1,0,0)");
		sprintf(labelstring,"(%.1f,%.1f,%.1f)",zszhrp[0][1],zszhrp[1][1],zszhrp[2][1]);
		fl_set_object_label(wyedir,labelstring);
		sprintf(labelstring,"(%.1f,%.1f,%.1f)",zszhrp[0][2],zszhrp[1][2],zszhrp[2][2]);
		fl_set_object_label(zeddir,labelstring);
}

/*
 *  Function:   hexagonal
 *  Description:        prints information for a hexagonal lattice
 * 					and updates parameters appropriately when the user
 *					wants to change the shape of the lattice
 *  Args IN:    lflag:  flag telling if routine should print (3) or not (0)
 *  Returns:  nothing
 */
hexagonal(lflag)
short lflag;
{
    if(lflag == PRINT) {
        if (VERBOSEFLAG) fprintf(stderr,"This group uses hexagonal co-ordinates:\n\t'X' = (1,0,0), 'Y' =(cos(120),sin(120),0), and 'Z' = (0,0,b).\n");
    }
		fl_set_object_label(xposition,"'x' = (1,0,0)");
		fl_set_object_label(wyeposition,"'wye'~(-.5,.855,0)");
		fl_set_object_label(zedposition,"'zed' = (0,0,b)");
		fl_set_object_label(xdir,"(1,0,0)");
		fl_set_object_label(wyedir,"(-.5,.855,0)");
		sprintf(labelstring,"(0,0,%.1f)",paramb);
		fl_set_object_label(zeddir,labelstring);
        MATEQUAL4(zszhrp,IDENTITYMATRIX4);
        zszhrp[0][1] = -0.5; zszhrp[1][1] = ROOTTOT;
	usedsliders = 1;
    zszhrp[2][2] = paramb;
}

/*
 *  Function:   ouax
 *  Description:    jumps to the current lattice routine (see above)
 *  Args IN:    lflag:  a flag to print or not to print
 *  Returns:  nothing
 */
ouax(lflag)
short lflag;
{
switch(lattype) {
    case 0:
        fprintf(stderr,"You must choose a group in order for this parameter to be functional.\n"
);
        MATEQUAL4(zszhrp,IDENTITYMATRIX4);
        break;
    case 1:
        triclinic(lflag);
        break;
    case 2: case 3:
        monoclinic(lflag);
        break;
    case 4: case 5: case 6: case 7:
        orthorhombic(lflag);
        break;
    case 8: case 9:
        tetragonal(lflag);
        break;
    case 10: case 11: case 12:
        cubic(lflag);
        break;
    case 13:
        rhombohedral(lflag);
        break;
    case 14:
        hexagonal(lflag);
        break;
    default:
        fprintf(stderr, "error in parser (bravais type)\n");
        exit(0);
    }
}
