/*____________________________________________________________________________
	Copyright (C) 1997 Network Associates Inc. and affiliated companies.
	All rights reserved.
	
	
	
	$Id: CPGPEncoderDecoder.cp,v 1.63 1999/04/10 11:14:14 jason Exp $
____________________________________________________________________________*/

#include <string.h>

#include <UMemoryMgr.h>

#include "PGPKeys.h"
#include "PGPUtilities.h"
#include "MacFiles.h"
#include "pgpMacMemory.h"
#include "MacStrings.h"
#include "MacErrors.h"
#include "PGPSharedEncryptDecrypt.h"
#include "PGPUserInterface.h"
#include "WarningAlert.h"
#include "pflPrefTypes.h"
#include "pgpOpenPrefs.h"
#include "pgpClientPrefs.h"
#include "pgpAdminPrefs.h"
#include "pgpWordWrap.h"
#include "pgpMemoryIO.h"
#include "pgpVersionHeader.h"
#include "CSecureMemory.h"
#include "pgpClientErrors.h"

#include "CTempFile.h"

#include "CPGPEncoderDecoder.h"


struct SDecodeData {
	Boolean				useCache;
	Boolean				FYEO;
	AnimatedCursorRef	cursor;
	Handle				handle;
	PGPKeySetRef		defaultKeys;
};


// Statics
Boolean					CPGPEncoderDecoder::sCheckForMissingKeys = true;
CSignPassphraseCache *	CPGPEncoderDecoder::sSigningPassphraseCache = nil;
CSignPassphraseCache *	CPGPEncoderDecoder::sDecryptPassphraseCache = nil;
PGPtlsContextRef		CPGPEncoderDecoder::sTLSContext
												= kInvalidPGPtlsContextRef;



// Constants
const ResIDT	STRx_PGPEncoderDecoder				=	12121;
const SInt16	kNoPublicKeysID						=	1;
const SInt16	kNoPrivateKeysID					=	2;
const SInt16	kAddKeysID							=	3;

const ResIDT	STRx_PGPEncoderDecoderErrorStrings	=	12122;
const SInt16	kErrorStringID						=	1;
const SInt16	kErrorStringAdminPrefsNotFoundID	=	2;

	void
CPGPEncoderDecoder::Init(
	PGPContextRef	inContext)
{
	PGPError	pgpError;
	
	sSigningPassphraseCache = new CSignPassphraseCache(inContext);
	sDecryptPassphraseCache = new CSignPassphraseCache(inContext);
	pgpError = PGPNewTLSContext(inContext, &sTLSContext);
	PGPThrowIfPGPError_(pgpError);
	
#if PGP_BUSINESS_SECURITY
	pgpError = PGPCheckAutoUpdateKeysFromServer(
						PGPGetContextMemoryMgr(inContext),
						true,
						nil,
						nil);
	PGPThrowIfPGPError_(pgpError);
#endif
}



	void
CPGPEncoderDecoder::Cleanup()
{
	delete sSigningPassphraseCache;
	delete sDecryptPassphraseCache;
	if (PGPtlsContextRefIsValid(sTLSContext)) {
		PGPFreeTLSContext(sTLSContext);
	}
}



	Boolean
CPGPEncoderDecoder::EncodeHandle(
	PGPContextRef				inContext,
	Handle						ioDataH,
	EEncodeOptions				inOptions,
	UInt32						inNumRecipients,
	const PGPRecipientSpec *	inRecipientList,
	Boolean						inUseCache)
{
	Boolean				result = false;
	OSErr				err;
	PGPError			pgpErr;
	AnimatedCursorRef	cursorRef = nil;
	PGPKeySetRef		defaultKeySet = kInvalidPGPKeySetRef;
	PGPKeySetRef		recipientSet = kInvalidPGPKeySetRef;
	PGPKeySetRef		newKeys = kInvalidPGPKeySetRef;
	PGPKeySetRef		selectedSet = kInvalidPGPKeySetRef;
	PGPKeyRef			signingKey = kInvalidPGPKeyRef;
	PGPOptionListRef	optionList = kInvalidPGPOptionListRef;
	PGPPrefRef			prefRef = kInvalidPGPPrefRef;
	PGPIORef			wrapInput = kInvalidPGPIORef;
	PGPIORef			wrapOutput = kInvalidPGPIORef;
	
	try {
		if (::GetHandleSize(ioDataH) > 0) {
			CSecureCString256	convEncryptPassphrase;
			CSecureMemory		signingBuffer(
									PGPGetContextMemoryMgr(inContext),
									256);
			PGPSize				signingPasskeySize;
			PGPUInt32			numPublicKeys = 1;
			PGPUInt32			numPrivateKeys = 1;
			Boolean				usePassphrase = true;
			Boolean				fyeo = false;
	
			// Check for missing keys
			if (sCheckForMissingKeys) {
				err = CountKeysInDefaultKeyring(	inContext,
													&numPublicKeys,
													&numPrivateKeys);
				PGPThrowIfOSError_(err);
				
				if ((numPublicKeys != 0) && (numPrivateKeys != 0)) {
					sCheckForMissingKeys = false;
				}
			}
			
			// Open default keyset
			pgpErr = PGPOpenDefaultKeyRings(	inContext,
												0,
												&defaultKeySet);
			PGPThrowIfPGPError_(pgpErr);

			// If encrypting, get the recipients
			if ((inOptions & encodeOptions_Encrypt)
			&& (! (inOptions & encodeOptions_ConvEncrypt))) {
				PGPRecipientOptions		options = kPGPRecipientOptionsHideFileOptions
											| kPGPRecipientOptionsShowFYEO;
				PGPRecipientSettings	settings =
											kPGPRecipientSettingsDefault;
				
				if (numPublicKeys == 0) {
					WarningAlert(	kWACautionAlertType,
									kWAOKStyle,
									STRx_PGPEncoderDecoder,
									kNoPublicKeysID);
				}

				if (! inUseCache) {
					options |= kPGPRecipientOptionsAlwaysShowDialog;
				}
				pgpErr = PGPClientRecipientDialog(	inContext,
													sTLSContext,
													defaultKeySet,
													inNumRecipients,
													inRecipientList,
													options,
													settings,
													&settings,
													&recipientSet,
													&newKeys);
				PGPThrowIfPGPError_(pgpErr);

				if (PGPRefIsValid(newKeys)) {
					pgpErr = PGPSelectKeysDialog(
								inContext,
								kPGPSelectKeysImportVariation,
								CString(CStringFromSTRx(	STRx_PGPEncoderDecoder,
															kAddKeysID)),							
								newKeys,							
								defaultKeySet,
								&selectedSet);
					if (pgpErr != kPGPError_UserAbort) {
						PGPThrowIfPGPError_(pgpErr);
						pgpErr = PGPSharedAddKeysToDefaultKeyring(
									selectedSet);
						PGPThrowIfPGPError_(pgpErr);
					}
				}
				
				if (settings & kPGPRecipientSettingsConvEncrypt) {
					inOptions &= ~encodeOptions_Encrypt;
					inOptions |= encodeOptions_ConvEncrypt;
				} else {
					PGPUInt32		numKeys;
					pgpErr	= PGPCountKeys( recipientSet, &numKeys);
					PGPThrowIfPGPError_(pgpErr);
				
					if ( numKeys == 0) {
						inOptions &= ~encodeOptions_Encrypt;
						if (! (inOptions & encodeOptions_Sign)) {
							PGPThrowOSError_(userCanceledErr);
						}
					}
				}
				
				if (settings & kPGPRecipientSettingsFYEO) {
					fyeo = true;
				}
			}
			
			// If conventionally encrypting, get passphrase
			if (inOptions & encodeOptions_ConvEncrypt) {
				char *	conventionalPassphrase;
				
				pgpErr = PGPConventionalEncryptionPassphraseDialog(
									inContext,
									PGPOUIOutputPassphrase(	inContext,
															&conventionalPassphrase),
									PGPOLastOption(inContext));
				PGPThrowIfPGPError_(pgpErr);
				strcpy(convEncryptPassphrase, conventionalPassphrase);
				PGPFreeData(conventionalPassphrase);
			}
			
			// If signing, get passphrase
			if (inOptions & encodeOptions_Sign) {
				PGPGetPassphraseSettings	userSettings;
				
				// Make sure we have a key to sign to
				if (numPrivateKeys == 0) {
					WarningAlert(	kWACautionAlertType,
									kWAOKStyle,
									STRx_PGPEncoderDecoder,
									kNoPrivateKeysID);
				}
				
				// Get passphrase
				if (! inUseCache) {
					sSigningPassphraseCache->Forget();
				}
				if (! sSigningPassphraseCache->GetPassphraseOrPasskey(defaultKeySet,
				signingBuffer.mMemory, &usePassphrase, &signingPasskeySize,
				&signingKey)) {
					usePassphrase = true;
					pgpErr = PGPClientSigningPassphraseDialog(
								inContext,
								defaultKeySet,
								nil,
								kPGPGetPassphraseOptionsHideFileOptions,
								userSettings,
								signingKey,
								(char *) signingBuffer.mMemory,
								&userSettings,
								&signingKey);
					if (pgpErr != kPGPError_KeyUnusableForSignature) {
						PGPThrowIfPGPError_(pgpErr);
					} else {
						PGPByte *	passkey = nil;
						
						pgpErr = PGPReconstitutionDialog(
										signingKey,
										defaultKeySet,
										kInvalidPGPtlsContextRef,
										&passkey,
										&signingPasskeySize);
						PGPThrowIfPGPError_(pgpErr);
						pgpCopyMemory(	passkey,
										signingBuffer.mMemory,
										signingPasskeySize);
						PGPFreeData(passkey);
						usePassphrase = false;
					}
					SetPassbufferCache(	true,
										signingKey,
										usePassphrase,
										signingBuffer.mMemory,
										signingPasskeySize);
				}
				
				// If we are only signing, wrap the input
				if ((! (inOptions & encodeOptions_Encrypt))
				&& (! (inOptions &encodeOptions_ConvEncrypt))) {
					PGPBoolean	wordWrapEnabled;
					
					pgpErr = PGPOpenClientPrefs(
								PGPGetContextMemoryMgr(inContext),
								&prefRef);
					PGPThrowIfPGPError_(pgpErr);
					pgpErr = PGPGetPrefBoolean(	prefRef,
												kPGPPrefWordWrapEnable,
												&wordWrapEnabled);
					PGPThrowIfPGPError_(pgpErr);
					
					if (wordWrapEnabled) {
						PGPUInt32		wordWrapWidth;
						PGPFileOffset	bufSize;
						PGPSize			unused;	
						
						{
							StHandleLocker	theLock(ioDataH);
							
							pgpErr = PGPGetPrefNumber(	prefRef,
														kPGPPrefWordWrapWidth,
														&wordWrapWidth);
							PGPThrowIfPGPError_(pgpErr);
							pgpErr = PGPNewMemoryIOFixedBuffer(
											PGPGetContextMemoryMgr(inContext),
											*ioDataH,
											::GetHandleSize(ioDataH),
											(PGPMemoryIORef*) &wrapInput);
							PGPThrowIfPGPError_(pgpErr);
							pgpErr = PGPNewMemoryIO(
										PGPGetContextMemoryMgr(inContext),
										(PGPMemoryIORef*) &wrapOutput);
							PGPThrowIfPGPError_(pgpErr);
							pgpErr = pgpWordWrapIO(	wrapInput,
													wrapOutput,
													wordWrapWidth,
													"\r");
							PGPThrowIfPGPError_(pgpErr);
							PGPFreeIO(wrapInput);
							wrapInput = kInvalidPGPIORef;
						}
						
						pgpErr = PGPIOSetPos(	wrapOutput,
												0);
						PGPThrowIfPGPError_(pgpErr);
						pgpErr = PGPIOGetEOF(	wrapOutput,
												&bufSize);
						PGPThrowIfPGPError_(pgpErr);
						::SetHandleSize(ioDataH, bufSize);
						PGPThrowIfMemError_();
						::HLock(ioDataH);
						pgpErr = PGPIORead(	wrapOutput,
											bufSize,
											*ioDataH,
											&unused);
						::HUnlock(ioDataH);
						PGPThrowIfPGPError_(pgpErr);
						PGPFreeIO(wrapOutput);
						wrapOutput = kInvalidPGPIORef;
					}
				}
			}

			// Load the cursor
			err = Get1AnimatedCursor(acur_BeachBall, &cursorRef);
			PGPThrowIfOSError_(err);
			
			// Do the encrypting
			PGPSize				bufSize;
			void *				buf;

			pgpErr = PGPGuaranteeMinimumEntropy(inContext);
			PGPThrowIfPGPError_(pgpErr);
			optionList = CreateOptionList(	inContext,
											inOptions,
											recipientSet,
											signingKey,
											convEncryptPassphrase,
											signingBuffer.mMemory,
											signingPasskeySize,
											usePassphrase);
			::HLock(ioDataH);
			pgpErr = PGPEncode(	inContext,
								PGPOInputBuffer(	inContext,
													*ioDataH,
													::GetHandleSize(ioDataH)),
								PGPOAllocatedOutputBuffer(	inContext,
															&buf,
															MAX_PGPSize,
															&bufSize),
								PGPOEventHandler(	inContext,
													PGPEncodeEventHandler,
													cursorRef),
								PGPOSendNullEvents(	inContext,
													500),
								PGPOForYourEyesOnly(inContext, fyeo),
								optionList,
								PGPOLastOption(inContext));
			::HUnlock(ioDataH);
			PGPThrowIfPGPError_(pgpErr);
			
			::PtrToXHand(buf, ioDataH, bufSize);
			PGPFreeData(buf);
								
			result = true;
		}
	}
	
	catch (CPGPException & error) {
		if (! error.IsCancelError()) {
				ShowError(error);
		}
	}

	catch(...) {
		ShowError(CPGPException(kPGPError_UnknownError));
	}
	
	// Cleanup
	if (cursorRef != nil) {
		DisposeAnimatedCursor(cursorRef);
	}
	if (PGPOptionListRefIsValid(optionList)) {
		PGPFreeOptionList(optionList);
	}
	if (PGPKeySetRefIsValid(selectedSet)) {
		PGPFreeKeySet(selectedSet);
	}
	if (PGPKeySetRefIsValid(newKeys)) {
		PGPFreeKeySet(newKeys);
	}
	if (PGPKeySetRefIsValid(recipientSet)) {
		PGPFreeKeySet(recipientSet);
	}
	if (PGPKeySetRefIsValid(defaultKeySet)) {
		PGPFreeKeySet(defaultKeySet);
	}
	if (PGPPrefRefIsValid(prefRef)) {
		PGPClosePrefFile(prefRef);
	}
	if (PGPIORefIsValid(wrapInput)) {
		PGPFreeIO(wrapInput);
	}
	if (PGPIORefIsValid(wrapOutput)) {
		PGPFreeIO(wrapOutput);
	}

	return result;
}



	Boolean
CPGPEncoderDecoder::DecodeHandle(
	PGPContextRef			inContext,
	Handle					inDataH,
	PGPKeySetRef			inKeySet,
	SDecodeSectionHandle &	outDecodeSectionsHandle,
	Boolean *				outFYEO,
	Boolean					inUseCache)
{
	Boolean				result = false;

	if (::GetHandleSize(inDataH) > 0) {
		OSErr				err;
		PGPError			pgpErr;
		PGPUInt32			numPublicKeys = 1;
		PGPUInt32			numPrivateKeys = 1;
		SDecodeData			decodeData = {inUseCache, false, nil, nil, inKeySet};

		// Check for missing keys
		try {
			if (sCheckForMissingKeys) {
				err = CountKeysInDefaultKeyring(	inContext,
													&numPublicKeys,
													&numPrivateKeys);
				PGPThrowIfOSError_(err);
				
				if (numPublicKeys == 0) {
					WarningAlert(	kWACautionAlertType,
									kWAOKStyle,
									STRx_PGPEncoderDecoder,
									kNoPublicKeysID);
				} else if (numPrivateKeys == 0) {
					WarningAlert(	kWACautionAlertType,
									kWAOKStyle,
									STRx_PGPEncoderDecoder,
									kNoPrivateKeysID);
				} else {
					sCheckForMissingKeys = false;
				}
			}
			
			// Get the cursor
			err = Get1AnimatedCursor(acur_BeachBall, &decodeData.cursor);
			PGPThrowIfOSError_(err);
			
			// Create Handle for storing decoded data
			decodeData.handle = ::NewHandle(0);
			PGPThrowIfMemFail_(decodeData.handle);
			
			::HLock(inDataH);
			::HLock(decodeData.handle);
			pgpErr = PGPDecode(	inContext,
								PGPOInputBuffer(	inContext,
													*inDataH,
													::GetHandleSize(inDataH)),
								PGPOEventHandler(	inContext,
													PGPDecodeEventHandler,
													&decodeData),
								PGPOSendNullEvents(	inContext,
													500),
								PGPOKeySetRef(	inContext,
												inKeySet),
								PGPOSendEventIfKeyFound(inContext, true),
								PGPOPassThroughIfUnrecognized(	inContext,
																true),
								PGPOLastOption(inContext));
			::HUnlock(decodeData.handle);
			::HUnlock(inDataH);
			PGPThrowIfPGPError_(pgpErr);
			
			outDecodeSectionsHandle =
					(SDecodeSectionHandle) decodeData.handle;
			*outFYEO = decodeData.FYEO;
			decodeData.handle = nil;
			result = true;
		}
		
		catch (CPGPException & error) {
			if (! error.IsCancelError()) {
					ShowError(error);
			}
		}

		catch(...) {
			ShowError(CPGPException(kPGPError_UnknownError));
		}
		
		// Cleanup
		if (decodeData.cursor != nil) {
			DisposeAnimatedCursor(decodeData.cursor);
		}
		if (decodeData.handle != nil) {
			::DisposeHandle(decodeData.handle);
		}
	}
	
	return result;
}



	void
CPGPEncoderDecoder::FreeDecodeSection(
	SDecodeSection *	inSection)
{
	if (inSection->buf != nil) {
		PGPFreeData(inSection->buf);
	}
	if (PGPKeySetRefIsValid(inSection->addKeysSet)) {
		PGPFreeKeySet(inSection->addKeysSet);
	}
}



	PGPOptionListRef
CPGPEncoderDecoder::CreateOptionList(
	PGPContextRef	inContext,
	EEncodeOptions	inOptions,
	PGPKeySetRef	inRecipients,
	PGPKeyRef		inSigningKey,
	char *			inConvEncryptPassphrase,
	void *			inPassBuffer,
	PGPSize			inPassKeySize,
	Boolean			inUsePassphrase)
{
	PGPOptionListRef	optionList;
	CString				commentString;
	PGPPrefRef			prefRef = kInvalidPGPPrefRef;
	PGPError			pgpErr;
	
	try {
		// Add admin comment
#if PGP_BUSINESS_SECURITY
		pgpErr = PGPOpenAdminPrefs(	PGPGetContextMemoryMgr(inContext),
									&prefRef );

		if (pgpErr != kPGPError_NoErr) {
			WarningAlert(	kWAStopAlertType,
							kWAOKStyle,
							STRx_PGPEncoderDecoderErrorStrings,
							kErrorStringID,
							CString(CStringFromInt(pgpErr)),
							CString(CStringFromSTRx(	STRx_PGPEncoderDecoderErrorStrings,
														kErrorStringAdminPrefsNotFoundID)));
			PGPThrowOSError_(userCanceledErr);
		}
		PGPGetPrefStringBuffer(	prefRef,
								kPGPPrefComments,
								commentString.GetMinStorage(),
								commentString);
		PGPClosePrefFile(prefRef);
#endif
		// We need the client prefs for the rest of the options
		pgpErr = PGPOpenClientPrefs(	PGPGetContextMemoryMgr(inContext),
										&prefRef);
		PGPThrowIfPGPError_(pgpErr);
		
		// If there was no admin comment, add the client comment
		if (commentString.GetLength() == 0) {
			PGPGetPrefStringBuffer(	prefRef,
									kPGPPrefComment,
									commentString.GetMinStorage(),
									commentString);
		}

		pgpErr = PGPNewOptionList(inContext, & optionList);
		PGPThrowIfPGPError_(pgpErr);
		if (commentString.GetLength() != 0) {
			PGPAppendOptionList(	optionList,
									PGPOCommentString(	inContext,
														commentString),
									PGPOLastOption(inContext));
		}
								
		// Add version string
		PGPAppendOptionList(	optionList,
								PGPOVersionString(	inContext,
													pgpVersionHeaderString),
								PGPOLastOption(inContext));
		
		// Add rest of options
		if ((inOptions & encodeOptions_Encrypt)
		&& (! (inOptions & encodeOptions_ConvEncrypt))) {
			PGPAppendOptionList(	optionList,
									PGPOEncryptToKeySet(	inContext,
															inRecipients),
									PGPOLastOption(inContext));

			PGPCipherAlgorithm *	allowedAlgorithms = nil;
			PGPSize					bufSize;
			
			PGPGetPrefData(	prefRef,
							kPGPPrefAllowedAlgorithmsList,
							&bufSize,
							&allowedAlgorithms);
			if (allowedAlgorithms != nil) {
				PGPAppendOptionList(
					optionList,
					PGPOPreferredAlgorithms(	inContext,
												allowedAlgorithms,
					bufSize / sizeof( PGPCipherAlgorithm ) ),
					PGPOLastOption(inContext));
				PGPDisposePrefData(	prefRef,
									allowedAlgorithms);
			}
		}
		if (inOptions & encodeOptions_ConvEncrypt) {
			PGPUInt32	preferredAlgorithm;
			
			pgpErr = PGPGetPrefNumber(	prefRef,
										kPGPPrefPreferredAlgorithm,
										&preferredAlgorithm);
			if (pgpErr != kPGPError_NoErr) {
				preferredAlgorithm = kPGPCipherAlgorithm_CAST5;
			}
			PGPAppendOptionList(	
							optionList,
							PGPOConventionalEncrypt(inContext, 
								PGPOPassphrase(inContext,
										inConvEncryptPassphrase),
								PGPOLastOption(inContext)),
							PGPOCipherAlgorithm(	inContext,
								(PGPCipherAlgorithm) preferredAlgorithm),
							PGPOLastOption(inContext));
		}
		if (inOptions & encodeOptions_Sign) {
			PGPAppendOptionList(
				optionList,
				PGPOSignWithKey(	inContext,
									inSigningKey,
									(inUsePassphrase) ? 
										PGPOPassphrase(
											inContext,
											(char *) inPassBuffer) :
										PGPOPasskeyBuffer(	inContext,
															inPassBuffer,
															inPassKeySize),
									PGPOLastOption(inContext)),
				PGPOLastOption(inContext));
		}
		if (inOptions & encodeOptions_TextOutput) {
			PGPAppendOptionList(	optionList,
									PGPOArmorOutput(inContext, true),
									PGPOLastOption(inContext));
		}
		if (inOptions & encodeOptions_TreatInputAsText) {
			PGPAppendOptionList(	optionList,
									PGPODataIsASCII(inContext, true),
									PGPOLastOption(inContext));
		}
		if (inOptions & encodeOptions_ClearSign) {
			PGPAppendOptionList(	optionList,
									PGPOClearSign(inContext, true),
									PGPOLastOption(inContext));
		}
		PGPClosePrefFile(prefRef);
	}
	
	catch (...) {
		if (PGPPrefRefIsValid(prefRef)) {
			PGPClosePrefFile(prefRef);
		}
		throw;
	}
	
	return optionList;
}



	void
CPGPEncoderDecoder::SetPassbufferCache(
	Boolean			inSigning,
	PGPKeyRef	 	inKey,
	PGPBoolean		inUsePassphrase,
	void *			inBuffer,
	PGPSize			inPasskeySize)
{
	PGPPrefRef				prefRef = kInvalidPGPPrefRef;
	
	try {
		CSignPassphraseCache *	theCache;
		PGPBoolean				cacheEnabled;
		PGPUInt32				cacheDuration = 0;
		PGPError				pgpErr;
		
		// Get the cache values
		pgpErr = PGPOpenClientPrefs(PGPGetContextMemoryMgr(PGPGetKeyContext(inKey)),
									&prefRef);
		PGPThrowIfPGPError_(pgpErr);
		
		if (inSigning) {
			theCache = sSigningPassphraseCache;
			pgpErr = PGPGetPrefBoolean(	prefRef,
										kPGPPrefSignCacheEnable,
										&cacheEnabled);
			PGPThrowIfPGPError_(pgpErr);
			if (cacheEnabled) {
				pgpErr = PGPGetPrefNumber(	prefRef,
											kPGPPrefSignCacheSeconds,
											&cacheDuration);
				PGPThrowIfPGPError_(pgpErr);
			}
		} else {
			theCache = sDecryptPassphraseCache;
			pgpErr = PGPGetPrefBoolean(	prefRef,
										kPGPPrefDecryptCacheEnable,
										&cacheEnabled);
			PGPThrowIfPGPError_(pgpErr);
			if (cacheEnabled) {
				pgpErr = PGPGetPrefNumber(	prefRef,
											kPGPPrefDecryptCacheSeconds,
											&cacheDuration);
				PGPThrowIfPGPError_(pgpErr);
			}
		}
		
		theCache->SetCacheSeconds(cacheDuration);
		if (inUsePassphrase) {
			theCache->RememberPassphrase((char *) inBuffer, inKey);
		} else {
			theCache->RememberPasskey(inBuffer, inPasskeySize, inKey);
		}
	}
	
	catch (...) {
	}

	if (PGPPrefRefIsValid(prefRef)) {
		PGPClosePrefFile(prefRef);
	}
}


	void
CPGPEncoderDecoder::ShowError(
	CPGPException &	inErr)
{
	WarningAlert(	kWAStopAlertType,
					kWAOKStyle,
					STRx_PGPEncoderDecoderErrorStrings,
					kErrorStringID,
					CString(CStringFromInt(inErr.GetError())),
					CString(inErr.what()));
}



	PGPError
CPGPEncoderDecoder::PGPDecodeEventHandler(
	PGPContextRef	inContext,
	PGPEvent *		inEventP,
	PGPUserValue	inUserValue)
{
	PGPError				result = kPGPError_NoErr;
	static PGPInt32			sSection;
	static Boolean			sAllocated;
	static PGPUInt32		sKeyIDCount = 0;
	static PGPKeySetRef		sRecipientKeySet = kInvalidPGPKeySetRef;
	static PGPKeyID *		sKeyIDList = nil;
	char *					tempPassphrase = nil;
	SDecodeSectionHandle	decodeSectionH = 
							(SDecodeSectionHandle) ((SDecodeData *)
									inUserValue)->handle;
	
	try {
		AnimateCursor(((SDecodeData *) inUserValue)->cursor);
		
		switch (inEventP->type) {
			case kPGPEvent_InitialEvent:
			{
				sSection = -1;
				sAllocated = false;
			}
			break;
			
			
			case kPGPEvent_EndLexEvent:
			{
				sAllocated = false;
				if (PGPKeySetRefIsValid(sRecipientKeySet)) {
					PGPFreeKeySet(sRecipientKeySet);
					sRecipientKeySet = kInvalidPGPKeySetRef;
				}
				if (IsntNull(sKeyIDList)) {
					delete[] sKeyIDList;
					sKeyIDList = nil;
				}
			}
			break;
			
			
			case kPGPEvent_AnalyzeEvent:
			{
				if (inEventP->data.analyzeData.sectionType ==
				kPGPAnalyze_Encrypted) {
					if (! sAllocated) {
						sSection++;
						// Unlock long enough to grow the handle
						::HUnlock((Handle) decodeSectionH);
						::SetHandleSize(	(Handle) decodeSectionH,
											(sSection + 1)
												* sizeof(SDecodeSection));
						::HLock((Handle) decodeSectionH);
						(*decodeSectionH)[sSection].buf = nil;
						(*decodeSectionH)[sSection].addKeysSet =
							kInvalidPGPKeySetRef;
						(*decodeSectionH)[sSection].sigDataValid = false;
						sAllocated = true;
					}
					(*decodeSectionH)[sSection].encrypted = true;
				}
			}
			break;
			
			
			case kPGPEvent_RecipientsEvent:
			{
				PGPUInt32	numFoundKeys = 0;
				
				if (PGPKeySetRefIsValid(inEventP->
				data.recipientsData.recipientSet)) {
					result = PGPIncKeySetRefCount(inEventP->
									data.recipientsData.recipientSet);
					if (IsntPGPError(result)) {
						sRecipientKeySet = inEventP->
							data.recipientsData.recipientSet;
						if (IsntNull(inEventP->
						data.recipientsData.keyIDArray)) {
							sKeyIDCount = inEventP->
								data.recipientsData.keyCount;
							sKeyIDList = new PGPKeyID[sKeyIDCount];
							if (sKeyIDList == nil) {
								result = kPGPError_OutOfMemory;
							} else {
								pgpCopyMemory(
									inEventP->data.recipientsData.keyIDArray,
									sKeyIDList,
									sKeyIDCount * sizeof(PGPKeyID));
							}
						}
					}
				}
			}
			break;
			
			
			case kPGPEvent_PassphraseEvent:
			{
				PGPKeyRef			keyRef = kInvalidPGPKeyRef;
				CSecureMemory		passbuffer(PGPGetContextMemoryMgr(
										inContext), 256);
				PGPSize				passkeySize;
				PGPBoolean			usePassphrase;

				if (! ((SDecodeData *) inUserValue)->useCache) {
					sDecryptPassphraseCache->Forget();
				}
				
				if (inEventP->data.passphraseData.fConventional) {
					usePassphrase = true;
					result = PGPConventionalDecryptionPassphraseDialog(
									inContext,
									PGPOUIOutputPassphrase(
										inContext,
										&tempPassphrase),
									PGPOLastOption(inContext));
					PGPThrowIfPGPError_(result);
					strcpy(	(char *) passbuffer.mMemory,
							tempPassphrase);
					PGPFreeData(tempPassphrase);
					tempPassphrase = nil;
				} else {
					// Get passphrase
					if (! sDecryptPassphraseCache->GetPassphraseOrPasskey(
						inEventP->data.passphraseData.keyset,
						passbuffer.mMemory, &usePassphrase, &passkeySize,
						&keyRef)) {
						PGPKeySetRef	newKeys = kInvalidPGPKeySetRef;
						
						usePassphrase = true;
						result = PGPClientDecryptionPassphraseDialog(
										inContext,
										sTLSContext,
										nil,
										sRecipientKeySet,
										sKeyIDList,
										sKeyIDCount,
										kInvalidPGPOptionListRef,
										&tempPassphrase,
										&keyRef,
										&newKeys);
						if (result != kPGPError_KeyUnusableForDecryption) {
							PGPThrowIfPGPError_(result);
							if (PGPKeySetRefIsValid(newKeys)) {
								if (! sAllocated) {
									sSection++;
									// Unlock long enough to grow the handle
									::HUnlock((Handle) decodeSectionH);
									::SetHandleSize(
										(Handle) decodeSectionH,
										(sSection + 1) * sizeof(
											SDecodeSection));
									::HLock((Handle) decodeSectionH);
									(*decodeSectionH)[sSection].buf = nil;
									(*decodeSectionH)[sSection].addKeysSet =
										kInvalidPGPKeySetRef;
									(*decodeSectionH)[sSection].sigDataValid =
										false;
									(*decodeSectionH)[sSection].encrypted =
										false;
									sAllocated = true;
								}
								if (PGPKeySetRefIsValid(
								(*decodeSectionH)[sSection].addKeysSet)) {
									result = PGPAddKeys(
												(*decodeSectionH)[sSection].
													addKeysSet,
												newKeys);
									PGPThrowIfPGPError_(result);
									result = PGPCommitKeyRingChanges(
												newKeys);
									PGPThrowIfPGPError_(result);
									PGPFreeKeySet((*decodeSectionH)[sSection].
										addKeysSet);
								}
								(*decodeSectionH)[sSection].addKeysSet = 
									newKeys;
							}
							strcpy(	(char *) passbuffer.mMemory,
									tempPassphrase);
							PGPFreeData(tempPassphrase);
							tempPassphrase = nil;
						} else {
							usePassphrase = false;
							result = PGPReconstitutionDialog(
								keyRef,
								((SDecodeData *) inUserValue)->defaultKeys,
								kInvalidPGPtlsContextRef,
								(PGPByte **) &tempPassphrase,
								&passkeySize);
							PGPThrowIfPGPError_(result);
							pgpCopyMemory(	tempPassphrase,
											passbuffer.mMemory,
											passkeySize);
							PGPFreeData(tempPassphrase);
							tempPassphrase = nil;
						}
						SetPassbufferCache(	false,
											keyRef,
											usePassphrase,
											passbuffer.mMemory,
											passkeySize);
					}
				}
					
				// Add the passphrase to the job
				result = PGPAddJobOptions(
							inEventP->job,
							(usePassphrase) ?
								PGPOPassphrase( inContext,
												(char *) passbuffer.mMemory) :
								PGPOPasskeyBuffer(	inContext,
													passbuffer.mMemory,
													passkeySize),
							PGPOLastOption(inContext));
				PGPThrowIfPGPError_(result);
			}
			break;
			
			
			case kPGPEvent_SignatureEvent:
			{
				if (! sAllocated) {
					sSection++;
					// Unlock long enough to grow the handle
					::HUnlock((Handle) decodeSectionH);
					::SetHandleSize(	(Handle) decodeSectionH,
										(sSection + 1)
											* sizeof(SDecodeSection));
					::HLock((Handle) decodeSectionH);
					(*decodeSectionH)[sSection].buf = nil;
					(*decodeSectionH)[sSection].addKeysSet =
						kInvalidPGPKeySetRef;
					(*decodeSectionH)[sSection].sigDataValid = false;
					(*decodeSectionH)[sSection].encrypted = false;
					sAllocated = true;
				}

				PGPBoolean		addedKeySetToJob;
				PGPKeySetRef	newKeySet;
				
				result = PGPSignerKeyLookup(	inContext,
												inEventP,
												kInvalidPGPKeySetRef,
												&addedKeySetToJob,
												&newKeySet);
				PGPThrowIfPGPError_(result);
				if (addedKeySetToJob) {
					if (PGPKeySetRefIsValid((*decodeSectionH)[sSection].
					addKeysSet)) {
						result = PGPAddKeys(
									(*decodeSectionH)[sSection].addKeysSet,
									newKeySet);
						PGPThrowIfPGPError_(result);
						result = PGPCommitKeyRingChanges(newKeySet);
						PGPThrowIfPGPError_(result);
						PGPFreeKeySet((*decodeSectionH)[sSection].addKeysSet);
					}
					(*decodeSectionH)[sSection].addKeysSet = newKeySet;
				} else {
					(*decodeSectionH)[sSection].sigData = inEventP->
							data.signatureData;
					(*decodeSectionH)[sSection].sigDataValid = true;
				}
			}
			break;
			
			
			case kPGPEvent_OutputEvent:
			{
				if (! sAllocated) {
					sSection++;
					// Unlock long enough to grow the handle
					::HUnlock((Handle) decodeSectionH);
					::SetHandleSize(	(Handle) decodeSectionH,
										(sSection + 1)
											* sizeof(SDecodeSection));
					::HLock((Handle) decodeSectionH);
					(*decodeSectionH)[sSection].buf = nil;
					(*decodeSectionH)[sSection].addKeysSet =
						kInvalidPGPKeySetRef;
					(*decodeSectionH)[sSection].sigDataValid = false;
					(*decodeSectionH)[sSection].encrypted = false;
					sAllocated = true;
				}
				if (inEventP->data.outputData.forYourEyesOnly) {
					((SDecodeData *) inUserValue)->FYEO = true;
				}
				result = PGPAddJobOptions(	inEventP->job,
											PGPOAllocatedOutputBuffer(
												inContext,
												&(*decodeSectionH)[sSection].
														buf,
												MAX_PGPSize,
												&(*decodeSectionH)[sSection].
														size),
											PGPOLastOption(inContext));
			}
			break;
			
			
			case kPGPEvent_KeyFoundEvent:
			{
				if (! sAllocated) {
					sSection++;
					// Unlock long enough to grow the handle
					::HUnlock((Handle) decodeSectionH);
					::SetHandleSize(	(Handle) decodeSectionH,
										(sSection + 1) * sizeof(SDecodeSection));
					::HLock((Handle) decodeSectionH);
					(*decodeSectionH)[sSection].buf = nil;
					(*decodeSectionH)[sSection].addKeysSet =
						kInvalidPGPKeySetRef;
					(*decodeSectionH)[sSection].sigDataValid = false;
					(*decodeSectionH)[sSection].encrypted = false;
					sAllocated = true;
				}

				if (PGPKeySetRefIsValid(
				(*decodeSectionH)[sSection].addKeysSet)) {
					result = PGPAddKeys(
								inEventP->data.keyFoundData.keySet,
								(*decodeSectionH)[sSection].addKeysSet);
					PGPThrowIfPGPError_(result);
					result = PGPCommitKeyRingChanges(
								(*decodeSectionH)[sSection].addKeysSet);
					PGPThrowIfPGPError_(result);
				} else {
					result = PGPIncKeySetRefCount(
								inEventP->data.keyFoundData.keySet);
					PGPThrowIfPGPError_(result);
					(*decodeSectionH)[sSection].addKeysSet = inEventP->
						data.keyFoundData.keySet;
				}
			}
			break;
		}
	}
	
	catch (...) {
		sAllocated = false;
		if (tempPassphrase != nil) {
			PGPFreeData(tempPassphrase);
		}
		if (PGPKeySetRefIsValid(sRecipientKeySet)) {
			PGPFreeKeySet(sRecipientKeySet);
			sRecipientKeySet = kInvalidPGPKeySetRef;
		}
		if (IsntNull(sKeyIDList)) {
			delete[] sKeyIDList;
			sKeyIDList = nil;
		}
	}
	
	return result;
}
