/*____________________________________________________________________________
	Copyright (C) 1997 Network Associates Inc. and affiliated companies.
	All rights reserved.

	$Id: CGroupsWindow.cp,v 1.43 1999/03/10 02:34:05 heller Exp $
____________________________________________________________________________*/

#include <Fonts.h>

#include <UGAColorRamp.h>
#include <UDrawingState.h>
#include <UDrawingUtils.h>
#include <UGraphicUtils.h>
#include <LTableMultiGeometry.h>
#include <LTableMultiSelector.h>
#include <LTableArrayStorage.h>
#include <LEditText.h>
#include <LDragTask.h>
#include <LDropFlag.h>
#include <PP_Messages.h>
#include <PP_KeyCodes.h>

#include "CGroupsWindow.h"

#include "CKeyTable.h"
#include "CKeyView.h"
#include "CPGPKeys.h"
#include "CKeyInfoWindow.h"
#include "ResourceConstants.h"

#include "CWarningAlert.h"
#include "MacFiles.h"
#include "MacStrings.h"
#include "CPGPStDialogHandler.h"

#include "pgpFileSpec.h"
#include "pgpMem.h"
#include "pgpKeys.h"
#include "pgpUtilities.h"
#include "pgpClientPrefs.h"
#include "pgpOpenPrefs.h"
#include "pgpClientLib.h"
#include "pgpSDKPrefs.h"


const Int16			kRowHeight					= 	18;
const ResIDT		kNewGroupDialogID			= 	155;
const Int16			kUserIDStringLength			= 	256;
const FlavorType	kGroupIDDragFlavor			= 	'gkID';
const FlavorType	kGroupItemDragFlavor		=	'giID';
const ResIDT		kGroupColumnTitleStringsID	= 	1021;
const Int16			kNameColumnWidth			=	286;
const Int16			kValidityColumnWidth		=	51;


enum	{
			kStringListID		= 1020,
			kBadGroupInfoStringID = 1,
			kDeleteConfirmStringID,
			kUnknownKeyStringID,
			kExpandSelectionID,
			kCollapseSelectionID,
			kExpandAllID,
			kCollapseAllID
		};	


#pragma mark --- Window Functions ---

CGroupsWindow::CGroupsWindow()
{
}

CGroupsWindow::CGroupsWindow(LStream *inStream)
	:	LWindow(inStream)
{
	mGroupsRef		= kInvalidPGPGroupSetRef;
	mGroupsFile		= kInvalidPGPFileSpecRef;
}

CGroupsWindow::~CGroupsWindow()
{
	if( PGPGroupSetRefIsValid(mGroupsRef) &&
		PGPFileSpecRefIsValid(mGroupsFile))
	{
		if(PGPGroupSetNeedsCommit(mGroupsRef))
			PGPSaveGroupSetToFile(mGroupsRef, mGroupsFile);
	}
	if(PGPGroupSetRefIsValid(mGroupsRef))
		PGPFreeGroupSet(mGroupsRef);
	if(PGPFileSpecRefIsValid(mGroupsFile))
		PGPFreeFileSpec(mGroupsFile);
}

	void
CGroupsWindow::FinishCreateSelf()
{
	LWindow::FinishCreateSelf();
	mGroupsTable = (CGroupsTable *)FindPaneByID(CGroupsTable::class_ID);
	{
		PGPError		err;
		PGPFileSpecRef	groupSpec;
		
		err = PGPsdkPrefGetFileSpec( gPGPContext, kPGPsdkPref_GroupsFile,
					&groupSpec );
		if( IsntPGPError( err ) )
		{
			mGroupsFile = groupSpec;
			err = PGPNewGroupSetFromFile(gPGPContext, groupSpec,
										&mGroupsRef);

			if ( IsntPGPError( err ) )
			{
				err	= PGPSortGroupSetStd( mGroupsRef,
						CPGPKeys::TheApp()->GetKeySet() );
			}
		}
		if( IsErr( err ) && !PGPGroupSetRefIsValid( mGroupsRef ) )
		{
			// We couldn't open the file, let's just make a group set
			err = PGPNewGroupSet(gPGPContext, &mGroupsRef);
			pgpAssertNoErr(err);
		}
		if(IsErr(err))
			ReportPGPError(err);
		pgpAssertAddrValid(mGroupsRef, PGPGroupSetRef);
		mGroupsTable->SetGroupSet(mGroupsRef, mGroupsFile);
	}
	SwitchTarget(mGroupsTable);
	Show();
}

	void
CGroupsWindow::AttemptClose()
{
	BroadcastMessage(kGroupsWindowClosed, NULL);
	LWindow::AttemptClose();
}

	void
CGroupsWindow::ListenToMessage(MessageT inMessage, void *ioParam)
{
	switch(inMessage)
	{
	}
}

	void
CGroupsWindow::InvalidateCaches()
{
	mGroupsTable->InvalidateCaches();
}


#pragma mark --- Table Functions ---

CGroupsTable::CGroupsTable(
	LStream	*inStream)
		:	CPGPHierarchyTable(inStream),
			LDragAndDrop(UQDGlobals::GetCurrentPort(), this)
{
	mTableGeometry	= new LTableMultiGeometry(this,	100, kRowHeight);
	mTableSelector	= new LTableMultiSelector(this);
	mTableStorage	= new LTableArrayStorage(this, (ulong)0);
	mGroupsRef	= kInvalidPGPGroupSetRef;
	mGroupsFile	= kInvalidPGPFileSpecRef;
	mSendingDrag = FALSE;
}

void
CGroupsTable::FinishCreateSelf()
{
	RemoveAllCols(false);
	InsertCols(kMaxColumnTypes, 1, NULL, 0, false);
	SetColWidth(kNameColumnWidth, 1, 1);
	SetColWidth(kValidityColumnWidth, 2, 2);
	SetColWidth(300, 3, 3);
}

CGroupsTable::~CGroupsTable()
{
}

	void
CGroupsTable::SetGroupSet(PGPGroupSetRef groupRef, PGPFileSpecRef groupsFile)
{
	mGroupsRef	= groupRef;
	mGroupsFile	= groupsFile;
	ResyncTable(TRUE, FALSE);
	RedrawTable();
}

	void
CGroupsTable::InvalidateCaches()
{
	RebuildTable();
}

	void
CGroupsTable::RebuildTable()
{
	RemoveAllRows(false);
	ResortGroups();
	ResyncTable(TRUE, FALSE);
	RedrawTable();
}

	void
CGroupsTable::ResyncTable(Boolean collapse, Boolean onlyUnmarked)
{
	PGPUInt32			numGroups,
						groupIndex,
						itemIndex,
						numKeyEntries,
						numTotalEntries;
	PGPGroupID			groupID;
	PGPError			err;
	GroupTableEntry		groupEntry;
	PGPGroupInfo		groupInfo;
	PGPGroupItem		groupItem;
	TableIndexT			lastGroupRow,
						lastRow,
						itemSubRow;
	
	if(!onlyUnmarked)
		RemoveAllRows(false);
	lastRow = lastGroupRow = 0;
	err = PGPCountGroupsInSet(mGroupsRef, &numGroups);
	pgpAssertNoErr(err);
	for(groupIndex = 0; groupIndex < numGroups ; groupIndex++)
	{
		err = PGPGetIndGroupID(mGroupsRef, groupIndex, &groupID);
		pgpAssertNoErr(err);
		err = PGPGetGroupInfo(mGroupsRef, groupID, &groupInfo);
		pgpAssertNoErr(err);
		++lastRow;
		if(groupInfo.userValue == 0 || !onlyUnmarked)
		{
			groupEntry.isTopGroup = TRUE;
			groupEntry.u.groupID = groupID;
			lastGroupRow = InsertSiblingRows(1, lastGroupRow, &groupEntry,
								sizeof(GroupTableEntry),
								TRUE, TRUE);
			PGPSetGroupUserValue(mGroupsRef, groupID, (PGPUserValue)1);
		}
		else if(!lastGroupRow)
			lastGroupRow = 1;
		if(lastRow > 1)
			lastGroupRow = lastRow;
		err = PGPCountGroupItems(mGroupsRef, groupID, FALSE,
							&numKeyEntries, &numTotalEntries);
		pgpAssertNoErr(err);
		if(numTotalEntries > 0)
		{
			for(itemIndex = 0; itemIndex < numTotalEntries; itemIndex++)
			{
				err = PGPGetIndGroupItem(mGroupsRef, groupID,
											itemIndex, &groupItem);
				pgpAssertNoErr(err);
				lastRow++;
				if(groupItem.userValue == 0 || !onlyUnmarked)
				{
					groupEntry.isTopGroup = FALSE;
					groupEntry.u.item.ownerGroupID = groupID;
					groupEntry.u.item.cachedKey = kInvalidPGPKeyRef;
					pgpCopyMemory(&groupItem, &groupEntry.u.item.groupItem,
									sizeof(PGPGroupItem));
									
					if(itemIndex == 0)
					{
						InsertChildRows(1, lastGroupRow, &groupEntry,
											sizeof(GroupTableEntry),
											FALSE, TRUE);
						itemSubRow = lastGroupRow + 1;
					}
					else
						itemSubRow = InsertSiblingRows(1, itemSubRow,
										&groupEntry, sizeof(GroupTableEntry),
										FALSE, TRUE);
					PGPSetIndGroupItemUserValue(mGroupsRef, groupID,
												itemIndex, (PGPUserValue)1);
				}
				else
				{
					if(itemIndex == 0)
						itemSubRow = lastGroupRow + 1;
					else
						itemSubRow = lastRow;
				}
			}
		}
	}
	if(collapse)
		CollapseGroups(FALSE);
}

	void
CGroupsTable::RedrawTable()
{
	Rect frame;
	
	CalcPortFrameRect(frame);
	InvalPortRect(&frame);
}

	void
CGroupsTable::CollapseGroups(Boolean all)
{
	TableIndexT	rows,
				cols,
				rowIndex;
	STableCell	cell(1, 1);
	Boolean		selectedOnly;
	Rect		frame;
	
	if(all)
		selectedOnly = FALSE;
	else
		selectedOnly = (mTableSelector->GetFirstSelectedRow() > 0);
	GetWideOpenTableSize(rows, cols);
	for(rowIndex = rows;rowIndex > 0;rowIndex--)
	{
		cell.row = GetExposedIndex(rowIndex);
		if(!selectedOnly || CellIsSelected(cell))
			if(IsCollapsable(rowIndex))
				DeepCollapseRow(rowIndex);
	}
	cell.row = cell.col = 1;
	ScrollCellIntoFrame(cell);
	
	CalcPortFrameRect(frame);
	InvalPortRect(&frame);
}

	void
CGroupsTable::ExpandGroups()
{
	TableIndexT	rows,
				cols,
				rowIndex;
	STableCell	cell(1, 1);
	Boolean		selectedOnly;
	
	selectedOnly = (mTableSelector->GetFirstSelectedRow() > 0);
	GetWideOpenTableSize(rows, cols);
	for(rowIndex = rows;rowIndex > 0;rowIndex--)
	{
		cell.row = GetExposedIndex(rowIndex);
		if(!selectedOnly || CellIsSelected(cell))
			if(IsCollapsable(rowIndex))
				DeepExpandRow(rowIndex);
	}
	RedrawTable();
}

	void
CGroupsTable::DrawCell(
	const STableCell	&inCell,
	const Rect			&inLocalRect)
{
	TableIndexT	woRow = mCollapsableTree->GetWideOpenIndex(inCell.row);
	Rect		frame;

	CalcLocalFrameRect( frame );
	{
		StDeviceLoop	devLoop(frame);		
		SInt16			depth;

		while(devLoop.NextDepth(depth))
		{
			STableCell			woCell(woRow, 1);
			Uint32				dataSize = sizeof(GroupTableEntry);
			GroupTableEntry		tableEntry;
			
			GetCellData(woCell, &tableEntry, dataSize);
			::RGBBackColor(&UGAColorRamp::GetColor(colorRamp_Gray1));
			::RGBForeColor(&UGAColorRamp::GetColor(colorRamp_Black));
			::TextMode(srcOr);
			switch(inCell.col-1)
			{
				case kAddressColumnID:
					DrawAddressColumnCell(&tableEntry, inLocalRect, depth);
					SetCellData(woCell, &tableEntry, dataSize);
					DrawDropFlag(inCell, woRow);
					break;
				case kValidityColumnID:
					DrawValidityColumnCell(&tableEntry, inLocalRect, depth);
					break;
				case kDescriptionColumnID:
					DrawDescriptionColumnCell(&tableEntry, inLocalRect, depth);
					break;
			}
			if(depth >= 8)
				RefreshBorder(inLocalRect);
		}
	}
}

	void
CGroupsTable::CacheKey(GroupTableEntry *tableEntry)
{
	PGPError	err;
	
	if(!PGPKeyRefIsValid(tableEntry->u.item.cachedKey) &&
		tableEntry->u.item.groupItem.type == kPGPGroupItem_KeyID)
	{
		err = PGPGetKeyByKeyID(CPGPKeys::TheApp()->GetKeySet(),
						&tableEntry->u.item.groupItem.u.key.keyID,
						tableEntry->u.item.groupItem.u.key.algorithm,
						&tableEntry->u.item.cachedKey);
	}
}

	void
CGroupsTable::DrawAddressColumnCell(
	GroupTableEntry		*gtr,
	Rect				cellRect,
	Int16				depth)
{
	PGPError		err = kPGPError_NoErr;
	Rect			sortColumnRect;
	Str255			pstr;
	ResIDT			iconID;
	Int16			slop;
	PGPGroupInfo	groupInfo;

	// Draw Background
	if(depth >= 8)
	{
		// Draw MacOS8-like background for cells, selected column
		::RGBForeColor(&UGAColorRamp::GetColor(colorRamp_Gray1));
		::PaintRect(&cellRect);
		::RGBForeColor(&UGAColorRamp::GetColor(colorRamp_Gray2));
		sortColumnRect = cellRect;
		sortColumnRect.left = cellRect.left + kDropFlagSlop;
		sortColumnRect.right = cellRect.right;
		::PaintRect(&sortColumnRect);
		::RGBForeColor(&UGAColorRamp::GetColor(colorRamp_Black));
	}
	else
	{
		::RGBForeColor(&UGAColorRamp::GetColor(colorRamp_Black));
		::RGBBackColor(&UGAColorRamp::GetColor(colorRamp_White));
		::EraseRect(&cellRect);
	}
	// Draw Info
	if(gtr->isTopGroup)
	{
		iconID = kDHGroupIconID;
		CKeyTable::DrawIcon(iconID, cellRect, 1);
		
		err = PGPGetGroupInfo(mGroupsRef, gtr->u.groupID, &groupInfo);
		pgpAssertNoErr(err);
		CToPString(groupInfo.name, pstr);
		::TextFont(kFontIDGeneva);
		::TextSize(9);
		::TextFace(0);
		::TruncString(	kNameColumnWidth -
						kDropFlagSlop - kLeftIndent -
						kIconWidth - kLeftBorder,
						pstr, truncMiddle);
		::MoveTo(cellRect.left + kDropFlagSlop + kLeftIndent +
						kIconWidth + kLeftBorder, cellRect.bottom -
						kBottomBorder);
		::DrawString(pstr);
	}
	else
	{
		pstr[0] = 0;
		::TextFace(0);
		if(gtr->u.item.groupItem.type == kPGPGroupItem_KeyID)
		{
			char		cstr[kUserIDStringLength];
			PGPSize		len;
			
			CacheKey(gtr);
			if(PGPKeyRefIsValid(gtr->u.item.cachedKey))
			{
				PGPBoolean				expired,
										revoked,
										disabled,
										secret;
				PGPPublicKeyAlgorithm	algorithm;
				PGPKeyRef				key;
				
				key 		= gtr->u.item.cachedKey;
				algorithm 	= (PGPPublicKeyAlgorithm)
									gtr->u.item.groupItem.u.key.algorithm;
				
				err = PGPGetKeyBoolean(	key, kPGPKeyPropIsExpired,
										&expired);
				pgpAssertNoErr(err);
				err = PGPGetKeyBoolean(	key, kPGPKeyPropIsRevoked,
										&revoked);
				pgpAssertNoErr(err);
				err = PGPGetKeyBoolean(	key, kPGPKeyPropIsDisabled,
										&disabled);
				pgpAssertNoErr(err);
				err = PGPGetKeyBoolean(key, kPGPKeyPropIsSecret,
										&secret);
				pgpAssertNoErr(err);

				if( revoked )
				{
					if(algorithm == kPGPPublicKeyAlgorithm_DSA)
						iconID = kDSARevokedKeyID;
					else
						iconID = kRSARevokedKeyID;
				}
				else if( expired )
				{
					if( algorithm == kPGPPublicKeyAlgorithm_DSA )
						iconID = kDSAExpiredKeyID;
					else
						iconID = kRSAExpiredKeyID;
				}
				else if( disabled )
				{
					if( algorithm == kPGPPublicKeyAlgorithm_DSA )
						iconID = kDSADisabledKeyID;
					else
						iconID = kRSADisabledKeyID;
				}
				else if(secret)
				{
					if(algorithm == kPGPPublicKeyAlgorithm_DSA)
						iconID = kDSAKeyPairIconID;
					else
						iconID = kRSAKeyPairIconID;
				}
				else
				{
					if(algorithm == kPGPPublicKeyAlgorithm_DSA)
						iconID = kDSAKeyIconID;
					else
						iconID = kRSAKeyIconID;
				}
				
				len = kUserIDStringLength;
				err = PGPGetPrimaryUserIDNameBuffer( key,
					kUserIDStringLength, cstr, &len);
				if(err != kPGPError_BufferTooSmall)
					pgpAssertNoErr(err);
				CToPString(cstr, pstr);
			}
			else
			{
				char	keyIDString[ kPGPMaxKeyIDStringSize ];
				
				iconID = kDSADisabledKeyID;
				::GetIndString(pstr, kStringListID, kUnknownKeyStringID);

				err = PGPGetKeyIDString( &gtr->u.item.groupItem.u.key.keyID,
									kPGPKeyIDString_Abbreviated, keyIDString);
				pgpAssertNoErr(err);
				if(IsntPGPError(err))
				{
					Str255		keyIDPString;
					
					CToPString(keyIDString, keyIDPString);
					AppendPString(keyIDPString, pstr);
				}
				::TextFace(italic);
			}
		}
		else
		{
			iconID = kDHGroupIconID;
			err = PGPGetGroupInfo(mGroupsRef, gtr->u.item.groupItem.u.group.id,
							&groupInfo);
			pgpAssertNoErr(err);
			if(IsntPGPError(err))
				CToPString(groupInfo.name, pstr);
		}
		CKeyTable::DrawIcon(iconID, cellRect, 2);
		
		::TextFont(kFontIDGeneva);
		::TextSize(9);
		slop = kDropFlagSlop + kLeftIndent + kLevelIndent + kIconWidth +
				kLeftBorder;
		::TruncString(	kNameColumnWidth - slop, pstr, truncMiddle);
		::MoveTo(cellRect.left + slop, cellRect.bottom - kBottomBorder);
		::DrawString(pstr);
	}
	::TextFace(0);
}

	void
CGroupsTable::DrawValidity(Int32 trustValue, Rect cellRect, Int16 depth)
{
	RGBColor	trustColor = { 52223, 52223, 65535 };
	Int16		width;
	PGPBoolean	showMarginal;
	PGPError	err;
	
	const Int16		kMaxValidity 			=	2;
	const Int16 	kBarRightBorder			=	10;
	const Int16	 	kBarLeftBorder			=	4;
	const Int16		kBarHeight				=	10;
	
	err = PGPGetPrefBoolean(gPrefRef,
							kPGPPrefDisplayMarginalValidity, &showMarginal);
	pgpAssertNoErr(err);
	if(!showMarginal)
	{
		ResIDT		iconID;

		if(trustValue > kMaxValidity)
		{
			iconID = kAxiomaticIconID;
		}
		else if(trustValue > 1)
		{
			iconID = kValidKeyIconID;
		}
		else if(trustValue == 1)
		{
			PGPBoolean	marginalIsInvalid;
		
			err = PGPGetPrefBoolean(gPrefRef,
							kPGPPrefMarginalIsInvalid,
							&marginalIsInvalid);
			pgpAssertNoErr(err);
			
			if( marginalIsInvalid )
			{
				iconID = kKeyAttrOffIconID;
			}
			else
			{
				iconID = kValidKeyIconID;
			}
		}
		else
		{
			iconID = kKeyAttrOffIconID;
		}

		CKeyTable::DrawIcon(iconID, cellRect, -2);
	}
	else
	{
		width = kValidityColumnWidth - kBarRightBorder
				- kBarLeftBorder;
		cellRect.left 	+=	kBarLeftBorder;
		cellRect.right	= 	cellRect.left + width;
		cellRect.bottom	=	cellRect.bottom - kBottomBorder;
		cellRect.top	=	cellRect.bottom - kBarHeight;
		
		::RGBForeColor(&UGAColorRamp::GetColor(colorRamp_Black));
		::FrameRect(&cellRect);
		::InsetRect(&cellRect, 1, 1);
		if(trustValue > kMaxValidity)
			trustValue = kMaxValidity;
		if(depth >= 8)
		{
			::RGBForeColor(&trustColor);
			::PaintRect(&cellRect);
		}
		
		cellRect.right -= width * (kMaxValidity - trustValue) /
				kMaxValidity;
		::RGBForeColor(&UGAColorRamp::GetColor(colorRamp_Gray9));
		::PaintRect(&cellRect);
		ApplyForeAndBackColors();
	}
}

	void
CGroupsTable::DrawValidityColumnCell(
	GroupTableEntry		*gtr,
	Rect				cellRect,
	Int16				depth)
{
	PGPError	err;
	PGPValidity	validity	= kPGPValidity_Unknown;
	PGPUInt32	notFound;
	PGPInt32	displayValidity	= 0;
	
	if(gtr->isTopGroup)
	{
		err = PGPGetGroupLowestValidity(mGroupsRef,
						gtr->u.groupID, CPGPKeys::TheApp()->GetKeySet(),
						&validity, &notFound);
		pgpAssertNoErr(err);
	}
	else
	{
		if(gtr->u.item.groupItem.type == kPGPGroupItem_KeyID)
		{
			CacheKey(gtr);
			if(PGPKeyRefIsValid(gtr->u.item.cachedKey))
			{
				err = PGPGetPrimaryUserIDValidity(gtr->u.item.cachedKey,
													&validity);
				pgpAssertNoErr(err);
			}
		}
		else
		{
			err = PGPGetGroupLowestValidity(mGroupsRef,
							gtr->u.item.groupItem.u.group.id,
							CPGPKeys::TheApp()->GetKeySet(),
							&validity, &notFound);
			pgpAssertNoErr(err);
		}
	}
	switch(validity)
	{
		default:
		case kPGPValidity_Unknown:
		case kPGPValidity_Invalid:
			displayValidity = 0;
			break;
		case kPGPValidity_Marginal:
			displayValidity = 1;
			break;
		case kPGPValidity_Complete:
			displayValidity = 2;
			break;
	}
	DrawValidity( displayValidity, cellRect, depth);
}

	void
CGroupsTable::DrawDescriptionColumnCell(
	GroupTableEntry		*gtr,
	Rect				cellRect,
	Int16				depth)
{
	Str255			str;
	PGPError		err;
	PGPGroupInfo	groupInfo;
	
	if(gtr->isTopGroup)
	{
		err = PGPGetGroupInfo(mGroupsRef, gtr->u.groupID, &groupInfo);
		pgpAssertNoErr(err);
		CToPString(groupInfo.description, str);
		::TextFont(kFontIDGeneva);
		::TextSize(9);
		::TextFace(0);
		::MoveTo(	cellRect.left + kLeftIndent,
					cellRect.bottom - kBottomBorder);
		::DrawString(str);
	}
}

	void
CGroupsTable::UpdateValidityColumn()
{
	TableIndexT	numRows, numCols;
	GetTableSize(numRows, numCols);
	
	STableCell	topCell(1, 2);
	STableCell 	botCell(numRows, 2);
	STableCell	cell;
	
	FocusDraw();
	for(cell = botCell; cell.row >= topCell.row; cell.row--) 
	{
		Rect cellRect;
		if(GetLocalCellRect(cell, cellRect))
			DrawCell(cell, cellRect);
	}
}

	void
CGroupsTable::RefreshBorder(Rect cellRect)
{
	Rect frame;
	
	CalcLocalFrameRect(frame);
	::RGBForeColor(&UGAColorRamp::GetColor(colorRamp_White));
	::MoveTo(frame.left, cellRect.bottom - 1);
	::LineTo(frame.right - 1, cellRect.bottom - 1);
	::RGBForeColor(&UGAColorRamp::GetColor(colorRamp_Black));
}

	void
CGroupsTable::ShowInfo()
{
	TableIndexT	numRows, numCols;
	GetTableSize(numRows, numCols);
	
	STableCell	topCell(1, 1);
	STableCell 	botCell(numRows, 1);
	STableCell	cell;
	PGPError		err;
			
	for(cell = botCell; cell.row >= topCell.row; cell.row--) 
	{
		if(CellIsSelected(cell)) 
		{
			TableIndexT		inWideOpenRow =
							mCollapsableTree->GetWideOpenIndex(cell.row);
			STableCell		woCell(inWideOpenRow, 1);
			GroupTableEntry	tableEntry;
			Uint32			dataSize = sizeof(GroupTableEntry);
			
			GetCellData(woCell, &tableEntry, dataSize);
			if(tableEntry.isTopGroup ||
				tableEntry.u.item.groupItem.type == kPGPGroupItem_Group)
			{
				PGPGroupInfo	groupInfo;
				PGPGroupID		id;
				
				if(tableEntry.isTopGroup)
					id = tableEntry.u.groupID;
				else
					id = tableEntry.u.item.groupItem.u.group.id;
				err = PGPGetGroupInfo(mGroupsRef, id, &groupInfo);
				pgpAssertNoErr(err);
				if(IsntPGPError(err))
				{
					if(EditGroup(&groupInfo))
					{
						err = PGPSetGroupName(mGroupsRef,
											id, groupInfo.name);
						pgpAssertNoErr(err);
						err = PGPSetGroupDescription(mGroupsRef,
											id, groupInfo.description);
						pgpAssertNoErr(err);
						Draw(nil);
					}
				}
			}
			else
			{
				if(PGPKeyRefIsValid(tableEntry.u.item.cachedKey))
				{
					if(!CKeyInfoWindow::OpenKeyInfo(
							tableEntry.u.item.cachedKey, 
							CPGPKeys::TheApp()->GetDefaultKeyView()->
							GetKeyTable()))
						break;	// Too many key windows, break out
				}
			}
		}
	}
}

	void
CGroupsTable::ClickCell(
	const STableCell&		inCell,
	const SMouseDownEvent&	inMouseDown)
{
	if(GetClickCount() == 2)
	{
		ShowInfo();
	}
	else if (LDropArea::DragAndDropIsPresent() && 
			::WaitMouseMoved(inMouseDown.macEvent.where))
		StartDrag(inCell, inMouseDown);
	else if(inCell.col == 1)
		CPGPHierarchyTable::ClickCell(inCell, inMouseDown);
}

	void
CGroupsTable::StartDrag(
	const STableCell&		inCell,
	const SMouseDownEvent&	inMouseDown)
{
	Rect			cellRect;
	TableIndexT		inWideOpenRow =
					mCollapsableTree->GetWideOpenIndex(inCell.row);
	STableCell		woCell(inWideOpenRow, 1);
	GroupTableEntry	tableEntry;
	Uint32			dataSize = sizeof(GroupTableEntry);
	
	GetCellData(woCell, &tableEntry, dataSize);

	mSendingDrag = TRUE;
	
	// Setup the drag rect
	GetLocalCellRect(inCell, cellRect);
	::InsetRect(&cellRect, 1, 1);
	cellRect.left = kDropFlagSlop;
	cellRect.right = kNameColumnWidth;
	if(tableEntry.isTopGroup)
	{
		LDragTask	drag(inMouseDown.macEvent, cellRect, 
							1, kGroupIDDragFlavor,
							&tableEntry.u.groupID,
							sizeof(PGPGroupID), flavorSenderOnly);
	}
	else
	{
		LDragTask	drag(inMouseDown.macEvent, cellRect, 
							1, kGroupItemDragFlavor,
							&tableEntry.u.item.groupItem,
							sizeof(PGPGroupItem), flavorSenderOnly);
	}
	
	mSendingDrag = FALSE;
}

	void
CGroupsTable::ClickSelf(
	const SMouseDownEvent	&inMouseDown)
{
	STableCell	hitCell;
	SPoint32	imagePt;
	
	LocalToImagePoint(inMouseDown.whereLocal, imagePt);
	
	if (GetCellHitBy(imagePt, hitCell)) {
										// Click is inside hitCell
										// Check if click is inside DropFlag
		TableIndexT	woRow = mCollapsableTree->GetWideOpenIndex(hitCell.row);
		Rect	flagRect;
		CalcCellFlagRect(hitCell, flagRect);
		
		if ((hitCell.col == 1) &&
			mCollapsableTree->IsCollapsable(woRow) &&
			::PtInRect(inMouseDown.whereLocal, &flagRect))
		{
										// Click is inside DropFlag
			FocusDraw();
			Boolean	expanded = mCollapsableTree->IsExpanded(woRow);
			if (LDropFlag::TrackClick(flagRect, inMouseDown.whereLocal,
									expanded)) {
										// Mouse released inside DropFlag
										//   so toggle the Row
				if (inMouseDown.macEvent.modifiers & optionKey) {
										// OptionKey down means to do
										//   a deep collapse/expand
					if (expanded) {
						DeepCollapseRow(woRow);
					} else {
						DeepExpandRow(woRow);
					}
				
				} else {				// Shallow collapse/expand
					if (expanded) {
						CollapseRow(woRow);
					} else {
						ExpandRow(woRow);
					}
				}
			}
	
		}
		else
		{
			if(hitCell.col == 1)
			{
				if (ClickSelect(hitCell, inMouseDown))
					ClickCell(hitCell, inMouseDown);
			}
		}
	} else {							// Click is outside of any Cell
		UnselectAllCells();
	}
}

	void
CGroupsTable::HiliteCellActively(
	const STableCell	&inCell,
	Boolean				/*inHilite*/)
{
	Rect	cellFrame;
	
	if(inCell.col != 1)
		return;
	if(GetLocalCellRect(inCell, cellFrame) && FocusExposed())
	{
		StDeviceLoop	devLoop(cellFrame);		
		SInt16			depth;

		while(devLoop.NextDepth(depth))
		{
			StColorPenState saveColorPen;   // Preserve color & pen state
			StColorPenState::Normalize();
			if(depth >= 8)
			{
				::RGBBackColor(&UGAColorRamp::GetColor(colorRamp_Gray2));
				::RGBForeColor(&UGAColorRamp::GetColor(colorRamp_Black));
			}
			else
			{
				::RGBBackColor(&UGAColorRamp::GetColor(colorRamp_White));
				::RGBForeColor(&UGAColorRamp::GetColor(colorRamp_Black));
			}
			cellFrame.left += kDropFlagSlop;
			cellFrame.right = cellFrame.left + kNameColumnWidth;
	        UDrawingUtils::SetHiliteModeOn();
			::InvertRect(&cellFrame);
		}
	}
}

	void
CGroupsTable::HiliteCellInactively(
	const STableCell	&inCell,
	Boolean			 	/*inHilite*/)
{
	Rect	cellFrame;
	
	if(inCell.col != 1)
		return;
	if(GetLocalCellRect(inCell, cellFrame) && FocusExposed())
	{
		StDeviceLoop	devLoop(cellFrame);		
		SInt16			depth;

		while(devLoop.NextDepth(depth))
		{
	        StColorPenState saveColorPen;   // Preserve color & pen state
	        StColorPenState::Normalize();
	        if(depth >= 8)
	        {
				::RGBBackColor(&UGAColorRamp::GetColor(colorRamp_Gray2));
				::RGBForeColor(&UGAColorRamp::GetColor(colorRamp_Black));
			}
			else
			{
				::RGBBackColor(&UGAColorRamp::GetColor(colorRamp_White));
				::RGBForeColor(&UGAColorRamp::GetColor(colorRamp_Black));
			}
			::PenMode(srcXor);
			cellFrame.left += kDropFlagSlop;
			cellFrame.right = cellFrame.left + kNameColumnWidth;
			UDrawingUtils::SetHiliteModeOn();
			::FrameRect(&cellFrame);
		}
	}
}

	void
CGroupsTable::HiliteCellDrag(
	const STableCell	&inCell)
{
	Rect	cellFrame;
	
	if(inCell.col != 1)
		return;
	if(GetLocalCellRect(inCell, cellFrame) && FocusExposed())
	{
		StDeviceLoop	devLoop(cellFrame);		
		SInt16			depth;

		while(devLoop.NextDepth(depth))
		{
	        StColorPenState saveColorPen;   // Preserve color & pen state
	        StColorPenState::Normalize();
	        if(depth >= 8)
	        {
				::RGBBackColor(&UGAColorRamp::GetColor(colorRamp_Gray2));
				::RGBForeColor(&UGAColorRamp::GetColor(colorRamp_Black));
			}
			else
			{
				::RGBBackColor(&UGAColorRamp::GetColor(colorRamp_White));
				::RGBForeColor(&UGAColorRamp::GetColor(colorRamp_Black));
			}
			cellFrame.left += kDropFlagSlop;
			cellFrame.right = cellFrame.left + kNameColumnWidth;
			cellFrame.bottom -= 1;
			::InvertRect(&cellFrame);
		}
	}
}

	void
CGroupsTable::DrawSelf()
{
	Rect frame;

	::RGBForeColor(&UGAColorRamp::GetColor(colorRamp_Gray1));
	::RGBBackColor(&UGAColorRamp::GetColor(colorRamp_White));
	CalcLocalFrameRect(frame);
	{
		StDeviceLoop	devLoop(frame);		
		SInt16			depth;

		while(devLoop.NextDepth(depth))
		{
			if(depth > 1)
				::PaintRect(&frame);
			else
				::EraseRect(&frame);
		}
	}
	LTableView::DrawSelf();
}

	Boolean
CGroupsTable::EditGroup(PGPGroupInfo *groupInfo)
{
	CPGPStDialogHandler	newGroupDialog(kNewGroupDialogID, this);
	LWindow				*ngd;
	MessageT			dialogMessage;
	Str255				pstr;
	LEditText			*mAddressEditField,
						*mDescriptionEditField;
	Boolean				success = FALSE;

	const PaneIDT	kAddressEditField		= 'eAdd';
	const PaneIDT	kDescEditField			= 'eDes';
	
	ngd = newGroupDialog.GetDialog();
	mAddressEditField = (LEditText *)ngd->FindPaneByID(kAddressEditField);
	mDescriptionEditField =
		(LEditText *)ngd->FindPaneByID(kDescEditField);
	if(groupInfo->id != kPGPInvalidGroupID)
	{
		CToPString(groupInfo->name, pstr);
		mAddressEditField->SetDescriptor(pstr);
		CToPString(groupInfo->description, pstr);
		mDescriptionEditField->SetDescriptor(pstr);
	}
	ngd->Show();
badInput:
	do
	{
		dialogMessage = newGroupDialog.DoDialog();
	} while(dialogMessage != msg_OK &&
			dialogMessage != msg_Cancel);
	if(dialogMessage == msg_OK)
	{
		mAddressEditField->GetDescriptor(pstr);
		if(pstr[0] == '\0')
		{
			CWarningAlert::Display(kWAStopAlertType, kWAOKStyle,
						kStringListID, kBadGroupInfoStringID);
			goto badInput;
		}
		PToCString(pstr, groupInfo->name);
		mDescriptionEditField->GetDescriptor(pstr);
		PToCString(pstr, groupInfo->description);
		success = TRUE;
	}
	return success;
}


	void
CGroupsTable::DeleteSelection()
{	
	Boolean		rebuild = FALSE;
	
	TableIndexT	numRows, numCols;
	GetTableSize(numRows, numCols);
	
	STableCell	topCell(1, 1);
	STableCell 	botCell(numRows, 1);
	STableCell	cell;
	
	for(cell = botCell; cell.row >= topCell.row; cell.row--) 
	{
		if(CellIsSelected(cell)) 
		{
			TableIndexT		inWideOpenRow =
								mCollapsableTree->GetWideOpenIndex(cell.row);
			STableCell		woCell(inWideOpenRow, 1);
			GroupTableEntry	tableEntry;
			PGPError		err;
			Uint32			dataSize = sizeof(GroupTableEntry);
			
			GetCellData(woCell, &tableEntry, dataSize);
			if(tableEntry.isTopGroup)
			{
				err = PGPDeleteGroup(mGroupsRef, tableEntry.u.groupID);
				pgpAssertNoErr(err);
				rebuild = TRUE;
				
			}
			else
			{
				PGPUInt32		numKeys,
								numTotal,
								itemIndex;
				PGPGroupItem	groupItem;
				
				RemoveRows(1, inWideOpenRow, true);
				err = PGPCountGroupItems(mGroupsRef,
							tableEntry.u.item.ownerGroupID,
							FALSE, &numKeys, &numTotal);
				pgpAssertNoErr(err);
				if(IsntPGPError(err) && numTotal > 0)
				{
					for(itemIndex = 0; itemIndex < numTotal; itemIndex++)
					{
						err = PGPGetIndGroupItem(mGroupsRef,
								tableEntry.u.item.ownerGroupID,
								itemIndex, &groupItem);
						pgpAssertNoErr(err);
						if(pgpMemoryEqual(&groupItem.u,
							&tableEntry.u.item.groupItem.u,
							sizeof(groupItem.u)))
						{
							err = PGPDeleteIndItemFromGroup(mGroupsRef,
										tableEntry.u.item.ownerGroupID,
										itemIndex);
							pgpAssertNoErr(err);
							break;
						}
					}
				}
			}
		}
	}
	if( rebuild )
		RebuildTable();
	UpdateValidityColumn();
	if(PGPGroupSetNeedsCommit(mGroupsRef))
		PGPSaveGroupSetToFile(mGroupsRef, mGroupsFile);
}

	void
CGroupsTable::UpdateFromServer()
{
	TableIndexT	numRows, numCols;
	GetTableSize(numRows, numCols);
	
	STableCell	topCell(1, 1);
	STableCell 	botCell(numRows, 1);
	STableCell	cell;
	
	for(cell = botCell; cell.row >= topCell.row; cell.row--) 
	{
		if(CellIsSelected(cell)) 
		{
			TableIndexT		inWideOpenRow =
								mCollapsableTree->GetWideOpenIndex(cell.row);
			STableCell		woCell(inWideOpenRow, 1);
			GroupTableEntry	tableEntry;
			Uint32			dataSize = sizeof(GroupTableEntry);
			PGPKeySetRef	keySet = kInvalidPGPKeySetRef;
			PGPError		err;
			
			keySet = NULL;
			GetCellData(woCell, &tableEntry, dataSize);
			if(tableEntry.isTopGroup)
			{
				err = PGPGetGroupFromServer(gPGPContext, gTLSContext,
						mGroupsRef, tableEntry.u.groupID, NULL, 0, &keySet);
				if(IsPGPError(err) && (err != kPGPError_UserAbort))
					ReportPGPError(err);
			}
			else
			{
				err = PGPGetKeyIDFromServer( gPGPContext, gTLSContext,
						&tableEntry.u.item.groupItem.u.key.keyID, &keySet);
				if(IsPGPError(err) && (err != kPGPError_UserAbort))
					ReportPGPError(err);
			}
			if(PGPKeySetRefIsValid(keySet))
			{
				CPGPKeys::TheApp()->GetDefaultKeyView()->GetKeyTable()->
					ImportKeysFromKeySet(keySet);
			}
		}
	}
}

	void
CGroupsTable::FindCommandStatus(
	CommandT	inCommand,
	Boolean		&outEnabled,
	Boolean		&outUsesMark,
	Char16		&outMark,
	Str255		outName)
{
	switch(inCommand) 
	{
		case cmd_Clear:
		case cmd_OptionClear:
			if(mTableSelector->GetFirstSelectedRow() > 0)
				outEnabled = true;
			break;
		case cmd_SelectAll:
		case cmd_NewGroup:
		case cmd_ImportGroups:
			outEnabled = true;
			break;
		case cmd_CollapseAll:
			outEnabled = true;
			if(mTableSelector->GetFirstSelectedRow() > 0)
				GetIndString(	outName,
								kStringListID,
								kCollapseSelectionID);
			else
				GetIndString(	outName,
								kStringListID,
								kCollapseAllID);
			break;
		case cmd_ExpandAll:
			outEnabled = true;
			if(mTableSelector->GetFirstSelectedRow() > 0)
				GetIndString(	outName,
								kStringListID,
								kExpandSelectionID);
			else
				GetIndString(	outName,
								kStringListID,
								kExpandAllID);
			break;
		case cmd_GroupProperties:
			if(mTableSelector->GetFirstSelectedRow() > 0)
				outEnabled = TRUE;
			break;
		case cmd_KSGetKeys:
			if(mTableSelector->GetFirstSelectedRow() > 0 &&
				gServerCallsPresent)
				outEnabled = TRUE;
			break;
		default:
			 LCommander::FindCommandStatus(inCommand, outEnabled,
										outUsesMark, outMark, outName);
			break;
	}
}

	Boolean
CGroupsTable::ObeyCommand(
	CommandT	inCommand,
	void		*ioParam)
{
	Boolean		cmdHandled = true,
				bypass = FALSE;

	switch (inCommand) 
	{
		case cmd_OptionClear:
			bypass = TRUE;
		case cmd_Clear:
		{
			if(bypass || CWarningAlert::Display(kWACautionAlertType,
							kWAOKCancelStyle, kStringListID,
							kDeleteConfirmStringID) == msg_OK)
			{
				DeleteSelection();
			}
			break;
		}
		case cmd_NewGroup:
		{
			PGPGroupInfo	groupInfo;
			PGPError		err;
			
			pgpClearMemory(&groupInfo, sizeof(PGPGroupInfo));
			if(EditGroup(&groupInfo))
			{
				PGPGroupID		newGroupID = kPGPInvalidGroupID;
				
				err = PGPNewGroup(mGroupsRef, groupInfo.name,
							groupInfo.description, &newGroupID);
				pgpAssertNoErr(err);
				if(IsntPGPError(err))
				{
					if ( IsntPGPError( err ) )
					{
						err	= PGPSortGroupSetStd( mGroupsRef,
								CPGPKeys::TheApp()->GetKeySet() );
					}

					ResyncTable(FALSE, TRUE);
				}
			}
			break;
		}
		case cmd_ImportGroups:
		{
			FSSpec				fsSpec;
			SFTypeList			typeList;
			
			typeList[0] = kPGPMacFileType_Groups;
			
			if( CustomGetFileWithShowAll( 1, typeList, &fsSpec ) )
				ImportGroupsFromFile( &fsSpec );
			break;
		}
		case cmd_CollapseAll:
			CollapseGroups(FALSE);
			break;
		case cmd_ExpandAll:
			ExpandGroups();
			break;
		case cmd_SelectAll:
			UnselectAllCells();
			SelectAllCells();
			break;
		case cmd_GroupProperties:
			ShowInfo();
			break;
		case cmd_KSGetKeys:
			UpdateFromServer();
			break;
		default:
			cmdHandled =  LCommander::ObeyCommand(inCommand, ioParam);
			break;
	}
	
	return cmdHandled;
}

	void
CGroupsTable::FocusDropArea()
{
	OutOfFocus(nil);
	FocusDraw();
}

	Boolean
CGroupsTable::ItemIsAcceptable(
	DragReference		inDragRef,
	ItemReference		inItemRef)
{
	Boolean			isAcceptable = FALSE;
	ushort			numFlavors;
	OSStatus		status;
	DragAttributes	attributes;
	
	GetDragAttributes(inDragRef, &attributes);
	status = CountDragItemFlavors(inDragRef, inItemRef, &numFlavors);
	if(IsntErr(status))
	{
		for(ushort flavorIndex = 1; flavorIndex <= numFlavors; flavorIndex++)
		{
			FlavorType	flavorType;
			
			status = GetFlavorType(inDragRef, inItemRef, flavorIndex,
									&flavorType);
			if(IsntErr(status))
			{
				switch(flavorType)
				{
					case 'TEXT':
					case kGroupIDDragFlavor:
					case kGroupItemDragFlavor:
						isAcceptable = TRUE;
						break;
					case flavorTypeHFS:
					{
						HFSFlavor	flavorData;
						ByteCount	dataSize;
						ByteCount	dataOffset;
						
						dataSize 	= sizeof(flavorData);
						dataOffset	= 0;
						
						status = GetFlavorData(inDragRef, inItemRef,
										flavorType, &flavorData,
										(Int32 *) &dataSize, dataOffset);
						if(IsntErr(status))
						{
							// Check for dragged folders and/or disks
							
							if(flavorData.fileType == kPGPMacFileType_Groups &&
								flavorData.fileCreator == kPGPMacFileCreator_Keys)
							{
								isAcceptable = true;
							}
						}
						break;
					}
				}
					
				if(isAcceptable)
					break;
			}
		}
	}
	return(isAcceptable);
}

	void
CGroupsTable::ResortGroups()
{
	PGPError	err;
	
	err	= PGPSortGroupSetStd( mGroupsRef,
			CPGPKeys::TheApp()->GetKeySet() );
	pgpAssertNoErr(err);
}

	PGPError
CGroupsTable::AddKeyIDToGroup(
	PGPKeyID			*keyID,
	PGPUInt32			algorithm,
	const STableCell	&groupCell)
{
	TableIndexT			woRow = GetWideOpenIndex(groupCell.row);
	STableCell			woCell(woRow, 1);
	Uint32				dataSize = sizeof(GroupTableEntry);
	GroupTableEntry		tableEntry;
	PGPError			err = kPGPError_UnknownError;
	
	GetCellData(woCell, &tableEntry, dataSize);
	if(tableEntry.isTopGroup)
	{
		PGPGroupItem	groupItem;
		
		groupItem.type 				= kPGPGroupItem_KeyID;
		groupItem.userValue 		= 0;
		groupItem.u.key.algorithm 	= (PGPPublicKeyAlgorithm) algorithm;
		groupItem.u.key.keyID		= *keyID;
		
		err = PGPAddItemToGroup(mGroupsRef, &groupItem, tableEntry.u.groupID);
		ResortGroups();
		ResyncTable(FALSE, TRUE);
		UpdateValidityColumn();
		if(PGPGroupSetNeedsCommit(mGroupsRef))
			PGPSaveGroupSetToFile(mGroupsRef, mGroupsFile);
	}
	return err;
}

	PGPError
CGroupsTable::AddGroupToGroup(
	PGPGroupID			groupID,
	const STableCell	&groupCell)
{
	TableIndexT			woRow = GetWideOpenIndex(groupCell.row);
	STableCell			woCell(woRow, 1);
	Uint32				dataSize = sizeof(GroupTableEntry);
	GroupTableEntry		tableEntry;
	PGPError			err = kPGPError_UnknownError;
	
	GetCellData(woCell, &tableEntry, dataSize);
	if(tableEntry.isTopGroup)
	{
		PGPGroupItem	groupItem;
		
		groupItem.type = kPGPGroupItem_Group;
		groupItem.userValue = 0;
		groupItem.u.group.id = groupID;
		err = PGPAddItemToGroup(mGroupsRef, &groupItem, tableEntry.u.groupID);
		ResortGroups();
		ResyncTable(FALSE, TRUE);
		UpdateValidityColumn();
		if(PGPGroupSetNeedsCommit(mGroupsRef))
			PGPSaveGroupSetToFile(mGroupsRef, mGroupsFile);
	}
	return err;
}

	PGPError
CGroupsTable::AddGroupItemToGroup(
	PGPGroupItem		*item,
	const STableCell	&groupCell)
{
	TableIndexT			woRow = GetWideOpenIndex(groupCell.row);
	STableCell			woCell(woRow, 1);
	Uint32				dataSize = sizeof(GroupTableEntry);
	GroupTableEntry		tableEntry;
	PGPError			err = kPGPError_UnknownError;
	
	GetCellData(woCell, &tableEntry, dataSize);
	if(tableEntry.isTopGroup)
	{
		item->userValue = 0;
		err = PGPAddItemToGroup(mGroupsRef, item, tableEntry.u.groupID);
		ResortGroups();
		ResyncTable(FALSE, TRUE);
		UpdateValidityColumn();
		if(PGPGroupSetNeedsCommit(mGroupsRef))
			PGPSaveGroupSetToFile(mGroupsRef, mGroupsFile);
	}
	return err;
}

	void
CGroupsTable::ReceiveDragItem(
	DragReference	inDragRef,
	DragAttributes	inDragAttrs,
	ItemReference	inItemRef,
	Rect			&/*inItemBounds*/)	// In Local coordinates
{
	Boolean			received = FALSE;
	ushort			numFlavors;
	OSStatus		status;
	Size			droppedSize,
					expectedSize;
	PGPError		err;
	
	status = CountDragItemFlavors(inDragRef, inItemRef, &numFlavors);
	if(IsntErr(status))
	{
		for(ushort flavorIndex = 1; flavorIndex <= numFlavors; flavorIndex++)
		{
			FlavorType	flavorType;
			
			status = GetFlavorType(inDragRef, inItemRef, flavorIndex,
									&flavorType);
			if(IsntErr(status))
			{
				status = GetFlavorDataSize(inDragRef, inItemRef,
											flavorType, &expectedSize);
				if(flavorType == 'TEXT')
				{
					PGPByte			*droppedBuf;
					
					if((mLastDragCell.row == 0) || (mLastDragCell.col != 1))
						return;
					if(IsntErr(status) && IsntNull(droppedBuf =
								(PGPByte *)pgpAlloc(expectedSize)))
					{
						droppedSize = expectedSize;
						status = GetFlavorData(inDragRef, inItemRef,
								flavorType, droppedBuf, &droppedSize, 0);
						pgpAssertNoErr(status);
						pgpAssert(droppedSize == expectedSize);
						if(IsntErr(status))
						{
							PGPKeySetRef	newKeySet;
							
							err = PGPImportKeySet( gPGPContext, &newKeySet,
									PGPOInputBuffer( gPGPContext,
											droppedBuf, droppedSize ),
									PGPOLastOption( gPGPContext ) );
							pgpAssertNoErr(err);
							if(IsntPGPError(err) && PGPKeySetRefIsValid(newKeySet))
							{
								PGPKeyIterRef	keyIter;
								PGPKeyListRef	keyList;
								
								err = PGPOrderKeySet(newKeySet,
												kPGPAnyOrdering,
												&keyList);
								if(IsntPGPError(err))
								{
									err = PGPNewKeyIter(keyList, &keyIter);
									if(IsntPGPError(err))
									{
										PGPKeyRef	key;
										PGPKeyID	keyID;
										
										while(IsntPGPError(
											PGPKeyIterNext(keyIter, &key))
											&& PGPKeyRefIsValid(key))
										{
											err = PGPGetKeyIDFromKey(key,
													&keyID);
											if(IsntPGPError(err) )
											{
												PGPInt32 algorithm;
												
												err = PGPGetKeyNumber(key,
													kPGPKeyPropAlgID, 
													&algorithm);
												pgpAssertNoErr(err);
												err = AddKeyIDToGroup(
													&keyID, algorithm,
													mLastDragCell);
												if(IsntPGPError(err))
													received = TRUE;
											}
										}
										PGPFreeKeyIter(keyIter);
									}
									PGPFreeKeyList(keyList);
								}
								PGPFreeKeySet(newKeySet);
							}
						}
						pgpFree(droppedBuf);
					}
				}
				else if(flavorType == kGroupIDDragFlavor)
				{
					PGPGroupID	groupID;
					
					if((mLastDragCell.row == 0) || (mLastDragCell.col != 1))
						return;
					if(IsntErr(status) && expectedSize == sizeof(PGPGroupID))
					{
						droppedSize = expectedSize;
						status = GetFlavorData(inDragRef, inItemRef,
								flavorType, &groupID, &droppedSize, 0);
						pgpAssertNoErr(status);
						if(IsntErr(status))
						{
							err = AddGroupToGroup(groupID, mLastDragCell);
							if(IsntPGPError(err))
								received = TRUE;
						}
					}
				}
				else if(flavorType == kGroupItemDragFlavor)
				{
					PGPGroupItem	item;
					
					if((mLastDragCell.row == 0) || (mLastDragCell.col != 1))
						return;
					if(IsntErr(status) && expectedSize == sizeof(PGPGroupItem))
					{
						droppedSize = expectedSize;
						status = GetFlavorData(inDragRef, inItemRef,
								flavorType, &item, &droppedSize, 0);
						pgpAssertNoErr(status);
						if(IsntErr(status))
						{
							err = AddGroupItemToGroup(&item, mLastDragCell);
							if(IsntPGPError(err))
								received = TRUE;
						}
					}
				}
				else if(flavorType == flavorTypeHFS)
				{
					HFSFlavor	flavorData;
					ByteCount	dataSize;
					
					dataSize 	= sizeof(flavorData);
					
					status = GetFlavorData(inDragRef, inItemRef, flavorType,
							&flavorData, (Int32 *) &dataSize, 0);
					if(IsntErr(status))
					{
						received = ImportGroupsFromFile(
										&flavorData.fileSpec );
					}
				}
				if(received)
					break;
			}
		}
	}
}

	Boolean
CGroupsTable::ImportGroupsFromFile(
	const FSSpec	*fsSpec )
{
	PGPGroupSetRef	mergeGroupSet;
	PGPError		err;
	
	err = PGPNewGroupSetFromFSSpec(gPGPContext, fsSpec, &mergeGroupSet);
	if( IsntPGPError( err ) )
	{
		err = PGPMergeGroupSets( mergeGroupSet, mGroupsRef );
		pgpAssertNoErr( err );
		PGPFreeGroupSet( mergeGroupSet );
		ResortGroups();
		ResyncTable( FALSE, TRUE );
		UpdateValidityColumn();
		if( PGPGroupSetNeedsCommit( mGroupsRef ) )
			PGPSaveGroupSetToFile( mGroupsRef, mGroupsFile );
		if( IsntPGPError( err ) )
			return TRUE;
	}
	return FALSE;
}

	void
CGroupsTable::HiliteDropArea(
	DragReference	inDragRef)
{
	Rect		dropRect;
	RgnHandle	dropRgn;
	
	CalcLocalFrameRect(dropRect);
	/*::PenSize(2,2);
	::RGBForeColor(&UGAColorRamp::GetColor(colorRamp_White));
	::FrameRect(&dropRect);
	::RGBForeColor(&UGAColorRamp::GetColor(colorRamp_Black));
	::RGBBackColor(&UGAColorRamp::GetColor(colorRamp_White));
	::PenSize(1,1);*/
	::RGBForeColor(&UGAColorRamp::GetColor(colorRamp_Black));
	::RGBBackColor(&UGAColorRamp::GetColor(colorRamp_Gray1));
	dropRgn = ::NewRgn();
	::RectRgn(dropRgn, &dropRect);
	::ShowDragHilite(inDragRef, dropRgn, true);
	::DisposeRgn(dropRgn);
}

	void
CGroupsTable::UnhiliteDropArea(
	DragReference	inDragRef)
{
	Rect		dropRect;
	//RgnHandle	dropRgn;

	CalcPortFrameRect(dropRect);
	::HideDragHilite(inDragRef);
	/*dropRgn = ::NewRgn();
	::OpenRgn();
	::PenSize(2,2);
	::FrameRect(&dropRect);
	::CloseRgn(dropRgn);
	InvalPortRgn(dropRgn);
	::DisposeRgn(dropRgn);*/
}

	void
CGroupsTable::EnterDropArea(
	DragReference	inDragRef,
	Boolean			inDragHasLeftSender)
{
	LDragAndDrop::EnterDropArea(inDragRef, inDragHasLeftSender);
	UnselectAllCells();
	mLastDragCell.SetCell(0,0);
}

	void
CGroupsTable::LeaveDropArea(
	DragReference	inDragRef)
{
	if(mLastDragCell.row != 0)
		HiliteCellDrag(mLastDragCell);
	LDragAndDrop::LeaveDropArea(inDragRef);
}

	void
CGroupsTable::InsideDropArea(
	DragReference	inDragRef)
{
	Point		mousePos;
	SPoint32	imagePt;
	STableCell	overCell;
	Boolean		foundCell;
	Uint32		nestingLevel;
	
	if(IsntErr(::GetDragMouse(inDragRef, &mousePos, nil)))
	{
		::GlobalToLocal(&mousePos);
		LocalToImagePoint(mousePos, imagePt);
		foundCell = GetCellHitBy(imagePt, overCell);
		if(foundCell)
			nestingLevel = GetNestingLevel(GetWideOpenIndex(overCell.row));
		if(foundCell && (overCell.col == 1) && (nestingLevel == 0))
		{
			if(overCell != mLastDragCell)
			{
				if(mLastDragCell.row != 0)
					HiliteCellDrag(mLastDragCell);
				mLastDragCell = overCell;
				HiliteCellDrag(mLastDragCell);
			}
		}
		else
		{
			if(mLastDragCell.row != 0)
			{
				HiliteCellDrag(mLastDragCell);
				mLastDragCell.SetCell(0,0);
			}
		}
	}
}

	void
CGroupsTable::ScrollImageBy(	Int32		inLeftDelta,
							Int32		inTopDelta,
							Boolean		inRefresh)
{
	if(inLeftDelta != 0)
	{
		/*mLabels->NotifyScrolled(inLeftDelta);
		if(inRefresh)
			mLabels->Refresh();*/
	}
	CPGPHierarchyTable::ScrollImageBy(inLeftDelta, inTopDelta, inRefresh);
}

	Boolean
CGroupsTable::HandleKeyPress(const EventRecord&	inKeyEvent)
{
	Boolean		keyHandled	= true;
	Int16		theKey		= inKeyEvent.message & charCodeMask;
	STableCell	cell;
	TableIndexT	rows, cols;
	
	if(inKeyEvent.modifiers & cmdKey)
		keyHandled = LCommander::HandleKeyPress(inKeyEvent);
	else
	{
		SetUpdateCommandStatus(TRUE);
		switch(theKey)
		{
			default:
				if(((theKey >= ' ') && (theKey <= 'Z')) ||
					((theKey >= 'a') && (theKey <= 'z')))
				{
				}
				else
					keyHandled = LCommander::HandleKeyPress(inKeyEvent);
				break;
			case char_UpArrow:
				cell = GetFirstSelectedCell();
				cell.col = 1;
				if(cell.row > 1)
					cell.row--;
				if(cell.row < 1)
					cell.row = 1;
				UnselectAllCells();
				ScrollCellIntoFrame(cell);
				SelectCell(cell);
				break;
			case char_DownArrow:
				GetTableSize(rows, cols);
				cell = GetFirstSelectedCell();
				cell.col = 1;
				cell.row++;
				if(cell.row > rows)
					cell.row = rows;
				UnselectAllCells();
				ScrollCellIntoFrame(cell);
				SelectCell(cell);
				break;
			case char_LeftArrow:
			case char_RightArrow:
				break;
			case char_Home:
				cell.row = cell.col = 1;
				ScrollCellIntoFrame(cell);
				break;
			case char_End:
				GetTableSize(rows, cols);
				cell.col = 1;
				cell.row = rows;
				ScrollCellIntoFrame(cell);
				break;
			case char_PageUp:
			{
				Rect	frameRect;
				
				CalcLocalFrameRect(frameRect);
				ScrollPinnedImageBy( 0,
					-UGraphicUtils::RectHeight(frameRect), true );
				break;
			}
			case char_PageDown:
			{
				Rect	frameRect;
				
				CalcLocalFrameRect(frameRect);
				ScrollPinnedImageBy( 0,
					UGraphicUtils::RectHeight(frameRect), true );
				break;
			}
			case char_Tab:
			case char_Return:
			case char_Enter:
				keyHandled = LCommander::HandleKeyPress(inKeyEvent);
				break;
			case char_FwdDelete:
			case char_Backspace:
				if(inKeyEvent.modifiers & optionKey)
					LCommander::GetTarget()->ProcessCommand(cmd_OptionClear);
				else
					LCommander::GetTarget()->ProcessCommand(cmd_Clear);
				break;
		}
	}
	return keyHandled;
}

