/*____________________________________________________________________________
	Copyright (C) 1997 Network Associates Inc. and affiliated companies.
	All rights reserved.
	
	
	
	$Id: CReconstitutionDialog.cp,v 1.31 1999/03/10 12:22:18 jason Exp $
____________________________________________________________________________*/

#include <string.h>

#include <LIconPane.h>
#include <LPushButton.h>
#include <UTableHelpers.h>
#include <URegistrar.h>
#include <PP_Messages.h>

#include "pgpFileSpec.h"
#include "pgpMem.h"
#include "pgpKeys.h"
#include "pgpShare.h"
#include "pgpShareFile.h"
#include "pgpUtilities.h"
#include "pgpUserInterface.h"
#include "pgpOpenPrefs.h"
#include "pflPrefTypes.h"
#include "pgpClientLibDialogs.h"

#include "PGPclientLibUtils.h"
#include "pgpClientLib.h"

#include "CString.h"
#include "CReconstitutionDialog.h"


static const PaneIDT	caption_SplitKey		=	'cKey';
static const PaneIDT	icon_SplitKey			=	'iKey';
static const PaneIDT	table_KeyShares			=	'tSha';
static const PaneIDT	caption_CollectedShares	=	'cCol';
static const PaneIDT	caption_SharesNeeded	=	'cNee';
static const PaneIDT	caption_Network			=	'cNet';
static const PaneIDT	view_Network			=	'vNet';
static const PaneIDT	caption_Status			=	'cSta';
static const PaneIDT	caption_Authenticated	=	'cAut';
static const PaneIDT	button_Network			=	'bNet';
static const PaneIDT	button_File				=	'bFil';
static const PaneIDT	button_Cancel			=	'bCAN';
static const PaneIDT	button_Continue			=	'bOK ';

static const MessageT	msg_Network				=	button_Network;
static const MessageT	msg_File				=	button_File;
static const MessageT	msg_Yield				=	'Yld ';

static const ResIDT		kReconstitutionDialogID	=	4769;
static const ResIDT		kDHUserIDIconID			=	3020;
static const ResIDT		kRSAUserIDIconID		=	3002;

enum {
	STRx_Reconstitution			=	4752,
	kErrorStringID				=	1,
	kWrongShareFileStringID,
	kSharePassphraseStringID,
	kListenStringID,
	kConnectStringID,
	kReceiveStringID,
	kCloseStringID,
	kStartNetworkStringID,
	kStopNetworkStringID,
	kAuthorizationStringID,
	kInvalidKeyStringID,
	kBadPassphraseStringID
};


struct DecryptSharesData {
	PGPKeySetRef			recipientSet;
	PGPKeyID				*keyIDList;
	PGPUInt32				keyIDCount;
	CReconstitutionDialog *	dialog;
};



class CWhackWindowKind
{
protected:
	short			mSaveWindowKind;
	WindowRecord	*mFrontWindow;
	
public:
			CWhackWindowKind( void )
			{
				mFrontWindow = (WindowRecord *) FrontWindow();
				mSaveWindowKind	= mFrontWindow->windowKind;
				mFrontWindow->windowKind = kApplicationWindowKind;
			}
			
	virtual ~CWhackWindowKind()
			{
				mFrontWindow->windowKind = mSaveWindowKind;
			}
};

	PGPError
PGPReconstitutionDialog(
	PGPKeyRef			keyToCollect,
	PGPKeySetRef		keySet,
	PGPtlsContextRef	tlsContext,
	PGPByte **			passKey,
	PGPSize *			passKeySize)
{
	PGPError			result;
	PGPclientLibState	state;
	
	PGPValidatePtr(passKey);
	*passKey = NULL;
	PGPValidateParam(PGPKeyRefIsValid(keyToCollect));
	PGPValidateParam(PGPKeySetRefIsValid(keySet));
	PGPValidatePtr(passKeySize);
	
	result = EnterPGPclientLib(PGPGetKeyContext(keyToCollect), &state);
	if (IsntPGPError(result)) {
		DialogRef	theDialog;

		RegisterClass_(CReconstitutionDialog);
		do {
			theDialog = CReconstitutionDialog::CreateDialog(
													kReconstitutionDialogID);
			if (IsntNull(theDialog)) {
				CReconstitutionDialog *	reconstitutionDialog;
				
				reconstitutionDialog = (CReconstitutionDialog *) 
											::GetWRefCon(theDialog);
				pgpAssertAddrValid(reconstitutionDialog, VoidAlign);
				
				result = reconstitutionDialog->SetInfo(	keyToCollect,
														keySet,
														tlsContext,
														passKey,
														passKeySize);
				if (IsntPGPError(result)) {
					result = reconstitutionDialog->DoDialog();
					if (result == kPGPError_BadPassphrase) {
						PGPFreeData(*passKey);
						*passKey = NULL;
						PGPUIWarningAlert(	kWACautionAlertType,
										kWAOKStyle,
										STRx_Reconstitution,
										kBadPassphraseStringID);
					}
				}
				
				delete reconstitutionDialog;
				::DisposeDialog(theDialog);
			} else {
				result = kPGPError_OutOfMemory;
			}
		} while (result == kPGPError_BadPassphrase);
	}
	ExitPGPclientLib(&state);
	
	return result;
}
	
	
	
CReconstitutionDialog::CReconstitutionDialog(
	LStream *	inStream)
	: CPGPModalGrafPortView(inStream), mShares(kInvalidPGPShareRef),
	  mSharesCollected(0), mSharesNeeded(1), mSKEPActive(false),
	  mSKEPPassphrase(nil), mPrefRef(kInvalidPGPPrefRef),
	  mTLSContext(kInvalidPGPtlsContextRef),
	  mDeferredDismissMessage(msg_Nothing)
{
	mWindowIsMoveable = TRUE;
}



CReconstitutionDialog::~CReconstitutionDialog()
{
	if (PGPShareRefIsValid(mShares)) {
		PGPFreeShares(mShares);
	}
	if (mSKEPPassphrase != nil) {
		PGPFreeData(mSKEPPassphrase);
	}
	if (PGPPrefRefIsValid(mPrefRef)) {
		PGPClosePrefFile(mPrefRef);
	}
}



	void
CReconstitutionDialog::FinishCreateSelf()
{
	CPGPModalGrafPortView::FinishCreateSelf();
	
	mShareholderList = (CShareholderList *) FindPaneByID(table_KeyShares);
	delete mShareholderList->GetTableSelector();
	mShareholderList->SetTableSelector(nil);
	mShareholderList->InsertCols(1, 1, nil, 0, TRUE);
	((LPushButton *) FindPaneByID(button_Network))->AddListener(this);
	((LPushButton *) FindPaneByID(button_File))->AddListener(this);
	((LPushButton *) FindPaneByID(button_Cancel))->AddListener(this);
	((LPushButton *) FindPaneByID(button_Continue))->AddListener(this);
	
	StartRepeating();
}
	
	MessageT
CReconstitutionDialog::HandleMessage(MessageT theMessage)
{
	PGPError	err = kPGPError_NoErr;
	
	switch (theMessage)
	{
		case msg_Network:
			err = CollectNetworkShares();
			theMessage = msg_Nothing;
			break;
		
		case msg_File:
			err = CollectFileShares();
			theMessage = msg_Nothing;
			break;
		
		case msg_OK:
		case msg_Cancel:
		{
			if( mSKEPActive )
			{
				// Cause SKEPReceiveShares() to exit with kPGPError_UserAbort
				mDeferredDismissMessage = theMessage;
				theMessage				= msg_Cancel;
				mExitSKEP				= TRUE;
			}
			else if( theMessage == msg_OK )
			{
				mDialogError = PGPGetPasskeyFromShares(	mShares, mPassKey,
														mPassKeySize );
				if( IsntPGPError( mDialogError ) && 
					! PGPPassphraseIsValid( mKeyToCollect,
						PGPOPasskeyBuffer( mContext, *mPassKey,
							*mPassKeySize ),
						PGPOLastOption( mContext )))
				{
					mDialogError = kPGPError_BadPassphrase;
				}
			}
			
			break;
		}	
			
		default:
			theMessage = CPGPModalGrafPortView::HandleMessage( theMessage );
			break;
	}

	if( IsPGPError( err ) )
	{
		ShowError( err );
	}
	
	return( theMessage );
}

	PGPError
CReconstitutionDialog::SetInfo(
	PGPKeyRef			inKeyToCollect,
	PGPKeySetRef		inKeySet,
	PGPtlsContextRef	inTLSContext,
	PGPByte **			outPassKey,
	PGPSize *			outPassKeySize)
{
	PGPError			result;
	Str255				userID;
	PGPSize				len;
	LIconPane *			splitKeyIcon;
	PGPInt32			algorithm;
	
	mKeySet = inKeySet;
	mKeyToCollect = inKeyToCollect;
	mTLSContext = inTLSContext;
	result = PGPGetKeyIDFromKey(mKeyToCollect, &mKeyToCollectKeyID);
	if (IsntPGPError(result)) {
		mContext = PGPGetKeyContext(mKeyToCollect);
		mPassKey = outPassKey;
		mPassKeySize = outPassKeySize;
		
		result = PGPGetPrimaryUserIDNameBuffer(	mKeyToCollect,
												sizeof(userID) - 1,
												(char *) &userID[1],
												&len);
		if (IsntPGPError(result)) {
			userID[0] = len - 1;
			FindPaneByID(caption_SplitKey)->SetDescriptor(userID);
			result = PGPGetKeyNumber(	mKeyToCollect,
										kPGPKeyPropAlgID,
										&algorithm);
			if (IsntPGPError(result)) {
				mAlgorithm = (PGPPublicKeyAlgorithm) algorithm;
				splitKeyIcon = (LIconPane *) FindPaneByID(icon_SplitKey);
				if (mAlgorithm == kPGPPublicKeyAlgorithm_DSA) {
					splitKeyIcon->SetIconID(kDHUserIDIconID);
				} else {
					splitKeyIcon->SetIconID(kRSAUserIDIconID);
				}
				
				result = PGPOpenClientPrefs(
							PGPGetContextMemoryMgr(mContext),
							&mPrefRef);
			}
		}
	}
	
	return result;
}



	void
CReconstitutionDialog::ListenToMessage(
	MessageT	inMessage,
	void *		ioParam)
{
	switch (inMessage)
	{
		case msg_Network:
		case msg_File:
			SetDismissMessage( inMessage );
			break;
			
		default:
			CPGPModalGrafPortView::ListenToMessage(inMessage, ioParam);
			break;
	}
}



	PGPError
CReconstitutionDialog::AddShares(
	CReconstitutionDialog *	inDialog,
	char *					inUserID,
	PGPShareRef *			inShares)
{
	PGPError				result = kPGPError_NoErr;
	ShareholderListEntry	cell;
	
	pgpClearMemory(&cell, sizeof(cell));
	cell.algorithm = inDialog->mAlgorithm;
	strcpy(cell.name, inUserID);
	cell.numShares = PGPGetNumberOfShares(*inShares);
	if (PGPShareRefIsValid(inDialog->mShares)) {
		PGPShareRef	combinedShares;
		
		result = PGPCombineShares(	inDialog->mShares,
									*inShares,
									&combinedShares);
		if (IsntPGPError(result)) {
			PGPFreeShares(inDialog->mShares);
			PGPFreeShares(*inShares);
			*inShares = kInvalidPGPShareRef;
			inDialog->mShares = combinedShares;
		}
	} else {
		PGPKeyID	keyID;

		result = PGPGetKeyIDFromShares(	*inShares, 
										&keyID);
		if (IsntPGPError(result)) {
			if (PGPCompareKeyIDs(&inDialog->mKeyToCollectKeyID, &keyID) != 0) {
				inDialog->UpdatePort();
				PGPUIWarningAlert(	kWACautionAlertType,
								kWAOKStyle,
								STRx_Reconstitution,
								kWrongShareFileStringID);
			} else {
				inDialog->mSharesNeeded = PGPGetShareThreshold(*inShares);
				inDialog->FindPaneByID(caption_SharesNeeded)->
					SetValue(inDialog->mSharesNeeded);	
				inDialog->mShares = *inShares;
				*inShares = kInvalidPGPShareRef;
			}
		}
	}
	if (IsntPGPError(result)) {
		*inShares = kInvalidPGPShareRef;
		inDialog->mSharesCollected += cell.numShares;
		if (inDialog->mSharesCollected == 
		PGPGetNumberOfShares(inDialog->mShares)) {
			inDialog->FindPaneByID(caption_CollectedShares)->
				SetValue(inDialog->mSharesCollected);
			inDialog->mShareholderList->InsertRows(	1,
													LArray::index_Last,
													&cell,
													sizeof(cell),
													TRUE);
		} else {
			inDialog->mSharesCollected -= cell.numShares;
		}
	}
		
	if (inDialog->mSharesCollected >= inDialog->mSharesNeeded) {
		inDialog->FindPaneByID(button_Continue)->Enable();
	}
	
	return result;
}

	PGPError
CReconstitutionDialog::CollectFileShares(void)
{
	PGPError	err = kPGPError_NoErr;
	FSSpec		fsSpec;

	UpdatePort();
	err = PGPSelectPGPFile( mContext, kPGPFileSelector_KeyShares,
						&fsSpec);
	if( IsntPGPError( err ) )
	{
		PFLFileSpecRef		fsRef;

		err = PFLNewFileSpecFromFSSpec( PGPGetContextMemoryMgr( mContext ),
											&fsSpec, &fsRef );
		if( IsntPGPError( err ) )
		{
			PGPShareFileRef		sfRef;

			err = PGPOpenShareFile( fsRef, &sfRef );
			if( IsntPGPError( err ) )
			{
				DecryptSharesData	dsd;
				char				userID[kMaxShareholderUserIDLength + 1];
				PGPSize				nameSize;

				dsd.recipientSet 	= kInvalidPGPKeySetRef;
				dsd.keyIDList 		= NULL;
				dsd.keyIDCount 		= 0;
				dsd.dialog 			= this;
				
				err = PGPGetShareFileUserID( sfRef,
							kMaxShareholderUserIDLength, userID, &nameSize);
				if( IsntPGPError( err ) )
				{
					PGPOptionListRef	decodeOpt;
					
					err = PGPBuildOptionList( mContext, &decodeOpt,
								PGPOKeySetRef( mContext, mKeySet ),
								PGPOEventHandler( mContext,
									DecryptSharesHandler, &dsd),
								PGPOLastOption( mContext ) );
					if( IsntPGPError( err ) )
					{	
						PGPShareRef	shares;
						
						err = PGPCopySharesFromFile( mContext, sfRef,
									decodeOpt, &shares );
						if( IsntPGPError( err ) )
						{
							err = AddShares(this, userID, &shares );
							
							if( PGPShareRefIsValid( shares ) )
								PGPFreeShares( shares );
						}
						
						if( PGPKeySetRefIsValid( dsd.recipientSet ) )
							PGPFreeKeySet( dsd.recipientSet );
				
						if( IsntNull( dsd.keyIDList ) )
							PGPFreeData( dsd.keyIDList );

						PGPFreeOptionList( decodeOpt );
					}
				}

				PGPFreeShareFile(sfRef);
			}
			
			PFLFreeFileSpec(fsRef);
		}
	}
	
	return( err );
}



	PGPError
CReconstitutionDialog::CollectNetworkShares()
{
	PGPError	err = kPGPError_NoErr;
	
	if( mSKEPActive )
	{	
		// Stopping network collection
		mExitSKEP = TRUE;
	}
	else
	{
		if( IsntNull( mSKEPPassphrase ) )
		{
			PGPFreeData( mSKEPPassphrase );
			mSKEPPassphrase = NULL;
		}
		
		UpdatePort();
		
		err = CheckForNonSplitPrivateKey();

		if (IsntPGPError( err ) )
		{
			CWhackWindowKind	whackWindowKind;
		
			err = PGPSigningPassphraseDialog( mContext, mKeySet, &mSKEPKey,
						PGPOUIDialogPrompt(	mContext, CString(CStringFromSTRx(
								STRx_Reconstitution, kAuthorizationStringID)) ),
						PGPOUIOutputPassphrase(	mContext, &mSKEPPassphrase ),
						PGPOLastOption( mContext ) );
		}
		
		if( IsntPGPError( err ) )
		{
			PGPskepRef	skep;

			FindPaneByID(caption_Network)->Hide();
			FindPaneByID(view_Network)->Show();
			FindPaneByID(button_Network)->SetDescriptor(
							CString(CStringFromSTRx(STRx_Reconstitution,
							kStopNetworkStringID)) );
			
			mExitSKEP 	= FALSE;
			mSKEPActive	= TRUE;
			
			err = PGPNewSKEP( mContext,  mTLSContext, &skep);
			if( IsntPGPError( err ) )
			{
				err = PGPskepSetEventHandler( skep, SKEPEventHandler, this );
				if( IsntPGPError( err ) )
				{
					err = PGPskepReceiveShares(	skep, mSKEPKey,
								mSKEPPassphrase);
				}
				
				PGPFreeSKEP( skep );
			}
			
			mSKEPActive = FALSE;
			
			FindPaneByID(view_Network)->Hide();
			FindPaneByID(caption_Network)->Show();
			FindPaneByID(button_Network)->SetDescriptor(
						CString(CStringFromSTRx(STRx_Reconstitution, kStartNetworkStringID)));
		}
	}
	
	return( err );
}

	PGPError
CReconstitutionDialog::DecryptSharesHandler(
	PGPContextRef	inContext,
	PGPEvent *		inEvent,
	void *			inUserValue)
{
	PGPError			result = kPGPError_NoErr;
	DecryptSharesData *	dsd = (DecryptSharesData *) inUserValue;
	
	switch (inEvent->type) {
		case kPGPEvent_PassphraseEvent:
		{
			CString				prompt(CStringFromSTRx(STRx_Reconstitution, kSharePassphraseStringID));
			char *				passphrase;
			PGPKeyRef			key;
			CWhackWindowKind	whackWindowKind;
			
			dsd->dialog->UpdatePort();
			if ( inEvent->data.passphraseData.fConventional) {
				result = PGPConventionalDecryptionPassphraseDialog(
								inContext,
								PGPOUIDialogPrompt(	inContext,
													prompt),
								PGPOUIOutputPassphrase(	inContext,
														&passphrase),
								PGPOLastOption(inContext));
			} else {
				result = PGPClientDecryptionPassphraseDialog(
								inContext,
								kInvalidPGPtlsContextRef,
								prompt,
								dsd->recipientSet,
								dsd->keyIDList,
								dsd->keyIDCount,
								kInvalidPGPOptionListRef,
								&passphrase,
								&key,
								NULL );
			}
			if (IsntPGPError(result)) {
				result = PGPAddJobOptions(	inEvent->job,
											PGPOPassphrase( inContext,
															passphrase),
											PGPOLastOption(inContext));
				PGPFreeData(passphrase);
			}
		}
		break;
		
		
		case kPGPEvent_RecipientsEvent:
		{
			dsd->recipientSet = inEvent->data.recipientsData.recipientSet;
			if (PGPKeySetRefIsValid(dsd->recipientSet)) {
				result = PGPIncKeySetRefCount(dsd->recipientSet);
			}
			
			if( IsntPGPError( result ) &&
				IsntNull( inEvent->data.recipientsData.keyIDArray ) )
			{
				PGPSize	dataSize;
				
				dsd->keyIDCount 	= inEvent->data.recipientsData.keyCount;
				dataSize			= dsd->keyIDCount * sizeof( PGPKeyID );
				
				dsd->keyIDList = (PGPKeyID *) PGPNewData(
									PGPGetContextMemoryMgr( inContext ),
									dataSize, 0 );
				if( IsntNull( dsd->keyIDList ) )
				{
					pgpCopyMemory( inEvent->data.recipientsData.keyIDArray,
							dsd->keyIDList, dataSize );
				}
				else
				{
					result = kPGPError_OutOfMemory;
				}
			}
		}
		break;
	}
	
	return result;
}



	PGPError
CReconstitutionDialog::SKEPEventHandler(
	PGPskepRef		skep,
	PGPskepEvent *	inEvent,
	PGPUserValue	inUserValue)
{
#pragma unused(skep)

	PGPError				err = kPGPError_NoErr;
	CReconstitutionDialog *	dialog = (CReconstitutionDialog *) inUserValue;
	
	if( dialog->mExitSKEP )
	{
		err = kPGPError_UserAbort;
	}
	else
	{
		switch (inEvent->type)
		{
			case kPGPskepEvent_ListenEvent:
			{
				dialog->FindPaneByID(caption_Status)->SetDescriptor(
							CString(CStringFromSTRx(STRx_Reconstitution, kListenStringID)));
			}
			break;
			
			
			case kPGPskepEvent_ConnectEvent:
			{
				dialog->FindPaneByID(caption_Status)->SetDescriptor(
						CString(CStringFromSTRx(STRx_Reconstitution, kConnectStringID)));
			}
			break;
			
			
			case kPGPskepEvent_AuthenticateEvent:
			{
				Str255		userID;
				PGPSize		fullSize;
				
				err = PGPGetPrimaryUserIDNameBuffer(
								inEvent->data.ad.remoteKey,
								sizeof(Str255) - 1,
								(char *) &userID[1],
								&fullSize);
				if( IsntPGPError( err ) )
				{
					PGPInt32	validity;

					userID[0] = fullSize - 1;
					dialog->FindPaneByID(caption_Authenticated)->
								SetDescriptor(userID);
					dialog->FindPaneByID(caption_Status)->SetDescriptor(
							CString(CStringFromSTRx(STRx_Reconstitution, kReceiveStringID)));
							
					err = PGPGetKeyNumber( inEvent->data.ad.remoteKey,
												kPGPKeyPropValidity,
												&validity);
					if( IsntPGPError( err ) &&
						validity != kPGPValidity_Complete )
					{
						PGPBoolean	isMarginalInvalid;
						
						err = PGPGetPrefBoolean( dialog->mPrefRef,
													kPGPPrefMarginalIsInvalid,
													&isMarginalInvalid);
						if( IsntPGPError( err ) &&
							( isMarginalInvalid ||
							  validity != kPGPValidity_Marginal ) )
						{
							err = PGPConfirmRemoteKeyDialog(
										CString(CStringFromSTRx(STRx_Reconstitution,
											kInvalidKeyStringID)),
										inEvent->data.ad.remoteHostname,
										inEvent->data.ad.remoteKey,
										inEvent->data.ad.tlsCipher, TRUE );
						}
					}
				}
			}
			break;
			
			case kPGPskepEvent_CloseEvent:
			{
				dialog->FindPaneByID(caption_Status)->SetDescriptor(
						CString(CStringFromSTRx(STRx_Reconstitution, kCloseStringID)));
			}
			break;
			
			case kPGPskepEvent_ShareEvent:
			{
				Str255	userID;
					
				dialog->FindPaneByID(caption_Authenticated)->
										GetDescriptor(userID);
				p2cstr(userID);
				dialog->FindPaneByID(caption_Authenticated)->SetDescriptor("\p");
				dialog->FindPaneByID(caption_Status)->SetDescriptor("\p");

				err = AddShares(dialog, (char *) userID, &inEvent->data.sd.shares );

				if( PGPShareRefIsValid( inEvent->data.sd.shares ) )
				{
					PGPFreeShares( inEvent->data.sd.shares );
				}
			}
			break;
			
			case kPGPskepEvent_NullEvent:
			{
				err = dialog->DoDialog();
			}
			break;
		}
	}
	
	return err;
}



	void
CReconstitutionDialog::SpendTime(
	const EventRecord &	inMacEvent)
{
#pragma unused(inMacEvent)

	if( mSKEPActive )
	{
		// Force ModalDialog() to always exit if we're in the SKEP event
		// handler.
		
		SetDismissMessage( msg_Yield );
	}
	else 
	{
		SetDismissMessage( mDeferredDismissMessage );
	}
}



	void
CReconstitutionDialog::ShowError(
	PGPError	inErr)
{
	if (inErr != kPGPError_UserAbort) {
		CString		theErrorString;
		
		PGPGetClientErrorString(	inErr,
									theErrorString.GetMinStorage() + 1,
									theErrorString);
		UpdatePort();
		PGPUIWarningAlert(	kWACautionAlertType,
						kWAOKStyle,
						STRx_Reconstitution,
						kErrorStringID,
						CString(CStringFromInt(inErr)),
						theErrorString);
	}
}



	PGPError
CReconstitutionDialog::CheckForNonSplitPrivateKey(void)
{
	PGPError		result;
	PGPKeyListRef	keyList;
	PGPBoolean		foundKey = false;
	
	foundKey = false;
	result = PGPOrderKeySet( mKeySet, kPGPAnyOrdering, &keyList );
	if( IsntPGPError( result ) )
	{
		PGPKeyIterRef	keyIterator;
		
		result = PGPNewKeyIter( keyList, &keyIterator );
		if( IsntPGPError( result ) )
		{
			PGPKeyRef	theKey;
			
			result = PGPKeyIterNext( keyIterator, &theKey );
			while( IsntPGPError( result ) )
			{
				PGPBoolean	canSign;
				
				result = PGPGetKeyBoolean( theKey, kPGPKeyPropCanSign,
							&canSign );
				if( IsntPGPError( result ) && ( canSign ) )
				{
					PGPBoolean	isSecretShared;
					
					result = PGPGetKeyBoolean( theKey,
								kPGPKeyPropIsSecretShared, &isSecretShared );
					if ( IsntPGPError( result ) && ( !isSecretShared ) )
					{
						foundKey = true;
						break;
					}
				}
				
				result = PGPKeyIterNext( keyIterator, &theKey );
			}
			
			if( result == kPGPError_EndOfIteration )
				result = kPGPError_NoErr;
				
			PGPFreeKeyIter( keyIterator );
		}
		
		PGPFreeKeyList( keyList );
	}
	
	if (IsntPGPError( result ) && ( !foundKey ) )
	{
		result = kPGPClientError_UnsplitPrivateKeyNotFound;
	}
	
	return( result );
}
