/*
	File:		MoreNetworkSetup.c

	Contains:	Simple wrappers to make Network Setup easier.

	Written by:	Quinn

	Copyright:	Copyright (C) 1998 by Apple Computer, Inc., all rights reserved.

				You may incorporate this Apple sample source code into your program(s) without
				restriction. This Apple sample source code has been provided "AS IS" and the
				responsibility for its operation is yours. You are not permitted to redistribute
				this Apple sample source code as "Apple sample source code" after having made
				changes. If you're going to re-distribute the source, we require that you make
				it clear in the source that the code was descended from Apple sample source
				code, but that you've made changes.

	Change History (most recent first):

         <7>     2/19/99    wprice  MODIFIED BY NETWORK ASSOCIATES
         <6>    10/11/98    Quinn   Fix bug in MNSIterateSet where it incremented the loop counter
                                    twice each iteration. Whoops.
         <6>    10/11/98    Quinn   Fix bug in MNSIterateSet where it incremented the loop counter
                                    twice each iteration. Whoops.
         <5>    10/11/98    Quinn   Convert "MorePrefix.h" to "MoreSetup.h".
         <4>     5/11/98    Quinn   Use MoreAssertQ instead of MoreAssert.
         <3>     5/11/98    Quinn   Fix header again.
         <2>     5/11/98    Quinn   Fix header.
         <1>     5/11/98    Quinn   First checked in.
*/

/////////////////////////////////////////////////////////////////
// MoreIsBetter Setup

#include "MoreSetup.h"

/////////////////////////////////////////////////////////////////
// Mac OS Interfaces

#include <Types.h>
#include <Files.h>
#include <Errors.h>
#include <Folders.h>
#include <Resources.h>
#include <Gestalt.h>
#include <CodeFragments.h>
#include <NetworkSetup.h>

/////////////////////////////////////////////////////////////////
// Our Prototypes

#include "MoreNetworkSetup.h"

/////////////////////////////////////////////////////////////////

#if TARGET_RT_MAC_CFM

	extern pascal Boolean IsNetworkSetupAvailable()
		// See comment in header file.
	{
		return (void *) OTCfgOpenDatabase != (void *) kUnresolvedCFragSymbolAddress;
	}

#else

	extern pascal Boolean IsNetworkSetupAvailable()
		// See comment in header file.
	{
		OSStatus junk;
		Boolean result;
		UInt32  response;
		CFragConnectionID connID;
		Ptr junkMain;
		Str255 junkMessage;
		
		#if TARGET_RT_MAC_CFM
			#error "IsNetworkSetupAvailable: This routine should not be used by CFM code."
		#endif
		
		result = false;
		
		if ( Gestalt(gestaltCFMAttr, (SInt32 *) &response) == noErr && (response & (1 << gestaltCFMPresent)) != 0) {
			if ( GetSharedLibrary("\pCfgOpenTpt", kAnyCFragArch, kReferenceCFrag, &connID, &junkMain, junkMessage) == noErr ) {
				result = true;
				junk = CloseConnection(&connID);
				MoreAssertQ(junk == noErr);
			}
		}
		
		return result;
	}

#endif

extern pascal Boolean  MNSValidDatabase(const MNSDatabadeRef *ref)
	// See comment in header file.
{
	return ref != nil && ref->dbRef != nil && ref->area != 0;
}

extern pascal Boolean  MNSDatabaseWritable(const MNSDatabadeRef *ref)
	// See comment in header file.
{
	return (ref->originalArea != 0);
}

extern pascal OSStatus MNSOpenDatabase(MNSDatabadeRef *ref, Boolean forWriting)
	// See comment in header file.
{
	OSStatus err;
	OSStatus junk;

	MoreAssertQ(ref != nil);
	
	ref->dbRef = nil;
	ref->area = 0;
	ref->originalArea = 0;
	
	err = OTCfgOpenDatabase(&ref->dbRef);
	if (err == noErr) {
		err = OTCfgGetCurrentArea(ref->dbRef, &ref->area);
		if (err == noErr) {
			if (forWriting) {
				ref->originalArea = ref->area;
				err = OTCfgBeginAreaModifications(ref->dbRef, ref->originalArea, &ref->area);
			} else {
				err = OTCfgOpenArea(ref->dbRef, ref->area);
			}
		}
		if (err != noErr) {
			junk = MNSCloseDatabase(ref, false);
			MoreAssertQ(junk == noErr);
		}
	}
	return err;	
}

extern pascal OSStatus MNSCloseDatabase(MNSDatabadeRef *ref, Boolean commit)
	// See comment in header file.
{
	OSStatus err;
	OSStatus err2;
	
	MoreAssertQ(ref != nil);
	MoreAssertQ(ref->dbRef != nil);
	MoreAssertQ(ref->originalArea == 0 || ref->area != 0);

	err = noErr;
	if (ref->area != 0) {
		if ( MNSDatabaseWritable(ref) ) {
			if ( commit ) {
				err = OTCfgCommitAreaModifications(ref->dbRef, ref->originalArea, ref->area);
			} else {
				err = OTCfgAbortAreaModifications(ref->dbRef, ref->originalArea);
			}
		} else {
			// Database was opened read-only, you're just not allowed to commit it.
			MoreAssertQ( ! commit );
			err = OTCfgCloseArea(ref->dbRef, ref->area);
		}
	}
	err2 = OTCfgCloseDatabase(&ref->dbRef);
	if (err == noErr) {
		err = err2;
	}
	ref->dbRef = nil;
	ref->area = 0;
	ref->originalArea = 0;

	return err;
}

extern pascal OSStatus MNSGetFixedSizePref(const MNSDatabadeRef *ref,
						const CfgEntityRef *entityID,
						OSType prefType,
						void *buffer, ByteCount prefSize)
	// See comment in header file.
{
	OSStatus err;
	OSStatus err2;
	CfgEntityAccessID prefsRefNum;
	ByteCount actualPrefSize;

	MoreAssertQ(MNSValidDatabase(ref));
	MoreAssertQ(entityID != nil);
	MoreAssertQ(buffer != nil);
	
	// Open the entity, read out the preference, and then
	// close it down.
	
	err = OTCfgOpenPrefs(ref->dbRef, entityID, false, &prefsRefNum);
	if (err == noErr) {
		err = OTCfgGetPrefsSize(prefsRefNum, prefType, &actualPrefSize);
		if (err == noErr && actualPrefSize != prefSize) {
			err = -1;
		}
		if (err == noErr) {
			err = OTCfgGetPrefs(prefsRefNum, prefType, buffer, prefSize);
		}
	
		err2 = OTCfgClosePrefs(prefsRefNum);
		if (err == noErr) {
			err = err2;
		}
	}
	
	return err;
}

extern pascal OSStatus MNSGetPref(const MNSDatabadeRef *ref,
						const CfgEntityRef *entityID,
						OSType prefType,
						void **buffer, ByteCount *prefSize)
	// See comment in header file.
{
	OSStatus err;
	OSStatus err2;
	CfgEntityAccessID prefsRefNum;
	ByteCount junkPrefSize;

	MoreAssertQ(MNSValidDatabase(ref));
	MoreAssertQ(entityID != nil);
	MoreAssertQ(buffer != nil);
	MoreAssertQ(*buffer == nil);
	
	if (prefSize == nil) {
		prefSize = &junkPrefSize;
	}

	// Open the entity, read out the preference, and then
	// close it down.
	
	*buffer = nil;	
	err = OTCfgOpenPrefs(ref->dbRef, entityID, false, &prefsRefNum);
	if (err == noErr) {
		err = OTCfgGetPrefsSize(prefsRefNum, prefType, prefSize);

		if (err == noErr) {
			*buffer = NewPtr(*prefSize);
			if (*buffer == nil) {
				MoreAssertQ(err == memFullErr);
				err = memFullErr;
			}
		}
		if (err == noErr) {
			err = OTCfgGetPrefs(prefsRefNum, prefType, *buffer, *prefSize);
		}
	
		err2 = OTCfgClosePrefs(prefsRefNum);
		if (err == noErr) {
			err = err2;
		}
	}
	
	// Clean up.
	
	if (err != noErr && *buffer != nil) {
		DisposePtr((char *)*buffer);
		MoreAssertQ(MemError() == noErr);
		*buffer = nil;
	}
	return err;
}

extern pascal OSStatus MNSSetPref(const MNSDatabadeRef *ref,
						const CfgEntityRef *entityID,
						OSType prefType,
						void *prefData, ByteCount prefSize)
	// See comment in header file.
{
	OSStatus err;
	OSStatus err2;
	CfgEntityAccessID prefsRefNum;

	MoreAssertQ(MNSValidDatabase(ref));
	MoreAssertQ(MNSDatabaseWritable(ref));
	MoreAssertQ(entityID != nil);
	//MoreAssertQ(prefData != nil);
	//MoreAssertQ(prefSize != nil);

	err = OTCfgOpenPrefs(ref->dbRef, entityID, true, &prefsRefNum);
	if (err == noErr) {
		err = OTCfgSetPrefs(prefsRefNum,  prefType, prefData, prefSize);
	
		err2 = OTCfgClosePrefs(prefsRefNum);
		if (err == noErr) {
			err = err2;
		}
	}
	
	return err;
}

extern pascal OSStatus MNSGetEntitiesList(const MNSDatabadeRef *ref,
								OSType entityClass, OSType entityType,
								ItemCount *entityCount,
								CfgEntityRef **entityIDs,
								CfgEntityInfo **entityInfos)
	// See comment in header file.
{
	OSStatus err;
	CfgEntityRef *paramIDs;
	CfgEntityInfo *paramInfos;

	MoreAssertQ(MNSValidDatabase(ref));
	MoreAssertQ(entityCount != nil);
	MoreAssertQ(entityIDs == nil   || *entityIDs == nil  );
	MoreAssertQ(entityInfos == nil || *entityInfos == nil);

	err = OTCfgGetEntitiesCount(ref->dbRef, ref->area, entityClass, entityType, entityCount);
	if (err == noErr) {
		if (entityIDs == nil) {
			paramIDs = nil;
		} else {
			*entityIDs = (CfgEntityRef *) NewPtr(*entityCount * sizeof(CfgEntityRef));
			if (*entityIDs == nil) {
				MoreAssertQ(err == memFullErr);
				err = memFullErr;
			}
			paramIDs = *entityIDs;
		}
		if (entityInfos == nil) {
			paramInfos = nil;
		} else {
			*entityInfos = (CfgEntityInfo *) NewPtr(*entityCount * sizeof(CfgEntityInfo));
			if (*entityInfos == nil) {
				MoreAssertQ(err == memFullErr);
				err = memFullErr;
			}
			paramInfos = *entityInfos;
		}
	}
	if (err == noErr) {
		err = OTCfgGetEntitiesList(ref->dbRef, ref->area, 
					entityClass, entityType, 
					entityCount, paramIDs, paramInfos);
	}
	
	// Clean up.
	
	if (err != noErr) {
		if (*entityIDs != nil) {
			DisposePtr( (Ptr) *entityIDs);
			MoreAssertQ(MemError() == noErr);
			*entityIDs = nil;
		}
		if (*entityInfos != nil) {
			DisposePtr( (Ptr) *entityInfos);
			MoreAssertQ(MemError() == noErr);
			*entityInfos = nil;
		}
	}

	return err;
}

extern pascal OSStatus MNSFindActiveSet(const MNSDatabadeRef *ref, CfgEntityRef *activeSet)
	// See comment in header file.
{
	OSStatus err;
	ItemCount setCount;
	CfgEntityRef *setEntities;
	Boolean found;
	ItemCount thisSetIndex;
	CfgSetsStruct thisStruct;

	MoreAssertQ(MNSValidDatabase(ref));
	MoreAssertQ(activeSet != nil);
	
	setEntities = nil;

	err = MNSGetEntitiesList(ref, kOTSetOfSettingsClass, kOTSetOfSettingsType, &setCount, &setEntities, nil);
	if (err == noErr) {
		thisSetIndex = 0;
		found = false;
		while (err == noErr && thisSetIndex < setCount && ! found) {
			err = MNSGetFixedSizePref(ref, &setEntities[thisSetIndex], kPrefsTypeStruct,
							&thisStruct, sizeof(thisStruct));
			if (err == noErr) {
				found = ((thisStruct.fFlags & (1 << kSetsIndexActive)) != 0);
				if ( ! found ) {
					thisSetIndex += 1;
				}
			}
		}
		if (err == noErr && ! found) {
			err = -2;
		}
	}
	if (err == noErr) {
		*activeSet = setEntities[thisSetIndex];
	}

	// Clean up.
	
	if (setEntities != nil) {
		DisposePtr( (Ptr) setEntities);
		MoreAssertQ(MemError() == noErr);
	}
	
	return err;
}

extern pascal OSStatus MNSIterateSet(const MNSDatabadeRef *ref,
									const CfgEntityRef *setEntity,
									MNSSetIterator iteratorProc,
									void *refcon,
									Boolean writeAfterIterate)
	// See comment in header file.
{
	OSStatus err;
	CfgSetsVector *vectorPrefData;
	ByteCount vectorPrefSize;
	ItemCount thisElementIndex;

	MoreAssertQ(MNSValidDatabase(ref));
	MoreAssertQ( !writeAfterIterate || MNSDatabaseWritable(ref) );
	MoreAssertQ(setEntity != nil);
	MoreAssertQ(iteratorProc != nil);
	
	vectorPrefData = nil;

	err = MNSGetPref(ref, setEntity, kPrefsTypeVector,
					&vectorPrefData, &vectorPrefSize);
	if (err == noErr) {
		for (thisElementIndex = 0; thisElementIndex < vectorPrefData->fCount; thisElementIndex++) {
			iteratorProc(ref, &vectorPrefData->fElements[thisElementIndex], refcon);
		}
		if (writeAfterIterate) {
			err = MNSSetPref(ref, setEntity, kPrefsTypeVector, vectorPrefData, vectorPrefSize);
		}
	}
	
	// Clean up.
	
	if (vectorPrefData != nil) {
		DisposePtr( (Ptr) vectorPrefData);
		MoreAssertQ(MemError() == noErr);
	}
	return err;
}
