/*____________________________________________________________________________
	Copyright (C) 1997 Network Associates Inc. and affiliated companies.
	All rights reserved.

	$Id: CTriangleControl.cp,v 1.2 1999/03/10 02:41:37 heller Exp $
____________________________________________________________________________*/
// ===========================================================================
//	File:					CTriangleControl.cp
//  Version:				1.0 - Dec 5, 1994
//
//	Copyright (C)1994 Mike Shields All rights reserved.
//	You may use the source code included in this archive in any application (commercial, 
//	shareware, freeware, postcardware, etc) when delivered an executable Macintosh 
//	application. You must attribute use of this source code in your copyright notice. 
//	You may not sell this source code in any form. This source code may be placed on 
//	publicly accessable archive sites and source code disks. It may not be placed on 
//	for profit archive sites and source code disks without the permission of the author, 
//	Mike Shields.
//	
//	    This source code is distributed in the hope that it will be useful,
//	    but WITHOUT ANY WARRANTY; without even the implied warranty of
//	    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// ===========================================================================
//	CTriangleControl.h		<- double-click + Command-D to see class declaration
//

// CTriangleControl is my implementation of the rotating triangle thingy you see in the Finder
// for opening folders in list views.

// Improvements coming soon:
// Add in the ability to set the hilite and standard colors from a 'PPob' resource and 
// programmatically. This way you can have colors other than the blues like the finder.

#include "CTriangleControl.h"

const RGBColor cBlue = { 0xCCCC, 0xCCCC, 0xFFFF };
const RGBColor cDarkBlue = { 0x6666, 0x6666, 0xFFFF };

// ---------------------------------------------------------------------------
//	CreateEditFieldStream
// ---------------------------------------------------------------------------
//	Create a new Triangle object from the data in a Stream

CTriangleControl *CTriangleControl::CreateTriangleControlStream(LStream *inStream)
{
	return (new CTriangleControl(inStream));
}

// ---------------------------------------------------------------------------
//	CTriangleControl
// ---------------------------------------------------------------------------
//	Default Constructor

CTriangleControl::CTriangleControl()
{
	mTriangleToDraw = NULL;
	mMaxValue = 1;
}


// ---------------------------------------------------------------------------
//	CTriangleControl(const CTriangleControl&)
// ---------------------------------------------------------------------------
//	Copy Constructor

CTriangleControl::CTriangleControl(const CTriangleControl &inOriginal)
		: LControl(inOriginal)
{
	InitTriangleControl();
}


// ---------------------------------------------------------------------------
//	CTriangleControl(SPaneInfo&, MessageT, Int32, Int32, Int32)
// ---------------------------------------------------------------------------
//	Construct from input parameters

CTriangleControl::CTriangleControl(const SPaneInfo &inPaneInfo, MessageT inValueMessage,
									Int32 inValue, Int32 inMinValue, Int32 inMaxValue)
		: LControl(inPaneInfo, inValueMessage, inValue, inMinValue, inMaxValue)
{
	InitTriangleControl();
}


// ---------------------------------------------------------------------------
//	CTriangleControl(LStream*)
// ---------------------------------------------------------------------------
//	Construct from a data stream

CTriangleControl::CTriangleControl(LStream *inStream)
		: LControl(inStream)
{
	mTriangleToDraw = NULL;
	mMaxValue = 1;
	InitTriangleControl();
}


// ---------------------------------------------------------------------------
//	InitTriangleControl
// ---------------------------------------------------------------------------
//	Private initializer

void CTriangleControl::InitTriangleControl()
{
	if(mTriangleToDraw)
		::KillPoly(mTriangleToDraw);
	// our tringle will always be drawn at fixed coords from the frame.
	// We use a polygon because I wanted a self contained object. 
	// I didn't want to rely on icons in the resource file to be loaded 
	// or anything like that.
	mTriangleToDraw = ::OpenPoly();
	if ( mValue == 0 )
	{
		// construct the "closed" triangle
		::MoveTo(3, 0);
		::LineTo(3, 10);
		::LineTo(8, 5);
		::LineTo(3, 0);
	}
	else
	{
		// construct the "open" triangle
		::MoveTo(0, 3);
		::LineTo(10, 3);
		::LineTo(5, 8);
		::LineTo(0, 3);
	}
	::ClosePoly();
}
	
// ---------------------------------------------------------------------------
//	~CTriangleControl
// ---------------------------------------------------------------------------
//	Destructor

CTriangleControl::~CTriangleControl()
{
}

// ---------------------------------------------------------------------------
//	SetValue
// ---------------------------------------------------------------------------
//	Specify the value for a Control
//
void CTriangleControl::SetValue(Int32 inValue)
{
	LControl::SetValue(inValue);
	
	// draw to reflect the change in state
	if ( FocusDraw() )
	{
		InitTriangleControl();
		DrawSelf();
	}
}

// ---------------------------------------------------------------------------
//	FindHotSpot
// ---------------------------------------------------------------------------
//	Determine which hot spot, if any, contains the specified point
//
Int16 CTriangleControl::FindHotSpot(Point /*inPoint*/)
{
	// the whole frame is the one and only hot spot
	return 1;
}

// ---------------------------------------------------------------------------
//	PointInHotSpot
// ---------------------------------------------------------------------------
//	Determine if a point is within a specified hot spot
//
Boolean CTriangleControl::PointInHotSpot(Point inPoint, Int16 /*inHotSpot*/)
{
	Point	portPt = inPoint;
	LocalToPortPoint(portPt);

	// the whole frame is the hot spot
	return PointIsInFrame(portPt.h, portPt.v);
}

// ---------------------------------------------------------------------------
//	HotSpotAction
// ---------------------------------------------------------------------------
//	Take action during mouse down tracking
//
void CTriangleControl::HotSpotAction(Int16 /*inHotSpot*/, Boolean inCurrInside, Boolean inPrevInside)
{
	// if we have moved in our out of our frame from where we were previously, then we
	// need to redraw ourselves to reflect the state change
	if ( inCurrInside != inPrevInside )
	{		
		if ( FocusDraw() )
		{
			DrawTriangle(inCurrInside);
		}
	}		
}

// ---------------------------------------------------------------------------
//	HotSpotResult
// ---------------------------------------------------------------------------
//	Perform result of clicking and releasing mouse inside a HotSpot
//
void CTriangleControl::HotSpotResult(Int16 /*inHotSpot*/)
{
	if ( FocusDraw() )
	{
		Rect	frame;
		CalcLocalFrameRect(frame);
		
		// Get rid of the current triangle
		::KillPoly(mTriangleToDraw);	
			
		// make intermediate triangle
		mTriangleToDraw = ::OpenPoly();
		::MoveTo(0, 8);
		::LineTo(8, 8);
		::LineTo(8, 0);
		::LineTo(0, 8);	
		::ClosePoly();
		
		::EraseRect(&frame);
		DrawTriangle(false);	
		
		// wait a few ticks to finish the animation
		for(short i=0;i<8;i++)
			::YieldToAnyThread();
		
		// make final triangle
		::KillPoly(mTriangleToDraw);	

		mTriangleToDraw = ::OpenPoly();
		if ( mValue == 1 )
		{
			// make the "unopened" triangle
			::MoveTo(3, 0);
			::LineTo(3, 10);
			::LineTo(8, 5);
			::LineTo(3, 0);
		}
		else
		{
			// make the "opened" triangle
			::MoveTo(0, 3);
			::LineTo(10, 3);
			::LineTo(5, 8);
			::LineTo(0, 3);
		}
		::ClosePoly();

		// draw the final triangle
		::EraseRect(&frame);
		DrawTriangle(false);
	}
	
	// set the value to our opposite
	SetValue((mValue ? 0 : 1 ));
}

// ---------------------------------------------------------------------------
//	DrawSelf
// ---------------------------------------------------------------------------
//	Draw the Triangle
//
void CTriangleControl::DrawSelf()
{
	DrawTriangle(false);
}

// ---------------------------------------------------------------------------
//	DrawTriangle
// ---------------------------------------------------------------------------
//	Draw the Triangle
//
void CTriangleControl::DrawTriangle(Boolean hilite)
{
	if ( mTriangleToDraw != NULL ) 
	{
		Rect	frame;
		if ( CalcLocalFrameRect(frame) ) 
		{	
			// set us up to draw. Make sure we draw in the correct spot in relation to 
			// where the frame is right now.  Restore the polygon after we are done.
			::OffsetPoly(mTriangleToDraw, frame.left, frame.top);
			
			//StPenState::Normalize();
			
			// We've got color, so draw the "blue" triangle
			if ( UEnvironment::HasFeature(env_SupportsColor) )
			{
				ApplyForeAndBackColors();
			
				// we fill with a different color depending on the hilite state
				::RGBForeColor((hilite ? &cDarkBlue : &cBlue));
				::FillPoly(mTriangleToDraw, &(UQDGlobals::GetQDGlobals()->black));
		
				ApplyForeAndBackColors();
				//StColorState::Normalize();
				::FramePoly(mTriangleToDraw);
			}
			else
			{
				if ( hilite )
				{
					::FillPoly(mTriangleToDraw, &(UQDGlobals::GetQDGlobals()->black));
				}
				::FramePoly(mTriangleToDraw);
			}
			
			// restore the polygon
			::OffsetPoly(mTriangleToDraw, -frame.left, -frame.top);
		}
	}
}

