/*____________________________________________________________________________
	Copyright (C) 1996-1998 Network Associates, Inc. and its affiliates.
	All rights reserved.
	
	$Id: CDrive.cp,v 1.10.8.2 1998/11/13 02:49:38 heller Exp $
____________________________________________________________________________*/

#include <stddef.h>
#include <Math64.h>
#include "MacEnvirons.h"

#include "CipherProc.h"
#include "CDrive.h"
#include "CipherContext.h"
#include "A4Stuff.h"
#include "SetupA4.h"
#include "PGPDisk.h"
#include "MacMemory.h"
#include "MacStrings.h"
#include "MacFiles.h"
#include "PGPDiskEncryptDecrypt.h"
#include "MacInterrupts.h"
#include "CCloseQueue.h"
#include "CChangeModDateQueue.h"
#include "UInt64.h"

#include "PGPDiskDRVRMain.h"
#include "AddDriveStruct.h"
#include "CBlindWriteQueue.h"


#define MAX( a, b ) 	( (a) >= (b) ? (a) : (b) )
#define MIN( a, b ) 	( (a) <= (b) ? (a) : (b) )


#if PGP_DEBUG
#include "MacEvents.h"
inline Boolean	AlphaLockIsDown( void )
					{ return( (GetAllModifiers() & alphaLock ) != 0 ); }
#endif


	static void
InitDeferredTask(
	DeferredTask *	defTask,
	DeferredTaskUPP	upp,
	long			param )
	{
	defTask->qLink		= 0;
	defTask->qType		= dtQType;
	defTask->dtFlags	= 0;
	defTask->dtParam	= param;
	defTask->dtAddr		= upp;
	defTask->dtReserved	= 0;
	}
	
	
#if CDRIVE_WATCHDOG
	static void
InitTMTask(
	TMTask *	tmTask,
	TimerUPP	upp)
	{
	pgpClearMemory( tmTask, sizeof( *tmTask ) );
	tmTask->tmAddr	= upp;
	}
#endif


#pragma mark -


void *					CDrive::sScratchBuffer			= nil;
ulong					CDrive::sScratchBufferSize		= 0;
CCloseQueue *			CDrive::sCloseQueue				= nil;
CChangeModDateQueue *	CDrive::sModDateQueue			= nil;


#if PGP_DEBUG

	void
CDrive::sAssertValid( const CDrive *drive )
	{
	pgpAssertAddrValid( drive, short);
	pgpAssert( drive->mMagic == kDriveMagic );
	}
	
#endif


/*_________________________________________________________________________
	Make a copy of the passed extents
_________________________________________________________________________*/
	OSErr
CDrive::InitExtents(
	const DiskExtent *	src,
	ulong				numExtents )
	{
	ulong	extentSize	= numExtents * sizeof( src[ 0 ] );
	OSErr	err	= noErr;
	
	mNumExtents	= numExtents;
	
	mExtents			= (const DiskExtent *)NewPtrSys( extentSize );
	if ( IsntNull( mExtents ) )
		{
		BlockMoveData( src, (Ptr)mExtents, extentSize );
		}
	else
		{
		err	= memFullErr;
		}
		
	return( err );
	}



/*_________________________________________________________________________
	Initialize the string which the user sees when using "Get Info".
	
	Important: do *not* use the file manager here.  Doing so can cause
	deadlock under interrupt driven driver I/O.
_________________________________________________________________________*/
	OSErr
CDrive::InitVolumeInfo( short fileRefNum )
	{
	FSSpec	spec;
	OSErr	err	= noErr;
	
	err	= GetSpecFromFCBTable( fileRefNum, &spec );
	if ( IsntErr( err ) )
		{
		Str255	volumeName;
		
		const VCB *	vcb	= GetVCBForVolume( spec.vRefNum );
		
		pgpAssert( IsntNull( vcb ) );
		CopyPString( vcb->vcbVN, volumeName );
		CopyPString( vcb->vcbVN, mVolumeName );
		
		if ( IsntErr( err ) )
			{
			// initialize mVolumeString to read:
			//  "PGPDisk file name on volume Fred" 
			CopyPString( "\p\322", mVolumeString);
			AppendPString( spec.name, mVolumeString);
			AppendPString( "\p\323 on  volume \322", mVolumeString);
			AppendPString( volumeName, mVolumeString);
			AppendPString( "\p\323", mVolumeString);
			
			// any additional info can be appended here:
			AppendPString( "\p version ", mVolumeString);
			NumVersion		version;
			Str255			versionString;
			*(ushort *)&version	= kPGPDiskDriverVersion;
			NumVersionToString( &version, versionString );
			AppendPString( versionString, mVolumeString);
			}
		}
	return( err );
	}
	
	
/*_________________________________________________________________________
	Initialize both cipher contexts and the variable which tracks which one
	has its bits flipped.
_________________________________________________________________________*/
	OSErr
CDrive::InitCipherContext(
	const CASTKey *			key,
	const PassphraseSalt *	salt)
	{
	OSErr	err	= noErr;
	
	pgpAssert( mCipherProcRef != kInvalidCipherProcRef );
	
	// initialize our two cipher contexts. One is correct and the other
	// has its bits flipped and we periodically reverse them
	mWhichCipherContextValid	= 0;
	mCipherContext[ 0 ].salt	= *salt;
	err	= CipherProc_CallInitContext( mCipherProcRef, &mCipherContext[ 0 ],
				key->keyBytes);
		
		
	// make the 2nd one be the same, then flip its bits
	mCipherContext[ 1 ]	= mCipherContext[ 0 ];
	FlipBytes( (void *)mCipherContext[ 1 ].castCFB.expandedKey.keyWords,
		sizeof( mCipherContext[ 1 ].castCFB.expandedKey.keyWords ) );
	
	return( err );
	}
	


/*_________________________________________________________________________
	Initialize everything using the supplied info.
	
	Important: any error conditions must return an appropriate error code.
_________________________________________________________________________*/
	OSStatus
CDrive::Init( const InitInfo * init )
	{
	const AddDriveStruct *	info	= init->info;
	OSStatus				err	= noErr;
	const VCB *				vcb	= nil;
	
	pgpAssertAddrValid( info, CDrive::InitInfo );
	pgpAssertAddrValid( init->pbArray, CDrive::MyPB );
	pgpAssert( init->numPBs >= info->numExtents );
	pgpAssert( init->cipherProcRef != kInvalidCipherProcRef );
	AssertFileRefNumIsValid( info->fileRefNum, "CDrive::CDrive" );
	pgpAssertAddrValid( info->extents, DiskExtent );
	pgpAssert( info->numExtents != 0 );


	// all drives share sScratchBuffer, but setting it is OK
	
	mCipherProcRef		= init->cipherProcRef;
	mFileRefNum			= info->fileRefNum;
	mPhysicalIcon		= info->physicalIcon;
	mMediaIcon			= info->mediaIcon;
	mWriteProtected		= info->mountReadOnly;
	mBreakUpPBs			= info->shouldBreakUpRequests;
	
	vcb	= GetVCBForFile( info->fileRefNum );
	pgpAssert( IsntNull( vcb ) );
	
	
	mBaseDriveVCB			= vcb;
	mBaseDriveNumber		= vcb->vcbDrvNum;
	mBaseDriverRefNum		= vcb->vcbDRefNum;
	
	const FlagsDrvQEl *	flagsEl	= (const FlagsDrvQEl *)
		( ((const char *)::GetDriveQElem( mBaseDriveNumber )) -
		offsetof( FlagsDrvQEl, el ) );
	mBaseDriveIsEjectable =
		( flagsEl->flags & kDriveIsNotEjectableMask ) == 0;
	
	SetPBs( init->pbArray, init->numPBs );
	
	if ( IsntErr( err ) )
		{
		err	= InitCipherContext( &info->decryptionKey, &info->salt );
		AssertNoErr( err, "CDrive::Init, InitCipherContext" );
		}

	if ( IsntErr( err ) )
		{
		err	= InitExtents( info->extents, info->numExtents );
		AssertNoErr( err, "CDrive::Init, InitExtents" );
		}

	if ( IsntErr( err ) )
		{
		err	= InitVolumeInfo( info->fileRefNum );
		AssertNoErr( err, "CDrive::Init, InitVolumeInfo" );
		}
	
	AssertNoErr( err, "CDrive::Init" );
	return( err );
	}





/*_________________________________________________________________________
	Full initialization will be done later via Init().
_________________________________________________________________________*/
CDrive::CDrive(  )
	{
	PrepareCallback();
	
#if PGP_DEBUG
	mMagic				= kDriveMagic;
#endif
	
	mVCB				= nil;
	mClientPB			= nil;
	mPBArray			= nil;
	mNumPBs				= 0;
	mBaseDriveNumber	= 0;
	mBaseDriverRefNum	= 0;
	mMyDQE				= nil;
	mCipherProcRef		= kInvalidCipherProcRef;
	mFileRefNum			= 0;
	mExtents			= 0;
	mNumExtents			= 0;
	mReadTuner			= kDefaultReadTunerStruct;
	mWriteTuner			= kDefaultWriteTunerStruct128;
	mVolumeString[ 0 ]	= 0;
	mHasBeenMounted		= false;
	mUnmountedAppPSN.highLongOfPSN	= 0;
	mUnmountedAppPSN.lowLongOfPSN	= kNoProcess;
	mLastWriteTime				= 0;
	mLastFileModDateUpdate		= 0;
	mBreakUpPBs			= TRUE;
	mDecryptLock		= false;
	mEncryptLock		= false;
#if CDRIVE_WATCHDOG
	mWatchdog.installed			= false;
	// note: supposedly, the problem we're working around is fixed in 7.6.
	// it's also fixed on certain machines with certain systems, (PCI Macs
	// for example), but it's not clear which ones. We should be able to set
	// this variable to false if we recognize a system without the problem
	mWatchdog.shouldUseWatchdog	= GetSystemVersion() < 0x0760;
#endif
	
	pgpClearMemory( &mCipherContext, sizeof( mCipherContext ) );
	pgpClearMemory( &mPhysicalIcon, sizeof( mPhysicalIcon ) );
	pgpClearMemory( &mMediaIcon, sizeof( mMediaIcon ) );

#if ALLOW_VM_PAGING
	mVirtualMemoryIsOn	= VirtualMemoryIsOn();
#endif

	mReinsertNotificationPending	= false;
	}
	
	
CDrive::~CDrive()
	{
	OSErr	err;
	
	if ( IsntNull( mExtents ) )
		{
		DisposePtr( (Ptr)mExtents );
		mExtents	= nil;
		}
	
	mPBArray	= nil;	// we didn't allocate it
	
	
	// don't leave key around in memory!
	pgpClearMemory( &mCipherContext[ 0 ], sizeof( mCipherContext ) );
	
	
	DrvQEl *	tempQElem;
	if( IsntNull( mMyDQE ) )
		{
		err	= RemoveDriveQEl( mMyDQE->el.dQDrive, &tempQElem);
		AssertNoErr( err, "CDrive::~CDrive()" );
		if ( IsntErr( err ) )
			{
			pgpAssert( tempQElem == &mMyDQE->el );
			}
		DisposePtr( (Ptr)mMyDQE );
		mMyDQE			= nil;
		}
	
	// the following don't belong to us
	mDCE			= nil;
	
#if PGP_DEBUG
	mMagic			= ~kDriveMagic;
#endif
	
	
	sCloseQueue->Close( mFileRefNum );
	mFileRefNum	= 0;
	}



	void
CDrive::sSetScratchBuffer(
	void *	buffer,
	ulong	bufferSize )
	{
	sScratchBuffer		= buffer;
	sScratchBufferSize	= bufferSize;
	}


	void
CDrive::SetPBs(
	void *	pbArray,
	ushort	numPBs )
	{
	mPBArray	= (MyPB *)pbArray;
	mNumPBs		= numPBs;
	}


	static Boolean
TunerValid( const TunerStruct *tuner )
	{
	Boolean	isValid	= TRUE;
	
	for( ushort index = 0; index < TunerStruct::kNumTune; ++ index )
		{
		if( tuner->maxBlocks[ index ] == 0 )
			{
			isValid	= false;
			break;
			}
		}
	
	return( isValid );
	}
	
	
	void
CDrive::SetReadTuner( const TunerStruct *tuner )
	{
	if ( TunerValid( tuner ) )
		{
		mReadTuner	= *tuner;
		}
	}
	
	
	void
CDrive::SetWriteTuner( const TunerStruct *tuner )
	{
	if ( TunerValid( tuner ) )
		{
		mWriteTuner	= *tuner;
		}
	}
	

	void
CDrive::DecryptPB( const MyPB *myPB )
	{
	ulong		numBlocks;
	uchar *		buffer;
	
	numBlocks	= GetPBNumBlocks( myPB );
	buffer		= (uchar *)myPB->ioPB.pb.ioParam.ioBuffer;
	
	CipherProc_CallDecrypt( mCipherProcRef, GetCipherContext(),
			myPB->startBlockOnSelf, numBlocks, buffer, buffer);
	}







	static asm void 
CallIODone(
	DCtlEntry *	dce,
	OSErr		result)
{
	FRALLOC +
	move.l	dce, a1
	move.w	result, d0
	FRFREE
	move.l	0x08FC, -(A7)
	RTS
	// jIODone returns to caller of this routine
}



/*_________________________________________________________________________
	See ScheduleRetry()
_________________________________________________________________________*/
	void
CDrive::RetryTMTask( MyPB *myPB )
	{
	RmvTime( (QElemPtr) &myPB->retry.tmTask );
	
	// must reset the posOffset because the driver we called may have whacked it
	SetPosOffset( &myPB->ioPB, myPB->retry.startBlock,
			myPB->ioPB.pb.ioParam.ioReqCount / kPGPDiskBlockSize );
	
	if ( mIODirection == kReadDirection )
		PBReadAsync( & myPB->ioPB.pb );
	else
		PBWriteAsync( & myPB->ioPB.pb );
	}


/*_________________________________________________________________________
	See ScheduleRetry()
_________________________________________________________________________*/
	void
CDrive::sRetryTMTask( void )
	{
	TMTask *tmTask	= (TMTask *)GetA1();
	
	EnterCodeResource();
	
	MyPB *	myPB	= (MyPB *)((Ptr)tmTask - offsetof( MyPB, retry.tmTask ));
	pgpAssert( myPB->magic == MyPB::kMagic );
	
	myPB->theDrive->RetryTMTask( myPB );
		
	ExitCodeResource();
	}


/*_________________________________________________________________________
	An error was returned by the base driver for a parameter block.   
	Try the I/O again. This case can occur in the following situations:
	- ioErr on a flaky disk
	- -4 error with FWB driver in certain situations (unknown cause)
	- possibly other errors
	
	As far as the rest of our code is concerned, the request never completed,
	so this retry need only try it again without noting the fact.  But we want
	to introduce a delay so that the problem situation is hopefully avoided.
_________________________________________________________________________*/
	void
CDrive::ScheduleRetry( MyPB *myPB )
	{
	myPB->retry.count	+= 1;
	
	InitTMTask( &myPB->retry.tmTask, (TimerUPP)sRetryTMTask );
	InsTime( (QElemPtr) &myPB->retry.tmTask );
	
	enum { kRetryDelay = 5 };	// milliseconds
	PrimeTime( (QElemPtr)&myPB->retry.tmTask, kRetryDelay );
	}
	
	
/*_________________________________________________________________________
	A parameter block has completed
_________________________________________________________________________*/
	void
CDrive::HandleCompletion( MyPB *myPB )
	{
	OSErr	err	= myPB->ioPB.pb.ioParam.ioResult;
	
	pgpa((
		pgpaAssert( myPB->magic == MyPB::kMagic ),
		pgpaAssert( myPB->ioPB.pb.ioParam.ioActCount ==
				myPB->ioPB.pb.ioParam.ioReqCount )
		));
	
	ushort	maxRetry	= MyPB::kMaxRetryCount;
	// FWB hack; may just be due to bad SCSI connection
	if ( err == -4 )
		{
		maxRetry	= MAX( maxRetry, 1000 );
		}
		
	if ( IsErr( err ) &&
		myPB->retry.count < maxRetry )
		{
		#if PGP_DEBUG
		if ( myPB->retry.count > 4 )
			{
			pgpDebugFmtMsg( ( pgpaFmtPrefix,
				"retry failed %lu times with error %ld (will retry)",
				(ulong)myPB->retry.count, (long)err ) );
			}
		#endif
		ScheduleRetry( myPB );
		}
	else
		{
		// remember the last non-normal error that occurred...should be
		// 'noErr' we'll return it as the result of the original client read
		if ( IsErr( err ) )
			{
			pgpDebugFmtMsg( ( pgpaFmtPrefix,
				"retry failed %lu times with error %l",
				(ulong)myPB->retry.count, (long)err ) );
			
			mResultErr	= err;
			}
			
		if ( mIODirection == kReadDirection )
			{
			HandleReadCompletion( myPB );
			}
		else
			{
			pgpAssert( mIODirection == kWriteDirection );
			HandleWriteCompletion( myPB );
			}
		}
	}


/*_________________________________________________________________________
	An I/O request has been completed.
	This hook proc is called by Device Manager
_________________________________________________________________________*/
	pascal void
CDrive::sCompletionProc( void )
	{
	EnterCallback();
	
	XIOPB *	pb;
	MyPB *			myPB;
	
	pb	= (XIOPB *)GetA0();
	
	myPB	= (MyPB *)(((char *)pb) - offsetof( MyPB, ioPB ));
	pgpAssert( myPB->magic == MyPB::kMagic );
	
	CDrive::sAssertValid( myPB->theDrive );
	myPB->theDrive->HandleCompletion( myPB );
	
	ExitCallback();
	}



	
/*_________________________________________________________________________
	Find the extent which contains the data corresponding to the block at
	offset 'blockIndexOnOurDrive' in our drive (not the base drive).
_________________________________________________________________________*/
	const DiskExtent *
CDrive::FindExtentContainingBlock(
	const ulong		blockIndex,
	ulong *			startBlockOfExtentPtr)
	{
	ulong	blocksCoveredSoFar		= 0;
	
	pgpAssert( blockIndex < GetLogicalSizeInBlocks() );
	
	for( ulong extentIndex = 0; extentIndex < mNumExtents; ++extentIndex )
		{
		const DiskExtent *	extent	= nil;
		
		extent	= &mExtents[ extentIndex ];
		
		if ( blockIndex < blocksCoveredSoFar + extent->numBlocks )
			{
			*startBlockOfExtentPtr	= blocksCoveredSoFar;
			return( extent );
			}
		
		blocksCoveredSoFar	+= extent->numBlocks;
		}
	
	pgpDebugFmtMsg( (pgpaFmtPrefix,
		"CDrive::FindExtentContainingOffset: illegal blockIndex %ul",
		blockIndex) );
	return( nil );
	}


	CDrive::MyPB *
CDrive::GetIndPB( ushort index )
	{
	pgpAssert( index < mNumPBs );
	return( &mPBArray[ index ] );
	}


	void
CDrive::SetPosOffset(
	XIOPB *	pb,
	ulong	startBlock,
	ulong	numBlocks)
	{
	if ( NeedWidePositioning( startBlock, numBlocks ) )
		{
		CUInt64		startOffset;
		
		pb->xpb.ioPosMode		= fsFromStart | kWidePosOffsetMask;
		
		// multiplication not yet defined for UInt64 so use equivalent
		// shift operator
		pgpAssert( kPGPDiskBlockSize == 512 );	// 2^9
		startOffset		= CUInt64( 0, startBlock ) << 9;
		pb->xpb.ioWPosOffset		= startOffset;
		}
	else
		{
		pb->pb.ioParam.ioPosMode	= fsFromStart;
		pb->pb.ioParam.ioPosOffset	= startBlock * kPGPDiskBlockSize;
		}
	}
	
/*_________________________________________________________________________
	Fill in a parameter block in preparation for I/O, taking into account
	whether wide positioning is needed.
_________________________________________________________________________*/
	void
CDrive::FillInPB(
	MyPB *			myPB,
	const void *	src,
	Ptr				dest,
	ulong			startBlock,
	ulong			numBlocks )
	{
	IOParam *	io;
	
	myPB->src			= src;
	myPB->theDrive		= this;
	myPB->retry.count	= 0;
	myPB->retry.startBlock	= startBlock;
	SetPBMagic( myPB );
	
	io	= &myPB->ioPB.pb.ioParam;
	io->ioCompletion	= sCompletionProc;
	io->ioVRefNum		= GetBaseDriveNumber();
	io->ioRefNum		= GetBaseDriverRefNum();
	io->ioBuffer		= dest;
	io->ioReqCount		= numBlocks * kPGPDiskBlockSize;
	
	SetPosOffset( &myPB->ioPB, startBlock, numBlocks );
	}


/*_________________________________________________________________________
	Is an ejectable base drive available?  The media could have been ejected,
	and new media inserted in its place.
_________________________________________________________________________*/
	Boolean
CDrive::BaseDriveIsAvailable()
{
	Boolean	isAvailable	= TRUE;
	
	if ( mBaseDriveIsEjectable )
	{
		pgpAssertAddrValid( mBaseDriveVCB, VCB );
		isAvailable	= ( mBaseDriveVCB->vcbDrvNum != 0 );
	}
	
	return( isAvailable );
}


typedef struct
	{
	NMRec	nm;
	Str255	msg;
	} MyNMRec;
	
	
	void
CDrive::NMProc( NMRec *nmRec )
	{
	NMRemove( nmRec );
	mReinsertNotificationPending	= false;
	
	if ( ! BaseDriveIsAvailable() )
	{
		NotifyUserToReinsert( nmRec->nmStr );
	}
	
	DisposePtr( (Ptr)nmRec );
	}
	
	pascal void
CDrive::sNMProc( NMRec *nmRec )
	{
	CDrive *	self	= (CDrive *)nmRec->nmRefCon;
	
	EnterCallback();
	CDrive::sAssertValid( self );
	self->NMProc( nmRec );
	ExitCallback();
	}


	void
CDrive::NotifyUserToReinsert( ConstStr255Param	msg )
	{
	if ( ! mReinsertNotificationPending )
		{
		MyNMRec *	myRec	= nil;
			
		myRec	= (MyNMRec *)NewPtrSysClear( sizeof( *myRec ) );
		if ( IsntNull( myRec ) )
			{
			CopyPString( msg, myRec->msg );
			
			myRec->nm.qType		= nmType;
			myRec->nm.nmResp	= sNMProc;
			myRec->nm.nmStr		= myRec->msg;
			myRec->nm.nmRefCon	= (long)this;
			
			mReinsertNotificationPending	= TRUE;
			NMInstall( &myRec->nm );
			}
		}
	}
	
	
/*_________________________________________________________________________
	If the base drive has been ejected, we want to tell the user to put it
	back immediately.  We do so by posting a notification manager task.
	The task will keep reposting the alert until the user reinserts the disk.
_________________________________________________________________________*/
	void
CDrive::CheckBaseDriveEjected( ConstStr255Param	messageTemplate )
	{
	if ( ( ! BaseDriveIsAvailable() ) && ( ! mReinsertNotificationPending ) )
		{
		Str255		msg;

		CopyPString( messageTemplate, msg );
		PrintPString( msg, msg, mVCB->vcbVN );
		PrintPString( msg, msg, mBaseDriveVCB->vcbVN );
		PrintPString( msg, msg, mBaseDriveVCB->vcbVN );
		
		NotifyUserToReinsert( msg );
		}
	}
		


/*_________________________________________________________________________
	Initialize any common stuff in preparation for servicing a client read
	or write.
_________________________________________________________________________*/
	void
CDrive::InitReadOrWrite(
	ParamBlockRec *		clientPB,
	DCtlEntry *			dce,
	IODirection			direction)
	{
	pgpAssert( mIOQueue.IsEmpty() );
	pgpAssert( mProcessingQueue.IsEmpty() );
	
#if INSTRUMENT
	Microseconds( &mStartTime );
#endif

	clientPB->ioParam.ioResult	= kAsyncIOInProgress;
	
	GetDateTime( &mLastAccessTime );
	
	mIODirection	= direction;
	mClientPB		= clientPB;
	mDCE			= dce;
	mResultErr		= noErr;

#if ALLOW_VM_PAGING
	mTreatAsVMRequest	= mVirtualMemoryIsOn && 
						PBIsSync( clientPB ) &&
						direction == kReadDirection &&
						clientPB->ioParam.ioReqCount <= 4096 ;
#endif
	}

	
	void
CDrive::UpdatePBAndCallIODone(
	OSErr		resultErr,
	ulong		actualCount)
	{
	pgpAssert( IsntNull( mClientPB ) );
	
	mClientPB->ioParam.ioResult		= resultErr;
	mClientPB->ioParam.ioActCount	= actualCount;
	
	mClientPB	= nil;
	CallIODone( mDCE, resultErr );
	}
	
	
	
	
#pragma mark -


#pragma mark -
	
/*_________________________________________________________________________
	We've finished a client io request.  Clean up and notify client
_________________________________________________________________________*/
	void
CDrive::MyIOIsDone(
	OSErr	resultErr,
	ulong	actualCount )
	{
	IOParam *	clientPB	= (IOParam *)mClientPB;
	
	pgpa((
		pgpaAssert( IsntNull( clientPB ) ),
		pgpaAssert( actualCount == clientPB->ioReqCount ),
		pgpaAssert( resultErr != kAsyncIOInProgress )
		));
	
	LogMsg( "IOD [..." /*]*/);
	
		UpdatePBAndCallIODone( resultErr, actualCount );
	
	LogMsg(/*[*/ "...]" );
	
	// to match original read or write
	LogMsg( /*[*/ "...]\r" );
	}
	
	
	
	
	Boolean
CDrive::IsValidRequest( const IOParam *pb)
	{
	Boolean	isValid	= TRUE;
	UInt32	loOffset;
	
	if ( PBUsesWidePositioning( pb ) )
		{
		loOffset	= ((XIOParam *)pb)->ioWPosOffset.lo;
		}
	else
		{
		loOffset	= pb->ioPosOffset;
		}
	
	if ( ( loOffset % kPGPDiskBlockSize ) != 0 )
		{
		isValid	= false;
		pgpDebugPStr(
			"\pCDrive::IsValidRequest: offset is not a multiple of 512" );
		}
	else if ( ( pb->ioReqCount % kPGPDiskBlockSize) != 0 )
		{
		isValid	= false;
		pgpDebugPStr(
			"\pCDrive::IsValidRequest: request is not a multiple of 512" );
		}
	else
		{
		isValid	= ( ::GetPBStartBlock( pb ) + ::GetPBNumBlocks( pb ) ) <=
						GetLogicalSizeInBlocks();
		pgpDebugMsgIf( ! isValid,
			"\pCDrive::IsValidRequest: request beyond end of drive" );
		}
	
	pgpAssert( isValid );
	
	return( isValid );
	}




#pragma mark -



	pascal void
CDrive::sKickStartDeferredTask()
	{
	// register A1 contains task.dtParam
	CDrive *	theDrive	= (CDrive *)GetA1();
	IODirection	direction;
	
	EnterCodeResource();
	
	LogMsg( "KSDT [..." );
	
	CDrive::sAssertValid( theDrive );

#if CDRIVE_WATCHDOG
	theDrive->RemoveWatchdog();
#endif
	
	direction	= theDrive->GetIODirection();
	pgpAssert( direction == kReadDirection || direction == kWriteDirection );
	
	if ( direction == kReadDirection )
		{
		theDrive->KickStartDeferredRead();
		}
	else
		{
		theDrive->KickStartDeferredWrite();
		}
	
	LogMsg( "...]" );
	
	ExitCodeResource();
	}


	void
CDrive::InstallDeferredTask( 
	DeferredTask *	task,
	DeferredTaskUPP	upp )
	{
#if CDRIVE_WATCHDOG
	// note: we should install the watchdog *before* the deferred task
	InstallWatchdog( task );
#endif

	InitDeferredTask( task, upp, (long)this );

	OSErr	err	= DTInstall( task );
	AssertNoErr( err, "InstallDeferredTask" );
	}
	
	


#if CDRIVE_WATCHDOG	// [


	

	
	void
CDrive::WatchdogTask()
	{
	QHdr *	dtQueueHdr	= LMGetDTQueue();
	OSErr	err	= noErr;
	
	err	= Dequeue( (QElem *)mWatchdog.watchTask, dtQueueHdr );
	if ( IsntErr( err ) )
		{
		pascal void		(*taskProc)( void );
		ulong			taskParam;
		
		taskProc		= mWatchdog.watchTask->dtAddr;
		taskParam		= mWatchdog.watchTask->dtParam;
		SetA1( (void *)taskParam );
		(*taskProc)();
		}
	else
		{
		// task not present...can occur if task was dequeued by deffered
		// task mgr but has not yet had chance to remove this watchdog
		}
	}
	
	
	pascal void
CDrive::sWatchdogTMTask()
	{
	WatchdogTMTask *myTask	= (WatchdogTMTask *)GetA1();
	
	EnterCodeResource();
	
	CDrive *	theDrive	= myTask->thisObject;
	CDrive::sAssertValid( theDrive );
	theDrive->WatchdogTask();
	
	ExitCodeResource();
	}
	
	
/*_________________________________________________________________________
	
_________________________________________________________________________*/
	void
CDrive::InstallWatchdog( DeferredTask *watchTask )
	{
	if ( mWatchdog.shouldUseWatchdog )
		{
		pgpAssert( ! mWatchdog.installed );
		
		mWatchdog.installed			= TRUE;
		mWatchdog.watchTask			= watchTask;
		mWatchdog.task.thisObject	= this;
		InitTMTask( &mWatchdog.task.tmTask, (TimerUPP)sWatchdogTMTask );
		InsTime( (QElemPtr) &mWatchdog.task.tmTask );
		
		// the length of the delay should be long enough so that the watchdog
		// ideally never runs except when the system is deadlocked.
		enum { kWatchdogDelay = 500 };
		PrimeTime( (QElemPtr)&mWatchdog.task.tmTask, kWatchdogDelay );
		}
	}


	void
CDrive::RemoveWatchdog( )
	{
	if ( mWatchdog.shouldUseWatchdog )
		{
		pgpAssert( mWatchdog.installed );
		
		// a simple check suffices; it is only removed in one place
		if ( mWatchdog.installed )
			{
			RmvTime( (QElemPtr) &mWatchdog.task.tmTask );
			mWatchdog.installed		= false;
			}
		}
	}


#endif	// ] USE_TIME_MGR_KICK
	
#pragma mark -


	
/*_________________________________________________________________________
	Insert a pb at index, moving existing pbs up as needed.
_________________________________________________________________________*/
	void
CDrive::InsertPB(
	const ushort	pbIndex,
	const ushort	numPBsInArray,
	const MyPB *	newPB)
	{
	pgpAssert( pbIndex < mNumPBs );
	pgpAssert( pbIndex <= numPBsInArray );
	
	ulong	numPBsToMove	= numPBsInArray - pbIndex;
	
	if ( numPBsToMove != 0 )
		{
		BlockMoveData( &mPBArray[ pbIndex ], &mPBArray[ pbIndex + 1 ],
			numPBsToMove * sizeof( mPBArray[ 0 ] )  );
		}
		
	mPBArray[ pbIndex ]	= *newPB;
	}


// find the highest bit position in number
	static ulong
GetHighestBitNumber(ulong number)
	{
	ulong	temp	= number;
	ulong	bits	= 0;
	
	while ( temp != 0 )
		{
		++bits;
		temp	>>= 1;
		}
	return( bits );
	}
	
	
/*_________________________________________________________________________
	For an io request of size 'numBlocks' blocks, what is the maximum size
	I/O we should perform.
_________________________________________________________________________*/
	ulong
CDrive::GetBlockLimitForSize(
	const TunerStruct *	tuner,
	ulong				numBlocks)
	{
	ushort	bits	= 0;
	
	bits	= GetHighestBitNumber( numBlocks );
	pgpAssert( bits != 0 );
	
	if ( bits > TunerStruct::kNumTune - 1 )
		bits	= TunerStruct::kNumTune - 1;
		
	ulong	max	= tuner->maxBlocks[ bits ];
	
	return( max );
	}


/*_________________________________________________________________________
	Break up 'myPB' into two.
	
	If 'firstPieceBlocks' is specified, use that size for the first one.
	Otherwise, split the request in two.
_________________________________________________________________________*/
	void
CDrive::BreakUpPB(
	MyPB *				myPB,
	MyPB *				outPB,
	ulong				firstPieceBlocks)
	{
	Ptr		ioBuffer			= myPB->ioPB.pb.ioParam.ioBuffer;
	ulong	startBlock			= GetPBStartBlock( myPB );
	ulong	requestBlocks		= GetPBNumBlocks( myPB );
	ulong	startBlockOnSelf	= myPB->startBlockOnSelf;
	ulong	numBlocksInFirst;
	
	pgpAssert( firstPieceBlocks == 0 || (firstPieceBlocks != 0 &&
					firstPieceBlocks < requestBlocks ) );
	
	*outPB	= *myPB;
	
	if ( firstPieceBlocks == 0)
		numBlocksInFirst	= requestBlocks / 2;
	else
		numBlocksInFirst	= firstPieceBlocks;
		
	pgpAssert( numBlocksInFirst >= 1 );
	
	FillInPB( myPB, myPB->src, ioBuffer, startBlock, numBlocksInFirst );
	myPB->startBlockOnSelf	= startBlockOnSelf;
	
	ulong	bytesInFirst	= numBlocksInFirst * kPGPDiskBlockSize;
	FillInPB( outPB, (Ptr)myPB->src + bytesInFirst, ioBuffer + bytesInFirst,
		startBlock + numBlocksInFirst, requestBlocks - numBlocksInFirst);
	outPB->startBlockOnSelf	= startBlockOnSelf + numBlocksInFirst;
	}
	
	
/*_________________________________________________________________________
	Split the indexth pb into two if it qualifies for breakup.
_________________________________________________________________________*/
	Boolean
CDrive::BreakUpPBIfNeeded(
	const ulong			clientRequestBlocks,
	MyPB *				myPB,
	MyPB *				outPB,
	const TunerStruct *	tuner)
	{
	ulong		maxBlocks;
	ulong		requestBlocks;
	Boolean		didBreak	= false;
	
	pgpAssertAddrValid( tuner, TunerStruct);
	
	requestBlocks	= GetPBNumBlocks( myPB );
	
	maxBlocks		= GetBlockLimitForSize( tuner, clientRequestBlocks );
	if ( requestBlocks > maxBlocks )
		{
		BreakUpPB( myPB, outPB );
		didBreak	= TRUE;
		}
	
	return( didBreak );
	}
	
	
/*_________________________________________________________________________
	Loop once through the list of read pbs, breaking up any that exceed our
	size limits. Does not attempt to recursively break things up again.
_________________________________________________________________________*/
	ulong
CDrive::BreakUpPBs(
	const ulong			clientRequestBlocks,
	const ushort		numPBsIn,
	const TunerStruct *	tuner)
	{
	ushort	pbIndex	= 0;
	ushort	numPBs	= numPBsIn;
	
	// loop until we've done all pbs or until we run out of parameter blocks
	while( pbIndex < numPBs && numPBs < mNumPBs)
		{
		MyPB *	curPB;
		MyPB	tempPB;
		Boolean	didBreak;
		
		curPB		= GetIndPB( pbIndex );
		didBreak	= BreakUpPBIfNeeded( clientRequestBlocks, curPB, &tempPB,
							tuner);
		if ( didBreak )
			{
			// advance past cur and insert new item after it
			++pbIndex;	
			InsertPB( pbIndex, numPBs, &tempPB );
			
			++numPBs;
			}
			
		++pbIndex;
		}
		
	return( numPBs );
	}
	
	
	void
CDrive::BreakUpPBsIfNeeded(
	const ulong			clientRequestBlocks,
	const ushort		numPBsIn,
	ushort *			numPBsOut,
	const TunerStruct *	tuner)
	{
	pgpAssert( numPBsIn != 0 );
	pgpAssertAddrValid( numPBsOut, ushort );
	
	if ( mBreakUpPBs )
		{
		ulong numPBs	= numPBsIn;
	
		// iterate until all pbs meet our size requirement or we run out of
		// pbs
		while ( numPBs < mNumPBs )
			{
			ulong	newNumPBs;
			Boolean	noChange;
			
			newNumPBs	= BreakUpPBs( clientRequestBlocks, numPBs, tuner);
			noChange	= ( newNumPBs == numPBs );
			numPBs		= newNumPBs;

			if ( noChange )
				break;
			}
		*numPBsOut	= numPBs;
		}
	else
		{
		*numPBsOut	= numPBsIn;
		}
	
	}


/*_________________________________________________________________________
	Push the PB at index down the list until the list is sorted.
	
	Keeping the PBs sorted by ascending start block should help minimize
	head seeking.
_________________________________________________________________________*/
	void
CDrive::InsertLastPB( const ushort indexX )
	{
	ulong	startBlock;
	
#if PGP_DEBUG
	// list should already be in sorted order, except for this last one
	for ( short pbIndex = 0; pbIndex < indexX - 1; ++pbIndex )
		{
		pgpAssert( GetPBStartBlock( &mPBArray[ pbIndex ] ) <
				GetPBStartBlock( &mPBArray[ pbIndex + 1 ] ) );
		}
#endif
	
	startBlock	= GetPBStartBlock( GetIndPB( indexX ) );
	for ( short pbIndex = indexX - 1; pbIndex >= 0; --pbIndex )
		{
		if ( startBlock < GetPBStartBlock( &mPBArray[ pbIndex ] ) )
			{
			MyPB	tempPB;
			
			// swap current element and following one
			tempPB					= mPBArray[ pbIndex ];
			mPBArray[ pbIndex ]		= mPBArray[ pbIndex + 1 ];
			mPBArray[ pbIndex + 1 ]	= tempPB;
			}
		else
			{
			// now in sorted order
			break;
			}
		}
#if PGP_DEBUG
	if ( indexX >= 1 )
		{
		// list should now be in sorted order
		for ( short pbIndex = 0; pbIndex < indexX; ++pbIndex )
			{
			pgpAssert( GetPBStartBlock( &mPBArray[ pbIndex ] ) <
					GetPBStartBlock( &mPBArray[ pbIndex + 1 ] ) );
			}
		}
#endif
	}
	
	
	
/*_________________________________________________________________________
	Accounting for fragmentation, fill in as many parameter blocks as needed
	to satisfy the I/O request.
_________________________________________________________________________*/
	void
CDrive::FillInAllPBsForRead(
	const UInt32		startOffsetInBlocks,
	const ByteCount		origRequestCountInBlocks,
	void *				buffer,
	ushort *			numPBsUsedPtr )
	{
	pgpAssert( startOffsetInBlocks + origRequestCountInBlocks <=
			GetLogicalSizeInBlocks() );
	
	pgpAssertAddrValid( numPBsUsedPtr, ushort);
	pgpAssert( origRequestCountInBlocks != 0);
	pgpAssert( mIOQueue.IsEmpty() );
	
	*numPBsUsedPtr	= 0;
	
	ulong	remainingBlocks	= origRequestCountInBlocks;
	ulong	curBlockOffset	= startOffsetInBlocks;
	Ptr		curBuffer	= (Ptr)buffer;
	
	while ( remainingBlocks != 0 )
		{
		MyPB *				myPB;
		const DiskExtent *	extent;
		ulong				startFileBlockOfExtent;
		
		pgpAssert( *numPBsUsedPtr < mNumPBs );
		
		myPB	= GetIndPB( *numPBsUsedPtr );
		
		extent = FindExtentContainingBlock( curBlockOffset,
						&startFileBlockOfExtent);
		pgpAssert( IsntNull( extent ) );
		
		ulong	offsetFromBase	= curBlockOffset - startFileBlockOfExtent;
		ulong	blockOffsetOnBaseDrive	= extent->diskBlockIndex +
						offsetFromBase;
		pgpAssert( extent->numBlocks >= offsetFromBase);
		ulong	numBlocksAvail		= extent->numBlocks - offsetFromBase;
		ulong	requestBlocks		= MIN( remainingBlocks, numBlocksAvail );
		
		pgpAssert( requestBlocks != 0 );

		FillInPB( myPB, curBuffer, curBuffer, blockOffsetOnBaseDrive,
						requestBlocks);
		myPB->startBlockOnSelf	= curBlockOffset;
		InsertLastPB( *numPBsUsedPtr );
		
		*numPBsUsedPtr	+= 1;
		curBuffer		+= requestBlocks * kPGPDiskBlockSize;
		remainingBlocks	-= requestBlocks;
		curBlockOffset	+= requestBlocks;
		}
	
	BreakUpPBsIfNeeded( origRequestCountInBlocks, *numPBsUsedPtr,
			numPBsUsedPtr, &mReadTuner);
	pgpAssert( *numPBsUsedPtr <= mNumPBs );
	}


		
	void
CDrive::QueueAllPBsForRead( ushort	numPBs )
	{
	pgpAssert( mProcessingQueue.IsEmpty() );
	pgpAssert( mIOQueue.IsEmpty() );
	
	for( ushort index = 0; index < numPBs; ++index )
		{
		mIOQueue.Enqueue( & GetIndPB( index )->ioQElem );
		}
	}
	
	
	void
CDrive::ReadPBs( ushort	numPBs )
	{
	pgpa((
		pgpaAssert( mProcessingQueue.IsEmpty() ),
		pgpaAssert( ! mIOQueue.IsEmpty() )
		));
	
	OSErr	err	= noErr;

	for( ushort index = 0; index < numPBs; ++index )
		{
		MyPB *	myPB;
		
		myPB	= GetIndPB( index );
		
	#if ALLOW_VM_PAGING	// [
		if ( mTreatAsVMRequest )
			{
			err		= PBReadSync( &myPB->ioPB.pb );
			mProcessingQueue.Enqueue( &myPB->processingQElem );
			mIOQueue.Dequeue( &myPB->ioQElem );
			}
		else
	#endif	// ] ALLOW_VM_PAGING
			{
			err		= PBReadAsync( &myPB->ioPB.pb );
			}
		AssertNoErr( err, "CDrive::ReadAllPBs" );
		}
	}




	void
CDrive::DecryptBlocksDeferredTask( )
	{
	do
		{
		pgpAssert( mDecryptLock );

		// it's OK to look directly at the front of the queue;
		// we the only code that can remove from the queue
		MyPB *		myPB;
		QElem *		qElem;
		qElem	= mProcessingQueue.PeekHead();
		pgpAssert( IsntNull( qElem ) );
		myPB	= ProcessingQueueElemToMyPB( qElem );
		
		DecryptPB( myPB );
		
		OSErr	err	= mProcessingQueue.Dequeue( qElem );
		AssertNoErr( err, "DecryptBlocksDeferredTask" );
		
		// check without critical section first to avoid disabling interrupts
		// if we're just looping.
		if ( mProcessingQueue.IsEmpty() )
			{
			Boolean	isEmpty;
			Boolean	done	= false;
			
			EnterCriticalSection();
				isEmpty	= mProcessingQueue.IsEmpty();
				if ( isEmpty )
					{
					// we're done if mIOQueue is empty as well
					// in either case, relinquish the lock, since there's
					// nothing available
					done	= mIOQueue.IsEmpty();
					mDecryptLock	= false;
					}
			ExitCriticalSection();
			
			if ( done )
				{
				MyIOIsDone( mResultErr, mClientPB->ioParam.ioReqCount );
				break;
				}
				
			if ( isEmpty )
				{
				break;
				}
			}
		} while ( TRUE );
	}
	
	
	pascal void
CDrive::sDecryptBlocksDeferredTask( void )
	{
	CDrive *	theDrive	= (CDrive *)GetA1();
	
	EnterCodeResource();
	
	CDrive::sAssertValid( theDrive );
	
#if CDRIVE_WATCHDOG
	theDrive->RemoveWatchdog();
#endif

	theDrive->DecryptBlocksDeferredTask( );
	
	ExitCodeResource();
	}
	
	
/*_________________________________________________________________________
	When a read finishes, we need to decrypt the data. However, we must not
	do so right now, because we may be in an interrupt and spending too much
	time in an interrupt is Bad.  So install a deferred task to perform the
	decryption.
_________________________________________________________________________*/
	void
CDrive::HandleReadCompletion( MyPB *myPB )
	{
	Boolean		decryptNow	= false;
	
	mProcessingQueue.Enqueue( &myPB->processingQElem );
	mIOQueue.Dequeue( &myPB->ioQElem );
	
	Boolean	haveLock	= ! TestAndSet( &mDecryptLock );
	if ( haveLock )
		{
		InstallDeferredTask( &mDecryptDeferredTask,
				sDecryptBlocksDeferredTask );
		}
	}



/*_________________________________________________________________________
	It's time to read all data needed to satisfy the client request.
_________________________________________________________________________*/
	void
CDrive::KickStartDeferredRead()
	{
	ParamBlockRec *		pb	= mClientPB;
	ushort				numPBsUsed;
	
	pgpAssert( IsntNull( pb ) );
	
	FillInAllPBsForRead( ::GetPBStartBlock( pb ), ::GetPBNumBlocks( pb ),
		pb->ioParam.ioBuffer, &numPBsUsed);
	
	QueueAllPBsForRead( numPBsUsed );
	ReadPBs( numPBsUsed );
	}
	
	
	
	
	
/*_________________________________________________________________________
	Service a client read request
_________________________________________________________________________*/
	void
CDrive::Read(
	ParamBlockRec *	clientPB,
	DCtlEntry *		dce )
	{
	pgpAssert( clientPB == (ParamBlockRec *)dce->dCtlQHdr.qHead );
	
	if ( ! BaseDriveIsAvailable() )
	{
		pgpDebugPStr( "\pCDrive::Read: base drive is not available;g" );
		mClientPB	= clientPB;
		MyIOIsDone( badUnitErr, 0UL );	// 'badUnitErr' is what base driver
										// returns
	}
	else if ( IsValidRequest( &clientPB->ioParam ) )
		{
		InitReadOrWrite( clientPB, dce, kReadDirection);

		// all requests have been queued and are being processed
		pgpAssert( clientPB->ioParam.ioResult == kAsyncIOInProgress );
		
	#if ALLOW_VM_PAGING	// [

		if ( mTreatAsVMRequest )
			{
			mDecryptLock	= TRUE;
			KickStartDeferredRead();
			DecryptBlocksDeferredTask();
			// ...done
			}
		else
	#endif	// ] ALLOW_VM_PAGING
		if ( InterruptsAreEnabled() )
			{
			// no reason to wait...start processing now
			// acquire the decrypt lock so that if I/O finishes before
			// returning, we can avoid the overhead of using a deferred
			// task as well as the problem that the I/O could complete
			// synchronously 
			mDecryptLock	= TRUE;
			
			KickStartDeferredRead();
			
			Boolean		isEmpty;
			EnterCriticalSection();
				isEmpty	= mProcessingQueue.IsEmpty();
				if ( isEmpty )
					{
					mDecryptLock	= false;
					}
			ExitCriticalSection();
			
			if ( ! isEmpty )
				{
				DecryptBlocksDeferredTask();
				}
			}
		else
			{
			InstallDeferredTask( &mKickStartDeferredTask,
					sKickStartDeferredTask );
			}
		}
	else
		{
		pgpDebugPStr( "\pCDrive::Read: invalid request;g" );
		mClientPB	= clientPB;
		MyIOIsDone( paramErr, 0UL );
		}
	}



#pragma mark -



/*_________________________________________________________________________
	Set up information needed to handle a write request.
_________________________________________________________________________*/
	void
CDrive::InitWrite( const IOParam *		pb)
	{
	pgpAssertAddrValid( pb->ioBuffer, VoidAlign );
	pgpAssert( pb->ioReqCount != 0 &&
			( pb->ioReqCount % kPGPDiskBlockSize) == 0 );
	
	mWriteStuff.curClientBuffer		= pb->ioBuffer;
	mWriteStuff.curClientBlock		= ::GetPBStartBlock( pb );
	mWriteStuff.blocksRemaining		= ::GetPBNumBlocks( pb );
	
	pgpAssert( mWriteStuff.curClientBlock < GetLogicalSizeInBlocks() );
	
	GetDateTime( &mLastWriteTime );
	}
	


/*_________________________________________________________________________
	We've been called at deferred task time.  We are free to use as much CPU
	time as we want.  Encrypt and write as much data as possible. Upon
	return from doing that, there may still be more outstanding data, but that
	will be handled by the last write call, which will call us again.
_________________________________________________________________________*/
	void
CDrive::KickStartDeferredWrite()
	{
	pgpa((
		pgpaAssert( mEncryptLock ),
		pgpaAssert( mProcessingQueue.IsEmpty() ),
		pgpaAssert( mIOQueue.IsEmpty() ),
		pgpaAssert( mWriteStuff.blocksRemaining != 0 ),
		pgpaAssert( mWriteStuff.curClientBlock <= GetLogicalSizeInBlocks() )
			));
					
	while ( TRUE )
		{
		Boolean		ioPending;
		
		pgpAssert( mEncryptLock );
		QueueForEncryption();
		EncryptAndWriteQueuedItems();
		
		EnterCriticalSection();
		ioPending	= ! mIOQueue.IsEmpty();
		if ( ioPending )
			{
			mEncryptLock	= false;
			}
		ExitCriticalSection();
		
		if ( ioPending )
			{
			// HandleWriteCompletion will reacquire lock
			break;
			}
		
		pgpAssert( mProcessingQueue.IsEmpty() );
		pgpAssert( mIOQueue.IsEmpty() );
		// io is *not* pending so either we're done or we need to process more
		if ( mWriteStuff.blocksRemaining == 0 )
			{
			mEncryptLock	= false;
			MyIOIsDone( mResultErr, mClientPB->ioParam.ioReqCount );
			break;
			}
		}
	}
	
	
	
	void
CDrive::HandleWriteCompletion( MyPB *myPB )
	{
	pgpAssert( IsntNull( mClientPB ) );
	pgpAssert( myPB->magic == MyPB::kMagic );
	
	mIOQueue.Dequeue( &myPB->ioQElem );
	
	Boolean		haveLock	= ! TestAndSet( &mEncryptLock );
	if ( haveLock )
		{
		Boolean		ioIsDone;
		
		EnterCriticalSection();
		ioIsDone	= mIOQueue.IsEmpty();
		if ( ! ioIsDone )
			{
			// the last IO request to complete will acquire the lock
			mEncryptLock	= false;
			}
		ExitCriticalSection();
		
		if ( ioIsDone )
			{
			// the processing queue must be empty; otherwise we could not
			// have acquired the lock
			pgpAssert( mProcessingQueue.IsEmpty() );
			
			// no need to retain lock; nothing else is running
			if ( mWriteStuff.blocksRemaining == 0 )
				{
				mEncryptLock	= false;
				MyIOIsDone( mResultErr, mClientPB->ioParam.ioReqCount );
				}
			else
				{
				// retain lock for KickStartDeferredWrite
				pgpAssert( mEncryptLock );
				InstallDeferredTask( &mKickStartDeferredTask,
						sKickStartDeferredTask );
				}
			}
		}
	}

		
/*_________________________________________________________________________
	Queue as many requests as possible for encryption (limited by size of
	our scratch buffer).
_________________________________________________________________________*/
	void
CDrive::QueueForEncryption( void )
	{
	ushort			numPBsUsed	= 0;
	ulong			remainingBlocks;
	ulong			blocksDoneThisTime;
	ulong			curBlockIndex;
	Ptr				curScratchBuffer	= (Ptr)sScratchBuffer;
	const void *	curClientBuffer		= mWriteStuff.curClientBuffer;
	
	pgpAssert( IsntNull( curScratchBuffer ) );
	pgpAssert( sScratchBufferSize != 0 );
	
	blocksDoneThisTime	= MIN( mWriteStuff.blocksRemaining,
			sScratchBufferSize / kPGPDiskBlockSize);
	curBlockIndex		= mWriteStuff.curClientBlock;
	pgpAssert( curBlockIndex + blocksDoneThisTime <= GetLogicalSizeInBlocks() );

	remainingBlocks		= blocksDoneThisTime;
	while ( remainingBlocks != 0 && numPBsUsed < mNumPBs )
		{
		MyPB *				myPB;
		const DiskExtent *	extent;
		ulong				startFileBlockOfExtent;
		
		pgpAssert( numPBsUsed < mNumPBs );
		
		myPB	= GetIndPB( numPBsUsed );
		myPB->theDrive	= this;
		SetPBMagic( myPB );
		
		extent	= FindExtentContainingBlock( curBlockIndex,
				&startFileBlockOfExtent);
		pgpAssert( IsntNull( extent ) );
		
		ulong	offsetFromBase = curBlockIndex - startFileBlockOfExtent;
		ulong	blockOffsetOnBaseDrive = extent->diskBlockIndex +
												offsetFromBase;
		ulong	numBlocksAvail = extent->numBlocks - offsetFromBase;
		ulong	requestBlocks = MIN( remainingBlocks, numBlocksAvail );

		FillInPB( myPB, curClientBuffer, curScratchBuffer,
					blockOffsetOnBaseDrive, requestBlocks);
		myPB->startBlockOnSelf	= curBlockIndex;
		InsertLastPB( numPBsUsed );
				
		curBlockIndex		+= requestBlocks;
		numPBsUsed			+= 1;
		curScratchBuffer	+= requestBlocks * kPGPDiskBlockSize;
		curClientBuffer		= (const void *)( (Ptr)curClientBuffer +
									requestBlocks * kPGPDiskBlockSize );
		remainingBlocks		-= requestBlocks;
		}

	pgpAssert( numPBsUsed <= mNumPBs );
	
	pgpAssert( blocksDoneThisTime <= mWriteStuff.blocksRemaining );
	mWriteStuff.blocksRemaining	-= blocksDoneThisTime;
	mWriteStuff.curClientBlock	= curBlockIndex;
	mWriteStuff.curClientBuffer	= curClientBuffer;
	
	BreakUpPBsIfNeeded( ::GetPBNumBlocks( mClientPB ), numPBsUsed,
			&numPBsUsed, &mWriteTuner );
	pgpAssert( numPBsUsed <= mNumPBs );
	
	// now queue them all up for processing
	for( ushort index = 0; index < numPBsUsed; ++index )
		{
		mProcessingQueue.Enqueue( &GetIndPB( index )->processingQElem );
		}
	}



	void
CDrive::EncryptAndWriteQueuedItems( void )
	{
	OSErr	err		= noErr;
	
	while ( ! mProcessingQueue.IsEmpty() )
		{
		MyPB *			myPB;
		ulong			numBlocks;
		const void *	src;
		void *			dest;
		
		myPB		= ProcessingQueueElemToMyPB( mProcessingQueue.PeekHead() );
		pgpAssert( myPB->magic == MyPB::kMagic );
		
		numBlocks	= GetPBNumBlocks( myPB );
		src			= myPB->src;
		dest		= myPB->ioPB.pb.ioParam.ioBuffer;
		
		CipherProc_CallEncrypt( mCipherProcRef, GetCipherContext(),
				myPB->startBlockOnSelf, numBlocks, src, dest );
		
		// CAUTION: it is crucial that we enqueue to mIOQueue before dequeing
		// from mProcessingQueue
		mIOQueue.Enqueue( &myPB->ioQElem );
		mProcessingQueue.Dequeue( &myPB->processingQElem );
		
		err	= PBWriteAsync( &myPB->ioPB.pb );
		AssertNoErr( err, "CDrive::EncryptAndWriteQueuedItems" );
		}
	}
	
	

	
	
/*_________________________________________________________________________
	Service a client write request.
	
	Writing is a bit more complicated than reading because we have to copy the
	client's buffer to a temporary buffer before encrypting it and then write
	out the temporary buffer.
_________________________________________________________________________*/
	void
CDrive::Write(
	ParamBlockRec *	clientPB,
	DCtlPtr			dce )
	{
	OSErr err = noErr;
	
	pgpAssert( ! mEncryptLock );
	
	if ( ! BaseDriveIsAvailable() )
	{
		pgpDebugPStr( "\pCDrive::Read: base drive is not available;g" );
		mClientPB	= clientPB;
		MyIOIsDone( badUnitErr, 0UL );	// 'badUnitErr' is what base driver
										// returns
	}
	else if ( IsValidRequest( &clientPB->ioParam ) )
		{
		if( IsWriteProtected() )
			{
			// This error is returned when writing to locked floppies and Zip'
			// disks. Writing to a CD-ROM returns a writErr (-20).
			err = wPrErr;
			}
		else
			{
			InitReadOrWrite( clientPB, dce, kWriteDirection );
			InitWrite( &clientPB->ioParam );
			
			pgpAssert( clientPB->ioParam.ioResult	== kAsyncIOInProgress );
			
			mEncryptLock	= TRUE;
			if ( InterruptsAreEnabled() )
				{
				KickStartDeferredWrite();
				}
			else
				{
				InstallDeferredTask( &mKickStartDeferredTask,
						sKickStartDeferredTask );
				}
			}
		}
	else
		{
		pgpDebugPStr( "\pCDrive::Read: invalid request" );
		err = paramErr;
		}

	if( IsErr( err ) )
		{
		mClientPB	= clientPB;
		MyIOIsDone( err, 0UL );
		}
	}

#pragma mark -


	const char *
CDrive::GetPhysicalIcon( ) const
	{
	return( (const char *)&mPhysicalIcon );
	}
	
	

	const char *
CDrive::GetMediaIcon( ) const
	{
	return( (const char *)&mMediaIcon );
	}




	ulong
CDrive::GetLogicalSizeInBlocks(  ) const
	{
	const DrvQEl *	dqe	= &mMyDQE->el;
	ulong			numBlocks;
	
	pgpAssertAddrValid( mMyDQE, FlagsDrvQEl );
	
	numBlocks	= dqe->dQDrvSz + (((ulong)dqe->dQDrvSz2) << 16);
	
	return( numBlocks );
	}
	
	
/*_________________________________________________________________________
	The driver is responsible for creating the drive queue element.
_________________________________________________________________________*/
	void
CDrive::SetDriveQElement(const FlagsDrvQEl *queueElement)
	{
	pgpAssertAddrValid( queueElement, FlagsDrvQEl );
		
	mMyDQE = queueElement;
	MacLeaks_RememberPtr( mMyDQE, kMacLeaks_NewPtrSysAllocType, 0,
			"somwhere", 0);
	}



/*_________________________________________________________________________
	Change all 1 bits to 0 and all 0 bits to 1 (logical NOT).
_________________________________________________________________________*/
	void
CDrive::FlipBytes( 
	void *		bytes,
	ulong		numBytes )
	{
	uchar *array	= (uchar *)bytes;
	
	pgpAssertAddrValid( array, UInt32 );
	
	for( ulong index = 0; index < numBytes; ++index )
		{
		array[ index ]	= ~ array[ index ];
		}
	}
	
	
/*_________________________________________________________________________
	Note: the caller has taken care to call this at a time when we're not
	servicing any I/O and a time during which no I/O can begin.
_________________________________________________________________________*/
	void
CDrive::ToggleKeyBits( void )
	{
	mWhichCipherContextValid	= ( mWhichCipherContextValid + 1 ) % 2;
	
	FlipBytes( (void *)mCipherContext[ 0 ].castCFB.expandedKey.keyWords,
		sizeof( mCipherContext[ 0 ].castCFB.expandedKey.keyWords ) );
		
	FlipBytes( (void *)mCipherContext[ 1 ].castCFB.expandedKey.keyWords,
		sizeof( mCipherContext[ 1 ].castCFB.expandedKey.keyWords ) );
	}



/*_________________________________________________________________________
	Return a pointer to the currently valid cipher context. One context is
	valid and the other has its bits flipped.
_________________________________________________________________________*/
	const CipherContext *
CDrive::GetCipherContext( void )
	{
	return( &mCipherContext[ mWhichCipherContextValid ] );
	}


/*_________________________________________________________________________
	If this is a writeable disk, change the file modification date to be the
	last write time to the disk.
_________________________________________________________________________*/
	void
CDrive::UpdateFileModificationDate( )
	{
	if ( ! IsWriteProtected() )
		{
		OSStatus		err	= noErr;
		const ulong		kMinSecondsBetweenUpdate	= 5;
		
		if ( ( mLastWriteTime > mLastFileModDateUpdate ) &&
				( mLastWriteTime - mLastFileModDateUpdate ) >=
					kMinSecondsBetweenUpdate )
			{
			FSSpec	spec;
			
			// we don't want to make a sync file mgr call, so use FCBs
			err	= GetSpecFromFCBTable( mFileRefNum, &spec );
			if ( IsntErr( err ) )
				{
				pgpAssert( IsntNull( sModDateQueue ) );
				sModDateQueue->ChangeModDate( &spec, mLastWriteTime );
				}
				
			GetDateTime( &mLastFileModDateUpdate );
			}
		}
	}
	

	OSErr
CDrive::sCreateSharedQueues()
	{
	OSErr	err	= noErr;
	
	if ( IsNull( sCloseQueue ) )
		{
		sCloseQueue	= new CCloseQueue();
		if ( IsNull( sCloseQueue ) )
			err	= memFullErr;
		}
		
	if ( IsntErr( err ) && IsNull( sModDateQueue ) )
		{
		sModDateQueue	= new CChangeModDateQueue();
		if ( IsNull( sModDateQueue ) )
			err	= memFullErr;
		}
		
	return( err );
	}
	
	
	void
CDrive::sCleanup()
	{
	delete sCloseQueue;
	sCloseQueue	= nil;
		
	delete sModDateQueue;
	sModDateQueue	= nil;
	}


/*_________________________________________________________________________
	Create a new CDrive object.  If an error occurs, destroy it and set it
	to nil.
_________________________________________________________________________*/
	OSErr
CDrive::sCreate(
	CDrive **			newDrive,
	const InitInfo *	info )
	{
	OSErr		err	= memFullErr;
	CDrive *	drive;
	
	err	= sCreateSharedQueues();
	if ( IsntErr( err ) )
		{
		drive	= new CDrive( );
		if ( IsntNull( drive )  )
			{
			err	= drive->Init( info );
			}
		
		*newDrive	= drive;
		}
	
	return( err );
	}















	
	
	
