/*                               -*- Mode: C -*- 
 * 
 * uSystem Version 4.3.2, Copyright (C) Peter A. Buhr and Richard A. Stroobosscher 1990
 * 
 * uKernel.c -- The routines which form the nucleus of the concurrency system.
 * 
 * Author           : Rick Stroobosscher
 * Created On       : Fri Feb  9 15:13:53 1990
 * Last Modified By : Peter A. Buhr
 * Last Modified On : Mon Mar  4 20:29:20 1991
 * Update Count     : 336
 */

#define  __U_KERNEL__

#if defined( mips ) || defined( __mips__ )
asm( ".set nomove" );
#endif

#include "uUnix.h"
#include "uSystem.h"
#include "uKernel.i"
#include "uMachine.i"
#include "uPeek.i"
#include "uQueue.i"
#include "uStack.i"

static inline void uAcquireTask( uTask task ) {

    /*
     * This function allows a processor to acquire task for execution.
     */

    uAcquireLock( &(task->mutex) );
} /* uAcquireTask */

static inline void uReleaseTask( uTask task ) {

    /*
     * This function allows a processor to release a task
     * so that another processor may execute it.
     */

    uReleaseLock( &(task->mutex) );
} /* uReleaseTask */

static inline uTask uAllocTask( int space ) {

    /*
     * This function allocates a task, coroutine and stack.
     */

    void *limit;
    uTask task;

    space  = U_CEILING( space, sizeof(double) ) + sizeof(double); /* move value up to multiple of sizeof(double) */
    limit = uMalloc( space + sizeof(struct uTaskD) );
    task  = (void *) U_FLOOR( (int) limit + space, sizeof(double) ); /* move value down to multiple of sizeof(double) */

    *task = U_TASK();
    
    task->StackData.limit = limit;
    task->StackData.base = (void *) ( (int) task - sizeof( double ) ); /* prime stack for first push */

    uPoke( task->StackData.limit, U_MAGIC );

    task->cortn = task;					/* task is also a coroutine */

    return( task );
} /* uAllocTask */

static inline void uFreeTask( uTask task ) {

    /*
     * This function deallocates the resources of a task.
     */

    uFree( task->StackData.limit );			/* remember, task info is at the top of the stack so free the limit */
} /* uFreeTask */

static inline uProcessor uAllocProcessor( void ) {

    /*
     * This function allocates a process descriptor.
     */

    uProcessor processor;

    processor = uMalloc( sizeof( struct uProcessorD ) ); /* allocate memory for a process descriptor */
    *processor = U_PROCESSOR();				/* initialize the processor descriptor */
    
    return( processor );				/* return a reference to the process */
} /* uAllocProcessor */

static inline void uFreeProcessor( uProcessor processor ) {

    /*
     * This function dellocates the resources of a process.
     */

    uFree( processor );					/* deallocate the processor descriptor */
} /* uFreeProcessor */

static inline uCluster uAllocCluster( void ) {

    /*
     * This function allocates a cluster descriptor.
     */

    uCluster cluster;

    cluster = uMalloc( sizeof( struct uClusterD ) );	/* allocate memory for a cluster descriptor */
    *cluster = U_CLUSTER();				/* initialize the cluster descriptor */

    /*
     * Set up the IO cluster stuff.
     * We really need to be able to derive classes.
     */

    FD_ZERO( &(cluster->rfds) );
    FD_ZERO( &(cluster->wfds) );
    FD_ZERO( &(cluster->efds) );

    return( cluster );					/* return a reference to the cluster descriptor */
} /* uAllocCluster */

static inline void uFreeCluster( uCluster cluster ) {

    /*
     * This function deallocates the resources of a cluster.
     */

    uFree( cluster );					/* deallocate the cluster descriptor */
} /* uFreeCluster */

char *uSignalError[] = {				/* error messages associated with signals */
    "HANGUP",
    "INTERRUPT",
    "QUIT",
    "ILLEGAL INSTRUCTION",
    "TRACE TRAP",
    "IOT INSTRUCTION",
    "EMT INSTRUCTION",
    "FLOATING POINT EXCEPTION",
    "KILL",
    "BUS ERROR",
    "SEGMENTATION VIOLATION",
    "BAD ARGUMENT TO SYSTEM CALL",
    "WRITE ON A PIPE WITH NO ONE TO READ IT",
    "ALARM CLOCK",
    "SOFTWARE TERMINATION SIGNAL",
    "URGENT CONDITION PRESENT ON SOCKET",
    "STOP",
    "STOP SIGNAL GENERATED FROM KEYBOARD",
    "CONTINUE AFTER STOP",
    "CHILD STATUS HAS CHANGED",
    "BACKGROUND READ ATTEMPTED FROM CONTROL TERMINAL",
    "BACKGROUND WRITE ATTEMPTED TO CONTROL TERMINAL",
    "IO IS POSSIBLE ON A DESCRIPTOR",
    "CPU TIME LIMIT EECEEDED",
    "FILE SIZE LIMIT EXCEEDED",
    "VIRTUAL TIME ALARM",
    "PROFILING TIMER ALARM",
    "USER BY PCI TO SIGNAL A WINDOW SIZE CHANGE",
    "USER DEFINED SIGNAL 1",
    "USER DEFINED SIGNAL 2"
    };

static int uSigChld( int a, int b, void *vsc ) {

    /*
     * This is a signal handler for the SIGCHLD signal.  It is installed by each
     * processor in the uSystem.  If any processor should die for any reason,
     * (floating point error, segmentation violation), its death will be caught by
     * its parent process.  When a process dies, its parent process receives the
     * SIGCHLD signal, and enters this handler.  If the process died abnormally,
     * then an error mesage is printed out and the uSystem error handling
     * facilities is invoked to handle the error.
     */

    int pid;
    int code;
    struct sigcontext *sc = vsc;
    
    pid = wait( &code );				/* find out who died and why they died */

    if ( code != 0 ) {
	uError( "uSigChld(%d, %d, 0x%x) : PROCESSOR DIED, CAUSE OF DEATH WAS %s.\n", a, b, sc, uSignalError[(code & 127) - 1] );
    } /* if */

    return( 0 );
} /* uSigChld */

#ifdef __U_MULTI__

#include "uCalibrate.h"
#include "uIdle.i"

#endif

static inline uTask uSelectTask( void ) {

    /*
     * This function selects a task from the ready queue associated with the
     * cluster on which the processor is executing.
     * If the processor has been scheduled for termination, this function
     * returns the null task reference, which terminates the processor.
     */
    
    uTask task;
    uTaskQueue *ready = &(uWorkProcessor->cluster->ready);

    for ( ;; ) {
      if ( uWorkProcessor->stop ) {			/* if the processor is scheduled for termination, */
	    return NULL;				/* return a reference to the null task. */
	} /* exit */
	uAcquireTaskQueue( ready );			/* lock the ready queue. */
      if ( ! uTaskQueueEmpty( ready ) ) {		/* if the ready queue is not empty, */
	    task = uDeleteTask( ready );		/* remove a task, */
	    uReleaseTaskQueue( ready );			/* unlock the ready queue, */
	    return task;				/* and return a reference to the task. */
	} /* exit */
	uReleaseTaskQueue( ready );			/* unlock the ready queue, */

#ifdef __U_MULTI__
	uIdle( ready );					/* idle the processor until work appears. */
#else
	uError( "uSelectTask() : NO READY TASK. SYSTEM IS EITHER DEADLOCKED OR ALL TASKS ARE WAITING.\n" );
#endif
	
    } /* for */
} /* uSelectTask */

static inline void uSave( void ) {

    /*
     * This function saves the context of a task or coroutine.
     */

    uStack stack = &(uWorkCoroutine->StackData);

    uPushFixedRegs();					/* push the fixed point registers onto the task stack */
    if ( uWorkCoroutine->save == U_SAVE_FLOAT ) {	/* if performing a floating point save... */
	uPushFloatRegs();				/* push the floating point registers on the stack also */
    } /* if */

    stack->pc = uReadReturnAddress();			/* save the return address */
    stack->sp = uReadStackPointer();			/* save the current stack pointer */
    stack->fp = uReadFramePointer();			/* save the current frame pointer */
} /* uSave */

static inline void uRestore( void ) {

    /*
     * This function restores the context of a task or coroutine.
     */

    uStack stack = &(uWorkCoroutine->StackData);

    uWriteReturnAddress( stack->pc );			/* restore the return address */
    uWriteStackPointer( stack->sp );			/* restore the stack pointer */
    uWriteFramePointer( stack->fp );			/* restore the frame pointer */

    if ( uWorkCoroutine->save == U_SAVE_FLOAT ) {	/* if the floating point registers are to be saved ... */
	uPopFloatRegs();				/* pop the floating point registers from the stack */
    } /* if */
    uPopFixedRegs();					/* pop the fixed point registers from the stack */
} /* uRestore */

static inline void uCallCoroutine( uCoroutine cortn ) {

    /*
     * This function initializes the context of
     * a new coroutine.
     */

    uStack stack= &(cortn->StackData);

    uWriteStackPointer( stack->sp );			/* install the new task stack pointer */
    uWriteFramePointer( stack->fp );			/* install the new task frame pointer */

    uCallUsingStack( cortn->begin );			/* call the task, arguments are already on the stack */
} /* uCallCoroutine */

static inline void uCallTask( uTask task ) {

    /*
     * This function initializes the context of a new coroutine.
     */

    uCallCoroutine( task->cortn );
} /* uCallTask */

static void uExecuteCoroutine( uCoroutine cortn ) {

    /*
     * This function switches the active thread of control from one task to another.
     */

    uSave();

#ifdef __U_DEBUG__

    uVerify();						/* verify that the stack is still okay */

#endif

    uWorkCoroutine = cortn;
    uWorkTask->cortn = cortn;
    
    if ( cortn->state == U_NEW ) {
	cortn->state = U_RUN;
	uCallCoroutine( cortn );
	uSuspendDie( NULL, 0 );
    } /* if */

    uRestore();
} /* uExecuteCoroutine */

static void uExecuteTask( uTask task ) {

    /*
     * This function switches the active thread of
     * control from one task to another.
     */

    uSave();

#ifdef __U_DEBUG__

    uVerify();						/* verify that the stack is still okay */

#endif

    uWorkTask = task;
    uWorkCoroutine = task->cortn;

    if ( task->state == U_NEW ) {
	task->state = U_RUN;
	task->cortn->state = U_RUN;
	uCallTask( task );
	uDie( NULL, 0 );
    } /* if */

    uRestore();
} /* uExecuteTask */

static void uScheduler( void ) {

    uTask task;

    while ( ( task = uSelectTask() ) != NULL ) {
	uAcquireTask( task );				/* acquire this task for execution */
	uExecuteTask( task );				/* execute this task */
	uReleaseTask( task );				/* release this task for execution */
    } /* while */
} /* uScheduler */

static inline void uBlockTask( void ) {
    uExecuteTask( uSchedulerTask );			/* resume the scheduler task */
} /* uBlockTask */

static inline void uWakeTask( uTask task ) {

    /*
     * This function adds a task to the ready queue associated with the
     * cluster on which the task is executing.
     */

    uTaskQueue *ready;
    
    ready = &(task->cluster->ready);			/* select the ready queue associated with the task's cluster */

    uAcquireTaskQueue( ready );				/* lock the ready queue */
    uInsertTask( ready, task );				/* add the task to the end of the ready queue */

#ifdef __U_MULTI__

    if ( ready->idle ) {				/* is there an idle processor that could execute this task ? */
	uProcessorStack *cpus;
	uProcessor processor;
	
	ready->idle = U_FALSE;				/* change ready queue idle state  */
	uReleaseTaskQueue( ready );			/* unlock the ready queue */
	cpus = &(task->cluster->cpus);			/* select the processor stack associated with the task's cluster */
	uAcquireProcessorStack( cpus );			/* lock the processor stack */
	processor = uProcessorStackTop( cpus );
	while ( processor != NULL ) {			/* search processor stack for an idle processor */
	    if ( processor->state == U_IDLE ) {		/* found one? */
		processor->state = U_BUSY;		/* mark it busy */
		kill( processor->pid, SIGALRM );	/* wake it up, as it is probably sleeping */
	    } /* if */
	    processor = uProcessorStackLink( processor );
	} /* while */
	uReleaseProcessorStack( cpus );			/* unlock the processor stack */
    } else {
	uReleaseTaskQueue( ready );			/* unlock the ready queue */
    } /* if */

#else

    uReleaseTaskQueue( ready );				/* unlock the ready queue */

#endif
} /* uWakeTask */

void uError( char *fmt, ... ) {

    /*
     * This function handles any errors that may appear during the execution of
     * uKernel code or user code.  Currently, this function simply
     * prints out a message indicating who caused the error, and what the
     * error was.  Then the uSystem shuts itself down, killing all remaining UNIX processes.
     */

    va_list argadr;

    extern int errno;

    va_start( argadr, fmt );

    fflush( stdout );
    fprintf( stderr, "*** uSYSTEM ERROR, " );
    _doprnt( fmt, argadr, stderr );
    fprintf( stderr, "*** UNIX ERRNO, %d\n", errno );
    fprintf( stderr, "Last uCoroutine to execute was %s.\n", uGetName( uThisCoroutine() ) );
    fprintf( stderr, "Last uTask to execute was 0x%x.\n", uThisTask() );
    fprintf( stderr, "uCoroutine and uTask lived on uCluster 0x%x.\n", uThisCluster() );
    fflush( stderr );

    va_end( argadr );

#ifdef __U_MULTI__
    {
	int group;

	group = getpgrp( 0 );
	setpgrp( 0, 0 );
	uKillGroup( group, SIGKILL );
	setpgrp( 0, group );
    }
#endif

    abort();
} /* uError */

inline void uVerify( void ) {

    /*
     * This function checks the stack of a coroutine to make sure that the
     * stack has not overflowed.  If it has, the function causes an error.
     */

    uStack stack = &(uWorkCoroutine->StackData);

    if ( uWorkTask != uSchedulerTask ) {
	
	if ( uPeek( stack->limit ) != U_MAGIC ) {	/* magic word has been overwritten? */
	    uError( "uVerify() : STACK OVERFLOW, MAGIC WORD DESTROYED FOR COROUTINE: %s.\n", uGetName(uWorkCoroutine) );
	} /* if */

	if ( uReadStackPointer() < stack->limit ) {	/* current stack pointer beyond stack limit? */
	    uError( "uVerify() : STACK OVERFLOW, BEYOND END OF STACK FOR COROUTINE: %s.\n", uGetName(uWorkCoroutine) );
	} /* if */

    } /* if */
} /* uVerify */

inline void uP( uSemaphore *sem ) {

    /*
     * This function implements the P operation on a semaphore.
     */

    uAcquireTaskQueue( sem );				/* lock the semaphore */
    if ( sem->len < 0 ) {				/* if the semaphore count is less than zero, */
	sem->len += 1;					/* simply increment the semaphore count and */
	uReleaseTaskQueue( sem );			/* unlock the semaphore, */
    } else {
	uInsertTask( sem, uWorkTask );			/* otherwise, insert a task on the semaphore blocked queue and */
	uReleaseTaskQueue( sem );			/* unlock the semaphore. */
	uBlockTask();					/* then call the kernel to block the current task */
    } /* if */
} /* uP */

inline void uV( uSemaphore *sem ) {

    /*
     * This function implements the V operation on a semaphore.
     */

    uTask task;

    uAcquireTaskQueue( sem );				/* lock the semaphore */
    if ( uTaskQueueEmpty( sem ) ) {			/* if the blocked queue is empty, */
	sem->len -= 1;					/* simply decrement the counter and */
	uReleaseTaskQueue( sem );			/* unlock the semaphore, */
    } else {
	task = uDeleteTask( sem );			/* otherwise, remove one task from the blocked queue, */
	uReleaseTaskQueue( sem );			/* unlock the semaphore and */
	uWakeTask( task );				/* make the task eligible for execution. */
    } /* if */
} /* uV */

inline int uC( uSemaphore *sem ) {

    /*
     * This function implements the C operation on a semaphore.
     */

    return -uTaskQueueLength( sem );

} /* uC */

static inline uTask uDoEmit( uCluster cluster, long space, void (*begin)(), long arglen, void *argadr ) {

    uTask task;
    uStack stack;

    task = uAllocTask( space );
    task->cluster = cluster;				/* set the cluster of residence for this task */

    task->begin = begin;				/* task begins execution at this address */
    task->owner = task;					/* task is its own coroutine owner */
    task->cortn = task;					/* task's current coroutine is itself */
    
    stack = &(task->StackData);

    uMakeFrameUsingStack( stack, argadr, arglen );

    uWakeTask( task );

    return( task );
} /* uDoEmit */

uTask uEmit( void *begin, ... ) {

    /*
     * This function emits a task with the default values for cluster, stack size,
     * and argument length.
     */

    va_list argadr;
#ifndef __U_DEBUG__
    uTask task;

    va_start( argadr, begin );

#else
    volatile uTask task;
    void           *msgp;
    int            msglen;

    va_start( argadr, begin );

    uExcept {
#endif

	task = uDoEmit( uThisCluster(), uGetStackSize(), (void (*)())begin, uGetArgLen(), argadr );
	
	va_end( argadr );
	
#ifdef __U_DEBUG__
    } uHandlers {
	uWhen( NULL, &uOutOfMemoryEx, &msgp, &msglen ) {
	    uEmitExMsg msg;

	    msg.msg = "uEmit: Cannot emit uTask due to memory allocation problems.\n";
	    msg.cluster = uThisCluster();
	    msg.space = uGetStackSize();
	    msg.begin = begin;
	    msg.arglen = uGetArgLen();

	    uRaise( uThisCluster(), &uEmitEx, &msg, sizeof( msg ) );
	} /* uWhen */
    } uEndExcept;
#endif

    return( task );					/* return a reference to the new task */
} /* uEmit */

uTask uLongEmit( uCluster where, long space, void *begin, long arglen, ... ) {

    /*
     * This function emits a task with
     * explicit values for cluster, stack size,
     * and argument length.
     */

    va_list argadr;
#ifndef __U_DEBUG__
    uTask task;

    va_start( argadr, arglen );

#else
    volatile uTask task;
    void           *msgp;
    int            msglen;

    va_start( argadr, arglen );

    uExcept {
#endif

	task = uDoEmit( where, space, (void (*)())begin, arglen, argadr );

	va_end( argadr );

#ifdef __U_DEBUG__
    } uHandlers {
	uWhen( NULL, &uOutOfMemoryEx, &msgp, &msglen ) {
	    uEmitExMsg msg;

	    msg.msg = "uLongEmit: Cannot emit uTask due to memory allocation problems.\n";
	    msg.cluster = where;
	    msg.space = space;
	    msg.begin = begin;
	    msg.arglen = arglen;

	    uRaise( where, &uEmitEx, &msg, sizeof( msg ) );
	} /* uWhen */
    } uEndExcept;
#endif

    return( task );					/* return a reference to the new task */
} /* uLongEmit */

void uAbsorb( uTask dier, void *rbuf, int rlen ) {

    /*
     * This function waits for the termination of a task,
     * obtains the terminating result, and
     * deallocates the resources of the terminated task.
     */

    uP( &(dier->done) );				/* wait for the task to complete */

#ifdef __U_DEBUG__

    if ( dier->ipc.slen > rlen ) {
	uAbsorbAreaTooShortExMsg msg;

	msg.head.msg = "uAbsorb: ABSORB MESSAGE AREA TOO SHORT TO RECEIVE ABSORBING TASK'S MESSAGE\n";
	msg.head.sbuf = dier->ipc.sbuf;
	msg.head.slen = dier->ipc.slen;
	msg.head.rbuf = rbuf;
	msg.head.rlen = rlen;
	msg.dier = dier;
	msg.absorber = uWorkTask;

	uV( &(dier->done) );	                        /* reset the semaphore so that the uAbsorb can be re-tried */

	uRaise( uWorkCoroutine, &uAbsorbAreaTooShortEx, &msg, sizeof( msg ) );
    } /* if */

#endif

    uCopy( dier->ipc.sbuf, rbuf, dier->ipc.slen );	/* copy the terminating result to the absorber */

    uAcquireTask( dier );

    uFreeTask( dier );
} /* uAbsorb */

void uDie( void *sbuf, int slen ) {

    /*
     * This function terminates the execution
     * of a task and returns a result to
     * an absorbing task.
     */

    uTask dier = uWorkTask;

    dier->ipc.sbuf = sbuf;				/* copy the address and length of the terminating result */
    dier->ipc.slen = slen;				/* into my task descriptor */

    uV( &(dier->done) );				/* signal the absorbing task */

    uBlockTask();					/* and block this task forever */
} /* uDie */

inline uTask uThisTask( void ) {

    /*
     * This function returns a reference to the
     * local task.
     */

    return( uWorkTask );
} /* uThisTask */

uTask uSend( uTask receiver, void *rbuf, int rlen, void *sbuf, int slen ) {

    /*
     * This function sends a message from one
     * task to another, and waits for a reply.
     */

    uTask sender = uWorkTask;

    sender->ipc.rbuf = rbuf;				/* move the address and length of the reply */
    sender->ipc.rlen = rlen;				/* into the task descriptor. */

    uP( &(receiver->ipc.snd) );				/* wait for the receiver task */

#ifdef __U_DEBUG__

    if ( slen > receiver->ipc.rlen ) {			/* if the send message is longer than receiver is willing to accept, */
	uSendMsgTooLongExMsg msg;

	msg.head.msg = "uSend: MESSAGE IS TOO LONG TO BE SENT FROM SENDING TASK\n";
	msg.head.sbuf = sbuf;
	msg.head.slen = slen;
	msg.head.rbuf = receiver->ipc.rbuf;
	msg.head.rlen = receiver->ipc.rlen;
	msg.sender = sender;
	msg.receiver = receiver;

	uV( &(receiver->ipc.snd) );			/* signal receiver to accept a new message */

	uRaise( uWorkCoroutine, &uSendMsgTooLongEx, &msg, sizeof( msg ) );
    } /* if */

#endif

    uCopy( sbuf, receiver->ipc.rbuf, slen );		/* copy the message to the receiver */
    receiver->ipc.from = sender;			/* tell the receiver who sent the message */
    sender->ipc.to = receiver;				/* remember who the message is sent to */

    uV( &(receiver->ipc.rcv) );				/* signal the receiver to continue */
    uP( &(sender->ipc.rcv) );				/* wait for the reply message */

#ifdef __U_DEBUG__
    /*
     * If rlen is "-1", then the reply area was too short,
     * raise an exception
     */
    if( sender->ipc.rlen == -1 ) {
	uReplyAreaTooShortExMsg msg;

	msg.head.msg = "uSend: REPLY AREA IS TOO SHORT TO RECEIVE REPLY MESSAGE.\n";
	msg.head.sbuf = NULL;                     /* can't access these fields in the receiver at this time */
	msg.head.slen = 0;			  /* can't access these fields in the receiver at this time */
	msg.head.rbuf = rbuf;
	msg.head.rlen = rlen;
	msg.sender = sender;
	msg.replier = receiver;

	uRaise( uWorkCoroutine, &uReplyAreaTooShortEx, &msg, sizeof( msg ) );
    } /* if */
#endif

    return( sender->ipc.from );				/* return the identity of the replier */
} /* uSend */

void uForward( uTask receiver, void *sbuf, int slen, uTask sender ) {

    /*
     * This function forwards a message from one task to another task.
     * The forwarding task may not reply to the message once it has
     * been forwarded.  The forward task must reply to the message,
     * or forward the message again.
     */

#ifdef __U_DEBUG__

    uTask forwarder = uWorkTask;
    
    if ( sender->ipc.to != forwarder ) {		/* if the sender did not send message to forwarder, error */
	uInvalidForwardExMsg msg;

	msg.head.head.msg = "uForward: FORWARDING TASK DID NOT RECEIVE MESSAGE FROM SPECIFIED RECEIVER TASK OR IT HAS ALREADY FORWARDED THE MESSAGE.\n";
	msg.head.head.sbuf = sbuf;
	msg.head.head.slen = slen;
	msg.head.head.rbuf = receiver->ipc.rbuf;
	msg.head.head.rlen = receiver->ipc.rlen;
	msg.head.sender = sender;
	msg.head.receiver = receiver;
	msg.forwarder = forwarder;

	uRaise( uWorkCoroutine, &uInvalidForwardEx, &msg, sizeof( msg ) );
    } /* if */

#endif

    uP( &(receiver->ipc.snd) );				/* wait for the receiver to be ready to receive a message */

#ifdef __U_DEBUG__

    if ( slen > receiver->ipc.rlen ) {			/* if message too long for the receiver, signal the */
	uForwardMsgTooLongExMsg msg;

	msg.head.msg = "uForward: MESSAGE IS TOO LONG TO BE FORWARDED FROM FORWARDING TASK\n";
	msg.head.sbuf = sbuf;
	msg.head.slen = slen;
	msg.head.rbuf = receiver->ipc.rbuf;
	msg.head.rlen = receiver->ipc.rlen;
	msg.sender = sender;
	msg.forwarder = forwarder;
	msg.receiver = receiver;

	uV( &(receiver->ipc.snd) );			/* signal receiver to accept a new message */

	uRaise( uWorkCoroutine, &uForwardMsgTooLongEx, &msg, sizeof( msg ) );
    } /* if */

#endif

    uCopy( sbuf, receiver->ipc.rbuf, slen );		/* copy the message */

    receiver->ipc.from = sender;			/* let the receiver know who the message came from originally */
    sender->ipc.to = receiver;				/* update the sender's memory of who it sent the message to */

    uV( &(receiver->ipc.rcv) );				/* signal the receiver that it has now received a message */
} /* uForward */

uTask uReceive( void *rbuf, int rlen ) {

    /*
     * This function allows a task to receive a message
     * from any other task.
     */

    uTask receiver = uWorkTask;

    receiver->ipc.rbuf = rbuf;				/* copy the receiver's address and length of the */
    receiver->ipc.rlen = rlen;				/* receive buffer into its task descriptor */

    uV( &(receiver->ipc.snd) );				/* signal a sender that i am ready to receive a message */
    uP( &(receiver->ipc.rcv) );				/* wait for a message */

    return( receiver->ipc.from );			/* return the identity of the sender */
} /* uReceive */

void uReply( uTask sender, void *sbuf, int slen ) {

    /*
     * This function replies to a previously
     * received message.
     */

    uTask replier = uWorkTask;

#ifdef __U_DEBUG__

    if ( sender->ipc.to != replier ) {			/* if the sender did not send a message to the replier, error */
	uNotReplyBlockedExMsg msg;

	msg.head.msg = "uReply: REPLYING TASK HAS NOT RECEIVED A MESSAGE FROM SENDER TASK OR IT HAS ALREADY BEEN REPLIED TO THAT TASK.\n";
	msg.head.sbuf = sbuf;
	msg.head.slen = slen;
	msg.head.rbuf = sender->ipc.rbuf;
	msg.head.rlen = sender->ipc.rlen;
	msg.sender = replier;
	msg.receiver = sender;

	uRaise( uWorkCoroutine, &uNotReplyBlockedEx, &msg, sizeof( msg ) );
    } /* if */

    /* 
     * if the reply message is too long for the sender,
     * then raise an exception in the sender
     * Do this by setting the "sender->ipc.rlen" field to -1.
     * This is checked when the sender wakes up.  If it is -1,
     * an exception is raised in the sender.
     */
    if ( slen > sender->ipc.rlen ) {
	sender->ipc.rlen = -1;
    } else {
	uCopy( sbuf, sender->ipc.rbuf, slen );		/* copy the message */
    } /* if */
#else
    uCopy( sbuf, sender->ipc.rbuf, slen );		/* copy the message */
#endif

    sender->ipc.to = NULL;				/* clear the sender's to flag, so I cannot reply again */
    sender->ipc.from = replier;				/* tell the sender who finally replied to the message */

    uV( &(sender->ipc.rcv) );				/* signal the sender that a reply has been sent */
} /* uReply */

inline void uYield( void ) {

    /*
     * This function allows a task to
     * explicitly give up its control
     * of a processor.
     */

    uWakeTask( uWorkTask );				/* put the task back on the ready queue */
    uBlockTask();					/* block this task temporarily */
} /* uYield */

uCluster uMigrate( uTask task, uCluster new ) {

    /*
     * This function allows a task to migrate
     * from one cluster to another.
     */

    uCluster old = task->cluster;			/* remember the old cluster */
    task->cluster = new;				/* change the task's cluster reference to the new cluster */
    if ( task == uThisTask() ) {			/* if migrating the calling task, */
	uYield();					/* reschedule the task on the new cluster. */
    } /* if */
    return old;						/* return a reference to the old cluster */
} /* uMigrate */

static inline uCoroutine uDoCocall( void *rbuf, int rlen, long space, void (*begin)(), long arglen, void *argadr ) {

    uCoroutine cortn;
    uStack stack;

    cortn = uAllocTask( space );

    cortn->begin = begin;				/* subroutine to be cocalled */

    stack = &(cortn->StackData);
    
    uMakeFrameUsingStack( stack, argadr, arglen );
    
    cortn->owner = uWorkTask;				/* task that owns this coroutine */

    uResume( cortn, rbuf, rlen, NULL, 0 );

    return( cortn );
} /* uDoCocall */

uCoroutine uCocall( void *rbuf, int rlen, void *begin, ... ) {

    va_list argadr;
#ifndef __U_DEBUG__
    uCoroutine cortn;

    va_start( argadr, begin );

#else
    volatile uCoroutine cortn;
    void                *msgp;
    int                 msglen;

    va_start( argadr, begin );

    uExcept {
#endif

	cortn = uDoCocall( rbuf, rlen, uGetStackSize(), (void (*)())begin, uGetArgLen(), argadr );

	va_end( argadr );

#ifdef __U_DEBUG__
    } uHandlers {
	uWhen( NULL, &uOutOfMemoryEx, &msgp, &msglen ) {
	    uCocallExMsg msg;

	    msg.msg = "uCocall: Cannot cocall uCoroutine due to memory allocation problems.\n";
	    msg.rbuf = rbuf;
	    msg.rlen = rlen;
	    msg.space = uGetStackSize();
	    msg.begin = begin;
	    msg.arglen = uGetArgLen();
	    
	    uRaise( uWorkCoroutine, &uCocallEx, &msg, sizeof( msg ) );
	} /* uWhen */
    } uEndExcept;
#endif

    return( cortn );
} /* uCocall */

uCoroutine uLongCocall( void *rbuf, int rlen, long space, void *begin, long arglen, ... ) {

    va_list argadr;
#ifndef __U_DEBUG__
    uCoroutine cortn;

    va_start( argadr, arglen );

#else
    volatile uCoroutine cortn;
    void                *msgp;
    int                 msglen;
    
    va_start( argadr, arglen );

    uExcept {
#endif

	cortn = uDoCocall( rbuf, rlen, space, (void (*)())begin, arglen, argadr );

	va_end( argadr );

#ifdef __U_DEBUG__
    } uHandlers {
	uWhen( NULL, &uOutOfMemoryEx, &msgp, &msglen ) {
	    uCocallExMsg msg;

	    msg.msg = "uLongCocall: Cannot cocall uCoroutine due to memory allocation problems.\n";
	    msg.rbuf = rbuf;
	    msg.rlen = rlen;
	    msg.space = space;
	    msg.begin = begin;
	    msg.arglen = arglen;
	    
	    uRaise( uWorkCoroutine, &uCocallEx, &msg, sizeof( msg ) );
	} /* uWhen */
    } uEndExcept;
#endif

    return( cortn );
} /* uLongCocall */

static inline void uDoResume( uCoroutine who, void *rbuf, int rlen, void *sbuf, int slen ) {

    uCoroutine resumer = uWorkCoroutine;

    resumer->rbuf = rbuf;
    resumer->rlen = rlen;

    uCopy( sbuf, who->rbuf, slen );

    who->restarter = resumer;

    uExecuteCoroutine( who );

    who = resumer->restarter;

    if ( who->state == U_OLD ) {
	uFreeTask( who );
    } /* if */
} /* uDoResume */

void uResume( uCoroutine who, void *rbuf, int rlen, void *sbuf, int slen ) {

#ifdef __U_DEBUG__
    
    if ( uWorkTask != who->owner ) {
	uBadCoroutineExMsg msg;

	msg.head.msg = "uResume: ATTEMPT TO RESUME A COROUTINE THAT WAS NOT COCALLED BY THIS TASK.\n";
	msg.head.sbuf = sbuf;
	msg.head.slen = slen;
	msg.head.rbuf = who->rbuf;
	msg.head.rlen = who->rlen;
	msg.restarter = who;
	msg.thistask = uWorkTask;

	uRaise( uWorkCoroutine, &uBadCoroutineEx, &msg, sizeof( msg ) );
    } /* if */
    
    if ( slen > who->rlen ) {
	uResumeMsgTooLongExMsg msg;

	msg.head.msg = "uResume: MESSAGE IS TOO LONG TO BE SENT FROM SENDING COROUTINE.\n";
	msg.head.sbuf = sbuf;
	msg.head.slen = slen;
	msg.head.rbuf = who->rbuf;
	msg.head.rlen = who->rlen;
	msg.resumed = who;
	msg.restarter = uWorkCoroutine;

	uRaise( uWorkCoroutine, &uResumeMsgTooLongEx, &msg, sizeof( msg ) );
    } /* if */

#endif

    who->resumer = uWorkCoroutine;

    uDoResume( who, rbuf, rlen, sbuf, slen );
} /* uResume */

void uSuspend( void *rbuf, int rlen, void *sbuf, int slen ) {

    uCoroutine resumer = uWorkCoroutine->resumer;

#ifdef __U_DEBUG__
    
    if ( uWorkTask != resumer->owner ) {
	uBadCoroutineExMsg msg;

	msg.head.msg = "uSuspend: ATTEMPT TO RESUME A COROUTINE THAT WAS NOT COCALLED BY THIS TASK.\n";
	msg.head.sbuf = sbuf;
	msg.head.slen = slen;
	msg.head.rbuf = resumer->rbuf;
	msg.head.rlen = resumer->rlen;
	msg.restarter = resumer;
	msg.thistask = uWorkTask;

	uRaise( uWorkCoroutine, &uBadCoroutineEx, &msg, sizeof( msg ) );
    } /* if */
    
    if ( slen > resumer->rlen ) {
	uSuspendMsgTooLongExMsg msg;

	msg.head.msg = "uSuspend: MESSAGE IS TOO LONG TO BE SENT FROM SENDING COROUTINE.\n";
	msg.head.sbuf = sbuf;
	msg.head.slen = slen;
	msg.head.rbuf = resumer->rbuf;
	msg.head.rlen = resumer->rlen;
	msg.resumed = resumer;
	msg.restarter = uWorkCoroutine;

	uRaise( uWorkCoroutine, &uSuspendMsgTooLongEx, &msg, sizeof( msg ) );
    } /* if */

#endif

    uWorkCoroutine->resumer = NULL;			/* break link with resumer */

    uDoResume( resumer, rbuf, rlen, sbuf, slen );
} /* uSuspend */

static inline void uDoResumeDie( uCoroutine who, void *sbuf, int slen ) {

    uCoroutine resumer = uWorkCoroutine;

    resumer->state = U_OLD;
    uCopy( sbuf, who->rbuf, slen );
    who->restarter = resumer;

    uExecuteCoroutine( who );
} /* uDoResumeDie */

void uResumeDie( uCoroutine who, void *sbuf, int slen ) {

#ifdef __U_DEBUG__
    
    if ( uWorkTask != who->owner ) {
	uBadCoroutineExMsg msg;

	msg.head.msg = "uResumeDie: ATTEMPT TO RESUME A COROUTINE THAT WAS NOT COCALLED BY THIS TASK.\n";
	msg.head.sbuf = sbuf;
	msg.head.slen = slen;
	msg.head.rbuf = who->rbuf;
	msg.head.rlen = who->rlen;
	msg.restarter = who;
	msg.thistask = uWorkTask;

	uRaise( uWorkCoroutine, &uBadCoroutineEx, &msg, sizeof( msg ) );
    } /* if */
    
    if ( slen > who->rlen ) {
	uResumeMsgTooLongExMsg msg;

	msg.head.msg = "uResumeDie: MESSAGE IS TOO LONG TO BE SENT FROM SENDING COROUTINE.\n";
	msg.head.sbuf = sbuf;
	msg.head.slen = slen;
	msg.head.rbuf = who->rbuf;
	msg.head.rlen = who->rlen;
	msg.resumed = who;
	msg.restarter = uWorkCoroutine;

	uRaise( uWorkCoroutine, &uResumeMsgTooLongEx, &msg, sizeof( msg ) );
    } /* if */

#endif

    /* who->resumer = uWorkCoroutine; */ /* this is being removed on a trial basis as it does not seem to make sense */

    uDoResumeDie( who, sbuf, slen );
} /* uResumeDie */

void uSuspendDie( void *sbuf, int slen ) {

    uCoroutine resumer = uWorkCoroutine->resumer;

#ifdef __U_DEBUG__
    if ( uWorkTask != resumer->owner ) {
	uBadCoroutineExMsg msg;

	msg.head.msg = "uSuspendDie: ATTEMPT TO RESUME A COROUTINE THAT WAS NOT COCALLED BY THIS TASK.\n";
	msg.head.sbuf = sbuf;
	msg.head.slen = slen;
	msg.head.rbuf = resumer->rbuf;
	msg.head.rlen = resumer->rlen;
	msg.restarter = resumer;
	msg.thistask = uWorkTask;

	uRaise( uWorkCoroutine, &uBadCoroutineEx, &msg, sizeof( msg ) );
    } /* if */
    
    if ( slen > resumer->rlen ) {
	uSuspendMsgTooLongExMsg msg;

	msg.head.msg = "uSuspendDie: MESSAGE IS TOO LONG TO BE SENT FROM SENDING COROUTINE.\n";
	msg.head.sbuf = sbuf;
	msg.head.slen = slen;
	msg.head.rbuf = resumer->rbuf;
	msg.head.rlen = resumer->rlen;
	msg.resumed = resumer;
	msg.restarter = uWorkCoroutine;

	uRaise( uWorkCoroutine, &uSuspendMsgTooLongEx, &msg, sizeof( msg ) );
    } /* if */

#endif

    uDoResumeDie( resumer, sbuf, slen );
} /* uSuspendDie */

uCoroutine uThisCoroutine( void ) {

    return( uWorkCoroutine );
} /* uThisCoroutine */

inline uProcessor uThisProcessor( void ) {

    /*
     * This function returns a reference to the
     * processor currently executing the calling task.
     */

    return( uWorkProcessor );
} /* uThisProcessor */

#ifdef __U_MULTI__

static void uCPU( uProcessor processor ) {

    uTask task;

#ifdef __U_PROFILE__

    /*
     * Initialize the profiling variables for this process.
     * I am not at all sure of how this code works, it was copied
     * from some code provided by Paul Larson.
     */
    
    char monfile[25];

    sprintf( monfile, "mon.%d", getpid() );
    _mon_file = monfile;
    monstartup( 2, etext );
    
#endif
    
    task = uAllocTask( 0 );				/* allocate a task descriptor without a stack */
    task->cluster = processor->cluster;
    task->mutex = U_LOCKED;
    task->state = U_RUN;

    /*
     * Initialize the private variables for this UNIX process.
     */
    
    uWorkProcessor = processor;				/* the working processor is the allocated processor */
    uWorkTask = task;					/* the working task is the allocated task */
    uWorkCoroutine = task->cortn;
    uSchedulerTask = task;
    
    /*
     * Now, this task must wait until it receives a message
     * indicating that this virtual processor must now
     * terminate execution.
     */
    
    uSetAlarm( uGetTimeSlice() );			/* set time slice to the cluster value */
    
    uScheduler();
    
    uSetAlarm( 0 );					/* turn off time slicing */
    
    uFreeTask( task );					/* deallocate the kernel task for this processor */
    
    uV( &(processor->done) );				/* signal that processor is finished */
    
    exit( 0 );						/* finally, this UNIX process terminates. */
} /* uCPU */

#endif

#ifdef __U_MULTI__

static uProcessor uEmitProcessor( uCluster cluster ) {

    /*
     * This function creates a new virtual processor on the specified cluster.
     * This function will return after the virtual processor
     * is created and just before the virtual processor will
     * start executing tasks.
     */

    volatile uProcessor processor;

#ifdef __U_DEBUG__
    void *msgp;
    int msglen;
#endif

    int size;
    int pid;
    void *sp;
    void *fp;
    uStack stack;

    /*
     * Allocate and initialize the resources necessary
     * to get a new virtual processor operational.
     */

#ifdef __U_DEBUG__
    uExcept {
#endif

	processor = uAllocProcessor();			/* allocate a processor descriptor */

#ifdef __U_DEBUG__
    } uHandlers {
	uWhen( NULL, &uOutOfMemoryEx, &msgp, &msglen ) {
	    uCreateProcessorExMsg msg;
	    
	    msg.msg = "uLongSetProcessors: uProcessor creation failed due to memory allocation problems.\n";
	    msg.num_proc = uLongGetProcessors( cluster );
	    
	    uRaise( cluster, &uCreateProcessorEx, &msg, sizeof(msg) );
	} /* uWhen */
    } uEndExcept;
#endif

    processor->cluster = cluster;

    /*
     * Before a new UNIX process can be forked, the stack has to be switched to
     * the UNIX process associated with the processor that is executing this task.
     * If the processor stack is not used, the new UNIX process will be executing
     * on the current task's shared stack, which causes some major difficulties.
     */

    /*
     * Save the parameters of the task frame, and calculate the size of the frame.
     */

    sp = uReadStackPointer();				/* save the task stack pointer */
    fp = uReadFramePointer();				/* save the task frame pointer */
    size = fp - sp;					/* calculate the size of this frame */

    /*
     * Save the frame of the current task.
     */
    
    stack = &(uWorkTask->StackData);			/* if the work task is the kernel task, */
    stack->sp = sp;					/* make sure the kernel task saves its sp and, */
    stack->fp = fp;					/* make sure the kernel task saves its fp */
    
    /*
     * Create a frame on the kernel task.
     */

    stack = &(uSchedulerTask->StackData);
    stack->sp -= sizeof( stack->fp );			/* make room to store the old frame pointer */
    uPoke( stack->sp, (int) stack->fp );		/* save the old frame pointer on the stack */
    stack->fp = stack->sp;				/* move the frame pointer to the new frame */
    stack->sp -= size;					/* make room on the stack for the frame */

    uCopy( sp, stack->sp, size );			/* copy the stack */

    /*
     * Now, the stack of the UNIX process is installed.
     */

    uWriteStackPointer( stack->sp );			/* write the new stack pointer */
    uWriteFramePointer( stack->fp );			/* write the new frame pointer */

    /*
     * Finally, a new UNIX process can be safely forked.
     */

    pid = uFork( uCPU, processor );

    /*
     * If the value of pid is zero, then this is the thread of the
     * new UNIX process.  Notice that pid will only be zero in the
     * BSD version of the uSystem.  The IRIX version of uFork()
     * actually starts a processor running the uCPU() function directly.
     */

    if ( pid == 0 ) {
	uCPU( processor );
    } /* if */

    /*
     * If a UNIX process cannot be allocate, cause an error to occur.
     */

    processor->pid = pid;

    /*
     * Remove the frame from the kernel task.
     */
    
    stack = &(uSchedulerTask->StackData);
    stack->sp = stack->fp;				/* move the UNIX stack pointer back a frame*/
    stack->fp = (void *) uPeek( stack->sp );		/* pop the old UNIX frame pointer from the stack */
    stack->sp += sizeof( stack->fp );			/* decrement the UNIX stack pointer */

    /*
     * Restore the frame of the working task.
     */

    stack = &(uWorkTask->StackData);
    sp = stack->sp;
    fp = stack->fp;

    uWriteStackPointer( sp );				/* write the task stack pointer */
    uWriteFramePointer( fp );				/* write the task frame pointer */

#ifdef __U_DEBUG__
    /* May raise exception if fork failed */
    if( processor->pid == -1 ) {
	uCreateProcessorExMsg msg;

	uFreeProcessor( processor );

	msg.msg = "uLongSetProcessors: Cannot emit processor due to failure of Unix process creation.\n";
	msg.num_proc = uLongGetProcessors( cluster );

	uRaise( cluster, &uCreateProcessorEx, &msg, sizeof( msg ) );
    } /* if */
#endif

    /*
     * Return a reference to the new process.
     */

    return( processor );
} /* uEmitProcessor */

#endif

#ifdef __U_MULTI__
    
static void uAbsorbProcessor( uProcessor processor ) {

    /*
     * This function absorbs a virtual processor.
     */

    processor->stop = U_TRUE;				/* schedule the processor for termination */
    kill( processor->pid, SIGALRM );			/* give it a nudge, to start termination */

    uP( &(processor->done) );				/* wait until it has terminated */

    uFreeProcessor( processor );			/* deallocate the processor */
} /* uAbsorbProcessor */

#endif

uCluster uCreateCluster( int num, long time ) {

    /*
     * This function emits a new cluster.
     */

#ifdef __U_MULTI__
    
#ifndef __U_DEBUG__

    uCluster cluster;

#else

    volatile uCluster cluster = NULL;
    void              *msgp;
    int               msglen;

    uExcept {

#endif

	cluster = uAllocCluster();
	cluster->timeslice = time;				/* set the time slice duration for this new cluster */

	uLongSetProcessors( cluster, num );

#ifdef __U_DEBUG__
    } uHandlers {
	uWhen( NULL, &uOutOfMemoryEx, &msgp, &msglen ) {
	    uCreateClusterExMsg msg;

	    msg.msg = "uCreateCluster: Cannot create cluster due to memory allocation problems\n";
	    msg.cv.TimeSlice = time;
	    msg.cv.ArgLen = U_ARG_LEN;
	    msg.cv.StackSize = U_STACK_SIZE;
	    msg.cv.Spin = U_SPIN;
	    msg.cv.Processors = num;

	    uRaise( uWorkCoroutine, &uCreateClusterEx, &msg, sizeof( msg ) );
	} /* uWhen */
	uWhen( cluster, &uCreateProcessorEx, &msgp, &msglen ) {
	    uCreateClusterExMsg msg;

	    msg.msg = "uCreateCluster: Cannot create cluster due to problems in uProcessor creation\n";
	    msg.cv.TimeSlice = time;
	    msg.cv.ArgLen = U_ARG_LEN;
	    msg.cv.StackSize = U_STACK_SIZE;
	    msg.cv.Spin = U_SPIN;
	    msg.cv.Processors = num;

	    uLongSetProcessors( cluster, 0 );
	    uFree( cluster );

	    uRaise( uWorkCoroutine, &uCreateClusterEx, &msg, sizeof( msg ) );
	} /* uWhen */
    } uEndExcept;
#endif

    return( cluster );					/* allocate and return a reference to the cluster descriptor */

#else

    return( uThisCluster() );				/* return a reference to the local cluster */

#endif
} /* uCreateCluster */

uCluster uLongCreateCluster( uClusterVars *cv ) {

    /*
     * This function emits a new cluster.
     */

#ifdef __U_MULTI__
    
#ifndef __U_DEBUG__

    uCluster cluster;

#else

    volatile uCluster cluster;
    void              *msgp;
    int               msglen;

    uExcept {

#endif

	cluster = uAllocCluster();
	cluster->timeslice = cv->TimeSlice;			/* set the time slice duration for this new cluster */
	cluster->spin = cv->Spin;				/* set the spin duration for this new cluster */
	cluster->stacksize = cv->StackSize;
	cluster->arglen = cv->ArgLen;

	uLongSetProcessors( cluster, cv->Processors );

#ifdef __U_DEBUG__
    } uHandlers {
	uWhen( NULL, &uOutOfMemoryEx, &msgp, &msglen ) {
	    uCreateClusterExMsg msg;

	    msg.msg = "uCreateCluster: Cannot create cluster due to memory allocation problems\n";
	    msg.cv = *cv;

	    uRaise( uWorkCoroutine, &uCreateClusterEx, &msg, sizeof( msg ) );
	} /* uWhen */
	uWhen( cluster, &uCreateProcessorEx, &msgp, &msglen ) {
	    uCreateClusterExMsg msg;

	    msg.msg = "uCreateCluster: Cannot create cluster due to problems in uProcessor creation\n";
	    msg.cv = *cv;

	    uLongSetProcessors( cluster, 0 );
	    uFree( cluster );

	    uRaise( uWorkCoroutine, &uCreateClusterEx, &msg, sizeof( msg ) );
	} /* uWhen */
    } uEndExcept;
#endif


    return( cluster );
    
#else

    return( uThisCluster() );				/* return a reference to the local cluster */

#endif
} /* uLongCreateCluster */

void uDestroyCluster( uCluster cluster ) {
    
    /*
     * This function destroys a cluster.
     */

#ifdef __U_MULTI__

    if ( uTaskQueueEmpty( &(cluster->ready) ) ) {	/* if no work on the ready queue */
	uLongSetProcessors( cluster, 0 );		/* remove all the processors */
	uFreeCluster( cluster );			/* free the cluster descriptor */
#ifdef __U_DEBUG__
    } else {
	uActiveTasksExMsg msg;

	msg = "uDestroyCluster: DESTROYING CLUSTER WITH ACTIVE TASKS\n";

	uRaise( cluster, &uActiveTasksEx, &msg, sizeof(msg) );
#endif
    } /* if */

#endif
} /* uDestroyCluster */

inline uCluster uThisCluster( void ) {

    /*
     * This function returns a reference to the local cluster.
     */

    return( uWorkTask->cluster );			/* return reference to the cluster associated with the working task */
} /* uThisCluster */

void uLongSetProcessors( uCluster cluster, int num ) {

#ifdef __U_MULTI__

    uProcessorStack *cpus = &(cluster->cpus);
    uProcessor processor;

#ifdef __U_DEBUG__

    if ( num < 0 ) {
	uCreateProcessorExMsg msg;

	msg.msg = "uLongSetProcessors: ATTEMPT TO SET NUMBER OF PROCESSORS TO VALUE LESS THAN 0\n";
	msg.num_proc = uLongGetProcessors( cluster );

	uRaise( cluster, &uCreateProcessorEx, &msg, sizeof( msg ) );
    } /* if */

#endif

    while ( num > uProcessorStackLength( cpus ) ) {	/* increase the number of processors on this cluster */
	processor = uEmitProcessor( cluster );
	uAcquireProcessorStack( cpus );
	uPushProcessor( cpus, processor );
	uReleaseProcessorStack( cpus );
    } /* while */

    while ( num < uProcessorStackLength( cpus ) ) {	/* decrease the number of processors on this cluster */
	uAcquireProcessorStack( cpus );
	processor = uPopProcessor( cpus );
	uReleaseProcessorStack( cpus );
	uAbsorbProcessor( processor );
    } /* while */

#endif
} /* uLongSetProcessors */

inline int uLongGetProcessors( uCluster cluster ) {
    return( uProcessorStackLength( &(cluster->cpus) ) );
} /* uLongGetProcessors */

inline void uSetProcessors( int num ) {
    uLongSetProcessors( uThisCluster(), num );
} /* uSetProcessors */

inline int uGetProcessors( void ) {
    return( uLongGetProcessors( uThisCluster() ) );
} /* uGetProcessors */

void uLongSetTimeSlice( uCluster cluster, long msecs ) {

    uProcessor processor;
    uProcessorStack *cpus;
    long osecs;
    
    /* read the old slice */
    osecs = cluster->timeslice;

    /* if the slice has changed */
    if ( msecs != osecs ) {

	/* write the new slice */
	cluster->timeslice = msecs;

	/* if the old slice was 0, or greater than the new slice, send a signal to each processor on this cluster */
	if ( osecs == 0 || osecs > msecs ) {
	    cpus = &(cluster->cpus);			/* grab a pointer to the beginning of the list. */
	    uAcquireProcessorStack( cpus );		/* acquire processor stack for entire duration of time slice change */
	    processor = uProcessorStackTop( cpus );
	    while ( processor != NULL ) {		/* run through the list of processors on this cluster */
		kill( processor->pid, SIGALRM );	/* and interrupt each so they will update their time slice duration */
		processor = uProcessorStackLink( processor );
	    } /* while */
	    uReleaseProcessorStack( cpus );		/* release processor stack */
	} /* if */
    } /* if */
} /* uLongSetTimeSlice */

inline long uLongGetTimeSlice( uCluster cluster ) {
    return( cluster->timeslice );
} /* uLongGetTimeSlice */

inline void uSetTimeSlice( long msecs ) {
    uLongSetTimeSlice( uThisCluster(), msecs );
} /* uSetTimeSlice */

inline long uGetTimeSlice( void ) {
    return( uLongGetTimeSlice( uThisCluster() ) );
} /* uGetTimeSlice */

inline void uLongSetSpin( uCluster cluster, long usecs ) {
    cluster->spin = usecs;
} /* uLongSetSpin */

inline long uLongGetSpin( uCluster cluster ) {
    return( cluster->spin );
} /* uLongGetSpin */

inline void uSetSpin( long usecs ) {
    uLongSetSpin( uThisCluster(), usecs );
} /* uSetSpin */

inline long uGetSpin( void ) {
    return( uLongGetSpin( uThisCluster() ) );
} /* uGetSpin */

inline void uLongSetStackSize( uCluster cluster, long size ) {
    cluster->stacksize = size;
} /* uLongSetStackSize */

inline long uLongGetStackSize( uCluster cluster ) {
    return( cluster->stacksize );
} /* uLongGetStackSize */

inline void uSetStackSize( long size ) {
    uLongSetStackSize( uThisCluster(), size );
} /* uSetStackSize */

inline long uGetStackSize( void ) {
    return( uLongGetStackSize( uThisCluster() ) );
} /* uGetStackSize */

inline void uLongSetArgLen( uCluster cluster, long len ) {
    cluster->arglen = len;
} /* uLongSetArgLen */

inline long uLongGetArgLen( uCluster cluster ) {
    return( cluster->arglen );
} /* uLongGetArgLen */

inline void uSetArgLen( long len ) {
    uLongSetArgLen( uThisCluster(), len );
} /* uSetArgLen */

inline long uGetArgLen( void ) {
    return( uLongGetArgLen( uThisCluster() ) );
} /* uGetArgLen */

inline int uLongReadyTasks( uCluster cluster ) {
    return( uTaskQueueLength( &(cluster->ready) )) ;
} /* uLongReadyTasks */
    
inline int uReadyTasks( void ) {
    return( uLongReadyTasks( uThisCluster() ) );
} /* uReadyTasks */

inline void uSetName( char *name ) {
    uWorkCoroutine->name = name;
} /* uSetName */

inline char *uGetName( uCoroutine cid ) {
    if ( cid->name == NULL ) {
 	sprintf( cid->hex, "0x%x", cid );
	cid->name = cid->hex;
    } /* if */
    return cid->name;
} /* uGetName */

void uSaveFixed( void ) {
    uWorkCoroutine->save = U_SAVE_FIXED;		/* save only the fixed point registers for this task */
} /* uSaveFixed */

void uSaveFloat( void ) {
    uWorkCoroutine->save = U_SAVE_FLOAT;		/* save fixed and floating pointer registers for this task */
} /* uSaveFloat */

static void uFirst( int argc, char *argv[], char *envp[] ) {

    void InstallStandardStream( void );
    void DestallStandardStream( void );
    
    int result = 0;
    uCluster cluster;

    uInstallStandardStream();				/* create the standard streams */

    cluster = uCreateCluster( 1, U_TIME_SLICE );	/* create the user cluster */

    uAbsorb( uLongEmit( cluster, uLongGetStackSize( cluster ), uStart, 0 ), NULL, 0 ); /* initialize the user cluster */

    /* execute the user application */
    uAbsorb( uLongEmit( cluster, uLongGetStackSize( cluster ), uMain, sizeof( argc ) + sizeof( argv ) + sizeof( envp ), argc, argv, envp ), &result, sizeof( result ) );

    uDestroyCluster( cluster );				/* destroy the user cluster */

    uDestallStandardStream();				/* destall the standard streams */

    uWorkProcessor->stop = U_TRUE;			/* schedule processor for termination */
    
    uDie( &result, sizeof( result ) );			/* return result from uMain */
} /* uFirst */

int uKernel( int argc, char **argv, char **envp ) {

    uCluster uSystemCluster;
    uProcessor uSystemProcessor;
    uTask uSystemTask;
    uTask uFirstTask;
    int result = 0;
    
    /*
     * Create and initialize hardware locks.
     */

    uCreateLocks();
    
    /*
     * Initialize the signal handlers for this UNIX process.
     * These signal handlers must be inherited by each UNIX process that
     * is subsequently forked.
     */

#ifdef __sgi__
    uSignal( SIGCHLD, (void (*)( int, ... )) uSigChld );
    uSignal( SIGALRM, (void (*)( int, ... )) uSigAlrm );
#else
    uSignal( SIGCHLD, uSigChld );
    uSignal( SIGALRM, uSigAlrm );
#endif
    
    /*
     * It is necessary to bootstrap the system cluster so that it works
     * in both uniprocessor and multiprocessor forms.
     */
    
    uSystemCluster = uAllocCluster(); 
    
    /*
     * It is necessary to bootstrap the system cluster's processor because it already exists.
     */

    uSystemProcessor = uAllocProcessor();		/* allocate the system processor, initialize it, */
    uSystemProcessor->cluster = uSystemCluster;		/* and add it to the system cluster */

    uSystemProcessor->pid = getpid();			/* retain the pid of this processor */

    uPushProcessor( &(uSystemCluster->cpus), uSystemProcessor ); /* push this processor on the system cluster */

    /*
     * It is necessary to bootstrap the system task on the system processor.
     */

    uSystemTask = uAllocTask( 0 );			/* allocate the system task and initialize it */
    uSystemTask->cluster = uSystemCluster;

    uSystemTask->mutex = U_LOCKED;			/* since this task is self starting acquire the task */
    uSystemTask->state = U_RUN;				/* and set its state to running */

    uWorkProcessor = uSystemProcessor;			/* set the local variables for the UNIX process */
    uWorkTask = uSystemTask;
    uWorkCoroutine = uWorkTask->cortn;
    uSchedulerTask = uWorkTask;

#ifdef __U_MULTI__					/* for efficiency reasons, the alarm is turned off on */
    uSetTimeSlice( 0 );					/* the system cluster in the multiprocessor version, */
#endif							/* don't want all those signals being generated. */
    
    uSetAlarm( uGetTimeSlice() );			/* start the alarm */

    uFirstTask = uEmit( uFirst, argc, argv, envp );	/* emit the system task */

    uScheduler();					/* call the scheduler to execute other tasks */
    
    uAbsorb( uFirstTask, &result, sizeof( result ) );	/* absorb the system task */

    uSetAlarm( 0 );					/* stop the alarm */
    
    uFreeTask( uSystemTask );				/* free the system resources */
    uFreeProcessor( uSystemProcessor );
    uFreeCluster( uSystemCluster );

    /*
     * Destroy hardware locks.
     */

    uDestroyLocks();
    
    return( result );					/* return the value from the user defined main */
} /* uKernel */

/* Local Variables: */
/* compile-command: "dmake" */
/* End: */
