/*                               -*- Mode: C -*- 
 * 
 * uSystem Version 4.3.2, Copyright (C) Peter A. Buhr and Richard A. Stroobosscher 1990
 * 
 * uKernel.h -- Declaration of objects that the uSystem makes available to the user.
 * 
 * Author           : Rick Stroobosscher
 * Created On       : Fri Feb  9 15:23:30 1990
 * Last Modified By : Peter A. Buhr
 * Last Modified On : Fri Mar  1 15:44:27 1991
 * Update Count     : 108
 */

#ifndef __U_KERNELH__
#define __U_KERNELH__

/*
 * USER CODE MAY BE WRITTEN TO DEPEND ON THE FOLLOWING MACROS AND TYPE DEFINITIONS
 * BUT NOT ON THEIR INTERNAL STRUCTURE:
 *
 *   NULL, uSemaphore, U_SEMAPHORE, uTask, uCoroutine, uCluster,
 *   uClusterVars, U_CLUSTER_VARS
 *
 * NO USER CODE SHOULD BE WRITTEN TO DEPEND ON ANY OTHER MACROS OR TYPE DEFINITIONS.
 */

/*
 * Macro definitions.
 */

#ifndef NULL
#define NULL 0
#endif
#define U_NULL 0

#define U_LOCKED ( (uLock) 1 )
#define U_UNLOCKED ( (uLock) 0 )

#define U_FALSE 0
#define U_TRUE 1

/*
 * Definition of a lock.
 */

typedef int uLock;

/*
 * Definition of a queue of tasks,
 * and its associated constructor.
 */

/* WARNING: monitor preprocessor depends on uTaskQueue */

typedef struct uTaskQueue {
    uLock mutex;					/* lock to ensure mutual exclusion */
    int len;						/* number of tasks on queue */
    int idle;						/* flag for idle processors */
    struct uTaskD *head;				/* pointer to first task in queue */
    struct uTaskD *tail;				/* pointer to last task in queue */
} uTaskQueue;

#ifdef __GNUC__
#define U_TASK_QUEUE() ( (uTaskQueue) { U_UNLOCKED, 0, U_FALSE, NULL } )
#else
#define U_TASK_QUEUE() { U_UNLOCKED, 0, U_FALSE, NULL }
#endif
    
/*
 * Definition of a semaphore,
 * and its associated constructor.
 */

typedef uTaskQueue uSemaphore;    

#ifdef __GNUC__
#define U_SEMAPHORE( V ) ( (uTaskQueue) { U_UNLOCKED, -(V), U_FALSE, NULL } )
#else
#define U_SEMAPHORE( V ) { U_UNLOCKED, -(V), U_FALSE, NULL }
#endif
    
/*
 * Definition of cluster variable control block,
 * and its associated constructor.
 */

typedef struct uClusterVars {
    long Processors;
    long TimeSlice;
    long Spin;
    long StackSize;
    long ArgLen;
} uClusterVars;

#ifdef __GNUC__
#define U_CLUSTER_VARS() ( (uClusterVars) { 1, U_TIME_SLICE, U_SPIN, U_STACK_SIZE, U_ARG_LEN } )
#else
#define U_CLUSTER_VARS() { 1, U_TIME_SLICE, U_SPIN, U_STACK_SIZE, U_ARG_LEN }
#endif
    
typedef struct uTaskD *uTask, *uCoroutine;
typedef struct uProcessorD *uProcessor;
typedef struct uClusterD *uCluster;

#ifdef __U_KERNEL__

/* Some cluster variables depend on some definitions */

/*
 * Definition of a stack of processors,
 * and its associated constructor.
 */

typedef struct uProcessorStack {
    uLock mutex;					/* lock to ensure mutual exclusion */
    int len;						/* number of processors on stack */
    struct uProcessorD *top;				/* pointer to processor on top of stack */
} uProcessorStack;

#ifdef __GNUC__
#define U_PROCESSOR_STACK() ( (uProcessorStack) { U_UNLOCKED, 0, NULL } )
#else
#define U_PROCESSOR_STACK() { U_UNLOCKED, 0, NULL }
#endif
    
/*
 * Definition of a stack descriptor,
 * and its associated constructor.
 */

typedef struct uStackD {
    void *base;						/* pointer to base of stack */
    void *limit;					/* pointer to limit of stack */
    void *sp;						/* current stack pointer */
    void *fp;						/* current frame pointer */
    void *pc;						/* return address */
} *uStack;

/*
 * Definition of an ipc descriptor,
 * and its associated constructor.
 */

typedef struct uMsgD {
    uSemaphore snd;					/* semaphore used by tasks sending messages */
    uSemaphore rcv;					/* semaphore used by task receiving message */
    struct uTaskD *to;					/* pointer to task receiving message */
    struct uTaskD *from;				/* pointer to task sending message */
    void *sbuf;						/* address of send or reply message */
    int slen;						/* length of send or reply message */
    void *rbuf;						/* address of receive message */
    int rlen;						/* length of receive message */
} *uMsg;

#ifdef __GNUC__
#define U_MSG() ( (struct uMsgD) { U_SEMAPHORE( 0 ), U_SEMAPHORE( 0 ), NULL } )
#else
#define U_MSG() { U_SEMAPHORE( 0 ), U_SEMAPHORE( 0 ), NULL }
#endif
    
/*
 * This definition is required since a forward
 * declaration of an Exception block is needed
 */

typedef struct uEBlockD *uEBlock;

/*
 * Definition of a task descriptor,
 * and its associated constructor.
 */

struct uTaskD {
    uLock mutex;					/* provide mutual exclusion on task execution */
    int state;						/* current state of task or coroutine */
    int save;						/* are the floating pointd registers saved */
    uSemaphore done;					/* done semaphore */
    struct uMsgD ipc;					/* IPC information */
    char *name;						/* name of the task */
    uEBlock exceptstack;				/* Exception stack for the coroutine */
    int rlen;						/* length of resume message buffer */
    void *rbuf;						/* address of resume message buffer */
    struct uStackD StackData;				/* stack information */
    struct uClusterD *cluster;				/* cluster task is executing on */
    struct uTaskD *cortn;				/* current coroutine executing */
    void (*begin)();					/* pointer to subroutine started as a coroutine */
    struct uTaskD *owner;				/* task that created this coroutine */
    struct uTaskD *restarter;				/* last coroutine to restart coroutine, used to deallocate resources */
    struct uTaskD *resumer;				/* last cocalling or resuming coroutine */
    struct uTaskD *link;				/* link for ready queue */
    char hex[11];					/* area big enough to hold 8 digit hex value, leading 0x and trailing \0 */
};

#ifdef __GNUC__
#define U_TASK() ( (struct uTaskD) { U_UNLOCKED, U_NEW, U_SAVE_FIXED, U_SEMAPHORE( 0 ), U_MSG(), NULL, NULL, 0 } )
#else
#define U_TASK() { U_UNLOCKED, U_NEW, U_SAVE_FIXED, U_SEMAPHORE( 0 ), U_MSG(), NULL, NULL, 0 }
#endif

#define U_SAVE_FIXED 0
#define U_SAVE_FLOAT 1

#define U_NEW 1
#define U_RUN 2
#define U_OLD 3

/*
 * Definition of a processor descriptor,
 * and its associated constructor.
 */

struct uProcessorD {
    int stop;						/* stop flag */
    int state;						/* state */
    uSemaphore done;					/* done semaphore */
    int pid;						/* process id */
    struct uClusterD *cluster;				/* pointer to cluster on which this processor executes */
    struct uProcessorD *link;				/* link used to chain processors together on processor stack */
};

#ifdef __GNUC__
#define U_PROCESSOR() ( (struct uProcessorD) { U_FALSE, U_BUSY, U_SEMAPHORE( 0 ) } )
#else
#define U_PROCESSOR() { U_FALSE, U_BUSY, U_SEMAPHORE( 0 ) }
#endif
    
#define U_BUSY 0
#define U_IDLE 1

/*
 * Definition of a cluster descriptor,
 * and its associated constructor.
 */

struct uClusterD {
    long arglen;					/* default argument size */
    long stacksize;					/* default stack size */
    long timeslice;					/* default processor time slice */
    long spin;						/* default processor spin before sleep time */
    struct uTaskQueue ready;				/* ready queue */
    struct uProcessorStack cpus;			/* list of processors for this cluster */
    uSemaphore mutex;
    int nfds;
    fd_set rfds;
    fd_set wfds;
    fd_set efds;
};

#ifdef __GNUC__
#define U_CLUSTER() ( (struct uClusterD) { U_ARG_LEN, U_STACK_SIZE, U_TIME_SLICE, U_SPIN, U_TASK_QUEUE(), U_PROCESSOR_STACK(), U_SEMAPHORE( 1 ), 0 } )
#else
#define U_CLUSTER() { U_ARG_LEN, U_STACK_SIZE, U_TIME_SLICE, U_SPIN, U_TASK_QUEUE(), U_PROCESSOR_STACK(), U_SEMAPHORE( 1 ), 0 }
#endif
    
#endif

/*
 * These macros return the ceiling and floor of values V
 * of multiples of value A.  In other words, if A is 8, and
 * V is 6, U_CEILING returns 8, while U_FLOOR returns 0.
 * If V is already a multiple of A, both macros return
 * the same value.
 */
    
#define U_CEILING( V, A ) ( ( (V) + (A) - 1 ) & ~ ( (A) - 1 ) )
#define U_FLOOR( V, A ) ( (V) & ~ ( (A) - 1 ) )

/*
 * These are the functions that the uKernel provides.
 */

extern void uError( char *, ... );

extern void uP( uSemaphore * );
extern void uV( uSemaphore * );
extern int uC( uSemaphore * );

extern uCluster uCreateCluster( int, long );
extern uCluster uLongCreateCluster( uClusterVars * );
extern void uDestroyCluster( uCluster );
extern uCluster uThisCluster( void );

/*
  The prototypes for these routines are getting more and more difficult to
  specify because of changes in ANSI C.  The problem is that void (*)() no
  longer matches any function type because of the promotion rules. Why isn't
  void(*)(...) allowed?  I wish ANSI would get this right.
  */

#if defined( __cplusplus ) || defined( c_plusplus )
    /*
     * Now C++ will not match a function pointer with void *
     */
    extern uTask uEmit( ... );			/* C++ 1.2.1 bug prevents the 1st parameter from being void (*)() */
    extern uTask uLongEmit( uCluster, long, void (*)(), long, ... );
    extern uCoroutine uCocall( void *, int, void (*)(), ... );
    extern uCoroutine uLongCocall( void *, int, long, void (*)(), long, ... );
#else
    extern uTask uEmit( void *, ... );
    extern uTask uLongEmit( uCluster, long, void *, long, ... );
    extern uCoroutine uCocall( void *, int, void *, ... );
    extern uCoroutine uLongCocall( void *, int, long, void *, long, ... );
    extern void uMain();
#endif

extern void uAbsorb( uTask, void *, int );
extern void uDie( void *, int );
extern uTask uThisTask( void );

extern void uSuspend( void *, int, void *, int );
extern void uResume( uCoroutine, void *, int, void *, int );
extern void uSuspendDie( void *, int );
extern void uResumeDie( uCoroutine, void *, int );
extern uCoroutine uThisCoroutine();

extern uTask uSend( uTask, void *, int, void *, int );
extern uTask uReceive( void *, int );
extern void uReply( uTask, void *, int );
extern void uForward( uTask, void *, int, uTask );

extern uCluster uMigrate( uTask, uCluster );

extern void uVerify( void );

extern void uYield( void );

extern void uLongSetTimeSlice( uCluster, long );
extern long uLongGetTimeSlice( uCluster );
extern void uSetTimeSlice( long );
extern long uGetTimeSlice( void );

extern void uLongSetSpin( uCluster, long );
extern long uLongGetSpin( uCluster );
extern void uSetSpin( long );
extern long uGetSpin( void );

extern void uLongSetStackSize( uCluster, long );
extern long uLongGetStackSize( uCluster );
extern void uSetStackSize( long );
extern long uGetStackSize( void );

extern void uLongSetArgLen( uCluster, long );
extern long uLongGetArgLen( uCluster );
extern void uSetArgLen( long );
extern long uGetArgLen( void );

extern uProcessor uThisProcessor( void );
extern void uLongSetProcessors( uCluster, int );
extern int uLongGetProcessors( uCluster );
extern void uSetProcessors( int );
extern int uGetProcessors( void );

extern void uSetName( char * );
extern char *uGetName( uCoroutine );

extern int uLongReadyTasks( uCluster );
extern int uReadyTasks( void );

extern void uSaveFixed( void );
extern void uSaveFloat( void );

extern int uKernel( int, char *[], char *[] );

#endif
