/*                               -*- Mode: C -*- 
 *
 * uSystem Version 4.3.2, Copyright (C) Hamish Macdonald 1990
 * 
 * uExcept.h -- uSystem exception handling declarations.
 * 
 * Author          : Hamish Macdonald
 * Created On      : Mon Jun 11 15:02:17 1990
 * Last Modified By: Peter A. Buhr
 * Last Modified On: Tue Dec 18 22:31:41 1990
 * Update Count    : 161
 */

#ifndef __U_EXCEPTH__
#define __U_EXCEPTH__

/*
 * Definition of an exception and its associated
 * constructor.  Exceptions can _only_ be initialized.  They
 * cannot be changed after they are initialized.
 */

typedef const struct _uException {
    const struct _uException *parent;			/* constant pointer to a constant uException */
} uException;

extern uException uAny;					/* The "catch-all" exception pointer */

/*
 * Initialization macro for the uException type.
 *
 * uExceptions _must_ be initialized when declared.
 *
 * uExceptions are constant, and cannot be changed after their declaration.
 *
 */

#define U_EXCEPTION( except ) { (const struct _uException*)except }

/*
 * Definition of the context for an exception
 * block.
 * The U_EXCEPT_CONTEXT_SIZE is a parameter found in the
 * uMachine.h header file.
 */

typedef unsigned long uExceptContext[U_EXCEPT_CONTEXT_SIZE];

/*
 * Definition of an exception block descriptor,
 * and its associated constructor.
 */ 
typedef struct uEBlockD {
    struct uEBlockD  *prev;
    uException *exception;
    void             *object;
    void             *data;
    int              len;
    void             *return_addr;
    int              reraise;
    uExceptContext   context;
} uEBlockD;

#ifndef __U_KERNEL__
typedef uEBlockD *uEBlock;
#endif

#define U_EBLOCKD( prev ) ( (uEBlockD) { prev, NULL, NULL, 0, NULL, U_FALSE } )

/*
 * Definition of macros for setting up exception handlers on
 * a tasks stack.
 */

#define uExcept                               \
    {                                         \
	uEBlockD _uExcpBlock;                 \
	if( !uExceptSave( &_uExcpBlock ) ) {

#define uHandlers                        \
            uExceptEnd( &_uExcpBlock );  \
	} else {                         \
            if( U_FALSE ) {

#define uWhen(_obj,_excp,_dataptr,_lenptr)       \
		uExceptEnd( &_uExcpBlock );      \
	    } } else if( uExceptCheck( &_uExcpBlock, (_obj), (_excp), (_dataptr), (_lenptr) ) ) { {

#define uEndExcept                                                                                      \
		uExceptEnd( &_uExcpBlock );                                                             \
            } else {                                                                                    \
		_uExcpBlock.reraise = U_TRUE;                                                           \
                uRaise( _uExcpBlock.object, _uExcpBlock.exception, _uExcpBlock.data, _uExcpBlock.len ); \
	    }                                                                                           \
        }                                                                                               \
    }

/*
 * This macro allows the user to access the raised exception within an exception handler.
 * This is most useful in an exception handler catching any exception (uAny).
 */
#define uRaisedException() (_uExcpBlock.exception)
/*
 * This macro allows the user to access the object on which the exception was raised within an exception handler.
 */
#define uRaisedInstance() (_uExcpBlock.object)

extern void uRaise( void*, uException*, void*, int );
extern void uCause( uTask, uException*, void*, int );
extern int uExceptSave( uEBlock );
extern void uExceptEnd( uEBlock );
extern int uExceptCheck( uEBlock, void*, uException*, void*, int* );

#endif
