/* $Header: sds_assem.c,v 1.15 93/01/13 16:57:36 vern Exp $ */

/**************************************************************************
 *                 ****** ISTK Release 1.2 *****                          *
 *                                                                        *
 *                                                                        *
 * This code has been produced by numerous authors at the CERN centre for *
 * high energy physics, Geneve, Switzerland, at the SSC laboratory in     *
 * Dallas, Texas, USA and at the Lawrence Berekeley Laboratory in         *
 * California, USA.                                                       *
 * The latter two institutions perform work under US Government contract. *
 * The intent of the work is to provide useful code for people who need   *
 * it, with an emphasis on free and collaborative exchange of ideas,      *
 * techniques and implementations.                                        *
 * Please read the disclaimer and copyright notices contained in the ISTK *
 * distribution and in distributed applications.                          *
 *                                                                        *
 **************************************************************************/



/* Reference release  Aug 10 1991 - C G Saltmarsh */
/* Has the basics used at CDG & SSC 1988-1991, plus vxworks
   support
*/



#include <string.h>
#include <stdlib.h>

#ifndef vms
#include <unistd.h>
#include <memory.h>
#endif

#if defined(vms)
#include "sdsgen.h"
#include "sds_externs.h"
#else
#include "Sds/sdsgen.h"
#include "Sds/sds_externs.h"
#endif

#define  SDS_INC_DIREC 32
#define  SDS_INC_SUBS  32
#define  SDS_INC_HEAP  256
#define  SDS_FIXED     -1

#define SDS_SAVE_STATE 1
#define SDS_RESTORE_STATE 2


static int sds_padbytes[4] = {0,0,0,0};

#if !defined(psos)
extern char  *strncpy();
#endif

#if !defined(hpux) && !defined(vxworks) && !defined(psos) && !defined(mips)
#if !defined(vms)
#if ! defined (__GCC_2__)
#if defined(__GNUC__) && __GNUC__ != 2
extern char  *memcpy();
#endif
#endif
#endif
#endif

extern char  *index();

extern int      sds_error;
extern char     sds_sizes[];
extern char     sds_align[];
extern char     sds_rbytes[];

/* globals to do dataset control */

short          allofl[MAX_SDS];
int            serial_stream[MAX_SDS];
int            sds_is_init = 0;
int            sds_re_init = 0;
int            load_source[MAX_SDS];
int           *dup_sizes[MAX_SDS];
char          *heap[MAX_SDS];
static int     current_direc_size[MAX_SDS];
static struct  sds_header *shead[MAX_SDS];
static char   *current_heap[MAX_SDS];
static int     heap_size[MAX_SDS];
char  ***element_start[MAX_SDS];
int    **varel_count[MAX_SDS];

struct type_list     *tlist[MAX_SDS];
static struct direc  *dptr_index[MAX_SDS];

/*  Forward declarations  */
#ifndef vxworks
off_t           lseek();
#endif
sds_handle sds_prepads();
char     **sds_saverestore();


/*********************************************************************/
void
sds_init()
/*********************************************************************/
{
  int  i;
  if (sds_is_init == 1 && sds_re_init == 0) return;
  for (i=0;i<MAX_SDS;i++) 
  {
    dptr_index[i] = DNULL;
    tlist[i] = TNULL;
    shead[i] = HNULL;
    heap[i] = NULL;
    dup_sizes[i] = INULL;
    serial_stream[i] = SDS_FILE_OP;
    element_start[i] = NULL;
    varel_count[i] = NULL;
  }
  sds_rinit();
  sds_re_init = 0;
  sds_is_init = 1;
}

/*********************************************************************/
void
sds_reinit_enable()
/*********************************************************************/
{
  sds_re_init = 1;
}

/*********************************************************************/
sds_handle
is_sds(name)
char  *name;
/*********************************************************************/
{
  sds_handle  i;
  struct direc *ddptr;

  if (sds_is_init == 0) return(SDS_NOT_INITIALISED);
  if (strcmp(name,"")) 
  {
    for (i=0;i<MAX_SDS;i++) 
    {
      ddptr = dptr_index[i];
      if ((ddptr != DNULL) && 
        !strcmp(name,sds_obind2name(i,(sds_code)0)))
        return(i);
      }
  }
  return(SDS_NO_SUCH_SDS);
}
/*********************************************************************/
sds_handle
next_sds()
/*********************************************************************/
{
  sds_handle sds;

  sds = 0;
  while (dptr_index[sds++] != DNULL)
    if (sds == MAX_SDS) return(SDS_NO_SPC);
  sds--;
  return(sds);
}
/*********************************************************************/
sds_handle
sds_duplicate(old_sds_index,name)
sds_handle  old_sds_index;
char *name;
/*********************************************************************/
{
  sds_handle  new_sds;
  sds_code  object,typecode;
  struct  direc  *dptr,*old_dptr;

  if (sds_is_init == 0) return(SDS_NOT_INITIALISED);

  if ((old_dptr = dptr_index[old_sds_index]) == DNULL)
    return(SDS_NO_SUCH_SDS);

  if (is_sds(name) > -1) return(SDS_DEJA_LA);

  if ((new_sds = sds_new_index(name)) < 0) return(new_sds);
  dptr = sds_direc_ptr(new_sds);

  dup_sizes[new_sds] = (int *)malloc(old_dptr[0].nelems * sizeof(int));

  if (dup_sizes[new_sds] == (char)0 )
  {
    sds_perror("malloc failure, sds_duplicate");
    exit(1);
  }


  for (object = 1;object < old_dptr[0].nelems; object++)
  {
    dup_sizes[new_sds][object] = old_dptr[object].nelems;
    sds_cleanup();
    if ((typecode = old_dptr[object].elemcod) & SDS_INDLIST)
      typecode = sds_duplicate_def(old_sds_index,new_sds,object);

    sds_declare_object(new_sds, 
      sds_obind2ptr(old_sds_index,object),
      sds_obind2name(old_sds_index,object),
      old_dptr[object].nelems,
      typecode);
   dptr[object].structype = old_dptr[object].structype;  
   if (old_dptr[object].illoca == SDS_DISJOINT_OBJECT)
     dptr[object].illoca = SDS_DISJOINT_OBJECT;  
    
  }
  return(new_sds);
}
/*********************************************************************/
sds_handle
sds_duplicate_def(source_sds_index,new_sds_index,object_index)
sds_handle  source_sds_index,new_sds_index;
sds_code  object_index;
/*********************************************************************/
{
  struct   direc    *dptr = sds_direc_ptr(source_sds_index);
  sds_code           typecode =  dptr[object_index].elemcod;

  return sds_type_duplicate_def(
                source_sds_index,new_sds_index,typecode);
}
/*********************************************************************/
void
sds_put_name(sds_index,object_index,name)
sds_handle  sds_index;
sds_code  object_index;
char  *name;
/*********************************************************************/
{
  char  *oname = sds_obind2name(sds_index,object_index);

  strncpy(oname,name,(int)strlen(oname));
}
/*********************************************************************/
sds_handle
sds_pname(sds_index,object_index,name)
sds_handle sds_index;
sds_code  object_index;
char  *name;
/*********************************************************************/
{
  struct  direc*  dptr = sds_direc_ptr(sds_index);

  if ((long)(dptr[object_index].obj_name = 
                         sds_add_to_heap(sds_index,name,','))
      < (long)0) return(sds_error);
  
  return(0);      
}
/*********************************************************************/
char  *
sds_obind2name(sds_index,object_index)
sds_handle sds_index;
sds_code object_index;
/*********************************************************************/
{
  return(sds_oname(sds_index,object_index,SDS_GEN_NAME));
}
/*********************************************************************/
sds_handle
sds_named_elements(sds_index,object_index)
sds_handle  sds_index;
sds_code  object_index;
/************************************************************************/
{
  struct  direc  *dptr = sds_direc_ptr(sds_index);

  if (dptr == DNULL)
    return(sds_error = SDS_NO_SUCH_SDS);
  return((int)(dptr[object_index].obj_name >> 16));
}
  
/*********************************************************************/
char  *
sds_oname(sds_index,object_index,elem)
sds_handle  sds_index;
sds_code  elem,object_index;
/*********************************************************************/
{
  struct  direc  *dptr = sds_direc_ptr(sds_index);
  char    *cptr;

  elem++;
  if (dptr == DNULL) 
  {
    sds_error = SDS_NO_SUCH_SDS;
    return(SNULL);
  }
  if (elem > (dptr[object_index].obj_name >> 16)) 
  {
    sds_error = SDS_NO_SUCH_OBJ;
    return(SNULL);
  }
  cptr = (heap[sds_index] + (dptr[object_index].obj_name & 0xffff));
  cptr = sds_jstr(cptr,elem);
  return(cptr);
}
/*********************************************************************/
sds_handle
sds_new_index(name)
char *name;
/*********************************************************************/
{

  struct direc *ddptr;
  sds_handle  sds;

  if (sds_is_init == 0) return(SDS_NOT_INITIALISED);
  if (is_sds(name) > -1) return(SDS_DEJA_LA);
  if ((sds = next_sds()) < 0) return(sds);

/*  Start-up space for the directory      */
  ddptr = dptr_index[sds] = 
    (struct direc *)calloc((unsigned int)SDS_INC_DIREC,
          (unsigned int)sizeof(struct direc));
  if (ddptr == (char)0 )
  {
    sds_perror("calloc failure, sds_new_index");
    exit(1);
  }
  allofl[sds] = SDS_DPTR_ALLOC;
  current_direc_size[sds] = SDS_INC_DIREC;

/*  Initialise direc[0]: the directory entry    */
  ddptr[0].nelems = (unsigned long)1;
  ddptr[0].elemsz = sizeof(struct direc);
  ddptr[0].align_type = sds_align[SDS_DIRECTORY_STRUCTURE];
  ddptr[0].offst = SDS_NOT_ASSEMBLED;

  tlist[sds] = TNULL;
  shead[sds] = HNULL;

  if ((long)(ddptr[0].obj_name = 
                   sds_add_to_heap(sds,name,(char)0))
      < (long)0) return(sds_error);

/*  And timestamp it for the hell of it      */
  sds_tstamp(sds,SDS_TIMESTAMP_ALL);
  return(sds);
}
/*********************************************************************/
sds_handle
sds_use(source_name,source_type,mode)
char *source_name;
sds_code source_type,mode;
/*   get access to an (alleged?) SDS outside process memory.
 Unless  this involves attaching to shared memory, this means I load
 into process memory.
*/
/*********************************************************************/
{
  struct direc *dptr = 0;
  sds_handle    sds = SDS_NO_SUCH_SDS;
  int           obj;

  if (sds_is_init == 0) return(sds_error = SDS_NOT_INITIALISED);

  if (source_type & SDS_FILE) 
  {
    if (source_type & SDS_DIREC_ONLY) 
    {
      if ((sds = sds_open(source_name)) < 0)
        return(sds_error = sds);
      if ((dptr = sds_load_direc(sds)) == DNULL)
        return(sds_error);
    }
    else 
    {
      if ((sds = next_sds()) < 0) 
        return(sds_error = sds);
      dptr = f2mem(source_name);
      if (dptr == DNULL) 
        return(sds_error);
      set_sys_vars(sds,dptr);
    }
  }
#ifdef SHMEM
  else if (source_type & SDS_SHARED_MEM) 
  {
    if (shm_q(source_name)) 
    { /* it exists! */
      if ((sds = next_sds()) < 0)  
        return(sds);
      if (mode & SDS_READ)
        dptr = (struct direc *)shm_attr(source_name);
      else dptr = (struct direc *)shm_attw(source_name);
    set_sys_vars(sds,dptr);
    }
    else return(sds_error = SDS_NO_SUCH_SDS);
  }
#endif
  else 
  {
    fprintf(stderr,"undefined sds source type: bailing out\n");
    exit(1);
  }
  load_source[sds] = source_type;
  /* clean up in case allocation flags have been left in file (not nice)
  */
  if (source_type != SDS_SHARED_MEM)
    for (obj = 0;obj < dptr[0].nelems; obj++)
      dptr[obj].illoca &= 
         ~(SDS_WAS_ALLOCATED | SDS_EXTERNAL_OBJECT | SDS_REALLOC);
  return(sds);
}

/*********************************************************************/
void
set_sys_vars(sds_index,dptr)
sds_handle  sds_index;
struct direc  *dptr;
{
/*********************************************************************/
/*  Some system variable to set so the functional interface
  will work...
*/
  dptr_index[sds_index] = dptr;
  current_direc_size[sds_index] = SDS_FIXED;
  shead[sds_index] =
    (struct sds_header *)((char *)dptr - (int)dptr[0].offst);
  if (shead[sds_index]->list_size)
    tlist[sds_index] = 
      (struct type_list *)((char *)shead[sds_index] + BASE_OFFSET);
  else
    tlist[sds_index] = TNULL;
  heap[sds_index] = 
    (char *)shead[sds_index] 
      + (int)shead[sds_index]->list_size
      + BASE_OFFSET;
}

/*********************************************************************/
sds_handle
sds_start_heap(sds_index)
sds_handle  sds_index;
/*********************************************************************/
{

  if (heap[sds_index] != NULL) return(0);

  current_heap[sds_index] = 
    heap[sds_index] = 
    malloc(SDS_INC_HEAP);

  if (current_heap[sds_index] == (char)0 )
  {
    sds_perror("malloc failure, sds_start_heap");
    exit(1);
  }
  heap_size[sds_index] = SDS_INC_HEAP;
  if (heap[sds_index] == NULL) return(SDS_NO_MEM);
  allofl[sds_index] |= SDS_HEAP_ALLOC;
  return(0);
}
/*********************************************************************/
sds_handle
sds_inc_heap(sds_index,count)
sds_handle sds_index;
int count;
/*********************************************************************/
{

  int  delta = current_heap[sds_index] - heap[sds_index];

  heap_size[sds_index] += count * SDS_INC_HEAP;

  current_heap[sds_index]  = 
    heap[sds_index] = 
    realloc(heap[sds_index],(unsigned)heap_size[sds_index]);
    if (current_heap[sds_index] == (char)0 )
    {
      sds_perror("realloc failure, sds_inc_heap");
      exit(1);
    }

  current_heap[sds_index] += delta;
  if (heap[sds_index] == NULL) return(SDS_NO_MEM);
  else return(0);
}
/*********************************************************************/
sds_handle
sds_add_to_heap(sds_index,buffer,delim)
sds_handle  sds_index;
char  *buffer,delim;
/*********************************************************************/
{

  long  temp,count,alloc_return = 0;
  int  incr,space_left,size = (int)strlen(buffer) + 1;

  if (heap[sds_index] == NULL) 
  {
    alloc_return = sds_start_heap(sds_index);
  }
  space_left = heap_size[sds_index] - 
      (current_heap[sds_index] - heap[sds_index]);
  if ((incr  = (size - space_left)) > 0) 
  {
    incr = 1 + incr/SDS_INC_HEAP;
    alloc_return = sds_inc_heap(sds_index,incr);
  }
  if ((sds_error = alloc_return) < 0) 
  {
    return(sds_error);
  }

  count = sds_namelist(current_heap[sds_index],buffer,delim);

  temp = (current_heap[sds_index] - heap[sds_index])
    + (count << 16);

  current_heap[sds_index] += size;

  return(temp);
}
/*********************************************************************/
struct direc *
sds_inc_direc_size(sds)
sds_handle  sds;
/*********************************************************************/
{
  current_direc_size[sds] += SDS_INC_DIREC;
  dptr_index[sds] = (struct direc *)
    realloc((char *)dptr_index[sds],
      (unsigned int)(current_direc_size[sds]*sizeof(struct direc)));
  if (dptr_index[sds] == (char)0 )
  {
    sds_perror("realloc failure, sds_inc_direc_size");
    exit(1);
  }
  return(dptr_index[sds]);
}
  
/*********************************************************************/
sds_handle
sds_tlist_add(sds,ty_list)
sds_handle  sds;
struct type_list *ty_list;
/*  Add a new tlist element to an SDS tlist           */
/*********************************************************************/
{
  return(sds_define_object(sds,ty_list,""));
}
/*********************************************************************/
long
tlist_len(tyl)
struct  type_list  *tyl;
/*********************************************************************/
{
  long  len = (long)0;
  if (tyl == TNULL) return(len);
  while ((tyl++)->elemcod != SDS_ENDLIST) len++;
  return(++len);
}
/*********************************************************************/
sds_handle 
sds_define_object(sds_index,ty_list,names)
sds_handle  sds_index;
struct type_list *ty_list;
char  *names;
/*********************************************************************/
{
  long new_size = 0,old_size = 0;
  struct type_list *size_align,tlist_head;
  struct type_list *new_list,*ttemp = ty_list,*oldlist;
  int  i;
  char  align;

  if (sds_direc_ptr(sds_index) == DNULL)
    return(SDS_NO_SUCH_SDS);

/*  How big is the new one? (NB: delimited by SDS_ENDLIST, not SDS_RETLIST */
  new_size = tlist_len(ty_list) + (long)2;
/*  (A size/namelist pair will be added:, and the size/alignment type
  pair:hence the '+2' */

/*      And the old one?                                                */
  if ((oldlist = tlist[sds_index]) != TNULL)
    old_size = tlist_len(oldlist) - (long)1;
/*  Here I subtract 1 to throw away the SDS_ENDLIST which marks the
  end of the type_list buffer
 */

/*  This is the first tlist entry: giving number of elements in the
  object and, if provided, a pointer to and length of its namelist
 */
  tlist_head.elemcod = SDS_LENLIST;
  if (names != (char *)0) 
  {
    tlist_head.nelems = sds_add_to_heap(sds_index,names,',');
  }
  else
    tlist_head.nelems = (unsigned long)0;

  if (old_size)/* There is an existing list, so get more space....  */
  {
     tlist[sds_index] = new_list =
      (struct type_list *)realloc((char *)tlist[sds_index],
        (unsigned)(new_size+old_size) * 
        (unsigned int)sizeof(struct type_list));
    if (tlist[sds_index] == (char)0 )
    {
      sds_perror("realloc failure, sds_define_object");
      exit(1);
    }
  }
  else 
  { /*  No old list, simple calloc needed...        */

    tlist[sds_index] = new_list =
      (struct type_list *)calloc((unsigned)new_size,
          (unsigned int)sizeof(struct type_list));
    if (tlist[sds_index] == (char)0 )
    {
      sds_perror("calloc failure, sds_define_object");
      exit(1);
    }
    allofl[sds_index] |= SDS_TLIST_ALLOC;
  }

/*  ...and put the new list on the end          */
  new_list += old_size;
  new_list->nelems = tlist_head.nelems;
  new_list->elemcod = tlist_head.elemcod;
  new_list++;

/*  the next slot is for the size/align pair, calculated soon  */
  size_align = new_list++;

  for (i=0;i<new_size - 2;i++) 
  {
    new_list->nelems = ttemp->nelems;
    new_list->elemcod = ttemp->elemcod; 
    if ((new_list->elemcod & SDS_LOCALLIST)) 
    {
      new_list->elemcod &= ~SDS_LOCALLIST;
      new_list->elemcod |= SDS_INDLIST;
      new_list->elemcod += old_size;
    }
    new_list++;
    ttemp++;
  }
  old_size |= SDS_INDLIST;
  size_align->elemcod = SDS_SIZE_ALIGN;
  size_align->nelems = sds_tlsize(sds_index,old_size,&align);
  size_align->elemcod |= (long)(align & 0xff);

/*  Return the pointer with its indirection flag      */
  return(old_size);
}  
/*********************************************************************/
sds_handle 
sds_bad_object(sds_index,object_index)
sds_handle sds_index;
sds_code object_index;
/*********************************************************************/
{
  struct direc *ddptr;
  if ((ddptr = sds_direc_ptr(sds_index)) == DNULL)
    return(sds_error = SDS_NO_SUCH_SDS);

  if (ddptr[0].nelems < object_index + 1)
    return(sds_error = SDS_NO_SUCH_OBJ);

  return(0);
}
/*********************************************************************/
sds_handle
sds_resize_object(sds_index,object_index,new_size)
sds_handle  sds_index;
sds_code object_index;
long new_size;
/*********************************************************************/
{

  struct direc   *dptr = sds_direc_ptr(sds_index);

  if ( dptr[0].offst != SDS_NOT_ASSEMBLED)
  {
      return sds_error = SDS_NOT_ASSEMBL;
  }
  if (dptr[object_index].nelems != new_size)
  {
      dptr[object_index].nelems = new_size;
      dptr[object_index].illoca |= SDS_REALLOC;
  }
  else 
  {
      return sds_error = SDS_CANNOT_RESIZE;
  }

  return 0;

}
/*********************************************************************/
void
sds_destroy(sds_index)
sds_handle sds_index;
/*********************************************************************/
{
  sds_handle     object;
  struct direc   *dptr = sds_direc_ptr(sds_index);

  if (dptr == DNULL)
    return;

  if (load_source[sds_index] == SDS_SHARED_MEM)
  {
#ifdef SHMEM
      sds_discard(sds_index);
/*      shm_destroy(sds_oname(sds_index,0,0)); */
#endif
  } 
  else 
  if (load_source[sds_index] == SDS_FILE)
  {
    free( (char *)get_header(sds_index) );
  } 
  else 
  {
      for (object = 1;object < dptr[0].nelems;object++)
        if (dptr[object].illoca & SDS_WAS_ALLOCATED)
          free(sds_obind2ptr(sds_index,object));
  }
  sds_discard(sds_index);
}
/*********************************************************************/
void
sds_discard(sds_index)
sds_handle  sds_index;
/*********************************************************************/
{ 
  struct direc *dptr = sds_direc_ptr(sds_index);
  int     fd, object_count,object;
  char  ***elstart;
  int    **varcount;
  int disob;

  if (dptr == DNULL)
    return;
  object_count = dptr[0].nelems;
/* Clean up the three malloc regions and the serial stream and 
   flag everything NULL
*/
  if (load_source[sds_index] == SDS_SHARED_MEM)
  {
#ifdef SHMEM
    shm_quit(dptr);
#endif
  }
#ifdef MEMMAP
  else
  if (load_source[sds_index] == SDS_MAPPED_MEM)
  {
    disob = 0;
    /* find out if an object has been searately mapped:
       if not, the whole file must have been */
    for (object=1; object<object_count;object++)
      if (dptr[object].illoca == SDS_DISJOINT_OBJECT)
        disob = 1;
    if (disob == 0 || dptr[0].illoca == SDS_DISJOINT_OBJECT)
      munmap((char *)get_header(sds_index), sds_dataset_size(sds_index));
    else
    {
      for (object=1; object<object_count;object++)
        if (dptr[object].illoca == SDS_DISJOINT_OBJECT)
          munmap((char *)sds_obind2ptr(sds_index,object),
                      dptr[object].nelems * dptr[object].elemsz);
    }
  } 
#endif
  else 
  {
    if ((dptr  != DNULL) && (allofl[sds_index] & SDS_DPTR_ALLOC)) 
    {
      allofl[sds_index] &= ~SDS_DPTR_ALLOC;
      free((char *)dptr);
    }
    if ((fd = serial_stream[sds_index]) != SDS_FILE_OP)
      close(fd);
    if ((tlist[sds_index] != TNULL) && (allofl[sds_index] & SDS_TLIST_ALLOC)) 
    {
      allofl[sds_index] &= ~SDS_TLIST_ALLOC;
      free((char *)tlist[sds_index]);
    }
    if ((heap[sds_index] != NULL) && (allofl[sds_index] & SDS_HEAP_ALLOC)) 
    {
      allofl[sds_index] &= ~SDS_HEAP_ALLOC;
      free(heap[sds_index]);
    }
    if ((shead[sds_index] != NULL) && (allofl[sds_index] & SDS_HEAD_ALLOC)) 
    {
      allofl[sds_index] &= ~SDS_HEAD_ALLOC;
      free(shead[sds_index]);
    }
    if ((dup_sizes[sds_index] != INULL) && (allofl[sds_index] & SDS_DUP_ALLOC)) 
    {
      allofl[sds_index] &= ~SDS_DUP_ALLOC;
      free(dup_sizes[sds_index]);
    }
  }
  elstart = element_start[sds_index];
  varcount =  varel_count[sds_index];
  if (elstart != NULL)
  {
    int i;
    for (i=1;i<object_count;i++)
    {
			if (dptr[i].illoca != SDS_DISJOINT_OBJECT)
			{
        if (elstart[i] != NULL)
          free(elstart[i]);
        if (varcount[i] != NULL)
          free((char *)varcount[i]);
			}
    }
    free(elstart);
    element_start[sds_index] = NULL;
    free((char *)varcount);
    varel_count[sds_index] = NULL;
  }
  shead[sds_index] = HNULL;
  dup_sizes[sds_index] = INULL;
  dptr_index[sds_index] = DNULL;
  serial_stream[sds_index] = SDS_FILE_OP;
  tlist[sds_index] = TNULL;
  heap[sds_index] = NULL;
}
/*********************************************************************/
long
sds_declare_object(sds_index,obj_ptr,name,number,code)
sds_handle sds_index;
void  *obj_ptr;
char  *name;
sds_code number;
sds_code code;
/*********************************************************************/
{
  struct direc *ddptr;
  sds_code      obj_index;
  sds_handle    tsz;
  unsigned char atype;
  
  if ((ddptr = sds_direc_ptr(sds_index)) == DNULL)
    return(sds_error = SDS_NO_SUCH_SDS);

/*  If I can't find this thing's size, it is a. not a known
  primitive and b. not a described complex object. So bug off.
*/
  tsz = sds_object_size(sds_index,code,&atype);
  /*
  if (code & SDS_INDLIST)
      tsz = sds_tlsize(sds_index,code,&atype);
    else
      tsz = sds_object_size(sds_index,code,&atype);
  */

   if (tsz == 0)
     return(sds_error = SDS_ZERO_LENGTH);

/*  Found it. Add one to directory list....      */
  obj_index = (int)ddptr[0].nelems++;

/*  May have to get more memory for the directory structure  */
  if (obj_index == current_direc_size[sds_index]) 
    ddptr = sds_inc_direc_size(sds_index);

/*  Fill in the definition of this object      */
  ddptr[obj_index].offst = (long)obj_ptr;
  ddptr[obj_index].elemcod = code;
  ddptr[obj_index].elemsz = tsz;
  ddptr[obj_index].nelems = number;
  ddptr[obj_index].illoca = 0;
  ddptr[obj_index].align_type = atype;
  ddptr[obj_index].structype = (short)0;

  if ((sds_error = sds_pname(sds_index,(long)obj_index,name))
    < 0) return(sds_error);

/*  And return its object number        */
  return(obj_index);
}
/*********************************************************************/
struct direc *
sds_direc_ptr(sds_index)
sds_handle  sds_index;
/*********************************************************************/
{
  if ((sds_index < 0) || (sds_index > MAX_SDS)) return(DNULL);
  return(dptr_index[sds_index]);
}

/*********************************************************************/
char **
sds_saverestore(sds_index,dptr, flag)
sds_handle sds_index;
struct direc *dptr;
/*********************************************************************/
{
  static char  *sds_state;
  static        sds_handle status;
  static char **object_pointer;
  int           i;
  int           junk = 0;

/*  I'm going to put the real object addresses in this pointer list        */
  if (flag == SDS_SAVE_STATE)
  {
    object_pointer = (char **)calloc((unsigned int)dptr[0].nelems,
            (unsigned int)sizeof(char *));
    if (object_pointer == (char)0 )
    {
      sds_perror("calloc failure, sds_make");
      exit(1);
    }
  
  /*  Get the addresses from the approved routine ...  */
  /*  Save the directory state in case it's NOT_ASSEMBLED */
    sds_state = (char *)dptr[0].offst;
    for (i=0;i < dptr[0].nelems;i++) 
      object_pointer[i] = sds_getp(dptr,i);
  
  
    if (sds_state == (char *)SDS_NOT_ASSEMBLED)
      status = offil(sds_index,(char **)0, 0, &junk);

    return object_pointer;
  }
  else
  {
/* leave things as before in case dataset expansion is required */
  
    if (sds_state == (char *)SDS_NOT_ASSEMBLED) 
    {
      dptr[0].offst = (unsigned long)sds_state;
      for (i=1;i < dptr[0].nelems;i++)
        dptr[i].offst = (unsigned long)object_pointer[i];
    }
    free((char *)object_pointer);
    return NULL;
  }
}

/*********************************************************************/
sds_handle
sds_assemble(sds_index,name,flg)
sds_handle sds_index;
sds_code flg;
char  *name;
/*  'assemble' means assemble the bits of an SDS into a coherent
  thing. That means the target place for assembly must be specified.
  In the simplest case - assemble in process memory - the only 
   action will be to allocate memory for any objects which don't
  yet have any - no new SDS is created. For assembly to *shared memory*
  a contiguous memory chunk has to be filled, so a new SDS is made.
  Assembling to file or streaming to a network do generate a new
  SDS but it is NOT accessible to the generating process without
  more work (eg sds_load) so no new SDS index is returned
*/
/*********************************************************************/
{
  struct direc *dptr;
  struct direc *new_dptr;
  int           new_offset,new_base;
  char         *cptr;
  int           cp_size,fd,sds_return = sds_index;
  unsigned long i;
  char        **object_pointer;
  
  sds_error = 0;
  if ((dptr = sds_direc_ptr(sds_index)) == DNULL)
    return(sds_error = SDS_NO_SUCH_SDS);

  if ((flg & SDS_FILE))  /* no new sds created */
  {
    object_pointer = sds_saverestore(sds_index,dptr,SDS_SAVE_STATE);
    if ((fd = sds_open_file(name,(int)flg)) >= 0) 
    {
      if (to_file(sds_index,fd,object_pointer) != 0) 
      {
        sds_error = SDS_FILE_WR;
        sds_return= sds_error;
      }
      close(fd);
    }
    else 
      sds_return = SDS_FILE_OP;
    sds_saverestore(sds_index,dptr,SDS_RESTORE_STATE);
  } 
  else if (flg & SDS_SYBASE)  /* no new sds created */
  {
#ifndef  SDSDB
    sds_error = SDS_TRANSFER_UNDEF;
    return(sds_error);
#else
    object_pointer = sds_saverestore(sds_index,dptr,SDS_SAVE_STATE);
    sds_return = sds_db_make(sds_index,name,name,flg,object_pointer);
    sds_saverestore(sds_index,dptr,SDS_RESTORE_STATE);
#endif
  }
#ifdef SHMEM
  else if (flg & SDS_SHARED_MEM)  /* a new sds is created */
  {
    if ((sds_return = next_sds()) >= 0)
    {
      object_pointer = sds_saverestore(sds_index,dptr,SDS_SAVE_STATE);

/*  Allocate the memory          */
      if ((cptr = (char *)shm_make(name,
                   sds_dataset_size(sds_index),0666)) == (char *)-1) 
      {
        sds_perror("shared memory make:");
        exit(1);
      }

/*  Make and load the sds header,tlist and heap    */
      cptr += sds_mem_header(sds_index,cptr);
      cptr += align_delta((int)cptr,dptr[0].align_type);

/*  How big is the directory?        */
      cp_size = (int)dptr[0].nelems*dptr[0].elemsz;

/*  Copy the directory over          */
      memcpy(cptr,(char *)dptr,cp_size);

      new_dptr = (struct direc *)cptr;
      new_base = (int)cptr;

      set_sys_vars(sds_return,(struct direc *)cptr);

      cptr += cp_size;

/*  And all the objects          */

      for (i=1;i < dptr[0].nelems;i++) 
      {
        new_dptr[i].illoca = SDS_EXTERNAL_OBJECT;
        cptr += align_delta(cptr,dptr[i].align_type);
        new_offset = new_dptr[0].offst + (int)cptr - new_base;
        if (new_offset != new_dptr[i].offst ) 
          new_dptr[i].offst = new_offset;
        cp_size = (int)dptr[i].nelems;

        if (dptr[i].illoca & SDS_REALLOC)
          if (dup_sizes[sds_index][i] < dptr[i].nelems)
            cp_size = dup_sizes[sds_index][i];

        cp_size *= dptr[i].elemsz;
        if (new_dptr[i].structype & SDS_RECORDS)
        {
          cptr += sds_copy_records(sds_return,cptr,
                     (sds_record_handle *)object_pointer[i]);
          new_dptr[i].structype &= ~SDS_RECORDS;
        }
        else 
        {
          if (object_pointer[i] != SDS_ALLOCATE)
            memcpy(cptr, object_pointer[i], cp_size);
          cptr += (int)dptr[i].nelems*dptr[i].elemsz;
        }
      }
      sds_saverestore(sds_index,dptr,SDS_RESTORE_STATE);
    }
  }
#endif
  else if (flg & SDS_PROC_MEM)  /* Just do necessary memory allocation */
  {
    for (i=1;i < dptr[0].nelems;i++) 
    {
      if (dptr[i].offst == (sds_code)SDS_ALLOCATE) 
      {
        dptr[i].offst = (sds_code)malloc(
          (unsigned)dptr[i].nelems*
          (unsigned)dptr[i].elemsz);

        if (dptr[i].offst == (sds_code)0 )
        {
          sds_perror("malloc failure, sds_assemble");
          exit(1);
        }
        dptr[i].illoca |= SDS_WAS_ALLOCATED;
      }
      else 
      {
        dptr[i].illoca |= SDS_EXTERNAL_OBJECT;
      }
    }
  }
  else /* this is an unknown target */ 
  {
    sds_return = 
    sds_error = SDS_TRANSFER_UNDEF;
  }
  return sds_return;
}
/*********************************************************************/
sds_handle
to_file(sds,fd,object_pointer)
sds_handle sds;
int fd;
char *object_pointer[];
/*********************************************************************/
{
  struct direc *dptr;
  unsigned long  i;
  sds_handle total_write = 0;
  sds_handle pads, ob_write;

  if ((dptr = sds_direc_ptr(sds)) == DNULL) 
    return(sds_error = SDS_NO_SUCH_SDS);

  if ((total_write = sds_write_header(fd,sds)) < 0)
    return(sds_error = SDS_FILE_WR);

  for (i=0;i < dptr[0].nelems;i++)
  {
     if (dptr[i].illoca != SDS_DISJOINT_OBJECT)
     {
       if ((pads = sds_prepads(&dptr[i], total_write)) > 0)
       {
         if (sds_write_data(fd, sds_padbytes, pads) != pads)
           return(sds_error = SDS_FILE_WR);
         else 
           total_write += pads;
       }
       if ((ob_write = sds_write_object(fd,sds,i,object_pointer[i])) < 0)
         return(sds_error = SDS_FILE_WR);
       else 
         total_write += ob_write;
    }
  }
  return(0);
}
/*********************************************************************/
sds_handle
sds_prepads(dptr, total_write)
struct direc *dptr;
sds_handle total_write;
/*********************************************************************/
{
  return (sds_handle)align_delta((int)total_write,dptr->align_type); 
}
/*********************************************************************/
sds_handle
sds_mem_header(sds,cptr)
sds_handle  sds;
char  *cptr;
/*********************************************************************/
{
  struct sds_header *sdsh = (struct sds_header *)cptr;
  sdsh->magic_number = (long)SDS_MAGIC;
  sdsh->version = (long)SDS_VERSION;
  sdsh->list_size = (short)tlist_size(tlist[sds]);
  sdsh->heap_size = (short)get_heap_size(sds);
  sdsh->heap_size += (short)align_delta((int)sdsh->heap_size,4);
  cptr += sizeof(struct sds_header);

  if (sdsh->list_size) /* there is a tlist  */
  {
    memcpy(cptr,(char *)tlist[sds],(int)sdsh->list_size);
    cptr += sdsh->list_size;
  }
  memcpy(cptr,heap[sds],(int)sdsh->heap_size);

  return((int)sdsh->list_size +(int)sdsh->heap_size + sizeof(struct sds_header));
}
/*********************************************************************/
sds_handle
sds_write_header(fd,sds)
sds_handle sds;
int  fd;
/*********************************************************************/
{
  struct sds_header sdsh;
  sds_handle ret_size;
  sdsh.magic_number = (long)SDS_MAGIC;
  sdsh.version = (long)SDS_VERSION;
  sdsh.list_size = (short)tlist_size(tlist[sds]);
  sdsh.heap_size = (short)get_heap_size(sds);
  ret_size =  sdsh.list_size + sdsh.heap_size + sizeof(struct sds_header);

  if (sds_write_data(fd,(char *)&sdsh,sizeof(struct sds_header))
     != sizeof(struct sds_header)) 
    return(SDS_FILE_WR);
  if (sdsh.list_size)
    if (sds_write_data(fd,(char *)tlist[sds],(int)sdsh.list_size)
     != (int)sdsh.list_size)
      return(SDS_FILE_WR);
  if (sds_write_data(fd,heap[sds],(int)sdsh.heap_size)
     != (int)sdsh.heap_size)
    return(SDS_FILE_WR);

  return ret_size;
}
/*********************************************************************/
sds_handle
sds_write_object(fd,sds,i,pointer)
/*********************************************************************/
int  fd;
sds_handle sds;
sds_code i;
void  *pointer;
{
  unsigned long  pad_size = 0;
  unsigned long  realloc_size = 0;
  unsigned long  total_size = 0;
  unsigned long  temp_size = 0;
  struct direc   *dptr = sds_direc_ptr(sds);
  unsigned char    pattern = 0xff;

  total_size = dptr[i].nelems*dptr[i].elemsz;
  realloc_size = total_size;

  if (i == 0) 
  {
/* Directory: do not copy over RECORD flags, and mark disjoint objects as
   unknown address */
    int obcount;
    pointer = malloc(total_size);
    memcpy(pointer, (char *)dptr, total_size);
    dptr = (struct direc *)pointer;
    for (obcount=0;obcount<dptr[0].nelems;obcount++)
    {
      if (dptr[obcount].structype == SDS_RECORDS)
        dptr[obcount].structype = SDS_NORMAL_OBJECT;
      if (dptr[obcount].illoca == SDS_DISJOINT_OBJECT)
        dptr[obcount].offst = SDS_IMPOSSIBLE_ADDRESS;
    }
  }
  else if (dptr[i].structype == SDS_RECORDS)
  {
    return sds_write_records(sds, fd, (sds_record_handle *)pointer);
  }

  if ((i != 0) && (dptr[i].illoca & SDS_REALLOC))
  {
    temp_size = dup_sizes[sds][i] * dptr[i].elemsz;
    if (temp_size < total_size)
    {
      realloc_size = temp_size;
      pad_size = total_size - realloc_size;
      pattern = 0x00;
    }
  }

  if (pointer == SDS_ALLOCATE) 
  {
      pad_size = total_size;
      realloc_size = 0;
  }

/*  If this thing is unallocated, I'll write ff bytes to
  indicate that there is undefined junk in it.
*/
  if (realloc_size != 0)
  {
    if (sds_write_data(fd,pointer,(int)realloc_size) != (int)realloc_size) 
    {
      perror("write object");
      return(sds_error = SDS_FILE_WR);
    }
  }
  if (pad_size != 0)
  {
    if (sds_write_pattern(fd,pad_size,pattern) < 0) 
    {
      perror("write null pattern to  object");
      return(sds_error = SDS_FILE_WR);
    }
  }
  if (i == 0)
    free(pointer);
  return total_size;
}

/*********************************************************************/
struct  direc *
sds_load_direc(sds)
/*********************************************************************/
sds_handle  sds;
{

  struct direc dtemp,*dptr;
  char  *list_ptr,*cptr;
  struct sds_header header;
  int  ierr,ndirecs,lhsize;
  int  fd = serial_stream[sds];
  int  o;

  if (fd < 0) 
  {
    sds_error = SDS_FILE_NOP;
    return(DNULL);
  }

/*  Read in the header ;     */
  if  ((ierr = sds_read_header(sds,&header)) < 0) 
  {
    close(fd);
    serial_stream[sds] = SDS_FILE_OP;
    sds_error = ierr;
    return(DNULL);
  }

  lhsize = (int)header.heap_size + (int)header.list_size;

  list_ptr = malloc((unsigned)lhsize);

  if (list_ptr == (char)0 )
  {
    sds_perror("malloc failure, sds_load_direc");
    exit(1);
  }
  sds_vread(fd,list_ptr,lhsize);

/*  Find size of directory        */
  sds_vread(fd,(char *)&dtemp,sizeof(struct direc));
  ndirecs = (int)dtemp.nelems;

/*  throw away previous directory      */
  if (dptr_index[sds] != DNULL) 
  {
    free((char *)dptr_index[sds]);
    dptr_index[sds] = DNULL;
  }

/*  New directory,tlist heap  and header go here    */
  cptr = malloc((unsigned int)(ndirecs*sizeof(struct direc)) +
      (unsigned int)BASE_OFFSET +  
      (unsigned int)header.heap_size +
      (unsigned int)header.list_size);

  if (cptr == (char)0 )
  {
    sds_perror("malloc failure, sds_load_direc 2");
    exit(1);
  }
  allofl[sds] = SDS_HEAD_ALLOC;


/*  Copy in header.....        */
  memcpy(cptr,(char *)&header,BASE_OFFSET);

  cptr += BASE_OFFSET;

/*  Copy in tlist and heap        */
  memcpy(cptr,list_ptr,lhsize);

  cptr += (int)(header.heap_size + header.list_size);

  dptr = (struct direc *)cptr;

/*  Copy in top directory.....      */
  memcpy(cptr,(char *)&dtemp,sizeof(struct direc));

  set_sys_vars(sds,(struct direc *)cptr);

  cptr += sizeof(struct direc);

/*  Read in rest of directory.....      */
  sds_vread(fd,cptr,sizeof(struct direc)*(ndirecs-1));

/*  reclaim some space.        */
  free(list_ptr);

  for (o = 1; o< dptr[0].nelems; o++)
    dptr[o].offst = (unsigned long)SDS_IMPOSSIBLE_ADDRESS;

  return(dptr);
}

/*********************************************************************/
sds_handle
sds_read_header(sds,header)
/*********************************************************************/
sds_handle  sds;
struct sds_header *header;
{
  int  fd = serial_stream[sds];

  if (fd < 0)
    return(sds_error = SDS_FILE_NOP);

  if (sds_read_data(fd,(char *)header,sizeof(struct sds_header))
         != sizeof(struct sds_header))
    return(sds_error = SDS_FILE_RD);

  if (header->magic_number != (long)SDS_MAGIC) 
    return(sds_error = SDS_NOT_SDS);

  if ((header->version & (long)0xffff) != ((long)SDS_VERSION & (long)0xffff)) 
    return(sds_error = SDS_BAD_VERSION);

  return(0);
}
/*********************************************************************/
sds_handle
sds_read_object(sds,obj,pointer,start,max)
sds_handle  sds;
sds_code obj;
void  *pointer;
long  start;
unsigned long  max;
/*********************************************************************/
{
  unsigned long  i,copy_size,offset;
  unsigned long  nelem,total_size;
  struct  sds_header  *head;
  struct direc *dptr = sds_direc_ptr(sds);
  int  fd = serial_stream[sds];
  int  rbytes;


  if (fd < 0)
    return(sds_error = SDS_FILE_NOP);

  if (dptr == DNULL)
    return(sds_error = SDS_NO_SUCH_SDS);
  
  if (obj >= dptr[0].nelems)
    return(sds_error = SDS_NO_SUCH_OBJ);
  if (start >= dptr[obj].nelems)
    return(sds_error = SDS_NO_SUCH_OBJ);

  head = get_header(sds);
  offset = BASE_OFFSET + head->list_size + head->heap_size;
  for (i=0;i<obj;i++) 
  {
    offset += dptr[i].nelems*dptr[i].elemsz;
    offset += align_delta((int)offset,dptr[i].align_type);
  }
  offset += start * dptr[obj].elemsz;

  total_size = dptr[obj].nelems*dptr[obj].elemsz;  

/* Don't try to copy more than max objects (else potential bang)   */
  if (max > (dptr[obj].nelems - start)) max = dptr[obj].nelems - start;
  dptr[obj].offst = (long)pointer;

/*  In bytes.....            */
  copy_size = max*dptr[obj].elemsz;

  if (lseek(fd,offset,L_SET) < 0) 
  {
    printf("lseek failure");
    perror("ach!");
    exit(1);
  }
  rbytes = sds_vread(fd,pointer,copy_size);
  if (rbytes != copy_size)
    nelem = SDS_FILE_RD;
  else
    nelem = rbytes/dptr[obj].elemsz;
  dptr[obj].offst = (long)pointer;

  return(nelem);
}
/*********************************************************************/
void
sds_close_stream(sds)
int  sds;
/*********************************************************************/
{
  int  fd;
  if ((fd = serial_stream[sds]) != SDS_FILE_OP)
    close(fd);
}
/*********************************************************************/
sds_handle
sds_open(filename)
/*********************************************************************/
char  *filename;
{

  int  fd;
  sds_handle sds;

  if ((fd = open(filename,O_RDONLY,0666)) < 0)
  {
    return(SDS_FILE_OP);
  }
  if ((sds = next_sds()) == SDS_NO_SUCH_SDS) return(SDS_NO_SPC);

/*  If there's already an input stream open for this SDS, close it  */
  if (serial_stream[sds]  !=  SDS_FILE_OP)
    close(serial_stream[sds]);

/*  Remember that you have this stream open        */
  serial_stream[sds] = fd;

  return(sds);
}
/*********************************************************************/
sds_handle
sds_open_file(name,flg)
int  flg;
char  *name;
/*  Here be System Dragons eg VMS's plethora of file type flags   */
/*********************************************************************/
{
  int  fflag = O_WRONLY,fd;
  fflag |= (flg & (int)SDS_APPEND)?O_APPEND:O_CREAT | O_TRUNC;

#ifdef vms
  if (!(fflag & O_APPEND)) 
  {
    if ((fd = creat(name,0666,"ctx=nocvt","ctx = bin","mrs = 2048","rfm=udf")) < 0) {
      return(sds_error = SDS_FILE_OP);
    }
  }
  else
#endif

  if ((fd = open(name,fflag,0666)) < 0) 
  {
    return(sds_error = SDS_FILE_OP);
  }

#ifdef vxworks
  lseek(fd,0,SEEK_END);  /* Go to end of file. */
#endif

  return(fd);
}

/*********************************************************************/
sds_handle
sds_write_pattern(fd,size,pattern)
int  fd;
unsigned long size;
char  pattern;
/*  Write crud to unallocated object going to eg file, although
  I'm not really sure why anyone would do it. Still, it's possible
  so one must do something rational-ish
*/
/*********************************************************************/
{
#define SDS_JUNK_BUFFER  256
  char  buffer[SDS_JUNK_BUFFER];
  int  i,mod = (int)size/SDS_JUNK_BUFFER;
  int  frac = (int)size%SDS_JUNK_BUFFER;
  memset(buffer,pattern,SDS_JUNK_BUFFER);
  if (mod > 0)
    for (i=0;i<mod;i++)
    {
      if (sds_write_data(fd,buffer,SDS_JUNK_BUFFER) < 0) 
      {
        return(sds_error = SDS_FILE_WR);
      }
    }
  if (sds_write_data(fd,buffer,frac) <0) 
  {
    return(sds_error = SDS_FILE_WR);
  }
  return(0);
}
/*********************************************************************/
sds_handle
sds_which(dptr)
struct direc *dptr;
/*   Given a directory pointer , which sds does it refer to ?     */
/*********************************************************************/
{
  if (dptr != DNULL) 
  {
    int i;
    for (i=0;i<MAX_SDS;i++)
      if (dptr == dptr_index[i])
        return(i);
  }
  return(sds_error = SDS_NO_SUCH_SDS);
}

/*********************************************************************/
sds_handle
get_heap_size(sds)
sds_handle  sds;
/*********************************************************************/
{
  struct  direc    *dptr = sds_direc_ptr(sds);
  struct   sds_header  *hptr = shead[sds];
  int  hsize,del;

  if (dptr == DNULL) 
  {
    sds_error = SDS_NO_SUCH_SDS;
    return(-1);
  }
  if (hptr == HNULL) 
  {
    hsize = 
      (current_heap[sds] - heap[sds]);
    del = align_delta(hsize,4);

    return(hsize + del);
  }
  else 
  {
    hsize = (int)(hptr->heap_size);
    return(hsize);
  }
}
/*********************************************************************/
struct  type_list *
get_tlist(sds)
sds_handle  sds;
/*********************************************************************/
{ return(tlist[sds]); }
/*********************************************************************/
struct  sds_header *
get_header(sds)
sds_handle  sds;
/*********************************************************************/
{ return(shead[sds]); }
/*********************************************************************/
char  *
get_heap(sds)
sds_handle  sds;
/*********************************************************************/
{ return(heap[sds]); }
/*********************************************************************/
float
sds_version(sds_index)
sds_handle sds_index;
/*********************************************************************/
{
  struct  sds_header   *head = get_header(sds_index);
  float version;
  sds_code ver;

  if (head == HNULL)
    ver = SDS_VERSION;
  else
    ver = head->version;

  version = (float)(head->version & 0xffff);
  version += (float)(head->version >> 16)/100.;

  return version;
}
/* Additions to support SDS to file descriptors. */

/*********************************************************************/
void
sds_elbow_socket(sds)
/* To 'elbow' or 'give some(thing)(one) the elbow' is an expression
   originating as far as I know in the South of England, perhaps among
   film persons, which means to throw away, discard. I have a fondness
   for it, which acounts for its appearance here. A special American
   language binding is available on request.
*/
/*********************************************************************/
sds_handle  sds;
{
  struct direc *dptr = sds_direc_ptr(sds);

/* Clean up the three malloc regions and the serial stream and 
   flag everything NULL
*/
  dptr = sds_direc_ptr(sds);
  if (load_source[sds] == SDS_SHARED_MEM)
  {
#ifdef SHMEM
    shm_quit(dptr);
#endif
         }
  else 
  {
    if ((dptr  != DNULL) && (allofl[sds] & SDS_DPTR_ALLOC)) 
    {
      allofl[sds] &= ~SDS_DPTR_ALLOC;
      free((char *)dptr);
    }
    if ((tlist[sds] != TNULL) && (allofl[sds] & SDS_TLIST_ALLOC)) 
    {
      allofl[sds] &= ~SDS_TLIST_ALLOC;
      free((char *)tlist[sds]);
    }
    if ((heap[sds] != NULL) && (allofl[sds] & SDS_HEAP_ALLOC)) 
    {
      allofl[sds] &= ~SDS_HEAP_ALLOC;
      free(heap[sds]);
    }
    if ((shead[sds] != NULL) && (allofl[sds] & SDS_HEAD_ALLOC)) 
    {
      allofl[sds] &= ~SDS_HEAD_ALLOC;
      free(shead[sds]);
    }
    if ((dup_sizes[sds] != INULL) && (allofl[sds] & SDS_DUP_ALLOC)) 
    {
      allofl[sds] &= ~SDS_DUP_ALLOC;
      free(dup_sizes[sds]);
    }
  }
  shead[sds] = HNULL;
  dup_sizes[sds] = INULL;
  dptr_index[sds] = DNULL;
  serial_stream[sds] = SDS_FILE_OP;
  tlist[sds] = TNULL;
  heap[sds] = NULL;
      }


/***********************************************************************/
sds_handle
write_sds(fd,sds_index)
int fd;
sds_handle sds_index;
/***********************************************************************/
{
  struct direc *dptr;
  int           sds_return = sds_index;
  char        **object_pointer;
  
  sds_error = 0;
  if ((dptr = sds_direc_ptr(sds_index)) == DNULL)
    return(sds_error = SDS_NO_SUCH_SDS);

  object_pointer = sds_saverestore(sds_index,dptr,SDS_SAVE_STATE);
  if (to_file(sds_index,fd,object_pointer) != 0) 
  {
    sds_error = SDS_FILE_WR;
    sds_return= sds_error;
  }
  sds_saverestore(sds_index,dptr,SDS_RESTORE_STATE);

  return(sds_return);
}

/***********************************************************************/
int write_sds2socket(fd,sds)
int fd;
sds_handle sds;
/***********************************************************************/
{
  return(write_sds(fd,sds));
}

/***********************************************************************/
int read_socket2sds(fd,mode)
int fd;
int mode;
/***********************************************************************/
{
  struct direc *dptr;

  extern char sds_rbytes[NARCS];

  struct direc      *ndptr;
  struct type_list  *ntlist;
  struct sds_header *nheader,*header;
  sds_handle         sds,orig_state;
  sds_handle         new_sds,old_sds;
  int                size,dsize,first,second;
  int                i,old,new;
  struct sds_odesc  *othing,*nthing;
  long               new_type;
  int                del;
  char               junk[32];
  int                element_addr;
  int                nread;
  struct sds_header  head;

  if (sds_is_init == 0) 
    return(sds_error = SDS_NOT_INITIALISED);
  
  if ((sds = next_sds()) < 0) 
    return(sds_error = sds);
  
  old_sds = sds_cload_direc(fd,&orig_state, &head);
  if (old_sds < 0) 
  {
    if (old_sds == SDS_GOOD_FORMAT)
    {
       sds = sds_na_load(fd, &head);
       return (sds);
    }
    printf("\nread_socket2sds: Server gone?");
    return( sds_error = old_sds );
  }
  header = get_header(old_sds);
  dptr = sds_direc_ptr(old_sds);
  
  
  /*   Duplicate this header for the new, native arch, dataset */
  
  size = (char *)dptr - (char *)header;
  
  /*  
   * the heap size may need to be increased (don't worry if it's too big)   
   */
  
  first = BASE_OFFSET + header->heap_size + header->list_size;
  dsize = align_delta(header->heap_size,RBYTE); 
  dsize = 0;
  size += dsize;
  second = size - first; 
  
  nheader = (struct sds_header *)malloc((unsigned)size +
          dptr[0].nelems * sizeof(struct direc));
  memcpy((char *)nheader,(char *)header,first);
  memcpy((char *)((int)nheader + first + dsize),
   (char *)((int)header + first),
   second);
  
  nheader->heap_size += dsize;
  nheader->magic_number = SDS_MAGIC;
  
  ndptr = (struct direc *)((char *)nheader +
         BASE_OFFSET +
         nheader->heap_size +
         nheader->list_size);
  memcpy((char *)ndptr,(char *)dptr,dptr[0].nelems * sizeof(struct direc));
  ndptr[0].offst = (long)((char *)ndptr - (char *)nheader);
  
  new_sds = next_sds();
  
  allofl[new_sds] = SDS_HEAD_ALLOC;
  set_sys_vars(new_sds,ndptr);
  
  fix_sizes_and_aligns(new_sds);
  
  ndptr[0].offst = SDS_NOT_ASSEMBLED;
  for (i=1;i<dptr[0].nelems;i++) 
  {
    ndptr[i].offst = (long)SDS_ALLOCATE;
    ndptr[i].illoca = (char)0;
  }
  
  /*  The complete descriptions - header, tlist and
  directories - are now an accurate description
  of both input and output sds's. It remains to convert the 
  objects, which are pointed to by the object_pointer
  list and described by the direc's
  */
  
  sds = sds_assemble(new_sds,"temp",SDS_PROC_MEM);
  if (sds != new_sds) 
  {
    printf("erk!,sdses not equal\n");
    exit(1);
  }
  ntlist = get_tlist(new_sds);
  ndptr = sds_direc_ptr(new_sds);
  
  del = 
    dptr[1].offst -
    dptr[0].offst -
    dptr[0].nelems * dptr[0].elemsz;
  
  if (del > 0) 
  {
    sds_vread(fd,junk,(unsigned)del);
  }
  element_addr = 0;
  del = 0;
  
  for (i=1;i<ndptr[0].nelems;i++) 
  {
    while((old = sds_resolve(old_sds,i,&othing,SDS_OBJECT)) >= 0) 
    {
      sds_alt_resolution_stack();
      new = sds_resolve(new_sds,i,&nthing,SDS_OBJECT);
      sds_main_resolution_stack();
      if (othing[old].nelems != nthing[new].nelems) 
      {
        printf("new <--> old sds mismatch!\n");
        exit(1);
      }
      if (element_addr != 0)
        del = (int)othing[old].address - element_addr;
      if (del > 0) 
      {
        nread = sds_vread(fd,junk,(unsigned)del);
      }
      new_type = convert(fd,&othing[old],
             &nthing[new],
             orig_state);
      if (ndptr[i].elemcod & SDS_INDLIST)
        ntlist[nthing[new].ind-1].elemcod = new_type;
      else
        ndptr[i].elemcod = new_type;
      element_addr = (int)othing[old].address +
                othing[old].nelems * othing[old].size;
    }
    
    sds_alt_resolution_stack();
    new = sds_resolve(new_sds,i,&nthing,SDS_OBJECT);
    sds_main_resolution_stack();
    
  }
  
  /* We don't need the wrong architecture sds */
  sds_elbow_socket(old_sds);

  return(sds);
}

