// $Header: glishd.cc,v 1.11 93/01/13 14:04:33 vern Exp $

#include <stdio.h>
#include <string.h>
#include <errno.h>

#include "Glish/Dict.h"
#include "Glish/Client.h"

#include "LocalExec.h"
#include "Channel.h"
#include "Reporter.h"
#include "system.h"

#if defined (SABER)
#include <libc.h>
#endif


extern "C" {
char* sys_errlist[];
int chdir( const char* path );
}


inline int streq( const char* a, const char* b )
	{
	return ! strcmp( a, b );
	}


declare(PDict,LocalExec);


char* prog_name;
Client* client;

PDict(LocalExec) clients;
int out_event_fd;	// connection to sequencer

void ping_client( Value* client_id );
void create_client( Value* argv );
void shell_command( Value* cmd );
void kill_client( Value* client_id );
void reap_clients();

int main( int argc, char** argv )
	{
	Client c( argc, argv );
	client = &c;

	prog_name = argv[0];
	++argv, --argc;

	if ( argc >= 2 && streq( argv[0], "-wd" ) )
		{
		if ( chdir( argv[1] ) < 0 )
			error->Report( "couldn't change to given directory: ",
					sys_errlist[errno] );

		argc -= 2, argv += 2;
		}

	for ( ; ; )
		{
		GlishEvent* e = c.NextEvent();

		if ( ! e )
			break;

		else if ( streq( e->name, "ping" ) )
			ping_client( e->value );

		else if ( streq( e->name, "client" ) )
			create_client( e->value );

		else if ( streq( e->name, "shell" ) )
			shell_command( e->value );

		else if ( streq( e->name, "kill" ) )
			kill_client( e->value );

		else
			c.Unrecognized();

		reap_clients();
		}
	}


void ping_client( Value* client_id )
	{
	char* id = client_id->StringVal();
	LocalExec* client = clients[id];

	if ( ! client )
		error->Report( "no such client ", id );
	else
		client->Ping();

	delete id;
	}


void create_client( Value* argv )
	{
	if ( argv->Type() != TYPE_STRING )
		{
		error->Report( "bad event value (", argv,
				") creating remote client" );
		return;
		}

	int argc = argv->Length();

	if ( argc <= 1 )
		{
		error->Report(
			"no arguments given for creating remote client" );
		return;
		}

	// First strip off the id.
	charptr* argv_ptr = argv->StringPtr();
	char* client_id = strdup( argv_ptr[0] );

	--argc;
	++argv_ptr;

	charptr* client_argv = new charptr[argc + 1];

	for ( int i = 0; i < argc; ++i )
		client_argv[i] = argv_ptr[i];

	client_argv[argc] = 0;

	const char* exec_name = which_executable( client_argv[0] );
	if ( ! exec_name )
		error->Report( "no such executable ", client_argv[0] );

	else
		{
		LocalExec* exec = new LocalExec( exec_name, client_argv );

		if ( exec->ExecError() )
			error->Report( "problem exec'ing client ", 
				client_argv[0], ": ", sys_errlist[errno] );

		clients.Insert( client_id, exec );
		}

	delete client_argv;
	}


void shell_command( Value* cmd )
	{
	char* command;
	if ( ! cmd->FieldVal( "command", command ) )
		error->Report( "remote glishd received bad shell command:",
				cmd );

	char* input;
	if ( ! cmd->FieldVal( "input", input ) )
		input = 0;

	FILE* shell = popen_with_input( command, input );

	if ( ! shell )
		{
		Value F( false );
		client->PostEvent( "fail", &F );
		}
	else
		{
		// ### This is an awful lot of events; much simpler would
		// be to slurp up the entire output into an array of strings
		// and send that back.  Value::AssignElements makes this
		// easy since it will grow the array as needed.  The
		// entire result could then be stuffed into a record.

		Value T( true );

		client->PostEvent( "okay", &T );
		char line_buf[8192];

		while ( fgets( line_buf, sizeof( line_buf ), shell ) )
			client->PostEvent( "shell_out", line_buf );

		client->PostEvent( "done", &T );

		// Cfront, in its infinite bugginess, complains about
		// "int assigned to enum bool" if we use
		//
		//	Value status_val( pclose( shell ) );
		//
		// here, so instead we create status_val dynamically.

		Value* status_val = new Value( pclose_with_input( shell ) );

		client->PostEvent( "status", status_val );

		Unref( status_val );
		}

	delete command;
	delete input;
	}


void kill_client( Value* client_id )
	{
	char* id = client_id->StringVal();
	LocalExec* client = clients[id];

	if ( ! client )
		error->Report( "no such client ", id );

	else
		{
		delete client;
		char* client_key = clients.Remove( id );
		delete client_key;
		}

	delete id;
	}


void reap_clients()
	{
	int terminated_pid;

	while ( (terminated_pid = reap_terminated_process()) )
		{ // remove terminated process from list of clients
		IterCookie* c = clients.InitForIteration();

		const char* key;
		LocalExec* exec;
		while ( (exec = clients.NextEntry( key, c )) )
			if ( exec->PID() == terminated_pid )
				{
				char* client_key = clients.Remove( key );
				delete client_key;
				break;
				}

		if ( ! exec )
			error->Report( "reaped unknown process" );
		}
	}
