// $Header: Stmt.h,v 1.6 92/12/01 10:25:22 vern Exp $


#ifndef stmt_h
#define stmt_h

#include "Glish/List.h"

#include "Func.h"
#include "Event.h"


class Stmt;
class EventDesignator;
class Agent;
class Task;
class Sequencer;

declare(PList,Stmt);
typedef PList(Stmt) stmt_list;

declare(PDict,stmt_list);
typedef PDict(stmt_list) stmt_list_dict;


typedef enum {
	FLOW_NEXT,		// continue on to next statement
	FLOW_LOOP,		// go to top of loop
	FLOW_BREAK,		// break out of loop
	FLOW_RETURN		// return from function
	} stmt_flow_type;

class Stmt : public GlishObject {
    public:
	// Exec() tells a statement to go ahead and execute.  We use
	// it as a wrapper around the actual execute member function
	// so we can reset the current line number and perform any
	// other "global" statement execution (such as setting the
	// control flow to default to FLOW_NEXT).
	//
	// Exec() returns a value associated with the statement or 0
	// if there is none, and in "flow" returns a stmt_flow_type
	// indicating control flow information.
	virtual Value* Exec( stmt_flow_type& flow );

	// Called when an event we've expressed interest in has arrived.
	// The argument specifies the Agent associated with the
	// event.
	virtual void Notify( Agent* agent );

    protected:
	// DoExec() does the real work of executing the statement.
	virtual Value* DoExec( stmt_flow_type& flow ) = 0;
	};


class SeqStmt : public Stmt {
    public:
	SeqStmt( Stmt* arg_lhs, Stmt* arg_rhs );

	Value* DoExec( stmt_flow_type& flow );
	void Describe( ostream& s ) const;

    protected:
	Stmt* lhs;
	Stmt* rhs;
	};


class WheneverStmt : public Stmt {
    public:
	WheneverStmt( event_list* arg_trigger, Stmt* arg_stmt,
			Sequencer* arg_sequencer );

	virtual ~WheneverStmt();

	Value* DoExec( stmt_flow_type& flow );
	void Notify( Agent* agent );
	void Describe( ostream& s ) const;

    protected:
	event_list* trigger;
	Stmt* stmt;
	Sequencer* sequencer;
	};


class LinkStmt : public Stmt {
    public:
	LinkStmt( event_list* source, event_list* sink, Sequencer* sequencer );

	Value* DoExec( stmt_flow_type& flow );
	void Describe( ostream& s ) const;

    protected:
	void MakeLink( Task* src, const char* source_event,
			Task* snk, const char* sink_event );

	virtual void LinkAction( Task* src, Value* v, bool is_local );

	event_list* source;
	event_list* sink;
	Sequencer* sequencer;
	};


class UnLinkStmt : public LinkStmt {
    public:
	UnLinkStmt( event_list* source, event_list* sink,
			Sequencer* sequencer );

    protected:
	void LinkAction( Task* src, Value* v, bool is_local );
	};

class AwaitStmt : public Stmt {
    public:
	AwaitStmt( event_list* arg_await_list, bool arg_only_flag,
		   event_list* arg_except_list,
		   Sequencer* arg_sequencer );

	Value* DoExec( stmt_flow_type& flow );
	void Describe( ostream& s ) const;

    protected:
	event_list* await_list;
	bool only_flag;
	event_list* except_list;
	Sequencer* sequencer;
	Stmt* except_stmt;
	};


class IfStmt : public Stmt {
    public:
	IfStmt( Expr* arg_expr,
		Stmt* arg_true_branch,
		Stmt* arg_false_branch );

	Value* DoExec( stmt_flow_type& flow );
	void Describe( ostream& s ) const;

    protected:
	Expr* expr;
	Stmt* true_branch;
	Stmt* false_branch;
	};


class ForStmt : public Stmt {
    public:
	ForStmt( Expr* index_expr, Expr* range_expr,
		 Stmt* body_stmt );

	Value* DoExec( stmt_flow_type& flow );
	void Describe( ostream& s ) const;

    protected:
	Expr* index;
	Expr* range;
	Stmt* body;
	};


class WhileStmt : public Stmt {
    public:
	WhileStmt( Expr* test_expr, Stmt* body_stmt );

	Value* DoExec( stmt_flow_type& flow );
	void Describe( ostream& s ) const;

    protected:
	Expr* test;
	Stmt* body;
	};


class PrintStmt : public Stmt {
    public:
	PrintStmt( parameter_list* arg_args )
		{
		args = arg_args;
		description = "print";
		}

	Value* DoExec( stmt_flow_type& flow );
	void Describe( ostream& s ) const;

    protected:
	parameter_list* args;
	};


class AssignStmt : public Stmt {
    public:
	AssignStmt( Expr* arg_lhs, Expr* arg_rhs )
		{
		lhs = arg_lhs;
		rhs = arg_rhs;
		description = ":=";
		}

	Value* DoExec( stmt_flow_type& flow );
	void Describe( ostream& s ) const;

    protected:
	Expr* lhs;
	Expr* rhs;
	};


class SendEventStmt : public Stmt {
    public:
	SendEventStmt( EventDesignator* sender, parameter_list* args );

	Value* DoExec( stmt_flow_type& flow );
	void Describe( ostream& s ) const;

    protected:
	EventDesignator* sender;
	parameter_list* args;
	};


class ExprStmt : public Stmt {
    public:
	ExprStmt( Expr* arg_expr )
		{ expr = arg_expr; description = "expression"; }

	Value* DoExec( stmt_flow_type& flow );
	void Describe( ostream& s ) const;
	void DescribeSelf( ostream& s ) const;

    protected:
	Expr* expr;
	};


class ExitStmt : public Stmt {
    public:
	ExitStmt( Expr* arg_status )
		{ description = "exit"; status = arg_status; }

	Value* DoExec( stmt_flow_type& flow );
	void Describe( ostream& s ) const;

    protected:
	Expr* status;
	};


class LoopStmt : public Stmt {
    public:
	LoopStmt()	{ description = "next"; }

	Value* DoExec( stmt_flow_type& flow );
	};


class BreakStmt : public Stmt {
    public:
	BreakStmt()	{ description = "break"; }

	Value* DoExec( stmt_flow_type& flow );
	};


class ReturnStmt : public Stmt {
    public:
	ReturnStmt( Expr* arg_retval )
		{ description = "return"; retval = arg_retval; }

	Value* DoExec( stmt_flow_type& flow );
	void Describe( ostream& s ) const;

    protected:
	Expr* retval;
	};


class NullStmt : public Stmt {
    public:
	NullStmt()	{ description = ";"; }

	Value* DoExec( stmt_flow_type& flow );
	};


extern Stmt* null_stmt;
extern Stmt* merge_stmts( Stmt* stmt1, Stmt* stmt2 );

#endif /* stmt_h */
