// $Header: Stmt.cc,v 1.10 93/01/13 15:34:57 vern Exp $

#include <stdio.h>
#include <stdlib.h>
#include <stream.h>
#include <string.h>

#include "Glish/Value.h"

#include "Stmt.h"
#include "BuiltIn.h"
#include "Task.h"
#include "Reporter.h"
#include "Sequencer.h"


Stmt* null_stmt;


Value* Stmt::Exec( stmt_flow_type& flow )
	{
	int prev_line_num = line_num;

	line_num = Line();
	flow = FLOW_NEXT;

	Value* result = DoExec( flow );

	line_num = prev_line_num;

	return result;
	}

void Stmt::Notify( Agent* /* agent */ )
	{
	}


SeqStmt::SeqStmt( Stmt* arg_lhs, Stmt* arg_rhs )
	{
	lhs = arg_lhs;
	rhs = arg_rhs;
	description = "sequence";
	}

Value* SeqStmt::DoExec( stmt_flow_type& flow )
	{
	Value* result = lhs->Exec( flow );

	if ( flow == FLOW_NEXT )
		{
		Unref( result );
		result = rhs->Exec( flow );
		}

	return result;
	}

void SeqStmt::Describe( ostream& s ) const
	{
	s << "{\n";
	lhs->Describe( s );
	s << "\n";
	rhs->Describe( s );
	s << "}\n";
	}


WheneverStmt::WheneverStmt( event_list* arg_trigger, Stmt* arg_stmt,
			Sequencer* arg_sequencer )
	{
	trigger = arg_trigger;
	stmt = arg_stmt;
	sequencer = arg_sequencer;

	description = "whenever";
	}

WheneverStmt::~WheneverStmt()
	{
	}

Value* WheneverStmt::DoExec( stmt_flow_type& /* flow */ )
	{
	Frame* frame = sequencer->CurrentFrame();

	loop_over_list( *trigger, i )
		(*trigger)[i]->Register( new Notifiee( this, frame ) );

	return 0;
	}

void WheneverStmt::Notify( Agent* /* agent */ )
	{
	stmt_flow_type flow;
	Unref( stmt->Exec( flow ) );

	if ( flow != FLOW_NEXT )
		warn->Report( "loop/break/return does not make sense inside",
				this );
	}

void WheneverStmt::Describe( ostream& s ) const
	{
	DescribeSelf( s );
	s << " ";
	describe_event_list( trigger, s );
	s << " do ";
	stmt->Describe( s );
	}


LinkStmt::LinkStmt( event_list* arg_source, event_list* arg_sink,
			Sequencer* arg_sequencer )
	{
	source = arg_source;
	sink = arg_sink;
	sequencer = arg_sequencer;

	description = "link";
	}

Value* LinkStmt::DoExec( stmt_flow_type& /* flow */ )
	{
	loop_over_list( *source, i )
		{
		EventDesignator* src = (*source)[i];
		Agent* src_agent = src->EventAgent( VAL_CONST );

		if ( ! src_agent )
			{
			error->Report( src->EventAgentExpr(),
					"is not an agent" );
			continue;
			}

		Task* src_task = src_agent->AgentTask();

		if ( ! src_task )
			{
			error->Report( src->EventAgentExpr(),
				"is not a client" );
			continue;
			}

		PList(char)* name_list = src->EventNames();

		if ( ! name_list )
			{
			error->Report( this,
				"linking of all events not yet supported" );
			continue;
			}

		loop_over_list( *sink, j )
			{
			EventDesignator* snk = (*sink)[j];
			Agent* snk_agent = snk->EventAgent( VAL_REF );

			if ( ! snk_agent )
				{
				error->Report( snk->EventAgentExpr(),
					"is not an agent" );
				continue;
				}

			Task* snk_task = snk_agent->AgentTask();

			if ( ! snk_task )
				{
				error->Report( snk->EventAgentExpr(),
					"is not a client" );
				continue;
				}

			PList(char)* sink_list = snk->EventNames();

			if ( sink_list->length() > 1 )
				error->Report(
				"multiple event names not allowed in \"to\":",
						this );

			loop_over_list( *name_list, k )
				{
				const char* name = (*name_list)[k];

				MakeLink( src_task, name, snk_task,
					sink_list ? (*sink_list)[0] : name );
				}

			delete_name_list( sink_list );
			snk->EventAgentDone();
			}

		delete_name_list( name_list );
		src->EventAgentDone();
		}

	return 0;
	}

void LinkStmt::Describe( ostream& s ) const
	{
	DescribeSelf( s );
	s << " ";
	describe_event_list( source, s );
	s << " to ";
	describe_event_list( sink, s );
	}

void LinkStmt::MakeLink( Task* src, const char* source_event,
			 Task* snk, const char* sink_event )
	{
	Value* v = create_record();

	v->SetField( "event", source_event );
	v->SetField( "new_name", sink_event );
	v->SetField( "task_id", snk->TaskID() );
	v->SetField( "receipient", snk->TaskID() );

	LinkAction( src, v, same_host( src, snk ) );

	Unref( v );
	}

void LinkStmt::LinkAction( Task* src, Value* v, bool is_local )
	{
	if ( is_local )
		src->SendSingleValueEvent( "*pipe-sink*", v, true );
	else
		src->SendSingleValueEvent( "*socket-sink*", v, true );
	}


UnLinkStmt::UnLinkStmt( event_list* arg_source, event_list* arg_sink,
			Sequencer* arg_sequencer )
: LinkStmt( arg_source, arg_sink, arg_sequencer )
	{
	description = "unlink";
	}

void UnLinkStmt::LinkAction( Task* src, Value* v, bool /* is_local */ )
	{
	src->SendSingleValueEvent( "*unlink-sink*", v, true );
	}


AwaitStmt::AwaitStmt( event_list* arg_await_list, bool arg_only_flag,
			event_list* arg_except_list, Sequencer* arg_sequencer )
	{
	await_list = arg_await_list;
	only_flag = arg_only_flag;
	except_list = arg_except_list;
	sequencer = arg_sequencer;
	except_stmt = null_stmt;

	description = "await";
	}

Value* AwaitStmt::DoExec( stmt_flow_type& /* flow */ )
	{
	loop_over_list( *await_list, i )
		(*await_list)[i]->Register( new Notifiee( this, 0 ) );

	if ( except_list )
		loop_over_list( *except_list, j )
			(*except_list)[j]->Register(
					new Notifiee( except_stmt, 0 ) );

	sequencer->Await( this, only_flag, except_stmt );

	return 0;
	}

void AwaitStmt::Describe( ostream& s ) const
	{
	s << "await ";

	loop_over_list( *await_list, i )
		{
		(*await_list)[i]->Describe( s );
		s << " ";
		}

	if ( except_list )
		{
		s << " except ";

		loop_over_list( *except_list, j )
			{
			(*except_list)[j]->Describe( s );
			s << " ";
			}
		}
	}


IfStmt::IfStmt( Expr* arg_expr, Stmt* arg_true_branch,
		Stmt* arg_false_branch )
	{
	expr = arg_expr;
	true_branch = arg_true_branch;
	false_branch = arg_false_branch;
	description = "if";
	}

Value* IfStmt::DoExec( stmt_flow_type& flow )
	{
	const Value* test_value = expr->ReadOnlyEval();
	bool take_true_branch = test_value->BoolVal();
	expr->ReadOnlyDone( test_value );

	Value* result = 0;

	if ( take_true_branch )
		{
		if ( true_branch )
			result = true_branch->Exec( flow );
		}

	else if ( false_branch )
		result = false_branch->Exec( flow );

	return result;
	}

void IfStmt::Describe( ostream& s ) const
	{
	s << "if ";
	expr->Describe( s );
	s << " ";

	if ( true_branch )
		true_branch->Describe( s );
	else
		s << " { } ";

	if ( false_branch )
		{
		s << "\nelse ";
		false_branch->Describe( s );
		}
	}


ForStmt::ForStmt( Expr* index_expr, Expr* range_expr,
		  Stmt* body_stmt )
	{
	index = index_expr;
	range = range_expr;
	body = body_stmt;
	description = "for";
	}

Value* ForStmt::DoExec( stmt_flow_type& flow )
	{
	Value* range_value = range->CopyEval();

	Value* result = 0;

	if ( ! range_value->IsNumeric() && range_value->Type() != TYPE_STRING )
		error->Report( "range (", range,
				") in for loop is not numeric or string" );

	else
		{
		int len = range_value->Length();

		for ( int i = 1; i <= len; ++i )
			{
			Value* loop_counter = new Value( i );
			Value* iter_value = (*range_value)[loop_counter];

			index->Assign( iter_value );

			Unref( result );
			result = body->Exec( flow );

			Unref( loop_counter );

			if ( flow == FLOW_BREAK || flow == FLOW_RETURN )
				break;
			}
		}

	Unref( range_value );

	if ( flow != FLOW_RETURN )
		flow = FLOW_NEXT;

	return result;
	}

void ForStmt::Describe( ostream& s ) const
	{
	s << "for ( ";
	index->Describe( s );
	s << " in ";
	range->Describe( s );
	s << " ) ";
	body->Describe( s );
	}


WhileStmt::WhileStmt( Expr* test_expr, Stmt* body_stmt )
	{
	test = test_expr;
	body = body_stmt;
	description = "while";
	}

Value* WhileStmt::DoExec( stmt_flow_type& flow )
	{
	Value* result = 0;

	while ( 1 )
		{
		const Value* test_value = test->ReadOnlyEval();
		bool do_test = test_value->BoolVal();
		test->ReadOnlyDone( test_value );

		if ( do_test )
			{
			Unref( result );
			result = body->Exec( flow );
			if ( flow == FLOW_BREAK || flow == FLOW_RETURN )
				break;
			}

		else
			break;
		}

	if ( flow != FLOW_RETURN )
		flow = FLOW_NEXT;

	return result;
	}

void WhileStmt::Describe( ostream& s ) const
	{
	s << "while ( ";
	test->Describe( s );
	s << " ) ";
	body->Describe( s );
	}


Value* PrintStmt::DoExec( stmt_flow_type& /* flow */ )
	{
	if ( args )
		{
		char* args_string = paste( args );
		message->Report( args_string );
		delete args_string;
		}

	else
		message->Report( "" );

	return 0;
	}

void PrintStmt::Describe( ostream& s ) const
	{
	s << "print ";

	describe_parameter_list( args, s );

	s << ";";
	}


Value* AssignStmt::DoExec( stmt_flow_type& /* flow */ )
	{
	lhs->Assign( rhs->CopyEval() );
	return 0;
	}

void AssignStmt::Describe( ostream& s ) const
	{
	lhs->Describe( s );
	s << " := ";
	rhs->Describe( s );
	s << ";";
	}


SendEventStmt::SendEventStmt( EventDesignator* arg_sender,
				parameter_list* arg_args )
	{
	sender = arg_sender;
	args = arg_args;
	description = "->";
	}

Value* SendEventStmt::DoExec( stmt_flow_type& /* flow */ )
	{
	sender->SendEvent( args );
	return 0;
	}

void SendEventStmt::Describe( ostream& s ) const
	{
	sender->Describe( s );
	s << "->(";
	describe_parameter_list( args, s );
	s << ")";
	}


Value* ExprStmt::DoExec( stmt_flow_type& /* flow */ )
	{
	return expr->CopyEval();
	}

void ExprStmt::Describe( ostream& s ) const
	{
	expr->Describe( s );
	s << ";";
	}

void ExprStmt::DescribeSelf( ostream& s ) const
	{
	expr->DescribeSelf( s );
	}

Value* ExitStmt::DoExec( stmt_flow_type& /* flow */ )
	{
	exit( status ? status->CopyEval()->IntVal() : 0 );
	return 0;
	}

void ExitStmt::Describe( ostream& s ) const
	{
	s << "exit";

	if ( status )
		{
		s << " ";
		status->Describe( s );
		}
	}

Value* LoopStmt::DoExec( stmt_flow_type& flow )
	{
	flow = FLOW_LOOP;
	return 0;
	}

Value* BreakStmt::DoExec( stmt_flow_type& flow )
	{
	flow = FLOW_BREAK;
	return 0;
	}

Value* ReturnStmt::DoExec( stmt_flow_type& flow )
	{
	flow = FLOW_RETURN;

	if ( retval )
		return retval->CopyEval();

	else
		return 0;
	}

void ReturnStmt::Describe( ostream& s ) const
	{
	s << "return";

	if ( retval )
		{
		s << " ";
		retval->Describe( s );
		}
	}

Value* NullStmt::DoExec( stmt_flow_type& /* flow */ )
	{
	return 0;
	}


Stmt* merge_stmts( Stmt* stmt1, Stmt* stmt2 )
	{
	if ( stmt1 == null_stmt )
		return stmt2;

	else if ( stmt2 == null_stmt )
		return stmt1;

	else
		return new SeqStmt( stmt1, stmt2 );
	}
