// $Header: Sequencer.cc,v 1.18 93/01/15 14:56:22 vern Locked $

#include <stdlib.h>
#include <string.h>
#include <stream.h>
#include <signal.h>


#define GLISH_VERSION "2.2"
#define GLISH_INIT_FILE "glish.init"
#define GLISH_RC_FILE ".glishrc"


// used for MAXHOSTNAMELEN
#include <sys/param.h>

#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN 64
#endif

extern "C" {
int gethostname( char* name, int namelen );
char* getcwd( char* buf, int size );
char* getenv( const char* );
int isatty( int fd );
int system( const char* string );
}


#include "Glish/Client.h"

#include "Reporter.h"
#include "Sequencer.h"
#include "Frame.h"
#include "BuiltIn.h"
#include "Task.h"
#include "input.h"
#include "Channel.h"
#include "Select.h"
#include "Socket.h"
#include "system.h"


// A Selectee corresponding to input for a Glish client ("task" in the
// older terminology).
class TaskSelectee : public Selectee {
    public:
	TaskSelectee( Sequencer* s, Task* t );
	int NotifyOfSelection();

    protected:
	Sequencer* sequencer;
	Task* task;
	};


// A Selectee corresponding to a new request of a client to connect
// to the sequencer.
class AcceptSelectee : public Selectee {
    public:
	AcceptSelectee( Sequencer* s, int conn_socket );
	int NotifyOfSelection();

    protected:
	Sequencer* sequencer;
	int connection_socket;
	};


// A Selectee corresponding to a Glish script's own client.
class ScriptSelectee : public Selectee {
public:
	ScriptSelectee( Client* client, Agent* agent, int conn_socket );
	int NotifyOfSelection();

protected:
	Client* script_client;
	Agent* script_agent;
	int connection_socket;
	};


// A special type of Client used for script clients.  It overrides
// FD_Change() to create or delete ScriptSelectee's as needed.
class ScriptClient : public Client {
public:
	ScriptClient( int& argc, char** argv );

	// Inform the ScriptClient as to which selector and agent
	// it should use for getting and propagating events.
	void SetInterface( Selector* selector, Agent* agent );

protected:
	void FD_Change( int fd, bool add_flag );

	Selector* selector;
	Agent* agent;
	};


// A special type of Agent used for script clients; when it receives
// an event, it propagates it via the ScriptClient object.
class ScriptAgent : public Agent {
public:
	ScriptAgent( Sequencer* s, Client* c ) : Agent(s)	{ client = c; }

	void SendEvent( const char* event_name, parameter_list* args,
			bool /* log */ )
		{
		Value* event_val = BuildEventValue( args, true );
		client->PostEvent( event_name, event_val );
		Unref( event_val );
		}

protected:
	Client* client;
	};


Notification::Notification( Agent* arg_notifier, const char* arg_field,
			    Value* arg_value, Notifiee* arg_notifiee )
	{
	notifier = arg_notifier;
	field = strdup( arg_field );
	value = arg_value;
	notifiee = arg_notifiee;

	Ref( value );
	}

Notification::~Notification()
	{
	delete field;
	Unref( value );
	}

void Notification::Describe( ostream& s ) const
	{
	s << "notification of ";
	notifier->DescribeSelf( s );
	s << "." << field << " (";
	value->DescribeSelf( s );
	s << ") for ";
	notifiee->stmt->DescribeSelf( s );
	}


Sequencer::Sequencer( int& argc, char**& argv )
	{
	init_reporters();
	init_values();

	// Create the global scope.
	scopes.append( new expr_dict );

	create_built_ins( this );

	null_stmt = new NullStmt;

	stmts = null_stmt;
	last_task_id = my_id = 1;

	await_stmt = except_stmt = 0;
	await_only_flag = false;
	pending_task = 0;

	maximize_num_fds();

	// avoid SIGPIPE's - they can occur if a client's termination
	// is not detected prior to sending an event to the client
	sigblock( sigmask( SIGPIPE ) );

	connection_socket = new AcceptSocket;
	mark_close_on_exec( connection_socket->FD() );

	selector = new Selector;

	accept_selectee = new AcceptSelectee( this, connection_socket->FD() );
	selector->AddSelectee( accept_selectee );

	connection_host = new char[MAXHOSTNAMELEN];
	if ( gethostname( connection_host, MAXHOSTNAMELEN ) < 0 )
		fatal->Report( "couldn't get local host name" );

	connection_port = new char[32];
	sprintf( connection_port, "%d", connection_socket->Port() );

	monitor_task = 0;
	last_notification = 0;

	num_active_processes = 0;
	verbose = 0;

	Expr* version_expr = InstallID( strdup( "version" ), GLOBAL_SCOPE );
	version_expr->Assign( new Value( GLISH_VERSION ) );

	script_client = new ScriptClient( argc, argv );

	if ( script_client->HasSequencerConnection() )
		{
		// Set up script agent to deal with incoming and outgoing
		// events.
		ScriptAgent* script_agent =
			new ScriptAgent( this, script_client );
		script_client->SetInterface( selector, script_agent );

		Expr* script_expr = InstallID( strdup( "script" ),
						GLOBAL_SCOPE );
		script_expr->Assign( script_agent->AgentRecord() );
		}

	name = argv[0];

	for ( ++argv, --argc; argc > 0; ++argv, --argc )
		{
		if ( ! strcmp( argv[0], "-v" ) )
			++verbose;

		else if ( strchr( argv[0], '=' ) )
			putenv( argv[0] );

		else
			break;
		}

	MakeEnvGlobal();
	BuildSuspendList();

	char* monitor_client_name = getenv( "glish_monitor" );
	if ( monitor_client_name )
		ActivateMonitor( monitor_client_name );

	char glish_init_file[256];
	char* glish_init_env = getenv( "glish_init" );

	if ( glish_init_env )
		strcpy( glish_init_file, glish_init_env );
	else
#ifdef GLISH_DIR
		sprintf( glish_init_file, "%s/%s", GLISH_DIR, GLISH_INIT_FILE );
#else
		strcpy( glish_init_file, GLISH_INIT_FILE );
#endif

	Parse( glish_init_file );

	const char* glish_rc;
	if ( (glish_rc = getenv( "GLISHRC" )) )
		Parse( glish_rc );

	else
		{
		FILE* glish_rc_file = fopen( GLISH_RC_FILE, "r" );
		const char* home;

		if ( glish_rc_file )
			{
			fclose( glish_rc_file );
			Parse( GLISH_RC_FILE );
			}

		else if ( (home = getenv( "HOME" )) )
			{
			char glish_rc_filename[256];
			sprintf( glish_rc_filename, "%s/%s",
					home, GLISH_RC_FILE );

			if ( (glish_rc_file = fopen( glish_rc_filename, "r")) )
				{
				fclose( glish_rc_file );
				Parse( glish_rc_filename );
				}
			}
		}

	if ( argc > 0 )
		{
		while ( argc > 0 && strcmp( argv[0], "--" ) )
			{
			Parse( argv[0] );
			++argv, --argc;
			}
		}

	else
		Parse( stdin );

	if ( argc > 0 )
		// Must have encountered "--".  Skip over it.
		++argv, --argc;

	MakeArgvGlobal( argv, argc );
	}


Sequencer::~Sequencer()
	{
	delete script_client;
	delete selector;
	delete connection_socket;
	delete connection_host;
	delete connection_port;
	}


void Sequencer::AddBuiltIn( BuiltIn* built_in )
	{
	Expr* id = InstallID( strdup( built_in->Name() ), GLOBAL_SCOPE );
	id->Assign( new Value( built_in ) );
	}


void Sequencer::QueueNotification( Notification* n )
	{
	if ( verbose > 1 )
		message->Report( "queueing", n );

	notification_queue.EnQueue( n );
	}


void Sequencer::PushScope()
	{
	scopes.append( new expr_dict );
	}

int Sequencer::PopScope()
	{
	int top_scope_pos = scopes.length() - 1;

	if ( top_scope_pos < 0 )
		fatal->Report( "scope underflow in Sequencer::PopScope" );

	expr_dict* top_scope = scopes[top_scope_pos];
	int frame_size = top_scope->Length();

	scopes.remove( top_scope );
	delete top_scope;

	return frame_size;
	}


Expr* Sequencer::InstallID( char* id, scope_type scope )
	{
	int scope_index;

	if ( scope == LOCAL_SCOPE )
		scope_index = scopes.length() - 1;
	else
		scope_index = 0;

	int frame_offset = scopes[scope_index]->Length();
	Expr* result = new VarExpr( id, scope, frame_offset, this );

	scopes[scope_index]->Insert( id, result );

	if ( scope == GLOBAL_SCOPE )
		global_frame.append( 0 );

	return result;
	}

Expr* Sequencer::LookupID( char* id, scope_type scope, bool do_install )
	{
	int scope_index;

	if ( scope == LOCAL_SCOPE )
		scope_index = scopes.length() - 1;
	else
		scope_index = 0;

	Expr* result = (*scopes[scope_index])[id];

	if ( ! result && do_install )
		{
		if ( scope == LOCAL_SCOPE )
			return LookupID( id, GLOBAL_SCOPE );
		else
			return InstallID( id, GLOBAL_SCOPE );
		}

	delete id;
	return result;
	}


void Sequencer::PushFrame( Frame* new_frame )
	{
	local_frames.append( new_frame );
	}

Frame* Sequencer::PopFrame()
	{
	int top_frame = local_frames.length() - 1;
	if ( top_frame < 0 )
		fatal->Report(
			"local frame stack underflow in Sequencer::PopFrame" );

	return local_frames.remove_nth( top_frame );
	}

Frame* Sequencer::CurrentFrame()
	{
	int top_frame = local_frames.length() - 1;
	if ( top_frame < 0 )
		return 0;

	return local_frames[top_frame];
	}


Value* Sequencer::FrameElement( scope_type scope, int frame_offset )
	{
	if ( scope == LOCAL_SCOPE )
		{
		int top_frame = local_frames.length() - 1;
		if ( top_frame < 0 )
			fatal->Report(
	    "local frame requested but none exist in Sequencer::FrameElement" );

		return local_frames[top_frame]->FrameElement( frame_offset );
		}

	else
		{
		if ( frame_offset < 0 || frame_offset >= global_frame.length() )
			fatal->Report(
			"bad global frame offset in Sequencer::FrameElement" );
		return global_frame[frame_offset];
		}
	}

void Sequencer::SetFrameElement( scope_type scope, int frame_offset,
					Value* value )
	{
	Value* prev_value;

	if ( scope == LOCAL_SCOPE )
		{
		int top_frame = local_frames.length() - 1;
		if ( top_frame < 0 )
			fatal->Report(
	"local frame requested but none exist in Sequencer::SetFrameElement" );

		Value*& frame_value =
			local_frames[top_frame]->FrameElement( frame_offset );
		prev_value = frame_value;
		frame_value = value;
		}

	else
		{
		if ( frame_offset < 0 || frame_offset >= global_frame.length() )
			fatal->Report(
		"bad global frame offset in Sequencer::SetFrameElement" );
		prev_value = global_frame.replace( frame_offset, value );
		}

	Unref( prev_value );
	}


char* Sequencer::RegisterTask( Task* new_task )
	{
	char buf[128];
	sprintf( buf, "task%d", ++last_task_id );

	char* new_ID = strdup( buf );

	ids_to_tasks.Insert( new_ID, new_task );

	return new_ID;
	}

void Sequencer::DeleteTask( Task* task )
	{
	(void) ids_to_tasks.Remove( task->TaskID() );
	}


void Sequencer::AddStmt( Stmt* addl_stmt )
	{
	stmts = merge_stmts( stmts, addl_stmt );
	}


Channel* Sequencer::GetHostDaemon( const char* host )
	{
	if ( ! host ||
	     ! strcmp( host, ConnectionHost() ) ||
	     ! strcmp( host, "localhost" ) )
		// request is for local host
		return 0;

	Channel* daemon_channel = daemons[host];

	if ( ! daemon_channel )
		daemon_channel = CreateDaemon( host );

	return daemon_channel;
	}


void Sequencer::Exec()
	{
	if ( interactive )
		return;

	if ( error->Count() > 0 )
		{
		message->Report( "execution aborted" );
		return;
		}

	stmt_flow_type flow;
	Unref( stmts->Exec( flow ) );

	EventLoop();
	}


void Sequencer::Await( Stmt* arg_await_stmt, bool only_flag,
			Stmt* arg_except_stmt )
	{
	Stmt* hold_await_stmt = await_stmt;
	bool hold_only_flag = await_only_flag;
	Stmt* hold_except_stmt = except_stmt;

	await_stmt = arg_await_stmt;
	await_only_flag = only_flag;
	except_stmt = arg_except_stmt;

	EventLoop();

	await_stmt = hold_await_stmt;
	await_only_flag = only_flag;
	except_stmt = hold_except_stmt;
	}


void Sequencer::NewConnection( Channel* connection_channel )
	{
	GlishEvent* establish_event =
		recv_event( connection_channel->ReadFD() );

	if ( ! establish_event )
		{
		error->Report( "new connection immediately broken" );
		return;
		}

	char* task_id = establish_event->value->StringVal();
	Task* task = ids_to_tasks[task_id];

	if ( ! task )
		{
		error->Report( "connection received from non-existent task ",
				task_id );
		Unref( establish_event );
		}

	else
		{
		AssociateTaskWithChannel( task, connection_channel );
		NewEvent( task, establish_event );
		}

	delete task_id;
	}


void Sequencer::AssociateTaskWithChannel( Task* task, Channel *chan )
	{
	task->SetChannel( chan, selector );
	task->SetActive();

	selector->AddSelectee( new TaskSelectee( this, task ) );

	// empty out buffer so subsequent select()'s will work
	if ( chan->DataInBuffer() )
		EmptyTaskChannel( task );
	}


int Sequencer::NewEvent( Task* task, GlishEvent* event )
	{
	if ( ! event )
		{ // task termination
		task->CloseChannel();

		if ( ! task->Active() )
			return 0;

		// Abnormal termination - no "done" message first.
		event = new GlishEvent( strdup( "fail" ),
					new Value( task->AgentID() ) );
		}

	char* event_name = event->name;
	Value* value = event->value;

	if ( verbose > 0 )
		message->Report( name, ": received event ",
				 task->Name(), ".", event_name, " ", value );

	if ( monitor_task && task != monitor_task )
		LogEvent( task->Name(), event_name, value, true );

	// If true, generate message if no interest in event.
	bool complain_if_no_interest = true;

	if ( ! strcmp( event_name, "established" ) )
		{
		// We already did the SetActive() when the channel
		// was established.
		complain_if_no_interest = false;
		}

	else if ( ! strcmp( event_name, "done" ) )
		{
		task->SetDone();
		complain_if_no_interest = false;
		}

	else if ( ! strcmp( event_name, "fail" ) )
		task->SetDone();

	else if ( ! strcmp( event_name, "*forward*" ) )
		{
		ForwardEvent( event_name, value );
		complain_if_no_interest = false;
		}


	bool ignore_event = false;
	bool await_finished = false;

	if ( await_stmt )
		{
		await_finished =
			task->HasRegisteredInterest( await_stmt, event_name );

		if ( ! await_finished && await_only_flag &&
		     ! task->HasRegisteredInterest( except_stmt, event_name ) )
			ignore_event = true;
		}

	if ( ignore_event )
		warn->Report( "event ", task->Name(), ".", event_name,
			      " ignored due to \"await\"" );

	else
		{
		// We're going to want to keep the event value as a field
		// in the task's AgentRecord.
		Ref( value );

		bool was_interest = task->CreateEvent( event_name, value );

		if ( ! was_interest && complain_if_no_interest )
			warn->Report( "event ", task->Name(), ".", event_name,
					" (", value, ") dropped" );

		RunQueue();	// process effects of CreateEvent()
		}

	Unref( event );

	if ( await_finished )
		{
		pending_task = task;

		// Make sure the pending task isn't delete'd before
		// we can exhaust its pending input.

		Ref( pending_task );

		return 1;
		}

	else
		return 0;
	}


void Sequencer::NewClientStarted()
	{
	++num_active_processes;
	}


bool Sequencer::ShouldSuspend( const char* task_var_ID )
	{
	if ( task_var_ID )
		return suspend_list[task_var_ID];
	else
		// This is an anonymous client - don't suspend.
		return false;
	}

int Sequencer::EmptyTaskChannel( Task* task, int force_read )
	{
	int status = 0;

	if ( task->Active() )
		{
		Channel* c = task->GetChannel();
		ChanState old_state = c->ChannelState();

		c->ChannelState() = CHAN_IN_USE;

		if ( force_read )
			status = NewEvent( task, recv_event( c->ReadFD() ) );

		while ( status == 0 &&
			c->ChannelState() == CHAN_IN_USE &&
			c->DataInBuffer() )
			{
			status = NewEvent( task, recv_event( c->ReadFD() ) );
			}

		if ( c->ChannelState() == CHAN_INVALID )
			{ // This happens iff the given task has exited
			selector->DeleteSelectee( c->ReadFD() );
			delete c;

			while ( reap_terminated_process() )
				;

			--num_active_processes;
			}

		else
			c->ChannelState() = old_state;
		}

	return status;
	}


void Sequencer::MakeEnvGlobal()
	{
	Value* env_value = create_record();

	extern char** environ;
	for ( char** env_ptr = environ; *env_ptr; ++env_ptr )
		{
		char* delim = strchr( *env_ptr, '=' );

		if ( delim )
			{
			*delim = '\0';
			env_value->AssignRecordElement( *env_ptr,
						    new Value( delim + 1 ) );
			*delim = '=';
			}
		else
			env_value->AssignRecordElement( *env_ptr,
							new Value( false ) );
		}

	Expr* env_expr = LookupID( strdup( "environ" ), GLOBAL_SCOPE );
	env_expr->Assign( env_value );
	}


void Sequencer::MakeArgvGlobal( char** argv, int argc )
	{
	Value* argv_value = new Value( argv, argc, COPY_ARRAY );
	Expr* argv_expr = LookupID( strdup( "argv" ), GLOBAL_SCOPE );
	argv_expr->Assign( argv_value );
	}


void Sequencer::BuildSuspendList()
	{
	char* suspend_env_list = getenv( "suspend" );

	if ( ! suspend_env_list )
		return;

	char* suspendee = strtok( suspend_env_list, " " );

	while ( suspendee )
		{
		suspend_list.Insert( suspendee, true );
		suspendee = strtok( 0, " " );
		}
	}


void Sequencer::Parse( FILE* file, const char* filename )
	{
	restart_yylex( file );

	yyin = file;
	current_sequencer = this;
	line_num = 1;
	input_file_name = filename;

	if ( isatty( fileno( yyin ) ) )
		{
		// We're about to enter the "interactive" loop, so
		// first execute any statements we've seen so far due
		// to .glishrc files.

		message->Report( "Glish version ", GLISH_VERSION, "." );

		Exec();

		interactive = true;
		}

	else
		interactive = false;

	if ( yyparse() )
		error->Report( "syntax errors parsing input" );

	input_file_name = 0;
	line_num = 0;
	}


void Sequencer::Parse( const char file[] )
	{
	FILE* f = fopen( file, "r" );

	if ( ! f )
		error->Report( "can't open file \"", file, "\"" );
	else
		Parse( f, file );
	}


Channel* Sequencer::CreateDaemon( const char* host )
	{
	char daemon_cmd[1024];
	char work_dir[MAXPATHLEN];

	if ( ! getcwd( work_dir, sizeof( work_dir ) ) )
		fatal->Report( "problems getting working directory:",
				work_dir );
#ifdef hpux
	static char rsh_cmd[] = "remsh";
#else
	static char rsh_cmd[] = "rsh";
#endif

	// We need a separate socket to accept connections from remote
	// sequencers; otherwise we can have a race in which we fire
	// up a remote sequencer but some previously fired up client
	// responds first.
	AcceptSocket remote_sequencer_connection_socket;

	sprintf( daemon_cmd,
	    "%s %s -n glishd -id glishd -host %s -port %d -wd %s &",
		rsh_cmd, host,  ConnectionHost(),
		remote_sequencer_connection_socket.Port(), work_dir );

	system( daemon_cmd );

	int daemon_conn =
		accept_connection( remote_sequencer_connection_socket.FD() );
	Channel* daemon_channel = new Channel( daemon_conn, daemon_conn );

	// read and discard daemon's "establish" event
	GlishEvent* e = recv_event( daemon_channel->ReadFD() );
	Unref( e );

	daemons.Insert( strdup( host ), daemon_channel );

	return daemon_channel;
	}


void Sequencer::ActivateMonitor( char* monitor_client_name )
	{
	TaskAttr* monitor_attrs =
		new TaskAttr( "*monitor*", "localhost",
				0, false, false, false );

	const_args_list monitor_args;
	monitor_args.append( new Value( monitor_client_name ) );

	monitor_task = new ClientTask( &monitor_args, monitor_attrs, this );

	if ( monitor_task->TaskError() )
		{
		Unref( monitor_task );
		monitor_task = 0;
		}
	}


void Sequencer::LogEvent( const char* id, const char* event_name,
			const Value* event_value, bool is_inbound )
	{
	if ( ! monitor_task )
		return;

	Value id_value( id );
	Value name_value( event_name );

	parameter_list args;

	ConstExpr id_expr( &id_value );
	ConstExpr name_expr( &name_value );
	ConstExpr value_expr( event_value );

	Parameter id_param( "id", VAL_VAL, &id_expr, false );
	Parameter name_param( "name", VAL_VAL, &name_expr, false );
	Parameter value_param( "value", VAL_VAL, &value_expr, false );

	args.insert( &name_param );
	args.insert( &id_param );
	args.insert( &value_param );

	const char* monitor_event_name = is_inbound ? "event_in" : "event_out";
	monitor_task->SendEvent( monitor_event_name, &args, false );
	}


void Sequencer::ForwardEvent( const char* event_name, Value* value )
	{
	char* receipient_id;
	char* new_event_name;

	if ( ! value->FieldVal( "receipient", receipient_id ) ||
	     ! value->FieldVal( "event", new_event_name ) )
		fatal->Report( "bad internal event \"", event_name, "\"" );

	Task* task = ids_to_tasks[receipient_id];

	if ( ! task )
		fatal->Report( "no such receipient ID in ", event_name,
				"internal event:", receipient_id );

	task->SendSingleValueEvent( new_event_name, value, true );

	delete receipient_id;
	delete new_event_name;
	}


void Sequencer::EventLoop()
	{
	RunQueue();

	if ( pending_task )
		{
		EmptyTaskChannel( pending_task );

		// We Ref()'d the pending_task when assigning it, to make
		// sure it didn't go away due to the effects of RunQueue().

		Unref( pending_task );

		pending_task = 0;
		}

	while ( num_active_processes > 0 && ! selector->DoSelection() )
		RunQueue();
	}


void Sequencer::RunQueue()
	{
	Notification* n;

	while ( (n = notification_queue.DeQueue()) )
		{
		if ( verbose > 1 )
			message->Report( "doing", n );

		if ( n->notifiee->frame )
			PushFrame( n->notifiee->frame );

		Value* notifier_val = n->notifier->AgentRecord();

		if ( notifier_val->Type() == TYPE_RECORD &&
		     notifier_val->HasRecordElement( n->field ) != n->value )
			// Need to assign the event value.
			notifier_val->AssignRecordElement( n->field, n->value );

		// There are a bunch of Ref's and Unref's here because the
		// Notify() call below can lead to a recursive call to this
		// routine (due to an "await" statement), so 'n' might
		// otherwise be deleted underneath our feet.
		Unref( last_notification );
		last_notification = n;

		Ref( n );
		n->notifiee->stmt->Notify( n->notifier );

		if ( n->notifiee->frame )
			(void) PopFrame();
		Unref( n );
		}
	}


TaskSelectee::TaskSelectee( Sequencer* s, Task* t )
    : Selectee( t->GetChannel()->ReadFD() )
	{
	sequencer = s;
	task = t;
	}

int TaskSelectee::NotifyOfSelection()
	{
	return sequencer->EmptyTaskChannel( task, 1 );
	}


AcceptSelectee::AcceptSelectee( Sequencer* s, int conn_socket )
    : Selectee( conn_socket )
	{
	sequencer = s;
	connection_socket = conn_socket;
	}

int AcceptSelectee::NotifyOfSelection()
	{
	int new_conn = accept_connection( connection_socket );
	mark_close_on_exec( new_conn );
	sequencer->NewConnection( new Channel( new_conn, new_conn ) );

	return 0;
	}


ScriptSelectee::ScriptSelectee( Client* client, Agent* agent, int conn_socket )
    : Selectee( conn_socket )
	{
	script_client = client;
	script_agent = agent;
	connection_socket = conn_socket;
	}

int ScriptSelectee::NotifyOfSelection()
	{
	fd_set fd_mask;

	FD_ZERO( &fd_mask );
	FD_SET( connection_socket, &fd_mask );

	GlishEvent* e = script_client->NextEvent( &fd_mask );

	if ( ! e )
		exit( 0 );

	script_agent->CreateEvent( e->name, e->value );

	return 0;
	}


ScriptClient::ScriptClient( int& argc, char** argv ) : Client( argc, argv )
	{
	selector = 0;
	agent = 0;
	}

void ScriptClient::SetInterface( Selector* s, Agent* a )
	{
	selector = s;
	agent = a;
	selector->AddSelectee( new ScriptSelectee( this, agent, read_fd ) );
	}

void ScriptClient::FD_Change( int fd, bool add_flag )
	{
	if ( ! agent )
		return;

	if ( add_flag )
		selector->AddSelectee( new ScriptSelectee( this, agent, fd ) );
	else
		selector->DeleteSelectee( fd );
	}
