// $Header: Expr.h,v 1.6 92/12/01 10:20:30 vern Exp $

#ifndef expr_h
#define expr_h

#include "Glish/Dict.h"
#include "Glish/Value.h"

class Stmt;
class Expr;
class Sequencer;
class ParameterPList;

declare(PList,Expr);
declare(PDict,Expr);

typedef PList(Expr) expr_list;


// Different scopes to use when resolving identifiers; used by the VarExpr
// and Sequencer classes.
typedef enum { LOCAL_SCOPE, GLOBAL_SCOPE } scope_type;


class Expr : public GlishObject {
    public:
	Expr( const char* desc )
		{ description = desc; }

	// Returns a copy of the present value of the event expression.
	// The caller is responsible for deleting the copy when done
	// using it.
	Value* CopyEval()
		{ return Eval( true ); }

	// Returns a read-only copy (i.e., the original) of the present
	// value of the event expression.  The caller is responsible for
	// later calling ReadOnlyDone() when the copy is no longer needed.
	const Value* ReadOnlyEval()
		{ return Eval( false ); }

	// Declares that the previously returned ReadOnlyEval() value
	// is no longer needed.
	void ReadOnlyDone( const Value* returned_value )
		{ Unref( (Value*) returned_value ); }

	// Returns the present value of the event expression.  If
	// "modifiable" is true then a modifiable version of the value is
	// returned; otherwise, a read-only copy.
	virtual Value* Eval( bool modifiable ) = 0;


	// Returns a reference to the value of the event expression.
	// If val_type is VAL_REF then a "ref" reference is returned,
	// otherwise a "const" reference.
	//
	// The reference should be Unref()'d once done using it.
	virtual Value* RefEval( value_type val_type );


	// Assigns a new value to the variable (LHS) corresponding
	// to this event expression, if appropriate.  The passed
	// value becomes the property of the Expr, which must
	// subsequently take care of garbage collecting it as necessary
	// (in particular, next time a value is Assign()'d, the value
	// should be deleted).
	//
	// Note that new_value can be nil (providing that index is nil,
	// too), in which case the old value
	// is deleted and the value set to nil.  Used for things like
	// formal parameters where it's desirable to free up the memory
	// used by their values as soon as the function call is complete,
	// rather than waiting for the next call to the function (and
	// subsequent assignment to the formal parameters).
	virtual void Assign( Value* new_value );

    protected:
	Value* CopyOrRefValue( const Value* value, bool modifiable );
	};


class VarExpr : public Expr {
    public:
	VarExpr( char* var_id, scope_type scope, int frame_offset,
			Sequencer* sequencer );

	~VarExpr();

	Value* Eval( bool modifiable );
	Value* RefEval( value_type val_type );

	void Assign( Value* new_value );

	const char* VarID()	{ return id; }

    protected:
	char* id;
	scope_type scope;
	int frame_offset;
	Sequencer* sequencer;
	};


class ConstExpr : public Expr {
    public:
	ConstExpr( const Value* const_value );

	Value* Eval( bool modifiable );
	void DescribeSelf( ostream& s ) const;

    protected:
	const Value* const_value;
	};


class UnaryExpr : public Expr {
    public:
	UnaryExpr( Expr* operand, const char* desc );

	Value* Eval( bool modifiable ) = 0;
	void Describe( ostream& s ) const;

    protected:
	Expr* op;
	};


class BinaryExpr : public Expr {
    public:
	BinaryExpr( Expr* op1, Expr* op2, const char* desc );

	Value* Eval( bool modifiable ) = 0;
	void Describe( ostream& s ) const;

    protected:
	Expr* left;
	Expr* right;
	};



class NegExpr : public UnaryExpr {
    public:
	NegExpr( Expr* operand );

	Value* Eval( bool modifiable );
	};


class NotExpr : public UnaryExpr {
    public:
	NotExpr( Expr* operand );

	Value* Eval( bool modifiable );
	};


class OrExpr : public BinaryExpr {
    public:
	OrExpr( Expr* op1, Expr* op2 );

	Value* Eval( bool modifiable );
	};


class AndExpr : public BinaryExpr {
    public:
	AndExpr( Expr* op1, Expr* op2 );

	Value* Eval( bool modifiable );
	};


class ConstructExpr : public Expr {
    public:
	ConstructExpr( ParameterPList* args );

	Value* Eval( bool modifiable );
	void Describe( ostream& s ) const;

    protected:
	Value* BuildArray();
	Value* BuildRecord();

	bool TypeCheck( const Value* values[], int num_values,
			glish_type& max_type );
	bool MaxNumeric( const Value* values[], int num_values,
				glish_type& max_type );
	bool AllEquivalent( const Value* values[], int num_values,
				glish_type& max_type );

	Value* ConstructArray( const Value* values[], int num_values,
				int total_length, glish_type max_type );

	bool is_array_constructor;
	ParameterPList* args;
	};


class ArrayRefExpr : public BinaryExpr {
    public:
	ArrayRefExpr( Expr* op1, Expr* op2 );

	Value* Eval( bool modifiable );
	Value* RefEval( value_type val_type );

	void Assign( Value* new_value );

	void Describe( ostream& s ) const;
	};


class RecordRefExpr : public UnaryExpr {
    public:
	RecordRefExpr( Expr* op, char* record_field );

	Value* Eval( bool modifiable );
	Value* RefEval( value_type val_type );

	void Assign( Value* new_value );

	void Describe( ostream& s ) const;

    protected:
	char* field;
	};


class RefExpr : public UnaryExpr {
    public:
	RefExpr( Expr* op, value_type type );

	Value* Eval( bool modifiable );
	void Assign( Value* new_value );

	void Describe( ostream& s ) const;

    protected:
	value_type type;
	};


class RangeExpr : public BinaryExpr {
    public:
	RangeExpr( Expr* op1, Expr* op2 );

	Value* Eval( bool modifiable );
	};


class CallExpr : public UnaryExpr {
    public:
	CallExpr( Expr* func, ParameterPList* args );

	Value* Eval( bool modifiable );
	void Describe( ostream& s ) const;

    protected:
	ParameterPList* args;
	};


typedef enum { EVENT_AGENT, EVENT_NAME, EVENT_VALUE } last_event_type;

class LastEventExpr : public Expr {
    public:
	LastEventExpr( Sequencer* sequencer, last_event_type type );

	Value* Eval( bool modifiable );
	Value* RefEval( value_type val_type );
	void Describe( ostream& s ) const;

    protected:
	Sequencer* sequencer;
	last_event_type type;
	};


extern void describe_expr_list( const expr_list* list, ostream& s );


#endif /* expr_h */
