// $Header: Client.cc,v 1.12 93/02/23 10:41:37 vern Locked $

#include <stdlib.h>
#include <osfcn.h>
#include <fcntl.h>
#include <ctype.h>
#include <string.h>
#include <errno.h>

#include <sys/types.h>

extern "C" {
#include <netinet/in.h>	// ditto
}

#include "Sds/sdsgen.h"
#include "Glish/Client.h"

#include "BuiltIn.h"
#include "Reporter.h"
#include "Socket.h"
#include "glish_event.h"
#include "system.h"

#ifdef SABER
#include <libc.h>
#endif


static Client* current_client;
static char* prog_name = "sequencer";


inline streq( const char* s1, const char* s2 )
	{
	return ! strcmp( s1, s2 );
	}


void Client_signal_handler()
	{
	current_client->HandlePing();
	}


class EventLink {
public:
	EventLink( char* task_id, char* new_name, int initial_activity )
		{
		id = task_id;
		name = new_name;
		fd = -1;
		active = initial_activity;
		}

	~EventLink()
		{
		delete id;
		delete name;
		}

	const char* ID() const		{ return id; }
	int FD() const			{ return fd; }
	int Active() const		{ return active; }
	const char* Name() const	{ return name; }

	void SetActive( int activity )	{ active = activity; }
	void SetFD( int new_fd )	{ fd = new_fd; }

private:
	char* id;
	char* name;
	int fd;
	int active;
	};


Client::Client( int& argc, char** argv ) : output_links( ORDERED )
	{
	char** orig_argv = argv;

	prog_name = argv[0];
	--argc, ++argv;	// remove program name from argument list

	sds_init();
	init_reporters();
	init_values();

	accept_socket = 0;

	bool suspend = false;

	if ( argc < 2 || ! streq( argv[0], "-id" ) )
		{
		// We weren't invoked by the sequencer.  Set things up
		// so that we receive "events" from stdin and send them
		// to stdout.
		have_sequencer_connection = false;
		client_name = prog_name;

		if ( argc >= 1 && streq( argv[0], "-noglish" ) )
			{
			// Create bogus IPC channels to associate with
			// read_fd and write_fd so that subsequent select()'s
			// will not show any activity.
			int pipe_fds[2];
			if ( pipe( pipe_fds ) < 0 )
				fatal->Report(
					"couldn't create pipe for -noglish" );

			read_fd = pipe_fds[0];

			if ( pipe( pipe_fds ) < 0 )
				fatal->Report(
					"couldn't create pipe for -noglish" );

			write_fd = pipe_fds[1];

			--argc, ++argv;
			}

		else
			{
			read_fd = fileno( stdin );
			write_fd = fileno( stdout );

			if ( argc >= 1 && streq( argv[0], "-glish" ) )
				// Eat up -glish argument.
				--argc, ++argv;
			}
		}

	else
		{
		have_sequencer_connection = true;
		client_name = argv[1];
		argc -= 2, argv += 2;

		if ( argc < 2 ||
		     (! streq( argv[0], "-host" ) &&
		      ! streq( argv[0], "-pipes" )) )
			fatal->Report(
		"invalid Client argument list - missing connection arguments" );

		if ( streq( argv[0], "-host" ) )
			{ // Socket connection.
			char* host = argv[1];
			argc -= 2, argv += 2;

			if ( argc < 2 || ! streq( argv[0], "-port" ) )
				fatal->Report(
			"invalid Client argument list - missing port number" );

			int port = atoi( argv[1] );
			argc -= 2, argv += 2;

			int socket = get_tcp_socket();

			if ( ! remote_connection( socket, host, port ) )
				fatal->Report(
			"could not establish Client connection to sequencer" );

			read_fd = write_fd = socket;
			}

		else
			{ // Pipe connection.
			++argv, --argc;	// skip -pipes argument

			if ( argc < 2 )
				fatal->Report(
			"invalid Client argument list - missing pipe fds" );

			read_fd = atoi( argv[0] );
			write_fd = atoi( argv[1] );

			argc -= 2, argv += 2;
			}

		if ( argc > 0 && streq( argv[0], "-suspend" ) )
			{
			suspend = true;
			--argc, ++argv;
			}
		}

	current_client = this;
	former_handler = (SigHandler)
		signal( SIGIO, SigHandler( Client_signal_handler ) );

	last_event = 0;

	if ( gethostname( local_host, sizeof( local_host ) ) < 0 )
		{
		fprintf( stderr, "%s: unknown local host" );
		strcpy( local_host, "<unknown>" );
		}

	if ( suspend )
		{
		int pid = int( getpid() );

		fprintf( stderr, "%s @ %s, pid %d: suspending ...\n",
			prog_name, local_host, pid );

		while ( suspend )
			sleep( 1 );	// allow debugger to attach
		}

	if ( have_sequencer_connection )
		{
		Value client_name_val( client_name );
		PostEvent( "established", &client_name_val );
		}

	for ( int i = 1; i <= argc; ++i )
		orig_argv[i] = *(argv++);

	orig_argv[i] = 0;
	argv = orig_argv;

	// Put argv[0] back into the argument count - it was removed
	// near the top of this routine.
	++argc;
	}

Client::~Client()
	{
	signal( SIGIO, former_handler );

	if ( have_sequencer_connection )
		PostEvent( "done", client_name );

	close( read_fd );
	close( write_fd );
	}

GlishEvent* Client::NextEvent()
	{
	if ( input_links.length() == 0 )
		// Only one input channel, okay to block reading it.
		return GetEvent( read_fd );

	fd_set input_fds;

	FD_ZERO( &input_fds );
	AddInputMask( &input_fds );

	while ( select( FD_SETSIZE, &input_fds, 0, 0, 0 ) < 0 )
		{
		if ( errno != EINTR )
			{
			fprintf( stderr, "%s: ", prog_name );
			perror( "error during select()" );
			return 0;
			}
		}

	return NextEvent( &input_fds );
	}

GlishEvent* Client::NextEvent( fd_set* mask )
	{
	if ( FD_ISSET( read_fd, mask ) )
		return GetEvent( read_fd );

	loop_over_list( input_links, i )
		{
		if ( FD_ISSET( input_links[i], mask ) )
			return GetEvent( input_links[i] );
		}

	PostEvent( "error", "bad call to Client::NextEvent" );

	return 0;
	}

void Client::Unrecognized()
	{
	if ( ! last_event )
		return;

	if ( last_event->name[0] == '*' )
		// Internal event - ignore.
		return;

	PostEvent( "unrecognized", last_event->name );
	}

void Client::PostEvent( const char* event_name, const Value* event_value )
	{
	if ( have_sequencer_connection )
		{
		event_link_list* l = output_links[event_name];

		if ( l )
			{
			int did_send = 0;

			loop_over_list( *l, i )
				{
				EventLink* el = (*l)[i];

				if ( el->Active() )
					{
					send_event( el->FD(), el->Name(),
							event_value );
					did_send = 1;
					}
				}

			// If we didn't send any events then it means that
			// all of the links are inactive.  Forward to
			// the sequencer instead.
			if ( ! did_send )
				send_event( write_fd, event_name, event_value );
			}
		else
			send_event( write_fd, event_name, event_value );
		}
	else
		message->Report( event_name, " ", event_value );
	}

void Client::PostEvent( GlishEvent* event )
	{
	PostEvent( event->name, event->value );
	}

void Client::PostEvent( const char* event_name, const char* event_value )
	{
	Value val( event_value );
	PostEvent( event_name, &val );
	}

void Client::PostEvent( const char* event_name, const char* event_fmt,
    const char* event_arg )
	{
	char buf[8192];
	sprintf( buf, event_fmt, event_arg );
	PostEvent( event_name, buf );
	}

void Client::AddInputMask( fd_set* mask )
	{
	FD_SET( read_fd, mask );

	// Now add in any fd's due to event links.
	loop_over_list( input_links, i )
		FD_SET( input_links[i], mask );
	}

int Client::HasClientInput( fd_set* mask )
	{
	if ( FD_ISSET( read_fd, mask ) )
		return 1;

	loop_over_list( input_links, i )
		{
		if ( FD_ISSET( input_links[i], mask ) )
			return 1;
		}

	return 0;
	}

GlishEvent* Client::GetEvent( int fd )
	{
	Unref( last_event );

	if ( have_sequencer_connection )
		last_event = recv_event( fd );
	else
		{
		char buf[512];
		if ( ! fgets( buf, sizeof( buf ), stdin ) )
			return 0;

		// Nuke final newline, if present.
		char* delim = strchr( buf, '\n' );

		if ( delim )
			*delim = '\0';

		// Find separation between event name and value.
		delim = strchr( buf, ' ' );
		Value* result;

		if ( ! delim )
			result = new Value( false );

		else
			{
			*delim++ = '\0';
			result = new Value( delim );
			}

		last_event = new GlishEvent( strdup( buf ), result );
		}

	if ( last_event )
		{
		const char* name = last_event->name;

		if ( streq( name, "terminate" ) )
			last_event = 0;

		else if ( name[0] == '*' )
			{
			if ( streq( name, "*pipe-source*" ) )
				OpenPipeSource( last_event->value );

			else if ( streq( name, "*pipe-sink*" ) )
				OpenPipeSink( last_event->value );

			else if ( streq( name, "*socket-source*" ) )
				OpenSocketSource( last_event->value );

			else if ( streq( name, "*socket-sink*" ) )
				OpenSocketSink( last_event->value );

			else if ( streq( name, "*unlink-sink*" ) )
				UnlinkSink( last_event->value );
			}
		}

	return last_event;
	}

void Client::FD_Change( int /* fd */, bool /* add_flag */ )
	{
	}

void Client::HandlePing()
	{
	}

void Client::OpenPipeSource( Value* v )
	{
	char* pipe_name;
	if ( ! v->FieldVal( "pipe", pipe_name ) )
		{
		PostEvent( "error", "bad internal link event" );
		return;
		}

	int pipe_fd = open( pipe_name, O_RDONLY );

	if ( pipe_fd < 0 )
		PostEvent( "error", "can't open pipe source %s", pipe_name );

	else
		{
		input_links.append( pipe_fd );

		// Now that we've opened both ends of the pipe we unlink
		// it from the file system to prevent it from hanging around
		// if we crash.
		if ( unlink( pipe_name ) < 0 )
			PostEvent( "error", "can't unlink pipe %s", pipe_name );
		}

	delete pipe_name;
	}

void Client::OpenPipeSink( Value* v )
	{
	int is_new;
	EventLink* el = AddOutputLink( v, 0, is_new );

	if ( ! el )
		return;

	el->SetActive( 1 );

	if ( ! is_new )
		return;

	char* pipe_name = make_named_pipe();

	if ( ! pipe_name )
		{
		// Fall back on sockets.
		OpenSocketSink( v );
		return;
		}

	// The following open() may block, so first post the response.
	v->SetField( "pipe", pipe_name );
	v->SetField( "event", "*pipe-source*" );
	PostEvent( "*forward*", v );

	int pipe_fd = open( pipe_name, O_WRONLY );

	if ( pipe_fd < 0 )
		PostEvent( "error", "can't open pipe sink %s", pipe_name );
	else
		el->SetFD( pipe_fd );
	}

void Client::OpenSocketSource( Value* v )
	{
	// Need to create a new connection.
	char* host;
	int port;

	if ( ! v->FieldVal( "host", host ) || ! v->FieldVal( "port", port ) )
		{
		PostEvent( "error", "bad internal link event" );
		return;
		}

	int socket = get_tcp_socket();

	if ( ! remote_connection( socket, host, port ) )
		PostEvent( "error", "can't open socket sink %s", host );

	else
		input_links.append( socket );

	delete host;
	}

void Client::OpenSocketSink( Value* v )
	{
	int is_new;
	EventLink* el = AddOutputLink( v, 0, is_new );

	if ( ! el )
		return;

	el->SetActive( 1 );

	if ( ! is_new )
		return;

	if ( ! accept_socket )
		accept_socket = new AcceptSocket();

	// Be sure to post the event *before* blocking on the socket
	// waiting for a connection.
	v->SetField( "host", local_host );
	v->SetField( "port", accept_socket->Port() );
	v->SetField( "event", "*socket-source*" );
	PostEvent( "*forward*", v );

	el->SetFD( accept_connection( accept_socket->FD() ) );
	}

void Client::UnlinkSink( Value* v )
	{
	int is_new;
	EventLink* el = AddOutputLink( v, 1, is_new );

	if ( ! el )
		return;

	if ( is_new )
		PostEvent( "error", "no link to %s", el->ID() );

	el->SetActive( 0 );
	}

EventLink* Client::AddOutputLink( Value* v, int want_active, int& is_new )
	{
	char* event_name;
	char* new_name;
	char* task_id;

	if ( ! v->FieldVal( "event", event_name ) ||
	     ! v->FieldVal( "new_name", new_name ) ||
	     ! v->FieldVal( "task_id", task_id ) )
		{
		PostEvent( "error", "bad internal link event" );
		return 0;
		}

	event_link_list* l = output_links[event_name];

	// Event link doesn't exist yet.
	if ( ! l )
		{
		l = new event_link_list;
		output_links.Insert( event_name, l );
		}

	else
		{ // Look for whether this event link already exists.
		loop_over_list( *l, i )
			{
			EventLink* el = (*l)[i];

			if ( el->Active() == want_active &&
			     streq( el->Name(), new_name ) &&
			     streq( el->ID(), task_id ) )
				{ // This one fits the bill.
				delete new_name;
				delete task_id;

				is_new = 0;

				return el;
				}
			}
		}

	EventLink* result = new EventLink( task_id, new_name, want_active );
	l->append( result );

	is_new = 1;

	return result;
	}

static int read_buffer( int fd, char* buffer, int buf_size )
	{
	int status = read( fd, buffer, buf_size );

	if ( status == 0 )
		return 0;

	else if ( status < 0 )
		{
		if ( errno != EPIPE && errno != ECONNRESET )
			error->Report( prog_name,
					": problem reading buffer, errno = ",
					errno );
		return 0;
		}

	else if ( status < buf_size )
		return read_buffer( fd, buffer + status, buf_size - status );

	return 1;
	}

static int write_buffer( int fd, char* buffer, int buf_size )
	{
	int status = write( fd, buffer, buf_size );

	if ( status < 0 )
		{
		if ( errno != EPIPE && errno != ECONNRESET )
			error->Report( prog_name,
					": problem writing buffer, errno = ",
					errno );
		return 0;
		}

	else if ( status < buf_size )
		return write_buffer( fd, buffer + status, buf_size - status );

	return 1;
	}


GlishEvent* recv_event( int fd )
	{
	struct event_header hdr;

	if ( ! read_buffer( fd, (char*) &hdr, sizeof( hdr ) ) )
		return 0;

	hdr.code = ntohl( hdr.code );
	int size = (int) ntohl( hdr.event_length );

	Value* result;

	if ( hdr.code == SDS_EVENT )
		{
		sds_cleanup();

		int sds = (int) sds_load_fd( fd, size, 8192 );

		if ( sds < 0 )
			{
			if ( sds != SDS_FILE_WR )
				error->Report( prog_name,
				": problem reading event, SDS error code = ",
						sds );
			return 0;
			}

		else
			result = read_value_from_SDS( sds );
		}

	else
		{
		char* value = new char[size+1];
		if ( size > 0 && ! read_buffer( fd, value, size ) )
			{
			error->Report( prog_name,
		": read only partial string event value from socket, errno = ",
					errno );
			delete value;
			return 0;
			}

		else
			{
			value[size] = '\0';
			result = split( value );
			}

		delete value;
		}

	return new GlishEvent( strdup( hdr.event_name ), result );
	}


static bool send_event_header( int fd, int code, int length, const char* name )
	{
	struct event_header hdr;

	hdr.code = htonl( (u_long) code );
	hdr.event_length = htonl( (u_long) length );

	if ( strlen( name ) >= MAX_EVENT_NAME_SIZE )
		warn->Report( prog_name,
				": event name \"", name, "\" truncated to ",
				MAX_EVENT_NAME_SIZE - 1, " characters" );

	strncpy( hdr.event_name, name, MAX_EVENT_NAME_SIZE - 1 );

	if ( ! write_buffer( fd, (char*) &hdr, sizeof( hdr ) ) )
		return false;

	return true;
	}

void send_event( int fd, const char* event_name, const Value* event_value )
	{
	event_value = event_value->Deref();

	if ( event_value->Type() == TYPE_STRING && event_value->Length() == 1 )
		{
		const_args_list a;
		a.append( event_value );
		char* string_val = paste( &a );

		int size = strlen( string_val );

		if ( ! send_event_header( fd, STRING_EVENT, size, event_name ) )
			return;

		if ( ! write_buffer( fd, string_val, size ) )
			return;
		}

	else
		{
		int sds = (int) sds_new_index( (char*) "" );

		if ( sds < 0 )
			error->Report( prog_name,
				": problem posting event, SDS error code = ",
					sds );

		del_list d;
		event_value->AddToSds( "", sds, &d );

		int size = (int) sds_dataset_size( sds );

		if ( send_event_header( fd, SDS_EVENT, size, event_name ) )
			{
			if ( write_sds2socket( fd, sds ) != sds &&
			     sds_error != SDS_FILE_WR )
				error->Report( prog_name,
			": problem sending event on socket, SDS error code = ",
						(int) sds_error );
			}

		sds_destroy( sds );
		sds_discard( sds );

		delete_list( &d );
		}
	}
