package Twilio::AvailablePhoneNumbers::Country;
use JSON;
use threads;
use threads::shared;
use Perlmazing;
use Twilio::Collection;
use Twilio::Codes;
use Twilio::Functions;
use parent 'Twilio::Collection';
our $current_limit = 30; # How much numbers Twilio shows when there are actually more numbers available

sub minimal {
	my $class = shift;
	bless {@_}, $class;
}

sub default_property {
	'country_code';
}

sub resource_url () {
	my $self = shift;
	my $name = ref $self;
	$name = substr $name, rindex($name, ':') + 1;
	$self->account_url."/AvailablePhoneNumbers/$self->{country_code}/Local.json";
}

sub list {
	my $self = shift;
	$self->use_instead(qw(local tollfree));
}

sub subresource_name {
	'available_phone_numbers';
}

sub subresource_module {
	'Twilio::AvailablePhoneNumbers::Country::AvailablePhoneNumber';
}

sub default_filter {
	'Contains';
}

sub new {
	my $class = shift;
	my $self = $class->SUPER::new(@_);
	$self->{subresource_uris} = {
		local		=> $self->account_uri."/AvailablePhoneNumbers/$self->{country_code}/Local.json",
		toll_free	=> $self->account_uri."/AvailablePhoneNumbers/$self->{country_code}/TollFree.json",
	};
	$self->classify_subresources;
	$self;
}

sub ResearchLocal {
	my $self = shift;
	if ($self->Twilio->use_threads) {
		$self->_research_threaded('Local', @_);
	} else {
		$self->_research('Local', @_);
	}
}

sub ResearchTollFree {
	my $self = shift;
	if ($self->Twilio->use_threads) {
		$self->_research_threaded('TollFree', @_);
	} else {
		$self->_research('TollFree', @_);
	}
}

sub _research {
	my $self = shift;
	my $type = shift;
	my @areas = @_;
	unless (@areas or $type eq 'TollFree') {
		@areas = sort numeric keys %{$country_codes->{$self->{country_code}}->{area_codes}};
	}
	my $iso_code = '+' . $country_codes->{$self->{country_code}}->{dialing_code};
	my @final;
	for my $area (@areas) {
		$self->trace("Looking for numbers starting with $area");
		my @results = $self->$type($iso_code.$area);
		if (@results >= $current_limit) {
			@results = $self->_deeper($type, $iso_code.$area);
		}
		next unless @results;
		push @final, @results;
	}
	sort {$a->phone_number cmp $b->phone_number } @final;
}

sub _research_threaded {
	my $self = shift;
	my $type = shift;
	my @areas = @_;
	unless (@areas or $type eq 'TollFree') {
		@areas = sort numeric keys %{$country_codes->{$self->{country_code}}->{area_codes}};
	}
	my $iso_code = '+' . $country_codes->{$self->{country_code}}->{dialing_code};
	my @final;
	for my $area (@areas) {
		$self->trace("Looking for numbers starting with $area");
		my @results = $self->$type($iso_code.$area);
		if (@results >= $current_limit) {
			$self->trace("I will research all numbers starting with $iso_code$area using ".$self->Twilio->use_threads." threads.");
			$self->trace("It can take a while!");
			my $minimal = Twilio::AvailablePhoneNumbers::Country->minimal (
				sid			=> $self->Twilio->sid,
				token		=> $self->Twilio->token,
				api_realm	=> $Twilio::api_realm,
				api_url		=> $Twilio::api_url,
				version		=> $self->Twilio->VERSION,
				url			=> $self->account_url."/AvailablePhoneNumbers/".$self->country_code,
				trace		=> $self->Twilio->{trace},
				threads		=> $self->Twilio->use_threads,
				trace_fh	=> $self->Twilio->get_trace_handler,
			);
			@results = $minimal->_deeper_threaded($type, $iso_code.$area);
			for my $r (@results) {
				$r = Twilio::AvailablePhoneNumbers::Country::AvailablePhoneNumber->new (
					Twilio	=> $self->Twilio,
					%$r,
				);
			}
		}
		next unless @results;
		push @final, @results;
	}
	while (1) {
		my $count = 0;
		for my $i (threads->list) {
			if ($i->is_running) {
				$count++;
			} else {
				eval {
					$i->join;
				}
			}
		}
		$self->trace("There are $count threads still running.");
		sleep .2;
		last unless $count;
	}
	sort {$a->phone_number cmp $b->phone_number } @final;
}

sub _deeper_threaded {
	my $self = shift;
	my $type = shift;
	my $filter = shift;
	$self->{started} = 0;
	$self->{pool} = [];
	$self->{results} = [];
	share $self->{started};
	share @{$self->{pool}};
	share @{$self->{results}};
	$self->{started} = 10;
	my $trace_fh = $self->{trace_fh};
	for my $i (0..9) {
		push @{$self->{pool}}, $filter.$i;
	}
	share $self->{started};
	for my $x (1..$self->{threads}) {
		threads->create (
			sub {
				sleep threads->tid / 20; # Helps yielding
				$self->{ua} = LWP::UserAgent->new (
					agent	=> "Twilio-Perl/$self->{version}",
					ssl_opts	=> {
						SSL_version		=> 'SSLv3',
						SSL_verify_mode	=> 0,
						verify_hostname	=> 0,
					}
				);
				$self->{ua}->credentials($self->{api_url}, $self->{api_realm}, $self->{sid}, $self->{token});
				while ($self->{started}) {
					my $q;
					{
						lock @{$self->{pool}};
						$q = shift @{$self->{pool}};
					}
					unless (defined $q) {
						sleep .2;
						next;
					}
					say $trace_fh "-: Researching $q" if $self->{trace};
					my $url = "$self->{url}/$type.json?Contains=".escape_uri(to_utf8 $q);
					my $tries = 0;
					CONNECT:
					$tries++;
					my $r = eval {
						$self->{ua}->get($url);
					};
					my $error = $@;
					if ($r and $r->is_success) {
						$r = from_json $r->decoded_content;
					} else {
						goto CONNECT unless $tries > 5;
						{
							lock $self->{started};
							$self->{started}--;
						}
						if ($error) {
							say "There was an error GETing $url: $error\n";
						} else {
							say "There was an error GETing $url:\nStatus: ".$r->status_line."\nBody:\n".$r->decoded_content;
						}
						threads->exit;
					}
					if (@{$r->{available_phone_numbers}} >= $current_limit) {
						{
							lock $self->{started};
							$self->{started} += 9;
						}
						{
							lock @{$self->{pool}};
							for my $i (0..9) {
								push @{$self->{pool}}, $q.$i;
							}
						}
						next;
					} else {
						{
							lock $self->{started};
							$self->{started}--;
						}
						{
							lock @{$self->{results}};
							for my $i (@{$r->{available_phone_numbers}}) {
								push @{$self->{results}}, shared_clone($i);
							}
						}
					}
				}
				threads->exit;
			},
		);
	}
	while (1) {
		my $count = 0;
		for my $i (threads->list) {
			$count++ if $i->is_running;
		}
		last unless $count;
		sleep .2;
	}
	@{$self->{results}};
}

sub _deeper {
	my $self = shift;
	my $type = shift;
	my $filter = shift;
	my @final;
	for my $i (0..9) {
		my $q = $filter.$i;
		$self->trace("Researching deeper into numbers starting with $q");
		my @results = $self->$type($q);
		if (@results >= $current_limit) {
			@results = $self->_deeper($type, $q);
		}
		next unless @results;
		push @final, @results;
	}
	@final;
}

1;
