#!perl

# t/manifest-write.t #

#   - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
#
#   Copyright © 2015 Van de Bugger
#
#   This file is part of perl-Dist-Zilla-Plugin-Manifest-Write.
#
#   perl-Dist-Zilla-Plugin-Manifest-Write is free software: you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by the Free Software
#   Foundation, either version 3 of the License, or (at your option) any later version.
#
#   perl-Dist-Zilla-Plugin-Manifest-Write is distributed in the hope that it will be useful, but
#   WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
#   PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License along with
#   perl-Dist-Zilla-Plugin-Manifest-Write. If not, see <http://www.gnu.org/licenses/>.
#
#   - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

use strict;
use warnings;
use autodie ':all';

use File::Temp;
use Path::Class;
use Test::DZil;
use Test::Fatal;
use Test::More;

my $version = qr{\d+\.\d\d\d(?:_\d\d\d)?};

use Data::Printer;

my $empty_dir = File::Temp::tempdir( CLEANUP => 1 );
my $dummy_pm  =
    "package Dummy;\n" .
    "# ABSTRACT: Dummy\n" .
    "# VERSION\n" .
    "1;\n";
my $ini_head  = {
    name                => 'Dummy',
    version             => '0.001',
    author              => 'John Doe',
    license             => 'GPL_3::or_later',
    copyright_holder    => 'John Doe',
    copyright_year      => '2007',
};


sub test {
    my ( %args ) = @_;
    subtest $args{ name } => sub {
        plan tests => 3;
        my $tzil = Builder->from_config(
            { dist_root => $empty_dir },
            {
                add_files => {
                    'source/lib/Dummy.pm' => $dummy_pm,
                    'source/dist.ini'     => dist_ini( $ini_head, @{ $args{ dist_ini } } ),
                },
            },
        );
        is( exception { $tzil->build }, undef, 'build must be successful' );
        my @manifest =  $tzil->built_in->file( 'MANIFEST' )->
            slurp( iomode => '<:encoding(UTF-8)', chomp => 1 );
        my $head = shift( @manifest );
        like(
            $head,
            qr{\A# This file was generated with Dist::Zilla::Plugin::Manifest::Write $version\z},
            'manifest head'
        );
        is_deeply( \@manifest, $args{ manifest }, 'manifest body' );
        done_testing();
    };
};

plan tests => 3;

#   No source providers are specified, files will be recognized as "3rd party".
test(
    name => 'no source providers',
    dist_ini => [
        'GatherDir',
        [ 'Manifest::Write' ],
        'MetaYAML',
    ],
    manifest => [
        'MANIFEST     #  metainfo file generated by Manifest::Write',
        'META.yml     #  metainfo file generated by MetaYAML',
        'dist.ini     # 3rd party file added by GatherDir',
        'lib/Dummy.pm # 3rd party file added by GatherDir',
    ],
);

#   Source provider is specified, files will should be recognized as "Dummy" (it is name of
#   distibution).
test(
    name => 'source provider specified',
    dist_ini => [
        'GatherDir',
        [ 'Manifest::Write', { source_provider => 'GatherDir' } ],
        'MetaYAML',
    ],
    manifest =>
    [
        'MANIFEST     #  metainfo file generated by Manifest::Write',
        'META.yml     #  metainfo file generated by MetaYAML',
        'dist.ini     #     Dummy file added by GatherDir',
        'lib/Dummy.pm #     Dummy file added by GatherDir',
    ],
);

#   Adding `OurPkgVersion` — it modifies Perl module.
test(
    name => 'adding munger',
    dist_ini => [
        'GatherDir',
        [ 'Manifest::Write', { source_provider => 'GatherDir' } ],
        'OurPkgVersion',
        'MetaYAML',
    ],
    manifest =>
    [
        'MANIFEST     #  metainfo file generated by Manifest::Write',
        'META.yml     #  metainfo file generated by MetaYAML',
        'dist.ini     #     Dummy file added by GatherDir',
        'lib/Dummy.pm #     Dummy file added by GatherDir and modified by OurPkgVersion',
    ],
);

done_testing();

exit( 0 );

# end of file #
