#-----------------------------------------------------------------------

=head1 NAME

Date::Ordinal - Conversion of dates to ordinal numbers and vice versa

=head1 SYNOPSIS

    use Date::Ordinal;
    
    $ord     = month2ord('January');           # $ord gets 1
    $mon     = ord2month('1');                 # $mon gets 'January'
    $mon     = ord2month(1);                   # $mon gets 'January'
    $mon     = ord2month('01');                # $mon gets 'January'

    
=cut

#-----------------------------------------------------------------------

    package Date::Ordinal;
#use strict;

#-----------------------------------------------------------------------

=head1 DESCRIPTION

The C<Locale::US> module provides access to the two-letter
codes for identifying states in the United States. You can either
access the codes via the L<conversion routines> (described below),
or with the two functions which return lists of all states codes or
all state names.

=cut

#-----------------------------------------------------------------------

require Exporter;
use Carp;

#-----------------------------------------------------------------------
#	Public Global Variables
#-----------------------------------------------------------------------
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);
$VERSION   = '1.00';
@ISA       = qw(Exporter);
@EXPORT    = qw(ord2month month2ord);

#-----------------------------------------------------------------------
#	Non-Private Global Variables
#-----------------------------------------------------------------------
@month=();

#=======================================================================

=head1 CONVERSION ROUTINES

There are two conversion routines: C<ord2month()> and C<month2ord()>.

=over 8

=item ord2month()

This function takes a month number [1..12] and returns a string
which contains the name of the month identified. If the number is
not a valid number, then C<undef> will be returned:

    $mon = ord2month('3');

=item month2ord()

This function takes a month name and returns the integer
corresponding to the month name, if such exists.
If the argument could not be identified as a month name,
then C<undef> will be returned:

    $ord = month2ord('March');

The case of the month name is not important.
See the section L<KNOWN BUGS AND LIMITATIONS> below.

=back

=cut

#=======================================================================
sub ord2month
{
    my $ord = shift;

    return undef unless (($ord >=1) && ($ord <=12));

    my $month_count;
    
    while (1+$month_count++ != $ord) { ; }

    return $month[$month_count-1];
}

sub month2ord
{
    my $month = shift;

    $state = "\u\L$state";

    my $month_count;

    for ($month_count=0; $month_count < 12; ++$month_count) {
	return ++$month_count if ($month eq $month[$month_count]);
    }

    return undef;
}

#=======================================================================

=head1 QUERY ROUTINES

There is one function (and a reference variant) which can be used to
obtain a list of all month names:


=over 8

=item C<all_month_names()>

    Returns a list of all month names;

=item C<all_month_names_ref()>

    Returns a reference to a list of all month names;

=back

=cut

#=======================================================================
sub all_month_names
{
    return @month;
}

sub all_month_names_ref
{
    return \@month;
}



#-----------------------------------------------------------------------


=head1 KNOWN BUGS AND LIMITATIONS

none

=head1 SEE ALSO

=over 4

=item Locale::US 


=item Date::Manip

=back

=head1 AUTHOR

Terrence Brannon E<lt>tbrannon@end70.comE<gt>

=head1 COPYRIGHT

Copyright (c) 1999 End70 Corporation

This module is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=cut

#-----------------------------------------------------------------------

#=======================================================================
# initialisation code - stuff the DATA into the CODES hash
#=======================================================================

@month=qw(
	  January February March April May June July
	  August September October November December
	  );


1;

