package CPAN::Testers::Schema;
our $VERSION = '0.004';
# ABSTRACT: Schema for CPANTesters database processed from test reports

#pod =head1 SYNOPSIS
#pod
#pod     my $schema = CPAN::Testers::Schema->connect( $dsn, $user, $pass );
#pod     my $rs = $schema->resultset( 'Stats' )->search( { dist => 'Test-Simple' } );
#pod     for my $row ( $rs->all ) {
#pod         if ( $row->state eq 'fail' ) {
#pod             say sprintf "Fail report from %s: http://cpantesters.org/cpan/report/%s",
#pod                 $row->tester, $row->guid;
#pod         }
#pod     }
#pod
#pod =head1 DESCRIPTION
#pod
#pod This is a L<DBIx::Class> Schema for the CPANTesters statistics database.
#pod This database is generated by processing the incoming data from L<the
#pod CPANTesters Metabase|http://metabase.cpantesters.org>, and extracting
#pod the useful fields like distribution, version, platform, and others (see
#pod L<CPAN::Testers::Schema::Result::Stats> for a full list).
#pod
#pod This is its own distribution so that it can be shared by the backend
#pod processing, data APIs, and the frontend web application.
#pod
#pod =head1 SEE ALSO
#pod
#pod =over 4
#pod
#pod =item L<CPAN::Testers::Schema::Result::Stats>
#pod
#pod =item L<DBIx::Class>
#pod
#pod =item L<http://github.com/cpan-testers/cpantesters-project>
#pod
#pod For an overview of how the CPANTesters project works, and for information about
#pod project goals and to get involved.
#pod
#pod =back
#pod
#pod =cut

use CPAN::Testers::Schema::Base;
use base 'DBIx::Class::Schema';

__PACKAGE__->load_namespaces;

#pod =method connect_from_config
#pod
#pod     my $schema = CPAN::Testers::Schema->connect_from_config;
#pod
#pod Connect to the MySQL database using a local MySQL configuration file
#pod in C<$HOME/.cpanstats.cnf>. This configuration file should look like:
#pod
#pod     [client]
#pod     host     = ""
#pod     database = cpanstats
#pod     user     = my_usr
#pod     password = my_pwd
#pod
#pod See L<DBD::mysql/mysql_read_default_file>.
#pod
#pod =cut

# Convenience connect method
sub connect_from_config ( $class ) {
    my $schema = $class->connect(
        "DBI:mysql:mysql_read_default_file=$ENV{HOME}/.cpanstats.cnf;".
        "mysql_read_default_group=application;mysql_enable_utf8=1",
        undef,  # user
        undef,  # pass
        {
            AutoCommit => 1,
            RaiseError => 1,
            mysql_enable_utf8 => 1,
            quote_char => '`',
            name_sep   => '.',
        },
    );
    return $schema;
}

1;

__END__

=pod

=head1 NAME

CPAN::Testers::Schema - Schema for CPANTesters database processed from test reports

=head1 VERSION

version 0.004

=head1 SYNOPSIS

    my $schema = CPAN::Testers::Schema->connect( $dsn, $user, $pass );
    my $rs = $schema->resultset( 'Stats' )->search( { dist => 'Test-Simple' } );
    for my $row ( $rs->all ) {
        if ( $row->state eq 'fail' ) {
            say sprintf "Fail report from %s: http://cpantesters.org/cpan/report/%s",
                $row->tester, $row->guid;
        }
    }

=head1 DESCRIPTION

This is a L<DBIx::Class> Schema for the CPANTesters statistics database.
This database is generated by processing the incoming data from L<the
CPANTesters Metabase|http://metabase.cpantesters.org>, and extracting
the useful fields like distribution, version, platform, and others (see
L<CPAN::Testers::Schema::Result::Stats> for a full list).

This is its own distribution so that it can be shared by the backend
processing, data APIs, and the frontend web application.

=head1 METHODS

=head2 connect_from_config

    my $schema = CPAN::Testers::Schema->connect_from_config;

Connect to the MySQL database using a local MySQL configuration file
in C<$HOME/.cpanstats.cnf>. This configuration file should look like:

    [client]
    host     = ""
    database = cpanstats
    user     = my_usr
    password = my_pwd

See L<DBD::mysql/mysql_read_default_file>.

=head1 SEE ALSO

=over 4

=item L<CPAN::Testers::Schema::Result::Stats>

=item L<DBIx::Class>

=item L<http://github.com/cpan-testers/cpantesters-project>

For an overview of how the CPANTesters project works, and for information about
project goals and to get involved.

=back

=head1 AUTHORS

=over 4

=item *

Oriol Soriano <oriolsoriano@gmail.com>

=item *

Doug Bell <preaction@cpan.org>

=back

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by Oriol Soriano, Doug Bell.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
