#!/usr/bin/perl

use strict;
use warnings;

use Test::More tests => 10;

use FindBin;
require "$FindBin::Bin/../util.pl";

my $test = Net::DRI::Test->new_epp(['LoginSecurity']);
my $dri = $test->dri();

####################################################################################################

my $rc;
my $expected;

# This is just needed to fill out internal structure
$test->set_response(<<'EOF');
<?xml version="1.0" encoding="UTF-8" standalone="no"?>
<epp xmlns="urn:ietf:params:xml:ns:epp-1.0">
  <greeting>
    <svID>Example EPP server epp.example.com</svID>
    <svDate>2000-06-08T22:00:00.0Z</svDate>
    <svcMenu>
      <version>1.0</version>
      <lang>en</lang>
      <lang>fr</lang>
      <objURI>urn:ietf:params:xml:ns:obj1</objURI>
      <objURI>urn:ietf:params:xml:ns:obj2</objURI>
      <objURI>urn:ietf:params:xml:ns:obj3</objURI>
      <svcExtension>
        <extURI>urn:ietf:params:xml:ns:epp:loginSec-1.0</extURI>
      </svcExtension>
    </svcMenu>
    <dcp>
      <access><all/></access>
      <statement>
        <purpose><admin/><prov/></purpose>
        <recipient><ours/><public/></recipient>
        <retention><stated/></retention>
      </statement>
    </dcp>
  </greeting>
</epp>
EOF
$rc=$dri->process('session','connect');

$test->set_response();
$rc=$dri->process('session','login',['ClientX','this is a long password', {}]); # last (empty) ref hash is guaranteed from call in Transport/Socket
$expected=$test->format_xml('
  <command>
    <login>
      <clID>ClientX</clID>
      <pw>\[LOGIN-SECURITY\]</pw>
      <options>
        <version>1.0</version>
        <lang>en</lang>
      </options>
      <svcs>
        <objURI>urn:ietf:params:xml:ns:obj1</objURI>
        <objURI>urn:ietf:params:xml:ns:obj2</objURI>
        <objURI>urn:ietf:params:xml:ns:obj3</objURI>
        <svcExtension>
          <extURI>urn:ietf:params:xml:ns:epp:loginSec-1.0</extURI>
        </svcExtension>
      </svcs>
    </login>
    <extension>
      <loginSec:loginSec xmlns:loginSec="urn:ietf:params:xml:ns:epp:loginSec-1.0">
        <loginSec:userAgent>
          <loginSec:app>Net::DRI/\S+ \(EPP/1.0\)</loginSec:app>
          <loginSec:tech>perl/\S+ XML::LibXML/\S+</loginSec:tech>
          <loginSec:os>\S+</loginSec:os>
        </loginSec:userAgent>
        <loginSec:pw>this is a long password</loginSec:pw>
      </loginSec:loginSec>
    </extension>
    <clTRID>ABC-12345</clTRID>
  </command>');
like($test->get_command(), qr/$expected/, 'login with long password');

eval {
    $rc=$dri->process('session','login',['ClientX','[LOGIN-SECURITY]', {}]); # last (empty) ref hash is guaranteed from call in Transport/Socket
};
like($@, qr/Invalid parameters: password can not be "\[LOGIN-SECURITY\]"/, 'use of trigger password is forbidden');


$rc=$dri->process('session','login',['ClientX','this is a long password', {client_newpassword=>'new password that is still long'}]);
$expected=$test->format_xml('
  <command>
    <login>
      <clID>ClientX</clID>
      <pw>\[LOGIN-SECURITY\]</pw>
      <newPW>\[LOGIN-SECURITY\]</newPW>
      <options>
        <version>1.0</version>
        <lang>en</lang>
      </options>
      <svcs>
        <objURI>urn:ietf:params:xml:ns:obj1</objURI>
        <objURI>urn:ietf:params:xml:ns:obj2</objURI>
        <objURI>urn:ietf:params:xml:ns:obj3</objURI>
        <svcExtension>
          <extURI>urn:ietf:params:xml:ns:epp:loginSec-1.0</extURI>
        </svcExtension>
      </svcs>
    </login>
    <extension>
      <loginSec:loginSec xmlns:loginSec="urn:ietf:params:xml:ns:epp:loginSec-1.0">
        <loginSec:userAgent>
          <loginSec:app>Net::DRI/\S+ \(EPP/1.0\)</loginSec:app>
          <loginSec:tech>perl/\S+ XML::LibXML/\S+</loginSec:tech>
          <loginSec:os>\S+</loginSec:os>
        </loginSec:userAgent>
        <loginSec:pw>this is a long password</loginSec:pw>
        <loginSec:newPW>new password that is still long</loginSec:newPW>
      </loginSec:loginSec>
    </extension>
    <clTRID>ABC-12345</clTRID>
  </command>');
like($test->get_command(), qr/$expected/, 'login with long password and change of password');


$rc=$dri->process('session','login',['ClientX','shortpassword', {client_newpassword=>'new password that is still long'}]);
$expected=$test->format_xml('
  <command>
    <login>
      <clID>ClientX</clID>
      <pw>shortpassword</pw>
      <newPW>\[LOGIN-SECURITY\]</newPW>
      <options>
        <version>1.0</version>
        <lang>en</lang>
      </options>
      <svcs>
        <objURI>urn:ietf:params:xml:ns:obj1</objURI>
        <objURI>urn:ietf:params:xml:ns:obj2</objURI>
        <objURI>urn:ietf:params:xml:ns:obj3</objURI>
        <svcExtension>
          <extURI>urn:ietf:params:xml:ns:epp:loginSec-1.0</extURI>
        </svcExtension>
      </svcs>
    </login>
    <extension>
      <loginSec:loginSec xmlns:loginSec="urn:ietf:params:xml:ns:epp:loginSec-1.0">
        <loginSec:userAgent>
          <loginSec:app>Net::DRI/\S+ \(EPP/1.0\)</loginSec:app>
          <loginSec:tech>perl/\S+ XML::LibXML/\S+</loginSec:tech>
          <loginSec:os>\S+</loginSec:os>
        </loginSec:userAgent>
        <loginSec:newPW>new password that is still long</loginSec:newPW>
      </loginSec:loginSec>
    </extension>
    <clTRID>ABC-12345</clTRID>
  </command>');
like($test->get_command(), qr/$expected/, 'login with short password and change of password');

####################################################################################################

$test->set_response(<<'EOF');
<?xml version="1.0" encoding="UTF-8" standalone="no"?>
<epp xmlns="urn:ietf:params:xml:ns:epp-1.0">
  <response>
    <result code="1000">
      <msg>Command completed successfully</msg>
    </result>
    <extension>
      <loginSec:loginSecData
        xmlns:loginSec=
          "urn:ietf:params:xml:ns:epp:loginSec-1.0">
        <loginSec:event
          type="password"
          level="warning"
          exDate="2018-04-01T22:00:00.0Z"
          lang="en">
          Password expiring in a week
        </loginSec:event>
      </loginSec:loginSecData>
    </extension>
    <trID>
      <clTRID>ABC-12345</clTRID>
      <svTRID>54321-XYZ</svTRID>
    </trID>
  </response>
</epp>
EOF
$rc=$dri->process('session','login', ['ClientX', 'PasswordY', {}]);
my @events1w = (
    {
        'lang' => 'en',
        'type' => 'password',
        'description' => 'Password expiring in a week',
        'exDate' => '2018-04-01T22:00:00.0Z'
    }
);


is_deeply($rc->get_data('session', 'login_security', 'warning'), \@events1w, 'parsing login_security event 1 warning');
is_deeply($rc->get_data('session', 'login_security', 'error'), [], 'parsing login_security event 1 error');


$test->set_response(<<'EOF');
<?xml version="1.0" encoding="UTF-8" standalone="no"?>
<epp xmlns="urn:ietf:params:xml:ns:epp-1.0">
  <response>
    <result code="2200">
      <msg>Authentication error</msg>
    </result>
    <extension>
      <loginSec:loginSecData
        xmlns:loginSec=
          "urn:ietf:params:xml:ns:epp:loginSec-1.0">
        <loginSec:event
          type="password"
          level="error"
          exDate="2018-03-26T22:00:00.0Z">
          Password has expired
        </loginSec:event>
        <loginSec:event
          type="newPW"
          level="error">
          New password does not meet complexity requirements
        </loginSec:event>
      </loginSec:loginSecData>
    </extension>
    <trID>
      <clTRID>ABC-12345</clTRID>
      <svTRID>54321-XYZ</svTRID>
    </trID>
  </response>
</epp>
EOF
$rc=$dri->process('session','login', ['ClientX', 'PasswordY', {}]);

my @events2e = (
    {
        'lang' => 'en',
        'type' => 'password',
        'description' => 'Password has expired',
        'exDate' => '2018-03-26T22:00:00.0Z'
    },
    {
        'lang' => 'en',
        'type' => 'newPW',
        'description' => 'New password does not meet complexity requirements',
    },
);

is_deeply($rc->get_data('session', 'login_security', 'warning'), [], 'parsing login_security event 2 warning');
is_deeply($rc->get_data('session', 'login_security', 'error'), \@events2e, 'parsing login_security event 2 error');


$test->set_response(<<'EOF');
<?xml version="1.0" encoding="UTF-8" standalone="no"?>
<epp xmlns="urn:ietf:params:xml:ns:epp-1.0">
  <response>
    <result code="1000">
      <msg>Command completed successfully</msg>
    </result>
    <extension>
      <loginSec:loginSecData
        xmlns:loginSec=
          "urn:ietf:params:xml:ns:epp:loginSec-1.0">
        <loginSec:event
          type="password"
          level="warning"
          exDate="2018-04-01T22:00:00.0Z"
          lang="en">
          Password expiration soon
        </loginSec:event>
        <loginSec:event
          type="certificate"
          level="warning"
          exDate="2018-04-02T22:00:00.0Z"/>
        <loginSec:event
          type="cipher"
          level="warning"
          value="TLS_RSA_WITH_AES_128_CBC_SHA">
          Non-PFS Cipher negotiated
        </loginSec:event>
        <loginSec:event
          type="tlsProtocol"
          level="warning"
          value="TLSv1.0">
          Insecure TLS protocol negotiated
        </loginSec:event>
        <loginSec:event
          type="stat"
          name="failedLogins"
          level="warning"
          value="100"
          duration="P1D">
          Excessive invalid daily logins
        </loginSec:event>
        <loginSec:event
          type="custom"
          name="myCustomEvent"
          level="warning">
          A custom login security event occured
        </loginSec:event>
      </loginSec:loginSecData>
    </extension>
    <trID>
      <clTRID>ABC-12345</clTRID>
      <svTRID>54321-XYZ</svTRID>
    </trID>
  </response>
</epp>
EOF
$rc=$dri->process('session','login', ['ClientX', 'PasswordY', {}]);

my @events3w = (
    {
        'lang' => 'en',
        'type' => 'password',
        'description' => 'Password expiration soon',
        'exDate' => '2018-04-01T22:00:00.0Z'
    },
    {
        'lang' => 'en',
        'type' => 'certificate',
        'exDate' => '2018-04-02T22:00:00.0Z'
    },
    {
        'lang' => 'en',
        'type' => 'cipher',
        'value' => 'TLS_RSA_WITH_AES_128_CBC_SHA',
        'description' => 'Non-PFS Cipher negotiated',
    },
    {
        'lang' => 'en',
        'type' => 'tlsProtocol',
        'value' => 'TLSv1.0',
        'description' => 'Insecure TLS protocol negotiated',
    },
    {
        'lang' => 'en',
        'type' => 'stat',
        'subtype' => 'failedLogins',
        'value' => '100',
        'duration' => 'P1D',
        'description' => 'Excessive invalid daily logins',
    },
    {
        'lang' => 'en',
        'type' => 'custom',
        'subtype' => 'myCustomEvent',
        'description' => 'A custom login security event occured',
    },
);

is_deeply($rc->get_data('session', 'login_security', 'warning'), \@events3w, 'parsing login_security event 3 warning');
is_deeply($rc->get_data('session', 'login_security', 'error'), [], 'parsing login_security event 3 error');

exit 0;
