# @(#)$Id: Root.pm 1116 2012-03-11 23:05:42Z pjf $

package CatalystX::Usul::Controller::Root;

use strict;
use warnings;
use version; our $VERSION = qv( sprintf '0.4.%d', q$Rev: 1116 $ =~ /\d+/gmx );
use parent qw(CatalystX::Usul::Controller);

use CatalystX::Usul::Constants;
use CatalystX::Usul::Functions qw(env_prefix is_member);
use MRO::Compat;

__PACKAGE__->config( imager_class => q(Imager), );

__PACKAGE__->mk_accessors( qw(default_namespace imager_class) );

sub auto : Private {
   return shift->next::method( @_ );
}

sub begin : Private {
   return shift->next::method( @_ );
}

sub end : Private {
   return shift->next::method( @_ );
}

sub render : ActionClass(RenderView) {
}

sub about : Chained(/) Args(0) Public {
   # Display license and authorship information
   my ($self, $c) = @_; $self->set_popup( $c, q(close) );

   return $c->model( $self->help_class )->form;
}

sub access_denied : Chained(/) Args Public {
   # The auto method has decided not to allow access to the requested room
   my ($self, $c, $ns, @rest) = @_;

   my $s = $c->stash; my $name = join SEP, @rest;

   unless ($s->{denied_namespace} = $ns and $s->{denied_action} = $name) {
      my $msg = 'Action namespace and/or action name not specified';

      return $self->error_page( $c, $msg );
   }

   $self->add_header( $c ); $self->reset_nav_menu( $c, q(back) );

   $c->action->name( q(cracker) );

   my $msg = 'Access denied to [_1] for [_2]';

   $self->log_warn( $self->loc( $s, $msg, $ns.SEP.$name, $s->{user} ) );
   $c->res->status( 403 );
   return FALSE;
}

sub action_closed : Chained(/) Args Public {
   # Requested page exists but is temporarily unavailable
   my ($self, $c, $ns, @rest) = @_;

   my $s = $c->stash; my $name = join SEP, @rest;

   unless ($s->{closed_namespace} = $ns and $s->{closed_action} = $name) {
      my $msg = 'Action namespace and/or action name not specified';

      return $self->error_page( $c, $msg );
   }

   $self->add_header( $c ); $self->reset_nav_menu( $c, q(back) );

   $self->log_warn( $self->loc( $s, 'Action [_1]/[_2] closed', $ns, $name ) );
   $c->res->status( 423 );
   return;
}

sub app_closed : Chained(/) Args HasActions {
   # Application has been closed by the administrators
   my ($self, $c) = @_; $self->add_header( $c );

   $self->reset_nav_menu( $c, q(blank) )->form;

   return FALSE;
}

sub app_reopen : ActionFor(app_closed.login) {
   # Open the application to users
   my ($self, $c) = @_; my $s = $c->stash; my $cfg = $c->config;

   $self->set_identity_model( $c ); $s->{user_model}->authenticate;

   $self->can( q(persist_state) )
      and $self->set_uri_query_params( $c, { realm => $s->{realm} } );

   if (is_member $cfg->{admin_role}, $c->user->roles) {
      $c->model( $self->global_class )->save;
      $self->redirect_to_path( $c, $s->{wanted} );
   }

   return TRUE;
}

sub captcha : Chained(/) Args(0) NoToken Public {
   # Dynamically generate a jpeg image displaying a random number
   my ($self, $c) = @_;

   return $c->model( $self->realm_class )->users->create_captcha;
}

sub company : Chained(/) Args(0) Public {
   # And now a short message from our sponsor
   my ($self, $c) = @_; return $self->set_popup( $c, q(close) );
}

sub feedback : Chained(/) Args HasActions {
   # Form to send an email to the site administrators
   my ($self, $c, @rest) = @_;

   $c->model( $self->help_class )->form( @rest );

   return $self->set_popup( $c, q(close) );
}

sub feedback_send : ActionFor(feedback.send) {
   # Send an email to the site administrators
   my ($self, $c) = @_; return $c->model( $self->help_class )->feedback_send;
}

sub help : Chained(/) Args Public {
   return shift->next::method( @_ );
}

sub imager : Chained(/) Args NoToken Public {
   my ($self, $c, @args) = @_; my $model = $c->model( $self->imager_class );

   my ($data, $type, $mtime)
      = $model->transform( [ @args ], $c->req->query_parameters );

   $data or return $self->error_page( $c, 'No body data generated' );

   $c->res->body( $data );
   $c->res->content_type( $type );
   $mtime and $c->res->headers->last_modified( $mtime );
# TODO: Work out what to do with expires header
#    $c->res->headers->expires( time() );
   return;
}

sub quit : Chained(/) Args(0) Public {
   my ($self, $c) = @_; my $s = $c->stash; my $cfg = $c->config;

   $ENV{ (env_prefix $cfg->{name}).q(_QUIT_OK) } and exit 0;

   $self->log_warn( $self->loc( $s, 'Quit attempted by [_1]', $s->{user} ) );

   return $self->redirect_to_path( $c, $self->default_namespace );
}

sub redirect_to_default : Chained(/) PathPart('') Args {
   my ($self, $c) = @_;

   return $self->redirect_to_path( $c, $self->default_namespace );
}

sub select_language : Chained(/) Args(0) Public {
   my ($self, $c) = @_; my $params = $c->req->params;

   if ($c->can( q(session) )) {
      $c->session( language => $params->{select_language} );
   }

   return $self->redirect_to_path( $c, $params->{referer} );
}

sub version {
   return $VERSION;
}

sub view_source : Chained(/) Args Public {
   # Display the source code with syntax highlighting
   my ($self, $c, $module) = @_;

   $module or return $self->error_page( $c, 'Module not specified' );

   $self->add_header( $c ); $self->reset_nav_menu( $c, q(close) );

   $c->model( $self->fs_class )->view_file( q(source), $module );
   return;
}

1;

__END__

=pod

=head1 Name

CatalystX::Usul::Controller::Root - Root Controller for the application

=head1 Version

0.1$Revision: 1116 $

=head1 Synopsis

   package MyApp::Controller::Root;

   use base qw(CatalystX::Usul::Controller::Root);

=head1 Description

Exposes some generic endpoints implemented in the base class

=head1 Subroutines/Methods

=head2 about

Display a simple popop window containing the copyright and license information

=head2 access_denied

The auto method redirects unauthorised users to this endpoint

=head2 action_closed

The requested endpoint exists but has been deactivated by the
administrators

=head2 app_closed

Display an information screen explaining that the application has been
closed by the administrators. An administrator must authenticate to
reopen the application

=head2 app_reopen

Action to reopen the application. Does this by setting the global
configuration attribute to false

=head2 auto

Calls method of same name in parent class

=head2 begin

Calls method of same name in parent class to stuff the stash with data
used by all pages

=head2 captcha

Dynamically generates a JPEG image used on the self registration
screen to defeat bots.

=head2 company

Some text about the company whose application this is.

=head2 end

Attempt to render a view

=head2 feedback

Display a popup window that lets the user send an email to the site
administrators

=head2 feedback_send

This private method is the action for the feedback controller

=head2 help

Calls method of same name in parent class

=head2 imager

Generates transformations of any image under the document root. Calls
L<transform|CatalystX::Usul::Model::Iamger/transform> and sets the
response object directly

=head2 quit

Called when collecting profile data using L<Devel::NYTProf> to stop
the server. The environment variable I<MYAPP_QUIT_OK> must be set to
true for this to work, so don't do that on a production server

=head2 redirect_to_default

Redirects to default controller. Matches any uri not matched by another action

=head2 render

Use the renderview action class

=head2 select_language

Handles the post request to select the language used. Stores the requested
language in the session and the redirects back to the original uri

=head2 version

Return the version number of this controller

=head2 view_source

View the source code for the current controller. Calls the
C<view_file> method in the L<CatalystX::Usul::Model::FileSystem> model
to display some source code with syntax highlighting

=head1 Diagnostics

Debug can be turned on/off from the tools menu

=head1 Configuration and Environment

Package variables contain the list of publicly accessible rooms

=head1 Dependencies

=over 3

=item L<CatalystX::Usul::Controller>

=back

=head1 Incompatibilities

There are no known incompatibilities in this module

=head1 Bugs and Limitations

There are no known bugs in this module.
Please report problems to the address below.
Patches are welcome

=head1 Author

Peter Flanigan, C<< <Support at RoxSoft.co.uk> >>

=head1 License and Copyright

Copyright (c) 2008 Peter Flanigan. All rights reserved

This program is free software; you can redistribute it and/or modify it
under the same terms as Perl itself. See L<perlartistic>

This program is distributed in the hope that it will be useful,
but WITHOUT WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE

=cut

# Local Variables:
# mode: perl
# tab-width: 3
# End:
