#  You may distribute under the terms of either the GNU General Public License
#  or the Artistic License (the same terms as Perl itself)
#
#  (C) Paul Evans, 2011 -- leonerd@leonerd.org.uk

package Tickit::Widget::Scroller::Item::RichText;

use strict;
use warnings;

use base qw( Tickit::Widget::Scroller::Item::Text );

our $VERSION = '0.03';

use Tickit::Utils qw( textwidth );

=head1 NAME

C<Tickit::Widget::Scroller::Item::RichText> - static text with render
attributes

=head1 SYNOPSIS

 use Tickit::Widget::Scroller;
 use Tickit::Widget::Scroller::Item::RichText;
 use String::Tagged;

 my $str = String::Tagged->new( "An important message" );
 $str->apply_tag( 3, 9, b => 1 );

 my $scroller = Tickit::Widget::Scroller->new;

 $scroller->push(
    Tickit::Widget::Scroller::Item::RichText->new( $str )
 );

=head1 DESCRIPTION

This subclass of L<Tickit::Widget::Scroller::Item::Text> draws static text
with rendering attributes, used to apply formatting. The attributes are stored
by supplying the text in an instance of a L<String::Tagged> object.

The recognised attributes are those of L<Tickit::Pen>, taking the same names
and values.

=cut

sub text
{
   my $self = shift;
   return $self->{text}->str;
}

sub render
{
   my $self = shift;
   my ( $win, %args ) = @_;

   my $text = $self->{text};
   my $chunks = $self->{chunks};

   foreach my $lineidx ( $args{firstline} .. $args{lastline} ) {
      my $spare = $args{width};

      my $indent = ( $lineidx && $self->{indent} ) ? $self->{indent} : 0;
      $spare -= $indent;

      $win->goto( $args{top} + $lineidx, 0 );
      $win->erasech( $indent, 1 ) if $indent;

      $text->iter_substr_nooverlap(
         sub {
            my ( $substr, %tags ) = @_;
            my %pen;
            $pen{$_} = $tags{$_} for @Tickit::Pen::ALL_ATTRS;
            $win->print( $substr, %pen );
            $spare -= textwidth $substr;
         },
         start => $chunks->[$lineidx][0],
         len   => $chunks->[$lineidx][1],
      );

      $win->erasech( $spare ) if $spare >  0;
   }
}

=head1 AUTHOR

Paul Evans <leonerd@leonerd.org.uk>

=cut

0x55AA;
