#  You may distribute under the terms of either the GNU General Public License
#  or the Artistic License (the same terms as Perl itself)
#
#  (C) Paul Evans, 2019 -- leonerd@leonerd.org.uk

package Future::AsyncAwait::Awaitable;

use strict;
use warnings;

our $VERSION = '0.34_001';

=head1 NAME

C<Future::AsyncAwait::Awaitable> - the interface required by C<Future::AsyncAwait>

=head1 DESCRIPTION

This module documents the method interface required by C<Future::AsyncAwait>
to operate on future instances returned by expressions invoked by the C<await>
keyword, and returned by functions declared by C<async sub>. This information
is largely of relevance to implementors of other module integrations, event
systems, or similar. It is not necessary to make regular use of the syntax
provided by the module when working with existing event systems.

=cut

=head1 CONSTRUCTORS

The following methods are expected to create new future instances. They make
use of the class set by the prevailing C<future_class> import argument, if
set, or default to C<Future> if not.

=head2 done

Generate a new immediate future that is successful. The future will already be
ready and have the list of values set as its result.

   $f = $CLASS->done( @results )

   # $f->is_ready will be true
   # $f->get will return @results

=head2 fail

Generate a new immediate future that is failed. The future will already be
ready and invoking the L</get> method will throw the given exception.

   $f = $CLASS->fail( $message )

   # $f->is_ready will be true
   # $f->get will throw $message

=head1 INSTANCE METHODS

=head2 new

Generate a new pending future of the same type as an existing one, which is
not modified by doing so.

   $new_f = $f->new

If the instance has any fields that are required for successful operation
(such as application-wide context or event system components) these ought to
be copied. The method should not 

=head2 done

Sets the success result of an existing still-pending future. It will only be
invoked on future instances that are currently pending.

   $f->done( @results )

   # $f->is_ready will now be true
   # $f->get will now return @results

=head2 fail

Sets the failure result of an existing still-pending future. It will only be
invoked on future instances that are currently pending.

   $f->fail( $message )

   # $f->is_ready will now be true
   # $f->get will now throw $message

=head2 is_ready

Returns true if a future is ready (successful, failed or cancelled); false if
still pending.

   $bool = $f->is_ready

=head2 is_cancelled

Returns true is a future has already been cancelled; false if still pending,
successful or failed.

   $bool = $f->is_cancelled

=head2 get

Yields the result of a successful future (or just the first value if called in
scalar context). Throws the failure message as an exception if called on a a
failed one. Will not be invoked on a pending or cancelled future.

   @result = $f->get
   $result = $f->get
   $f->get

=head2 on_ready

Attach a new CODE reference to be invoked when the future becomes ready (by
success or failure). The arguments and context that C<$code> is invoked with
are unspecified.

   $f->on_ready( $code )

=head2 on_cancel

Attach a new CODE reference to be invoked when the future is cancelled. The
arguments and context that C<$code> is invoked with are unspecified.

   $f1->on_cancel( $f2 )

=cut

=head1 AUTHOR

Paul Evans <leonerd@leonerd.org.uk>

=cut

0x55AA;
