package Poker::Eval;
use strict;
use warnings FATAL => 'all';
use Poker::Score::High;
use Algorithm::Combinatorics qw(combinations);
use Moo;

=head1 NAME

Poker::Eval - Evaluate and score hole cards in relation to community cards. Base class for specific game types. 

=head1 VERSION

Version 0.01

=cut

our $VERSION = '0.01';


=head1 SYNOPSIS

    This is just a base class. Poker::Eval::Omaha shows a real example.

    use Poker::Dealer;
    use Poker::Eval::Omaha; # Omaha rules
    use Poker::Score::High; # Highball scoring system
    use feature qw(say);

    # Create dealer and shuffle deck
    my $dealer = Poker::Dealer->new;
    $dealer->shuffle_deck;

    # Create Omaha evaluation object
    my $eval = Poker::Eval::Omaha->new(
      scorer => Poker::Score::High->new, # standard highball 
    );

    # deal out five community cards
    $eval->community_cards( $dealer->deal_up(5) );

    # deal yourself four cards
    my $hole_cards = $dealer->deal_down(4);

    # score of the best hand you can make (higher is better)
    say $eval->best_hand_score($hole_cards);

    # name of the best hand you can make (e.g. 'Two Pair')
    say $eval->best_hand_name($hole_cards);

    # cards of best hand you can make in human-readable form. 
    say $eval->best_hand_flat($hole_cards);
  
    # array_ref of Poker::Card objects.
    $array_ref = $eval->best_hand_cards($hole_cards);

=head1 INTRODUCTION

In most poker variants, Holdem and Stud for example, any combination of hole and community cards can be used to make the best hand.  Poker::Eval::Community is the correct subclass for those games.  But in Omaha, your best hand is made using EXACTLY two hole cards and EXACTLY three community cards, so Poker::Eval::Omaha is what you want. Other subclasses include Badugi, Chinese, and Wild.

=head1 SEE ALSO

Poker::Eval::Community, Poker::Eval::Omaha, Poker::Eval::Wild, Poker::Eval::Badugi, Poker::Eval::Chinese, Poker::Eval::Bitch, Poker::Eval::Badugi27, Poker::Score, Poker::Dealer

=head1 ATTRIBUTES

=head2 community_cards

Array ref of Poker::Card objects representing community cards
=cut


has 'community_cards' => (
  is      => 'rw',
  isa => sub { die "Not an array ref!" unless ref($_[0]) eq 'ARRAY' },
  builder => '_build_community_cards',
);

sub _build_community_cards { 
  return [];
};

=head2 scorer

Required attribute that identifies the scoring system. Must be a Poker::Score 
object.  See Poker::Score for available options.

=cut

has 'scorer' => (
  is      => 'rw',
  isa => sub { die "Not an Score object!" unless $_[0]->isa('Poker::Score') },
);

=head1 METHODS

=head2 best_hand_name

Name of the best hand you can make (e.g. 'Two Pair')

=cut

sub best_hand_name {
  my ($self, $hole) = @_;
  return $self->best_hand($hole)->{name};
}

=head2 best_hand_score

Numerical score of the best hand you can make

=cut


sub best_hand_score {
  my ($self, $hole) = @_;
  return $self->best_hand($hole)->{score};
} 

=head2 best_hand_cards

Array_ref of Poker::Card objects representing the best hand you can make.

=cut

sub best_hand_cards {
  my ($self, $hole) = @_;
  return $self->best_hand($hole)->{hand};
} 

sub flatten {
  my ($self, $cards) = @_;
  return join('', map { $_->rank . $_->suit } @{ $cards });
}

=head2 best_hand_flat

Cards of the best hand you can make in human-readable form.

=cut

sub best_hand_flat {
  my ($self, $hole) = @_;
  return $self->flatten($self->best_hand_cards($hole)); 
} 

=head2 best_hand

All of the above returned in a single hash ref for convenience and efficiency.

=cut

=head2 community_flat

Community cards in human-readable form.

=cut

sub community_flat {
  my $self = shift;
  return $self->flatten($self->community_cards);
} 

sub BUILD { }

=head1 BUGS

Probably.  Only developer tested so far.

=head1 AUTHOR

Nathaniel Graham, C<< <ngraham at cpan.org> >>

=head1 LICENSE AND COPYRIGHT

Copyright 2016 Nathaniel Graham.

This program is free software; you can redistribute it and/or modify it
under the terms of the the Artistic License (2.0). You may obtain a
copy of the full license at:

L<http://www.perlfoundation.org/artistic_license_2_0>

=cut

1;
