package # hide from PAUSE
App::YTDL::Arguments;

use warnings;
use strict;
use 5.010000;

use Exporter qw( import );
our @EXPORT_OK = qw( from_arguments_to_choices );

use Term::ANSIScreen       qw( :cursor :screen );
use Term::Choose           qw( choose );
use URI                    qw();
use URI::Escape            qw( uri_unescape );

use if $^O eq 'MSWin32', 'Win32::Console::ANSI';

use App::YTDL::ChooseVideos qw( choose_videos );
use App::YTDL::DataExtract  qw( prepare_info );
use App::YTDL::GetData      qw( get_yt_list_info get_vimeo_list_info get_download_info get_youtube_list_info ); #



sub from_arguments_to_choices {
    my ( $opt, @ids ) = @_;
    my $info = {};
    my $invalid_char = $opt->{invalid_char};
    my $more = 0;
    for my $webpage_url ( @ids ) {
        if ( my $uploader_id = _youtube_user_id( $opt, $webpage_url ) ) {
            my $ex = 'youtube';
            my $tmp = _yt_list_info( $opt, 'CL', $uploader_id );
            #my $tmp = _youtube_list_info( $opt, 'CL', $uploader_id );
            next if ! defined $tmp;
            my ( $vi ) = keys %{$tmp->{$ex}};
            my $prompt = $tmp->{$ex}{$vi}{uploader};
            choose_videos( $opt, $info, $tmp, $ex, $prompt );
        }
        elsif ( my $playlist_id = _youtube_playlist_id( $opt, $webpage_url ) ) {
            my $ex = 'youtube';
            my $tmp = _yt_list_info( $opt, 'PL', $playlist_id );
            next if ! defined $tmp;
            my $prompt = $playlist_id;
            choose_videos( $opt, $info, $tmp, $ex, $prompt );
        }
        elsif ( my $more_ids = _youtube_more_ids( $opt, $webpage_url ) ) {
            my $ex = 'youtube';
            my $tmp = _yt_more_url_info( $opt, $more_ids );
            next if ! defined $tmp;
            my $prompt = $more_ids;
            choose_videos( $opt, $info, $tmp, $ex, $prompt );
        }
        elsif ( my $video_id = _youtube_video_id( $opt, $webpage_url )  ) {
            my $ex = 'youtube';
            $info->{$ex}{$video_id}{extractor}   = 'youtube';
            $info->{$ex}{$video_id}{webpage_url} = sprintf 'https://www.youtube.com/watch?v=%s', $video_id;
        }
        elsif ( my $vimeo_uploader_id = _vimeo_uploader_id( $opt, $webpage_url )  ) {
            my $ex = 'vimeo';
            my $tmp = _vimeo_list_info( $opt, $vimeo_uploader_id );
            next if ! defined $tmp;
            my ( $vi ) = keys %{$tmp->{$ex}};
            my $prompt = $tmp->{$ex}{$vi}{uploader} || $vimeo_uploader_id;
            choose_videos( $opt, $info, $tmp, $ex, $prompt );
        }
        else {
            _generic( $opt, $info, $webpage_url );
        }
    }
    return $info;
}


sub _generic {
    my ( $opt, $info, $webpage_url ) = @_;
    my $message = "Fetching download info: ";
    my $tmp = get_download_info( $opt, $webpage_url, $message );
    next if ! defined $tmp;
    my ( $ex ) = keys %$tmp;
    my @keys = keys %{$tmp->{$ex}};
    if ( @keys == 1 ) {
        my $video_id = $keys[0];
        $info->{$ex}{$video_id} = $tmp->{$ex}{$video_id};
    }
    else {
        my $prompt = $webpage_url;
        choose_videos( $opt, $info, $tmp, $ex, $prompt );
    }
}


sub _youtube_video_id {
    my ( $opt, $webpage_url ) = @_;
    my $invalid_char = $opt->{invalid_char};
    return     if ! $opt->{fast_list_youtube};
    return     if ! $webpage_url;
    return $1  if   $webpage_url =~ m{^([\p{PerlWord}-]{11})\z};
    return     if   $webpage_url !~ $opt->{yt_regexp};
    return $1  if   $webpage_url =~ m{/.*?[?&;!](?:v|video_id)=([^$invalid_char]+)};
    return $1  if   $webpage_url =~ m{/(?:e|v|embed)/([^$invalid_char]+)};
    return $1  if   $webpage_url =~ m{#p/(?:u|search)/\d+/([^&?/]+)};
    return $1  if   $webpage_url =~ m{youtu.be/([^$invalid_char]+)};
    return;
}

sub _youtube_playlist_id {
    my ( $opt, $webpage_url ) = @_;
    my $invalid_char = $opt->{invalid_char};
    return    if ! $opt->{fast_list_youtube};
    return    if ! $webpage_url;
    return $1 if   $webpage_url =~ m{^p#(?:[FP]L)?([^$invalid_char]+)\z};
    return    if   $webpage_url !~ $opt->{yt_regexp};
    return $1 if   $webpage_url =~ m{/.*?[?&;!]list=([^$invalid_char]+)};
    return;
}

sub _youtube_user_id {
    my ( $opt, $webpage_url ) = @_;
    my $invalid_char = $opt->{invalid_char};
    # return    if ! $opt->{small_list_size};
    return    if ! $opt->{fast_list_youtube};
    return    if ! $webpage_url;
    return $1 if   $webpage_url =~ m{^c#([^$invalid_char]+)\z};
    return    if   $webpage_url !~ $opt->{yt_regexp};
    return $1 if   $webpage_url =~ m{/user/([^$invalid_char]+)};
    return $1 if   $webpage_url =~ m{/channel/([^$invalid_char]+)}; # ?
    return;
}

sub _youtube_more_ids {
    my ( $opt, $webpage_url ) = @_;
    my $invalid_char = $opt->{invalid_char};
    return    if ! $opt->{fast_list_youtube};
    return    if ! $webpage_url;
    return    if   $webpage_url !~ $opt->{yt_regexp};
    return $1 if   uri_unescape( $webpage_url ) =~ m{youtu\.?be.*video_ids=([^$invalid_char]+(?:,[^$invalid_char]+)*)};
    return;
}

sub _vimeo_uploader_id {
    my ( $opt, $webpage_url ) = @_;
    return    if ! $opt->{fast_list_vimeo};
    return    if ! $webpage_url;
    return $1 if   $webpage_url =~ m{https?://vimeo\.com/(?![0-9]+(?:$|[?#/]))([^/]+)(?:/videos|[#?]|$)};
    return;
}


sub _yt_list_info {
    my( $opt, $type, $list_id ) = @_;
    printf "Fetching %s info ... \n", $type eq 'PL' ? 'playlist' : 'channel';
    my $url = URI->new( $type eq 'PL'
        ? 'https://gdata.youtube.com/feeds/api/playlists/' . $list_id
        : 'https://gdata.youtube.com/feeds/api/users/'     . $list_id . '/uploads'
    );
    my $ex = 'youtube';
    my $tmp_info = {};
    my $start_index = 1;
    my $max_results = 50;
    my $count_entries = $max_results;

    while ( $count_entries == $max_results ) {
        $url->query_form( 'start-index' => $start_index, 'max-results' => $max_results, 'v' => $opt->{yt_api_v}, 'alt' => 'json' );
        $start_index += $max_results;
        my $h_ref = get_yt_list_info( $opt, $url->as_string );
        if ( ! defined $h_ref || ! defined $h_ref->{feed}{entry} || ! @{$h_ref->{feed}{entry}} ) {
            push @{$opt->{error_get_download_infos}}, sprintf "%s: %s", $type, $list_id;
            my $prompt = "Error list info: $type - $url";
            choose( [ 'ENTER to continue' ], { prompt => $prompt } );
            return;
        }
        $count_entries = @{$h_ref->{feed}{entry}};
        for my $entry ( @{$h_ref->{feed}{entry}} ) {
            my $video_id = $entry->{'media$group'}{'yt$videoid'}{'$t'};
            $tmp_info->{$ex}{$video_id} = {
                video_id    => $video_id,
                title       => $entry->{'title'}{'$t'},
                uploader    => $entry->{'author'}[0]{'name'}{'$t'},
                published   => $entry->{'published'}{'$t'}, #
                duration    => $entry->{'media$group'}{'yt$duration'}{'seconds'},
                view_count  => $entry->{'yt$statistics'}{'viewCount'},
                webpage_url => sprintf 'https://www.youtube.com/watch?v=%s', $video_id,
            };
            $tmp_info->{$ex}{$video_id}{playlist_id} = $list_id if $type && $type eq 'PL';
            prepare_info( $opt, $tmp_info, $ex, $video_id );
        }
        last if $opt->{small_list_size} && $start_index > 50;
        last if $start_index >= $h_ref->{feed}{'openSearch$totalResults'}{'$t'};
    }
    if ( ! keys %{$tmp_info->{$ex}} ) {
        my $prompt = "No videos found: $type - $url";
        choose( [ 'Continue with ENTER' ], { prompt => $prompt } );
    }
    my $up = keys %{$tmp_info->{$ex}};
    print up( $up + 2 ), cldown;
    return $tmp_info;
}


sub _youtube_list_info { # ###
    my( $opt, $type, $list_id ) = @_;
    printf "Fetching info data ... \n";
    #my $url = 'https://www.youtube.com/user/' . $list_id . '/videos?sort=dd&view=0&flow=list';
    my $url = 'https://www.youtube.com/user/' . $list_id . '/videos';
    my $ex = 'youtube';
    my $tmp_info = {};
    my $a_ref = get_youtube_list_info( $opt, $url );
    if ( ! defined $a_ref ) {
        push @{$opt->{error_get_download_infos}}, sprintf "%s: %s", $type, $list_id;
        my $prompt = "Error list info: $type - $url";
        choose( [ 'ENTER to continue' ], { prompt => $prompt } );
        return;
    }
    for my $entry ( @{$a_ref} ) {
        my $video_id = $entry->{video_id};
        $tmp_info->{$ex}{$video_id} = {
            video_id        => $video_id,
            title           => $entry->{title},
            upload_date_rel => $entry->{upload_date_rel},
            uploader        => $entry->{uploader},
            view_count      => $entry->{view_count},
            duration        => $entry->{duration},
            date_sort       => $entry->{date_sort},
            webpage_url     => $entry->{webpage_url},
        };
        prepare_info( $opt, $tmp_info, $ex, $video_id );
    }
    if ( ! keys %{$tmp_info->{$ex}} ) {
        my $prompt = "No videos found: $type - $url";
        choose( [ 'Continue with ENTER' ], { prompt => $prompt } );
    }
    my $up = keys %{$tmp_info->{$ex}};
    print up( $up + 2 ), cldown;
    return $tmp_info;
}


sub _vimeo_list_info {
    my( $opt, $list_id ) = @_;
    printf "Fetching info ... \n";
    my $page_nr = 1;
    my $ex = 'vimeo';
    my $tmp_info = {};
    my $videos = 0;
    my $url;
    while ( 1 ) {
        $url = sprintf 'https://vimeo.com/%s/videos/page:%d/sort:date', $list_id, $page_nr;
        my ( $a_ref, $next ) = get_vimeo_list_info( $opt, $url );
        if ( ! defined $a_ref ) {
            push @{$opt->{error_get_download_infos}}, sprintf "%s: error", $list_id;
            return;
        }
        if ( ! @$a_ref ) {
            push @{$opt->{error_get_download_infos}}, sprintf "%s: no videos found.", $url;
        }
        for my $entry ( @{$a_ref} ) {
            my $video_id = $entry->{video_id};
            $tmp_info->{$ex}{$video_id} = {
                video_id    => $video_id,
                title       => $entry->{title},
                published   => $entry->{published},
                uploader    => $entry->{uploader},
                webpage_url => $entry->{webpage_url},
            };
            prepare_info( $opt, $tmp_info, $ex, $video_id );
        }
        $videos += @{$a_ref};
        last if $opt->{small_list_size} && $videos >= 48;
        last if ! $next;
        $page_nr++;
    }
    if ( ! keys %{$tmp_info->{$ex}} ) {
        my $prompt = "No videos found: $url";
        choose( [ 'Continue with ENTER' ], { prompt => $prompt } );
    }
    my $up = keys %{$tmp_info->{$ex}};
    print up( $up + 2 ), cldown;
    return $tmp_info;
}


sub _yt_more_url_info {
    my ( $opt, $more_ids ) = @_;
    my $tmp_info = {};
    my $ex = 'youtube';
    for my $video_id ( split /,/, $more_ids ) {
        my $url = URI->new( 'https://gdata.youtube.com/feeds/api/videos/' . $video_id );
        $url->query_form( 'v' => $opt->{yt_api_v}, 'alt' => 'json' );
        my $h_ref = get_yt_list_info( $opt, $url );
        if ( ! defined $h_ref ) {
            my $err_msg = 'Video group: ' . $more_ids . ' - ' . $video_id . '   ' . $url;
            push @{$opt->{error_get_download_infos}}, $err_msg;
            next;
        }
        my $entry = $h_ref->{feed}{entry};
        my $video_id = $entry->{'media$group'}{'yt$videoid'}{'$t'};
        $tmp_info->{$ex}{$video_id} = {
            video_id    => $video_id,
            title       => $entry->{'title'}{'$t'},
            author      => $entry->{'author'}[0]{'name'}{'$t'},
            published   => $entry->{'published'}{'$t'},
            seconds     => $entry->{'media$group'}{'yt$duration'}{'seconds'},
            view_count  => $entry->{'yt$statistics'}{'viewCount'},
            webpage_url => sprintf 'https://www.youtube.com/watch?v=%s', $video_id,
        };
        prepare_info( $opt, $tmp_info, $ex, $video_id );
    }
    return $tmp_info;
}




1;


__END__
