# NAME

Game::Entities

# SYNOPSIS

    my $ECS = Game::Entities.new;

    # Create an entity with a possibly empty set of components
    $guid = $ECS->create(
        Collidable->new,
        Drawable->new,
        Consumable->new,
    );

    # CRUD operations on an entity's components
    $ECS->add( $guid => Equipable->new );      # It is now equipable
    $ECS->delete( $guid => 'Consumable' );     # It is no longer consumable
    $item = $ECS->get( $guid => 'Equipable' );

    # A system operates on sets of entities defined by a view
    $view = $ECS->view(qw( Drawable Equipable ));

    # You can iterate over the set with a callback
    # The callback will get the GUID and the components in the order requested
    $view->each( sub ( $guid, $draw, $item ) {
        draw_equipment( $draw, $item );
    });

    # Or you can do the same iterating over the components
    for ( $view->componens ) {
        my ( $draw, $item ) = @$_;
        draw_equipment( $draw, $item );
    }

    # Or over the entity GUIDs
    for my $guid ( $view->entities ) {
        my ( $draw, $item ) = $ECS->get( $guid => qw( Drawable Equipable ) );
        draw_equipment( $draw, $item );
    }

    # Or you can iterate over the view directly and get both
    for ( @{ $view } ) {
        my ( $guid, $draw, $item ) = ( $_->[0], @{ $_->[1] } );

        if ( $ECS->get( $guid => 'Consumable' ) ) {
            say 'This is equipment you can eat!';
        }

        ...
    }

    # Delete the entity and all its components
    $ECS->delete($guid);

    $ECS->clear; # Delete all entities and components

# DESCRIPTION

Game::Entities is a minimalistic entity manager designed for applications
using an ECS architecture.

If you don't know what this means, Mick West's
[Evolve Your Hierarchy](http://cowboyprogramming.com/2007/01/05/evolve-your-heirachy)
might be a good place to start.

## On Stability

This distribution is currently **experimental**, and as such, its API might
still change without warning. Any change, breaking or not, will be noted in
the change log, so if you wish to use it, please pin your dependencies and
make sure to check the change log before upgrading.

# CONCEPTS

Throughout this documentation, there are a couple of key concepts that will
be used repeatedly:

## GUID

Entities are represented by an opaque global unique identifier: a GUID. GUIDs
used by Game::Entities are opaque to the user, and represent a particular
version of an entity which will remain valid as long as that entity is not
deleted (with `delete` or `clear`).

Each Game::Entities registry supports up to 1,048,575 (2^20 - 1) simultaneous
valid entity GUIDs.

## Valid entities

An entity's GUID is valid from the time it is created with `create` until the
time it is deleted. An entity's GUID can be stored and used as an identifier
anywhere in the program at any point during this time, but must not be used
outside it.

## Components

As far as Game::Entities is concerned, any reference of any type can be used
as a component and added to an entity. This includes blessed and non-blessd
references.

Entities can have any number of components attached to them, but they will
only ever have one component of any one type (as identified by `ref`).

## Views

A group of components defines a view, which is composed of all entities that
have all of that view's components. This means that all the entities for the
view for components `A`, `B`, and `C` will have _at least_ all those
components, but could of course have any others as well.

The main purpose of views is to make it possible to iterate as fast as
possible over any group of entities that have a set of common components.

# METHODS

## new

    $ECS = Game::Entities->new;

Creates a new entity registry. The constructor takes no arguments.

## create

    $guid = $ECS->create(@components)

Creates a new entity and returns its GUID. If called with a list of components,
these will be added to the entity before returning.

## add

    $ECS->add( $guid, @components );

Takes an entity's GUID and a component, and adds that component to the
specified entity. If the entity already had a component of that type, calling
this method will silently overwrite it.

Multiple components can be specified, and they will all be added in the order
they were provided.

## get

    $component  = $ECS->get( $guid, $component  );
    @components = $ECS->get( $guid, @components );

Takes an entity's GUID and a component, and retrieves the component of that
type from the specified entity, if it exists. If the entity had no component
of that type, this method returns undefined instead.

Multiple components can be specified, and they will all be retrieved and
returned in the order they were provided.

When called with a single target component, it will return a scalar. When
called with a list of components, it will return a list.

## delete

    $ECS->delete( $guid );
    $ECS->delete( $guid, @components );

When called with only an entity's GUID, it deletes all components from the
entity, and marks that entity's GUID as invalid.

When called with one or more additional components, those will be removed
from the specified entity in that order. Deleting a component from an entity
is an idempotent process.

## check

    $bool = $ECS->check( $guid => $component );

Takes an entity's GUID and a component and returns a truthy value if the
specified entity has a component of that type, or a falsy value otherwise.

## valid

    $ECS->valid($guid);

Takes an entity's GUID and returns a truthy value if the specified entity is
valid, or a falsy value otherwise. An entity is valid if it has been created
and not yet deleted.

## created

    $count = $ECS->created;

Returns the number of entities that have been created. Calling `clear`
resets this number.

## alive

    $count = $ECS->alive;

Returns the number of created entities that are still valid. That is:
entities that have been created and not yet deleted.

## clear

    $ECS->clear;

Resets the internal storage of the registry. Calling this method leaves no
trace from the previous state.

## view

    $view = $ECS->view;
    $view = $ECS->view(@components);

Takes a set of one or more components, and returns an internal object
representing a _view_ for that specific set of components. Entities in the
view are in no specific order, and this order is not guaranteed to remain the
same.

As a special case, calling this method with no components will generate a
view which is guaranteed to always have all valid entities.

Once a view has been created, it should remain valid as long as none of the
components in the view's set are added or deleted from any entity. Once this
is done, the data returned by the view object is no longer guaranteed to be
accurate. For this reason, it is not recommended to keep hold of view objects
for longer than it takes to run an iteration.

The view object can be used as an array reference to iterate over the pair
of entity GUIDs and component sets. When used in this way, each value will
be an array reference holding the guid as the first value, and a nested array
reference with the list of components as the second value.

    for my $pair ( $ECS->view(@components)->@* ) {
        my ( $guid, @components ) = ( $pair->[0], $pair->[1]->@* );
        # Do something with the data
    }

The pairs returned when used in this way are returned by calling `pairs` in
[List::Util](https://metacpan.org/pod/List%3A%3AUtil#pairs).

Apart from this, the interface for the view object is documented below:

### each

    $view->each( sub ( $guid, @components ) { ... } );

Takes a code reference which will be called once per entity in the view.

The code reference will be called with the GUID for the current entity as the
first parameter, and the components in the requested set in the
order they were specified.

If the view was created with no components (in other words, if it's a view
of all entities), the list of components passed to the coe reference will be
empty.

Within the callback, it is safe to add or delete entities, as well as to add
or remove components from those entities.

### entities

    @guids = $view->entities;

Returns a list of only the GUIDs of the entities in this view.

### components

    @components = $view->components;

Returns a list of array references, each of which will hold the list of
components for a single entity in the order provided when the view was
created.

Useful for iterating like

    my $view = $ECS->view(qw( A B C ));
    for ( $view->components ) {
        my ( $a, $b, $c ) = @$_;
        ...
    }

# COPYRIGHT AND LICENSE

Copyright 2021 José Joaquín Atria

This library is free software; you can redistribute it and/or modify it under
the Artistic License 2.0.
