# Before `make' is performed this script should be runnable with
# `make test'. After `make install' it should work as `perl test.pl'
 
use strict;
use vars qw($loaded);

BEGIN { $| = 1; print "1..10032\n"; }
END   {print "not ok 1\n" unless $loaded; }

my $ok_count = 1;
sub ok {
  shift or print "not ";
  print "ok $ok_count\n";
  ++$ok_count;
}

use Math::Interpolate qw(derivatives constant_interpolate
			 linear_interpolate robust_interpolate);

# If we got here, then the package being tested was loaded.
$loaded = 1;
ok(1);								#  1

# Open the data filehandle for the answers.
no strict 'refs';
my $data_fh = \*{"::DATA"};
use strict;

sub Compare {
  my ($a, $b) = @_;
  if ($a == 0) {
    return $a == $b;
  }
  abs(($a - $b)/$a) <= 1.0e-6;
}

# Check the derivates from two points.
my @dy = derivatives([1, 2], [10, 20]);
ok( $dy[0] == 10 );						#  2
ok( $dy[1] == 10 );						#  3

# Check the derivative generating code.
my @x  = (  5,  7,  8, 12, 15);
my @y  = (-10, -9, 10, 10, 12);
@dy = derivatives(\@x, \@y);
foreach my $result_dy ( @dy ) {
  my $answer_dy = <$data_fh>;
  ok( Compare($result_dy, $answer_dy) );			# 4-8
}

# Skip a line in __DATA__.
<$data_fh>;

# Calculate the derivates for a reversed @y.
@dy = derivatives(\@x, [reverse @y]);
foreach my $result_dy ( @dy ) {
  my $answer_dy = <$data_fh>;
  ok( Compare($result_dy, $answer_dy) );			# 9-13
}

# Skip one line in __DATA__
<$data_fh>;

# Check the extrapolation code.
ok( linear_interpolate(1, [2, 3], [10, 15]) == 5 );		# 14

# Check the constant interpolation code.
ok( constant_interpolate(0,   \@x, \@y) == -10 );		# 15
ok( constant_interpolate(6,   \@x, \@y) == -10 );		# 16
ok( constant_interpolate(7,   \@x, \@y) ==  -9 );		# 17
ok( constant_interpolate(7.5, \@x, \@y) ==  -9 );		# 18
ok( constant_interpolate(8,   \@x, \@y) ==  10 );		# 19
ok( constant_interpolate(12,   \@x, \@y) == 10 );		# 20
ok( constant_interpolate(15,   \@x, \@y) == 12 );		# 21
ok( constant_interpolate(99,   \@x, \@y) == 12 );		# 22

# Set the derivates to some odd values and calulate the resulting
# interpolated curve.
@dy = (-2,  2, -2,  2, -2);

# Calculate the linear and robust interpolated curve letting the
# robust_interpolate calculate the derivatives.
my $i = 0;
while (1) {
  my $x = $x[0] + 0.01*($i++);
  last unless $x <= $x[-1];
  my ($l_y, $l_dy) = linear_interpolate($x, \@x, \@y);
  my ($r_y, $r_dy) = robust_interpolate($x, \@x, \@y);
#  print "$x\t$l_y\t$l_dy\t$r_y\t$r_dy\n";
  my @answers = split ' ', <$data_fh>;
  ok( Compare($x,    $answers[0]) );
  ok( Compare($l_y,  $answers[1]) );
  ok( Compare($l_dy, $answers[2]) );
  ok( Compare($r_y,  $answers[3]) );
  ok( Compare($r_dy, $answers[4]) );
}

# Skip one line in __DATA__
<$data_fh>;

# Calculate the linear and robust interpolated curve using the derivates
# we set ourselves.
$i = 0;
while (1) {
  my $x = $x[0] + 0.01*($i++);
  last unless $x <= $x[-1];
  my ($l_y, $l_dy) = linear_interpolate($x, \@x, \@y);
  my ($r_y, $r_dy) = robust_interpolate($x, \@x, \@y, \@dy);
#  print "$x\t$l_y\t$l_dy\t$r_y\t$r_dy\n";
  my @answers = split ' ', <$data_fh>;
  ok( Compare($x,    $answers[0]) );
  ok( Compare($l_y,  $answers[1]) );
  ok( Compare($l_dy, $answers[2]) );
  ok( Compare($r_y,  $answers[3]) );
  ok( Compare($r_dy, $answers[4]) );
}

exit 0;

__DATA__
0.398796498905908
0.626886145404664
0.207650273224044
0.32
1.01333333333333

-1.8
-0.2
-0.0498687664041995
-0.484201537147737
-0.229472859102489

5	-10	0.5	-10	0.5
5.01	-9.995	0.5	-9.99600799496208	0.399604917959649
5.02	-9.99	0.5	-9.99200789343858	0.40041579768342
5.03	-9.985	0.5	-9.98799967077286	0.401229148073759
5.04	-9.98	0.5	-9.98398330220806	0.402044979178022
5.05	-9.975	0.5	-9.9799587628866	0.402863301094697
5.06	-9.97	0.5	-9.97592602784963	0.4036841239737
5.07	-9.965	0.5	-9.97188507203658	0.404507458016712
5.08	-9.96	0.5	-9.96783587028458	0.405333313477472
5.09	-9.955	0.5	-9.96377839732796	0.406161700662117
5.1	-9.95	0.5	-9.95971262779774	0.406992629929495
5.11	-9.945	0.5	-9.95563853622106	0.40782611169148
5.12	-9.94	0.5	-9.95155609702071	0.408662156413317
5.13	-9.935	0.5	-9.94746528451454	0.409500774613936
5.14	-9.93	0.5	-9.94336607291493	0.410341976866288
5.15	-9.925	0.5	-9.93925843632829	0.411185773797675
5.16	-9.92	0.5	-9.93514234875445	0.412032176090095
5.17	-9.915	0.5	-9.93101778408617	0.412881194480565
5.18	-9.91	0.5	-9.92688471610854	0.413732839761476
5.19	-9.905	0.5	-9.92274311849848	0.414587122780925
5.2	-9.9	0.5	-9.91859296482412	0.415444054443069
5.21	-9.895	0.5	-9.91443422854428	0.41630364570847
5.22	-9.89	0.5	-9.91026688300789	0.417165907594438
5.23	-9.885	0.5	-9.90609090145341	0.4180308511754
5.24	-9.88	0.5	-9.9019062570083	0.41889848758324
5.25	-9.875	0.5	-9.89771292268837	0.419768828007662
5.26	-9.87	0.5	-9.89351087139727	0.420641883696559
5.27	-9.865	0.5	-9.88930007592587	0.421517665956364
5.28	-9.86	0.5	-9.88508050895169	0.422396186152424
5.29	-9.855	0.5	-9.8808521430383	0.423277455709367
5.3	-9.85	0.5	-9.8766149506347	0.424161486111471
5.31	-9.845	0.5	-9.87236890407477	0.425048288903038
5.32	-9.84	0.5	-9.86811397557666	0.425937875688782
5.33	-9.835	0.5	-9.86385013724214	0.426830258134188
5.34	-9.83	0.5	-9.85957736105603	0.427725447965914
5.35	-9.825	0.5	-9.85529561888558	0.428623456972164
5.36	-9.82	0.5	-9.85100488247985	0.429524297003081
5.37	-9.815	0.5	-9.84670512346907	0.430427979971142
5.38	-9.81	0.5	-9.84239631336406	0.431334517851538
5.39	-9.805	0.5	-9.83807842355554	0.432243922682589
5.4	-9.8	0.5	-9.83375142531357	0.433156206566129
5.41	-9.795	0.5	-9.82941528978683	0.434071381667914
5.42	-9.79	0.5	-9.82506998800206	0.434989460218031
5.43	-9.785	0.5	-9.82071549086334	0.4359104545113
5.44	-9.78	0.5	-9.81635176915149	0.436834376907692
5.45	-9.775	0.5	-9.81197879352343	0.437761239832736
5.46	-9.77	0.5	-9.80759653451145	0.438691055777947
5.47	-9.765	0.5	-9.80320496252262	0.439623837301235
5.48	-9.76	0.5	-9.79880404783809	0.440559597027343
5.49	-9.755	0.5	-9.79439376061243	0.441498347648259
5.5	-9.75	0.5	-9.78997407087295	0.442440101923655
5.51	-9.745	0.5	-9.78554494851902	0.443384872681326
5.52	-9.74	0.5	-9.7811063633214	0.444332672817612
5.53	-9.735	0.5	-9.77665828492153	0.445283515297851
5.54	-9.73	0.5	-9.77220068283085	0.446237413156815
5.55	-9.725	0.5	-9.76773352643012	0.447194379499161
5.56	-9.72	0.5	-9.76325678496868	0.44815442749988
5.57	-9.715	0.5	-9.75877042756379	0.449117570404749
5.58	-9.71	0.5	-9.75427442319986	0.450083821530787
5.59	-9.705	0.5	-9.74976874072781	0.451053194266722
5.6	-9.7	0.5	-9.7452533488643	0.452025702073445
5.61	-9.695	0.5	-9.740728216191	0.453001358484484
5.62	-9.69	0.5	-9.73619331115391	0.453980177106477
5.63	-9.685	0.5	-9.73164860206258	0.454962171619634
5.64	-9.68	0.5	-9.72709405708938	0.455947355778233
5.65	-9.675	0.5	-9.72252964426877	0.456935743411083
5.66	-9.67	0.5	-9.71795533149657	0.457927348422026
5.67	-9.665	0.5	-9.71337108652915	0.458922184790416
5.68	-9.66	0.5	-9.70877687698273	0.459920266571611
5.69	-9.655	0.5	-9.70417267033258	0.460921607897476
5.7	-9.65	0.5	-9.69955843391228	0.461926222976879
5.71	-9.645	0.5	-9.69493413491292	0.462934126096195
5.72	-9.64	0.5	-9.69029974038235	0.463945331619818
5.73	-9.635	0.5	-9.68565521722437	0.464959853990662
5.74	-9.63	0.5	-9.68100053219798	0.465977707730693
5.75	-9.625	0.5	-9.67633565191653	0.466998907441434
5.76	-9.62	0.5	-9.67166054284698	0.468023467804496
5.77	-9.615	0.5	-9.66697517130907	0.469051403582108
5.78	-9.61	0.5	-9.66227950347449	0.47008272961764
5.79	-9.605	0.5	-9.65757350536611	0.471117460836144
5.8	-9.6	0.5	-9.65285714285714	0.472155612244897
5.81	-9.595	0.5	-9.64813038167029	0.473197198933944
5.82	-9.59	0.5	-9.64339318737696	0.474242236076634
5.83	-9.585	0.5	-9.6386455253964	0.475290738930196
5.84	-9.58	0.5	-9.63388736099486	0.476342722836274
5.85	-9.575	0.5	-9.62911865928475	0.477398203221499
5.86	-9.57	0.5	-9.6243393852238	0.478457195598057
5.87	-9.565	0.5	-9.61954950361417	0.479519715564248
5.88	-9.56	0.5	-9.61474897910161	0.480585778805069
5.89	-9.555	0.5	-9.60993777617459	0.481655401092796
5.9	-9.55	0.5	-9.60511585916341	0.482728598287553
5.91	-9.545	0.5	-9.60028319223933	0.483805386337917
5.92	-9.54	0.5	-9.59543973941368	0.484885781281499
5.93	-9.535	0.5	-9.59058546453697	0.485969799245545
5.94	-9.53	0.5	-9.58572033129799	0.487057456447538
5.95	-9.525	0.5	-9.58084430322288	0.488148769195805
5.96	-9.52	0.5	-9.57595734367426	0.489243753890125
5.97	-9.515	0.5	-9.57105941585029	0.490342427022351
5.98	-9.51	0.5	-9.56615048278375	0.491444805177024
5.99	-9.505	0.5	-9.56123050734109	0.492550905032007
6	-9.5	0.5	-9.55629945222155	0.493660743359108
6.01	-9.495	0.5	-9.55135727995613	0.494774337024718
6.02	-9.49	0.5	-9.54640395290673	0.495891702990457
6.03	-9.485	0.5	-9.54143943326514	0.497012858313811
6.04	-9.48	0.5	-9.53646368305209	0.498137820148792
6.05	-9.475	0.5	-9.5314766641163	0.499266605746587
6.06	-9.47	0.5	-9.52647833813346	0.500399232456223
6.07	-9.465	0.5	-9.52146866660532	0.501535717725233
6.08	-9.46	0.5	-9.51644761085862	0.502676079100329
6.09	-9.455	0.5	-9.51141513204415	0.503820334228076
6.1	-9.45	0.5	-9.50637119113573	0.50496850085558
6.11	-9.445	0.5	-9.50131574892922	0.506120596831168
6.12	-9.44	0.5	-9.49624876604146	0.50727664010509
6.13	-9.435	0.5	-9.49117020290929	0.508436648730215
6.14	-9.43	0.5	-9.48608001978851	0.509600640862733
6.15	-9.425	0.5	-9.48097817675282	0.510768634762871
6.16	-9.42	0.5	-9.47586463369282	0.511940648795605
6.17	-9.415	0.5	-9.47073935031491	0.51311670143138
6.18	-9.41	0.5	-9.46560228614027	0.514296811246846
6.19	-9.405	0.5	-9.46045340050378	0.515480996925584
6.2	-9.4	0.5	-9.45529265255293	0.516669277258847
6.21	-9.395	0.5	-9.45012000124677	0.517861671146306
6.22	-9.39	0.5	-9.4449354053548	0.519058197596805
6.23	-9.385	0.5	-9.4397388234559	0.520258875729113
6.24	-9.38	0.5	-9.43453021393719	0.521463724772688
6.25	-9.375	0.5	-9.42930953499295	0.522672764068454
6.26	-9.37	0.5	-9.42407674462349	0.523886013069568
6.27	-9.365	0.5	-9.418831800634	0.525103491342211
6.28	-9.36	0.5	-9.41357466063348	0.526325218566369
6.29	-9.355	0.5	-9.40830528203354	0.527551214536632
6.3	-9.35	0.5	-9.40302362204724	0.528781499162998
6.31	-9.345	0.5	-9.39772963768801	0.530016092471678
6.32	-9.34	0.5	-9.39242328576841	0.531255014605913
6.33	-9.335	0.5	-9.38710452289895	0.532498285826793
6.34	-9.33	0.5	-9.38177330548699	0.533745926514085
6.35	-9.325	0.5	-9.37642958973547	0.534997957167077
6.36	-9.32	0.5	-9.37107333164172	0.536254398405406
6.37	-9.315	0.5	-9.36570448699628	0.53751527096992
6.38	-9.31	0.5	-9.3603230113817	0.538780595723524
6.39	-9.305	0.5	-9.35492886017125	0.540050393652048
6.4	-9.3	0.5	-9.34952198852772	0.541324685865118
6.41	-9.295	0.5	-9.34410235140223	0.542603493597031
6.42	-9.29	0.5	-9.33866990353287	0.54388683820764
6.43	-9.285	0.5	-9.33322459944354	0.545174741183246
6.44	-9.28	0.5	-9.32776639344262	0.546467224137497
6.45	-9.275	0.5	-9.32229523962173	0.547764308812296
6.46	-9.27	0.5	-9.31681109185442	0.549066017078716
6.47	-9.265	0.5	-9.31131390379486	0.55037237093792
6.48	-9.26	0.5	-9.30580362887659	0.551683392522089
6.49	-9.255	0.5	-9.30028022031114	0.552999104095364
6.5	-9.25	0.5	-9.29474363108675	0.55431952805479
6.51	-9.245	0.5	-9.289193813967	0.555644686931266
6.52	-9.24	0.5	-9.28363072148953	0.556974603390508
6.53	-9.235	0.5	-9.27805430596459	0.55830930023402
6.54	-9.23	0.5	-9.27246451947379	0.559648800400071
6.55	-9.225	0.5	-9.26686131386861	0.560993126964675
6.56	-9.22	0.5	-9.26124464076913	0.562342303142595
6.57	-9.215	0.5	-9.25561445156255	0.563696352288334
6.58	-9.21	0.5	-9.24997069740184	0.565055297897155
6.59	-9.205	0.5	-9.24431332920429	0.566419163606093
6.6	-9.2	0.5	-9.23864229765013	0.56778797319499
6.61	-9.195	0.5	-9.23295755318106	0.569161750587527
6.62	-9.19	0.5	-9.22725904599882	0.570540519852266
6.63	-9.185	0.5	-9.22154672606374	0.571924305203713
6.64	-9.18	0.5	-9.21582054309327	0.573313131003374
6.65	-9.175	0.5	-9.2100804465605	0.574707021760834
6.66	-9.17	0.5	-9.20432638569268	0.576106002134824
6.67	-9.165	0.5	-9.19855830946973	0.577510096934333
6.68	-9.16	0.5	-9.19277616662273	0.578919331119691
6.69	-9.155	0.5	-9.18697990563236	0.580333729803684
6.7	-9.15	0.5	-9.18116947472745	0.581753318252674
6.71	-9.145	0.5	-9.17534482188337	0.583178121887723
6.72	-9.14	0.5	-9.16950589482051	0.584608166285738
6.73	-9.135	0.5	-9.16365264100269	0.586043477180611
6.74	-9.13	0.5	-9.15778500763561	0.587484080464386
6.75	-9.125	0.5	-9.15190294166528	0.588930002188415
6.76	-9.12	0.5	-9.14600638977636	0.590381268564547
6.77	-9.115	0.5	-9.14009529839059	0.591837905966306
6.78	-9.11	0.5	-9.13416961366518	0.5932999409301
6.79	-9.105	0.5	-9.12822928149113	0.594767400156424
6.8	-9.1	0.5	-9.12227424749164	0.596240310511068
6.81	-9.095	0.5	-9.11630445702039	0.597718699026371
6.82	-9.09	0.5	-9.11031985515993	0.599202592902443
6.83	-9.085	0.5	-9.10432038671993	0.600692019508418
6.84	-9.08	0.5	-9.09830599623555	0.602187006383722
6.85	-9.075	0.5	-9.09227662796567	0.60368758123934
6.86	-9.07	0.5	-9.08623222589123	0.605193771959109
6.87	-9.065	0.5	-9.08017273371344	0.606705606601
6.88	-9.06	0.5	-9.07409809485205	0.60822311339844
6.89	-9.055	0.5	-9.0680082524436	0.609746320761615
6.9	-9.05	0.5	-9.06190314933965	0.611275257278812
6.91	-9.045	0.5	-9.05578272810497	0.61280995171775
6.92	-9.04	0.5	-9.04964693101575	0.614350433026944
6.93	-9.035	0.5	-9.04349570005779	0.615896730337067
6.94	-9.03	0.5	-9.03732897692465	0.617448872962319
6.95	-9.025	0.5	-9.03114670301585	0.619006890401819
6.96	-9.02	0.5	-9.02494881943497	0.62057081234102
6.97	-9.015	0.5	-9.0187352669878	0.622140668653108
6.98	-9.01	0.5	-9.01250598618047	0.623716489400433
6.99	-9.005	0.5	-9.00626091721752	0.625298304835947
7	-9	19	-9	19
7.01	-8.81	19	-8.98829572635871	1.71041442503272
7.02	-8.62000000000001	19	-8.96586282638561	2.77260823807361
7.03	-8.43	19	-8.93291473999276	3.8134485317473
7.04	-8.24	19	-8.88966509772637	4.83291623199987
7.05	-8.05	19	-8.83632772097965	5.83099224347547
7.06	-7.85999999999999	19	-8.77311662220603	6.80765744948759
7.07	-7.66999999999999	19	-8.70024600513255	7.76289271199009
7.08	-7.48	19	-8.61793026497366	8.6966788715492
7.09	-7.29	19	-8.52638398864517	9.60899674731436
7.1	-7.10000000000001	19	-8.42582195497862	10.4998271369897
7.11	-6.91000000000001	19	-8.31645913493586	11.3691508168051
7.12	-6.72	19	-8.1985106918239	12.2169485414876
7.13	-6.53	19	-8.07219198151016	13.0432010442322
7.14	-6.33999999999999	19	-7.93771855263786	13.8478890366733
7.15	-6.14999999999999	19	-7.79530614684182	14.6309932088554
7.16	-5.96	19	-7.64517069896446	15.3924942292045
7.17	-5.77	19	-7.48752833727218	16.1323727444987
7.18	-5.58000000000001	19	-7.32259538367193	16.8506093798391
7.19	-5.39000000000001	19	-7.15058835392818	17.5471847386211
7.2	-5.2	19	-6.97172395788008	18.2220794025045
7.21	-5.01	19	-6.78621909965901	18.8752739313849
7.22	-4.81999999999999	19	-6.59429087790628	19.5067488633643
7.23	-4.62999999999999	19	-6.39615658599136	20.1164847147213
7.24	-4.44	19	-6.19203371223009	20.7044619798823
7.25	-4.25	19	-5.98213994010346	21.2706611313918
7.26	-4.06	19	-5.76669314847655	21.8150626198832
7.27	-3.87000000000001	19	-5.54591141181778	22.3376468740488
7.28	-3.68	19	-5.32001300041846	22.8383943006108
7.29	-3.49	19	-5.08921638061271	23.3172852842911
7.3	-3.29999999999999	19	-4.85374021499748	23.7743001877824
7.31	-3.10999999999999	19	-4.61380336265312	24.2094193517177
7.32	-2.91999999999999	19	-4.36962487936405	24.6226230946412
7.33	-2.73	19	-4.1214240178398	25.0138917129781
7.34	-2.54	19	-3.8694202279364	25.3832054810051
7.35	-2.35000000000001	19	-3.61383315687795	25.7305446508202
7.36	-2.16000000000001	19	-3.35488264947861	26.0558894523129
7.37	-1.97	19	-3.09278874836478	26.3592200931343
7.38	-1.78	19	-2.82777169419773	26.6405167586669
7.39	-1.58999999999999	19	-2.56005192589629	26.8997596119948
7.4	-1.39999999999999	19	-2.28985008086017	27.1369287938732
7.41	-1.21	19	-2.0173869951932	27.3520044226985
7.42	-1.02	19	-1.7428837039272	27.5449665944782
7.43	-0.830000000000005	19	-1.46656144124601	27.7157953828003
7.44	-0.640000000000009	19	-1.18864164070977	27.8644708388033
7.45	-0.449999999999996	19	-0.90934593547963	27.9909729911454
7.46	-0.260000000000002	19	-0.628896158542725	28.0952818459748
7.47	-0.0699999999999878	19	-0.3475143429373	28.1773773868985
7.48	0.120000000000008	19	-0.0654227219784622	28.2372395749524
7.49	0.310000000000004	19	0.217156270516131	28.2748483485702
7.5	0.5	19	0.5	28.2901836235535
7.51	0.689999999999996	19	0.782885630971388	28.2832252930402
7.52	0.879999999999992	19	1.06559012674648	28.2539532274749
7.53	1.07	19	1.34789024923232	28.2023472745773
7.54	1.26	19	1.62956255869934	28.1283872593117
7.55	1.45000000000001	19	1.9103834135538	28.0320529838564
7.56	1.64000000000001	19	2.19012897010959	27.9133242275722
7.57	1.83000000000001	19	2.46857518236006	27.7721807469723
7.58	2.02	19	2.7454978017493	27.6086022756906
7.59	2.21	19	3.02067237694321	27.4225685244508
7.6	2.39999999999999	19	3.29387425360027	27.2140591810358
7.61	2.58999999999999	19	3.56487857414192	26.9830539102559
7.62	2.78	19	3.83346027752277	26.729532353918
7.63	2.97	19	4.09939409900024	26.4534741307944
7.64	3.16000000000001	19	4.3624545699043	26.1548588365912
7.65	3.35000000000001	19	4.62241601740641	25.8336660439174
7.66	3.54	19	4.87905256428858	25.489875302253
7.67	3.73	19	5.1321381287118	25.1234661379182
7.68	3.91999999999999	19	5.38144642398437	24.7344180540413
7.69	4.10999999999999	19	5.62675095832975	24.3227105305276
7.7	4.3	19	5.86782503465423	23.8883230240275
7.71	4.49	19	6.10444175031416	23.4312349679053
7.72	4.68000000000001	19	6.33637399688302	22.9514257722068
7.73	4.87000000000001	19	6.56339445991797	22.4488748236286
7.74	5.06	19	6.7852756187263	21.9235614854853
7.75	5.25	19	7.00178974613139	21.3754650976783
7.76	5.44	19	7.21270890823844	20.8045649766635
7.77	5.62999999999999	19	7.41780496419984	20.2108404154197
7.78	5.82	19	7.61684956598026	19.5942706834167
7.79	6.01	19	7.80961415812136	18.9548350265828
7.8	6.20000000000001	19	7.9958699775063	18.292512667273
7.81	6.39000000000001	19	8.17538805312371	17.607282804237
7.82	6.58000000000001	19	8.34793920583161	16.8991246125867
7.83	6.77	19	8.51329404812083	16.1680172437642
7.84	6.96	19	8.67122298387808	15.4139398255092
7.85	7.14999999999999	19	8.82149620814889	14.6368714618271
7.86	7.33999999999999	19	8.96388370689997	13.8367912329564
7.87	7.53	19	9.0981552567815	13.0136781953359
7.88	7.72	19	9.22408042488887	12.167511381573
7.89	7.91000000000001	19	9.34142856852429	11.2982698004104
7.9	8.10000000000001	19	9.4499688349579	10.4059324366938
7.91	8.29	19	9.5494701611887	9.49047825133934
7.92	8.48	19	9.63970127370506	8.55188618130064
7.93	8.66999999999999	19	9.72043068824496	7.59013513953625
7.94	8.85999999999999	19	9.79142670955581	6.60520401497679
7.95	9.05	19	9.85245743115409	5.59707167249198
7.96	9.24	19	9.9032907350845	4.56571695285806
7.97	9.43000000000001	19	9.94369429167894	3.51111867272424
7.98	9.62000000000001	19	9.97343555931498	2.4332556245804
7.99	9.81	19	9.99228178417419	1.33210657672334
8	10	0	10	0
8.01	10	0	10.0020627654738	0.204904212620628
8.02	10	0	10.0040981121231	0.202166513215011
8.03	10	0	10.0061061237069	0.199437204453212
8.04	10	0	10.0080868842795	0.196716315910997
8.05	10	0	10.0100404781915	0.194003877294554
8.06	10	0	10.0119669900913	0.191299918441194
8.07	10	0	10.0138665049261	0.188604469320039
8.08	10	0	10.0157391079438	0.185917560032725
8.09	10	0	10.0175848846936	0.183239220814103
8.1	10	0	10.0194039210282	0.180569482032951
8.11	10	0	10.0211963031041	0.177908374192683
8.12	10	0	10.0229621173839	0.175255927932067
8.13	10	0	10.0247014506372	0.172612174025947
8.14	10	0	10.026414389942	0.169977143385968
8.15	10	0	10.028101022686	0.167350867061308
8.16	10	0	10.0297614365682	0.164733376239409
8.17	10	0	10.0313957196002	0.162124702246723
8.18	10	0	10.0330039601074	0.159524876549449
8.19	10	0	10.0345862467308	0.156933930754287
8.2	10	0	10.036142668428	0.15435189660919
8.21	10	0	10.0376733144748	0.15177880600412
8.22	10	0	10.0391782744666	0.149214690971815
8.23	10	0	10.04065763832	0.146659583688552
8.24	10	0	10.042111496274	0.144113516474925
8.25	10	0	10.0435399388913	0.141576521796618
8.26	10	0	10.0449430570604	0.139048632265191
8.27	10	0	10.0463209419963	0.136529880638858
8.28	10	0	10.0476736852425	0.134020299823294
8.29	10	0	10.0490013786723	0.131519922872425
8.3	10	0	10.0503041144902	0.129028782989221
8.31	10	0	10.0515819852335	0.126546913526522
8.32	10	0	10.0528350837738	0.124074347987833
8.33	10	0	10.0540635033185	0.121611120028153
8.34	10	0	10.0552673374123	0.119157263454792
8.35	10	0	10.0564466799387	0.116712812228201
8.36	10	0	10.0576016251217	0.114277800462804
8.37	10	0	10.058732267527	0.111852262427838
8.38	10	0	10.0598387020641	0.109436232548198
8.39	10	0	10.0609210239871	0.107029745405279
8.4	10	0	10.0619793288971	0.104632835737841
8.41	10	0	10.0630137127431	0.102245538442861
8.42	10	0	10.064024271824	0.0998678885764025
8.43	10	0	10.0650111027901	0.0974999213544854
8.44	10	0	10.0659743026445	0.0951416721539625
8.45	10	0	10.066913968745	0.0927931765134019
8.46	10	0	10.0678301988056	0.0904544701339747
8.47	10	0	10.0687230908981	0.0881255888803498
8.48	10	0	10.0695927434537	0.0858065687815914
8.49	10	0	10.0704392552649	0.0834974460320651
8.5	10	0	10.0712627254867	0.0811982569923489
8.51	10	0	10.0720632536388	0.0789090381901494
8.52	10	0	10.0728409396069	0.0766298263212254
8.53	10	0	10.0735958836445	0.0743606582503152
8.54	10	0	10.0743281863747	0.072101571012073
8.55	10	0	10.0750379487916	0.0698526018120074
8.56	10	0	10.0757252722623	0.0676137880274305
8.57	10	0	10.0763902585286	0.0653851672084085
8.58	10	0	10.0770330097087	0.0631667770787226
8.59	10	0	10.0776536282988	0.060958655536833
8.6	10	0	10.078252217175	0.058760840656851
8.61	10	0	10.078828879595	0.0565733706895169
8.62	10	0	10.0793837191999	0.0543962840631837
8.63	10	0	10.0799168400161	0.0522296193848085
8.64	10	0	10.0804283464567	0.0500734154409483
8.65	10	0	10.0809183433239	0.047927711198765
8.66	10	0	10.0813869358101	0.0457925458070344
8.67	10	0	10.0818342295006	0.0436679585971634
8.68	10	0	10.0822603303744	0.0415539890842131
8.69	10	0	10.0826653448069	0.0394506769679289
8.7	10	0	10.0830493795713	0.0373580621337774
8.71	10	0	10.0834125418407	0.0352761846539897
8.72	10	0	10.0837549391898	0.0332050847886124
8.73	10	0	10.0840766795967	0.0311448029865642
8.74	10	0	10.0843778714452	0.0290953798867004
8.75	10	0	10.0846586235262	0.0270568563188844
8.76	10	0	10.08491904504	0.0250292733050659
8.77	10	0	10.0851592455982	0.0230126720603673
8.78	10	0	10.0853793352251	0.0210070939941711
8.79	10	0	10.0855794243606	0.0190125807112316
8.8	10	0	10.0857596238613	0.0170291740127653
8.81	10	0	10.0859200450028	0.0150569158975823
8.82	10	0	10.0860607994818	0.0130958485631967
8.83	10	0	10.086181999418	0.0111460144069571
8.84	10	0	10.0862837573559	0.00920745602718892
8.85	10	0	10.0863661862671	0.00728021622433391
8.86	10	0	10.0864293995523	0.00536433800210349
8.87	10	0	10.0864735110431	0.00345986456863818
8.88	10	0	10.0864986350043	0.00156683933767525
8.89	10	0	10.0865048861359	-0.000314694070277437
8.9	10	0	10.086492379575	-0.00218469182675679
8.91	10	0	10.0864612308983	-0.00404310989415498
8.92	10	0	10.0864115561236	-0.00588990402452021
8.93	10	0	10.0863434717124	-0.00772502975834998
8.94	10	0	10.086257094572	-0.0095484424233761
8.95	10	0	10.0861525420571	-0.0113600971333419
8.96	10	0	10.0860299319728	-0.0131599487867708
8.97	10	0	10.0858893825759	-0.0149479520657267
8.98	10	0	10.0857310125778	-0.0167240614345665
8.99	10	0	10.0855549411461	-0.0184882311386835
9	10	0	10.0853612879072	-0.0202404152032429
9.01	10	0	10.0851501729481	-0.0219805674319088
9.02	10	0	10.0849217168193	-0.0237086414055614
9.03	10	0	10.0846760405363	-0.0254245904810079
9.04	10	0	10.0844132655822	-0.0271283677896817
9.05	10	0	10.0841335139101	-0.0288199262363363
9.06	10	0	10.0838369079448	-0.0304992184977261
9.07	10	0	10.0835235705859	-0.0321661970212827
9.08	10	0	10.0831936252094	-0.0338208140237789
9.09	10	0	10.0828471956705	-0.0354630214899848
9.1	10	0	10.0824844063055	-0.0370927711713154
9.11	10	0	10.0821053819346	-0.0387100145844677
9.12	10	0	10.0817102478638	-0.0403147030100496
9.13	10	0	10.0812991298875	-0.0419067874911983
9.14	10	0	10.080872154291	-0.0434862188321913
9.15	10	0	10.0804294478528	-0.0450529475970449
9.16	10	0	10.0799711378466	-0.046606924108106
9.17	10	0	10.0794973520445	-0.0481480984446325
9.18	10	0	10.0790082187188	-0.0496764204413645
9.19	10	0	10.0785038666448	-0.0511918396870861
9.2	10	0	10.0779844251031	-0.0526943055231759
9.21	10	0	10.077450023882	-0.05418376704215
9.22	10	0	10.0769007932804	-0.0556601730861919
9.23	10	0	10.0763368641099	-0.0571234722456754
9.24	10	0	10.0757583676973	-0.0585736128576743
9.25	10	0	10.0751654358876	-0.0600105430044644
9.26	10	0	10.074558201046	-0.0614342105120137
9.27	10	0	10.0739367960609	-0.0628445629484624
9.28	10	0	10.0733013543462	-0.0642415476225923
9.29	10	0	10.0726520098441	-0.0656251115822853
9.3	10	0	10.0719888970276	-0.0669952016129755
9.31	10	0	10.0713121509032	-0.0683517642360786
9.32	10	0	10.0706219070134	-0.0696947457074267
9.33	10	0	10.0699183014396	-0.07102409201568
9.34	10	0	10.0692014708045	-0.0723397488807312
9.35	10	0	10.0684715522752	-0.0736416617521029
9.36	10	0	10.0677286835655	-0.0749297758073216
9.37	10	0	10.0669730029388	-0.0762040359503002
9.38	10	0	10.0662046492108	-0.0774643868096853
9.39	10	0	10.0654237617523	-0.0787107727372166
9.4	10	0	10.0646304804922	-0.0799431378060552
9.41	10	0	10.0638249459197	-0.0811614258091122
9.42	10	0	10.0630072990878	-0.0823655802573613
9.43	10	0	10.0621776816156	-0.0835555443781397
9.44	10	0	10.0613362356915	-0.0847312611134376
9.45	10	0	10.0604831040758	-0.085892673118175
9.46	10	0	10.0596184301037	-0.0870397227584677
9.47	10	0	10.0587423576882	-0.088172352109879
9.48	10	0	10.0578550313231	-0.0892905029556614
9.49	10	0	10.0569565960858	-0.0903941167849835
9.5	10	0	10.0560471976401	-0.0914831347911463
9.51	10	0	10.0551269822397	-0.0925574978697858
9.52	10	0	10.0541960967305	-0.0936171466170632
9.53	10	0	10.0532546885541	-0.0946620213278421
9.54	10	0	10.0523029057507	-0.0956920619938527
9.55	10	0	10.051340896962	-0.0967072083018434
9.56	10	0	10.0503688114345	-0.0977073996317184
9.57	10	0	10.0493867990224	-0.0986925750546629
9.58	10	0	10.0483950101906	-0.0996626733312545
9.59	10	0	10.0473935960181	-0.100617632909561
9.6	10	0	10.0463827082009	-0.101557391923224
9.61	10	0	10.0453624990554	-0.102481888189532
9.62	10	0	10.0443331215212	-0.103391059207474
9.63	10	0	10.0432947291648	-0.104284842155785
9.64	10	0	10.0422474761821	-0.105163173890971
9.65	10	0	10.0411915174025	-0.106025990945329
9.66	10	0	10.0401270082914	-0.106873229524944
9.67	10	0	10.039054104954	-0.107704825507675
9.68	10	0	10.0379729641381	-0.108520714441127
9.69	10	0	10.0368837432379	-0.109320831540607
9.7	10	0	10.0357866002971	-0.110105111687069
9.71	10	0	10.0346816940121	-0.110873489425035
9.72	10	0	10.0335691837359	-0.111625898960516
9.73	10	0	10.0324492294807	-0.112362274158899
9.74	10	0	10.031321991922	-0.113082548542837
9.75	10	0	10.0301876324019	-0.113786655290109
9.76	10	0	10.0290463129322	-0.114474527231477
9.77	10	0	10.0278981961983	-0.115146096848517
9.78	10	0	10.0267434455625	-0.115801296271437
9.79	10	0	10.0255822250676	-0.116440057276886
9.8	10	0	10.0244146994403	-0.117062311285737
9.81	10	0	10.0232410340949	-0.117667989360859
9.82	10	0	10.0220613951371	-0.118257022204875
9.83	10	0	10.0208759493671	-0.118829340157896
9.84	10	0	10.0196848642836	-0.119384873195245
9.85	10	0	10.0184883080875	-0.119923550925165
9.86	10	0	10.0172864496853	-0.120445302586501
9.87	10	0	10.0160794586931	-0.120950057046381
9.88	10	0	10.01486750544	-0.121437742797857
9.89	10	0	10.0136507609724	-0.121908287957556
9.9	10	0	10.012429397057	-0.122361620263289
9.91	10	0	10.0112035861854	-0.12279766707166
9.92	10	0	10.0099735015773	-0.123216355355644
9.93	10	0	10.0087393171847	-0.123617611702159
9.94	10	0	10.0075012076957	-0.124001362309611
9.95	10	0	10.0062593485382	-0.124367532985424
9.96	10	0	10.0050139158843	-0.124716049143553
9.97	10	0	10.0037650866536	-0.12504683580198
9.98	10	0	10.0025130385178	-0.125359817580181
9.99	10	0	10.0012579499042	-0.12565491869659
10	10	0	10	-0.125932062966031
10.01	10	0	9.9987393687563	-0.126191173797138
10.02	10	0	9.99747623689214	-0.126432174189749
10.03	10	0	9.99621078589862	-0.126654986732295
10.04	10	0	9.99494319804303	-0.126859533599146
10.05	10	0	9.99367365637299	-0.127045736547963
10.06	10	0	9.99240234472063	-0.127213516917009
10.07	10	0	9.99112944770681	-0.127362795622451
10.08	10	0	9.98985515074537	-0.127493493155642
10.09	10	0	9.98857964004735	-0.127605529580375
10.1	10	0	9.9873031026253	-0.127698824530125
10.11	10	0	9.98602572629761	-0.127773297205264
10.12	10	0	9.98474769969284	-0.127828866370256
10.13	10	0	9.98346921225411	-0.127865450350835
10.14	10	0	9.98219045424345	-0.127882967031158
10.15	10	0	9.9809116167463	-0.127881333850934
10.16	10	0	9.97963289167594	-0.127860467802537
10.17	10	0	9.97835447177795	-0.127820285428094
10.18	10	0	9.97707655063478	-0.127760702816551
10.19	10	0	9.97579932267025	-0.127681635600716
10.2	10	0	9.97452298315415	-0.127582998954281
10.21	10	0	9.97324772820684	-0.127464707588819
10.22	10	0	9.9719737548039	-0.127326675750765
10.23	10	0	9.97070126078076	-0.127168817218361
10.24	10	0	9.96943044483744	-0.12699104529859
10.25	10	0	9.96816150654325	-0.126793272824081
10.26	10	0	9.96689464634157	-0.126575412149992
10.27	10	0	9.96563006555461	-0.126337375150863
10.28	10	0	9.96436796638828	-0.126079073217457
10.29	10	0	9.96310855193698	-0.125800417253565
10.3	10	0	9.96185202618854	-0.125501317672791
10.31	10	0	9.96059859402912	-0.125181684395315
10.32	10	0	9.95934846124816	-0.124841426844625
10.33	10	0	9.95810183454334	-0.12448045394423
10.34	10	0	9.95685892152565	-0.124098674114341
10.35	10	0	9.95561993072438	-0.123695995268533
10.36	10	0	9.95438507159226	-0.123272324810375
10.37	10	0	9.95315455451052	-0.12282756963004
10.38	10	0	9.9519285907941	-0.122361636100882
10.39	10	0	9.95070739269677	-0.121874430075993
10.4	10	0	9.94949117341641	-0.121365856884728
10.41	10	0	9.94828014710023	-0.120835821329205
10.42	10	0	9.94707452885007	-0.120284227680776
10.43	10	0	9.94587453472774	-0.119710979676478
10.44	10	0	9.94468038176034	-0.119115980515443
10.45	10	0	9.9434922879457	-0.11849913285529
10.46	10	0	9.94231047225779	-0.117860338808486
10.47	10	0	9.94113515465221	-0.117199499938681
10.48	10	0	9.93996655607167	-0.116516517257003
10.49	10	0	9.93880489845153	-0.115811291218344
10.5	10	0	9.93765040472544	-0.115083721717594
10.51	10	0	9.93650329883089	-0.114333708085864
10.52	10	0	9.93536380571489	-0.113561149086668
10.53	10	0	9.93423215133968	-0.112765942912078
10.54	10	0	9.93310856268844	-0.111947987178853
10.55	10	0	9.9319932677711	-0.111107178924527
10.56	10	0	9.93088649563011	-0.110243414603477
10.57	10	0	9.9297884763463	-0.109356590082955
10.58	10	0	9.9286994410448	-0.108446600639084
10.59	10	0	9.92761962190093	-0.107513340952833
10.6	10	0	9.92654925214621	-0.10655670510595
10.61	10	0	9.92548856607435	-0.105576586576863
10.62	10	0	9.92443779904731	-0.10457287823656
10.63	10	0	9.92339718750139	-0.103545472344419
10.64	10	0	9.92236696895338	-0.102494260544019
10.65	10	0	9.92134738200672	-0.101419133858908
10.66	10	0	9.92033866635775	-0.100319982688341
10.67	10	0	9.91934106280193	-0.0991966968029864
10.68	10	0	9.91835481324021	-0.0980491653405918
10.69	10	0	9.9173801606853	-0.0968772768016188
10.7	10	0	9.91641734926816	-0.0956809190448425
10.71	10	0	9.91546662424435	-0.0944599792829138
10.72	10	0	9.91452823200057	-0.0932143440778898
10.73	10	0	9.91360242006118	-0.091943899336719
10.74	10	0	9.91268943709473	-0.0906485303067041
10.75	10	0	9.91178953292065	-0.0893281215709139
10.76	10	0	9.91090295851587	-0.0879825570435668
10.77	10	0	9.91002996602154	-0.0866117199653732
10.78	10	0	9.90917080874978	-0.0852154928988429
10.79	10	0	9.90832574119052	-0.0837937577235498
10.8	10	0	9.90749501901829	-0.0823463956313615
10.81	10	0	9.90667889909921	-0.0808732871216298
10.82	10	0	9.90587763949783	-0.0793743119963399
10.83	10	0	9.90509149948425	-0.0778493493552235
10.84	10	0	9.90432073954106	-0.0762982775908262
10.85	10	0	9.9035656213705	-0.0747209743835402
10.86	10	0	9.90282640790158	-0.0731173166965934
10.87	10	0	9.90210336329728	-0.0714871807709962
10.88	10	0	9.90139675296183	-0.0698304421204525
10.89	10	0	9.90070684354794	-0.0681469755262188
10.9	10	0	9.90003390296422	-0.0664366550319311
10.91	10	0	9.89937820038255	-0.0646993539383826
10.92	10	0	9.89874000624552	-0.0629349447982605
10.93	10	0	9.89811959227399	-0.061143299410841
10.94	10	0	9.89751723147458	-0.0593242888166347
10.95	10	0	9.89693319814734	-0.0574777832919947
10.96	10	0	9.8963677678934	-0.0556036523436732
10.97	10	0	9.89582121762268	-0.0537017647033419
10.98	10	0	9.89529382556171	-0.0517719883220511
10.99	10	0	9.89478587126141	-0.0498141903646643
11	10	0	9.89429763560501	-0.0478282372042251
11.01	10	0	9.89382940081599	-0.0458139944162913
11.02	10	0	9.89338145046611	-0.0437713267732153
11.03	10	0	9.89295406948341	-0.0417000982383777
11.04	10	0	9.89254754416039	-0.0396001719603763
11.05	10	0	9.89216216216216	-0.0374714102671569
11.06	10	0	9.89179821253467	-0.0353136746601062
11.07	10	0	9.89145598571302	-0.033126825808086
11.08	10	0	9.8911357735298	-0.0309107235414248
11.09	10	0	9.89083786922352	-0.0286652268458502
11.1	10	0	9.8905625674471	-0.0263901938563768
11.11	10	0	9.89031016427635	-0.0240854818511404
11.12	10	0	9.89008095721865	-0.0217509472451775
11.13	10	0	9.88987524522154	-0.0193864455841594
11.14	10	0	9.88969332868148	-0.0169918315380612
11.15	10	0	9.88953550945267	-0.0145669588947887
11.16	10	0	9.88940209085581	-0.0121116805537442
11.17	10	0	9.88929337768714	-0.00962584851933949
11.18	10	0	9.8892096762273	-0.00710931389445635
11.19	10	0	9.88915129425046	-0.00456192687384342
11.2	10	0	9.88911854103343	-0.00198353673746563
11.21	10	0	9.8891117273648	0.000626008156209702
11.22	10	0	9.88913116555421	0.00326686037697991
11.23	10	0	9.88917716944168	0.00593917342973734
11.24	10	0	9.88925005440696	0.00864310176134909
11.25	10	0	9.88935013737905	0.0113788007676095
11.26	10	0	9.88947773684563	0.0141464268002403
11.27	10	0	9.88963317286275	0.0169461371739573
11.28	10	0	9.88981676706442	0.0197780901735972
11.29	10	0	9.8900288426724	0.0226424450613011
11.3	10	0	9.89026972450599	0.0255393620837724
11.31	10	0	9.89053973899188	0.0284690024795831
11.32	10	0	9.89083921417417	0.031431528486554
11.33	10	0	9.89116847972435	0.0344271033491903
11.34	10	0	9.89152786695142	0.037455891326193
11.35	10	0	9.89191770881208	0.0405180576980254
11.36	10	0	9.89233833992095	0.04361376877455
11.37	10	0	9.89279009656096	0.0467431919027344
11.38	10	0	9.8932733166937	0.0499064954744146
11.39	10	0	9.89378833996995	0.0531038489341435
11.4	10	0	9.89433550774024	0.0563354227870906
11.41	10	0	9.89491516306546	0.0596013886070217
11.42	10	0	9.89552765072765	0.0629019190443448
11.43	10	0	9.89617331724074	0.0662371878342288
11.44	10	0	9.89685251086148	0.0696073698047876
11.45	10	0	9.89756558160041	0.0730126408853481
11.46	10	0	9.89831288123289	0.0764531781147811
11.47	10	0	9.89909476331027	0.079929159649907
11.48	10	0	9.89991158317106	0.0834407647739837
11.49	10	0	9.90076369795231	0.0869881739052541
11.5	10	0	9.90165146660094	0.0905715686055906
11.51	10	0	9.90257524988524	0.0941911315891987
11.52	10	0	9.90353541040645	0.0978470467314056
11.53	10	0	9.90453231261042	0.101539499077528
11.54	10	0	9.9055663227993	0.10526867485181
11.55	10	0	9.90663780914345	0.109034761466459
11.56	10	0	9.90774714169333	0.112837947530741
11.57	10	0	9.90889469239148	0.116678422860176
11.58	10	0	9.91008083508471	0.120556378485798
11.59	10	0	9.91130594553623	0.124472006663515
11.6	10	0	9.91257040143799	0.128425500883544
11.61	10	0	9.91387458242302	0.132417055879932
11.62	10	0	9.91521887007799	0.136446867640162
11.63	10	0	9.91660364795573	0.140515133414844
11.64	10	0	9.91802930158794	0.144622051727511
11.65	10	0	9.91949621849795	0.148767822384469
11.66	10	0	9.92100478821363	0.15295264648477
11.67	10	0	9.92255540228032	0.157176726430258
11.68	10	0	9.92414845427395	0.161440265935713
11.69	10	0	9.92578433981421	0.165743470039074
11.7	10	0	9.92746345657782	0.170086545111779
11.71	10	0	9.92918620431191	0.174469698869179
11.72	10	0	9.93095298484756	0.178893140381053
11.73	10	0	9.93276420211335	0.183357080082224
11.74	10	0	9.93462026214905	0.187861729783258
11.75	10	0	9.93652157311951	0.192407302681285
11.76	10	0	9.93846854532849	0.196994013370897
11.77	10	0	9.94046159123275	0.201622077855157
11.78	10	0	9.94250112545616	0.206291713556705
11.79	10	0	9.94458756480396	0.211003139328966
11.8	10	0	9.94672132827713	0.215756575467471
11.81	10	0	9.94890283708687	0.220552243721271
11.82	10	0	9.95113251466916	0.225390367304457
11.83	10	0	9.95341078669952	0.230271170907799
11.84	10	0	9.95573808110781	0.235194880710471
11.85	10	0	9.95811482809316	0.240161724391913
11.86	10	0	9.96054146013905	0.24517193114378
11.87	10	0	9.96301841202849	0.250225731682015
11.88	10	0	9.96554612085931	0.255323358259018
11.89	10	0	9.96812502605963	0.260465044675966
11.9	10	0	9.97075556940333	0.265651026295195
11.91	10	0	9.97343819502582	0.270881540052744
11.92	10	0	9.97617334943975	0.276156824470995
11.93	10	0	9.978961481551	0.281477119671431
11.94	10	0	9.98180304267471	0.286842667387519
11.95	10	0	9.98469848655148	0.292253710977719
11.96	10	0	9.98764826936365	0.297710495438605
11.97	10	0	9.9906528497518	0.303213267418114
11.98	10	0	9.99371268883127	0.308762275228927
11.99	10	0	9.99682825020893	0.314357768861958
12	10	0.666666666666667	10	0.666666666666667
12.01	10.0066666666667	0.666666666666667	10.0032115555556	0.322311111111111
12.02	10.0133333333333	0.666666666666667	10.0064462222222	0.324622222222223
12.03	10.02	0.666666666666667	10.009704	0.326933333333334
12.04	10.0266666666667	0.666666666666667	10.0129848888889	0.329244444444444
12.05	10.0333333333333	0.666666666666667	10.0162888888889	0.331555555555555
12.06	10.04	0.666666666666667	10.019616	0.333866666666667
12.07	10.0466666666667	0.666666666666667	10.0229662222222	0.336177777777778
12.08	10.0533333333333	0.666666666666667	10.0263395555556	0.338488888888888
12.09	10.06	0.666666666666667	10.029736	0.3408
12.1	10.0666666666667	0.666666666666667	10.0331555555556	0.343111111111112
12.11	10.0733333333333	0.666666666666667	10.0365982222222	0.345422222222222
12.12	10.08	0.666666666666667	10.040064	0.347733333333333
12.13	10.0866666666667	0.666666666666667	10.0435528888889	0.350044444444445
12.14	10.0933333333333	0.666666666666667	10.0470648888889	0.352355555555556
12.15	10.1	0.666666666666667	10.0506	0.354666666666666
12.16	10.1066666666667	0.666666666666667	10.0541582222222	0.356977777777778
12.17	10.1133333333333	0.666666666666667	10.0577395555556	0.359288888888889
12.18	10.12	0.666666666666667	10.061344	0.361599999999999
12.19	10.1266666666667	0.666666666666667	10.0649715555556	0.363911111111111
12.2	10.1333333333333	0.666666666666667	10.0686222222222	0.366222222222222
12.21	10.14	0.666666666666667	10.072296	0.368533333333334
12.22	10.1466666666667	0.666666666666667	10.0759928888889	0.370844444444445
12.23	10.1533333333333	0.666666666666667	10.0797128888889	0.373155555555556
12.24	10.16	0.666666666666667	10.083456	0.375466666666667
12.25	10.1666666666667	0.666666666666667	10.0872222222222	0.377777777777777
12.26	10.1733333333333	0.666666666666667	10.0910115555556	0.380088888888889
12.27	10.18	0.666666666666667	10.094824	0.3824
12.28	10.1866666666667	0.666666666666667	10.0986595555556	0.384711111111111
12.29	10.1933333333333	0.666666666666667	10.1025182222222	0.387022222222222
12.3	10.2	0.666666666666667	10.1064	0.389333333333334
12.31	10.2066666666667	0.666666666666667	10.1103048888889	0.391644444444445
12.32	10.2133333333333	0.666666666666667	10.1142328888889	0.393955555555556
12.33	10.22	0.666666666666667	10.118184	0.396266666666666
12.34	10.2266666666667	0.666666666666667	10.1221582222222	0.398577777777777
12.35	10.2333333333333	0.666666666666667	10.1261555555556	0.400888888888889
12.36	10.24	0.666666666666667	10.130176	0.4032
12.37	10.2466666666667	0.666666666666667	10.1342195555556	0.405511111111111
12.38	10.2533333333333	0.666666666666667	10.1382862222222	0.407822222222222
12.39	10.26	0.666666666666667	10.142376	0.410133333333333
12.4	10.2666666666667	0.666666666666667	10.1464888888889	0.412444444444445
12.41	10.2733333333333	0.666666666666667	10.1506248888889	0.414755555555556
12.42	10.28	0.666666666666667	10.154784	0.417066666666667
12.43	10.2866666666667	0.666666666666667	10.1589662222222	0.419377777777778
12.44	10.2933333333333	0.666666666666667	10.1631715555556	0.421688888888889
12.45	10.3	0.666666666666667	10.1674	0.423999999999999
12.46	10.3066666666667	0.666666666666667	10.1716515555556	0.426311111111111
12.47	10.3133333333333	0.666666666666667	10.1759262222222	0.428622222222222
12.48	10.32	0.666666666666667	10.180224	0.430933333333333
12.49	10.3266666666667	0.666666666666667	10.1845448888889	0.433244444444444
12.5	10.3333333333333	0.666666666666667	10.1888888888889	0.435555555555556
12.51	10.34	0.666666666666667	10.193256	0.437866666666667
12.52	10.3466666666667	0.666666666666667	10.1976462222222	0.440177777777778
12.53	10.3533333333333	0.666666666666667	10.2020595555556	0.442488888888889
12.54	10.36	0.666666666666667	10.206496	0.444799999999999
12.55	10.3666666666667	0.666666666666667	10.2109555555556	0.447111111111111
12.56	10.3733333333333	0.666666666666667	10.2154382222222	0.449422222222222
12.57	10.38	0.666666666666667	10.219944	0.451733333333334
12.58	10.3866666666667	0.666666666666667	10.2244728888889	0.454044444444444
12.59	10.3933333333333	0.666666666666667	10.2290248888889	0.456355555555555
12.6	10.4	0.666666666666667	10.2336	0.458666666666667
12.61	10.4066666666667	0.666666666666667	10.2381982222222	0.460977777777778
12.62	10.4133333333333	0.666666666666667	10.2428195555556	0.46328888888889
12.63	10.42	0.666666666666667	10.247464	0.4656
12.64	10.4266666666667	0.666666666666667	10.2521315555556	0.467911111111111
12.65	10.4333333333333	0.666666666666667	10.2568222222222	0.470222222222222
12.66	10.44	0.666666666666667	10.261536	0.472533333333333
12.67	10.4466666666667	0.666666666666667	10.2662728888889	0.474844444444445
12.68	10.4533333333333	0.666666666666667	10.2710328888889	0.477155555555556
12.69	10.46	0.666666666666667	10.275816	0.479466666666667
12.7	10.4666666666667	0.666666666666667	10.2806222222222	0.481777777777778
12.71	10.4733333333333	0.666666666666667	10.2854515555556	0.48408888888889
12.72	10.48	0.666666666666667	10.290304	0.486399999999999
12.73	10.4866666666667	0.666666666666667	10.2951795555556	0.488711111111111
12.74	10.4933333333333	0.666666666666667	10.3000782222222	0.491022222222222
12.75	10.5	0.666666666666667	10.305	0.493333333333333
12.76	10.5066666666667	0.666666666666667	10.3099448888889	0.495644444444444
12.77	10.5133333333333	0.666666666666667	10.3149128888889	0.497955555555556
12.78	10.52	0.666666666666667	10.319904	0.500266666666667
12.79	10.5266666666667	0.666666666666667	10.3249182222222	0.502577777777777
12.8	10.5333333333333	0.666666666666667	10.3299555555556	0.504888888888889
12.81	10.54	0.666666666666667	10.335016	0.507200000000001
12.82	10.5466666666667	0.666666666666667	10.3400995555556	0.509511111111111
12.83	10.5533333333333	0.666666666666667	10.3452062222222	0.511822222222223
12.84	10.56	0.666666666666667	10.350336	0.514133333333333
12.85	10.5666666666667	0.666666666666667	10.3554888888889	0.516444444444445
12.86	10.5733333333333	0.666666666666667	10.3606648888889	0.518755555555555
12.87	10.58	0.666666666666667	10.365864	0.521066666666667
12.88	10.5866666666667	0.666666666666667	10.3710862222222	0.523377777777778
12.89	10.5933333333333	0.666666666666667	10.3763315555556	0.525688888888889
12.9	10.6	0.666666666666667	10.3816	0.528
12.91	10.6066666666667	0.666666666666667	10.3868915555556	0.530311111111111
12.92	10.6133333333333	0.666666666666667	10.3922062222222	0.532622222222223
12.93	10.62	0.666666666666667	10.397544	0.534933333333333
12.94	10.6266666666667	0.666666666666667	10.4029048888889	0.537244444444445
12.95	10.6333333333333	0.666666666666667	10.4082888888889	0.539555555555555
12.96	10.64	0.666666666666667	10.413696	0.541866666666667
12.97	10.6466666666667	0.666666666666667	10.4191262222222	0.544177777777778
12.98	10.6533333333333	0.666666666666667	10.4245795555556	0.546488888888889
12.99	10.66	0.666666666666667	10.430056	0.5488
13	10.6666666666667	0.666666666666667	10.4355555555556	0.551111111111111
13.01	10.6733333333333	0.666666666666667	10.4410782222222	0.553422222222222
13.02	10.68	0.666666666666667	10.446624	0.555733333333334
13.03	10.6866666666667	0.666666666666667	10.4521928888889	0.558044444444444
13.04	10.6933333333333	0.666666666666667	10.4577848888889	0.560355555555556
13.05	10.7	0.666666666666667	10.4634	0.562666666666667
13.06	10.7066666666667	0.666666666666667	10.4690382222222	0.564977777777778
13.07	10.7133333333333	0.666666666666667	10.4746995555556	0.567288888888889
13.08	10.72	0.666666666666667	10.480384	0.569600000000001
13.09	10.7266666666667	0.666666666666667	10.4860915555556	0.571911111111111
13.1	10.7333333333333	0.666666666666667	10.4918222222222	0.574222222222222
13.11	10.74	0.666666666666667	10.497576	0.576533333333333
13.12	10.7466666666667	0.666666666666667	10.5033528888889	0.578844444444445
13.13	10.7533333333333	0.666666666666667	10.5091528888889	0.581155555555555
13.14	10.76	0.666666666666667	10.514976	0.583466666666667
13.15	10.7666666666667	0.666666666666667	10.5208222222222	0.585777777777778
13.16	10.7733333333333	0.666666666666667	10.5266915555556	0.588088888888889
13.17	10.78	0.666666666666667	10.532584	0.5904
13.18	10.7866666666667	0.666666666666667	10.5384995555556	0.592711111111112
13.19	10.7933333333333	0.666666666666667	10.5444382222222	0.595022222222222
13.2	10.8	0.666666666666667	10.5504	0.597333333333333
13.21	10.8066666666667	0.666666666666667	10.5563848888889	0.599644444444445
13.22	10.8133333333333	0.666666666666667	10.5623928888889	0.601955555555556
13.23	10.82	0.666666666666667	10.568424	0.604266666666667
13.24	10.8266666666667	0.666666666666667	10.5744782222222	0.606577777777778
13.25	10.8333333333333	0.666666666666667	10.5805555555556	0.608888888888889
13.26	10.84	0.666666666666667	10.586656	0.6112
13.27	10.8466666666667	0.666666666666667	10.5927795555556	0.613511111111111
13.28	10.8533333333333	0.666666666666667	10.5989262222222	0.615822222222222
13.29	10.86	0.666666666666667	10.605096	0.618133333333333
13.3	10.8666666666667	0.666666666666667	10.6112888888889	0.620444444444445
13.31	10.8733333333333	0.666666666666667	10.6175048888889	0.622755555555556
13.32	10.88	0.666666666666667	10.623744	0.625066666666667
13.33	10.8866666666667	0.666666666666667	10.6300062222222	0.627377777777778
13.34	10.8933333333333	0.666666666666667	10.6362915555556	0.629688888888889
13.35	10.9	0.666666666666667	10.6426	0.632
13.36	10.9066666666667	0.666666666666667	10.6489315555556	0.634311111111111
13.37	10.9133333333333	0.666666666666667	10.6552862222222	0.636622222222223
13.38	10.92	0.666666666666667	10.661664	0.638933333333333
13.39	10.9266666666667	0.666666666666667	10.6680648888889	0.641244444444444
13.4	10.9333333333333	0.666666666666667	10.6744888888889	0.643555555555556
13.41	10.94	0.666666666666667	10.680936	0.645866666666666
13.42	10.9466666666667	0.666666666666667	10.6874062222222	0.648177777777778
13.43	10.9533333333333	0.666666666666667	10.6938995555556	0.650488888888889
13.44	10.96	0.666666666666667	10.700416	0.6528
13.45	10.9666666666667	0.666666666666667	10.7069555555556	0.655111111111111
13.46	10.9733333333333	0.666666666666667	10.7135182222222	0.657422222222223
13.47	10.98	0.666666666666667	10.720104	0.659733333333334
13.48	10.9866666666667	0.666666666666667	10.7267128888889	0.662044444444445
13.49	10.9933333333333	0.666666666666667	10.7333448888889	0.664355555555555
13.5	11	0.666666666666667	10.74	0.666666666666667
13.51	11.0066666666667	0.666666666666667	10.7466782222222	0.668977777777777
13.52	11.0133333333333	0.666666666666667	10.7533795555556	0.671288888888889
13.53	11.02	0.666666666666667	10.760104	0.6736
13.54	11.0266666666667	0.666666666666667	10.7668515555556	0.675911111111111
13.55	11.0333333333333	0.666666666666667	10.7736222222222	0.678222222222222
13.56	11.04	0.666666666666667	10.780416	0.680533333333334
13.57	11.0466666666667	0.666666666666667	10.7872328888889	0.682844444444444
13.58	11.0533333333333	0.666666666666667	10.7940728888889	0.685155555555556
13.59	11.06	0.666666666666667	10.800936	0.687466666666667
13.6	11.0666666666667	0.666666666666667	10.8078222222222	0.689777777777778
13.61	11.0733333333333	0.666666666666667	10.8147315555556	0.692088888888888
13.62	11.08	0.666666666666667	10.821664	0.6944
13.63	11.0866666666667	0.666666666666667	10.8286195555556	0.696711111111111
13.64	11.0933333333333	0.666666666666667	10.8355982222222	0.699022222222222
13.65	11.1	0.666666666666667	10.8426	0.701333333333334
13.66	11.1066666666667	0.666666666666667	10.8496248888889	0.703644444444444
13.67	11.1133333333333	0.666666666666667	10.8566728888889	0.705955555555555
13.68	11.12	0.666666666666667	10.863744	0.708266666666667
13.69	11.1266666666667	0.666666666666667	10.8708382222222	0.710577777777778
13.7	11.1333333333333	0.666666666666667	10.8779555555556	0.712888888888889
13.71	11.14	0.666666666666667	10.885096	0.715200000000001
13.72	11.1466666666667	0.666666666666667	10.8922595555556	0.717511111111111
13.73	11.1533333333333	0.666666666666667	10.8994462222222	0.719822222222222
13.74	11.16	0.666666666666667	10.906656	0.722133333333333
13.75	11.1666666666667	0.666666666666667	10.9138888888889	0.724444444444445
13.76	11.1733333333333	0.666666666666667	10.9211448888889	0.726755555555555
13.77	11.18	0.666666666666667	10.928424	0.729066666666667
13.78	11.1866666666667	0.666666666666667	10.9357262222222	0.731377777777778
13.79	11.1933333333333	0.666666666666667	10.9430515555556	0.733688888888889
13.8	11.2	0.666666666666667	10.9504	0.736
13.81	11.2066666666667	0.666666666666667	10.9577715555556	0.738311111111112
13.82	11.2133333333333	0.666666666666667	10.9651662222222	0.740622222222222
13.83	11.22	0.666666666666667	10.972584	0.742933333333334
13.84	11.2266666666667	0.666666666666667	10.9800248888889	0.745244444444445
13.85	11.2333333333333	0.666666666666667	10.9874888888889	0.747555555555556
13.86	11.24	0.666666666666667	10.994976	0.749866666666666
13.87	11.2466666666667	0.666666666666667	11.0024862222222	0.752177777777778
13.88	11.2533333333333	0.666666666666667	11.0100195555556	0.754488888888889
13.89	11.26	0.666666666666667	11.017576	0.7568
13.9	11.2666666666667	0.666666666666667	11.0251555555556	0.759111111111111
13.91	11.2733333333333	0.666666666666667	11.0327582222222	0.761422222222222
13.92	11.28	0.666666666666667	11.040384	0.763733333333333
13.93	11.2866666666667	0.666666666666667	11.0480328888889	0.766044444444445
13.94	11.2933333333333	0.666666666666667	11.0557048888889	0.768355555555555
13.95	11.3	0.666666666666667	11.0634	0.770666666666667
13.96	11.3066666666667	0.666666666666667	11.0711182222222	0.772977777777778
13.97	11.3133333333333	0.666666666666667	11.0788595555556	0.775288888888889
13.98	11.32	0.666666666666667	11.086624	0.7776
13.99	11.3266666666667	0.666666666666667	11.0944115555556	0.779911111111111
14	11.3333333333333	0.666666666666667	11.1022222222222	0.782222222222222
14.01	11.34	0.666666666666667	11.110056	0.784533333333333
14.02	11.3466666666667	0.666666666666667	11.1179128888889	0.786844444444444
14.03	11.3533333333333	0.666666666666667	11.1257928888889	0.789155555555555
14.04	11.36	0.666666666666667	11.133696	0.791466666666666
14.05	11.3666666666667	0.666666666666667	11.1416222222222	0.793777777777778
14.06	11.3733333333333	0.666666666666667	11.1495715555556	0.796088888888889
14.07	11.38	0.666666666666667	11.157544	0.7984
14.08	11.3866666666667	0.666666666666667	11.1655395555556	0.800711111111111
14.09	11.3933333333333	0.666666666666667	11.1735582222222	0.803022222222223
14.1	11.4	0.666666666666667	11.1816	0.805333333333333
14.11	11.4066666666667	0.666666666666667	11.1896648888889	0.807644444444444
14.12	11.4133333333333	0.666666666666667	11.1977528888889	0.809955555555556
14.13	11.42	0.666666666666667	11.205864	0.812266666666667
14.14	11.4266666666667	0.666666666666667	11.2139982222222	0.814577777777777
14.15	11.4333333333333	0.666666666666667	11.2221555555556	0.816888888888889
14.16	11.44	0.666666666666667	11.230336	0.8192
14.17	11.4466666666667	0.666666666666667	11.2385395555556	0.821511111111111
14.18	11.4533333333333	0.666666666666667	11.2467662222222	0.823822222222222
14.19	11.46	0.666666666666667	11.255016	0.826133333333334
14.2	11.4666666666667	0.666666666666667	11.2632888888889	0.828444444444444
14.21	11.4733333333333	0.666666666666667	11.2715848888889	0.830755555555556
14.22	11.48	0.666666666666667	11.279904	0.833066666666667
14.23	11.4866666666667	0.666666666666667	11.2882462222222	0.835377777777778
14.24	11.4933333333333	0.666666666666667	11.2966115555556	0.837688888888888
14.25	11.5	0.666666666666667	11.305	0.84
14.26	11.5066666666667	0.666666666666667	11.3134115555556	0.842311111111111
14.27	11.5133333333333	0.666666666666667	11.3218462222222	0.844622222222222
14.28	11.52	0.666666666666667	11.330304	0.846933333333333
14.29	11.5266666666667	0.666666666666667	11.3387848888889	0.849244444444444
14.3	11.5333333333333	0.666666666666667	11.3472888888889	0.851555555555555
14.31	11.54	0.666666666666667	11.355816	0.853866666666667
14.32	11.5466666666667	0.666666666666667	11.3643662222222	0.856177777777778
14.33	11.5533333333333	0.666666666666667	11.3729395555556	0.858488888888889
14.34	11.56	0.666666666666667	11.381536	0.860799999999999
14.35	11.5666666666667	0.666666666666667	11.3901555555556	0.863111111111111
14.36	11.5733333333333	0.666666666666667	11.3987982222222	0.865422222222223
14.37	11.58	0.666666666666667	11.407464	0.867733333333334
14.38	11.5866666666667	0.666666666666667	11.4161528888889	0.870044444444445
14.39	11.5933333333333	0.666666666666667	11.4248648888889	0.872355555555556
14.4	11.6	0.666666666666667	11.4336	0.874666666666667
14.41	11.6066666666667	0.666666666666667	11.4423582222222	0.876977777777777
14.42	11.6133333333333	0.666666666666667	11.4511395555556	0.879288888888889
14.43	11.62	0.666666666666667	11.459944	0.8816
14.44	11.6266666666667	0.666666666666667	11.4687715555556	0.883911111111111
14.45	11.6333333333333	0.666666666666667	11.4776222222222	0.886222222222222
14.46	11.64	0.666666666666667	11.486496	0.888533333333334
14.47	11.6466666666667	0.666666666666667	11.4953928888889	0.890844444444445
14.48	11.6533333333333	0.666666666666667	11.5043128888889	0.893155555555556
14.49	11.66	0.666666666666667	11.513256	0.895466666666667
14.5	11.6666666666667	0.666666666666667	11.5222222222222	0.897777777777778
14.51	11.6733333333333	0.666666666666667	11.5312115555556	0.900088888888888
14.52	11.68	0.666666666666667	11.540224	0.9024
14.53	11.6866666666667	0.666666666666667	11.5492595555556	0.904711111111111
14.54	11.6933333333333	0.666666666666667	11.5583182222222	0.907022222222223
14.55	11.7	0.666666666666667	11.5674	0.909333333333333
14.56	11.7066666666667	0.666666666666667	11.5765048888889	0.911644444444445
14.57	11.7133333333333	0.666666666666667	11.5856328888889	0.913955555555556
14.58	11.72	0.666666666666667	11.594784	0.916266666666667
14.59	11.7266666666667	0.666666666666667	11.6039582222222	0.918577777777777
14.6	11.7333333333333	0.666666666666667	11.6131555555556	0.920888888888889
14.61	11.74	0.666666666666667	11.622376	0.923199999999999
14.62	11.7466666666667	0.666666666666667	11.6316195555556	0.925511111111111
14.63	11.7533333333333	0.666666666666667	11.6408862222222	0.927822222222222
14.64	11.76	0.666666666666667	11.650176	0.930133333333334
14.65	11.7666666666667	0.666666666666667	11.6594888888889	0.932444444444444
14.66	11.7733333333333	0.666666666666667	11.6688248888889	0.934755555555556
14.67	11.78	0.666666666666667	11.678184	0.937066666666667
14.68	11.7866666666667	0.666666666666667	11.6875662222222	0.939377777777778
14.69	11.7933333333333	0.666666666666667	11.6969715555556	0.941688888888888
14.7	11.8	0.666666666666667	11.7064	0.944
14.71	11.8066666666667	0.666666666666667	11.7158515555556	0.946311111111111
14.72	11.8133333333333	0.666666666666667	11.7253262222222	0.948622222222222
14.73	11.82	0.666666666666667	11.734824	0.950933333333333
14.74	11.8266666666667	0.666666666666667	11.7443448888889	0.953244444444445
14.75	11.8333333333333	0.666666666666667	11.7538888888889	0.955555555555555
14.76	11.84	0.666666666666667	11.763456	0.957866666666667
14.77	11.8466666666667	0.666666666666667	11.7730462222222	0.960177777777778
14.78	11.8533333333333	0.666666666666667	11.7826595555556	0.962488888888889
14.79	11.86	0.666666666666667	11.792296	0.964799999999999
14.8	11.8666666666667	0.666666666666667	11.8019555555556	0.967111111111111
14.81	11.8733333333333	0.666666666666667	11.8116382222222	0.969422222222222
14.82	11.88	0.666666666666667	11.821344	0.971733333333333
14.83	11.8866666666667	0.666666666666667	11.8310728888889	0.974044444444444
14.84	11.8933333333333	0.666666666666667	11.8408248888889	0.976355555555556
14.85	11.9	0.666666666666667	11.8506	0.978666666666666
14.86	11.9066666666667	0.666666666666667	11.8603982222222	0.980977777777778
14.87	11.9133333333333	0.666666666666667	11.8702195555556	0.983288888888889
14.88	11.92	0.666666666666667	11.880064	0.985600000000001
14.89	11.9266666666667	0.666666666666667	11.8899315555556	0.98791111111111
14.9	11.9333333333333	0.666666666666667	11.8998222222222	0.990222222222222
14.91	11.94	0.666666666666667	11.909736	0.992533333333333
14.92	11.9466666666667	0.666666666666667	11.9196728888889	0.994844444444444
14.93	11.9533333333333	0.666666666666667	11.9296328888889	0.997155555555555
14.94	11.96	0.666666666666667	11.939616	0.999466666666667
14.95	11.9666666666667	0.666666666666667	11.9496222222222	1.00177777777778
14.96	11.9733333333333	0.666666666666667	11.9596515555556	1.00408888888889
14.97	11.98	0.666666666666667	11.969704	1.0064
14.98	11.9866666666667	0.666666666666667	11.9797795555556	1.00871111111111
14.99	11.9933333333333	0.666666666666667	11.9898782222222	1.01102222222222
15	12	0.666666666666667	12	0.666666666666667

5	-10	0.5	-10	0.5
5.01	-9.995	0.5	-10.019792358804	-1.9585407445834
5.02	-9.99	0.5	-10.0391721854305	-1.91749265383097
5.03	-9.985	0.5	-10.0581435643564	-1.87685030879325
5.04	-9.98	0.5	-10.0767105263158	-1.83660837950139
5.05	-9.975	0.5	-10.0948770491803	-1.79676162321956
5.06	-9.97	0.5	-10.1126470588235	-1.75730488273741
5.07	-9.965	0.5	-10.1300244299674	-1.71823308470116
5.08	-9.96	0.5	-10.147012987013	-1.6795412379828
5.09	-9.955	0.5	-10.1636165048544	-1.64122443208597
5.1	-9.95	0.5	-10.1798387096774	-1.60327783558793
5.11	-9.945	0.5	-10.1956832797428	-1.56569669461647
5.12	-9.94	0.5	-10.2111538461538	-1.52847633136095
5.13	-9.935	0.5	-10.2262539936102	-1.49161214261654
5.14	-9.93	0.5	-10.2409872611465	-1.45509959836099
5.15	-9.925	0.5	-10.2553571428571	-1.41893424036281
5.16	-9.92	0.5	-10.2693670886076	-1.38311168082038
5.17	-9.915	0.5	-10.2830205047319	-1.34762760103096
5.18	-9.91	0.5	-10.296320754717	-1.312477750089
5.19	-9.905	0.5	-10.3092711598746	-1.27765794361297
5.2	-9.9	0.5	-10.321875	-1.2431640625
5.21	-9.895	0.5	-10.3341355140187	-1.20899205170757
5.22	-9.89	0.5	-10.3460559006211	-1.17513791906177
5.23	-9.885	0.5	-10.3576393188855	-1.14159773409119
5.24	-9.88	0.5	-10.3688888888889	-1.10836762688614
5.25	-9.875	0.5	-10.3798076923077	-1.07544378698225
5.26	-9.87	0.5	-10.3903987730061	-1.04282246226806
5.27	-9.865	0.5	-10.4006651376147	-1.010499957916
5.28	-9.86	0.5	-10.4106097560976	-0.978472635336108
5.29	-9.855	0.5	-10.42023556231	-0.946736911151967
5.3	-9.85	0.5	-10.4295454545455	-0.915289256198347
5.31	-9.845	0.5	-10.4385422960725	-0.884126194540029
5.32	-9.84	0.5	-10.4472289156627	-0.853244302511249
5.33	-9.835	0.5	-10.4556081081081	-0.822640207775343
5.34	-9.83	0.5	-10.4636826347305	-0.79231058840403
5.35	-9.825	0.5	-10.4714552238806	-0.762252171975942
5.36	-9.82	0.5	-10.4789285714286	-0.732461734693877
5.37	-9.815	0.5	-10.4861053412463	-0.702936100520388
5.38	-9.81	0.5	-10.4929881656805	-0.673672140331221
5.39	-9.805	0.5	-10.4995796460177	-0.644666771086225
5.4	-9.8	0.5	-10.5058823529412	-0.6159169550173
5.41	-9.795	0.5	-10.5118988269795	-0.587419698832999
5.42	-9.79	0.5	-10.5176315789474	-0.559172052939366
5.43	-9.785	0.5	-10.523083090379	-0.531171110676675
5.44	-9.78	0.5	-10.5282558139535	-0.503414007571659
5.45	-9.775	0.5	-10.533152173913	-0.475897920604915
5.46	-9.77	0.5	-10.537774566474	-0.448620067493067
5.47	-9.765	0.5	-10.5421253602305	-0.421577705985434
5.48	-9.76	0.5	-10.5462068965517	-0.394768133174791
5.49	-9.755	0.5	-10.5500214899713	-0.368188684821963
5.5	-9.75	0.5	-10.5535714285714	-0.341836734693878
5.51	-9.745	0.5	-10.556858974359	-0.315709693914823
5.52	-9.74	0.5	-10.5598863636364	-0.28980501033058
5.53	-9.735	0.5	-10.5626558073654	-0.264120167885144
5.54	-9.73	0.5	-10.5651694915254	-0.238652686009767
5.55	-9.725	0.5	-10.5674295774648	-0.213400119024003
5.56	-9.72	0.5	-10.5694382022472	-0.18836005554854
5.57	-9.715	0.5	-10.5711974789916	-0.163530117929524
5.58	-9.71	0.5	-10.5727094972067	-0.138907961674105
5.59	-9.705	0.5	-10.5739763231198	-0.114491274896998
5.6	-9.7	0.5	-10.575	-0.0902777777777785
5.61	-9.695	0.5	-10.5757825484765	-0.0662652220286822
5.62	-9.69	0.5	-10.5763259668508	-0.0424513903726988
5.63	-9.685	0.5	-10.576632231405	-0.0188340960316921
5.64	-9.68	0.5	-10.5767032967033	0.00458881777563031
5.65	-9.675	0.5	-10.5765410958904	0.0278194783261407
5.66	-9.67	0.5	-10.5761475409836	0.0508599838753027
5.67	-9.665	0.5	-10.5755245231608	0.0737124041309979
5.68	-9.66	0.5	-10.5746739130435	0.0963787807183358
5.69	-9.655	0.5	-10.5735975609756	0.118861127635668
5.7	-9.65	0.5	-10.5722972972973	0.141161431701973
5.71	-9.645	0.5	-10.5707749326146	0.163281652995837
5.72	-9.64	0.5	-10.5690322580645	0.18522372528616
5.73	-9.635	0.5	-10.5670710455764	0.206989556454802
5.74	-9.63	0.5	-10.5648930481283	0.228581028911322
5.75	-9.625	0.5	-10.5625	0.25
5.76	-9.62	0.5	-10.5598936170213	0.271248302399275
5.77	-9.615	0.5	-10.557075596817	0.292327744513786
5.78	-9.61	0.5	-10.5540476190476	0.313240110859159
5.79	-9.605	0.5	-10.5508113456464	0.333987162439693
5.8	-9.6	0.5	-10.5473684210526	0.354570637119113
5.81	-9.595	0.5	-10.5437204724409	0.374992249984501
5.82	-9.59	0.5	-10.5398691099476	0.395253693703572
5.83	-9.585	0.5	-10.535815926893	0.415356638875444
5.84	-9.58	0.5	-10.5315625	0.435302734375
5.85	-9.575	0.5	-10.5271103896104	0.45509360769101
5.86	-9.57	0.5	-10.5224611398964	0.474730865258128
5.87	-9.565	0.5	-10.5176162790698	0.494216092782886
5.88	-9.56	0.5	-10.5125773195876	0.513550855563822
5.89	-9.555	0.5	-10.5073457583548	0.532736698805849
5.9	-9.55	0.5	-10.5019230769231	0.551775147928995
5.91	-9.545	0.5	-10.496310741688	0.570667708871606
5.92	-9.54	0.5	-10.4905102040816	0.589415868388171
5.93	-9.535	0.5	-10.4845229007634	0.608021094341821
5.94	-9.53	0.5	-10.4783502538071	0.626484835991652
5.95	-9.525	0.5	-10.4719936708861	0.644808524274956
5.96	-9.52	0.5	-10.4654545454545	0.662993572084481
5.97	-9.515	0.5	-10.458734256927	0.681041374540794
5.98	-9.51	0.5	-10.4518341708543	0.698953309259868
5.99	-9.505	0.5	-10.4447556390977	0.716730736615976
6	-9.5	0.5	-10.4375	0.734375
6.01	-9.495	0.5	-10.4300685785536	0.751887426073221
6.02	-9.49	0.5	-10.4224626865672	0.769269325016707
6.03	-9.485	0.5	-10.4146836228288	0.786521990776374
6.04	-9.48	0.5	-10.4067326732673	0.803646701303794
6.05	-9.475	0.5	-10.3986111111111	0.820644718792866
6.06	-9.47	0.5	-10.3903201970443	0.837517289912398
6.07	-9.465	0.5	-10.3818611793612	0.854265646034688
6.08	-9.46	0.5	-10.3732352941176	0.870891003460208
6.09	-9.455	0.5	-10.3644437652812	0.887394563638429
6.1	-9.45	0.5	-10.355487804878	0.903777513384889
6.11	-9.445	0.5	-10.3463686131387	0.920041025094572
6.12	-9.44	0.5	-10.3370873786408	0.936186256951645
6.13	-9.435	0.5	-10.3276452784504	0.952214353135681
6.14	-9.43	0.5	-10.3180434782609	0.968126444024366
6.15	-9.425	0.5	-10.3082831325301	0.983923646392801
6.16	-9.42	0.5	-10.2983653846154	0.999607063609468
6.17	-9.415	0.5	-10.2882913669065	1.01517778582889
6.18	-9.41	0.5	-10.2780622009569	1.03063689018109
6.19	-9.405	0.5	-10.2676789976134	1.04598544095784
6.2	-9.4	0.5	-10.2571428571429	1.06122448979592
6.21	-9.395	0.5	-10.2464548693587	1.07635507585717
6.22	-9.39	0.5	-10.2356161137441	1.0913782260057
6.23	-9.385	0.5	-10.2246276595745	1.10629495498214
6.24	-9.38	0.5	-10.2134905660377	1.12110626557494
6.25	-9.375	0.5	-10.2022058823529	1.13581314878893
6.26	-9.37	0.5	-10.1907746478873	1.15041658401111
6.27	-9.365	0.5	-10.1791978922717	1.16491753917369
6.28	-9.36	0.5	-10.167476635514	1.17931697091449
6.29	-9.355	0.5	-10.1556118881119	1.19361582473471
6.3	-9.35	0.5	-10.1436046511628	1.20781503515414
6.31	-9.345	0.5	-10.1314559164733	1.22191552586388
6.32	-9.34	0.5	-10.1191666666667	1.23591820987654
6.33	-9.335	0.5	-10.1067378752887	1.24982398967406
6.34	-9.33	0.5	-10.0941705069124	1.2636337573531
6.35	-9.325	0.5	-10.0814655172414	1.27734839476813
6.36	-9.32	0.5	-10.068623853211	1.29096877367225
6.37	-9.315	0.5	-10.0556464530892	1.30449575585566
6.38	-9.31	0.5	-10.0425342465753	1.31793019328204
6.39	-9.305	0.5	-10.0292881548975	1.33127292822266
6.4	-9.3	0.5	-10.0159090909091	1.34452479338843
6.41	-9.295	0.5	-10.0023979591837	1.35768661205979
6.42	-9.29	0.5	-9.9887556561086	1.37075919821461
6.43	-9.285	0.5	-9.97498306997743	1.38374335665405
6.44	-9.28	0.5	-9.96108108108108	1.39663988312637
6.45	-9.275	0.5	-9.94705056179775	1.40944956444893
6.46	-9.27	0.5	-9.93289237668161	1.42217317862816
6.47	-9.265	0.5	-9.91860738255034	1.4348114949777
6.48	-9.26	0.5	-9.90419642857143	1.44736527423469
6.49	-9.255	0.5	-9.88966035634744	1.45983526867426
6.5	-9.25	0.5	-9.875	1.47222222222222
6.51	-9.245	0.5	-9.86021618625277	1.48452687056602
6.52	-9.24	0.5	-9.84530973451328	1.496749941264
6.53	-9.235	0.5	-9.83028145695364	1.5088921538529
6.54	-9.23	0.5	-9.81513215859031	1.52095421995381
6.55	-9.225	0.5	-9.79986263736264	1.5329368433764
6.56	-9.22	0.5	-9.78447368421052	1.54484072022161
6.57	-9.215	0.5	-9.76896608315098	1.55666653898271
6.58	-9.21	0.5	-9.75334061135371	1.56841498064492
6.59	-9.205	0.5	-9.73759803921569	1.58008671878337
6.6	-9.2	0.5	-9.72173913043478	1.59168241965973
6.61	-9.195	0.5	-9.70576464208243	1.60320274231723
6.62	-9.19	0.5	-9.68967532467532	1.61464833867431
6.63	-9.185	0.5	-9.67347192224622	1.62601985361689
6.64	-9.18	0.5	-9.65715517241379	1.63731792508918
6.65	-9.175	0.5	-9.64072580645161	1.64854318418314
6.66	-9.17	0.5	-9.62418454935622	1.65969625522666
6.67	-9.165	0.5	-9.60753211991435	1.67077775587031
6.68	-9.16	0.5	-9.59076923076923	1.68178829717291
6.69	-9.155	0.5	-9.57389658848614	1.69272848368574
6.7	-9.15	0.5	-9.55691489361702	1.70359891353554
6.71	-9.145	0.5	-9.53982484076433	1.71440017850623
6.72	-9.14	0.5	-9.52262711864407	1.72513286411951
6.73	-9.135	0.5	-9.50532241014799	1.73579754971416
6.74	-9.13	0.5	-9.48791139240506	1.74639480852427
6.75	-9.125	0.5	-9.4703947368421	1.75692520775623
6.76	-9.12	0.5	-9.4527731092437	1.76738930866464
6.77	-9.115	0.5	-9.43504716981132	1.77778766662711
6.78	-9.11	0.5	-9.41721757322176	1.78812083121794
6.79	-9.105	0.5	-9.39928496868476	1.79838934628074
6.8	-9.1	0.5	-9.38125	1.80859375
6.81	-9.095	0.5	-9.3631133056133	1.81873457497158
6.82	-9.09	0.5	-9.3448755186722	1.82881234827224
6.83	-9.085	0.5	-9.32653726708075	1.83882759152811
6.84	-9.08	0.5	-9.30809917355372	1.84878082098217
6.85	-9.075	0.5	-9.2895618556701	1.85867254756084
6.86	-9.07	0.5	-9.27092592592593	1.86850327693949
6.87	-9.065	0.5	-9.25219199178645	1.87827350960707
6.88	-9.06	0.5	-9.23336065573771	1.88798374092986
6.89	-9.055	0.5	-9.21443251533742	1.8976344612142
6.9	-9.05	0.5	-9.19540816326531	1.90722615576843
6.91	-9.045	0.5	-9.17628818737271	1.91675930496389
6.92	-9.04	0.5	-9.15707317073171	1.92623438429506
6.93	-9.035	0.5	-9.13776369168357	1.93565186443886
6.94	-9.03	0.5	-9.11836032388664	1.94501221131309
6.95	-9.025	0.5	-9.09886363636364	1.95431588613407
6.96	-9.02	0.5	-9.07927419354839	1.96356334547346
6.97	-9.015	0.5	-9.05959255533199	1.97275504131428
6.98	-9.01	0.5	-9.03981927710843	1.98189142110611
6.99	-9.005	0.5	-9.01995490981964	1.99097292781957
7	-9	19	-9	19
7.01	-8.81	19	-8.97524875954199	2.94774832833166
7.02	-8.62000000000001	19	-8.94109521988528	3.88044068453433
7.03	-8.43	19	-8.89769051724138	4.79796158673539
7.04	-8.24	19	-8.84518694817658	5.70019444372811
7.05	-8.05	19	-8.78373798076923	6.58702154215975
7.06	-7.85999999999999	19	-8.71349826589595	7.4583240335461
7.07	-7.66999999999999	19	-8.63462364864865	8.31398192111032
7.08	-7.48	19	-8.54727117988394	9.15387404644412
7.09	-7.29	19	-8.45159912790698	9.97787807598701
7.1	-7.10000000000001	19	-8.34776699029127	10.7858704873221
7.11	-6.91000000000001	19	-8.23593550583658	11.5777265552847
7.12	-6.72	19	-8.11626666666666	12.3533203378818
7.13	-6.53	19	-7.98892373046875	13.1125246620178
7.14	-6.33999999999999	19	-7.8540712328767	13.8552111090261
7.15	-6.14999999999999	19	-7.71187499999999	14.58125
7.16	-5.96	19	-7.56250216110019	15.2905103809234
7.17	-5.77	19	-7.40612116141732	15.982860007595
7.18	-5.58000000000001	19	-7.24290177514793	16.6581653303456
7.19	-5.39000000000001	19	-7.07301511857708	17.316291478542
7.2	-5.2	19	-6.89663366336633	17.957102244878
7.21	-5.01	19	-6.71393125	18.5804600694444
7.22	-4.81999999999999	19	-6.52508310139164	19.1862260235802
7.23	-4.62999999999999	19	-6.33026583665338	19.7742597934954
7.24	-4.44	19	-6.12965748502994	20.3444196636667
7.25	-4.25	19	-5.9234375	20.8965625
7.26	-4.06	19	-5.7117867735471	21.4305437327561
7.27	-3.87000000000001	19	-5.49488765060242	21.9462173392364
7.28	-3.68	19	-5.27292394366197	22.443435826225
7.29	-3.49	19	-5.04608094758064	22.9220502121813
7.3	-3.29999999999999	19	-4.81454545454544	23.3819100091828
7.31	-3.10999999999999	19	-4.57850576923076	23.8228632046092
7.32	-2.91999999999999	19	-4.33815172413792	24.2447562425684
7.33	-2.73	19	-4.09367469512195	24.6474340050565
7.34	-2.54	19	-3.84526761710795	25.0307397928497
7.35	-2.35000000000001	19	-3.59312500000001	25.3945153061224
7.36	-2.16000000000001	19	-3.33744294478529	25.7386006247883
7.37	-1.97	19	-3.07841915983606	26.0628341885582
7.38	-1.78	19	-2.81625297741273	26.367052776712
7.39	-1.58999999999999	19	-2.55114537037036	26.6510914875781
7.4	-1.39999999999999	19	-2.28329896907216	26.9147837177171
7.41	-1.21	19	-2.01291807851239	27.1579611408032
7.42	-1.02	19	-1.74020869565218	27.3804536862004
7.43	-0.830000000000005	19	-1.46537852697096	27.5820895172259
7.44	-0.640000000000009	19	-1.18863700623702	27.7626950090983
7.45	-0.449999999999996	19	-0.910195312499994	27.9220947265625
7.46	-0.260000000000002	19	-0.630266388308979	28.0601114011881
7.47	-0.0699999999999878	19	-0.349064958158978	28.1765659083349
7.48	0.120000000000008	19	-0.0668075471697993	28.2712772437799
7.49	0.310000000000004	19	0.216287500000006	28.3440625
7.5	0.5	19	0.5	28.3947368421053
7.51	0.689999999999996	19	0.784107911392399	28.4231134834161
7.52	0.879999999999992	19	1.06838731501056	28.4290036606788
7.53	1.07	19	1.3526123940678	28.4122166089127
7.54	1.26	19	1.63655541401274	28.3725595358838
7.55	1.45000000000001	19	1.91998670212768	28.3098375961974
7.56	1.64000000000001	19	2.20267462686569	28.2238538650033
7.57	1.83000000000001	19	2.48438557692308	28.1144093113083
7.58	2.02	19	2.76488394004283	27.9813027708871
7.59	2.21	19	3.04393208154506	27.8243309187865
7.6	2.39999999999999	19	3.32129032258064	27.6432882414152
7.61	2.58999999999999	19	3.59671691810343	27.437967008212
7.62	2.78	19	3.86996803455724	27.2081572428849
7.63	2.97	19	4.14079772727272	26.9536466942149
7.64	3.16000000000001	19	4.40895791757051	26.6742208064144
7.65	3.35000000000001	19	4.67419836956523	26.3696626890359
7.66	3.54	19	4.93626666666667	26.0397530864198
7.67	3.73	19	5.19490818777292	25.6842703466753
7.68	3.91999999999999	19	5.44986608315098	25.3029903901862
7.69	4.10999999999999	19	5.70088124999999	24.8956866776316
7.7	4.3	19	5.94769230769231	24.4621301775148
7.71	4.49	19	6.19003557268722	24.002089333191
7.72	4.68000000000001	19	6.4276450331126	23.5153300293846
7.73	4.87000000000001	19	6.66025232300886	23.0016155581878
7.74	5.06	19	6.8875866962306	22.4607065845301
7.75	5.25	19	7.109375	21.8923611111111
7.76	5.44	19	7.3253416481069	21.2963344427855
7.77	5.62999999999999	19	7.53520859374999	20.6723791503907
7.78	5.82	19	7.73869530201343	20.0202450340075
7.79	6.01	19	7.9355187219731	19.3396790856442
7.8	6.20000000000001	19	8.12539325842698	18.6304254513319
7.81	6.39000000000001	19	8.30803074324325	17.8922253926223
7.82	6.58000000000001	19	8.48314040632055	17.1248172474764
7.83	6.77	19	8.65042884615385	16.3279363905325
7.84	6.96	19	8.8096	15.5013151927438
7.85	7.14999999999999	19	8.96035511363636	14.6446829803719
7.86	7.33999999999999	19	9.10239271070614	13.7577659933272
7.87	7.53	19	9.23540856164384	12.8402873428411
7.88	7.72	19	9.35909565217391	11.8919669684609
7.89	7.91000000000001	19	9.47314415137615	10.9125215943523
7.9	8.10000000000001	19	9.57724137931035	9.90166468489889
7.91	8.29	19	9.67107177419355	8.85910639958375
7.92	8.48	19	9.7543168591224	7.78455354714145
7.93	8.66999999999999	19	9.82665520833333	6.67770953896608
7.94	8.85999999999999	19	9.88776241299304	5.53827434176178
7.95	9.05	19	9.93731104651163	4.36594442942129
7.96	9.24	19	9.97497062937063	3.16041273411904
7.97	9.43000000000001	19	10.0004075934579	1.92136859660224
7.98	9.62000000000001	19	10.0132852459016	0.648497715667776
7.99	9.81	19	10.0132637323944	-0.658517903193836
8	10	0	10	0
8.01	10	0	9.98005	-1.99
8.02	10	0	9.9602	-1.98
8.03	10	0	9.94045	-1.97
8.04	10	0	9.9208	-1.96
8.05	10	0	9.90125	-1.95
8.06	10	0	9.8818	-1.94
8.07	10	0	9.86245	-1.93
8.08	10	0	9.8432	-1.92
8.09	10	0	9.82405	-1.91
8.1	10	0	9.805	-1.9
8.11	10	0	9.78605	-1.89
8.12	10	0	9.7672	-1.88
8.13	10	0	9.74845	-1.87
8.14	10	0	9.7298	-1.86
8.15	10	0	9.71125	-1.85
8.16	10	0	9.6928	-1.84
8.17	10	0	9.67445	-1.83
8.18	10	0	9.6562	-1.82
8.19	10	0	9.63805	-1.81
8.2	10	0	9.62	-1.8
8.21	10	0	9.60205	-1.79
8.22	10	0	9.5842	-1.78
8.23	10	0	9.56645	-1.77
8.24	10	0	9.5488	-1.76
8.25	10	0	9.53125	-1.75
8.26	10	0	9.5138	-1.74
8.27	10	0	9.49645	-1.73
8.28	10	0	9.4792	-1.72
8.29	10	0	9.46205	-1.71
8.3	10	0	9.445	-1.7
8.31	10	0	9.42805	-1.69
8.32	10	0	9.4112	-1.68
8.33	10	0	9.39445	-1.67
8.34	10	0	9.3778	-1.66
8.35	10	0	9.36125	-1.65
8.36	10	0	9.3448	-1.64
8.37	10	0	9.32845	-1.63
8.38	10	0	9.3122	-1.62
8.39	10	0	9.29605	-1.61
8.4	10	0	9.28	-1.6
8.41	10	0	9.26405	-1.59
8.42	10	0	9.2482	-1.58
8.43	10	0	9.23245	-1.57
8.44	10	0	9.2168	-1.56
8.45	10	0	9.20125	-1.55
8.46	10	0	9.1858	-1.54
8.47	10	0	9.17045	-1.53
8.48	10	0	9.1552	-1.52
8.49	10	0	9.14005	-1.51
8.5	10	0	9.125	-1.5
8.51	10	0	9.11005	-1.49
8.52	10	0	9.0952	-1.48
8.53	10	0	9.08045	-1.47
8.54	10	0	9.0658	-1.46
8.55	10	0	9.05125	-1.45
8.56	10	0	9.0368	-1.44
8.57	10	0	9.02245	-1.43
8.58	10	0	9.0082	-1.42
8.59	10	0	8.99405	-1.41
8.6	10	0	8.98	-1.4
8.61	10	0	8.96605	-1.39
8.62	10	0	8.9522	-1.38
8.63	10	0	8.93845	-1.37
8.64	10	0	8.9248	-1.36
8.65	10	0	8.91125	-1.35
8.66	10	0	8.8978	-1.34
8.67	10	0	8.88445	-1.33
8.68	10	0	8.8712	-1.32
8.69	10	0	8.85805	-1.31
8.7	10	0	8.845	-1.3
8.71	10	0	8.83205	-1.29
8.72	10	0	8.8192	-1.28
8.73	10	0	8.80645	-1.27
8.74	10	0	8.7938	-1.26
8.75	10	0	8.78125	-1.25
8.76	10	0	8.7688	-1.24
8.77	10	0	8.75645	-1.23
8.78	10	0	8.7442	-1.22
8.79	10	0	8.73205	-1.21
8.8	10	0	8.72	-1.2
8.81	10	0	8.70805	-1.19
8.82	10	0	8.6962	-1.18
8.83	10	0	8.68445	-1.17
8.84	10	0	8.6728	-1.16
8.85	10	0	8.66125	-1.15
8.86	10	0	8.6498	-1.14
8.87	10	0	8.63845	-1.13
8.88	10	0	8.6272	-1.12
8.89	10	0	8.61605	-1.11
8.9	10	0	8.605	-1.1
8.91	10	0	8.59405	-1.09
8.92	10	0	8.5832	-1.08
8.93	10	0	8.57245	-1.07
8.94	10	0	8.5618	-1.06
8.95	10	0	8.55125	-1.05
8.96	10	0	8.5408	-1.04
8.97	10	0	8.53045	-1.03
8.98	10	0	8.5202	-1.02
8.99	10	0	8.51005	-1.01
9	10	0	8.5	-1
9.01	10	0	8.49005	-0.99
9.02	10	0	8.4802	-0.98
9.03	10	0	8.47045	-0.969999999999999
9.04	10	0	8.4608	-0.960000000000001
9.05	10	0	8.45125	-0.949999999999999
9.06	10	0	8.4418	-0.94
9.07	10	0	8.43245	-0.93
9.08	10	0	8.4232	-0.92
9.09	10	0	8.41405	-0.91
9.1	10	0	8.405	-0.9
9.11	10	0	8.39605	-0.890000000000001
9.12	10	0	8.3872	-0.879999999999999
9.13	10	0	8.37845	-0.870000000000001
9.14	10	0	8.3698	-0.859999999999999
9.15	10	0	8.36125	-0.85
9.16	10	0	8.3528	-0.84
9.17	10	0	8.34445	-0.83
9.18	10	0	8.3362	-0.82
9.19	10	0	8.32805	-0.809999999999999
9.2	10	0	8.32	-0.800000000000001
9.21	10	0	8.31205	-0.789999999999999
9.22	10	0	8.3042	-0.780000000000001
9.23	10	0	8.29645	-0.77
9.24	10	0	8.2888	-0.76
9.25	10	0	8.28125	-0.75
9.26	10	0	8.2738	-0.74
9.27	10	0	8.26645	-0.73
9.28	10	0	8.2592	-0.719999999999999
9.29	10	0	8.25205	-0.710000000000001
9.3	10	0	8.245	-0.699999999999999
9.31	10	0	8.23805	-0.69
9.32	10	0	8.2312	-0.68
9.33	10	0	8.22445	-0.67
9.34	10	0	8.2178	-0.66
9.35	10	0	8.21125	-0.649999999999999
9.36	10	0	8.2048	-0.640000000000001
9.37	10	0	8.19845	-0.629999999999999
9.38	10	0	8.1922	-0.620000000000001
9.39	10	0	8.18605	-0.609999999999999
9.4	10	0	8.18	-0.6
9.41	10	0	8.17405	-0.59
9.42	10	0	8.1682	-0.58
9.43	10	0	8.16245	-0.57
9.44	10	0	8.1568	-0.559999999999999
9.45	10	0	8.15125	-0.550000000000001
9.46	10	0	8.1458	-0.539999999999999
9.47	10	0	8.14045	-0.530000000000001
9.48	10	0	8.1352	-0.52
9.49	10	0	8.13005	-0.51
9.5	10	0	8.125	-0.5
9.51	10	0	8.12005	-0.49
9.52	10	0	8.1152	-0.48
9.53	10	0	8.11045	-0.469999999999999
9.54	10	0	8.1058	-0.460000000000001
9.55	10	0	8.10125	-0.449999999999999
9.56	10	0	8.0968	-0.44
9.57	10	0	8.09245	-0.43
9.58	10	0	8.0882	-0.42
9.59	10	0	8.08405	-0.41
9.6	10	0	8.08	-0.399999999999999
9.61	10	0	8.07605	-0.390000000000001
9.62	10	0	8.0722	-0.379999999999999
9.63	10	0	8.06845	-0.370000000000001
9.64	10	0	8.0648	-0.359999999999999
9.65	10	0	8.06125	-0.35
9.66	10	0	8.0578	-0.34
9.67	10	0	8.05445	-0.33
9.68	10	0	8.0512	-0.32
9.69	10	0	8.04805	-0.309999999999999
9.7	10	0	8.045	-0.300000000000001
9.71	10	0	8.04205	-0.289999999999999
9.72	10	0	8.0392	-0.280000000000001
9.73	10	0	8.03645	-0.27
9.74	10	0	8.0338	-0.26
9.75	10	0	8.03125	-0.25
9.76	10	0	8.0288	-0.24
9.77	10	0	8.02645	-0.23
9.78	10	0	8.0242	-0.219999999999999
9.79	10	0	8.02205	-0.210000000000001
9.8	10	0	8.02	-0.199999999999999
9.81	10	0	8.01805	-0.19
9.82	10	0	8.0162	-0.18
9.83	10	0	8.01445	-0.17
9.84	10	0	8.0128	-0.16
9.85	10	0	8.01125	-0.149999999999999
9.86	10	0	8.0098	-0.140000000000001
9.87	10	0	8.00845	-0.129999999999999
9.88	10	0	8.0072	-0.120000000000001
9.89	10	0	8.00605	-0.109999999999999
9.9	10	0	8.005	-0.0999999999999996
9.91	10	0	8.00405	-0.0899999999999999
9.92	10	0	8.0032	-0.0800000000000001
9.93	10	0	8.00245	-0.0700000000000003
9.94	10	0	8.0018	-0.0599999999999987
9.95	10	0	8.00125	-0.0500000000000007
9.96	10	0	8.0008	-0.0399999999999991
9.97	10	0	8.00045	-0.0300000000000011
9.98	10	0	8.0002	-0.0199999999999996
9.99	10	0	8.00005	-0.00999999999999979
10	10	0	8	0
10.01	10	0	8.00005	0.00999999999999979
10.02	10	0	8.0002	0.0199999999999996
10.03	10	0	8.00045	0.0300000000000011
10.04	10	0	8.0008	0.0399999999999991
10.05	10	0	8.00125	0.0500000000000007
10.06	10	0	8.0018	0.0600000000000005
10.07	10	0	8.00245	0.0700000000000003
10.08	10	0	8.0032	0.0800000000000001
10.09	10	0	8.00405	0.0899999999999999
10.1	10	0	8.005	0.100000000000001
10.11	10	0	8.00605	0.109999999999999
10.12	10	0	8.0072	0.120000000000001
10.13	10	0	8.00845	0.129999999999999
10.14	10	0	8.0098	0.140000000000001
10.15	10	0	8.01125	0.15
10.16	10	0	8.0128	0.16
10.17	10	0	8.01445	0.17
10.18	10	0	8.0162	0.18
10.19	10	0	8.01805	0.190000000000001
10.2	10	0	8.02	0.199999999999999
10.21	10	0	8.02205	0.210000000000001
10.22	10	0	8.0242	0.219999999999999
10.23	10	0	8.02645	0.23
10.24	10	0	8.0288	0.24
10.25	10	0	8.03125	0.25
10.26	10	0	8.0338	0.26
10.27	10	0	8.03645	0.27
10.28	10	0	8.0392	0.280000000000001
10.29	10	0	8.04205	0.289999999999999
10.3	10	0	8.045	0.300000000000001
10.31	10	0	8.04805	0.31
10.32	10	0	8.0512	0.32
10.33	10	0	8.05445	0.33
10.34	10	0	8.0578	0.34
10.35	10	0	8.06125	0.350000000000001
10.36	10	0	8.0648	0.359999999999999
10.37	10	0	8.06845	0.370000000000001
10.38	10	0	8.0722	0.379999999999999
10.39	10	0	8.07605	0.390000000000001
10.4	10	0	8.08	0.4
10.41	10	0	8.08405	0.41
10.42	10	0	8.0882	0.42
10.43	10	0	8.09245	0.43
10.44	10	0	8.0968	0.440000000000001
10.45	10	0	8.10125	0.449999999999999
10.46	10	0	8.1058	0.460000000000001
10.47	10	0	8.11045	0.469999999999999
10.48	10	0	8.1152	0.48
10.49	10	0	8.12005	0.49
10.5	10	0	8.125	0.5
10.51	10	0	8.13005	0.51
10.52	10	0	8.1352	0.52
10.53	10	0	8.14045	0.530000000000001
10.54	10	0	8.1458	0.539999999999999
10.55	10	0	8.15125	0.550000000000001
10.56	10	0	8.1568	0.56
10.57	10	0	8.16245	0.57
10.58	10	0	8.1682	0.58
10.59	10	0	8.17405	0.59
10.6	10	0	8.18	0.600000000000001
10.61	10	0	8.18605	0.609999999999999
10.62	10	0	8.1922	0.620000000000001
10.63	10	0	8.19845	0.629999999999999
10.64	10	0	8.2048	0.640000000000001
10.65	10	0	8.21125	0.65
10.66	10	0	8.2178	0.66
10.67	10	0	8.22445	0.67
10.68	10	0	8.2312	0.68
10.69	10	0	8.23805	0.690000000000001
10.7	10	0	8.245	0.699999999999999
10.71	10	0	8.25205	0.710000000000001
10.72	10	0	8.2592	0.719999999999999
10.73	10	0	8.26645	0.73
10.74	10	0	8.2738	0.74
10.75	10	0	8.28125	0.75
10.76	10	0	8.2888	0.76
10.77	10	0	8.29645	0.77
10.78	10	0	8.3042	0.780000000000001
10.79	10	0	8.31205	0.789999999999999
10.8	10	0	8.32	0.800000000000001
10.81	10	0	8.32805	0.81
10.82	10	0	8.3362	0.82
10.83	10	0	8.34445	0.83
10.84	10	0	8.3528	0.84
10.85	10	0	8.36125	0.850000000000001
10.86	10	0	8.3698	0.859999999999999
10.87	10	0	8.37845	0.870000000000001
10.88	10	0	8.3872	0.879999999999999
10.89	10	0	8.39605	0.890000000000001
10.9	10	0	8.405	0.9
10.91	10	0	8.41405	0.91
10.92	10	0	8.4232	0.92
10.93	10	0	8.43245	0.93
10.94	10	0	8.4418	0.940000000000001
10.95	10	0	8.45125	0.949999999999999
10.96	10	0	8.4608	0.960000000000001
10.97	10	0	8.47045	0.969999999999999
10.98	10	0	8.4802	0.98
10.99	10	0	8.49005	0.99
11	10	0	8.5	1
11.01	10	0	8.51005	1.01
11.02	10	0	8.5202	1.02
11.03	10	0	8.53045	1.03
11.04	10	0	8.5408	1.04
11.05	10	0	8.55125	1.05
11.06	10	0	8.5618	1.06
11.07	10	0	8.57245	1.07
11.08	10	0	8.5832	1.08
11.09	10	0	8.59405	1.09
11.1	10	0	8.605	1.1
11.11	10	0	8.61605	1.11
11.12	10	0	8.6272	1.12
11.13	10	0	8.63845	1.13
11.14	10	0	8.6498	1.14
11.15	10	0	8.66125	1.15
11.16	10	0	8.6728	1.16
11.17	10	0	8.68445	1.17
11.18	10	0	8.6962	1.18
11.19	10	0	8.70805	1.19
11.2	10	0	8.72	1.2
11.21	10	0	8.73205	1.21
11.22	10	0	8.7442	1.22
11.23	10	0	8.75645	1.23
11.24	10	0	8.7688	1.24
11.25	10	0	8.78125	1.25
11.26	10	0	8.7938	1.26
11.27	10	0	8.80645	1.27
11.28	10	0	8.8192	1.28
11.29	10	0	8.83205	1.29
11.3	10	0	8.845	1.3
11.31	10	0	8.85805	1.31
11.32	10	0	8.8712	1.32
11.33	10	0	8.88445	1.33
11.34	10	0	8.8978	1.34
11.35	10	0	8.91125	1.35
11.36	10	0	8.9248	1.36
11.37	10	0	8.93845	1.37
11.38	10	0	8.9522	1.38
11.39	10	0	8.96605	1.39
11.4	10	0	8.98	1.4
11.41	10	0	8.99405	1.41
11.42	10	0	9.0082	1.42
11.43	10	0	9.02245	1.43
11.44	10	0	9.0368	1.44
11.45	10	0	9.05125	1.45
11.46	10	0	9.0658	1.46
11.47	10	0	9.08045	1.47
11.48	10	0	9.0952	1.48
11.49	10	0	9.11005	1.49
11.5	10	0	9.125	1.5
11.51	10	0	9.14005	1.51
11.52	10	0	9.1552	1.52
11.53	10	0	9.17045	1.53
11.54	10	0	9.1858	1.54
11.55	10	0	9.20125	1.55
11.56	10	0	9.2168	1.56
11.57	10	0	9.23245	1.57
11.58	10	0	9.2482	1.58
11.59	10	0	9.26405	1.59
11.6	10	0	9.28	1.6
11.61	10	0	9.29605	1.61
11.62	10	0	9.3122	1.62
11.63	10	0	9.32845	1.63
11.64	10	0	9.3448	1.64
11.65	10	0	9.36125	1.65
11.66	10	0	9.3778	1.66
11.67	10	0	9.39445	1.67
11.68	10	0	9.4112	1.68
11.69	10	0	9.42805	1.69
11.7	10	0	9.445	1.7
11.71	10	0	9.46205	1.71
11.72	10	0	9.4792	1.72
11.73	10	0	9.49645	1.73
11.74	10	0	9.5138	1.74
11.75	10	0	9.53125	1.75
11.76	10	0	9.5488	1.76
11.77	10	0	9.56645	1.77
11.78	10	0	9.5842	1.78
11.79	10	0	9.60205	1.79
11.8	10	0	9.62	1.8
11.81	10	0	9.63805	1.81
11.82	10	0	9.6562	1.82
11.83	10	0	9.67445	1.83
11.84	10	0	9.6928	1.84
11.85	10	0	9.71125	1.85
11.86	10	0	9.7298	1.86
11.87	10	0	9.74845	1.87
11.88	10	0	9.7672	1.88
11.89	10	0	9.78605	1.89
11.9	10	0	9.805	1.9
11.91	10	0	9.82405	1.91
11.92	10	0	9.8432	1.92
11.93	10	0	9.86245	1.93
11.94	10	0	9.8818	1.94
11.95	10	0	9.90125	1.95
11.96	10	0	9.9208	1.96
11.97	10	0	9.94045	1.97
11.98	10	0	9.9602	1.98
11.99	10	0	9.98005	1.99
12	10	0.666666666666667	10	0.666666666666667
12.01	10.0066666666667	0.666666666666667	10.0199777406789	1.99554441970154
12.02	10.0133333333333	0.666666666666667	10.0399108138239	1.99106646830945
12.03	10.02	0.666666666666667	10.0597989949749	1.98656599580819
12.04	10.0266666666667	0.666666666666667	10.0796420581655	1.98204285092263
12.05	10.0333333333333	0.666666666666667	10.0994397759104	1.97749688110538
12.06	10.04	0.666666666666667	10.1191919191919	1.97292793252389
12.07	10.0466666666667	0.666666666666667	10.138898257448	1.96833585004744
12.08	10.0533333333333	0.666666666666667	10.1585585585586	1.96372047723399
12.09	10.06	0.666666666666667	10.1781725888325	1.95908165631683
12.1	10.0666666666667	0.666666666666667	10.1977401129944	1.95441922819113
12.11	10.0733333333333	0.666666666666667	10.2172608941709	1.94973303240027
12.12	10.08	0.666666666666667	10.2367346938776	1.94502290712203
12.13	10.0866666666667	0.666666666666667	10.2561612720045	1.94028868915466
12.14	10.0933333333333	0.666666666666667	10.2755403868032	1.9355302139027
12.15	10.1	0.666666666666667	10.2948717948718	1.9307473153627
12.16	10.1066666666667	0.666666666666667	10.3141552511416	1.92593982610871
12.17	10.1133333333333	0.666666666666667	10.3333905088622	1.92110757727768
12.18	10.12	0.666666666666667	10.3525773195876	1.91625039855458
12.19	10.1266666666667	0.666666666666667	10.3717154331612	1.91136811815741
12.2	10.1333333333333	0.666666666666667	10.3908045977011	1.90646056282204
12.21	10.14	0.666666666666667	10.4098445595855	1.90152755778679
12.22	10.1466666666667	0.666666666666667	10.4288350634371	1.8965689267769
12.23	10.1533333333333	0.666666666666667	10.4477758521086	1.89158449198878
12.24	10.16	0.666666666666667	10.4666666666667	1.88657407407407
12.25	10.1666666666667	0.666666666666667	10.4855072463768	1.8815374921235
12.26	10.1733333333333	0.666666666666667	10.5042973286876	1.87647456365056
12.27	10.18	0.666666666666667	10.5230366492147	1.87138510457498
12.28	10.1866666666667	0.666666666666667	10.5417249417249	1.86626892920599
12.29	10.1933333333333	0.666666666666667	10.5603619381203	1.86112585022538
12.3	10.2	0.666666666666667	10.5789473684211	1.85595567867036
12.31	10.2066666666667	0.666666666666667	10.5974809607499	1.8507582239162
12.32	10.2133333333333	0.666666666666667	10.6159624413146	1.84553329365867
12.33	10.22	0.666666666666667	10.6343915343915	1.84028069389621
12.34	10.2266666666667	0.666666666666667	10.6527679623086	1.835000228912
12.35	10.2333333333333	0.666666666666667	10.6710914454277	1.82969170125564
12.36	10.24	0.666666666666667	10.6893617021277	1.82435491172476
12.37	10.2466666666667	0.666666666666667	10.7075784487863	1.81898965934629
12.38	10.2533333333333	0.666666666666667	10.7257413997628	1.81359574135755
12.39	10.26	0.666666666666667	10.7438502673797	1.80817295318711
12.4	10.2666666666667	0.666666666666667	10.7619047619048	1.80272108843537
12.41	10.2733333333333	0.666666666666667	10.7799045915325	1.79723993885495
12.42	10.28	0.666666666666667	10.7978494623656	1.79172929433075
12.43	10.2866666666667	0.666666666666667	10.8157390783962	1.78618894285988
12.44	10.2933333333333	0.666666666666667	10.8335731414868	1.7806186705312
12.45	10.3	0.666666666666667	10.8513513513514	1.77501826150475
12.46	10.3066666666667	0.666666666666667	10.8690734055355	1.76938749799076
12.47	10.3133333333333	0.666666666666667	10.8867389993972	1.76372616022855
12.48	10.32	0.666666666666667	10.904347826087	1.75803402646503
12.49	10.3266666666667	0.666666666666667	10.9218995765275	1.75231087293301
12.5	10.3333333333333	0.666666666666667	10.9393939393939	1.7465564738292
12.51	10.34	0.666666666666667	10.9568306010929	1.74077060129197
12.52	10.3466666666667	0.666666666666667	10.9742092457421	1.73495302537873
12.53	10.3533333333333	0.666666666666667	10.9915295551493	1.72910351404314
12.54	10.36	0.666666666666667	11.0087912087912	1.72322183311194
12.55	10.3666666666667	0.666666666666667	11.0259938837921	1.71730774626154
12.56	10.3733333333333	0.666666666666667	11.043137254902	1.71136101499423
12.57	10.38	0.666666666666667	11.0602209944751	1.70538139861421
12.58	10.3866666666667	0.666666666666667	11.0772447724477	1.69936865420315
12.59	10.3933333333333	0.666666666666667	11.0942082563155	1.69332253659559
12.6	10.4	0.666666666666667	11.1111111111111	1.68724279835391
12.61	10.4066666666667	0.666666666666667	11.1279529993816	1.68112918974303
12.62	10.4133333333333	0.666666666666667	11.1447335811648	1.67498145870473
12.63	10.42	0.666666666666667	11.1614525139665	1.66879935083175
12.64	10.4266666666667	0.666666666666667	11.1781094527363	1.66258260934135
12.65	10.4333333333333	0.666666666666667	11.1947040498442	1.65633097504877
12.66	10.44	0.666666666666667	11.2112359550562	1.65004418634011
12.67	10.4466666666667	0.666666666666667	11.2277048155097	1.64372197914503
12.68	10.4533333333333	0.666666666666667	11.2441102756892	1.637364086909
12.69	10.46	0.666666666666667	11.2604519774011	1.63097024056518
12.7	10.4666666666667	0.666666666666667	11.2767295597484	1.62454016850599
12.71	10.4733333333333	0.666666666666667	11.2929426591052	1.61807359655423
12.72	10.48	0.666666666666667	11.3090909090909	1.61157024793388
12.73	10.4866666666667	0.666666666666667	11.325173940544	1.60502984324047
12.74	10.4933333333333	0.666666666666667	11.3411913814956	1.59845210041107
12.75	10.5	0.666666666666667	11.3571428571429	1.59183673469388
12.76	10.5066666666667	0.666666666666667	11.3730279898219	1.58518345861741
12.77	10.5133333333333	0.666666666666667	11.3888463989802	1.57849198195926
12.78	10.52	0.666666666666667	11.4045977011494	1.57176201171445
12.79	10.5266666666667	0.666666666666667	11.4202815099168	1.56499325206337
12.8	10.5333333333333	0.666666666666667	11.4358974358974	1.55818540433925
12.81	10.54	0.666666666666667	11.4514450867052	1.55133816699522
12.82	10.5466666666667	0.666666666666667	11.4669240669241	1.54445123557093
12.83	10.5533333333333	0.666666666666667	11.4823339780787	1.53752430265867
12.84	10.56	0.666666666666667	11.4976744186047	1.53055705786912
12.85	10.5666666666667	0.666666666666667	11.5129449838188	1.52354918779652
12.86	10.5733333333333	0.666666666666667	11.5281452658885	1.51650037598349
12.87	10.58	0.666666666666667	11.5432748538012	1.50941030288522
12.88	10.5866666666667	0.666666666666667	11.5583333333333	1.50227864583333
12.89	10.5933333333333	0.666666666666667	11.5733202870189	1.49510507899914
12.9	10.6	0.666666666666667	11.5882352941176	1.4878892733564
12.91	10.6066666666667	0.666666666666667	11.6030779305828	1.48063089664365
12.92	10.6133333333333	0.666666666666667	11.6178477690289	1.47332961332589
12.93	10.62	0.666666666666667	11.6325443786982	1.46598508455586
12.94	10.6266666666667	0.666666666666667	11.6471673254282	1.45859696813469
12.95	10.6333333333333	0.666666666666667	11.6617161716172	1.45116491847205
12.96	10.64	0.666666666666667	11.6761904761905	1.44368858654573
12.97	10.6466666666667	0.666666666666667	11.6905897945659	1.43616761986069
12.98	10.6533333333333	0.666666666666667	11.7049136786189	1.42860166240747
12.99	10.66	0.666666666666667	11.7191616766467	1.4209903546201
13	10.6666666666667	0.666666666666667	11.7333333333333	1.41333333333333
13.01	10.6733333333333	0.666666666666667	11.7474281897128	1.40563023173936
13.02	10.68	0.666666666666667	11.7614457831325	1.39788067934388
13.03	10.6866666666667	0.666666666666667	11.7753856472166	1.39008430192152
13.04	10.6933333333333	0.666666666666667	11.789247311828	1.38224072147069
13.05	10.7	0.666666666666667	11.8030303030303	1.37434955616774
13.06	10.7066666666667	0.666666666666667	11.8167341430499	1.3664104203205
13.07	10.7133333333333	0.666666666666667	11.8303583502366	1.35842292432116
13.08	10.72	0.666666666666667	11.8439024390244	1.35038667459845
13.09	10.7266666666667	0.666666666666667	11.8573659198914	1.34230127356919
13.1	10.7333333333333	0.666666666666667	11.8707482993197	1.33416631958906
13.11	10.74	0.666666666666667	11.8840490797546	1.32598140690278
13.12	10.7466666666667	0.666666666666667	11.8972677595628	1.31774612559348
13.13	10.7533333333333	0.666666666666667	11.9104038329911	1.30946006153137
13.14	10.76	0.666666666666667	11.9234567901235	1.3011227963217
13.15	10.7666666666667	0.666666666666667	11.9364261168385	1.29273390725192
13.16	10.7733333333333	0.666666666666667	11.9493112947658	1.28429296723812
13.17	10.78	0.666666666666667	11.9621118012422	1.27579954477065
13.18	10.7866666666667	0.666666666666667	11.9748271092669	1.26725320385898
13.19	10.7933333333333	0.666666666666667	11.9874566874567	1.25865350397575
13.2	10.8	0.666666666666667	12	1.25
13.21	10.8066666666667	0.666666666666667	12.012456506611	1.24129224215957
13.22	10.8133333333333	0.666666666666667	12.0248256624826	1.23252977597264
13.23	10.82	0.666666666666667	12.037106918239	1.22371214218847
13.24	10.8266666666667	0.666666666666667	12.049299719888	1.21483887672716
13.25	10.8333333333333	0.666666666666667	12.0614035087719	1.20590951061865
13.26	10.84	0.666666666666667	12.073417721519	1.19692356994072
13.27	10.8466666666667	0.666666666666667	12.085341789993	1.18788057575609
13.28	10.8533333333333	0.666666666666667	12.0971751412429	1.17878004404865
13.29	10.86	0.666666666666667	12.1089171974522	1.16962148565865
13.3	10.8666666666667	0.666666666666667	12.1205673758865	1.16040440621699
13.31	10.8733333333333	0.666666666666667	12.1321250888415	1.1511283060785
13.32	10.88	0.666666666666667	12.1435897435897	1.14179268025422
13.33	10.8866666666667	0.666666666666667	12.1549607423269	1.13239701834266
13.34	10.8933333333333	0.666666666666667	12.1662374821173	1.12294080446008
13.35	10.9	0.666666666666667	12.1774193548387	1.11342351716962
13.36	10.9066666666667	0.666666666666667	12.1885057471264	1.10384462940943
13.37	10.9133333333333	0.666666666666667	12.1994960403168	1.09420360841975
13.38	10.92	0.666666666666667	12.2103896103896	1.08449991566875
13.39	10.9266666666667	0.666666666666667	12.2211858279103	1.07473300677737
13.4	10.9333333333333	0.666666666666667	12.231884057971	1.06490233144297
13.41	10.94	0.666666666666667	12.2424836601307	1.05500733336181
13.42	10.9466666666667	0.666666666666667	12.2529839883552	1.04504745015033
13.43	10.9533333333333	0.666666666666667	12.2633843909555	1.03502211326525
13.44	10.96	0.666666666666667	12.2736842105263	1.02493074792244
13.45	10.9666666666667	0.666666666666667	12.2838827838828	1.01477277301453
13.46	10.9733333333333	0.666666666666667	12.2939794419971	1.00454760102725
13.47	10.98	0.666666666666667	12.3039735099338	0.994254637954475
13.48	10.9866666666667	0.666666666666667	12.3138643067847	0.983893283211945
13.49	10.9933333333333	0.666666666666667	12.3236511456024	0.973462929549674
13.5	11	0.666666666666667	12.3333333333333	0.962962962962963
13.51	11.0066666666667	0.666666666666667	12.3429101707498	0.952392762602037
13.52	11.0133333333333	0.666666666666667	12.352380952381	0.941751700680273
13.53	11.02	0.666666666666667	12.361744966443	0.931039142380974
13.54	11.0266666666667	0.666666666666667	12.3710014947683	0.920254445762699
13.55	11.0333333333333	0.666666666666667	12.3801498127341	0.909396961663089
13.56	11.04	0.666666666666667	12.3891891891892	0.898466033601168
13.57	11.0466666666667	0.666666666666667	12.3981188863807	0.88746099767812
13.58	11.0533333333333	0.666666666666667	12.4069381598793	0.876381182476471
13.59	11.06	0.666666666666667	12.4156462585034	0.865225908957688
13.6	11.0666666666667	0.666666666666667	12.4242424242424	0.853994490358127
13.61	11.0733333333333	0.666666666666667	12.4327258921792	0.84268623208334
13.62	11.08	0.666666666666667	12.441095890411	0.831300431600674
13.63	11.0866666666667	0.666666666666667	12.4493516399695	0.819836378330165
13.64	11.0933333333333	0.666666666666667	12.4574923547401	0.808293353533653
13.65	11.1	0.666666666666667	12.4655172413793	0.79667063020214
13.66	11.1066666666667	0.666666666666667	12.473425499232	0.78496747294131
13.67	11.1133333333333	0.666666666666667	12.4812163202463	0.773183137855199
13.68	11.12	0.666666666666667	12.4888888888889	0.761316872427984
13.69	11.1266666666667	0.666666666666667	12.4964423820572	0.749367915403844
13.7	11.1333333333333	0.666666666666667	12.5038759689922	0.737335496664863
13.71	11.14	0.666666666666667	12.5111888111888	0.725218837106948
13.72	11.1466666666667	0.666666666666667	12.5183800623053	0.713017148513698
13.73	11.1533333333333	0.666666666666667	12.5254488680718	0.700729633428216
13.74	11.16	0.666666666666667	12.5323943661972	0.688355485022813
13.75	11.1666666666667	0.666666666666667	12.5392156862745	0.675893886966551
13.76	11.1733333333333	0.666666666666667	12.5459119496855	0.663344013290614
13.77	11.18	0.666666666666667	12.5524822695035	0.65070502825143
13.78	11.1866666666667	0.666666666666667	12.5589257503949	0.637976086191536
13.79	11.1933333333333	0.666666666666667	12.5652414885194	0.625156331398114
13.8	11.2	0.666666666666667	12.5714285714286	0.612244897959183
13.81	11.2066666666667	0.666666666666667	12.5774860779634	0.599240909617359
13.82	11.2133333333333	0.666666666666667	12.5834130781499	0.586143479621193
13.83	11.22	0.666666666666667	12.5892086330935	0.572951710573987
13.84	11.2266666666667	0.666666666666667	12.5948717948718	0.559664694280079
13.85	11.2333333333333	0.666666666666667	12.6004016064257	0.546281511588523
13.86	11.24	0.666666666666667	12.6057971014493	0.532801232234125
13.87	11.2466666666667	0.666666666666667	12.6110573042776	0.51922291467578
13.88	11.2533333333333	0.666666666666667	12.6161812297735	0.505545605932069
13.89	11.26	0.666666666666667	12.6211678832117	0.491768341414033
13.9	11.2666666666667	0.666666666666667	12.6260162601626	0.477890144755106
13.91	11.2733333333333	0.666666666666667	12.6307253463733	0.463910027638126
13.92	11.28	0.666666666666667	12.6352941176471	0.449826989619377
13.93	11.2866666666667	0.666666666666667	12.6397215397215	0.435640017949601
13.94	11.2933333333333	0.666666666666667	12.6440065681445	0.421348087391913
13.95	11.3	0.666666666666667	12.6481481481481	0.406950160036578
13.96	11.3066666666667	0.666666666666667	12.6521452145215	0.392445185112569
13.97	11.3133333333333	0.666666666666667	12.6559966914806	0.377832098795838
13.98	11.32	0.666666666666667	12.6597014925373	0.363109824014256
13.99	11.3266666666667	0.666666666666667	12.6632585203658	0.348277270249148
14	11.3333333333333	0.666666666666667	12.6666666666667	0.333333333333333
14.01	11.34	0.666666666666667	12.6699248120301	0.318276895245633
14.02	11.3466666666667	0.666666666666667	12.6730318257956	0.303106823901754
14.03	11.3533333333333	0.666666666666667	12.675986565911	0.287821972941478
14.04	11.36	0.666666666666667	12.6787878787879	0.272421181512089
14.05	11.3666666666667	0.666666666666667	12.6814345991561	0.256903274047961
14.06	11.3733333333333	0.666666666666667	12.6839255499154	0.241267060046209
14.07	11.38	0.666666666666667	12.6862595419847	0.225511333838354
14.08	11.3866666666667	0.666666666666667	12.6884353741497	0.209634874357906
14.09	11.3933333333333	0.666666666666667	12.6904518329071	0.193636444903771
14.1	11.4	0.666666666666667	12.6923076923077	0.177514792899409
14.11	11.4066666666667	0.666666666666667	12.6940017137961	0.161268649647659
14.12	11.4133333333333	0.666666666666667	12.6955326460481	0.144896730081126
14.13	11.42	0.666666666666667	12.6968992248062	0.128397732508061
14.14	11.4266666666667	0.666666666666667	12.6981001727116	0.111770338353601
14.15	11.4333333333333	0.666666666666667	12.6991341991342	0.0950132118963282
14.16	11.44	0.666666666666667	12.7	0.0781249999999999
14.17	11.4466666666667	0.666666666666667	12.7006962576153	0.0611043318403789
14.18	11.4533333333333	0.666666666666667	12.7012216404887	0.0439498186270484
14.19	11.46	0.666666666666667	12.7015748031496	0.0266600533201076
14.2	11.4666666666667	0.666666666666667	12.7017543859649	0.00923361034164194
14.21	11.4733333333333	0.666666666666667	12.7017590149516	-0.00833095471812972
14.22	11.48	0.666666666666667	12.7015873015873	-0.0260351054001859
14.23	11.4866666666667	0.666666666666667	12.7012378426172	-0.0438803246815692
14.24	11.4933333333333	0.666666666666667	12.7007092198582	-0.0618681152859518
14.25	11.5	0.666666666666667	12.7	-0.08
14.26	11.5066666666667	0.666666666666667	12.6991087344029	-0.0982775219956717
14.27	11.5133333333333	0.666666666666667	12.6980339588919	-0.116702245158569
14.28	11.52	0.666666666666667	12.6967741935484	-0.135275754422475
14.29	11.5266666666667	0.666666666666667	12.6953279424978	-0.153999656110221
14.3	11.5333333333333	0.666666666666667	12.6936936936937	-0.172875578280985
14.31	11.54	0.666666666666667	12.6918699186992	-0.191905171084232
14.32	11.5466666666667	0.666666666666667	12.6898550724638	-0.211090107120353
14.33	11.5533333333333	0.666666666666667	12.6876475930972	-0.230432081808215
14.34	11.56	0.666666666666667	12.6852459016393	-0.249932813759742
14.35	11.5666666666667	0.666666666666667	12.6826484018265	-0.269594045161693
14.36	11.5733333333333	0.666666666666667	12.6798534798535	-0.289417542164794
14.37	11.58	0.666666666666667	12.6768595041322	-0.309405095280379
14.38	11.5866666666667	0.666666666666667	12.673664825046	-0.329558519784705
14.39	11.5933333333333	0.666666666666667	12.6702677746999	-0.349879656131144
14.4	11.6	0.666666666666667	12.6666666666667	-0.370370370370371
14.41	11.6066666666667	0.666666666666667	12.6628597957289	-0.391032554578772
14.42	11.6133333333333	0.666666666666667	12.6588454376164	-0.411868127295236
14.43	11.62	0.666666666666667	12.6546218487395	-0.432879033966527
14.44	11.6266666666667	0.666666666666667	12.6501872659176	-0.454067247401421
14.45	11.6333333333333	0.666666666666667	12.6455399061033	-0.475434768233818
14.46	11.64	0.666666666666667	12.6406779661017	-0.496983625395003
14.47	11.6466666666667	0.666666666666667	12.6355996222852	-0.518715876595325
14.48	11.6533333333333	0.666666666666667	12.630303030303	-0.540633608815428
14.49	11.66	0.666666666666667	12.6247863247863	-0.562738938807315
14.5	11.6666666666667	0.666666666666667	12.6190476190476	-0.585034013605442
14.51	11.6733333333333	0.666666666666667	12.6130850047755	-0.607521011048075
14.52	11.68	0.666666666666667	12.6068965517241	-0.630202140309154
14.53	11.6866666666667	0.666666666666667	12.6004803073967	-0.653079642440901
14.54	11.6933333333333	0.666666666666667	12.5938342967245	-0.676155790927419
14.55	11.7	0.666666666666667	12.5869565217391	-0.699432892249529
14.56	11.7066666666667	0.666666666666667	12.5798449612403	-0.722913286461151
14.57	11.7133333333333	0.666666666666667	12.5724975704568	-0.746599347777429
14.58	11.72	0.666666666666667	12.5649122807018	-0.770493485174926
14.59	11.7266666666667	0.666666666666667	12.5570869990225	-0.794598143004159
14.6	11.7333333333333	0.666666666666667	12.5490196078431	-0.818915801614762
14.61	11.74	0.666666666666667	12.5407079646018	-0.843448977993577
14.62	11.7466666666667	0.666666666666667	12.5321499013807	-0.868200226415978
14.63	11.7533333333333	0.666666666666667	12.5233432245302	-0.893172139110733
14.64	11.76	0.666666666666667	12.5142857142857	-0.918367346938776
14.65	11.7666666666667	0.666666666666667	12.5049751243781	-0.943788520086138
14.66	11.7733333333333	0.666666666666667	12.4954091816367	-0.96943836877144
14.67	11.78	0.666666666666667	12.4855855855856	-0.995319643968292
14.68	11.7866666666667	0.666666666666667	12.4755020080321	-1.02143513814293
14.69	11.7933333333333	0.666666666666667	12.4651560926485	-1.04778768600749
14.7	11.8	0.666666666666667	12.4545454545455	-1.07438016528926
14.71	11.8066666666667	0.666666666666667	12.4436676798379	-1.10121549751635
14.72	11.8133333333333	0.666666666666667	12.4325203252033	-1.12829664882015
14.73	11.82	0.666666666666667	12.4211009174312	-1.15562663075499
14.74	11.8266666666667	0.666666666666667	12.4094069529652	-1.18320850113541
14.75	11.8333333333333	0.666666666666667	12.3974358974359	-1.21104536489152
14.76	11.84	0.666666666666667	12.3851851851852	-1.23914037494284
14.77	11.8466666666667	0.666666666666667	12.3726522187822	-1.26749673309115
14.78	11.8533333333333	0.666666666666667	12.35983436853	-1.29611769093271
14.79	11.86	0.666666666666667	12.3467289719626	-1.32500655079047
14.8	11.8666666666667	0.666666666666667	12.3333333333333	-1.35416666666667
14.81	11.8733333333333	0.666666666666667	12.319644723093	-1.38360144521641
14.82	11.88	0.666666666666667	12.3056603773585	-1.41331434674261
14.83	11.8866666666667	0.666666666666667	12.2913774973712	-1.44330888621308
14.84	11.8933333333333	0.666666666666667	12.2767932489451	-1.47358863430006
14.85	11.9	0.666666666666667	12.2619047619048	-1.50415721844293
14.86	11.9066666666667	0.666666666666667	12.2467091295117	-1.53501832393471
14.87	11.9133333333333	0.666666666666667	12.2312034078807	-1.56617569503279
14.88	11.92	0.666666666666667	12.2153846153846	-1.59763313609468
14.89	11.9266666666667	0.666666666666667	12.1992497320472	-1.62939451273939
14.9	11.9333333333333	0.666666666666667	12.1827956989247	-1.66146375303503
14.91	11.94	0.666666666666667	12.1660194174757	-1.69384484871336
14.92	11.9466666666667	0.666666666666667	12.1489177489178	-1.72654185641199
14.93	11.9533333333333	0.666666666666667	12.1314875135722	-1.75955889894499
14.94	11.96	0.666666666666667	12.1137254901961	-1.79290016660259
14.95	11.9666666666667	0.666666666666667	12.0956284153005	-1.8265699184807
14.96	11.9733333333333	0.666666666666667	12.0771929824561	-1.86057248384119
14.97	11.98	0.666666666666667	12.0584158415842	-1.89491226350358
14.98	11.9866666666667	0.666666666666667	12.039293598234	-1.9295937312691
14.99	11.9933333333333	0.666666666666667	12.0198228128461	-1.96462143537783
15	12	0.666666666666667	12	0.666666666666667
