#!perl

use strict;
use warnings;
$|=1;
use Test::More tests => 1;
use DBI;
#use DBD::SQLite;
use File::Spec;
use File::Path;
use File::Basename;

# rebuild cpanstats db

my $f = File::Spec->catfile('t','_DBDIR','test.db');
unlink $f if -f $f;
mkpath( dirname($f) );

my $dbh = DBI->connect("dbi:SQLite:dbname=$f", '', '', {AutoCommit=>1});
$dbh->do(q{
  CREATE TABLE cpanstats (
                          id            INTEGER PRIMARY KEY,
                          state         TEXT,
                          postdate      TEXT,
                          tester        TEXT,
                          dist          TEXT,
                          version       TEXT,
                          platform      TEXT,
                          perl          TEXT,
                          osname        TEXT,
                          osvers        TEXT,
                          fulldate      TEXT
  )
});

while(<DATA>){
  chomp;
  $dbh->do('INSERT INTO cpanstats ( id, state, postdate, tester, dist, version, platform, perl, osname, osvers, fulldate ) VALUES ( ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ? )', {}, split(/\|/,$_) );
}

$dbh->do(q{ CREATE INDEX distverstate ON cpanstats (dist, version, state) });
$dbh->do(q{ CREATE INDEX ixdate ON cpanstats (postdate) });
$dbh->do(q{ CREATE INDEX ixperl ON cpanstats (perl) });
$dbh->do(q{ CREATE INDEX ixplat ON cpanstats (platform) });

my ($ct) = $dbh->selectrow_array('select count(*) from cpanstats');

$dbh->disconnect;

is($ct, 14034, "row count for cpanstats");

#select * from cpanstats where state='cpan' and dist in (SELECT dist from ixlatest where author='DCANTRELL')
# sqlite> select * from cpanstats where postdate=200901 order by dist limit 20;
# id|state|postdate|tester|dist|version|platform|perl|osname|osvers|date
__DATA__
3031|pass|199910|schinder@pobox.com|Statistics-ChiSquare|0.2|PA-RISC1.1|5.4.4|hpux|10.20|199910191643
3032|pass|199910|schinder@pobox.com|Statistics-ChiSquare|0.2|sun4-solaris|5.5.3|solaris|2.7|199910191644
3121|pass|199910|schinder@pobox.com|Statistics-ChiSquare|0.2|ppc-linux|5.5.3|linux|2.2.10|199910221334
3147|pass|199910|ljmoore@freespace.net (Larry Moore)|Statistics-ChiSquare|0.2|MacOS_CFM68K|5.4.0|MacOS|7.6|199910230929
18566|pass|200102|a.calpini@alos.it ("alos.it/a.calpini")|Data-Compare|0.01|MSWin32-x86-multi-thread|5.6.0|MSWin32|4.0|200102091428
18584|pass|200102|schinder@pobox.com|Data-Hexdumper|0.01|ppc-linux|5.6.0|linux|2.2.14|200102091544
18585|pass|200102|schinder@pobox.com|Data-Hexdumper|0.01|sun4-solaris|5.6.0|solaris|2.7|200102091544
18586|pass|200102|schinder@pobox.com|Data-Hexdumper|0.01|PA-RISC1.1|5.6.0|hpux|10.20|200102091544
18661|pass|200102|a.calpini@alos.it ("alos.it/a.calpini")|Data-Hexdumper|0.01|MSWin32-x86-multi-thread|5.6.0|MSWin32|4.0|200102121237
19470|pass|200103|a.calpini@alos.it ("alos.it/a.calpini")|Statistics-ChiSquare|0.2|MSWin32-x86-multi-thread|5.6.0|MSWin32|4.0|200103011637
21002|unknown|200104|cpantest@quaday.com (Tim Ayers)|Pony|1.01|sparc-openbsd|5.6.0|openbsd|2.8|200104050832
21167|pass|200104|cpantest@quaday.com (Tim Ayers)|Tie-Scalar-Decay|1.0|sparc-openbsd|5.6.0|openbsd|2.8|200104110838
22016|pass|200104|kriegjcb@CMC01SE.rz.ruhr-uni-bochum.de|Tie-Scalar-Decay|1.1|i686-linux|5.7.1|linux|2.2.16|200104251735
22040|unknown|200104|kriegjcb@sunu991.ruhr-uni-bochum.de|Data-Compare|0.02|sun4-solaris|5.6.1|solaris|2.8|200104260801
22045|pass|200104|Jost.Krieger@ruhr-uni-bochum.de|Data-Compare|0.02|sun4-solaris|5.6.1|solaris|2.8|200104260812
22158|pass|200104|cpantest@quaday.com (Tim Ayers)|Data-Compare|0.02|sparc-openbsd|5.6.0|openbsd|2.8|200104270739
22176|pass|200104|kriegjcb@CMC01SE.rz.ruhr-uni-bochum.de|Data-Compare|0.02|i686-linux|5.7.1|linux|2.2.16|200104271607
22192|pass|200104|schinder@pobox.com|Tie-Scalar-Decay|1.1|sun4-solaris|5.6.1|solaris|2.7|200104271416
22193|pass|200104|schinder@pobox.com|Tie-Scalar-Decay|1.1|PA-RISC1.1|5.6.1|hpux|10.20|200104271416
22194|fail|200104|schinder@pobox.com|Tie-Scalar-Decay|1.1|ppc-linux|5.6.1|linux|2.4.2|200104271420
23486|pass|200105|Jost.Krieger@ruhr-uni-bochum.de|Tie-Scalar-Decay|1.1.1|sun4-solaris|5.6.1|solaris|2.8|200105221052
23554|pass|200105|cpantest@quaday.com (Tim Ayers)|Tie-Scalar-Decay|1.1.1|OpenBSD.sparc-openbsd|5.6.1|openbsd|2.8|200105230932
24779|unknown|200106|code@leegoddard.com (Lee Goddard)|Acme-Pony|1.1.1|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200106202115
24988|pass|200106|Jost.Krieger@ruhr-uni-bochum.de|Tie-Hash-Rank|1.0.1|sun4-solaris|5.6.1|solaris|2.8|200106281517
25189|unknown|200107|Jost.Krieger@ruhr-uni-bochum.de|Acme-Pony|1.1.1|sun4-solaris|5.6.1|solaris|2.8|200107091033
27937|unknown|200110|Jost.Krieger@ruhr-uni-bochum.de|Acme-Pony|1.1.2|sun4-solaris|5.6.1|solaris|2.8|200110101407
29413|pass|200111|gellyfish@gellyfish.com (Jonathan Stowe)|Tie-Hash-Transactional|1.0|i686-linux|5.6.1|linux|2.2.16-22|200111112039
30073|pass|200111|ethan@r220-1.rz.RWTH-Aachen.DE (Tassilo von Parseval)|Statistics-ChiSquare|0.3|i686-linux-thread|5.7.2|linux|2.4.13|200111240806
30092|pass|200111|riechert@pobox.com (Andreas Marcel Riechert)|Statistics-ChiSquare|0.3|i686-linux|5.6.1|linux|2.2.14-5.0|200111241833
30240|pass|200111|randy@theoryx5.uwinnipeg.ca (Randy Kobes)|Statistics-ChiSquare|0.3|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200111252029
31844|unknown|200201|riechert@pobox.com (Andreas Marcel Riechert)|Acme-Pony|1.1.2|i686-linux|5.6.1|linux|2.2.14-5.0|200201051646
32918|pass|200201|easmith@beatrice.rutgers.edu (Ed Allen Smith)|Data-Compare|0.02|IP30-R12000-irix|5.6.0|irix|6.5|200201231304
36386|pass|200204|autrijus@www.autrijus.org (Autrijus Tang)|Data-Compare|0.02|i386-freebsd|5.7.3 patch 15354|freebsd|4.5-release|200204011435
36442|unknown|200204|autrijus@www.autrijus.org (Autrijus Tang)|Acme-Pony|1.1.2|i386-freebsd|5.7.3 patch 15354|freebsd|4.5-release|200204021614
48917|pass|200207|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Statistics-ChiSquare|0.3|sun4-solaris|5.8.0|solaris|2.8|200207310932
48959|pass|200207|perl_test@logicalchaos.org ((Automated Perl Test Account))|Statistics-ChiSquare|0.3|i386-linux|5.6.1|linux|2.4.8-11mdkenterprise|200207310518
50072|pass|200208|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Hexdumper|0.01|sun4-solaris|5.8.0|solaris|2.8|200208051136
50073|pass|200208|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Tie-Hash-Transactional|1.0|sun4-solaris|5.8.0|solaris|2.8|200208051138
50115|pass|200208|perl_test@logicalchaos.org ((Automated Perl Test Account))|Data-Hexdumper|0.01|i386-linux|5.6.1|linux|2.4.8-11mdkenterprise|200208050731
50116|pass|200208|perl_test@logicalchaos.org ((Automated Perl Test Account))|Tie-Hash-Transactional|1.0|i386-linux|5.6.1|linux|2.4.8-11mdkenterprise|200208050731
50564|pass|200208|perl_test@logicalchaos.org ((Automated Perl Test Account))|Data-Compare|0.02|i386-linux|5.6.1|linux|2.4.8-11mdkenterprise|200208060203
53617|pass|200208|alian@cpan.org|Data-Compare|0.02|i386-netbsd|5.8.0|netbsd|1.5.2|200208141515
54328|pass|200208|smoke@nambit.net (Dans Testorama)|Data-Compare|0.02|i386-freebsd|5.6.1|freebsd|4.2-release|200208162340
54333|pass|200208|smoke@nambit.net (Dans Testorama)|Data-Compare|0.02|i386-freebsd-thread-multi|5.8.0|freebsd|4.2-release|200208162342
54350|unknown|200208|smoke@nambit.net (Dans Testorama)|Acme-Pony|1.1.2|i386-freebsd|5.5.3|freebsd|4.0-current|200208170011
54351|unknown|200208|smoke@nambit.net (Dans Testorama)|Acme-Pony|1.1.2|i386-freebsd|5.6.1|freebsd|4.2-release|200208170011
54352|unknown|200208|smoke@nambit.net (Dans Testorama)|Acme-Pony|1.1.2|i386-freebsd-thread-multi|5.8.0|freebsd|4.2-release|200208170012
54628|pass|200208|smoke@nambit.net (Dans Testorama)|Statistics-ChiSquare|0.3|i386-freebsd|5.5.3|freebsd|4.0-current|200208170322
54629|pass|200208|smoke@nambit.net (Dans Testorama)|Statistics-ChiSquare|0.3|i386-freebsd|5.6.1|freebsd|4.2-release|200208170322
54630|pass|200208|smoke@nambit.net (Dans Testorama)|Statistics-ChiSquare|0.3|i386-freebsd-thread-multi|5.8.0|freebsd|4.2-release|200208170322
55844|pass|200208|alian@cpan.org|Data-Hexdumper|0.01|i386-netbsd|5.8.0|netbsd|1.5.2|200208190038
56068|pass|200208|perl_test@logicalchaos.org ((Automated Perl Test Account))|Data-Compare|0.02|i386-linux|5.6.1|linux|2.4.16-6mdksmp|200208191423
56910|pass|200208|alian@cpan.org (alian)|Data-Compare|0.02|OpenBSD.i386-openbsd|5.8.0|openbsd|3.0|200208221001
56921|pass|200208|alian@cpan.org (alian)|Data-Hexdumper|0.01|OpenBSD.i386-openbsd|5.8.0|openbsd|3.0|200208221028
58539|pass|200208|alian@cpan.org (alian)|Data-Compare|0.02|i386-freebsd|5.8.0|freebsd|4.6.2-release|200208301816
58554|pass|200208|alian@cpan.org (alian)|Data-Hexdumper|0.01|i386-freebsd|5.8.0|freebsd|4.6.2-release|200208301832
60246|unknown|200209|perl_test@logicalchaos.org ((Automated Perl Test Account))|Acme-Licence|1.0|i386-linux|5.6.1|linux|2.4.16-6mdksmp|200209190454
60251|unknown|200209|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Acme-Licence|1.0|sun4-solaris|5.8.0|solaris|2.8|200209191322
60401|pass|200209|perl_test@logicalchaos.org ((Automated Perl Test Account))|Data-Hexdumper|1.0.1|i386-linux|5.6.1|linux|2.4.16-6mdksmp|200209200810
60428|pass|200209|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Hexdumper|1.0.1|sun4-solaris|5.8.0|solaris|2.8|200209201635
61369|pass|200209|root@thesafrans.com ((root))|Statistics-ChiSquare|0.3|i686-linux|5.8.0|linux|2.4.7-10|200209290847
61871|pass|200209|jlatour@www.arens.nl (Jeroen Latour)|Statistics-ChiSquare|0.3|i586-linux|5.8.0|linux|2.2.16c32_iii|200209301534
61914|pass|200209|jlatour@www.arens.nl (Jeroen Latour)|Data-Compare|0.02|i586-linux|5.8.0|linux|2.2.16c32_iii|200209301558
61923|pass|200209|jlatour@www.arens.nl (Jeroen Latour)|Data-Hexdumper|1.0.1|i586-linux|5.8.0|linux|2.2.16c32_iii|200209301602
62459|unknown|200210|jlatour@www.arens.nl (Jeroen Latour)|Acme-Pony|1.1.2|i586-linux|5.8.0|linux|2.2.16c32_iii|200210012348
62470|unknown|200210|jlatour@www.arens.nl (Jeroen Latour)|Acme-Licence|1.0|i586-linux|5.8.0|linux|2.2.16c32_iii|200210020052
62479|unknown|200210|perl_test@logicalchaos.org ((Automated Perl Test Account))|Acme-Pony|1.1.2|i386-linux|5.6.1|linux|2.4.16-6mdksmp|200210011738
64602|unknown|200210|root@godzilla.uni.to (root)|Acme-Pony|1.1.2|i386-linux|5.6.1|linux|2.4.17-0.13smp|200210161825
68081|pass|200211|alian@cpan.org|Statistics-ChiSquare|0.3|i386-netbsd|5.8.0|netbsd|1.5.2|200211151932
68422|pass|200211|alian@cpan.org (Alian - CPANPLUS/auto)|Statistics-ChiSquare|0.3|OpenBSD.i386-openbsd|5.8.0|openbsd|3.0|200211152330
70145|pass|200211|alian@cpan.org (Alian - CPANPLUS/auto)|Data-Hexdumper|1.0.1|i386-netbsd|5.8.0|netbsd|1.5.2|200211210114
70469|pass|200211|alian@cpan.org|Data-Compare|0.02|sparc-netbsd|5.6.1|netbsd|1.5.1|200211211430
70495|pass|200211|alian@cpan.org (Alian - CPANPLUS/auto)|Statistics-ChiSquare|0.3|i386-freebsd|5.8.0|freebsd|4.6.2-release|200211211354
70505|pass|200211|alian@cpan.org|Data-Hexdumper|1.0.1|sparc-netbsd|5.6.1|netbsd|1.5.1|200211211512
78607|pass|200302|alian@cpan.org ((Cpan Tester - CPAN++ Automate ))|Data-Compare|0.02|i386-netbsd-thread-multi|5.8.0|netbsd|1.6|200302110230
78663|pass|200302|alian@cpan.org ((CPAN Tester = CPAN++ automate))|Data-Compare|0.02|ppc-linux|5.8.0|linux|2.4.18-0.9a|200302110322
78789|pass|200302|alian@cpan.org ((CPAN Tester''CPAN++ automate))|Data-Compare|0.02|sun4-solaris|5.8.0|solaris|2.7|200302110526
79039|pass|200302|alian@cpan.org ((CPAN Tester_CPAN++ automate))|Data-Compare|0.02|ppc-linux|5.8.0|linux|2.2.16|200302110959
79143|pass|200302|alian@cpan.org|Data-Compare|0.02|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200302111124
79430|pass|200302|alian@cpan.org ((alian))|Data-Compare|0.02|sparc-linux|5.6.1|linux|2.4.18|200302120456
79656|pass|200302|alian@cpan.org ((CPAN Tester / CPAN++ automate))|Data-Compare|0.02|sparc-netbsd|5.8.0|netbsd|1.5.3|200302130551
86049|pass|200303|root@localhost.localdomain ((root))|Data-Compare|0.02|i386-linux|5.8.0|linux|2.4.18-11smp|200303300513
90155|pass|200305|alian@cpan.org ((alian))|Data-Hexdumper|1.0.1|i686-linux-thread-multi-64all-ld|5.8.0|linux|2.4.20|200305131230
91242|pass|200305|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Bryar|1.0|sun4-solaris|5.8.0|solaris|2.8|200305242112
91335|pass|200305|cpantester@calaquendi.net ((Jeroen Latour))|Bryar|1.0|i586-linux|5.8.0|linux|2.2.16c32_iii|200305260746
91402|fail|200305|crazyinsomniac@yahoo.com (DH)|Bryar|1.0|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200305270008
92820|pass|200306|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Bryar|1.2|sun4-solaris|5.8.0|solaris|2.8|200306150155
94030|pass|200306|cpantester@calaquendi.net ((Jeroen Latour))|NestedMap|1.0|i586-linux|5.8.0|linux|2.2.16c32_iii|200306261252
94034|pass|200306|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|NestedMap|1.0|sun4-solaris|5.8.0|solaris|2.8|200306261239
94252|pass|200306|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|File-Find-Rule-Permissions|1.0|sun4-solaris|5.8.0|solaris|2.8|200306300842
94315|pass|200306|cpantester@calaquendi.net ((Jeroen Latour))|File-Find-Rule-Permissions|1.0|i586-linux|5.8.0|linux|2.2.16c32_iii|200306302104
95944|pass|200307|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|File-Find-Rule-Permissions|1.1|sun4-solaris|5.8.0|solaris|2.8|200307230034
97303|pass|200308|cpantester@calaquendi.net ((Jeroen Latour))|File-Find-Rule-Permissions|1.1|i586-linux|5.8.0|linux|2.2.16c32_iii|200308061042
97862|pass|200308|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Hexdumper|1.1|sun4-solaris|5.8.0|solaris|2.8|200308101707
98649|fail|200308|cpantester@calaquendi.net ((Jeroen Latour))|Bryar|2.0|i586-linux|5.8.0|linux|2.2.16c32_iii|200308161636
98691|fail|200308|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Bryar|2.0|sun4-solaris|5.8.0|solaris|2.8|200308161658
99060|pass|200308|root@dropbear ((root))|Data-Hexdumper|1.1|i586-linux|5.8.0|linux|2.4.20-22em|200308221803
99725|pass|200308|cpantester@calaquendi.net ((Jeroen Latour))|Tie-Hash-Longest|1.0.reupload-again-because-im-stupid|i586-linux|5.8.0|linux|2.2.16c32_iii|200308290931
99860|pass|200308|crazyinsomniac@yahoo.com|Tie-Hash-Longest|1.0.reupload-again-because-im-stupid|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200308310346
100133|pass|200309|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Tie-Hash-Longest|1.0.reupload-again-because-im-stupid|sun4-solaris|5.8.0|solaris|2.8|200309011741
100444|pass|200309|jlatour@calaquendi.net|Acme-Scurvy-Whoreson-BilgeRat|1.0|i586-linux|5.8.0|linux|2.2.16c32_iii|200309030257
100450|pass|200309|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Acme-Scurvy-Whoreson-BilgeRat|1.0|sun4-solaris|5.8.0|solaris|2.8|200309030244
103212|pass|200309|Perl_Test@LogicalChaos.org|Data-Hexdumper|1.1|i386-linux|5.8.0|linux|2.4.20-2mdkenterprise|200309280407
104500|unknown|200310|kriegjcb@mi.ruhr-uni-bochum.de ((Jost Krieger))|Acme-Pony|1.1.2|sun4-solaris|5.8.1|solaris|2.8|200310061309
105242|pass|200310|webmaster@thr.msu.edu|Data-Compare|0.02|i386-freebsd|5.6.1|freebsd|5.1-current|200310091009
105391|pass|200310|webmaster@thr.msu.edu|Data-Hexdumper|1.1|i386-freebsd|5.6.1|freebsd|5.1-current|200310091842
105564|pass|200310|webmaster@thr.msu.edu|File-Find-Rule-Permissions|1.1|i386-freebsd|5.6.1|freebsd|5.1-current|200310100200
107669|pass|200310|webmaster@thr.msu.edu|Tie-Hash-Longest|1.0.reupload-again-because-im-stupid|i386-freebsd|5.6.1|freebsd|5.1-current|200310191312
107670|pass|200310|webmaster@thr.msu.edu|Tie-Hash-Transactional|1.0|i386-freebsd|5.6.1|freebsd|5.1-current|200310191312
107671|pass|200310|webmaster@thr.msu.edu|Tie-Hash-Rank|1.0.1|i386-freebsd|5.6.1|freebsd|5.1-current|200310191312
107672|pass|200310|webmaster@thr.msu.edu|Tie-Scalar-Decay|1.1.1|i386-freebsd|5.6.1|freebsd|5.1-current|200310191313
107712|pass|200310|Perl_Test@LogicalChaos.org|Tie-Hash-Rank|1.0.1|i386-linux|5.8.0|linux|2.4.20-2mdkenterprise|200310191223
107713|pass|200310|Perl_Test@LogicalChaos.org|Tie-Scalar-Decay|1.1.1|i386-linux|5.8.0|linux|2.4.20-2mdkenterprise|200310191223
107906|pass|200310|nothingmuch@woobling.org|Games-Dice-Advanced|1.0|darwin|5.8.1|darwin|6.8|200310210201
108039|pass|200310|webmaster@thr.msu.edu|Games-Dice-Advanced|1.0|i386-freebsd|5.6.1|freebsd|5.1-current|200310211714
108140|pass|200310|Perl_Test@LogicalChaos.org|Games-Dice-Advanced|1.0|i386-linux|5.8.0|linux|2.4.20-2mdkenterprise|200310220104
108339|pass|200310|barbie@missbarbell.co.uk (Barbie)|Games-Dice-Advanced|1.0|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200310222256
108348|pass|200310|nothingmuch@woobling.org|Net-Random|1.0|darwin|5.8.1|darwin|6.8|200310230107
108357|pass|200310|Jost.Krieger+smokeback@sunu991.rz.ruhr-uni-bochum.de ((Jost Krieger))|Net-Random|1.0|sun4-solaris|5.8.1|solaris|2.8|200310230036
108976|fail|200310|barbie@missbarbell.co.uk (Barbie)|Net-Random|1.0|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200310280741
111603|pass|200311|nothingmuch@woobling.org|Sub-WrapPackages|1.0|darwin|5.8.1|darwin|6.8|200311152328
111628|pass|200311|Jost.Krieger+smokeback@sunu991.rz.ruhr-uni-bochum.de ((Jost Krieger))|Sub-WrapPackages|1.0|sun4-solaris|5.8.1|solaris|2.8|200311161149
111640|pass|200311|nothingmuch@woobling.org|Statistics-ChiSquare|0.5|darwin|5.8.1|darwin|6.8|200311161944
111706|pass|200311|nothingmuch@woobling.org|Statistics-ChiSquare|0.5|powerpc-linux|5.8.0|linux|2.4.23-pre5|200311172329
111711|pass|200311|nothingmuch@woobling.org|Tie-Hash-Longest|1.1|darwin|5.8.1|darwin|6.8|200311180043
111714|pass|200311|nothingmuch@woobling.org|Tie-Hash-Longest|1.1|powerpc-linux|5.8.0|linux|2.4.23-pre5|200311180020
111767|pass|200311|Jost.Krieger+smokeback@sunu991.rz.ruhr-uni-bochum.de ((Jost Krieger))|Statistics-ChiSquare|0.5|sun4-solaris|5.8.1|solaris|2.8|200311181021
111776|pass|200311|Jost.Krieger+smokeback@sunu991.rz.ruhr-uni-bochum.de ((Jost Krieger))|Tie-Hash-Longest|1.1|sun4-solaris|5.8.1|solaris|2.8|200311181049
111919|pass|200311|webmaster@thr.msu.edu|Sub-WrapPackages|1.0|i386-freebsd|5.8.2|freebsd|5.1-current|200311191538
111932|pass|200311|webmaster@thr.msu.edu|Statistics-ChiSquare|0.5|i386-freebsd|5.8.2|freebsd|5.1-current|200311191552
111946|pass|200311|webmaster@thr.msu.edu|Tie-Hash-Longest|1.1|i386-freebsd|5.8.2|freebsd|5.1-current|200311191635
111994|pass|200311|barbie@missbarbell.co.uk (Barbie)|Tie-Hash-Longest|1.1|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200311200824
112009|pass|200311|barbie@missbarbell.co.uk (Barbie)|Statistics-ChiSquare|0.5|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200311200824
112018|pass|200311|barbie@missbarbell.co.uk (Barbie)|Sub-WrapPackages|1.0|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200311200824
112080|pass|200311|nothingmuch@woobling.org|Statistics-SerialCorrelation|1.0|darwin|5.8.1|darwin|6.8|200311211439
112103|pass|200311|nothingmuch@woobling.org|Statistics-SerialCorrelation|1.0|powerpc-linux|5.8.0|linux|2.4.23-pre5|200311211803
112146|pass|200311|webmaster@thr.msu.edu|Statistics-SerialCorrelation|1.0|i386-freebsd|5.8.2|freebsd|5.1-current|200311220119
112172|pass|200311|Perl_Test@LogicalChaos.org|Sub-WrapPackages|1.0|i386-linux|5.8.0|linux|2.4.20-2mdkenterprise|200311220236
112185|pass|200311|Perl_Test@LogicalChaos.org|Statistics-ChiSquare|0.5|i386-linux|5.8.0|linux|2.4.20-2mdkenterprise|200311220248
112197|pass|200311|Perl_Test@LogicalChaos.org|Tie-Hash-Longest|1.1|i386-linux|5.8.0|linux|2.4.20-2mdkenterprise|200311220323
112240|pass|200311|Perl_Test@LogicalChaos.org|Statistics-SerialCorrelation|1.0|i386-linux|5.8.0|linux|2.4.20-2mdkenterprise|200311220439
112272|pass|200311|Jost.Krieger+smokeback@sunu991.rz.ruhr-uni-bochum.de ((Jost Krieger))|Statistics-SerialCorrelation|1.0|sun4-solaris|5.8.1|solaris|2.8|200311221038
112788|pass|200311|barbie@missbarbell.co.uk (Barbie)|Statistics-SerialCorrelation|1.0|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200311270726
113523|pass|200312|root@uk-wrk-0017 ((root))|Sub-WrapPackages|1.0|i386-linux|5.8.1|linux|2.4.22-xfs+ti1211|200312021956
114594|fail|200312|Jost.Krieger+smokeback@sunu991.rz.ruhr-uni-bochum.de ((Jost Krieger))|Bryar|2.1|sun4-solaris|5.8.1|solaris|2.8|200312121042
115002|pass|200312|barbie@missbarbell.co.uk (Barbie)|Bryar|2.1|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200312152024
115405|pass|200312|Jost.Krieger+smokeback@sunu991.rz.ruhr-uni-bochum.de ((Jost Krieger))|File-Find-Rule-Permissions|1.2|sun4-solaris|5.8.1|solaris|2.8|200312191034
115506|pass|200312|webmaster@thr.msu.edu|File-Find-Rule-Permissions|1.2|i386-freebsd|5.6.1|freebsd|5.2-beta|200312200410
115565|na|200312|barbie@missbarbell.co.uk (Barbie)|File-Find-Rule-Permissions|1.2|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200312201228
115890|pass|200312|Jost.Krieger+smokeback@sunu991.rz.ruhr-uni-bochum.de ((Jost Krieger))|Data-Compare|0.03|sun4-solaris|5.8.1|solaris|2.8|200312241039
115892|pass|200312|Jost.Krieger+smokeback@sunu991.rz.ruhr-uni-bochum.de ((Jost Krieger))|Data-Compare|0.04|sun4-solaris|5.8.1|solaris|2.8|200312241040
115938|pass|200312|barbie@missbarbell.co.uk (Barbie)|Data-Compare|0.04|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200312242048
115941|pass|200312|barbie@missbarbell.co.uk (Barbie)|Data-Compare|0.03|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200312242048
115967|pass|200312|lithron@bellsouth.net|Data-Compare|0.04|cygwin-thread-multi-64int|5.8.2|cygwin|1.5.5(0.9432)|200312250300
115981|pass|200312|webmaster@thr.msu.edu|Data-Compare|0.03|i386-freebsd|5.8.2|freebsd|5.2-current|200312251145
115983|pass|200312|webmaster@thr.msu.edu|Data-Compare|0.04|i386-freebsd|5.8.2|freebsd|5.2-current|200312251146
116371|pass|200312|Jost.Krieger+smokeback@sunu991.rz.ruhr-uni-bochum.de ((Jost Krieger))|Data-Compare|0.05|sun4-solaris|5.8.1|solaris|2.8|200312311027
116578|pass|200401|webmaster@thr.msu.edu|Data-Compare|0.05|i386-freebsd|5.8.2|freebsd|5.2-current|200401021841
117144|pass|200401|barbie@missbarbell.co.uk (Barbie)|Data-Compare|0.05|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200401062214
117236|pass|200401|Jost.Krieger@ruhr-uni-bochum.de|Data-Compare|0.06|sun4-solaris|5.8.1|solaris|2.8|200401070853
117337|pass|200401|jlatour@calaquendi.net|Data-Compare|0.06|i586-linux|5.8.2|linux|2.2.16c37_iii|200401080005
117380|pass|200401|webmaster@thr.msu.edu|Data-Compare|0.06|i386-freebsd|5.8.2|freebsd|5.2-current|200401080041
117485|pass|200401|jlatour@calaquendi.net|Data-Compare|0.07|i586-linux|5.8.2|linux|2.2.16c37_iii|200401081057
117505|pass|200401|Jost.Krieger@ruhr-uni-bochum.de|Data-Compare|0.07|sun4-solaris|5.8.1|solaris|2.8|200401081034
117607|pass|200401|webmaster@thr.msu.edu|Data-Compare|0.07|i386-freebsd|5.8.2|freebsd|5.2-current|200401090408
117898|pass|200401|barbie@missbarbell.co.uk (Barbie)|Data-Compare|0.07|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200401100851
117907|pass|200401|barbie@missbarbell.co.uk (Barbie)|Data-Compare|0.06|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200401100851
119387|pass|200401|Jost.Krieger@ruhr-uni-bochum.de|Data-Compare|0.08|sun4-solaris|5.8.1|solaris|2.8|200401210824
119444|pass|200401|steve@gandalf.workpc.tds.net ((root))|Data-Compare|0.08|i386-linux|5.8.1|linux|2.4.21-2.elsmp|200401211242
119557|pass|200401|barbie@missbarbell.co.uk (Barbie)|Data-Compare|0.08|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200401222135
119724|pass|200401|steve@gandalf.workpc.tds.net ((root))|Bryar|2.3|i386-linux|5.8.1|linux|2.4.21-2.elsmp|200401231111
119747|pass|200401|Jost.Krieger+smokeback@sunu991.rz.ruhr-uni-bochum.de ((Jost Krieger))|Bryar|2.2|sun4-solaris|5.8.1|solaris|2.8|200401241057
119757|pass|200401|Jost.Krieger+smokeback@sunu991.rz.ruhr-uni-bochum.de ((Jost Krieger))|Statistics-SerialCorrelation|1.1|sun4-solaris|5.8.1|solaris|2.8|200401241132
119881|pass|200401|webmaster@thr.msu.edu|Statistics-SerialCorrelation|1.1|i386-freebsd|5.8.2|freebsd|5.2-current|200401260203
120817|pass|200402|barbie@missbarbell.co.uk (Barbie)|Bryar|2.2|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200402030754
120850|pass|200402|barbie@missbarbell.co.uk (Barbie)|Bryar|2.3|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200402030754
120895|pass|200402|barbie@missbarbell.co.uk (Barbie)|Statistics-SerialCorrelation|1.1|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200402030755
121670|pass|200402|nothingmuch@woobling.org|Bryar|2.4|ppc-darwin-thread-multi|5.8.3|darwin|7.2.0|200402081917
121763|pass|200402|Jost.Krieger+smokeback@sunu991.rz.ruhr-uni-bochum.de ((Jost Krieger))|Bryar|2.5|sun4-solaris|5.8.1|solaris|2.8|200402091052
122036|pass|200402|nothingmuch@woobling.org|Bryar|2.6|ppc-darwin-thread-multi|5.8.3|darwin|7.2.0|200402111538
122175|pass|200402|barbie@missbarbell.co.uk (Barbie)|Bryar|2.5|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200402120813
122192|pass|200402|barbie@missbarbell.co.uk (Barbie)|Bryar|2.4|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200402120813
122308|pass|200402|barbie@missbarbell.co.uk (Barbie)|Bryar|2.6|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200402130742
123333|pass|200402|nothingmuch@woobling.org|Data-Hexdumper|1.2|ppc-darwin-thread-multi|5.8.3|darwin|7.2.0|200402201702
123394|pass|200402|cpansmoke@alternation.net|Data-Hexdumper|1.2|i586-linux|5.8.0|linux|2.4.22-4tr|200402210507
123399|pass|200402|Jost.Krieger+smokeback@sunu991.rz.ruhr-uni-bochum.de ((Jost Krieger))|Data-Hexdumper|1.2|sun4-solaris|5.8.1|solaris|2.8|200402211022
123609|pass|200402|webmaster@thr.msu.edu|Data-Hexdumper|1.2|i386-freebsd|5.8.2|freebsd|5.2-current|200402221849
123735|pass|200402|nothingmuch@woobling.org|Data-Compare|0.09|ppc-darwin-thread-multi|5.8.3|darwin|7.2.0|200402240107
123796|pass|200402|Jost.Krieger+smokeback@sunu991.rz.ruhr-uni-bochum.de ((Jost Krieger))|Data-Compare|0.09|sun4-solaris|5.8.1|solaris|2.8|200402241052
123817|pass|200402|cpansmoke@alternation.net|Data-Compare|0.09|i586-linux|5.8.0|linux|2.4.22-4tr|200402240807
123896|pass|200402|webmaster@thr.msu.edu|Data-Compare|0.09|i386-freebsd|5.8.2|freebsd|5.2-current|200402241850
124187|pass|200402|barbie@missbarbell.co.uk (Barbie)|Data-Hexdumper|1.2|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200402260808
124420|pass|200402|barbie@missbarbell.co.uk (Barbie)|Data-Compare|0.09|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200402270721
127016|pass|200403|nothingmuch@woobling.org|Data-Compare|0.1|ppc-darwin-thread-multi|5.8.3|darwin|7.2.0|200403131335
127052|pass|200403|cpansmoke@alternation.net|Data-Compare|0.1|i586-linux|5.8.0|linux|2.4.22-4tr|200403131807
127114|pass|200403|kriegjcb@sunu991.rz.ruhr-uni-bochum.de|Data-Compare|0.1|sun4-solaris|5.8.1|solaris|2.8|200403141030
127238|pass|200403|webmaster@thr.msu.edu|Data-Compare|0.1|i386-freebsd|5.8.2|freebsd|5.2-current|200403141830
128050|pass|200403|barbie@missbarbell.co.uk (Barbie)|Data-Compare|0.1|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200403180756
129983|pass|200403|Jost.Krieger@ruhr-uni-bochum.de|Data-Compare|0.1|sun4-solaris-thread-multi|5.8.3|solaris|2.8|200403291616
132100|pass|200404|cpansmoke@alternation.net|Bryar|2.6|i586-linux|5.8.0|linux|2.4.22-4tr|200404090008
132861|pass|200404|kriegjcb@sunu991.rz.ruhr-uni-bochum.de|Bryar|2.6|sun4-solaris|5.8.1|solaris|2.8|200404150923
142676|pass|200406|Jost.Krieger@ruhr-uni-bochum.de|Data-Compare|0.11|sun4-solaris-thread-multi|5.8.3|solaris|2.8|200406030737
142964|pass|200406|rrwo@cpan.org|Data-Compare|0.11|MSWin32-x86-multi-thread|5.8.3|MSWin32|4.0|200406040942
146822|pass|200407|cpansmoke@alternation.net|Data-Compare|0.11|i586-linux|5.8.0|linux|2.4.22-4tr|200407050908
147721|pass|200407|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Statistics-ChiSquare|0.5|sun4-solaris-thread-multi|5.8.3|solaris|2.8|200407080346
148267|pass|200407|rrwo@cpan.org|Class-CanBeA|1.0|MSWin32-x86-multi-thread|5.8.3|MSWin32|4.0|200407110109
148277|pass|200407|rrwo@cpan.org|Class-CanBeA|1.0|MSWin32-x86-multi-thread|5.8.4|MSWin32|4.0|200407110250
148287|pass|200407|cpansmoke@alternation.net|Class-CanBeA|1.0|i586-linux|5.8.0|linux|2.4.22-4tr|200407110408
148986|pass|200407|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Class-CanBeA|1.0|sun4-solaris-thread-multi|5.8.3|solaris|2.8|200407140502
151471|pass|200407|webmaster@thr.msu.edu|Data-Compare|0.11|i386-freebsd-64int|5.8.4|freebsd|5.2-current|200407301504
154082|pass|200408|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Number-Phone|1.0|sun4-solaris-thread-multi|5.8.3|solaris|2.8|200408182140
154114|pass|200408|cpantester@thr.msu.edu|Number-Phone|1.0|i386-freebsd-64int|5.8.5|freebsd|5.2-current|200408190127
154121|pass|200408|rrwo@cpan.org|Number-Phone|1.0|MSWin32-x86-multi-thread|5.8.4|MSWin32|4.0|200408190341
154183|pass|200408|martin.thurn@ngc.com ("Thurn, Martin")|Number-Phone|1.0|sun4-solaris|5.9.1|solaris|2.8|200408190754
155673|unknown|200409|jim@jimandkoka.com|Acme-Pony|1.1.2|darwin-2level|5.8.4|darwin|7.3.0|200409021509
156532|pass|200409|Jost.Krieger+perl@rub.de|Net-Random|1.1|sun4-solaris-thread-multi|5.8.3|solaris|2.8|200409071326
156605|pass|200409|alian@cpan.org ((CPAN Tester + CPAN++ automate))|Net-Random|1.1|ppc-linux|5.8.4|linux|2.4.19-4a|200409072209
156606|pass|200409|alian@cpan.org ((CPAN Tester. CPAN++ automate))|Net-Random|1.1|OpenBSD.i386-openbsd|5.8.4|openbsd|3.5|200409080011
156917|fail|200409|alian@cpan.org ((Cpan Tester - CPAN++ Automate ))|Net-Random|1.1|i386-netbsd-64int|5.8.4|netbsd|1.6|200409090006
157366|pass|200409|barbie@missbarbell.co.uk (Barbie)|Net-Random|1.1|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200409110708
158211|pass|200409|Jost.Krieger+perl@rub.de|Number-Phone|1.1|sun4-solaris-thread-multi|5.8.3|solaris|2.8|200409160411
158232|pass|200409|alian@cpan.org ((Cpan Tester - CPAN++ Automate ))|Number-Phone|1.1|i386-netbsd-64int|5.8.4|netbsd|1.6|200409161150
158233|pass|200409|alian@cpan.org ((CPAN Tester. CPAN++ automate))|Number-Phone|1.1|OpenBSD.i386-openbsd|5.8.4|openbsd|3.5|200409161349
158237|pass|200409|alian@cpan.org ((alian))|Number-Phone|1.1|sparc64-linux|5.8.5|linux|2.4.18|200409161204
158254|pass|200409|martin.thurn@ngc.com ("Thurn, Martin")|Number-Phone|1.1|sun4-solaris|5.9.1|solaris|2.8|200409160520
158727|pass|200409|cpantester@thr.msu.edu|Net-Random|1.1|i386-freebsd-64int|5.8.5|freebsd|5.2-current|200409171223
158921|pass|200409|cpantester@thr.msu.edu|Number-Phone|1.1|i386-freebsd-64int|5.8.5|freebsd|5.2-current|200409180135
159531|pass|200409|alian@cpan.org ((CPAN Tester + CPAN++ automate))|Number-Phone|1.1|ppc-linux|5.8.4|linux|2.4.19-4a|200409210855
159704|pass|200409|barbie@missbarbell.co.uk (Barbie)|Number-Phone|1.1|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200409220830
160038|pass|200409|Jost.Krieger+perl@rub.de|Number-Phone|1.2|sun4-solaris-thread-multi|5.8.3|solaris|2.8|200409230357
160051|pass|200409|alian@cpan.org ((CPAN Tester + CPAN++ automate))|Number-Phone|1.2|ppc-linux|5.8.4|linux|2.4.19-4a|200409230919
160093|pass|200409|martin.thurn@ngc.com ("Thurn, Martin")|Number-Phone|1.2|sun4-solaris|5.9.1|solaris|2.8|200409230617
162192|pass|200410|Jost.Krieger+perl@rub.de|Number-Phone|1.2001|sun4-solaris-thread-multi|5.8.3|solaris|2.8|200410132331
162785|pass|200410|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Bryar|2.8_01|sun4-solaris-thread-multi|5.8.3|solaris|2.8|200410190938
162948|pass|200410|Jost.Krieger+perl@rub.de|Number-Phone|1.3|sun4-solaris-thread-multi|5.8.3|solaris|2.8|200410210359
163103|pass|200410|Jost.Krieger+perl@rub.de|Bryar|2.6|sun4-solaris-thread-multi|5.8.3|solaris|2.8|200410230659
163394|pass|200410|Jost.Krieger+perl@rub.de|Acme-Scurvy-Whoreson-BilgeRat|1.1|sun4-solaris-thread-multi|5.8.3|solaris|2.8|200410260359
163426|pass|200410|shlomif@iglu.org.il (Shlomi Fish)|Number-Phone|1.3|i686-linux|5.8.5|linux|2.6.8.1-10mdksmp|200410261443
163536|pass|200410|barbie@missbarbell.co.uk|Number-Phone|1.2|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200410270912
164481|pass|200411|cpantester@thr.msu.edu|Number-Phone|1.2|i386-freebsd-64int|5.8.5|freebsd|5.2-current|200411021233
164948|pass|200411|cpantester@thr.msu.edu|Number-Phone|1.2001|i386-freebsd-64int|5.8.5|freebsd|5.2-current|200411031032
165117|pass|200411|corion+cpansmoker@corion.net|Acme-Scurvy-Whoreson-BilgeRat|1.1|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200411031909
165645|pass|200411|saper@cpan.org|Data-Compare|0.11|i386-linux|5.8.1|linux|2.4.18-23mdksmp|200411071232
165739|pass|200411|corion-cpansmoker@corion.net|Data-Transactional|0.1|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200411071010
165993|pass|200411|cpantester@thr.msu.edu|Data-Transactional|0.1|i386-freebsd-64int|5.8.5|freebsd|5.2-current|200411090940
166096|pass|200411|saper@cpan.org|Data-Compare|0.13|i386-linux|5.8.1|linux|2.4.18-23mdksmp|200411110826
166129|pass|200411|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Compare|0.12|sun4-solaris-thread-multi|5.8.3|solaris|2.8|200411100845
166130|pass|200411|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Compare|0.13|sun4-solaris-thread-multi|5.8.3|solaris|2.8|200411100846
166207|pass|200411|barbie@missbarbell.co.uk|Number-Phone|1.2001|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200411101259
166420|pass|200411|rrwo@cpan.org|Data-Compare|0.13|MSWin32-x86-multi-thread|5.8.4|MSWin32|4.0|200411110047
166421|pass|200411|rrwo@cpan.org|Data-Compare|0.12|MSWin32-x86-multi-thread|5.8.4|MSWin32|4.0|200411110048
166465|pass|200411|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Number-Phone|1.3001|sun4-solaris-thread-multi|5.8.3|solaris|2.8|200411111217
166521|pass|200411|barbie@missbarbell.co.uk|Number-Phone|1.3|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200411111855
166525|fail|200411|barbie@missbarbell.co.uk|Bryar|2.8_01|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200411111855
166546|fail|200411|barbie@missbarbell.co.uk|Data-Compare|0.13|MSWin32-x86-multi-thread|5.6.1|MSWin32|4.0|200411111855
166588|pass|200411|cpantester@thr.msu.edu|Data-Compare|0.12|i386-freebsd-64int|5.8.5|freebsd|5.2-current|200411111416
166590|pass|200411|cpantester@thr.msu.edu|Data-Compare|0.13|i386-freebsd-64int|5.8.5|freebsd|5.2-current|200411111416
166610|pass|200411|corion-cpansmoker@corion.net|Number-Phone|1.3001|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200411112036
166652|pass|200411|cpantester@thr.msu.edu|Number-Phone|1.3001|i386-freebsd-64int|5.8.5|freebsd|5.2-current|200411111530
166717|pass|200411|Jost.Krieger+cpanplus@rub.de|Data-Transactional|0.1|sun4-solaris-thread-multi|5.8.5|solaris|2.9|200411120929
167036|pass|200411|shlomif@iglu.org.il (Shlomi Fish)|Number-Phone|1.3001|i686-linux|5.8.5|linux|2.6.8.1-10mdksmp|200411132058
169429|pass|200411|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Number-Phone|1.3002|sun4-solaris-thread-multi|5.8.5|solaris|2.9|200411261633
169442|pass|200411|nothingmuch@woobling.org|Number-Phone|1.3002|i686-linux|5.9.1|linux|2.6.8.1|200411262046
169444|pass|200411|nothingmuch@woobling.org|Number-Phone|1.3002|i386-linux-thread|5.8.4|linux|2.6.7-gentoo-r11|200411262047
169521|fail|200411|nothingmuch@woobling.org|Number-Phone|1.3002|i686-linux|5.5.4|linux|2.6.8.1|200411262045
169522|pass|200411|nothingmuch@woobling.org|Number-Phone|1.3002|i686-linux|5.6.2|linux|2.6.8.1|200411262046
169524|pass|200411|nothingmuch@woobling.org|Number-Phone|1.3002|i686-linux-thread-multi|5.9.1|linux|2.6.8.1|200411262047
169526|pass|200411|nothingmuch@woobling.org|Number-Phone|1.3002|i686-linux|5.8.5|linux|2.6.8.1|200411262048
169528|pass|200411|nothingmuch@woobling.org|Number-Phone|1.3002|i686-linux-thread-multi|5.8.5|linux|2.6.8.1|200411262048
169559|pass|200411|corion-cpansmoker@corion.net|Number-Phone|1.3002|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200411270047
169740|pass|200411|saper@cpan.org|Number-Phone|1.3002|i386-linux|5.8.1|linux|2.4.18-23mdksmp|200411271546
169753|pass|200411|saper@cpan.org|Number-Phone|1.3002|darwin|5.6.0|darwin|6.0|200411271602
172027|pass|200412|Jost.Krieger+cpanplus@rub.de|Number-Phone|1.3003|sun4-solaris-thread-multi|5.8.5|solaris|2.9|200412060512
172039|fail|200412|nothingmuch@woobling.org|Number-Phone|1.3003|i686-linux|5.5.4|linux|2.6.8.1|200412061054
172045|pass|200412|nothingmuch@woobling.org|Number-Phone|1.3003|i686-linux|5.6.2|linux|2.6.8.1|200412061106
172055|pass|200412|nothingmuch@woobling.org|Number-Phone|1.3003|i686-linux|5.9.2 patch 23609|linux|2.6.8.1|200412061115
172072|pass|200412|nothingmuch@woobling.org|Number-Phone|1.3003|i686-linux-thread-multi|5.9.2 patch 23609|linux|2.6.8.1|200412061138
172080|pass|200412|nothingmuch@woobling.org|Number-Phone|1.3003|i686-linux|5.8.6|linux|2.6.8.1|200412061149
172095|pass|200412|nothingmuch@woobling.org|Number-Phone|1.3003|i686-linux-thread-multi|5.8.6|linux|2.6.8.1|200412061203
172107|pass|200412|saper@cpan.org|Number-Phone|1.3003|i386-linux|5.8.1|linux|2.4.18-23mdksmp|200412061110
172118|pass|200412|nothingmuch@woobling.org|Number-Phone|1.3003|i386-linux-thread|5.8.4|linux|2.6.7-gentoo-r11|200412061222
172178|pass|200412|corion-cpansmoker@corion.net|Number-Phone|1.3003|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200412061934
172666|pass|200412|foo@bar.com|Acme-Pony|1.1.2|darwin-thread-multi-2level|5.8.1 RC3|darwin|7.0|200412081422
172735|fail|200412|nothingmuch@woobling.org|Data-Compare|0.13|i686-linux|5.6.2|linux|2.6.8.1|200412090318
172774|pass|200412|nothingmuch@woobling.org|Data-Compare|0.13|i686-linux|5.8.6|linux|2.6.8.1|200412090439
172783|pass|200412|nothingmuch@woobling.org|Data-Compare|0.13|i686-linux-thread-multi|5.8.6|linux|2.6.8.1|200412090454
174147|pass|200412|nothingmuch@woobling.org|Sub-WrapPackages|1.0|i686-linux|5.6.2|linux|2.6.8.1|200412151759
174466|pass|200412|nothingmuch@woobling.org|Sub-WrapPackages|1.0|i686-linux|5.9.2 patch 23656|linux|2.6.8.1|200412170002
174476|pass|200412|nothingmuch@woobling.org|Sub-WrapPackages|1.0|i686-linux-thread-multi|5.9.2 patch 23656|linux|2.6.8.1|200412170021
174501|pass|200412|nothingmuch@woobling.org|Sub-WrapPackages|1.0|i686-linux-thread-multi|5.8.6|linux|2.6.8.1|200412170049
174828|pass|200412|saper@cpan.org|Sub-WrapPackages|1.0|darwin|5.6.0|darwin|6.0|200412190038
175191|pass|200412|nothingmuch@woobling.org|Class-CanBeA|1.1|i686-linux|5.6.2|linux|2.6.8.1|200412211016
175207|pass|200412|nothingmuch@woobling.org|Class-CanBeA|1.1|i686-linux|5.9.2 patch 23662|linux|2.6.8.1|200412211029
175225|pass|200412|nothingmuch@woobling.org|Class-CanBeA|1.1|i686-linux-thread-multi|5.9.2 patch 23662|linux|2.6.8.1|200412211043
175238|pass|200412|nothingmuch@woobling.org|Class-CanBeA|1.1|i386-linux-thread|5.8.4|linux|2.6.7-gentoo-r11|200412211101
175264|pass|200412|nothingmuch@woobling.org|Class-CanBeA|1.1|i686-linux|5.8.6|linux|2.6.8.1|200412211123
175282|pass|200412|nothingmuch@woobling.org|Class-CanBeA|1.1|i686-linux-thread-multi|5.8.6|linux|2.6.8.1|200412211139
175392|pass|200412|rrwo@cpan.org|Class-CanBeA|1.1|MSWin32-x86-multi-thread|5.8.4|MSWin32|4.0|200412211514
175604|pass|200412|Jost.Krieger@ruhr-uni-bochum.de|Class-CanBeA|1.1|sun4-solaris-thread-multi|5.8.5|solaris|2.9|200412221554
175780|pass|200412|saper@cpan.org|Class-CanBeA|1.1|i386-linux|5.8.1|linux|2.4.18-23mdksmp|200412230008
176594|pass|200501|rrwo@cpan.org|Data-Compare|0.13|MSWin32-x86-multi-thread|5.8.4|MSWin32|4.0|200501011906
176599|pass|200501|rrwo@cpan.org|Data-Compare|0.13|MSWin32-x86-multi-thread|5.8.4|MSWin32|4.0|200501011908
177521|pass|200501|cpantester@thr.msu.edu|Number-Phone|1.3003|i386-freebsd-64int|5.8.5|freebsd|5.2-current|200501080223
177682|pass|200501|cpantester@thr.msu.edu|Sub-WrapPackages|1.0|i386-freebsd-64int|5.8.5|freebsd|5.2-current|200501080846
177830|pass|200501|cpantester@thr.msu.edu|Class-CanBeA|1.1|i386-freebsd-64int|5.8.5|freebsd|5.2-current|200501090029
178996|pass|200501|saper@cpan.org|Data-Compare|0.13|i386-linux|5.8.5|linux|2.4.19-44mdkenterprise|200501162222
179170|pass|200501|root@fisharerojo.org ((Charlie &))|Acme-Scurvy-Whoreson-BilgeRat|1.1|i386-openbsd|5.8.5|openbsd|3.6|200501180850
179209|pass|200501|shlomif@iglu.org.il (Shlomi Fish)|Sub-WrapPackages|1.0|i386-linux|5.8.5|linux|2.4.19-44mdkenterprise|200501181930
179679|pass|200501|rrwo@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|MSWin32-x86-multi-thread|5.8.6|MSWin32|4.0|200501201640
180545|pass|200501|saperghis@aacom.fr|Data-Compare|0.13|i386-linux|5.8.3|linux|2.6.0-1mdkenterprise|200501241408
180718|pass|200501|root@fisharerojo.org ((Charlie &))|Bryar|2.6|i386-openbsd|5.8.5|openbsd|3.6|200501250751
181235|fail|200501|corion-cpansmoker@corion.net|Palm-SMS|0.02|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200501290957
181709|pass|200502|cpansmoke@alternation.net|Sub-WrapPackages|1.0|i586-linux|5.8.3|linux|2.4.21-13trfirewallsmp|200502020612
181897|pass|200502|saper@cpan.org|Sub-WrapPackages|1.0|i386-linux|5.8.5|linux|2.4.19-44mdkenterprise|200502022306
182231|pass|200502|cpantester@thr.msu.edu|Acme-Scurvy-Whoreson-BilgeRat|1.1|i386-freebsd-64int|5.8.6|freebsd|5.3-stable|200502031858
185115|pass|200502|rrwo@cpan.org|Data-Compare|0.13|cygwin-thread-multi-64int|5.8.6|cygwin|1.5.12(0.11642)|200502182335
189095|pass|200503|saper@cpan.org|Data-Hexdumper|1.2|i386-linux|5.8.5|linux|2.4.19-44mdkenterprise|200503100108
196329|pass|200504|leeym@utopia.leeym.com ((Yen-Ming Lee))|Data-Compare|0.13|i386-freebsd-64int|5.8.6|freebsd|5.4-prerelease|200504032333
196772|pass|200504|cpan@calaquendi.net|Data-Compare|0.13|i386-freebsd-64int|5.8.6|freebsd|4.10-release-p3|200504042125
197527|pass|200504|leeym@utopia.leeym.com ((Yen-Ming Lee))|Number-Phone|1.3003|i386-freebsd-64int|5.8.6|freebsd|5.4-prerelease|200504081244
197600|pass|200504|rrwo@cpan.org|Number-Phone|1.3003|cygwin-thread-multi-64int|5.8.6|cygwin|1.5.12(0.11642)|200504081254
197676|pass|200504|saper@cpan.org|Number-Phone|1.3003|i386-linux|5.8.5|linux|2.4.19-44mdkenterprise|200504082105
197683|pass|200504|leeym@utopia.leeym.com ((Yen-Ming Lee))|Number-Phone|1.3003|i386-freebsd-64int|5.8.6|freebsd|5.4-prerelease|200504090406
198129|pass|200504|saper@cpan.org|Number-Phone|1.3003|sun4-solaris|5.8.5|solaris|2.9|200504111111
198209|pass|200504|corion-cpanplus@corion.net|Number-Phone|1.3003|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200504111758
198491|pass|200504|leeym@utopia.leeym.com ((Yen-Ming Lee))|Number-Phone|1.3004|i386-freebsd-64int|5.8.6|freebsd|5.4-prerelease|200504131128
198591|pass|200504|corion-cpanplus@corion.net|Number-Phone|1.3004|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200504132021
208372|pass|200505|saper@cpan.org|Number-Phone|1.4|sun4-solaris|5.8.5|solaris|2.9|200505191028
208402|pass|200505|imacat@mail.imacat.idv.tw|Number-Phone|1.4|cygwin-thread-multi-64int|5.8.6|cygwin|1.5.12(0.11642)|200505192028
208414|pass|200505|imacat@mail.imacat.idv.tw|Number-Phone|1.4|MSWin32-x86-multi-thread|5.8.6|MSWin32|4.0|200505192055
208419|pass|200505|imacat@mail.imacat.idv.tw|Number-Phone|1.4|i686-linux-thread-multi-64int-ld|5.8.6|linux|2.4.30|200505192057
208639|pass|200505|Jost.Krieger@ruhr-uni-bochum.de|Number-Phone|1.4|sun4-solaris-thread-multi|5.8.5|solaris|2.9|200505200649
208931|pass|200505|corion-cpanplus@corion.net|Number-Phone|1.4|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200505202142
209198|pass|200505|saper@cpan.org|Number-Phone|1.4|i386-linux|5.8.5|linux|2.4.19-44mdkenterprise|200505220037
211199|pass|200505|root@houseabsolute.urth.org ((root))|Data-Compare|0.13|i386-linux|5.8.4|linux|2.4.27-ti1211|200505310008
211443|fail|200506|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Palm-SMS|0.01|sun4-solaris-thread-multi|5.8.5|solaris|2.9|200506020717
214964|pass|200506|saper@cpan.org|Number-Phone|1.4|darwin|5.6.0|darwin|6.0|200506120128
219764|pass|200506|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Number-Phone|1.4001|sun4-solaris-thread-multi|5.8.5|solaris|2.9|200506281457
220045|pass|200506|imacat@mail.imacat.idv.tw|Number-Phone|1.4001|i686-linux-thread-multi-64int-ld|5.8.7|linux|2.4.30|200506291710
220096|pass|200506|saper@cpan.org|Number-Phone|1.4001|sun4-solaris|5.8.5|solaris|2.9|200506291213
220127|pass|200506|imacat@mail.imacat.idv.tw|Number-Phone|1.4001|cygwin-thread-multi-64int|5.8.6|cygwin|1.5.12(0.11642)|200506292034
220138|pass|200506|imacat@mail.imacat.idv.tw|Number-Phone|1.4001|MSWin32-x86-multi-thread|5.8.7|MSWin32|5.0|200506292041
220280|pass|200506|corion-cpanplus@corion.net|Number-Phone|1.4001|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200506300845
220308|pass|200506|corion-cpanplus@corion.net|Data-Compare|0.13|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200506300919
220438|pass|200506|imacat@mail.imacat.idv.tw|Data-Compare|0.13|i686-linux-thread-multi-64int-ld|5.8.7|linux|2.4.30|200506302053
220485|pass|200506|imacat@mail.imacat.idv.tw|Data-Compare|0.13|cygwin-thread-multi-64int|5.8.6|cygwin|1.5.12(0.11642)|200506302147
220553|pass|200507|saper@cpan.org|Number-Phone|1.4001|i386-linux|5.8.5|linux|2.4.19-44mdkenterprise|200507010001
220664|pass|200507|root@berthoudwireless.net ((root))|Data-Compare|0.13|i686-linux|5.8.7|linux|2.6.12-gentoo-r2|200507010103
223927|pass|200507|saper@cpan.org|Data-Compare|0.13|sun4-solaris|5.8.5|solaris|2.9|200507130954
226474|pass|200507|saper@cpan.org|Number-Phone|1.4001|darwin|5.6.0|darwin|6.0|200507210008
228104|pass|200507|imacat@mail.imacat.idv.tw|Data-Compare|0.13|cygwin-thread-multi-64int|5.8.7|cygwin|1.5.18(0.13242)|200507272149
229324|pass|200508|leeym@utopia.leeym.com ((Yen-Ming Lee))|Data-Compare|0.13|i386-freebsd-64int|5.8.7|freebsd|5.4-stable|200508011634
230507|pass|200508|imacat@mail.imacat.idv.tw|Number-Phone|1.4002|cygwin-thread-multi-64int|5.8.7|cygwin|1.5.18(0.13242)|200508032142
230641|pass|200508|imacat@mail.imacat.idv.tw|Number-Phone|1.4002|i686-linux-thread-multi-64int-ld|5.8.7|linux|2.4.30|200508042018
230643|pass|200508|imacat@mail.imacat.idv.tw|Number-Phone|1.4002|MSWin32-x86-multi-thread|5.8.7|MSWin32|5.0|200508042019
231459|pass|200508|corion-cpanplus@corion.net|Number-Phone|1.4002|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200508072159
232232|pass|200508|saper@cpan.org|Number-Phone|1.4002|sun4-solaris|5.8.5|solaris|2.9|200508091108
233794|pass|200508|leeym@utopia.leeym.com ((Yen-Ming Lee))|Data-Hexdumper|1.2|i386-freebsd-64int|5.8.7|freebsd|5.4-stable|200508141701
234470|pass|200508|imacat@mail.imacat.idv.tw|Data-Hexdumper|1.2|i686-linux-thread-multi-64int-ld|5.8.7|linux|2.4.30|200508170017
236166|pass|200508|gwood@email.mot.com ((George W Wood))|Data-Compare|0.13|i686-linux|5.8.6|linux|2.4.21-20.elsmp|200508220342
236819|pass|200508|leeym@utopia.leeym.com ((Yen-Ming Lee))|Number-Phone|1.4003|i386-freebsd-64int|5.8.7|freebsd|5.4-stable|200508241729
236907|pass|200508|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Number-Phone|1.4003|sun4-solaris-thread-multi|5.8.5|solaris|2.9|200508241616
236968|pass|200508|saper@cpan.org|Number-Phone|1.4003|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200508242102
237109|pass|200508|imacat@mail.imacat.idv.tw|Number-Phone|1.4003|cygwin-thread-multi-64int|5.8.7|cygwin|1.5.18(0.13242)|200508250747
237124|pass|200508|imacat@mail.imacat.idv.tw|Number-Phone|1.4003|i686-linux-thread-multi-64int-ld|5.8.7|linux|2.4.30|200508250805
237270|pass|200508|rrwo@cpan.org|Number-Phone|1.4003|MSWin32-x86-multi-thread|5.8.7|MSWin32|5.1|200508251439
244310|pass|200509|meneldor@metallibrary.ru|Data-Hexdumper|1.2|i386-linux|5.8.3|linux|2.4.21-9.elsmp|200509161951
244808|pass|200509|ultradm@cpan.org|Data-Compare|0.13|i686-linux-thread|5.8.6|linux|2.6.11-gentoo-r6|200509172039
245054|pass|200509|corion-cpanplus@corion.net|Data-Hexdumper|1.2|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200509172204
245305|pass|200509|njh@despammed.com|Data-Hexdumper|1.2|ppc-linux|5.8.3|linux|2.6.6-1.1326smp|200509171902
246450|pass|200509|david@landgren.net|Data-Hexdumper|1.2|i386-freebsd|5.8.1|freebsd|5.4-stable|200509191213
247422|pass|200509|njh@despammed.com|Data-Compare|0.13|i386-linux|5.8.7|linux|2.4.27-ti1211|200509201402
258137|pass|200510|leeym@utopia.leeym.com ((Yen-Ming Lee))|Number-Phone|1.4004|i386-freebsd-64int|5.8.7|freebsd|5.4-stable|200510121351
258161|pass|200510|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Number-Phone|1.4004|sun4-solaris-thread-multi|5.8.5|solaris|2.9|200510120819
258306|pass|200510|corion-cpanplus@corion.net|Number-Phone|1.4004|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200510122021
258774|pass|200510|njh@despammed.com|Number-Phone|1.4004|i386-linux|5.8.5|linux|2.4.21-25.elsmp|200510131058
258922|pass|200510|saper@cpan.org|Number-Phone|1.4004|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200510132040
259968|pass|200510|saper@cpan.org|Number-Phone|1.5|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200510161745
260021|pass|200510|imacat@mail.imacat.idv.tw|Number-Phone|1.5|i686-linux-thread-multi-64int-ld|5.8.7|linux|2.4.30|200510170414
260029|pass|200510|imacat@mail.imacat.idv.tw|Number-Phone|1.5|cygwin-thread-multi-64int|5.8.7|cygwin|1.5.18(0.13242)|200510170421
260051|pass|200510|leeym@utopia.leeym.com ((Yen-Ming Lee))|Number-Phone|1.5|i386-freebsd-64int|5.8.7|freebsd|5.4-stable|200510170844
260191|pass|200510|imacat@mail.imacat.idv.tw|Number-Phone|1.4004|cygwin-thread-multi-64int|5.8.7|cygwin|1.5.18(0.13242)|200510180825
260587|pass|200510|corion-cpanplus@corion.net|Number-Phone|1.5|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200510192027
260898|pass|200510|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Number-Phone|1.5|sun4-solaris-thread-multi|5.8.5|solaris|2.9|200510202001
261368|pass|200510|njh@despammed.com|Number-Phone|1.5|i386-linux|5.8.6|linux|2.4.21-27.0.2.elsmp|200510221455
261541|pass|200510|njh@despammed.com|Number-Phone|1.5|ppc-linux|5.8.3|linux|2.6.6-1.1326smp|200510221844
262207|pass|200510|njh@despammed.com|Number-Phone|1.5|i486-linux-gnu|5.8.7|linux|2.6.14-rc4|200510251548
264405|pass|200511|leeym@utopia.leeym.com ((Yen-Ming Lee))|Data-Compare|0.13|i386-freebsd-64int|5.8.7|freebsd|5.4-stable|200511010235
265861|pass|200511|njh@despammed.com|Data-Compare|0.13|ppc-linux|5.8.3|linux|2.6.6-1.1326smp|200511051614
267538|pass|200511|leeym@utopia.leeym.com ((Yen-Ming Lee))|Data-Compare|0.13|i386-freebsd-64int|5.8.7|freebsd|5.4-stable|200511120225
269770|pass|200511|postmaster@amsfep18-int.chello.nl|Data-Compare|0.13|MSWin32-x86-multi-thread|5.8.4|MSWin32|4.0|200511201354
275357|pass|200512|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Net-Random|1.2|sun4-solaris-thread-multi|5.8.5|solaris|2.9|200512140828
275393|pass|200512|leeym@utopia.leeym.com ((Yen-Ming Lee))|Net-Random|1.2|i386-freebsd-64int|5.8.7|freebsd|6.0-stable|200512141740
275676|pass|200512|njh@despammed.com|Net-Random|1.2|i386-linux|5.8.6|linux|2.6.9-22.18.bz155725.elsmp|200512150658
275830|pass|200512|corion-cpanplus@corion.net|Net-Random|1.2|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200512151939
279001|pass|200512|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Class-CanBeA|1.2|sun4-solaris-thread-multi|5.8.5|solaris|2.9|200512291152
279781|pass|200601|corion-cpanplus@corion.net|Class-CanBeA|1.2|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200601011841
281710|pass|200601|saper@cpan.org|Class-CanBeA|1.2|i686-linux|5.6.2|linux|2.6.8.1-10mdk|200601110339
281711|pass|200601|saper@cpan.org|Class-CanBeA|1.2|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200601110339
286555|pass|200601|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|File-Overwrite|1.0|sun4-solaris-thread-multi|5.8.5|solaris|2.9|200601301202
287234|pass|200602|leeym@utopia.leeym.com ((Yen-Ming Lee))|File-Overwrite|1.0|i386-freebsd-64int|5.8.7|freebsd|6.0-stable|200602012357
289121|pass|200602|gam3-cpanplus@gam3.net|Data-Compare|0.13|i486-linux-gnu|5.8.7|linux|2.6.14-rc4|200602101344
300432|pass|200603|leeym@utopia.leeym.com ((Yen-Ming Lee))|Data-Compare|0.13|i386-freebsd-64int|5.8.8|freebsd|6.1-prerelease|200603200436
305984|pass|200604|edepagne@184dhcp145.pl.eso.org ((root))|Statistics-ChiSquare|0.5|i386-linux|5.8.7|linux|2.6.3-28mdk-i686-up-4gb|200604080602
308015|pass|200604|root@chorny.net|Statistics-ChiSquare|0.5|i686-linux|5.8.8|linux|2.4.20-8smp|200604180521
312967|pass|200605|leeym@utopia.leeym.com ((Yen-Ming Lee))|Data-Compare|0.13|i386-freebsd-64int|5.8.8|freebsd|6.1-prerelease|200605092239
314112|pass|200605|Jost.Krieger+perl@rub.de|Acme-Pony|1.1.2|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200605151414
326166|pass|200606|root@webigail.pooq.com ((root))|Data-Compare|0.13|i386-linux|5.8.8|linux|2.6.12-12mdksmp|200606301715
326827|pass|200607|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Tie-STDOUT|1.01|sun4-solaris-thread-multi|5.8.5|solaris|2.9|200607041255
326894|pass|200607|imacat@mail.imacat.idv.tw|Tie-STDOUT|1.0|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200607050456
326896|pass|200607|imacat@mail.imacat.idv.tw|Tie-STDOUT|1.01|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200607050458
327028|pass|200607|imacat@mail.imacat.idv.tw|Tie-STDOUT|1.02|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200607060241
327156|pass|200607|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Tie-STDOUT|1.02|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200607061747
330218|pass|200607|imacat@mail.imacat.idv.tw|Data-Compare|0.13|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200607190322
331278|pass|200607|imacat@mail.imacat.idv.tw|Data-Hexdumper|1.2|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200607200146
331975|pass|200607|imacat@mail.imacat.idv.tw|Number-Phone|1.5|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200607202026
332024|pass|200607|imacat@mail.imacat.idv.tw|Number-Phone|1.4004|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200607201820
333143|pass|200607|imacat@mail.imacat.idv.tw|Net-Random|1.2|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200607220710
333371|pass|200607|imacat@mail.imacat.idv.tw|Class-CanBeA|1.2|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200607221801
334134|pass|200607|imacat@mail.imacat.idv.tw|File-Overwrite|1.0|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200607231648
335574|pass|200607|imacat@mail.imacat.idv.tw|Data-Compare|0.13|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200607251327
336203|pass|200607|imacat@mail.imacat.idv.tw|Number-Phone|1.5|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200607260008
336692|pass|200607|imacat@mail.imacat.idv.tw|Net-Random|1.2|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200607261742
336789|pass|200607|imacat@mail.imacat.idv.tw|Class-CanBeA|1.2|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200607262132
337018|pass|200607|imacat@mail.imacat.idv.tw|File-Overwrite|1.0|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200607270205
337872|pass|200607|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Sub-WrapPackages|1.1|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200607271444
338614|pass|200607|imacat@mail.imacat.idv.tw|Tie-STDOUT|1.02|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200607281708
338982|pass|200607|imacat@mail.imacat.idv.tw|Sub-WrapPackages|1.1|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200607291323
340236|pass|200608|imacat@mail.imacat.idv.tw|Sub-WrapPackages|1.2|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200608030100
340401|pass|200608|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Sub-WrapPackages|1.2|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200608031203
340715|pass|200608|jworkman@pimpworks.org (Jeff Workman)|Acme-Licence|1.0|i386-netbsd|5.8.3|netbsd|1.6.2|200608031600
341364|pass|200608|corion-cpanplus@corion.net|Sub-WrapPackages|1.2|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200608050855
342214|pass|200608|cogent@cpan.org|Data-Compare|0.13|darwin-thread-multi-64int-2level|5.8.8|darwin|8.7.0|200608100437
344460|pass|200608|imacat@mail.imacat.idv.tw|Number-Phone|1.51|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200608240204
344713|pass|200608|imacat@mail.imacat.idv.tw|Number-Phone|1.52|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200608250509
344714|pass|200608|imacat@mail.imacat.idv.tw|Number-Phone|1.54|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200608250524
344715|fail|200608|imacat@mail.imacat.idv.tw|Number-Phone-UK-DetailedLocations|1.0|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200608250524
344716|fail|200608|imacat@mail.imacat.idv.tw|Number-Phone|1.53|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200608250527
344828|pass|200608|imacat@mail.imacat.idv.tw|Number-Phone|1.55|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200608260505
344829|pass|200608|imacat@mail.imacat.idv.tw|Number-Phone-UK-DetailedLocations|1.2|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200608260515
348551|pass|200609|imacat@mail.imacat.idv.tw|Games-Dice-Advanced|1.1|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200609140041
348707|pass|200609|chris@bingosnet.co.uk|Games-Dice-Advanced|1.1|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.0|200609140529
349313|pass|200609|alexchorny@gmail.com (Alexandr Ciornii)|Games-Dice-Advanced|1.1|i686-linux|5.8.8|linux|2.4.20-8smp|200609141102
351783|pass|200609|gabor@pti.co.il ("Gabor Szabo")|Data-Compare|0.13|i486-linux-gnu|5.8.7|linux|2.6.10|200609251531
353009|pass|200609|mark.gardner@bms.com|Sub-WrapPackages|1.2|i686-linux-thread-multi|5.8.8|linux|2.4.21-37.0.1.el|200609281424
356781|pass|200610|chris@bingosnet.co.uk|Data-Compare|0.13|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.0|200610082338
357072|pass|200610|chris@bingosnet.co.uk|Data-Compare|0.13|i386-freebsd-64int|5.8.5|freebsd|5.3-release|200610091142
358244|pass|200610|chris@bingosnet.co.uk|Data-Compare|0.13|i686-linux-thread-multi-64int-ld|5.8.7|linux|2.4.20-8|200610111335
358966|pass|200610|chris@bingosnet.co.uk|Data-Compare|0.13|alpha-freebsd|5.8.7|freebsd|6.0-release|200610121131
359236|pass|200610|chris@bingosnet.co.uk|Data-Compare|0.13|cygwin-thread-multi-64int|5.8.7|cygwin|1.5.18(0.13242)|200610121735
361330|pass|200610|chris@bingosnet.co.uk|Data-Compare|0.13|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200610181038
365977|pass|200610|chris@bingosnet.co.uk|Data-Compare|0.13|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.18(0.13242)|200610301814
366628|pass|200610|chris@bingosnet.co.uk|Data-Compare|0.13|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200610311115
367382|pass|200611|andreas.koenig.gmwojprw@franz.ak.mind.de ((Andreas J. Koenig))|Data-Compare|0.14|i686-linux-64int|5.9.5|linux|2.6.17-2-k7|200611012205
367439|pass|200611|chris@bingosnet.co.uk|Data-Compare|0.14|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.0|200611020324
367783|pass|200611|imacat@mail.imacat.idv.tw|Data-Compare|0.14|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200611030258
367810|pass|200611|alexchorny@gmail.com|Data-Compare|0.14|MSWin32-x86-multi-thread|5.8.7|MSWin32|5.0|200611020952
367872|pass|200611|chris@bingosnet.co.uk|Data-Compare|0.14|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200611022046
368276|pass|200611|fayland@gmail.com|Data-Compare|0.14|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200611030209
368354|pass|200611|chris@bingosnet.co.uk|Data-Compare|0.14|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.18(0.13242)|200611031157
368655|pass|200611|chris@bingosnet.co.uk|Data-Compare|0.14|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200611031750
370342|pass|200611|davidp@preshweb.co.uk ("David Precious")|Data-Compare|0.14|i486-linux|5.8.7|linux|2.4.31|200611081605
370768|pass|200611|bladebrown@mail.ru|Data-Hexdumper|1.2|cygwin-thread-multi-64int|5.8.7|cygwin|1.5.18(0.13242)|200611110950
370898|pass|200611|chris@bingosnet.co.uk|Bryar|2.6|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.0|200611112020
371195|pass|200611|chris@bingosnet.co.uk|Bryar|2.6|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200611121102
371918|fail|200611|alexchorny@gmail.com|Bryar|2.6|MSWin32-x86-multi-thread|5.8.7|MSWin32|5.0|200611121255
372376|pass|200611|chris@bingosnet.co.uk|Bryar|2.6|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.18(0.13242)|200611132232
373000|pass|200611|chris@bingosnet.co.uk|Bryar|2.6|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.21(0.15642)|200611141002
373148|pass|200611|chris@bingosnet.co.uk|Bryar|2.6|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200611140821
381086|pass|200612|gooba42@gmail.com|Games-Dice-Advanced|1.1|cygwin-thread-multi-64int|5.8.7|cygwin|1.5.18(0.13242)|200612010507
382411|pass|200612|cogent@cpan.org|Data-Compare|0.14|darwin-thread-multi-64int-2level|5.8.8|darwin|8.7.0|200612030522
383534|unknown|200612|srezic@cpan.org|Acme-Pony|1.1.2|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200612052257
385883|pass|200612|chris@bingosnet.co.uk|Number-Phone|1.56|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.0|200612130007
386086|pass|200612|chris@bingosnet.co.uk|Number-Phone|1.56|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200612132153
386324|pass|200612|chris@bingosnet.co.uk|Number-Phone|1.56|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.21(0.15642)|200612141113
386332|pass|200612|chris@bingosnet.co.uk|Number-Phone|1.56|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200612141052
386341|pass|200612|chris@bingosnet.co.uk|Number-Phone|1.56|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200612141125
386581|pass|200612|corion-cpanplus@corion.net|Number-Phone|1.56|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200612142241
387488|pass|200612|chris@bingosnet.co.uk|Number-Phone|1.56|sun4-solaris-64int|5.8.8|solaris|2.7|200612161308
388186|pass|200612|chris@bingosnet.co.uk|Number-Phone|1.56|alpha-freebsd|5.8.7|freebsd|6.0-release|200612181024
393599|pass|200701|saper@cpan.org|Sub-WrapPackages|1.2|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200701021141
393934|pass|200701|autarch@urth.org|Data-Compare|0.14|i486-linux-gnu|5.8.8|linux|2.6.15.7|200701030556
394575|pass|200701|chris@bingosnet.co.uk|Data-Compare|0.14|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200701041619
395005|pass|200701|chris@bingosnet.co.uk|Data-Compare|0.14|sun4-solaris-64int|5.8.8|solaris|2.7|200701060943
395212|pass|200701|chris@bingosnet.co.uk|Data-Compare|0.14|alpha-freebsd|5.8.7|freebsd|6.0-release|200701061407
395489|pass|200701|njh@despammed.com|Data-Compare|0.14|i486-linux-gnu|5.8.7|linux|2.6.10|200701070017
395731|pass|200701|chris@bingosnet.co.uk|Data-Compare|0.14|i386-pc-solaris2.11|5.8.7|solaris|2.11|200701071231
396008|pass|200701|claco@chrislaco.com|Data-Compare|0.14|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200701080049
396170|pass|200701|chris@bingosnet.co.uk|Data-Compare|0.14|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-rc1|200701081709
396447|pass|200701|chris@bingosnet.co.uk|Data-Compare|0.14|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200701090751
398116|pass|200701|saper@cpan.org|Data-Compare|0.14|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200701111152
400138|pass|200701|chris@bingosnet.co.uk|Data-Compare|0.14|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.21(0.15642)|200701161135
401271|pass|200701|chris@bingosnet.co.uk|Data-Compare|0.14|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.23(0.15642)|200701171904
401652|pass|200701|james@mastros.biz|Number-Phone|1.56|i486-linux-gnu|5.8.8|linux|2.6.18.3|200701180143
402258|pass|200701|chris@bingosnet.co.uk|Data-Compare|0.14|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200701191454
402576|pass|200701|chris@bingosnet.co.uk|Bryar|3.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200701210243
402577|pass|200701|chris@bingosnet.co.uk|Bryar|3.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.0|200701210247
402614|fail|200701|chris@bingosnet.co.uk|Net-Random|1.3|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200701211110
402616|fail|200701|chris@bingosnet.co.uk|Net-Random|1.3|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200701211114
402617|fail|200701|chris@bingosnet.co.uk|Net-Random|1.3|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.0|200701211115
402745|fail|200701|chris@bingosnet.co.uk|Bryar|3.0|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.23(0.15642)|200701212346
402746|fail|200701|chris@bingosnet.co.uk|Net-Random|1.3|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.23(0.15642)|200701212348
402878|fail|200701|saper@cpan.org|Net-Random|1.3|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200701220934
403051|pass|200701|chris@bingosnet.co.uk|Net-Random|1.3|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-rc1|200701221027
403054|pass|200701|chris@bingosnet.co.uk|Data-Compare|0.14|alpha-freebsd|5.8.7|freebsd|6.0-release|200701221135
403141|pass|200701|imacat@mail.imacat.idv.tw|Net-Random|1.3|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200701222020
403217|pass|200701|chris@bingosnet.co.uk|Net-Random|1.3|alpha-freebsd|5.8.7|freebsd|6.0-release|200701221247
403247|pass|200701|chris@bingosnet.co.uk|Data-Compare|0.14|i686-linux-thread-multi-64int-ld|5.9.4|linux|2.4.26|200701221245
403469|fail|200701|chris@bingosnet.co.uk|Net-Random|1.3|i686-linux-thread-multi-64int-ld|5.9.4|linux|2.4.26|200701221428
403807|fail|200701|corion-cpanplus@corion.net|Bryar|3.0|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200701221903
403808|fail|200701|corion-cpanplus@corion.net|Net-Random|1.3|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200701221903
404101|fail|200701|chris@bingosnet.co.uk|Net-Random|1.3|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200701231253
404652|pass|200701|chris@bingosnet.co.uk|Bryar|3.0|i386-openbsd|5.8.8|openbsd|4.0|200701241445
404654|fail|200701|chris@bingosnet.co.uk|Net-Random|1.3|i386-openbsd|5.8.8|openbsd|4.0|200701241445
404727|pass|200701|chris@bingosnet.co.uk|Bryar|3.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200701241624
404728|fail|200701|chris@bingosnet.co.uk|Net-Random|1.3|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200701241624
404938|pass|200701|chris@bingosnet.co.uk|Bryar|3.0|i386-pc-solaris2.11|5.8.7|solaris|2.11|200701242140
404939|fail|200701|chris@bingosnet.co.uk|Net-Random|1.3|i386-pc-solaris2.11|5.8.7|solaris|2.11|200701242143
405024|fail|200701|bricas@cpan.org|Net-Random|1.3|i486-linux-gnu|5.8.8|linux|2.6.15.7|200701241254
405090|fail|200701|bricas@cpan.org|Bryar|3.0|i486-linux-gnu|5.8.8|linux|2.6.15.7|200701241253
406240|pass|200701|corion-cpanplus@corion.net|XML-Tiny|1.0|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200701271603
406297|pass|200701|imacat@mail.imacat.idv.tw|XML-Tiny|1.0|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200701280232
406369|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200701281727
406372|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.0|200701281730
406388|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200701281814
406449|pass|200701|saper@cpan.org|Number-Phone|1.56|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200701282030
406568|pass|200701|saper@cpan.org|XML-Tiny|1.0|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200701291002
406701|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200701291118
406775|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.0|i686-linux-thread-multi-64int-ld|5.9.4|linux|2.4.26|200701291146
407142|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.0|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.23(0.15642)|200701291804
407210|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200701291938
407352|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.0|i386-pc-solaris2.11|5.8.7|solaris|2.11|200701292244
407472|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.01|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200701300837
407503|pass|200701|saper@cpan.org|XML-Tiny|1.01|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200701301003
407507|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.01|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.0|200701300910
407521|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.01|i386-pc-solaris2.11|5.8.7|solaris|2.11|200701300936
407580|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.0|alpha-freebsd|5.8.7|freebsd|6.0-release|200701301052
407581|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.01|alpha-freebsd|5.8.7|freebsd|6.0-release|200701301053
407664|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.01|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200701301210
407864|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.01|i686-linux-thread-multi-64int-ld|5.9.4|linux|2.4.26|200701301354
408226|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.02|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200701310829
408246|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.02|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.0|200701310844
408304|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.02|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200701310925
408321|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.0|i386-openbsd|5.8.8|openbsd|4.0|200701310943
408322|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.01|i386-openbsd|5.8.8|openbsd|4.0|200701310944
408323|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.02|i386-openbsd|5.8.8|openbsd|4.0|200701310944
408325|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.02|i686-linux-thread-multi-64int-ld|5.9.4|linux|2.4.26|200701310944
408457|pass|200701|chris@bingosnet.co.uk|Data-Compare|0.14|i386-openbsd|5.8.8|openbsd|4.0|200701311200
408469|pass|200701|imacat@mail.imacat.idv.tw|XML-Tiny|1.01|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200701312008
408518|pass|200701|saper@cpan.org|XML-Tiny|1.02|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200701311345
408519|pass|200701|saper@cpan.org|XML-Tiny|1.02|i686-linux|5.4.4|linux|2.6.8.1-10mdk|200701311346
408662|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-rc1|200701311611
408663|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.01|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-rc1|200701311611
408664|pass|200701|chris@bingosnet.co.uk|XML-Tiny|1.02|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-rc1|200701311611
408851|pass|200701|castaway@desert-island.dynodns.net|Data-Compare|0.14|i686-linux-thread|5.8.7|linux|2.6.15.1|200701311544
409022|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.01|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200702010958
409023|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.02|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200702010958
409131|pass|200702|imacat@mail.imacat.idv.tw|XML-Tiny|1.02|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.14|200702012013
409233|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.02|i386-pc-solaris2.11|5.8.7|solaris|2.11|200702011557
409389|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.02|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200702011906
409390|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.01|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200702011906
409399|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.01|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.23(0.15642)|200702012025
409400|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.02|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.23(0.15642)|200702012025
409816|pass|200702|jrockway@cpan.org|Data-Compare|0.14|i486-linux-gnu|5.8.8|linux|2.6.18.3|200702021305
409919|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.03|i686-linux-thread-multi-64int-ld|5.9.4|linux|2.4.26|200702030740
409976|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.03|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.0|200702031408
409980|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.03|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200702031502
410113|pass|200702|smueller@cpan.org ("Steffen Mueller")|XML-Tiny|1.02|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200702032114
410364|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.03|i386-openbsd|5.8.8|openbsd|4.0|200702041215
410380|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.03|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200702041245
410746|pass|200702|saper@cpan.org|XML-Tiny|1.03|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200702051015
410765|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.03|i386-pc-solaris2.11|5.8.7|solaris|2.11|200702050953
410944|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.02|alpha-freebsd|5.8.7|freebsd|6.0-release|200702051437
410945|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.03|alpha-freebsd|5.8.7|freebsd|6.0-release|200702051437
411263|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.03|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-rc1|200702060952
411314|pass|200702|imacat@mail.imacat.idv.tw|XML-Tiny|1.03|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.29|200702062123
411349|pass|200702|imacat@mail.imacat.idv.tw|XML-Tiny|1.03|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.29|200702062145
411552|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.03|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.23(0.15642)|200702062002
411568|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.03|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200702062036
411918|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.03|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200702071720
413290|pass|200702|david@cantrell.org.uk|Tie-Scalar-Decay|1.1.1|i686-linux|5.8.8|linux|2.4.27-3-686|200702112331
413292|pass|200702|david@cantrell.org.uk|Tie-Hash-Rank|1.0.1|i686-linux|5.8.8|linux|2.4.27-3-686|200702112335
413328|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.04|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200702120203
413330|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.04|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200702120204
413341|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.04|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.0|200702120314
413385|pass|200702|saper@cpan.org|XML-Tiny|1.04|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200702121120
413428|pass|200702|saper@cpan.org|XML-Tiny|1.04|i686-linux|5.4.4|linux|2.6.8.1-10mdk|200702121203
413448|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.04|alpha-freebsd|5.8.7|freebsd|6.0-release|200702121122
413729|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.04|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200702121534
413903|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.04|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.23(0.15642)|200702121813
414124|pass|200702|chris@bingosnet.co.uk|Data-Compare|0.14|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200702122206
414424|pass|200702|imacat@mail.imacat.idv.tw|XML-Tiny|1.04|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.29|200702132233
414887|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.04|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200702140850
415005|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.04|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-rc1|200702140853
415088|pass|200702|saper@cpan.org|XML-Tiny|1.04|i686-linux|5.6.2|linux|2.6.8.1-10mdk|200702141401
415282|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.04|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200702142048
415613|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.04|i386-openbsd|5.8.8|openbsd|4.0|200702151026
416151|pass|200702|david@cantrell.org.uk|Number-Phone|1.56|i686-linux|5.6.2|linux|2.4.27-3-686|200702160107
416160|pass|200702|david@cantrell.org.uk|Number-Phone-UK-DetailedLocations|1.2|i686-linux|5.6.2|linux|2.4.27-3-686|200702160110
416168|pass|200702|david@cantrell.org.uk|Number-Phone|1.56|i686-linux|5.6.2|linux|2.4.27-3-686|200702160110
416390|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.04|i386-pc-solaris2.11|5.8.7|solaris|2.11|200702161340
420805|pass|200702|chris@bingosnet.co.uk|Data-Compare|0.14|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200702241548
421144|pass|200702|chris@bingosnet.co.uk|Number-Phone|1.5601|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.0|200702242015
421152|pass|200702|chris@bingosnet.co.uk|Number-Phone|1.5601|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200702242018
421157|pass|200702|chris@bingosnet.co.uk|Number-Phone|1.5601|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200702242023
421523|pass|200702|david@cantrell.org.uk|Data-Compare|0.14|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200702251914
421525|fail|200702|david@cantrell.org.uk|Data-Compare|0.14|i686-linux|5.6.2|linux|2.4.27-3-686|200702251914
421552|pass|200702|david@cantrell.org.uk|Number-Phone|1.5601|i686-linux|5.6.2|linux|2.4.27-3-686|200702251920
421581|pass|200702|david@cantrell.org.uk|Number-Phone|1.5601|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200702251927
421634|pass|200702|david@cantrell.org.uk|Data-Compare|0.14|i686-linux|5.8.8|linux|2.4.27-3-686|200702251941
421661|pass|200702|david@cantrell.org.uk|Number-Phone|1.5601|i686-linux|5.8.8|linux|2.4.27-3-686|200702251947
422037|pass|200702|chris@bingosnet.co.uk|Number-Phone|1.5601|alpha-freebsd|5.8.7|freebsd|6.0-release|200702261104
422104|pass|200702|chris@bingosnet.co.uk|Number-Phone|1.5601|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200702261121
422151|pass|200702|chris@bingosnet.co.uk|Number-Phone|1.5601|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200702261228
422169|pass|200702|imacat@mail.imacat.idv.tw|Number-Phone|1.5601|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.29|200702262037
422879|pass|200702|chris@bingosnet.co.uk|Number-Phone|1.5601|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200702270133
423054|pass|200702|saper@cpan.org|Data-Compare|0.15|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200702271007
423090|pass|200702|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Compare|0.15|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200702270927
423091|pass|200702|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Number-Phone|1.5601|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200702270928
423199|pass|200702|chris@bingosnet.co.uk|Number-Phone|1.5601|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-rc1|200702270958
423480|pass|200702|david@cantrell.org.uk|Data-Compare|0.15|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200702271507
423670|pass|200702|david@cantrell.org.uk|Data-Compare|0.15|i686-linux|5.8.8|linux|2.4.27-3-686|200702271636
424064|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.05|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200702280418
424067|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.05|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200702280420
424100|pass|200702|corion-cpanplus@corion.net|Data-Compare|0.16|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200702280837
424204|pass|200702|Jost.Krieger@ruhr-uni-bochum.de|Data-Compare|0.16|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200702281107
424249|pass|200702|chris@bingosnet.co.uk|XML-Tiny|1.05|i386-pc-solaris2.11|5.8.7|solaris|2.11|200702281202
424258|fail|200702|Jost.Krieger@ruhr-uni-bochum.de|Net-Random|1.3|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200702281235
424590|pass|200702|david@cantrell.org.uk|XML-Tiny|1.05|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200702281549
424641|pass|200702|david@cantrell.org.uk|Data-Compare|0.16|i686-linux|5.6.2|linux|2.4.27-3-686|200702281449
424664|pass|200702|david@cantrell.org.uk|Data-Compare|0.16|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200702281502
424839|pass|200702|david@cantrell.org.uk|XML-Tiny|1.05|i686-linux|5.6.2|linux|2.4.27-3-686|200702281459
425056|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.05|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200703010144
425143|pass|200703|chris@bingosnet.co.uk|Net-Random|1.4|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200703010319
425147|pass|200703|chris@bingosnet.co.uk|Net-Random|1.4|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200703010323
425457|pass|200703|saper@cpan.org|Data-Compare|0.16|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200703011023
425610|pass|200703|david@cantrell.org.uk|Data-Compare|0.16|i686-linux|5.8.8|linux|2.4.27-3-686|200703011040
425615|pass|200703|david@cantrell.org.uk|XML-Tiny|1.05|i686-linux|5.8.8|linux|2.4.27-3-686|200703011042
425718|pass|200703|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Compare|0.16|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200703011108
425719|pass|200703|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Net-Random|1.4|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200703011109
425723|pass|200703|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|XML-Tiny|1.05|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200703011109
425770|pass|200703|david@cantrell.org.uk|Data-Compare|0.16|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200703011115
425784|pass|200703|david@cantrell.org.uk|Data-Compare|0.16|i686-linux|5.6.2|linux|2.4.27-3-686|200703011121
425799|pass|200703|chris@bingosnet.co.uk|Net-Random|1.4|i386-openbsd|5.8.8|openbsd|4.0|200703011122
425819|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.05|i386-openbsd|5.8.8|openbsd|4.0|200703011129
425969|pass|200703|imacat@mail.imacat.idv.tw|XML-Tiny|1.05|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.29|200703012155
425979|pass|200703|chris@bingosnet.co.uk|Net-Random|1.4|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200703011402
426002|pass|200703|david@cantrell.org.uk|Net-Random|1.4|i686-linux|5.6.2|linux|2.4.27-3-686|200703011412
426092|pass|200703|david@cantrell.org.uk|Data-Compare|0.16|i686-linux|5.8.8|linux|2.4.27-3-686|200703011503
426155|pass|200703|david@cantrell.org.uk|Net-Random|1.4|i686-linux|5.8.8|linux|2.4.27-3-686|200703011528
426308|pass|200703|david@cantrell.org.uk|Net-Random|1.4|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200703011415
426482|pass|200703|chris@bingosnet.co.uk|Net-Random|1.4|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200703012329
426483|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.05|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200703012335
426651|pass|200703|saper@cpan.org|Net-Random|1.4|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200703021008
426712|pass|200703|saper@cpan.org|XML-Tiny|1.05|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200703021050
426868|pass|200703|imacat@mail.imacat.idv.tw|Net-Random|1.4|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.29|200703022135
426952|pass|200703|david@cantrell.org.uk|Data-Compare|0.16|i686-linux|5.6.2|linux|2.4.27-3-686|200703021425
426980|pass|200703|david@cantrell.org.uk|Data-Compare|0.16|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200703021435
427087|pass|200703|david@cantrell.org.uk|Data-Compare|0.16|i686-linux|5.8.8|linux|2.4.27-3-686|200703021536
427212|pass|200703|chris@bingosnet.co.uk|Net-Random|1.4|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-rc1|200703021558
427213|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.05|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-rc1|200703021559
427431|pass|200703|chris@bingosnet.co.uk|Net-Random|1.4|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200703031108
427432|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.05|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200703031108
428389|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.06|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200703050518
428390|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.06|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200703050520
428391|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.06|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200703050521
428489|pass|200703|chris@bingosnet.co.uk|Net-Random|1.4|alpha-freebsd|5.8.7|freebsd|6.0-release|200703051004
428490|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.05|alpha-freebsd|5.8.7|freebsd|6.0-release|200703051005
428491|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.06|alpha-freebsd|5.8.7|freebsd|6.0-release|200703051005
428598|pass|200703|srezic@iconmobile.com (Slaven Rezic)|Data-Compare|0.16|i686-linux|5.8.7|linux|2.4.20-18.8|200703051233
428962|pass|200703|david@cantrell.org.uk|XML-Tiny|1.06|i686-linux|5.8.8|linux|2.4.27-3-686|200703051539
429094|pass|200703|david@cantrell.org.uk|XML-Tiny|1.06|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200703051734
429161|pass|200703|chris@bingosnet.co.uk|Net-Random|1.4|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200703051912
429162|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.05|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200703051912
429163|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.06|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200703051913
429176|pass|200703|chris@bingosnet.co.uk|Net-Random|1.4|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.23(0.15642)|200703051926
429179|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.05|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.23(0.15642)|200703051927
429180|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.06|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.23(0.15642)|200703051928
429695|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.06|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.0|200703061105
429805|pass|200703|imacat@mail.imacat.idv.tw|XML-Tiny|1.06|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.29|200703062107
430026|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.06|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200703061525
430283|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.05|i686-linux-thread-multi-64int-ld|5.9.4|linux|2.4.26|200703061713
430284|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.06|i686-linux-thread-multi-64int-ld|5.9.4|linux|2.4.26|200703061714
430700|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.06|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200703070516
431402|pass|200703|njh@bandsman.co.uk|Data-Compare|0.16|i386-linux|5.8.8|linux|2.6.9-34.elsmp|200703080651
432496|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.06|i386-pc-solaris2.11|5.8.7|solaris|2.11|200703090229
432921|pass|200703|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|XML-Tiny|1.06|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200703091413
433143|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.06|i386-openbsd|5.8.8|openbsd|4.0|200703091651
433802|pass|200703|chris@bingosnet.co.uk|XML-Tiny|1.06|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-rc1|200703101135
436498|pass|200703|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.0|200703130946
436583|pass|200703|Jost.Krieger@ruhr-uni-bochum.de|Net-Random|1.4|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200703131155
436826|pass|200703|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|i686-linux|5.6.2|linux|2.4.27-3-686|200703131620
436857|pass|200703|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200703131628
436923|fail|200703|david@cantrell.org.uk|Number-Phone|1.5601|i686-linux|5.8.8|linux|2.4.27-3-686|200703131515
437019|pass|200703|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.23(0.15642)|200703131809
437354|pass|200703|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|XML-DoubleEncodedEntities|1.0|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200703141014
437425|pass|200703|imacat@mail.imacat.idv.tw|XML-DoubleEncodedEntities|1.0|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.29|200703141958
437858|fail|200703|chris@bingosnet.co.uk|Number-Phone|1.5601|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200703150458
437859|pass|200703|chris@bingosnet.co.uk|Number-Phone|1.57|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200703150506
437965|pass|200703|chris@bingosnet.co.uk|Number-Phone|1.57|i386-openbsd|5.8.8|openbsd|4.0|200703150950
437967|pass|200703|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Number-Phone|1.57|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200703151000
437969|pass|200703|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Number-Phone-UK-DetailedLocations|1.3|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200703151001
437970|pass|200703|chris@bingosnet.co.uk|Number-Phone-UK-DetailedLocations|1.3|i386-openbsd|5.8.8|openbsd|4.0|200703150952
437971|pass|200703|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-openbsd|5.8.8|openbsd|4.0|200703150953
438217|pass|200703|chris@bingosnet.co.uk|Number-Phone|1.57|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200703151453
438220|pass|200703|chris@bingosnet.co.uk|Number-Phone-UK-DetailedLocations|1.3|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200703151453
438278|pass|200703|david@cantrell.org.uk|Number-Phone|1.57|i686-linux|5.6.2|linux|2.4.27-3-686|200703151537
438279|pass|200703|david@cantrell.org.uk|Number-Phone-UK-DetailedLocations|1.3|i686-linux|5.6.2|linux|2.4.27-3-686|200703151539
438390|pass|200703|david@cantrell.org.uk|Number-Phone|1.57|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200703151638
438391|pass|200703|david@cantrell.org.uk|Number-Phone-UK-DetailedLocations|1.3|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200703151638
438441|pass|200703|david@cantrell.org.uk|Number-Phone|1.57|i686-linux|5.8.8|linux|2.4.27-3-686|200703151714
438442|pass|200703|david@cantrell.org.uk|Number-Phone-UK-DetailedLocations|1.3|i686-linux|5.8.8|linux|2.4.27-3-686|200703151716
438515|pass|200703|chris@bingosnet.co.uk|Number-Phone|1.57|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.0|200703151958
438516|pass|200703|chris@bingosnet.co.uk|Number-Phone-UK-DetailedLocations|1.3|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.0|200703152000
438527|pass|200703|david@cantrell.org.uk|Number-Phone|1.57|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200703151616
439309|pass|200703|imacat@mail.imacat.idv.tw|Number-Phone|1.57|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.29|200703170044
439389|pass|200703|chris@bingosnet.co.uk|Number-Phone|1.57|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200703170812
439391|pass|200703|chris@bingosnet.co.uk|Number-Phone-UK-DetailedLocations|1.3|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200703170813
439392|pass|200703|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200703170813
439763|pass|200703|imacat@mail.imacat.idv.tw|Number-Phone-UK-DetailedLocations|1.3|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.29|200703171627
439973|pass|200703|david@cantrell.org.uk|Data-Compare|0.16|i686-linux|5.8.8|linux|2.4.27-3-686|200703171908
440130|pass|200703|david@cantrell.org.uk|Data-Compare|0.16|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200703171754
440280|fail|200703|corion-cpanplus@corion.net|Number-Phone|1.57|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200703172321
440293|pass|200703|chris@bingosnet.co.uk|Number-Phone|1.57|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-rc1|200703172124
440294|pass|200703|chris@bingosnet.co.uk|Number-Phone-UK-DetailedLocations|1.3|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-rc1|200703172126
440295|pass|200703|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-rc1|200703172126
440390|pass|200703|chris@bingosnet.co.uk|Data-Hexdumper|1.2|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200703181013
440713|pass|200703|david@cantrell.org.uk|Data-Compare|0.16|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200703181558
440867|pass|200703|david@cantrell.org.uk|Data-Compare|0.16|i686-linux|5.8.8|linux|2.4.27-3-686|200703181921
441193|pass|200703|jrockway@cpan.org|Data-Compare|0.16|i486-linux-gnu|5.8.8|linux|2.6.18.3|200703181659
441296|pass|200703|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200703191312
441475|pass|200703|david@cantrell.org.uk|Data-Compare|0.16|i686-linux|5.8.8|linux|2.4.27-3-686|200703192317
441520|pass|200703|david@cantrell.org.uk|Data-Hexdumper|1.2|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200703192340
441568|pass|200703|david@cantrell.org.uk|Data-Hexdumper|1.2|i686-linux|5.8.8|linux|2.4.27-3-686|200703200007
441785|pass|200703|chris@bingosnet.co.uk|Number-Phone|1.57|alpha-freebsd|5.8.7|freebsd|6.0-release|200703201026
441798|pass|200703|chris@bingosnet.co.uk|Number-Phone-UK-DetailedLocations|1.3|alpha-freebsd|5.8.7|freebsd|6.0-release|200703201027
441880|pass|200703|imacat@mail.imacat.idv.tw|Data-Hexdumper|1.2|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.29|200703201956
441964|pass|200703|davidp@preshweb.co.uk ("David Precious")|Data-Compare|0.16|i486-linux|5.8.7|linux|2.4.31|200703201224
442206|pass|200703|corion-cpanplus@corion.net|Number-Phone|1.57|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200703202030
442207|pass|200703|corion-cpanplus@corion.net|Number-Phone-UK-DetailedLocations|1.3|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200703202030
442408|pass|200703|chris@bingosnet.co.uk|Number-Phone|1.57|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200703210008
442410|pass|200703|chris@bingosnet.co.uk|Number-Phone-UK-DetailedLocations|1.3|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200703210010
442474|pass|200703|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200703210214
442479|pass|200703|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200703210218
442482|pass|200703|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200703210224
442587|pass|200703|chris@bingosnet.co.uk|Data-Hexdumper|1.2|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200703210706
442624|pass|200703|wyant@cpan.org|XML-DoubleEncodedEntities|1.0|darwin-2level|5.8.8|darwin|7.9.0|200703210259
442968|pass|200703|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200703211540
442976|pass|200703|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|i686-linux|5.6.2|linux|2.4.27-3-686|200703211618
443027|pass|200703|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|i686-linux|5.8.8|linux|2.4.27-3-686|200703211731
443143|pass|200703|chris@bingosnet.co.uk|Number-Phone|1.57|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200703212141
443144|pass|200703|chris@bingosnet.co.uk|Number-Phone-UK-DetailedLocations|1.3|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200703212142
443425|pass|200703|chris@bingosnet.co.uk|Data-Hexdumper|1.2|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200703220318
443561|pass|200703|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200703220515
444751|pass|200703|chris@bingosnet.co.uk|Data-Hexdumper|1.2|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.4-release|200703232117
444790|pass|200703|dither@gmail.com|Data-Compare|0.16|i386-freebsd-64int|5.8.8|freebsd|5.4-release|200703231735
444941|pass|200703|chris@bingosnet.co.uk|Data-Hexdumper|1.2|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.0|200703240104
445126|pass|200703|chris@bingosnet.co.uk|Data-Hexdumper|1.2|i386-openbsd|5.8.8|openbsd|4.0|200703240725
445864|pass|200703|chris@bingosnet.co.uk|Data-Hexdumper|1.2|i386-pc-solaris2.11|5.8.7|solaris|2.11|200703242336
447207|pass|200703|david@cantrell.org.uk|Data-Hexdumper|1.2|i686-linux|5.6.2|linux|2.4.27-3-686|200703261737
447526|pass|200703|david@cantrell.org.uk|Data-Hexdumper|1.2|i686-linux|5.8.8|linux|2.4.27-3-686|200703262159
447776|pass|200703|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|i686-linux|5.6.2|linux|2.4.27-3-686|200703271458
447879|pass|200703|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200703271609
449979|pass|200703|david@cantrell.org.uk|Data-Compare|0.16|i686-linux|5.6.2|linux|2.4.27-3-686|200703301451
450622|pass|200703|david@cantrell.org.uk|Data-Compare|0.16|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200703302101
454327|pass|200704|chris@bingosnet.co.uk|Acme-Pony|1.1.2|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200704051030
454539|pass|200704|david@cantrell.org.uk|Data-Compare|0.16|i686-linux|5.6.2|linux|2.4.27-3-686|200704051512
454755|pass|200704|david@cantrell.org.uk|Data-Compare|0.16|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200704051622
458609|pass|200704|tom@eborcom.com ("Tom Hukins")|Sub-WrapPackages|1.2|i686-linux|5.9.5 patch 30836|linux|2.6.17-11-generic|200704121137
458708|pass|200704|tom@eborcom.com ("Tom Hukins")|XML-Tiny|1.06|i686-linux|5.9.5 patch 30836|linux|2.6.17-11-generic|200704121420
459149|pass|200704|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Net-Random|2.0|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200704130952
459445|pass|200704|chris@bingosnet.co.uk|Net-Random|2.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200704131703
459556|pass|200704|chris@bingosnet.co.uk|Net-Random|2.0|i386-pc-solaris2.11|5.8.7|solaris|2.11|200704131846
459577|pass|200704|chris@bingosnet.co.uk|Net-Random|2.0|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.23(0.15642)|200704131927
460125|pass|200704|imacat@mail.imacat.idv.tw|Net-Random|2.0|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.16.29|200704142351
460308|pass|200704|david@cantrell.org.uk|Net-Random|2.0|i686-linux|5.6.2|linux|2.4.27-3-686|200704142209
460619|pass|200704|david@cantrell.org.uk|Net-Random|2.0|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200704151540
461008|pass|200704|chris@bingosnet.co.uk|Net-Random|2.0|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200704161042
461251|pass|200704|chris@bingosnet.co.uk|Net-Random|2.0|alpha-linux-thread-multi|5.8.8|linux|2.6.16.19|200704161338
462048|pass|200704|saper@cpan.org|Net-Random|2.0|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200704171032
462055|pass|200704|chris@bingosnet.co.uk|Net-Random|2.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200704170903
462970|pass|200704|alexchorny@gmail.com (Alexandr Ciornii)|Data-Compare|0.16|i686-linux|5.8.8|linux|2.4.20-8smp|200704180113
463006|pass|200704|chris@bingosnet.co.uk|Net-Random|2.0|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.21(0.15642)|200704181224
463921|pass|200704|chris@bingosnet.co.uk|Net-Random|2.0|i386-openbsd|5.8.8|openbsd|4.0|200704190037
465406|pass|200704|david@cantrell.org.uk|Net-Random|2.0|i686-linux|5.8.3|linux|2.4.22-physmem-patch3-5um|200704201622
466836|pass|200704|david@cantrell.org.uk|File-Overwrite|1.0|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200704231400
468218|pass|200704|david@cantrell.org.uk|Class-CanBeA|1.2|i686-linux|5.6.2|linux|2.4.27-3-686|200704251738
468223|pass|200704|david@cantrell.org.uk|Class-CanBeA|1.2|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200704251844
474057|pass|200705|xdg@hyperbolic.net (DAGOLDEN)|Net-Random|2.0|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200705040829
475133|pass|200705|chaddai.fouche@gmail.com ("Chaddai Fouche")|Data-Compare|0.16|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200705052338
475141|pass|200705|chaddai.fouche@gmail.com ("Chaddai Fouche")|Data-Compare|0.16|MSWin32-x86-multi-thread|5.9.5|MSWin32|4.0|200705052344
475387|pass|200705|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.16|i686-linux-64int|5.9.5|linux|2.6.18-4-k7|200705061119
475972|pass|200705|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.16|i686-linux-64int|5.8.8 patch 30961|linux|2.6.18-4-k7|200705070015
476806|fail|200705|bsmith@cpan.org|Bryar|3.0|i486-linux-gnu|5.9.5 patch 31148|linux|2.6.17.13.vtrl22.01|200705061925
476985|pass|200705|david@cantrell.org.uk|Data-Compare|0.16|i686-linux|5.6.2|linux|2.4.27-3-686|200705071450
477045|pass|200705|david@cantrell.org.uk|Data-Compare|0.16|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200705071614
478961|fail|200705|bsmith@cpan.org|Bryar|3.0|i486-linux-gnu|5.9.5 patch 31148|linux|2.6.17.13.vtrl22.01|200705091231
483520|pass|200705|bsmith@cpan.org|Data-Compare|0.16|i486-linux-gnu|5.9.5 patch 31148|linux|2.6.17.13.vtrl22.01|200705131130
484000|pass|200705|bsmith@cpan.org|Data-Hexdumper|1.2|i486-linux-gnu|5.9.5 patch 31148|linux|2.6.17.13.vtrl22.01|200705131435
484013|fail|200705|bsmith@cpan.org|Data-Transactional|0.1|i486-linux-gnu|5.9.5 patch 31148|linux|2.6.17.13.vtrl22.01|200705131441
498580|pass|200705|tom@eborcom.com ("Tom Hukins")|Data-Compare|0.16|i686-linux|5.9.5 patch 30836|linux|2.6.17-11-generic|200705301013
499954|pass|200705|stro@cpan.org|Number-Phone-UK-DetailedLocations|1.2|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200705311327
499957|pass|200705|stro@cpan.org|Number-Phone-UK-DetailedLocations|1.3|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200705311330
500760|pass|200706|chris@bingosnet.co.uk|Number-Phone|1.58|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.4.26|200706020634
500763|pass|200706|chris@bingosnet.co.uk|Number-Phone|1.58|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200706020637
501025|pass|200706|david@cantrell.org.uk|Number-Phone|1.58|i686-linux|5.6.2|linux|2.4.27-3-686|200706031142
501028|pass|200706|david@cantrell.org.uk|Number-Phone|1.58|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200706031245
501042|pass|200706|imacat@mail.imacat.idv.tw|Number-Phone|1.58|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200706031911
501405|pass|200706|saper@cpan.org|Number-Phone|1.58|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200706041055
501492|pass|200706|stro@cpan.org|Number-Phone|1.58|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200706031924
501643|pass|200706|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Number-Phone|1.58|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200706041402
501841|pass|200706|chris@bingosnet.co.uk|Number-Phone|1.58|i386-pc-solaris2.11|5.8.7|solaris|2.11|200706041812
502234|pass|200706|chris@bingosnet.co.uk|Number-Phone|1.58|alpha-linux-thread-multi|5.8.8|linux|2.6.16.19|200706050932
502780|pass|200706|chris@bingosnet.co.uk|Number-Phone|1.58|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200706060804
503556|pass|200706|chris@bingosnet.co.uk|Number-Phone|1.58|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200706061615
503784|pass|200706|david@cantrell.org.uk|Data-Compare|0.16|i686-linux|5.9.5 patch 31335|linux|2.4.27-3-686|200706062047
504111|pass|200706|chris@bingosnet.co.uk|Number-Phone|1.58|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200706071044
504704|pass|200706|chris@bingosnet.co.uk|Number-Phone|1.58|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200706071858
519559|pass|200706|david@cantrell.org.uk|NestedMap|1.0|i686-linux|5.8.8|linux|2.4.27-3-686|200706261547
527714|pass|200707|stro@cpan.org|Tie-Hash-Rank|1.0.1|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200707041839
527792|pass|200707|stro@cpan.org|Data-Compare|0.16|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200707041757
528095|pass|200707|stro@cpan.org|Acme-Pony|1.1.2|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200707041749
532760|pass|200707|david@cantrell.org.uk|Data-Compare|0.16|i686-linux|5.8.8|linux|2.4.27-3-686|200707101103
532768|pass|200707|david@cantrell.org.uk|Data-Compare|0.16|i686-linux|5.9.5 patch 31416|linux|2.4.27-3-686|200707101104
536645|pass|200707|david@cantrell.org.uk|Class-CanBeA|1.2|i686-linux|5.9.5|linux|2.4.27-3-686|200707142021
537892|pass|200707|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.16|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200707152130
538989|pass|200707|overlordq@gmail.com|Data-Compare|0.16|i486-linux-gnu|5.8.8|linux|2.6.18.3|200707161626
541445|pass|200707|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|i686-linux|5.6.2|linux|2.4.27-3-686|200707182344
541446|pass|200707|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|i686-linux|5.8.8|linux|2.4.27-3-686|200707182344
541447|pass|200707|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|i686-linux|5.9.5|linux|2.4.27-3-686|200707182345
541450|pass|200707|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200707190052
546402|pass|200707|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.16|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200707250634
546525|pass|200707|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.06|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200707250749
551041|pass|200707|david@cantrell.org.uk|Palm-TreoPhoneCallDB|1|i686-linux|5.9.5|linux|2.4.27-3-686|200707301143
551134|pass|200707|david@cantrell.org.uk|Palm-TreoPhoneCallDB|1|i686-linux|5.8.8|linux|2.4.27-3-686|200707301418
551151|pass|200707|david@cantrell.org.uk|Palm-TreoPhoneCallDB|1|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200707301537
551157|fail|200707|david@cantrell.org.uk|Palm-TreoPhoneCallDB|1|i686-linux|5.6.2|linux|2.4.27-3-686|200707301412
551158|pass|200707|david@cantrell.org.uk|Data-Hexdumper|1.2|i686-linux|5.6.2|linux|2.4.27-3-686|200707301408
551428|pass|200707|chris@bingosnet.co.uk|Palm-TreoPhoneCallDB|1.1|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200707310541
551444|pass|200707|andreas.koenig.gmwojprw@franz.ak.mind.de|Palm-TreoPhoneCallDB|1.1|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200707310704
551498|pass|200707|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Palm-TreoPhoneCallDB|1.1|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200707310826
551733|pass|200707|ferreira@cpan.org ("Adriano Ferreira")|Data-Compare|0.16|cygwin|5.9.5|cygwin|1.5.24(0.15642)|200707311148
551763|pass|200707|david@cantrell.org.uk|Palm-TreoPhoneCallDB|1.1|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200707311721
551794|pass|200707|tom@eborcom.com ("Tom Hukins")|Palm-TreoPhoneCallDB|1.1|i686-linux|5.9.5 patch 31576|linux|2.6.17-11-generic|200707311337
551801|pass|200707|david@cantrell.org.uk|Palm-TreoPhoneCallDB|1.1|i686-linux|5.8.8|linux|2.4.27-3-686|200707311725
551813|pass|200707|david@cantrell.org.uk|Palm-TreoPhoneCallDB|1.1|i686-linux|5.6.2|linux|2.4.27-3-686|200707311818
551815|pass|200707|david@cantrell.org.uk|Palm-TreoPhoneCallDB|1.1|i686-linux|5.9.5|linux|2.4.27-3-686|200707311828
552449|pass|200708|imacat@mail.imacat.idv.tw|Palm-TreoPhoneCallDB|1.1|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200708020036
552542|pass|200708|stro@cpan.org|Palm-TreoPhoneCallDB|1.1|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200708011914
553871|pass|200708|chris@bingosnet.co.uk|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200708032142
553873|pass|200708|chris@bingosnet.co.uk|Palm-TreoPhoneCallDB|1|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200708032144
554693|pass|200708|chris@bingosnet.co.uk|Palm-TreoPhoneCallDB|1.1|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200708041837
554699|pass|200708|chris@bingosnet.co.uk|Palm-TreoPhoneCallDB|1|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200708041840
555029|pass|200708|chris@bingosnet.co.uk|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200708050148
555030|pass|200708|chris@bingosnet.co.uk|Palm-TreoPhoneCallDB|1|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200708050150
555430|pass|200708|chris@bingosnet.co.uk|Palm-TreoPhoneCallDB|1.1|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200708051356
555431|pass|200708|chris@bingosnet.co.uk|Palm-TreoPhoneCallDB|1|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200708051400
555684|fail|200708|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.16|i686-linux-64int|5.8.3|linux|2.4.22-rc2-ac2|200708051931
555827|pass|200708|chris@bingosnet.co.uk|Palm-TreoPhoneCallDB|1.1|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200708052212
555828|pass|200708|chris@bingosnet.co.uk|Palm-TreoPhoneCallDB|1|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200708052214
556260|pass|200708|chris@bingosnet.co.uk|Palm-TreoPhoneCallDB|1.1|alpha-linux-thread-multi|5.8.8|linux|2.6.16.19|200708061125
556303|pass|200708|chris@bingosnet.co.uk|Palm-TreoPhoneCallDB|1.1|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200708061143
558672|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200708072108
558814|pass|200708|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200708080159
558828|pass|200708|david@cantrell.org.uk|XML-Tiny|1.06|i686-linux|5.8.3|linux|2.4.22-physmem-patch3-5um|200708071720
558868|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200708080137
558881|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200708080141
559090|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200708080854
559174|pass|200708|saper@cpan.org|Data-Compare|0.17|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200708081011
559259|pass|200708|david@cantrell.org.uk|Data-Compare|0.17|OpenBSD.i386-openbsd|5.8.8|openbsd|3.5|200708081147
559501|pass|200708|david@cantrell.org.uk|Data-Compare|0.17|i686-linux|5.8.8|linux|2.4.27-3-686|200708081457
559543|pass|200708|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Compare|0.17|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200708081416
559576|pass|200708|tom@eborcom.com ("Tom Hukins")|Data-Compare|0.17|i686-linux|5.9.5 patch 31576|linux|2.6.17-11-generic|200708080757
559775|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200708081635
559788|pass|200708|david@cantrell.org.uk|Data-Compare|0.17|i686-linux|5.6.2|linux|2.4.27-3-686|200708081706
559811|pass|200708|david@cantrell.org.uk|Data-Compare|0.17|i686-linux|5.9.5|linux|2.4.27-3-686|200708081751
559866|pass|200708|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-64int|5.9.4|linux|2.6.17-2-k7|200708081955
559868|pass|200708|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-64int|5.9.3|linux|2.6.14|200708081956
559872|pass|200708|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-64int|5.8.8 patch 30961|linux|2.6.18-4-k7|200708081958
559876|pass|200708|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200708081959
559878|pass|200708|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200708082000
559880|pass|200708|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-thread-multi-64int|5.8.4|linux|2.6.18-4-k7|200708082001
559882|pass|200708|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-thread-multi-64int|5.8.3|linux|2.6.18-4-k7|200708082002
559883|pass|200708|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200708082003
559905|pass|200708|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-64int|5.9.1 patch 21538|linux|2.4.22-rc2-ac2|200708081957
559906|pass|200708|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-thread-multi-64int|5.9.2|linux|2.6.18-4-k7|200708081956
559907|pass|200708|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200708081958
560131|pass|200708|feldt@nhn.ou.edu|Data-Compare|0.17|sun4-solaris-thread-multi|5.8.8|solaris|2.8|200708081419
560273|pass|200708|g.grigelionis@computer.org|Data-Compare|0.17|i586-linux-thread-multi|5.8.8|linux|2.6.16|200708081741
561378|pass|200708|stro@cpan.org|Data-Compare|0.17|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200708091302
561436|pass|200708|imacat@mail.imacat.idv.tw|Data-Compare|0.17|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200708092112
561571|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200708091651
561620|pass|200708|david@cantrell.org.uk|File-Find-Rule-Permissions|1.2|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200708091620
561639|pass|200708|david@cantrell.org.uk|XML-Tiny|1.06|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200708091626
561654|pass|200708|david@cantrell.org.uk|File-Find-Rule-Permissions|1.2|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200708091612
561872|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200708092054
562017|pass|200708|david@cantrell.org.uk|Data-Compare|0.17|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200708092047
562255|pass|200708|david@cantrell.org.uk|Tie-Scalar-Decay|1.1.1|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200708092050
563659|pass|200708|david@cantrell.org.uk|Data-Compare|0.17|darwin-2level|5.9.5|darwin|8.10.1|200708101841
563736|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i386-pc-solaris2.11|5.8.7|solaris|2.11|200708101903
564951|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200708111221
565096|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200708111422
566789|pass|200708|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.06|i686-linux-thread-multi-64int|5.8.8 patch 31223|linux|2.6.22-1-k7|200708121117
567459|pass|200708|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-thread-multi-64int|5.8.8 patch 31223|linux|2.6.22-1-k7|200708121328
569962|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i386-netbsd-thread-multi-64int|5.9.5 patch 31701|netbsd|3.1|200708131004
570482|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|alpha-linux-thread-multi|5.8.8|linux|2.6.16.19|200708131205
571994|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i386-freebsd-thread-multi-64int|5.9.5|freebsd|6.2-release|200708131852
573512|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|alpha-linux-thread-multi|5.9.5 patch 31711|linux|2.6.16.19|200708141248
574134|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200708141634
577657|pass|200708|david@cantrell.org.uk|Number-Phone|1.58|darwin-2level|5.9.5|darwin|8.10.1|200708160846
577933|pass|200708|chris@bingosnet.co.uk|Number-Phone|1.58|i86pc-solaris-thread-multi-64int|5.9.5|solaris|2.10|200708161122
578018|pass|200708|david@cantrell.org.uk|Number-Phone|1.58|i686-linux|5.8.8|linux|2.4.27-3-686|200708161225
578081|pass|200708|david@cantrell.org.uk|Number-Phone|1.58|i686-linux|5.9.5|linux|2.4.27-3-686|200708161239
578089|pass|200708|david@cantrell.org.uk|Number-Phone|1.58|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200708161137
581816|pass|200708|chris@bingosnet.co.uk|Number-Phone|1.58|i686-linux-thread-multi-64int|5.9.5|linux|2.6.16.38|200708172240
582029|pass|200708|tony@develop-help.com|Number-Phone|1.58|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200708181009
582646|fail|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.0|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200708190002
582647|fail|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.0|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200708190003
582649|fail|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.0|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200708190004
582650|fail|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.0|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200708190005
582652|fail|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.0|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200708190006
582653|fail|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.0|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200708190007
582654|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200708190009
582655|fail|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.0|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200708190010
583002|pass|200708|chris@bingosnet.co.uk|Number-Phone|1.58|OpenBSD.i386-openbsd-thread-multi-64int|5.9.5|openbsd|4.1|200708190601
583066|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200708190837
583126|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|OpenBSD.i386-openbsd-thread-multi-64int|5.9.5|openbsd|4.1|200708191235
583431|pass|200708|chris@bingosnet.co.uk|Number-Phone|1.58|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200708191551
583694|fail|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200708191732
583699|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200708191734
583850|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200708191937
584790|pass|200708|imacat@mail.imacat.idv.tw|CPAN-FindDependencies|1.0|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200708201327
584903|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200708200824
584909|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200708200827
584915|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200708200829
584924|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200708200832
584932|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200708200834
584938|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200708200837
584941|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200708200841
584952|pass|200708|chris@bingosnet.co.uk|Number-Phone|1.58|i386-freebsd-thread-multi-64int|5.9.5|freebsd|6.2-release|200708200342
585019|pass|200708|saper@cpan.org|CPAN-FindDependencies|1.02|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200708201042
585127|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.0|i386-freebsd-thread-multi-64int|5.9.5|freebsd|6.2-release|200708200543
585132|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|i386-freebsd-thread-multi-64int|5.9.5|freebsd|6.2-release|200708200545
585137|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|i386-freebsd-thread-multi-64int|5.9.5|freebsd|6.2-release|200708200547
585170|pass|200708|chris@bingosnet.co.uk|Number-Phone|1.58|alpha-linux-thread-multi|5.8.8|linux|2.6.16.19|200708201137
585345|pass|200708|imacat@mail.imacat.idv.tw|CPAN-FindDependencies|1.01|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200708201950
585388|fail|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.0|alpha-linux-thread-multi|5.8.8|linux|2.6.16.19|200708201318
585397|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|alpha-linux-thread-multi|5.8.8|linux|2.6.16.19|200708201321
585398|pass|200708|imacat@mail.imacat.idv.tw|CPAN-FindDependencies|1.02|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200708202021
585401|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|alpha-linux-thread-multi|5.8.8|linux|2.6.16.19|200708201324
585417|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200708201331
585420|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200708201334
585423|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200708201338
585425|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200708201341
585428|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200708201344
585430|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200708201348
585431|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200708201350
585436|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200708201353
585809|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200708201719
585816|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200708201722
585822|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200708201724
585836|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200708201727
585840|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200708201729
585844|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200708201731
585847|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200708201733
585854|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200708201736
586798|fail|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.0|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200708210028
586800|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200708210031
586802|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200708210034
587051|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200708210635
587052|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200708210635
587053|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200708210636
587054|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200708210636
587055|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200708210636
587056|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200708210636
587057|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200708210636
587058|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200708210637
587059|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200708210637
587240|pass|200708|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|CPAN-FindDependencies|1.02|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200708210843
587241|pass|200708|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Hexdumper|1.3|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200708210844
587255|pass|200708|saper@cpan.org|Data-Hexdumper|1.3|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200708211047
587301|pass|200708|plu@pqpq.de|Data-Compare|0.17|i386-freebsd|5.8.8|freebsd|6.1-release-p6|200708211044
587322|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|alpha-linux-thread-multi|5.9.5 patch 31739|linux|2.6.16.19|200708211012
587418|pass|200708|david@cantrell.org.uk|Data-Compare|0.17|i386-freebsd|5.9.5|freebsd|6.2-release|200708211028
587508|pass|200708|chris@bingosnet.co.uk|Number-Phone|1.58|alpha-linux-thread-multi|5.9.5 patch 31739|linux|2.6.16.19|200708211103
587652|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.0|alpha-linux-thread-multi|5.9.5 patch 31739|linux|2.6.16.19|200708211244
587655|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|alpha-linux-thread-multi|5.9.5 patch 31739|linux|2.6.16.19|200708211245
587656|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|alpha-linux-thread-multi|5.9.5 patch 31739|linux|2.6.16.19|200708211247
587657|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|alpha-linux-thread-multi|5.9.5 patch 31739|linux|2.6.16.19|200708211247
587803|pass|200708|david@cantrell.org.uk|Data-Hexdumper|1.3|i686-linux|5.8.8|linux|2.4.27-3-686|200708211459
587851|pass|200708|david@cantrell.org.uk|Data-Hexdumper|1.3|i686-linux|5.6.2|linux|2.4.27-3-686|200708211525
587855|pass|200708|david@cantrell.org.uk|Data-Hexdumper|1.3|i686-linux|5.9.5|linux|2.4.27-3-686|200708211529
587856|pass|200708|chris@bingosnet.co.uk|Number-Phone|1.58|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200708211534
587942|pass|200708|plu@pqpq.de|Number-Phone|1.58|i386-freebsd|5.8.8|freebsd|6.1-release-p6|200708211120
588007|pass|200708|plu@pqpq.de|Data-Compare|0.17|i386-freebsd|5.8.8|freebsd|6.1-release-p6|200708211358
588093|pass|200708|david@cantrell.org.uk|Data-Hexdumper|1.3|darwin-2level|5.9.5|darwin|8.10.1|200708211615
588147|pass|200708|plu@pqpq.de|CPAN-FindDependencies|1.01|i386-freebsd|5.8.8|freebsd|6.1-release-p6|200708211533
588180|pass|200708|plu@pqpq.de|Number-Phone|1.58|i386-freebsd|5.8.8|freebsd|6.1-release-p6|200708211424
588194|pass|200708|tom@eborcom.com ("Tom Hukins")|Data-Hexdumper|1.3|i686-linux|5.9.5 patch 31576|linux|2.6.17-11-generic|200708211135
588231|fail|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.0|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200708211653
588253|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200708211656
588256|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200708211658
588258|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200708211658
588287|pass|200708|plu@pqpq.de|CPAN-FindDependencies|1.02|i386-freebsd|5.8.8|freebsd|6.1-release-p6|200708211533
588363|pass|200708|david@cantrell.org.uk|Data-Hexdumper|1.3|i386-freebsd|5.9.5|freebsd|6.2-release|200708211830
588480|fail|200708|plu@pqpq.de|CPAN-FindDependencies|1.0|i386-freebsd|5.8.8|freebsd|6.1-release-p6|200708211532
588481|pass|200708|plu@pqpq.de|Data-Hexdumper|1.3|i386-freebsd|5.8.8|freebsd|6.1-release-p6|200708211534
588709|pass|200708|david@cantrell.org.uk|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux|5.6.2|linux|2.4.27-3-686|200708212106
588724|pass|200708|david@cantrell.org.uk|Bryar|3.0|i686-linux|5.6.2|linux|2.4.27-3-686|200708212109
588730|pass|200708|david@cantrell.org.uk|Data-Hexdumper|1.0.1|i686-linux|5.6.2|linux|2.4.27-3-686|200708212110
588735|pass|200708|david@cantrell.org.uk|File-Overwrite|1.0|i686-linux|5.6.2|linux|2.4.27-3-686|200708212110
588736|pass|200708|david@cantrell.org.uk|Data-Transactional|0.1|i686-linux|5.6.2|linux|2.4.27-3-686|200708212110
588738|pass|200708|david@cantrell.org.uk|File-Find-Rule-Permissions|1.2|i686-linux|5.6.2|linux|2.4.27-3-686|200708212110
588739|pass|200708|david@cantrell.org.uk|Games-Dice-Advanced|1.1|i686-linux|5.6.2|linux|2.4.27-3-686|200708212111
588740|pass|200708|david@cantrell.org.uk|NestedMap|1.0|i686-linux|5.6.2|linux|2.4.27-3-686|200708212111
588744|pass|200708|david@cantrell.org.uk|Net-Random|2.0|i686-linux|5.6.2|linux|2.4.27-3-686|200708212113
588751|pass|200708|david@cantrell.org.uk|Number-Phone-UK-DetailedLocations|1.3|i686-linux|5.6.2|linux|2.4.27-3-686|200708212116
588764|pass|200708|david@cantrell.org.uk|Statistics-ChiSquare|0.5|i686-linux|5.6.2|linux|2.4.27-3-686|200708212117
588765|pass|200708|david@cantrell.org.uk|Sub-WrapPackages|1.2|i686-linux|5.6.2|linux|2.4.27-3-686|200708212118
588766|pass|200708|david@cantrell.org.uk|Statistics-SerialCorrelation|1.1|i686-linux|5.6.2|linux|2.4.27-3-686|200708212117
588767|pass|200708|david@cantrell.org.uk|Tie-Hash-Longest|1.1|i686-linux|5.6.2|linux|2.4.27-3-686|200708212118
588768|pass|200708|david@cantrell.org.uk|Tie-Hash-Rank|1.0.1|i686-linux|5.6.2|linux|2.4.27-3-686|200708212118
588819|fail|200708|david@cantrell.org.uk|Tie-STDOUT|1.02|i686-linux|5.6.2|linux|2.4.27-3-686|200708212145
588820|pass|200708|david@cantrell.org.uk|Tie-Scalar-Decay|1.1.1|i686-linux|5.6.2|linux|2.4.27-3-686|200708212146
588822|pass|200708|david@cantrell.org.uk|Tie-Scalar-Decay|1.1|i686-linux|5.6.2|linux|2.4.27-3-686|200708212146
588824|pass|200708|david@cantrell.org.uk|XML-Tiny|1.06|i686-linux|5.6.2|linux|2.4.27-3-686|200708212146
588886|pass|200708|chris@bingosnet.co.uk|Number-Phone|1.58|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200708212205
589058|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.0|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200708212319
589059|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200708212319
589061|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200708212320
589063|unknown|200708|david@cantrell.org.uk|Acme-Licence|1.0|darwin-2level|5.9.5|darwin|8.10.1|200708212318
589066|pass|200708|david@cantrell.org.uk|Acme-Scurvy-Whoreson-BilgeRat|1.1|darwin-2level|5.9.5|darwin|8.10.1|200708212318
589067|unknown|200708|david@cantrell.org.uk|Acme-Pony|1.1.2|darwin-2level|5.9.5|darwin|8.10.1|200708212318
589072|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200708212320
589087|fail|200708|david@cantrell.org.uk|Bryar|3.0|darwin-2level|5.9.5|darwin|8.10.1|200708212323
589089|pass|200708|david@cantrell.org.uk|Class-CanBeA|1.2|darwin-2level|5.9.5|darwin|8.10.1|200708212323
589093|fail|200708|david@cantrell.org.uk|Data-Transactional|0.1|darwin-2level|5.9.5|darwin|8.10.1|200708212323
589095|pass|200708|david@cantrell.org.uk|File-Overwrite|1.0|darwin-2level|5.9.5|darwin|8.10.1|200708212324
589096|pass|200708|david@cantrell.org.uk|File-Find-Rule-Permissions|1.2|darwin-2level|5.9.5|darwin|8.10.1|200708212323
589099|pass|200708|david@cantrell.org.uk|Net-Random|2.0|darwin-2level|5.9.5|darwin|8.10.1|200708212324
589100|pass|200708|david@cantrell.org.uk|Games-Dice-Advanced|1.1|darwin-2level|5.9.5|darwin|8.10.1|200708212324
589102|pass|200708|david@cantrell.org.uk|NestedMap|1.0|darwin-2level|5.9.5|darwin|8.10.1|200708212324
589105|pass|200708|david@cantrell.org.uk|Number-Phone-UK-DetailedLocations|1.3|darwin-2level|5.9.5|darwin|8.10.1|200708212327
589106|pass|200708|david@cantrell.org.uk|Palm-TreoPhoneCallDB|1.1|darwin-2level|5.9.5|darwin|8.10.1|200708212328
589107|pass|200708|david@cantrell.org.uk|Palm-TreoPhoneCallDB|1|darwin-2level|5.9.5|darwin|8.10.1|200708212328
589110|pass|200708|david@cantrell.org.uk|Statistics-ChiSquare|0.5|darwin-2level|5.9.5|darwin|8.10.1|200708212328
589112|pass|200708|david@cantrell.org.uk|Statistics-SerialCorrelation|1.1|darwin-2level|5.9.5|darwin|8.10.1|200708212328
589114|pass|200708|david@cantrell.org.uk|Tie-Hash-Longest|1.1|darwin-2level|5.9.5|darwin|8.10.1|200708212328
589115|pass|200708|david@cantrell.org.uk|Sub-WrapPackages|1.2|darwin-2level|5.9.5|darwin|8.10.1|200708212328
589116|pass|200708|david@cantrell.org.uk|Tie-STDOUT|1.02|darwin-2level|5.9.5|darwin|8.10.1|200708212328
589117|pass|200708|david@cantrell.org.uk|Tie-Hash-Rank|1.0.1|darwin-2level|5.9.5|darwin|8.10.1|200708212328
589119|pass|200708|david@cantrell.org.uk|Tie-Scalar-Decay|1.1.1|darwin-2level|5.9.5|darwin|8.10.1|200708212329
589120|pass|200708|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|darwin-2level|5.9.5|darwin|8.10.1|200708212329
589121|pass|200708|david@cantrell.org.uk|Tie-Scalar-Decay|1.1|darwin-2level|5.9.5|darwin|8.10.1|200708212329
589122|pass|200708|david@cantrell.org.uk|XML-Tiny|1.06|darwin-2level|5.9.5|darwin|8.10.1|200708212329
589147|pass|200708|david@cantrell.org.uk|Data-Hexdumper|1.2|darwin-2level|5.9.5|darwin|8.10.1|200708212323
589149|pass|200708|david@cantrell.org.uk|Data-Hexdumper|1.0.1|darwin-2level|5.9.5|darwin|8.10.1|200708212323
589707|pass|200708|david@cantrell.org.uk|Data-Hexdumper|1.3|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200708220732
589902|pass|200708|stro@cpan.org|Data-Hexdumper|1.3|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200708221013
589905|pass|200708|stro@cpan.org|CPAN-FindDependencies|1.0|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200708220951
589906|pass|200708|stro@cpan.org|CPAN-FindDependencies|1.02|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200708220952
589960|pass|200708|stro@cpan.org|CPAN-FindDependencies|1.01|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200708220951
590091|pass|200708|chaddai.fouche@gmail.com ("Chaddai Fouche")|Data-Compare|0.17|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200708221618
590117|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.0|i386-pc-solaris2.11|5.8.7|solaris|2.11|200708221558
590120|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|i386-pc-solaris2.11|5.8.7|solaris|2.11|200708221600
590122|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|i386-pc-solaris2.11|5.8.7|solaris|2.11|200708221601
590124|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|i386-pc-solaris2.11|5.8.7|solaris|2.11|200708221601
590675|pass|200708|imacat@mail.imacat.idv.tw|Data-Hexdumper|1.3|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200708230522
591020|pass|200708|tony@develop-help.com|Data-Compare|0.17|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200708230944
591446|pass|200708|bob@randomness.org.uk ("bob walker")|Bryar|3.0|i86pc-solaris|5.8.8|solaris|2.10|200708231015
591701|pass|200708|cpan@pqpq.de|CPAN-FindDependencies|1.02|i386-freebsd-thread-multi|5.9.5 patch 31749|freebsd|6.1-release-p6|200708231335
591731|pass|200708|cpan@pqpq.de|Data-Hexdumper|1.3|i386-freebsd-thread-multi|5.9.5 patch 31749|freebsd|6.1-release-p6|200708231335
591734|pass|200708|cpan@pqpq.de|CPAN-FindDependencies|1.0|i386-freebsd-thread-multi|5.9.5 patch 31749|freebsd|6.1-release-p6|200708231334
591735|pass|200708|cpan@pqpq.de|CPAN-FindDependencies|1.01|i386-freebsd-thread-multi|5.9.5 patch 31749|freebsd|6.1-release-p6|200708231335
591769|pass|200708|cpan@pqpq.de|Data-Compare|0.17|i386-freebsd-thread-multi|5.9.5 patch 31749|freebsd|6.1-release-p6|200708231210
592026|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.0|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200708231648
592027|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200708231649
592028|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200708231651
592029|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200708231651
592644|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200708232350
592919|pass|200708|cogent@cpan.org|Data-Compare|0.17|darwin-thread-multi-64int-2level|5.8.8|darwin|8.7.0|200708231440
593087|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i86pc-solaris-thread-multi-64int|5.9.5|solaris|2.10|200708240501
593443|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|i86pc-solaris-thread-multi-64int|5.9.5|solaris|2.10|200708241048
593448|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|i86pc-solaris-thread-multi-64int|5.9.5|solaris|2.10|200708241050
593461|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.0|i86pc-solaris-thread-multi-64int|5.9.5|solaris|2.10|200708241047
593475|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|i86pc-solaris-thread-multi-64int|5.9.5|solaris|2.10|200708241050
594276|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200708250644
594281|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200708250648
594282|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200708250649
594391|pass|200708|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|1.02|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200708250940
594684|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200708251155
594685|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200708251156
594686|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200708251156
595047|pass|200708|srezic@cpan.org|CPAN-FindDependencies|1.02|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200708251106
595064|pass|200708|chris@bingosnet.co.uk|Data-Compare|0.17|i686-linux-thread-multi-64int|5.9.5|linux|2.6.16.38|200708251908
595274|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.01|i686-linux-thread-multi-64int|5.9.5|linux|2.6.16.38|200708260006
595275|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|i686-linux-thread-multi-64int|5.9.5|linux|2.6.16.38|200708260007
595276|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|i686-linux-thread-multi-64int|5.9.5|linux|2.6.16.38|200708260007
595889|pass|200708|chris@bingosnet.co.uk|CPAN-FindDependencies|1.02|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200708262118
595890|pass|200708|chris@bingosnet.co.uk|Data-Hexdumper|1.3|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200708262118
597202|pass|200708|chris@bingosnet.co.uk|Tie-STDOUT|1.03|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200708281836
597204|pass|200708|chris@bingosnet.co.uk|Tie-STDOUT|1.03|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200708281837
597205|pass|200708|chris@bingosnet.co.uk|Tie-STDOUT|1.03|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200708281837
597207|pass|200708|chris@bingosnet.co.uk|Tie-STDOUT|1.03|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200708281837
597208|pass|200708|chris@bingosnet.co.uk|Tie-STDOUT|1.03|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200708281838
597209|pass|200708|chris@bingosnet.co.uk|Tie-STDOUT|1.03|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200708281838
597210|pass|200708|chris@bingosnet.co.uk|Tie-STDOUT|1.03|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200708281838
597211|pass|200708|chris@bingosnet.co.uk|Tie-STDOUT|1.03|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200708281839
597213|pass|200708|chris@bingosnet.co.uk|Tie-STDOUT|1.03|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200708281839
597630|pass|200708|chris@bingosnet.co.uk|Data-Transactional|1.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200708290537
597669|pass|200708|chris@bingosnet.co.uk|Data-Transactional|1.0|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200708290641
597670|pass|200708|chris@bingosnet.co.uk|Data-Transactional|1.0|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200708290642
597671|pass|200708|chris@bingosnet.co.uk|Data-Transactional|1.0|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200708290643
597672|pass|200708|chris@bingosnet.co.uk|Data-Transactional|1.0|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200708290643
597673|pass|200708|chris@bingosnet.co.uk|Data-Transactional|1.0|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200708290644
597674|pass|200708|chris@bingosnet.co.uk|Data-Transactional|1.0|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200708290644
597675|pass|200708|chris@bingosnet.co.uk|Data-Transactional|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200708290645
597676|pass|200708|chris@bingosnet.co.uk|Data-Transactional|1.0|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200708290646
597713|pass|200708|david@cantrell.org.uk|Tie-STDOUT|1.03|darwin-2level|5.9.5|darwin|8.10.1|200708290946
597716|pass|200708|david@cantrell.org.uk|Tie-STDOUT|1.03|i686-linux|5.8.8|linux|2.4.27-3-686|200708290951
597730|pass|200708|david@cantrell.org.uk|Tie-STDOUT|1.03|i686-linux|5.9.5|linux|2.4.27-3-686|200708290959
597780|pass|200708|david@cantrell.org.uk|Data-Transactional|1.0|i686-linux|5.6.2|linux|2.4.27-3-686|200708291105
597790|pass|200708|david@cantrell.org.uk|Data-Transactional|1.0|darwin-2level|5.9.5|darwin|8.10.1|200708291109
597795|pass|200708|david@cantrell.org.uk|Data-Transactional|1.0|i686-linux|5.8.8|linux|2.4.27-3-686|200708291110
597813|pass|200708|david@cantrell.org.uk|Data-Transactional|1.0|i686-linux|5.9.5|linux|2.4.27-3-686|200708291219
597905|pass|200708|stro@cpan.org|Data-Transactional|1.0|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200708291330
597961|pass|200708|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Transactional|1.0|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200708291427
597962|pass|200708|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Tie-STDOUT|1.03|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200708291427
598003|pass|200708|david@cantrell.org.uk|Tie-STDOUT|1.03|i386-freebsd|5.9.5|freebsd|6.2-release|200708290937
598061|pass|200708|imacat@mail.imacat.idv.tw|Tie-STDOUT|1.03|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200708292348
598091|pass|200708|david@cantrell.org.uk|Data-Transactional|1.0|i386-freebsd|5.9.5|freebsd|6.2-release|200708291141
598406|pass|200708|imacat@mail.imacat.idv.tw|Data-Transactional|1.0|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200708301923
598875|pass|200708|tony@develop-help.com|Data-Transactional|1.0|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200708311047
598876|na|200708|tony@develop-help.com|Tie-STDOUT|1.03|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200708311048
598901|pass|200708|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200708310354
598903|pass|200708|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200708310355
598905|pass|200708|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200708310356
598907|pass|200708|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200708310358
598909|pass|200708|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200708310359
598911|pass|200708|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200708310400
598913|pass|200708|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200708310401
598915|pass|200708|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200708310402
599214|pass|200708|david@cantrell.org.uk|Tie-STDOUT|1.03|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200708311420
599278|pass|200708|david@cantrell.org.uk|Data-Transactional|1.0|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200708311520
599522|pass|200709|cpan@pqpq.de|Data-Transactional|1.0|i386-freebsd-thread-multi|5.9.5 patch 31749|freebsd|6.1-release-p6|200709010933
599523|pass|200709|cpan@pqpq.de|Tie-STDOUT|1.03|i386-freebsd-thread-multi|5.9.5 patch 31749|freebsd|6.1-release-p6|200709010933
599621|pass|200709|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|i386-freebsd|5.9.5|freebsd|6.2-release|200709010951
599788|pass|200709|chris@bingosnet.co.uk|Data-Transactional|1.0|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200709011328
599789|pass|200709|chris@bingosnet.co.uk|Tie-STDOUT|1.03|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200709011328
599841|pass|200709|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200709011415
599842|pass|200709|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-64int|5.9.4|linux|2.6.17-2-k7|200709011417
599847|pass|200709|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200709011421
599849|pass|200709|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200709011422
599850|pass|200709|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200709011423
599851|pass|200709|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int|5.8.4|linux|2.6.18-4-k7|200709011424
599852|pass|200709|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int|5.8.3|linux|2.6.18-4-k7|200709011425
599853|pass|200709|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200709011426
599857|pass|200709|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-64int|5.9.3|linux|2.6.14|200709011418
599858|pass|200709|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int|5.8.8 patch 31223|linux|2.6.22-1-k7|200709011420
600229|pass|200709|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200709011840
600617|pass|200709|chris@bingosnet.co.uk|Data-Transactional|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.9.5|openbsd|4.1|200709012018
600618|pass|200709|chris@bingosnet.co.uk|Data-Compare|0.17|OpenBSD.i386-openbsd-thread-multi-64int|5.9.5|openbsd|4.1|200709012018
600620|pass|200709|chris@bingosnet.co.uk|Tie-STDOUT|1.03|OpenBSD.i386-openbsd-thread-multi-64int|5.9.5|openbsd|4.1|200709012019
600828|pass|200709|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200709012036
601889|pass|200709|chris@bingosnet.co.uk|Acme-Licence|1.0|i386-netbsd-thread-multi-64int|5.9.5|netbsd|3.1|200709021301
602062|pass|200709|chris@bingosnet.co.uk|Acme-Pony|1.1.2|i386-netbsd-thread-multi-64int|5.9.5|netbsd|3.1|200709021344
602128|pass|200709|chris@bingosnet.co.uk|Acme-Scurvy-Whoreson-BilgeRat|1.1|i386-netbsd-thread-multi-64int|5.9.5|netbsd|3.1|200709021403
602983|pass|200709|bob@randomness.org.uk ("bob walker")|Tie-STDOUT|1.03|i86pc-solaris|5.8.8|solaris|2.10|200709021453
603004|pass|200709|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.9.5|openbsd|4.1|200709020706
603202|pass|200709|jrockway@cpan.org|Data-Compare|0.17|i486-linux-gnu|5.8.8|linux|2.6.18.3|200709011804
603343|pass|200709|chris@bingosnet.co.uk|Data-Transactional|1.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200709021923
603344|pass|200709|chris@bingosnet.co.uk|Tie-STDOUT|1.03|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200709021923
603532|pass|200709|corion-cpanplus@corion.net|Data-Compare|0.17|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200709022206
603534|pass|200709|corion-cpanplus@corion.net|Data-Transactional|1.0|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200709022207
603536|fail|200709|corion-cpanplus@corion.net|Tie-STDOUT|1.03|MSWin32-x86-multi-thread|5.8.5|MSWin32|4.0|200709022208
603917|pass|200709|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200709030032
604525|pass|200709|cpan@pqpq.de|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi|5.9.5 patch 31749|freebsd|6.1-release-p6|200709031103
604902|pass|200709|chris@bingosnet.co.uk|Data-Transactional|1.0|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200709031349
604903|pass|200709|chris@bingosnet.co.uk|Tie-STDOUT|1.03|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200709031350
604907|pass|200709|saper@cpan.org|Tie-STDOUT|1.03|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200709031450
605243|pass|200709|chris@bingosnet.co.uk|Data-Transactional|1.0|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200709031624
605244|pass|200709|chris@bingosnet.co.uk|Tie-STDOUT|1.03|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200709031624
605308|pass|200709|chris@bingosnet.co.uk|Data-Transactional|1.0|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200709031812
605309|pass|200709|chris@bingosnet.co.uk|Tie-STDOUT|1.03|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200709031813
606619|pass|200709|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200709040455
606840|pass|200709|chris@bingosnet.co.uk|Data-Transactional|1.0|i386-pc-solaris2.11|5.8.7|solaris|2.11|200709040750
606845|pass|200709|chris@bingosnet.co.uk|Tie-STDOUT|1.03|i386-pc-solaris2.11|5.8.7|solaris|2.11|200709040750
607296|pass|200709|chris@bingosnet.co.uk|Data-Transactional|1.0|alpha-linux-thread-multi|5.8.8|linux|2.6.16.19|200709041124
607298|pass|200709|chris@bingosnet.co.uk|Tie-STDOUT|1.03|alpha-linux-thread-multi|5.8.8|linux|2.6.16.19|200709041124
607602|pass|200709|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-pc-solaris2.11|5.8.7|solaris|2.11|200709041352
607854|pass|200709|chris@bingosnet.co.uk|Data-Transactional|1.0|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200709041631
607857|pass|200709|chris@bingosnet.co.uk|Tie-STDOUT|1.03|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200709041631
607951|pass|200709|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|alpha-linux-thread-multi|5.8.8|linux|2.6.16.19|200709041750
609071|pass|200709|chris@bingosnet.co.uk|Data-Transactional|1.0|i86pc-solaris-thread-multi-64int|5.9.5|solaris|2.10|200709050035
609076|pass|200709|chris@bingosnet.co.uk|Tie-STDOUT|1.03|i86pc-solaris-thread-multi-64int|5.9.5|solaris|2.10|200709050036
609166|pass|200709|cpan@pqpq.de|Tie-STDOUT|1.03|i86pc-solaris|5.9.5 patch 31788|solaris|2.10|200709042059
609202|pass|200709|cpan@pqpq.de|Data-Transactional|1.0|i86pc-solaris|5.9.5 patch 31788|solaris|2.10|200709042059
609215|pass|200709|cpan@pqpq.de|Data-Compare|0.17|i86pc-solaris|5.9.5 patch 31788|solaris|2.10|200709042059
610943|pass|200709|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200709052027
611293|pass|200709|perlgeek@perlgeek.net|Tie-STDOUT|1.03|i686-linux|5.8.8|linux|2.6.9-42.0.3.elsmp|200709060223
611964|pass|200709|davidp@preshweb.co.uk ("David Precious")|Data-Compare|0.17|i486-linux|5.8.7|linux|2.4.31|200709061158
612850|pass|200709|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.06|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200709062106
616595|pass|200709|cpan@0x64.nl|XML-Tiny|1.06|x86_64-linux-gnu|5.8.8|linux|2.6.18-1-amd64|200709081211
617035|pass|200709|david@cantrell.org.uk|Data-Compare|0.02|i686-linux|5.6.2|linux|2.4.27-3-686|200709090041
617180|pass|200709|andy@hexten.net|Data-Hexdumper|1.3|i486-linux-gnu|5.8.8|linux|2.6.15.7|200709090338
617430|pass|200709|smueller@cpan.org ("Steffen Mueller")|XML-Tiny|1.06|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200709091052
620597|pass|200709|david@cantrell.org.uk|Data-Compare|0.02|darwin-2level|5.9.5|darwin|8.10.1|200709102055
622464|pass|200709|kamensky@cpan.org ("Moshe Kamensky")|Data-Compare|0.17|sun4-solaris|5.8.8|solaris|2.8|200709111136
623174|pass|200709|andy@hexten.net|Data-Hexdumper|1.3|i486-linux-gnu|5.8.8|linux|2.6.15.7|200709112108
625331|pass|200709|david@cantrell.org.uk|Data-Compare|0.17|sun4-solaris|5.9.5|solaris|2.9|200709121723
631722|pass|200709|chris@bingosnet.co.uk|Data-Compare|0.17|i386-freebsd-thread-multi-64int|5.8.8|freebsd|5.3-release|200709151723
632261|pass|200709|gabor@pti.co.il ("Gabor Szabo")|XML-DoubleEncodedEntities|1.0|i686-linux|5.8.8|linux|2.6.20-16-generic|200709151533
632453|pass|200709|gabor@pti.co.il ("Gabor Szabo")|Data-Compare|0.17|i686-linux|5.8.8|linux|2.6.20-16-generic|200709151221
634970|pass|200709|andy@hexten.net|Data-Hexdumper|1.3|i486-linux-gnu|5.8.8|linux|2.6.15.7|200709160900
636217|pass|200709|cpan@pqpq.de|Data-Compare|0.17|i386-freebsd|5.10.0 patch 31806|freebsd|6.1-release-p6|200709162224
644503|pass|200709|srezic@cpan.org|CPAN-FindDependencies|1.02|amd64-freebsd|5.10.0 patch 31894|freebsd|6.2-release|200709200002
645430|pass|200709|chris@bingosnet.co.uk|Data-Compare|0.17|i686-linux-thread-multi-64int|5.10.0 patch 31906|linux|2.6.16.38|200709201232
647235|pass|200709|chris@bingosnet.co.uk|Data-Compare|0.17|darwin-thread-multi-2level|5.8.6|darwin|8.10.0|200709210757
649448|pass|200709|chris@bingosnet.co.uk|Data-Compare|0.17|darwin-thread-multi-2level|5.8.5|darwin|8.10.0|200709220425
649692|pass|200709|srezic@cpan.org|CPAN-FindDependencies|1.02|i386-freebsd|5.10.0 patch 31938|freebsd|6.1-release|200709220244
650269|pass|200709|chris@bingosnet.co.uk|Data-Compare|0.17|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200709221114
651644|pass|200709|chris@bingosnet.co.uk|Data-Compare|0.17|darwin-thread-multi-2level|5.8.4|darwin|8.10.0|200709230013
653290|unknown|200709|david@cantrell.org.uk|Acme-Licence|1.0|i386-freebsd|5.9.5|freebsd|6.2-release|200709231412
653303|unknown|200709|david@cantrell.org.uk|Acme-Pony|1.1.2|i386-freebsd|5.9.5|freebsd|6.2-release|200709231415
653315|pass|200709|david@cantrell.org.uk|Acme-Scurvy-Whoreson-BilgeRat|1.1|i386-freebsd|5.9.5|freebsd|6.2-release|200709231416
653355|fail|200709|david@cantrell.org.uk|Bryar|3.0|i386-freebsd|5.9.5|freebsd|6.2-release|200709231422
653360|pass|200709|david@cantrell.org.uk|Class-CanBeA|1.2|i386-freebsd|5.9.5|freebsd|6.2-release|200709231423
653373|pass|200709|david@cantrell.org.uk|Data-Hexdumper|1.0.1|i386-freebsd|5.9.5|freebsd|6.2-release|200709231425
653383|pass|200709|david@cantrell.org.uk|Data-Hexdumper|1.2|i386-freebsd|5.9.5|freebsd|6.2-release|200709231426
653410|fail|200709|david@cantrell.org.uk|Data-Transactional|0.1|i386-freebsd|5.9.5|freebsd|6.2-release|200709231428
653422|pass|200709|david@cantrell.org.uk|File-Find-Rule-Permissions|1.2|i386-freebsd|5.9.5|freebsd|6.2-release|200709231429
653424|pass|200709|chris@bingosnet.co.uk|Acme-Pony|1.1.2|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200709231454
653426|pass|200709|david@cantrell.org.uk|File-Overwrite|1.0|i386-freebsd|5.9.5|freebsd|6.2-release|200709231431
653429|pass|200709|david@cantrell.org.uk|Games-Dice-Advanced|1.1|i386-freebsd|5.9.5|freebsd|6.2-release|200709231432
653434|pass|200709|david@cantrell.org.uk|NestedMap|1.0|i386-freebsd|5.9.5|freebsd|6.2-release|200709231434
653448|pass|200709|david@cantrell.org.uk|Net-Random|2.0|i386-freebsd|5.9.5|freebsd|6.2-release|200709231438
653454|pass|200709|david@cantrell.org.uk|Number-Phone|1.58|i386-freebsd|5.9.5|freebsd|6.2-release|200709231439
653465|pass|200709|david@cantrell.org.uk|Number-Phone-UK-DetailedLocations|1.3|i386-freebsd|5.9.5|freebsd|6.2-release|200709231442
653473|pass|200709|david@cantrell.org.uk|Palm-TreoPhoneCallDB|1.1|i386-freebsd|5.9.5|freebsd|6.2-release|200709231444
653479|pass|200709|david@cantrell.org.uk|Palm-TreoPhoneCallDB|1|i386-freebsd|5.9.5|freebsd|6.2-release|200709231446
653484|pass|200709|david@cantrell.org.uk|Statistics-ChiSquare|0.5|i386-freebsd|5.9.5|freebsd|6.2-release|200709231449
653494|pass|200709|david@cantrell.org.uk|Statistics-SerialCorrelation|1.1|i386-freebsd|5.9.5|freebsd|6.2-release|200709231450
653506|pass|200709|david@cantrell.org.uk|Tie-Hash-Longest|1.1|i386-freebsd|5.9.5|freebsd|6.2-release|200709231453
653512|pass|200709|david@cantrell.org.uk|Tie-Hash-Rank|1.0.1|i386-freebsd|5.9.5|freebsd|6.2-release|200709231454
653518|pass|200709|david@cantrell.org.uk|Tie-STDOUT|1.02|i386-freebsd|5.9.5|freebsd|6.2-release|200709231456
653529|pass|200709|david@cantrell.org.uk|Tie-Scalar-Decay|1.1.1|i386-freebsd|5.9.5|freebsd|6.2-release|200709231457
653534|pass|200709|david@cantrell.org.uk|Tie-Scalar-Decay|1.1|i386-freebsd|5.9.5|freebsd|6.2-release|200709231459
653538|pass|200709|david@cantrell.org.uk|XML-Tiny|1.06|i386-freebsd|5.9.5|freebsd|6.2-release|200709231500
653652|pass|200709|david@cantrell.org.uk|Sub-WrapPackages|1.2|i386-freebsd|5.9.5|freebsd|6.2-release|200709231452
663263|pass|200710|imacat@mail.imacat.idv.tw|Devel-CheckOS|0.9|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200710012251
663600|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|0.92|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200710012338
663601|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|0.92|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200710012335
663602|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|0.92|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200710012334
663613|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|0.92|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200710012336
663622|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|0.92|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200710012337
663623|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|0.92|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200710012337
663624|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|0.92|i686-linux-thread-multi-64int|5.8.8 patch 31223|linux|2.6.22-1-k7|200710012336
664271|pass|200710|cpan@pqpq.de|Devel-CheckOS|0.92|i386-freebsd|5.10.0 patch 31806|freebsd|6.1-release-p6|200710020911
664272|pass|200710|Jost.Krieger@ruhr-uni-bochum.de|Acme-Pony|1.1.2|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200710020710
664273|pass|200710|cpan@pqpq.de|Devel-CheckOS|0.91|i386-freebsd|5.10.0 patch 31806|freebsd|6.1-release-p6|200710020911
664275|pass|200710|cpan@pqpq.de|Devel-CheckOS|0.9|i386-freebsd|5.10.0 patch 31806|freebsd|6.1-release-p6|200710020911
664608|pass|200710|david@cantrell.org.uk|Devel-CheckOS|0.91|darwin-2level|5.9.5|darwin|8.10.1|200710021141
664666|pass|200710|david@cantrell.org.uk|Devel-CheckOS|0.92|darwin-2level|5.9.5|darwin|8.10.1|200710021152
664742|pass|200710|srezic@cpan.org|Data-Compare|0.17|i686-linux|5.10.0 patch 31937|linux|2.6.18-4-486|200710021313
664771|pass|200710|david@cantrell.org.uk|Devel-CheckOS|0.91|sun4-solaris|5.9.5|solaris|2.9|200710020627
664779|pass|200710|david@cantrell.org.uk|Devel-CheckOS|0.91|i386-freebsd|5.9.5|freebsd|6.2-release|200710021131
664789|pass|200710|david@cantrell.org.uk|Devel-CheckOS|0.91|i686-linux|5.8.8|linux|2.4.27-3-686|200710021233
664790|pass|200710|david@cantrell.org.uk|Devel-CheckOS|0.91|i686-linux|5.9.5|linux|2.4.27-3-686|200710021234
665010|pass|200710|david@cantrell.org.uk|Devel-CheckOS|0.92|i686-linux|5.9.5|linux|2.4.27-3-686|200710021358
665011|pass|200710|david@cantrell.org.uk|Devel-CheckOS|0.91|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200710021258
665091|pass|200710|david@cantrell.org.uk|Devel-CheckOS|0.92|i686-linux|5.8.8|linux|2.4.27-3-686|200710021421
665092|pass|200710|david@cantrell.org.uk|Devel-CheckOS|0.91|i686-linux|5.6.2|linux|2.4.27-3-686|200710021421
665097|pass|200710|david@cantrell.org.uk|Devel-CheckOS|0.92|i386-freebsd|5.9.5|freebsd|6.2-release|200710021324
665102|pass|200710|david@cantrell.org.uk|Devel-CheckOS|0.92|sun4-solaris|5.9.5|solaris|2.9|200710020824
665182|pass|200710|david@cantrell.org.uk|Devel-CheckOS|0.92|i686-linux|5.6.2|linux|2.4.27-3-686|200710021445
665351|pass|200710|srezic@cpan.org|Devel-CheckOS|0.92|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200710020837
665385|pass|200710|saper@cpan.org|Devel-CheckOS|0.92|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200710021621
665491|pass|200710|srezic@cpan.org|Devel-CheckOS|0.92|amd64-freebsd|5.10.0 patch 31995|freebsd|6.2-release|200710020837
665546|pass|200710|david@cantrell.org.uk|Devel-CheckOS|0.92|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200710021501
665823|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.0|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200710021809
665827|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.0|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200710021807
665828|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.0|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200710021808
665829|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.0|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200710021808
665831|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.0|i686-linux-thread-multi-64int|5.8.8 patch 31223|linux|2.6.22-1-k7|200710021806
665832|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.0|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200710021805
665833|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.0|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200710021804
666004|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux|5.10.0|linux|2.6.22-1-k7|200710021710
666037|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.9|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200710021839
666053|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.9|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200710021846
666078|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.9|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200710021853
666102|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|File-Find-Rule-Permissions|1.3|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200710021958
666107|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|File-Find-Rule-Permissions|1.3|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200710021959
666113|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.9|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200710021901
666149|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.9|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200710021908
666179|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.9|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200710021915
666199|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|File-Find-Rule-Permissions|1.3|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200710022002
666204|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.9|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200710021922
666222|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|File-Find-Rule-Permissions|1.3|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200710022000
666228|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|File-Find-Rule-Permissions|1.3|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200710022001
666229|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|File-Find-Rule-Permissions|1.3|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200710022001
666230|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|File-Find-Rule-Permissions|1.3|i686-linux-thread-multi-64int|5.8.8 patch 31223|linux|2.6.22-1-k7|200710022000
666239|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.9|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200710021929
666247|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.9|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200710021931
666540|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.06|i686-linux|5.10.0|linux|2.6.22-1-k7|200710022105
667743|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.0|i686-linux|5.10.0|linux|2.6.22-1-k7|200710031215
667823|pass|200710|imacat@mail.imacat.idv.tw|Devel-CheckOS|0.91|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200710031934
667829|pass|200710|imacat@mail.imacat.idv.tw|Devel-CheckOS|0.92|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200710031944
667886|pass|200710|saper@cpan.org|File-Find-Rule-Permissions|1.3|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200710031453
668899|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.0|i686-linux|5.8.8|linux|2.4.27-3-686|200710032339
668904|pass|200710|david@cantrell.org.uk|File-Find-Rule-Permissions|1.3|i686-linux|5.8.8|linux|2.4.27-3-686|200710032341
668977|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.0|darwin-2level|5.9.5|darwin|8.10.1|200710040024
668982|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.0|sun4-solaris|5.9.5|solaris|2.9|200710040024
668983|pass|200710|david@cantrell.org.uk|File-Find-Rule-Permissions|1.3|darwin-2level|5.9.5|darwin|8.10.1|200710040024
668985|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.0|i386-freebsd|5.9.5|freebsd|6.2-release|200710040025
668989|pass|200710|david@cantrell.org.uk|File-Find-Rule-Permissions|1.3|sun4-solaris|5.9.5|solaris|2.9|200710040027
668990|pass|200710|david@cantrell.org.uk|File-Find-Rule-Permissions|1.3|i386-freebsd|5.9.5|freebsd|6.2-release|200710040028
669006|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.0|i686-linux|5.9.5|linux|2.4.27-3-686|200710040037
669009|pass|200710|david@cantrell.org.uk|File-Find-Rule-Permissions|1.3|i686-linux|5.9.5|linux|2.4.27-3-686|200710040038
669010|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.0|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200710040038
669019|pass|200710|david@cantrell.org.uk|File-Find-Rule-Permissions|1.3|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200710040043
669023|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.0|i686-linux|5.6.2|linux|2.4.27-3-686|200710040045
669026|pass|200710|david@cantrell.org.uk|File-Find-Rule-Permissions|1.3|i686-linux|5.6.2|linux|2.4.27-3-686|200710040046
669230|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux|5.10.0|linux|2.6.22-1-k7|200710040801
669588|pass|200710|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.0|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200710042126
669592|pass|200710|imacat@mail.imacat.idv.tw|File-Find-Rule-Permissions|1.3|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200710042127
669812|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.91|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200710041746
669813|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.91|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200710041744
669814|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.91|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200710041748
669815|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.91|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200710041750
669816|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.91|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200710041752
669817|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.91|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200710041754
669818|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.91|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200710041756
669820|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.91|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200710041758
669822|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.91|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200710041800
670143|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.92|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200710042349
670144|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.92|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200710042352
670145|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.92|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200710042353
670146|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.92|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200710042355
670148|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.92|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200710042358
670150|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.92|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200710050004
670153|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.92|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200710050007
670155|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.92|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200710050009
670157|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|0.92|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200710050012
670386|pass|200710|alexchorny@gmail.com|File-Find-Rule-Permissions|1.3|i686-linux|5.10.0 patch 32015|linux|2.4.20-8smp|200710040629
670692|pass|200710|alexchorny@gmail.com|Devel-CheckOS|1.0|i686-linux|5.10.0 patch 32015|linux|2.4.20-8smp|200710040629
670971|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.06|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200710051147
671194|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.0|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200710051318
671198|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.0|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200710051320
671199|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.0|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200710051323
671200|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.0|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200710051325
671201|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.0|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200710051327
671202|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.0|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200710051329
671215|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.0|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200710051331
671220|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.0|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200710051333
671223|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200710051335
671243|pass|200710|rlane10@wi.rr.com|Number-Phone|1.58|i386-linux|5.8.8|linux|2.6.9-34.elsmp|200710050721
671301|pass|200710|chris@bingosnet.co.uk|File-Find-Rule-Permissions|1.3|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200710051410
671306|pass|200710|chris@bingosnet.co.uk|File-Find-Rule-Permissions|1.3|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200710051413
671308|pass|200710|chris@bingosnet.co.uk|File-Find-Rule-Permissions|1.3|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200710051415
671312|pass|200710|chris@bingosnet.co.uk|File-Find-Rule-Permissions|1.3|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200710051417
671318|pass|200710|chris@bingosnet.co.uk|File-Find-Rule-Permissions|1.3|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200710051420
671322|pass|200710|chris@bingosnet.co.uk|File-Find-Rule-Permissions|1.3|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200710051422
671325|pass|200710|chris@bingosnet.co.uk|File-Find-Rule-Permissions|1.3|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200710051425
671326|pass|200710|chris@bingosnet.co.uk|File-Find-Rule-Permissions|1.3|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200710051427
671328|pass|200710|chris@bingosnet.co.uk|File-Find-Rule-Permissions|1.3|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200710051429
671367|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.1|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200710051606
671369|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.1|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200710051607
671370|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.1|i686-linux-thread-multi-64int|5.8.8 patch 31223|linux|2.6.22-1-k7|200710051608
671371|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.1|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200710051608
671374|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.1|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200710051609
671377|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.1|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200710051609
671400|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.1|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200710051610
671477|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200710051757
673261|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200710061431
674252|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200710061335
674519|fail|200710|icerider70@gmail.com ("ICERIDER")|Devel-CheckOS|1.1|i486-linux-gnu|5.8.8|linux|2.6.18.3|200710061502
674615|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.1|darwin-2level|5.9.5|darwin|8.10.1|200710062344
674617|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.1|sun4-solaris|5.9.5|solaris|2.9|200710061745
674669|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.1|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200710070004
674672|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.1|i686-linux|5.9.5|linux|2.4.27-3-686|200710070004
674674|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.1|i386-freebsd|5.9.5|freebsd|6.2-release|200710070005
674727|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.1|i686-linux|5.6.2|linux|2.4.27-3-686|200710070039
674728|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.1|i686-linux|5.8.8|linux|2.4.27-3-686|200710070039
675923|pass|200710|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.1|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200710080206
676370|pass|200710|alexchorny@gmail.com|Devel-CheckOS|1.1|i686-linux|5.10.0 patch 32015|linux|2.4.20-8smp|200710071351
676967|pass|200710|saper@cpan.org|Devel-CheckOS|1.1|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200710081119
677780|pass|200710|srezic@cpan.org|Data-Compare|0.17|amd64-freebsd|5.10.0 patch 32039|freebsd|6.2-release|200710082304
677917|pass|200710|srezic@cpan.org|Data-Compare|0.17|amd64-freebsd|5.6.2|freebsd|6.2-release|200710082309
679862|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.1|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200710101039
679863|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.1|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200710101041
679870|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.1|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200710101043
679875|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.1|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200710101045
679883|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.1|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200710101048
679889|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.1|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200710101050
679894|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.1|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200710101052
679895|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.1|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200710101054
679896|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.1|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200710101056
680299|pass|200710|david@valuesofn.com|Data-Compare|0.17|darwin-2level|5.8.8|darwin|8.8.2|200710101043
681705|pass|200710|icerider70@gmail.com ("ICERIDER")|Devel-CheckOS|1.1|i486-linux-gnu|5.8.8|linux|2.6.18.3|200710111716
682486|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200710121126
683801|pass|200710|srezic@cpan.org|XML-Tiny|1.06|amd64-freebsd|5.10.0 patch 32096|freebsd|6.2-release|200710131421
683802|pass|200710|srezic@cpan.org|XML-Tiny|1.06|amd64-freebsd|5.6.2|freebsd|6.2-release|200710131422
683803|pass|200710|srezic@cpan.org|XML-Tiny|1.06|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200710131421
683804|pass|200710|srezic@cpan.org|XML-Tiny|1.06|amd64-freebsd|5.5.5|freebsd|6.2-release|200710131422
685543|pass|200710|srezic@cpan.org|Devel-CheckOS|1.1|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200710141046
685544|pass|200710|srezic@cpan.org|Devel-CheckOS|1.1|amd64-freebsd|5.10.0 patch 32096|freebsd|6.2-release|200710141045
685545|pass|200710|srezic@cpan.org|Devel-CheckOS|1.1|amd64-freebsd|5.6.2|freebsd|6.2-release|200710141046
686176|unknown|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-AssertLib|0.1|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200710141723
686178|unknown|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-AssertLib|0.1|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200710141723
686180|unknown|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-AssertLib|0.1|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200710141724
686183|unknown|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-AssertLib|0.1|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200710141724
686207|unknown|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-AssertLib|0.1|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200710141723
686209|unknown|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-AssertLib|0.1|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200710141723
686210|unknown|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-AssertLib|0.1|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200710141724
686293|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.2|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200710141815
686295|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.2|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200710141815
686300|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.2|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200710141815
686304|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.2|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200710141816
686308|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.2|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200710141816
686373|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.2|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200710141816
686378|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.2|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200710141815
686488|unknown|200710|srezic@cpan.org|Devel-AssertLib|0.1|amd64-freebsd|5.10.0 patch 32096|freebsd|6.2-release|200710141935
686622|unknown|200710|srezic@cpan.org|Devel-AssertLib|0.1|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200710142014
686730|unknown|200710|srezic@cpan.org|Devel-AssertLib|0.1|amd64-freebsd|5.6.2|freebsd|6.2-release|200710142111
686818|pass|200710|srezic@cpan.org|Devel-CheckOS|1.2|amd64-freebsd|5.10.0 patch 32096|freebsd|6.2-release|200710150044
686938|pass|200710|srezic@cpan.org|Devel-CheckOS|1.2|amd64-freebsd|5.6.2|freebsd|6.2-release|200710150113
686952|unknown|200710|srezic@cpan.org|Devel-AssertLib|0.1|amd64-freebsd|5.5.5|freebsd|6.2-release|200710150117
687066|pass|200710|andy@hexten.net|Data-Hexdumper|1.3|i86pc-solaris-64int|5.8.4|solaris|2.11|200710150226
687689|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|alpha-linux-thread-multi|5.8.8|linux|2.6.16.19|200710151304
687691|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|alpha-linux-thread-multi|5.8.8|linux|2.6.16.19|200710151305
687712|pass|200710|xdg@hyperbolic.net (DAGOLDEN)|Devel-CheckOS|1.2|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200710150820
687716|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200710151333
687717|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200710151334
688024|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.2|i686-linux|5.9.5|linux|2.4.27-3-686|200710151516
688025|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.2|i686-linux|5.8.8|linux|2.4.27-3-686|200710151516
688026|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.2|darwin-2level|5.9.5|darwin|8.10.1|200710151508
688030|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.2|i686-linux|5.6.2|linux|2.4.27-3-686|200710151507
688096|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.2|sun4-solaris|5.9.5|solaris|2.9|200710150937
688180|pass|200710|saper@cpan.org|Devel-AssertLib|0.1|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200710151711
688233|pass|200710|imacat@mail.imacat.idv.tw|Devel-AssertLib|0.1|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200710152339
688356|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.2|i386-freebsd|5.9.5|freebsd|6.2-release|200710151316
688419|pass|200710|david@cantrell.org.uk|Devel-CheckOS|1.2|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200710151750
689795|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200710160742
689801|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200710160743
690422|pass|200710|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.2|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200710162118
690817|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|i686-linux-thread-multi-64int|5.10.0 patch 31906|linux|2.6.16.38|200710161710
690828|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|i686-linux-thread-multi-64int|5.10.0 patch 31906|linux|2.6.16.38|200710161711
691859|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200710170645
691860|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200710170645
691861|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200710170645
691862|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200710170646
691863|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200710170646
691864|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200710170646
691865|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200710170646
691866|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200710170647
691867|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200710170647
691868|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200710170647
691869|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200710170648
691870|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200710170649
691871|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200710170649
691872|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200710170650
691873|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200710170650
691874|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200710170651
691875|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200710170651
691876|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200710170652
692213|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200710171009
692214|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200710171010
692388|pass|200710|cpan@pqpq.de|Devel-CheckOS|1.2|i386-freebsd|5.10.0 patch 31806|freebsd|6.1-release-p6|200710171254
692389|pass|200710|cpan@pqpq.de|Devel-AssertLib|0.1|i386-freebsd|5.10.0 patch 31806|freebsd|6.1-release-p6|200710171254
693524|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200710180053
693559|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200710180055
693630|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|OpenBSD.i386-openbsd-thread-multi-64int|5.9.5|openbsd|4.1|200710180016
693641|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|OpenBSD.i386-openbsd-thread-multi-64int|5.9.5|openbsd|4.1|200710180017
693720|pass|200710|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd|5.10.0 patch 31938|freebsd|6.1-release|200710180126
693798|pass|200710|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200710180126
693802|pass|200710|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd|5.6.2|freebsd|6.1-release|200710180127
695204|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200710181755
695210|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200710181758
695352|pass|200710|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|sun4-solaris|5.9.5|solaris|2.9|200710181429
695818|pass|200710|xdg@hyperbolic.net (DAGOLDEN)|Devel-CheckLib|0.2|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200710181841
695827|pass|200710|xdg@hyperbolic.net (DAGOLDEN)|Devel-CheckLib|0.2|i686-linux-64int-ld|5.8.8|linux|2.6.17-10-generic|200710181845
695959|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.2|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200710190300
696149|pass|200710|cpan@pqpq.de|XML-DoubleEncodedEntities|1.0|i386-freebsd|5.10.0 patch 31806|freebsd|6.1-release-p6|200710190702
696435|pass|200710|cpan@pqpq.de|Devel-CheckLib|0.2|i386-freebsd|5.10.0 patch 31806|freebsd|6.1-release-p6|200710190814
696830|pass|200710|saper@cpan.org|Devel-CheckLib|0.2|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200710191054
696970|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200710191005
696972|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200710191005
696974|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200710191006
697028|pass|200710|icerider70@gmail.com ("ICERIDER")|Devel-CheckOS|1.2|i486-linux-gnu|5.8.8|linux|2.6.18.3|200710190612
697605|pass|200710|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200710191626
697844|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200710191859
697845|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200710191900
697846|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200710191900
698349|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200710200033
698350|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200710200035
698351|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200710200037
698796|fail|200710|mr.muskrat@perlgeek.net ("Mr. Muskrat")|Devel-CheckLib|0.2|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200710200031
698896|unknown|200710|mr.muskrat@perlgeek.net ("Mr. Muskrat")|Devel-AssertLib|0.1|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200710200030
698912|pass|200710|mr.muskrat@perlgeek.net ("Mr. Muskrat")|Devel-CheckOS|1.2|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200710200032
699054|pass|200710|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200710201144
699218|pass|200710|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200710200134
699476|pass|200710|david@cantrell.org.uk|Devel-CheckLib|0.2|darwin-2level|5.9.5|darwin|8.10.1|200710201531
699498|pass|200710|david@cantrell.org.uk|Devel-CheckLib|0.2|i686-linux|5.9.5|linux|2.4.27-3-686|200710201535
699502|pass|200710|david@cantrell.org.uk|Devel-CheckLib|0.2|i686-linux|5.8.8|linux|2.4.27-3-686|200710201535
699537|pass|200710|imacat@mail.imacat.idv.tw|Devel-CheckLib|0.2|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200710202240
699565|pass|200710|icerider70@gmail.com ("ICERIDER")|XML-DoubleEncodedEntities|1.0|i486-linux-gnu|5.8.8|linux|2.6.18.3|200710200957
699613|pass|200710|mmusgrove@cpan.org|Devel-CheckOS|1.2|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200710200902
699617|pass|200710|mmusgrove@cpan.org|Devel-CheckLib|0.2|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200710200937
699640|pass|200710|david@cantrell.org.uk|Devel-CheckLib|0.2|i686-linux|5.6.2|linux|2.4.27-3-686|200710201631
699764|pass|200710|david@cantrell.org.uk|Devel-CheckLib|0.2|i386-freebsd|5.9.5|freebsd|6.2-release|200710201454
699784|pass|200710|david@cantrell.org.uk|Devel-CheckLib|0.2|sun4-solaris|5.9.5|solaris|2.9|200710201118
699895|pass|200710|david@cantrell.org.uk|Devel-CheckLib|0.2|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200710201649
700529|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200710202034
700532|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200710202035
700533|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200710202036
700907|pass|200710|mmusgrove@cpan.org|XML-DoubleEncodedEntities|1.0|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200710201315
701122|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200710202334
701123|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200710202335
701125|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200710202336
701127|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200710202336
701130|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200710202337
701131|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200710202337
701133|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200710202338
701134|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200710202339
701136|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200710202339
701604|pass|200710|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200710210601
702100|pass|200710|chris@bingosnet.co.uk|Devel-AssertLib|0.1|i386-pc-solaris2.11|5.8.7|solaris|2.11|200710211441
702101|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|i386-pc-solaris2.11|5.8.7|solaris|2.11|200710211441
702102|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|i386-pc-solaris2.11|5.8.7|solaris|2.11|200710211442
702158|pass|200710|bob@randomness.org.uk ("bob walker")|Devel-CheckLib|0.2|i86pc-solaris|5.8.8|solaris|2.10|200710211502
702598|fail|200710|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Devel-AssertLib|0.1|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200710211728
702757|fail|200710|srezic@cpan.org|Devel-CheckLib|0.2|i386-freebsd|5.10.0 patch 31938|freebsd|6.1-release|200710212020
702758|pass|200710|srezic@cpan.org|Devel-CheckLib|0.2|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200710212020
702759|fail|200710|srezic@cpan.org|Devel-CheckLib|0.2|i386-freebsd|5.6.2|freebsd|6.1-release|200710212021
702855|fail|200710|srezic@cpan.org|Devel-CheckLib|0.2|i386-freebsd|5.5.5|freebsd|6.1-release|200710212111
703302|pass|200710|bob@randomness.org.uk|Devel-CheckOS|1.2|i386-openbsd|5.8.8|openbsd|4.2|200710212258
703306|pass|200710|bob@randomness.org.uk|Devel-CheckLib|0.2|i386-openbsd|5.8.8|openbsd|4.2|200710212259
704075|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200710221130
704614|pass|200710|Jost.Krieger@ruhr-uni-bochum.de|Acme-Pony|1.1.2|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200710221350
705556|pass|200710|andy@hexten.net|Data-Compare|0.17|i486-linux-gnu|5.8.8|linux|2.6.15.7|200710222023
706080|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|i86pc-solaris-thread-multi-64int|5.9.5|solaris|2.10|200710222325
707010|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.2|i686-linux-thread-multi|5.10.0|linux|2.6.22-1-k7|200710230655
707013|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.06|i686-linux-thread-multi|5.10.0|linux|2.6.22-1-k7|200710230655
707652|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-thread-multi|5.10.0|linux|2.6.22-1-k7|200710230718
708218|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-thread-multi|5.10.0|linux|2.6.22-1-k7|200710230351
708858|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200710231428
708917|pass|200710|Jost.Krieger@ruhr-uni-bochum.de|Devel-CheckOS|1.2|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200710231407
708922|pass|200710|Jost.Krieger@ruhr-uni-bochum.de|File-Find-Rule-Permissions|1.3|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200710231407
709511|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200710231759
710694|fail|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.2|200710241103
710738|pass|200710|chris@bingosnet.co.uk|Devel-CheckOS|1.2|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.2|200710241111
711237|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.07|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200710241518
711238|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.07|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200710241519
711240|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.07|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200710241519
711243|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.07|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200710241520
711245|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.07|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200710241519
711247|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.07|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200710241519
711778|pass|200710|srezic@cpan.org|XML-Tiny|1.07|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200710242156
711936|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200710242158
711937|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200710242158
711939|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200710242158
711940|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200710242159
711941|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200710242159
711942|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200710242200
711943|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200710242200
711944|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200710242200
711945|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200710242201
712027|pass|200710|srezic@cpan.org|XML-Tiny|1.07|i386-freebsd|5.10.0 patch 32171|freebsd|6.1-release|200710242237
712321|pass|200710|srezic@cpan.org|XML-Tiny|1.07|i386-freebsd|5.5.5|freebsd|6.1-release|200710250037
712334|pass|200710|srezic@cpan.org|XML-Tiny|1.07|i386-freebsd|5.6.2|freebsd|6.1-release|200710242345
712381|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-thread-multi-64int|5.10.0|linux|2.6.22-1-k7|200710250128
713464|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.2|i686-linux-thread-multi-64int|5.10.0|linux|2.6.22-1-k7|200710250735
713625|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.07|i686-linux-thread-multi-64int|5.10.0|linux|2.6.22-1-k7|200710250848
713802|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int|5.10.0|linux|2.6.22-1-k7|200710250829
714082|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200710251121
714083|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200710251121
714227|pass|200710|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|XML-Tiny|1.07|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200710251225
714305|pass|200710|david@cantrell.org.uk|XML-Tiny|1.07|darwin-2level|5.9.5|darwin|8.10.1|200710251427
714319|pass|200710|david@cantrell.org.uk|XML-Tiny|1.07|i686-linux|5.6.2|linux|2.4.27-3-686|200710251428
714330|pass|200710|david@cantrell.org.uk|XML-Tiny|1.07|i686-linux|5.9.5|linux|2.4.27-3-686|200710251430
714400|pass|200710|david@cantrell.org.uk|XML-Tiny|1.07|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200710251353
714407|pass|200710|david@cantrell.org.uk|XML-Tiny|1.07|sun4-solaris|5.9.5|solaris|2.9|200710250851
714443|pass|200710|david@cantrell.org.uk|XML-Tiny|1.07|i686-linux|5.8.8|linux|2.4.27-3-686|200710251513
714470|pass|200710|david@cantrell.org.uk|XML-Tiny|1.07|i386-freebsd|5.9.5|freebsd|6.2-release|200710251113
714489|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.08|i686-linux-thread-multi-64int|5.10.0|linux|2.6.22-1-k7|200710251631
714490|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.08|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200710251633
714492|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.08|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200710251633
714493|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.08|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200710251634
714498|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.08|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200710251632
714500|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.08|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200710251632
714524|pass|200710|andy@hexten.net|Data-Hexdumper|1.3|powerpc-linux-gnu|5.8.8|linux|2.6.17-10-powerpc64-smp|200710251645
714870|pass|200710|imacat@mail.imacat.idv.tw|XML-Tiny|1.07|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200710260119
714943|pass|200710|srezic@cpan.org|XML-Tiny|1.08|amd64-freebsd|5.10.0 patch 32096|freebsd|6.2-release|200710252029
715122|pass|200710|srezic@cpan.org|XML-Tiny|1.08|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200710252155
715413|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200710252223
715420|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200710252224
715423|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200710252225
715425|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200710252225
715427|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200710252226
715428|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200710252227
715432|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200710252227
715434|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200710252228
715435|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200710252229
715463|pass|200710|srezic@cpan.org|XML-Tiny|1.08|amd64-freebsd|5.6.2|freebsd|6.2-release|200710252317
715653|pass|200710|srezic@cpan.org|XML-Tiny|1.08|amd64-freebsd|5.5.5|freebsd|6.2-release|200710260039
715764|pass|200710|srezic@cpan.org|Data-Compare|0.17|amd64-freebsd|5.10.0 patch 32096|freebsd|6.2-release|200710260041
715840|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.08|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200710260247
715852|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.2|OpenBSD.i386-openbsd-thread-multi-64int|5.9.5|openbsd|4.1|200710260233
715853|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|OpenBSD.i386-openbsd-thread-multi-64int|5.9.5|openbsd|4.1|200710260233
716103|pass|200710|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|XML-Tiny|1.08|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200710260752
716290|pass|200710|Jost.Krieger@ruhr-uni-bochum.de|Acme-Pony|1.1.2|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200710260935
716541|pass|200710|Jost.Krieger@ruhr-uni-bochum.de|Acme-Pony|1.1.2|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200710261317
716957|pass|200710|david@cantrell.org.uk|XML-Tiny|1.08|darwin-2level|5.9.5|darwin|8.10.1|200710261635
716962|pass|200710|david@cantrell.org.uk|XML-Tiny|1.08|i686-linux|5.9.5|linux|2.4.27-3-686|200710261636
716977|pass|200710|david@cantrell.org.uk|XML-Tiny|1.08|i686-linux|5.8.8|linux|2.4.27-3-686|200710261638
716993|pass|200710|david@cantrell.org.uk|XML-Tiny|1.08|i386-freebsd|5.9.5|freebsd|6.2-release|200710261351
717087|pass|200710|david@cantrell.org.uk|XML-Tiny|1.08|i686-linux|5.6.2|linux|2.4.27-3-686|200710261725
717191|pass|200710|imacat@mail.imacat.idv.tw|XML-Tiny|1.08|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200710270055
717424|pass|200710|david@cantrell.org.uk|XML-Tiny|1.08|sun4-solaris|5.9.5|solaris|2.9|200710261445
717447|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200710262045
717448|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200710262046
717474|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200710262133
717475|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200710262134
717581|pass|200710|icerider70@gmail.com ("ICERIDER")|XML-Tiny|1.07|i486-linux-gnu|5.8.8|linux|2.6.18.3|200710261644
718615|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200710271354
718617|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200710271354
719002|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200710270712
719003|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200710270712
719671|pass|200710|mmusgrove@cpan.org|XML-Tiny|1.08|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200710280102
719929|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200710280342
719930|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200710280342
721013|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i386-netbsd-thread-multi-64int|5.10.0 patch 32199|netbsd|3.1|200710281256
721014|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i386-netbsd-thread-multi-64int|5.10.0 patch 32199|netbsd|3.1|200710281256
721312|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i386-pc-solaris2.11|5.8.7|solaris|2.11|200710281527
721313|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i386-pc-solaris2.11|5.8.7|solaris|2.11|200710281528
722085|pass|200710|chris@bingosnet.co.uk|Data-Compare|0.17|i386-netbsd-thread-multi-64int|5.10.0 patch 32199|netbsd|3.1|200710281954
722872|pass|200710|mmusgrove@cpan.org|Data-Compare|0.17|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200710290333
723241|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200710291142
723243|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200710291142
723796|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.09|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200710291858
723799|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.09|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200710291859
723801|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.09|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200710291900
723806|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.09|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200710291901
723808|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.09|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200710291902
723809|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.09|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200710291903
724183|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i86pc-solaris-thread-multi-64int|5.9.5|solaris|2.10|200710292135
724187|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i86pc-solaris-thread-multi-64int|5.9.5|solaris|2.10|200710292136
724259|pass|200710|srezic@cpan.org|XML-Tiny|1.09|amd64-freebsd|5.10.0 patch 32096|freebsd|6.2-release|200710292255
724288|pass|200710|srezic@cpan.org|XML-Tiny|1.09|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200710292317
724381|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.10|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200710292352
724384|pass|200710|srezic@cpan.org|XML-Tiny|1.09|amd64-freebsd|5.6.2|freebsd|6.2-release|200710292347
724386|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.10|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200710292353
724392|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.10|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200710292353
724403|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.10|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200710292353
724404|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.10|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200710292353
724405|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.10|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200710292354
724510|pass|200710|srezic@cpan.org|XML-Tiny|1.09|amd64-freebsd|5.5.5|freebsd|6.2-release|200710300021
724537|pass|200710|srezic@cpan.org|XML-Tiny|1.10|amd64-freebsd|5.5.5|freebsd|6.2-release|200710300027
724568|pass|200710|srezic@cpan.org|XML-Tiny|1.10|amd64-freebsd|5.6.2|freebsd|6.2-release|200710300026
724572|pass|200710|srezic@cpan.org|XML-Tiny|1.10|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200710300026
724585|pass|200710|srezic@cpan.org|XML-Tiny|1.10|amd64-freebsd|5.10.0 patch 32096|freebsd|6.2-release|200710300025
724754|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.09|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200710300127
724755|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.09|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200710300128
724756|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.09|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200710300128
724757|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.09|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200710300128
724758|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.09|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200710300129
724759|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.09|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200710300129
724761|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.09|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200710300129
724762|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.09|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200710300130
724767|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.09|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200710300130
725114|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.10|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200710300556
725116|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.10|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200710300556
725117|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.10|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200710300557
725118|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.10|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200710300557
725119|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.10|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200710300557
725120|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.10|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200710300558
725123|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.10|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200710300558
725125|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.10|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200710300558
725127|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.10|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200710300559
725167|pass|200710|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|XML-Tiny|1.10|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200710300816
725313|pass|200710|cpan@0x64.nl|Bryar|3.0|x86_64-linux-gnu|5.8.8|linux|2.6.17-2-vserver-amd64|200710300124
725387|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200710301051
725389|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.09|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200710301051
725391|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.10|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200710301052
725392|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200710301051
725433|pass|200710|david@cantrell.org.uk|XML-Tiny|1.09|i686-linux|5.8.8|linux|2.4.27-3-686|200710301118
725438|pass|200710|david@cantrell.org.uk|XML-Tiny|1.09|darwin-2level|5.9.5|darwin|8.10.1|200710301119
725448|pass|200710|david@cantrell.org.uk|XML-Tiny|1.09|i686-linux|5.9.5|linux|2.4.27-3-686|200710301124
725561|pass|200710|david@cantrell.org.uk|XML-Tiny|1.10|i686-linux|5.9.5|linux|2.4.27-3-686|200710301228
725576|pass|200710|david@cantrell.org.uk|XML-Tiny|1.09|i686-linux|5.6.2|linux|2.4.27-3-686|200710301230
725595|pass|200710|david@cantrell.org.uk|XML-Tiny|1.10|i686-linux|5.8.8|linux|2.4.27-3-686|200710301232
725596|pass|200710|david@cantrell.org.uk|XML-Tiny|1.10|darwin-2level|5.9.5|darwin|8.10.1|200710301232
725706|pass|200710|david@cantrell.org.uk|XML-Tiny|1.09|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200710301302
725747|pass|200710|david@cantrell.org.uk|XML-Tiny|1.10|i686-linux|5.6.2|linux|2.4.27-3-686|200710301249
725750|pass|200710|david@cantrell.org.uk|XML-Tiny|1.09|sun4-solaris|5.9.5|solaris|2.9|200710300813
725806|pass|200710|david@cantrell.org.uk|XML-Tiny|1.09|i386-freebsd|5.9.5|freebsd|6.2-release|200710301029
725849|pass|200710|david@cantrell.org.uk|XML-Tiny|1.10|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200710301411
725881|pass|200710|david@cantrell.org.uk|XML-Tiny|1.10|sun4-solaris|5.9.5|solaris|2.9|200710300937
725967|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.3|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200710301630
725968|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.3|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200710301631
725969|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.3|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200710301631
725970|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.3|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200710301632
725978|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.3|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200710301631
725980|pass|200710|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.3|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200710301631
726169|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.07|i686-linux-thread-multi-64int-ld|5.10.0 patch 32202|linux|2.6.16.38|200710301718
726170|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|i686-linux-thread-multi-64int-ld|5.10.0 patch 32202|linux|2.6.16.38|200710301718
726171|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.09|i686-linux-thread-multi-64int-ld|5.10.0 patch 32202|linux|2.6.16.38|200710301718
726172|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.10|i686-linux-thread-multi-64int-ld|5.10.0 patch 32202|linux|2.6.16.38|200710301719
726508|pass|200710|srezic@cpan.org|Devel-CheckLib|0.3|amd64-freebsd|5.10.0 patch 32200|freebsd|6.2-release|200710302036
726675|pass|200710|srezic@cpan.org|XML-Tiny|1.10|amd64-freebsd|5.10.0 patch 32200|freebsd|6.2-release|200710302055
726727|pass|200710|srezic@cpan.org|Devel-CheckLib|0.3|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200710302109
726791|pass|200710|srezic@cpan.org|Devel-CheckLib|0.3|amd64-freebsd|5.6.2|freebsd|6.2-release|200710302200
726828|pass|200710|david@cantrell.org.uk|XML-Tiny|1.10|i386-freebsd|5.9.5|freebsd|6.2-release|200710301059
726836|pass|200710|srezic@cpan.org|Devel-CheckLib|0.3|amd64-freebsd|5.5.5|freebsd|6.2-release|200710302227
726852|fail|200710|srezic@cpan.org|XML-Tiny|1.10|amd64-freebsd|5.5.5|freebsd|6.2-release|200710302232
727114|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.3|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200710310013
727115|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.3|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200710310014
727116|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.3|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200710310014
727117|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.3|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200710310014
727118|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.3|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200710310015
727120|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.3|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200710310015
727121|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.3|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200710310015
727122|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.3|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200710310016
727123|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.3|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200710310016
727192|pass|200710|chris@bingosnet.co.uk|Data-Compare|0.17|i686-linux-thread-multi-64int-ld|5.10.0 patch 32202|linux|2.6.16.38|200710310050
727341|pass|200710|mmusgrove@cpan.org|XML-Tiny|1.10|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200710310337
727446|pass|200710|mmusgrove@cpan.org|XML-Tiny|1.10|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200710310008
727554|pass|200710|mmusgrove@cpan.org|Data-Compare|0.17|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200710310116
727762|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.3|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200710311018
727763|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200710311019
727764|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.09|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200710311019
727765|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.10|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200710311020
727781|pass|200710|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Devel-CheckLib|0.3|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200710311052
727820|pass|200710|Jost.Krieger@ruhr-uni-bochum.de|Acme-Pony|1.1.2|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200710311203
728112|pass|200711|imacat@mail.imacat.idv.tw|XML-Tiny|1.09|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200711010038
728139|pass|200711|imacat@mail.imacat.idv.tw|XML-Tiny|1.10|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200711010058
728217|pass|200711|imacat@mail.imacat.idv.tw|Devel-CheckLib|0.3|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200711010151
728275|pass|200710|saper@cpan.org|XML-Tiny|1.08|i386-linux|5.8.5|linux|2.6.3-25mdk-i686-up-4gb|200710311957
728443|pass|200710|david@cantrell.org.uk|Devel-CheckLib|0.3|i686-linux|5.9.5|linux|2.4.27-3-686|200710312032
728512|pass|200710|chris@bingosnet.co.uk|Devel-CheckLib|0.3|OpenBSD.i386-openbsd-thread-multi-64int|5.9.5|openbsd|4.1|200710312101
728514|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.08|OpenBSD.i386-openbsd-thread-multi-64int|5.9.5|openbsd|4.1|200710312101
728517|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.09|OpenBSD.i386-openbsd-thread-multi-64int|5.9.5|openbsd|4.1|200710312102
728518|pass|200710|chris@bingosnet.co.uk|XML-Tiny|1.10|OpenBSD.i386-openbsd-thread-multi-64int|5.9.5|openbsd|4.1|200710312102
728552|pass|200710|david@cantrell.org.uk|Devel-CheckLib|0.3|i686-linux|5.6.2|linux|2.4.27-3-686|200710312115
728555|pass|200710|david@cantrell.org.uk|Devel-CheckLib|0.3|darwin-2level|5.9.5|darwin|8.10.1|200710312116
728567|pass|200710|david@cantrell.org.uk|Devel-CheckLib|0.3|i686-linux|5.8.8|linux|2.4.27-3-686|200710312117
728568|pass|200710|david@cantrell.org.uk|Devel-CheckLib|0.3|i386-freebsd|5.9.5|freebsd|6.2-release|200710312115
728604|pass|200710|david@cantrell.org.uk|Devel-CheckLib|0.3|sun4-solaris|5.9.5|solaris|2.9|200710311636
728605|pass|200710|david@cantrell.org.uk|Devel-CheckLib|0.3|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200710312136
728884|pass|200711|srezic@cpan.org|Data-Compare|0.17|amd64-freebsd|5.10.0 patch 32200|freebsd|6.2-release|200711010041
729616|pass|200711|icerider70@gmail.com ("ICERIDER")|XML-Tiny|1.10|i486-linux-gnu|5.8.8|linux|2.6.18.3|200711010424
730066|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200711010550
730773|pass|200711|icerider70@gmail.com ("ICERIDER")|Devel-CheckLib|0.3|i486-linux-gnu|5.8.8|linux|2.6.18.3|200711010651
731697|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200711011807
731701|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200711011807
731719|pass|200711|chris@bingosnet.co.uk|Devel-CheckLib|0.3|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200711011714
731720|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.08|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200711011714
731721|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.09|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200711011715
731722|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.10|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200711011715
731730|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.9.4|linux|2.6.17-2-k7|200711011808
731782|fail|200711|david@cantrell.org.uk|Palm-SMS|0.02|darwin-2level|5.9.5|darwin|8.10.1|200711011729
732214|pass|200711|srezic@cpan.org|XML-Tiny|1.11|amd64-freebsd|5.6.2|freebsd|6.2-release|200711012140
732229|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200711012145
732356|pass|200711|srezic@cpan.org|XML-Tiny|1.11|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200711012145
732438|pass|200711|srezic@cpan.org|XML-Tiny|1.11|amd64-freebsd|5.5.5|freebsd|6.2-release|200711012223
732646|pass|200711|srezic@cpan.org|XML-Tiny|1.11|amd64-freebsd|5.10.0 patch 32200|freebsd|6.2-release|200711012233
732819|pass|200711|chris@bingosnet.co.uk|Devel-CheckLib|0.3|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0_rc3|200711012325
732820|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.08|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0_rc3|200711012325
732821|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.09|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0_rc3|200711012326
732822|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.10|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0_rc3|200711012326
733089|pass|200711|mmusgrove@cpan.org|Devel-CheckLib|0.3|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200711020241
733397|pass|200711|chris@bingosnet.co.uk|Data-Compare|0.17|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0_rc3|200711020511
733409|pass|200711|mmusgrove@cpan.org|XML-Tiny|1.11|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200711020026
733855|pass|200711|david@cantrell.org.uk|XML-Tiny|1.11|i686-linux|5.6.2|linux|2.4.27-3-686|200711021208
733946|pass|200711|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|XML-Tiny|1.11|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200711021252
734213|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200711021542
734243|pass|200711|david@cantrell.org.uk|XML-Tiny|1.11|darwin-2level|5.9.5|darwin|8.10.1|200711021558
734247|pass|200711|david@cantrell.org.uk|XML-Tiny|1.11|i686-linux|5.9.5|linux|2.4.27-3-686|200711021559
734259|pass|200711|david@cantrell.org.uk|XML-Tiny|1.11|i386-freebsd|5.9.5|freebsd|6.2-release|200711021552
734350|pass|200711|chris@bingosnet.co.uk|Devel-CheckLib|0.3|MSWin32-x86-multi-thread|5.10.0 patchlevel 32202|MSWin32|5.2|200711021633
734384|pass|200711|david@cantrell.org.uk|XML-Tiny|1.11|i686-linux|5.8.8|linux|2.4.27-3-686|200711021559
734440|pass|200711|chris@bingosnet.co.uk|Devel-CheckLib|0.3|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200711021640
734442|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.09|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200711021640
734443|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.10|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200711021641
734445|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200711021641
734649|pass|200711|andy@hexten.net|Devel-CheckOS|1.2|i486-linux-gnu|5.8.8|linux|2.6.15.7|200711021844
734964|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200711022109
734969|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200711022109
734970|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200711022110
734983|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200711022111
734990|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200711022111
734991|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200711022111
734996|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200711022112
735000|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200711022110
735001|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200711022110
735073|pass|200711|david@cantrell.org.uk|XML-Tiny|1.11|sun4-solaris|5.9.5|solaris|2.9|200711021627
735204|pass|200711|cpan@0x64.nl|XML-Tiny|1.11|x86_64-linux-gnu|5.8.8|linux|2.6.17-2-vserver-amd64|200711022008
735291|pass|200711|david@cantrell.org.uk|XML-Tiny|1.11|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200711022208
735637|pass|200711|david@cantrell.org.uk|Data-Compare|0.02|i386-freebsd|5.9.5|freebsd|6.2-release|200711030027
735719|pass|200711|chris@bingosnet.co.uk|Devel-CheckLib|0.3|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200711030139
735720|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.09|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200711030140
735721|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.10|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200711030140
735724|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200711030141
735927|pass|200711|mmusgrove@cpan.org|Devel-CheckLib|0.3|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200711030313
736341|pass|200711|srezic@cpan.org|CPAN-FindDependencies|1.02|amd64-freebsd|5.10.0 patch 32200|freebsd|6.2-release|200711031033
736770|pass|200711|alexchorny@gmail.com|Devel-CheckLib|0.3|i686-linux|5.10.0 patch 32015|linux|2.4.20-8smp|200711030412
736772|pass|200711|alexchorny@gmail.com|XML-Tiny|1.11|i686-linux|5.10.0 patch 32015|linux|2.4.20-8smp|200711030412
736795|pass|200711|mmusgrove@cpan.org|XML-Tiny|1.11|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200711030733
737084|pass|200711|chris@bingosnet.co.uk|Devel-CheckLib|0.3|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200711030337
737086|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.09|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200711030337
737087|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.10|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200711030338
737089|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200711030338
737280|pass|200711|imacat@mail.imacat.idv.tw|XML-Tiny|1.11|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.20.12|200711032258
738907|pass|200711|chris@bingosnet.co.uk|Devel-CheckLib|0.3|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200711040155
738908|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.09|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200711040155
738909|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.10|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200711040155
738910|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200711040156
739569|pass|200711|chris@bingosnet.co.uk|Devel-CheckLib|0.3|i386-pc-solaris2.11|5.8.7|solaris|2.11|200711041250
739570|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.09|i386-pc-solaris2.11|5.8.7|solaris|2.11|200711041250
739571|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.10|i386-pc-solaris2.11|5.8.7|solaris|2.11|200711041251
739572|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-pc-solaris2.11|5.8.7|solaris|2.11|200711041251
740713|pass|200711|chris@bingosnet.co.uk|Devel-CheckLib|0.3|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200711050044
740714|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.09|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200711050044
740715|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.10|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200711050044
740716|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200711050045
741522|pass|200711|cpan@pqpq.de|XML-Tiny|1.11|i386-freebsd|5.10.0 patch 31806|freebsd|6.1-release-p6|200711050851
741523|pass|200711|cpan@pqpq.de|XML-Tiny|1.10|i386-freebsd|5.10.0 patch 31806|freebsd|6.1-release-p6|200711050851
741524|pass|200711|cpan@pqpq.de|XML-Tiny|1.09|i386-freebsd|5.10.0 patch 31806|freebsd|6.1-release-p6|200711050851
741525|pass|200711|cpan@pqpq.de|Devel-CheckLib|0.3|i386-freebsd|5.10.0 patch 31806|freebsd|6.1-release-p6|200711050851
741809|pass|200711|Jost.Krieger@ruhr-uni-bochum.de|Acme-Pony|1.1.2|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200711051058
742656|pass|200711|andy@hexten.net|Data-Hexdumper|1.3|darwin-2level|5.9.5|darwin|9.0.0|200711051831
743026|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.10|i86pc-solaris-thread-multi-64int|5.9.5|solaris|2.10|200711052057
743028|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|i86pc-solaris-thread-multi-64int|5.9.5|solaris|2.10|200711052058
743035|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.09|i86pc-solaris-thread-multi-64int|5.9.5|solaris|2.10|200711052057
743052|pass|200711|chris@bingosnet.co.uk|Devel-CheckLib|0.3|i86pc-solaris-thread-multi-64int|5.9.5|solaris|2.10|200711052056
743914|pass|200711|mmusgrove@cpan.org ("Matthew Musgrove")|Data-Compare|0.17|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200711060535
745365|pass|200711|chris@bingosnet.co.uk|Data-Compare|0.17|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200711061836
746713|pass|200711|chris@bingosnet.co.uk|Devel-CheckLib|0.3|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200711070246
746714|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200711070246
747954|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.10.0 patch 32202|linux|2.6.16.38|200711071243
748739|pass|200711|andy@hexten.net|Data-Compare|0.17|i486-linux-gnu|5.8.8|linux|2.6.15.7|200711071835
749141|pass|200711|bob@randomness.org.uk ("bob walker")|Devel-CheckOS|1.2|i86pc-solaris|5.8.8|solaris|2.10|200711072025
749292|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.3|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200711072209
749304|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.3|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200711072211
749306|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.3|i686-linux-64int|5.9.4|linux|2.6.17-2-k7|200711072212
749311|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.3|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200711072214
749312|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.3|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200711072215
749316|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.3|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200711072216
749324|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.3|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200711072218
749330|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.3|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200711072219
749605|pass|200711|bob@randomness.org.uk ("bob walker")|CPAN-FindDependencies|1.02|i86pc-solaris|5.8.8|solaris|2.10|200711072344
749902|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.41|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200711080739
749905|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.41|i686-linux-64int|5.9.4|linux|2.6.17-2-k7|200711080739
749907|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.41|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200711080740
749909|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.41|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200711080741
749910|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.41|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200711080741
749912|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.41|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200711080742
749914|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.41|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200711080743
749917|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200711080643
749954|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.41|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200711080738
750004|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|1.1|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200711080815
750021|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|1.1|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200711080824
750025|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|1.1|i686-linux-64int|5.9.4|linux|2.6.17-2-k7|200711080830
750026|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|1.1|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200711080834
750042|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|1.1|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200711080840
750060|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|1.1|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200711080848
750106|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|1.1|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200711080859
750197|pass|200711|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|CPAN-FindDependencies|1.1|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200711080817
750202|pass|200711|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Devel-CheckOS|1.41|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200711080818
750450|pass|200711|bob@randomness.org.uk ("bob walker")|CPAN-FindDependencies|1.1|i86pc-solaris|5.8.8|solaris|2.10|200711081027
751024|pass|200711|Jost.Krieger@ruhr-uni-bochum.de|Acme-Pony|1.1.2|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200711081552
751025|pass|200711|andy@hexten.net|Data-Hexdumper|1.3|darwin-thread-multi-2level|5.10.0 patch 32245|darwin|9.0.0|200711081554
751606|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.3|darwin-2level|5.9.5|darwin|8.10.1|200711081833
751640|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.3|i686-linux|5.9.5|linux|2.4.27-3-686|200711081842
751643|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.3|i686-linux|5.8.8|linux|2.4.27-3-686|200711081843
751647|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.3|i686-linux|5.6.2|linux|2.4.27-3-686|200711081843
751663|pass|200711|srezic@cpan.org|CPAN-FindDependencies|1.1|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200711081939
751724|pass|200711|srezic@cpan.org|Devel-CheckOS|1.41|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200711081941
751804|pass|200711|srezic@cpan.org|Devel-CheckOS|1.41|amd64-freebsd|5.10.0 patch 32200|freebsd|6.2-release|200711082040
751901|pass|200711|srezic@cpan.org|CPAN-FindDependencies|1.1|amd64-freebsd|5.10.0 patch 32200|freebsd|6.2-release|200711082039
752045|pass|200711|srezic@cpan.org|Devel-CheckOS|1.41|amd64-freebsd|5.6.2|freebsd|6.2-release|200711082125
752282|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.4|darwin-2level|5.9.5|darwin|8.10.1|200711090012
752289|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.41|darwin-2level|5.9.5|darwin|8.10.1|200711090014
752314|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|OpenBSD.i386-openbsd-thread-multi-64int|5.9.5|openbsd|4.1|200711090031
752315|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|OpenBSD.i386-openbsd-thread-multi-64int|5.9.5|openbsd|4.1|200711090032
752317|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|OpenBSD.i386-openbsd-thread-multi-64int|5.9.5|openbsd|4.1|200711090033
752318|pass|200711|chris@bingosnet.co.uk|XML-Tiny|1.11|OpenBSD.i386-openbsd-thread-multi-64int|5.9.5|openbsd|4.1|200711090034
752322|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.4|i686-linux|5.8.8|linux|2.4.27-3-686|200711090038
752326|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.4|i686-linux|5.9.5|linux|2.4.27-3-686|200711090039
752338|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.41|i686-linux|5.8.8|linux|2.4.27-3-686|200711090045
752342|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.41|i686-linux|5.9.5|linux|2.4.27-3-686|200711090046
752367|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.4|i686-linux|5.6.2|linux|2.4.27-3-686|200711090058
752369|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.41|i686-linux|5.6.2|linux|2.4.27-3-686|200711090100
752699|fail|200711|mmusgrove@cpan.org ("Matthew Musgrove")|Devel-CheckOS|1.3|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200711090104
752721|fail|200711|mmusgrove@cpan.org ("Matthew Musgrove")|Devel-CheckOS|1.4|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200711090112
752722|fail|200711|mmusgrove@cpan.org ("Matthew Musgrove")|Devel-CheckOS|1.41|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200711090113
752727|pass|200711|mmusgrove@cpan.org ("Matthew Musgrove")|CPAN-FindDependencies|1.1|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200711090120
753143|fail|200711|mmusgrove@cpan.org|Devel-CheckOS|1.3|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200711090606
753276|na|200711|mmusgrove@cpan.org|Devel-CheckOS|1.4|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200711090711
753277|fail|200711|mmusgrove@cpan.org|Devel-CheckOS|1.41|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200711090713
753292|pass|200711|mmusgrove@cpan.org|CPAN-FindDependencies|1.1|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200711090726
754375|pass|200711|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.3|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200711100554
754643|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200711100042
754646|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200711100044
754648|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200711100046
754981|pass|200711|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.41|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200711101304
754998|pass|200711|imacat@mail.imacat.idv.tw|CPAN-FindDependencies|1.1|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200711101311
755487|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.3|sun4-solaris|5.9.5|solaris|2.9|200711100348
755495|fail|200711|david@cantrell.org.uk|Devel-CheckOS|1.3|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200711100951
755563|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0_rc3|200711101013
755568|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0_rc3|200711101014
755573|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0_rc3|200711101015
758030|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|darwin-thread-multi-2level|5.8.1|darwin|8.10.0|200711110901
758033|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|darwin-thread-multi-2level|5.8.2|darwin|8.10.0|200711110907
758036|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|darwin-thread-multi-2level|5.8.3|darwin|8.10.0|200711110912
758039|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|darwin-thread-multi-2level|5.8.4|darwin|8.10.0|200711110918
758050|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|darwin-thread-multi-2level|5.8.5|darwin|8.10.0|200711110923
758056|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|darwin-thread-multi-2level|5.8.6|darwin|8.10.0|200711110928
758060|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|darwin-thread-multi-2level|5.8.7|darwin|8.10.0|200711110932
758061|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|darwin-thread-multi-2level|5.8.8|darwin|8.10.0|200711110937
758062|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|darwin-thread-multi|5.6.2|darwin|8.10.0|200711110938
758063|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|darwin-thread-multi-2level|5.8.1|darwin|8.10.0|200711110940
758065|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|darwin-thread-multi-2level|5.8.2|darwin|8.10.0|200711110941
758066|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|darwin-thread-multi-2level|5.8.3|darwin|8.10.0|200711110942
758068|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|darwin-thread-multi-2level|5.8.4|darwin|8.10.0|200711110943
758070|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|darwin-thread-multi-2level|5.8.5|darwin|8.10.0|200711110944
758072|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|darwin-thread-multi-2level|5.8.6|darwin|8.10.0|200711110946
758073|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|darwin-thread-multi-2level|5.8.7|darwin|8.10.0|200711110947
758074|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|darwin-thread-multi-2level|5.8.8|darwin|8.10.0|200711110948
758077|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|darwin-thread-multi|5.6.2|darwin|8.10.0|200711110952
758078|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|darwin-thread-multi-2level|5.8.1|darwin|8.10.0|200711110953
758081|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|darwin-thread-multi-2level|5.8.2|darwin|8.10.0|200711110955
758082|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|darwin-thread-multi-2level|5.8.3|darwin|8.10.0|200711110956
758083|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|darwin-thread-multi-2level|5.8.4|darwin|8.10.0|200711110958
758084|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|darwin-thread-multi-2level|5.8.5|darwin|8.10.0|200711110959
758085|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|darwin-thread-multi-2level|5.8.6|darwin|8.10.0|200711111000
758087|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|darwin-thread-multi-2level|5.8.7|darwin|8.10.0|200711111001
758088|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|darwin-thread-multi-2level|5.8.8|darwin|8.10.0|200711111002
758230|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200711111148
758232|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200711111149
758233|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200711111149
758235|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200711111151
758239|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200711111152
758242|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200711111153
758243|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200711111154
758246|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200711111155
758247|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200711111156
758365|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200711111346
758366|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200711111347
758367|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200711111348
758725|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200711111840
758726|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200711111840
758727|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200711111841
758728|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200711111841
758729|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200711111842
758730|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200711111842
758731|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200711111843
758732|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200711111844
758735|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200711111844
758765|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200711111859
758779|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200711111903
758783|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200711111907
758798|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200711111911
758810|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200711111914
758828|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200711111918
758842|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200711111922
758852|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200711111925
758997|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-64int|5.8.8 patch 32273|linux|2.6.22-1-k7|200711112053
759211|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 32273|linux|2.6.22-1-k7|200711112249
759785|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.41|sun4-solaris|5.9.5|solaris|2.9|200711120107
759793|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.41|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200711120012
759916|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 32273|linux|2.6.22-1-k7|200711120108
760130|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.3|i686-linux-64int|5.8.8 patch 32273|linux|2.6.22-1-k7|200711120257
761118|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200711120944
761122|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200711120944
761123|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|i686-linux-thread-multi-64int|5.8.8|linux|2.6.21.5-smp|200711120945
761727|pass|200711|Jost.Krieger@ruhr-uni-bochum.de|Devel-CheckOS|1.41|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200711121152
761946|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200711121258
761955|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200711121300
761962|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200711121301
762712|pass|200711|chris@bingosnet.co.uk|Data-Compare|0.17|i386-netbsd-thread-multi-64int|5.10.0 patch 32280|netbsd|3.1|200711121713
764083|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.41|i386-freebsd|5.9.5|freebsd|6.2-release|200711122250
764588|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200711130237
764591|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|i686-linux-thread-multi-64int|5.9.5|linux|2.6.21.5-smp|200711130237
764854|pass|200711|icerider70@gmail.com ("ICERIDER")|Data-Compare|0.17|i486-linux-gnu|5.8.8|linux|2.6.22-3-amd64|200711130039
765192|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|i386-pc-solaris2.11|5.8.7|solaris|2.11|200711131007
765193|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|i386-pc-solaris2.11|5.8.7|solaris|2.11|200711131008
765194|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|i386-pc-solaris2.11|5.8.7|solaris|2.11|200711131009
765970|pass|200711|andy@hexten.net|Data-Hexdumper|1.3|i386-linux|5.8.5|linux|2.6.9-42.elsmp|200711131828
766337|pass|200711|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|amd64-freebsd|5.10.0 patch 32200|freebsd|6.2-release|200711132151
766737|pass|200711|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200711140001
766767|pass|200711|icerider70@gmail.com ("ICERIDER")|XML-DoubleEncodedEntities|1.0|i486-linux-gnu|5.8.8|linux|2.6.22-3-amd64|200711140002
767072|pass|200711|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|amd64-freebsd|5.6.2|freebsd|6.2-release|200711140233
767176|pass|200711|bob@randomness.org.uk ("bob walker")|XML-Tiny|1.11|i86pc-solaris|5.8.8|solaris|2.10|200711140157
768950|pass|200711|andy@hexten.net|Data-Hexdumper|1.3|i486-linux-gnu|5.8.8|linux|2.6.22-3-amd64|200711142150
769960|pass|200711|chris@bingosnet.co.uk|CPAN-FindDependencies|1.1|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200711150217
769974|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.3|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200711150218
770001|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.41|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200711150224
770227|pass|200711|andy@hexten.net|Data-Hexdumper|1.3|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200711150336
771720|pass|200711|Jost.Krieger@ruhr-uni-bochum.de|Acme-Pony|1.1.2|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200711151421
773882|pass|200711|andy@hexten.net|Devel-CheckOS|1.41|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200711160111
774066|pass|200711|chris@bingosnet.co.uk|Data-Compare|0.17|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200711160119
775025|pass|200711|andy@hexten.net|Devel-CheckOS|1.41|i486-linux-gnu|5.8.8|linux|2.6.15.7|200711161355
776151|pass|200711|andy@hexten.net|Data-Hexdumper|1.3|i686-linux-thread-multi|5.10.0 patch 32330|linux|2.6.22-14-generic|200711161734
779725|pass|200711|srezic@cpan.org|CPAN-FindDependencies|1.1|amd64-freebsd|5.10.0 patch 32367|freebsd|6.2-release|200711172259
779882|pass|200711|mmusgrove@cpan.org ("Matthew Musgrove")|XML-DoubleEncodedEntities|1.0|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200711171714
780203|pass|200711|srezic@cpan.org|Data-Compare|0.17|amd64-freebsd|5.10.0 patch 32367|freebsd|6.2-release|200711180053
780237|pass|200711|mmusgrove@cpan.org|XML-DoubleEncodedEntities|1.0|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200711171759
782724|pass|200711|srezic@cpan.org|CPAN-FindDependencies|1.1|amd64-freebsd-thread-multi|5.10.0 patch 32367|freebsd|6.2-release|200711181939
783177|pass|200711|srezic@cpan.org|Data-Compare|0.17|amd64-freebsd-thread-multi|5.10.0 patch 32367|freebsd|6.2-release|200711182121
783899|pass|200711|chris@bingosnet.co.uk|Data-Compare|0.17|i686-linux-thread-multi-64int|5.10.0|linux|2.6.16.38|200711190054
784863|pass|200711|Jost.Krieger@ruhr-uni-bochum.de|Acme-Pony|1.1.2|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200711191100
790887|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200711202149
790891|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200711202150
790893|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200711202150
790896|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200711202151
790898|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200711202151
790899|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200711202152
790900|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200711202153
791726|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-64int|5.8.8 patch 32389|linux|2.6.22-1-k7|200711210525
791728|pass|200711|icerider70@gmail.com ("ICERIDER")|Devel-CheckOS|1.42|i486-linux-gnu|5.8.8|linux|2.6.22-3-amd64|200711210436
792257|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200711211005
792263|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200711211005
792268|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200711211006
792275|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200711211006
792278|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200711211007
792281|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200711211008
792287|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200711211008
792289|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200711211009
792294|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200711211009
792332|pass|200711|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Devel-CheckOS|1.42|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200711211015
792411|pass|200711|srezic@cpan.org|Devel-CheckOS|1.42|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200711211155
792711|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200711211315
792839|pass|200711|Jost.Krieger@ruhr-uni-bochum.de|Acme-Pony|1.1.2|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200711211437
793079|pass|200711|Jost.Krieger@ruhr-uni-bochum.de|Devel-CheckOS|1.42|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200711211551
793685|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0_rc3|200711211914
793833|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.42|darwin-2level|5.9.5|darwin|8.10.1|200711211958
793845|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.42|i686-linux|5.8.8|linux|2.4.27-3-686|200711212002
793890|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.42|i686-linux|5.6.2|linux|2.4.27-3-686|200711212013
793933|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.42|i686-linux|5.9.5|linux|2.4.27-3-686|200711212018
793969|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.42|i386-freebsd|5.9.5|freebsd|6.2-release|200711212013
794720|fail|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Bryar|3.0|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200711220351
794722|fail|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Bryar|3.0|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200711220351
794724|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Bryar|3.0|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200711220352
794732|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Bryar|3.0|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200711220354
794757|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Bryar|3.0|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200711220352
794758|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Bryar|3.0|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200711220353
794759|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Bryar|3.0|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200711220353
794883|fail|200711|srezic@cpan.org|Bryar|3.0|amd64-freebsd|5.10.0 patch 32367|freebsd|6.2-release|200711220756
794928|na|200711|srezic@cpan.org|Bryar|3.0|amd64-freebsd|5.5.5|freebsd|6.2-release|200711220815
794967|pass|200711|srezic@cpan.org|Bryar|3.0|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200711220802
794968|pass|200711|srezic@cpan.org|Bryar|3.0|amd64-freebsd|5.6.2|freebsd|6.2-release|200711220809
795134|fail|200711|srezic@cpan.org|Bryar|3.0|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200711221015
796071|fail|200711|icerider70@gmail.com ("ICERIDER")|Bryar|3.0|i486-linux-gnu|5.8.8|linux|2.6.22-3-amd64|200711220952
796188|fail|200711|icerider70@gmail.com ("ICERIDER")|Bryar|3.0|i486-linux-gnu|5.8.8|linux|2.6.22-3-amd64|200711220952
796262|pass|200711|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.42|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200711230041
796506|pass|200711|chris@bingosnet.co.uk|Bryar|3.0|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200711221727
796514|pass|200711|chris@bingosnet.co.uk|Bryar|3.0|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200711221732
796522|pass|200711|chris@bingosnet.co.uk|Bryar|3.0|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200711221737
796553|pass|200711|chris@bingosnet.co.uk|Bryar|3.0|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200711221742
796572|pass|200711|chris@bingosnet.co.uk|Bryar|3.0|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200711221748
796574|pass|200711|chris@bingosnet.co.uk|Bryar|3.0|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200711221753
796578|pass|200711|chris@bingosnet.co.uk|Bryar|3.0|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200711221758
796586|pass|200711|chris@bingosnet.co.uk|Bryar|3.0|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200711221803
796609|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0_rc3|200711221805
796624|pass|200711|chris@bingosnet.co.uk|Bryar|3.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200711221809
796632|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.42|sun4-solaris|5.9.5|solaris|2.9|200711221409
796633|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.42|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200711222008
796792|pass|200711|taro-nishino@sam.hi-ho.ne.jp (Taro Nishino)|Devel-CheckOS|1.42|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200711230101
796943|pass|200711|taro-nishino@sam.hi-ho.ne.jp (Taro Nishino)|Devel-CheckOS|1.42|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200711230101
797187|pass|200711|david@cantrell.org.uk|Class-CanBeA|1.2|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230002
797200|pass|200711|david@cantrell.org.uk|CPAN-FindDependencies|1.1|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230006
797203|pass|200711|david@cantrell.org.uk|Data-Compare|0.17|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230006
797204|pass|200711|david@cantrell.org.uk|Data-Transactional|1.0|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230007
797205|pass|200711|david@cantrell.org.uk|Data-Hexdumper|1.3|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230006
797207|pass|200711|david@cantrell.org.uk|Devel-CheckLib|0.3|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230007
797208|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.42|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230007
797226|pass|200711|david@cantrell.org.uk|File-Overwrite|1.0|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230021
797227|pass|200711|david@cantrell.org.uk|Games-Dice-Advanced|1.1|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230021
797228|pass|200711|david@cantrell.org.uk|NestedMap|1.0|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230022
797240|pass|200711|david@cantrell.org.uk|Net-Random|2.0|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230026
797246|pass|200711|david@cantrell.org.uk|Number-Phone|1.58|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230028
797251|pass|200711|david@cantrell.org.uk|Number-Phone-UK-DetailedLocations|1.3|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230029
797265|pass|200711|david@cantrell.org.uk|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230031
797266|pass|200711|david@cantrell.org.uk|Statistics-SerialCorrelation|1.1|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230031
797268|pass|200711|david@cantrell.org.uk|Tie-STDOUT|1.03|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230031
797269|pass|200711|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230031
797270|pass|200711|david@cantrell.org.uk|Sub-WrapPackages|1.2|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230031
797271|pass|200711|david@cantrell.org.uk|XML-Tiny|1.11|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711230032
797770|fail|200711|chris@bingosnet.co.uk|Bryar|3.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0_rc3|200711230741
798215|pass|200711|david@cantrell.org.uk|Bryar|3.0|i686-linux|5.8.8|linux|2.4.27-3-686|200711231110
798251|fail|200711|david@cantrell.org.uk|Bryar|3.0|i686-linux-thread-multi|5.10.0 patch 32448|linux|2.4.27-3-686|200711231131
798475|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200711231503
799278|fail|200711|david@cantrell.org.uk|Bryar|3.0|sun4-solaris|5.9.5|solaris|2.9|200711231241
799280|fail|200711|david@cantrell.org.uk|Bryar|3.0|alpha-netbsd|5.9.5|netbsd|2.1.0_stable|200711231841
799737|fail|200711|david@cantrell.org.uk|Bryar|3.0|darwin-thread-multi-2level|5.10.0 patch 32468|darwin|8.10.1|200711232131
799879|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200711232234
800041|pass|200711|david@cantrell.org.uk|CPAN-FindDependencies|1.1|sun4-solaris-thread-multi|5.10.0 patch 32468|solaris|2.9|200711231702
800120|pass|200711|david@cantrell.org.uk|Devel-CheckOS|1.42|sun4-solaris-thread-multi|5.10.0 patch 32468|solaris|2.9|200711231717
800122|pass|200711|david@cantrell.org.uk|File-Find-Rule-Permissions|1.3|sun4-solaris-thread-multi|5.10.0 patch 32468|solaris|2.9|200711231717
800244|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200711240055
800714|pass|200711|chris@bingosnet.co.uk|Bryar|3.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200711240531
802110|pass|200711|chris@bingosnet.co.uk|Bryar|3.0|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200711241858
802164|pass|200711|andy@hexten.net|Data-Hexdumper|1.3|i686-linux-thread-multi|5.10.0 patch 32478|linux|2.6.18-5-686|200711241915
804772|pass|200711|andy@hexten.net|Data-Hexdumper|1.3|i386-freebsd-thread-multi|5.10.0 patch 32487|freebsd|6.2-release|200711251905
805243|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-pc-solaris2.11|5.8.7|solaris|2.11|200711252121
805468|pass|200711|andy@hexten.net|Data-Hexdumper|1.3|ppc-linux-thread-multi|5.10.0 patch 32376|linux|2.6.20-16-powerpc|200711252222
805729|pass|200711|david@cantrell.org.uk|Tie-STDOUT|1.03|alpha-netbsd|5.10.0 patch 31856|netbsd|2.1.0_stable|200711260042
805754|pass|200711|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|alpha-netbsd|5.10.0 patch 31856|netbsd|2.1.0_stable|200711260050
805787|pass|200711|david@cantrell.org.uk|Net-Random|2.0|alpha-netbsd|5.10.0 patch 31856|netbsd|2.1.0_stable|200711260103
806233|pass|200711|chris@bingosnet.co.uk|Bryar|3.0|i386-pc-solaris2.11|5.8.7|solaris|2.11|200711260744
807683|pass|200711|Jost.Krieger@ruhr-uni-bochum.de|Acme-Pony|1.1.2|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200711261548
807867|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200711261640
807929|pass|200711|stro@cpan.org|Devel-CheckOS|1.42|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200711261758
809202|pass|200711|david@cantrell.org.uk|Data-Hexdumper|1.3|darwin-thread-multi-2level|5.10.0 patch 32468|darwin|8.10.1|200711262249
810317|pass|200711|chris@bingosnet.co.uk|Bryar|3.0|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200711270828
811314|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200711271352
811999|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200711271804
813087|pass|200711|andy@hexten.net|Data-Hexdumper|1.3|i486-linux-gnu|5.8.8|linux|2.6.15.7|200711280142
813495|fail|200711|chris@bingosnet.co.uk|Bryar|3.0|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200711280649
813940|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.3|i686-linux-thread-multi|5.10.0|linux|2.6.22-1-k7|200711281215
814046|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-thread-multi|5.10.0|linux|2.6.22-1-k7|200711281320
814406|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200711281443
814407|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|darwin-thread-multi|5.6.2|darwin|8.10.0|200711281444
814413|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|darwin-thread-multi-2level|5.8.1|darwin|8.10.0|200711281445
814415|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|darwin-thread-multi-2level|5.8.2|darwin|8.10.0|200711281446
814417|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|darwin-thread-multi-2level|5.8.3|darwin|8.10.0|200711281447
814420|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|darwin-thread-multi-2level|5.8.4|darwin|8.10.0|200711281448
814424|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|darwin-thread-multi-2level|5.8.5|darwin|8.10.0|200711281449
814427|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|darwin-thread-multi-2level|5.8.6|darwin|8.10.0|200711281450
814429|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|darwin-thread-multi-2level|5.8.7|darwin|8.10.0|200711281451
814433|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|darwin-thread-multi-2level|5.8.8|darwin|8.10.0|200711281452
815957|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi|5.10.0|linux|2.6.22-1-k7|200711290131
816430|fail|200711|chris@bingosnet.co.uk|Bryar|3.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200711290454
818090|pass|200711|srezic@cpan.org|Data-Compare|0.17|i386-freebsd|5.10.0 patch 32478|freebsd|6.1-release|200711291851
818305|pass|200711|srezic@cpan.org|Data-Compare|0.17|i386-freebsd|5.6.2|freebsd|6.1-release|200711292028
819571|pass|200711|alexchorny@gmail.com|Data-Hexdumper|1.3|i686-linux|5.10.0 patch 32015|linux|2.4.20-8smp|200711300129
819990|pass|200711|chris@bingosnet.co.uk|Devel-CheckOS|1.42|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.2|200711301328
820335|pass|200711|alexchorny@gmail.com|Data-Compare|0.17|i686-linux|5.10.0 patch 32015|linux|2.4.20-8smp|200711300740
820981|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux|5.10.0|linux|2.6.22-1-k7|200711302032
821716|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux|5.10.0|linux|2.6.22-1-k7|200711302146
821881|pass|200711|david@cantrell.org.uk|Data-Compare|0.17|darwin-thread-multi-2level|5.10.0 patch 32468|darwin|8.10.1|200711302135
822556|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux|5.10.0|linux|2.6.22-1-k7|200711302357
822812|pass|200711|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.3|i686-linux|5.10.0|linux|2.6.22-1-k7|200711302254
823787|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200712010557
825915|pass|200712|david@cantrell.org.uk|Data-Compare|0.17|alpha-netbsd|5.10.0 patch 31856|netbsd|2.1.0_stable|200712012335
825953|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|cygwin-thread-multi-64int|5.10.0|cygwin|1.5.24(0.15642)|200712020017
826485|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0_rc3|200712020732
828013|pass|200712|david@cantrell.org.uk|Data-Compare|0.17|sun4-solaris-thread-multi|5.10.0 patch 32468|solaris|2.9|200712021707
828319|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0_rc3|200712030142
829840|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|1.99_01|i686-linux|5.10.0|linux|2.6.22-1-k7|200712031905
829850|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|1.99_01|i686-linux-thread-multi|5.10.0|linux|2.6.22-1-k7|200712031910
829873|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|1.99_01|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200712031916
829878|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|1.99_01|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200712031921
829881|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|1.99_01|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200712031926
829889|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|1.99_01|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200712031932
829908|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|1.99_01|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200712031938
829926|pass|200712|srezic@cpan.org|Data-Compare|0.17|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200712031942
830133|pass|200712|srezic@cpan.org|CPAN-FindDependencies|1.99_01|amd64-freebsd|5.10.0 patch 32559|freebsd|6.2-release|200712032137
830134|pass|200712|srezic@cpan.org|CPAN-FindDependencies|1.99_01|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200712032135
830173|pass|200712|xdg@hyperbolic.net (DAGOLDEN)|CPAN-FindDependencies|1.99_01|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200712031555
830197|pass|200712|srezic@cpan.org|CPAN-FindDependencies|1.99_01|amd64-freebsd-thread-multi|5.10.0 patch 32367|freebsd|6.2-release|200712032138
831557|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200712040554
831803|pass|200712|chris@bingosnet.co.uk|Devel-CheckOS|1.42|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200712040900
831868|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200712040923
831873|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200712040927
831878|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200712040930
831888|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200712040934
831893|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200712040938
831897|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200712040942
831902|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200712040946
831907|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200712040949
831993|pass|200712|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|CPAN-FindDependencies|1.99_01|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200712041045
832037|pass|200712|srezic@cpan.org|Devel-CheckOS|1.42|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200712041214
832173|pass|200712|icerider70@gmail.com ("ICERIDER")|CPAN-FindDependencies|1.99_01|i486-linux-gnu|5.8.8|linux|2.6.22-3-amd64|200712040644
832268|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200712041351
832504|pass|200712|srezic@cpan.org|CPAN-FindDependencies|1.99_01|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200712041628
832963|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200712040744
834171|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200712050347
834470|pass|200712|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200712050739
834935|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200712050000
835613|pass|200712|imacat@mail.imacat.idv.tw|CPAN-FindDependencies|1.99_01|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200712060104
835862|pass|200712|srezic@cpan.org|CPAN-FindDependencies|1.99_01|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200712051931
836161|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200712050931
836650|pass|200712|david@cantrell.org.uk|Devel-CheckOS|1.42|darwin-thread-multi-2level|5.10.0 patch 32468|darwin|8.10.1|200712052301
836743|pass|200712|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200712051300
837303|pass|200712|srezic@cpan.org|Data-Compare|0.17|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200712061010
837491|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|i386-pc-solaris2.11|5.8.7|solaris|2.11|200712061102
838315|pass|200712|dillman@ria.buffalo.edu|Devel-CheckOS|1.42|sun4-solaris|5.8.8|solaris|2.9|200712061058
838624|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|i386-pc-solaris2.11-thread-multi-64int|5.10.0|solaris|2.11|200712062244
838827|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|i386-pc-solaris2.11-thread-multi-64int|5.10.0|solaris|2.11|200712070122
839373|pass|200712|notbenh@cpan.org|Devel-CheckOS|1.42|i486-linux-gnu|5.8.8|linux|2.6.15.7|200712061109
839420|pass|200712|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-pc-solaris2.11-thread-multi-64int|5.10.0|solaris|2.11|200712070844
840134|pass|200712|andy@hexten.net|Devel-CheckOS|1.42|darwin-thread-multi-2level|5.10.0 patch 32560|darwin|9.1.0|200712071523
840932|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200712072327
840955|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|darwin-thread-multi-2level|5.8.1|darwin|8.10.0|200712072337
840967|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|darwin-thread-multi-2level|5.8.2|darwin|8.10.0|200712072342
840982|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|darwin-thread-multi-2level|5.8.3|darwin|8.10.0|200712072348
841002|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|darwin-thread-multi-2level|5.8.7|darwin|8.10.0|200712080006
841011|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|darwin-thread-multi-2level|5.8.8|darwin|8.10.0|200712080012
841362|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200712080341
841879|pass|200712|chris@bingosnet.co.uk|Devel-CheckOS|1.42|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200712080232
841964|pass|200712|g.grigelionis@computer.org|Devel-CheckOS|1.42|i586-linux-thread-multi|5.8.8|linux|2.6.16|200712080029
842674|pass|200712|david@cantrell.org.uk|Devel-CheckOS|1.42|alpha-netbsd|5.10.0 patch 31856|netbsd|2.1.0_stable|200712082045
842797|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200712082257
843205|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200712090321
843500|pass|200712|cpan@pqpq.de|Data-Compare|0.17|amd64-freebsd|5.10.0 patch 32559|freebsd|6.2-release|200712090923
843671|pass|200712|cpan@pqpq.de|CPAN-FindDependencies|1.99_01|amd64-freebsd|5.10.0 patch 32559|freebsd|6.2-release|200712091035
844619|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200712091711
846267|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200712100820
846955|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200712101233
847107|pass|200712|d@0x64.nl|Devel-CheckOS|1.42|x86_64-linux-gnu|5.8.8|linux|2.6.17-2-vserver-amd64|200712101305
847826|pass|200712|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200712101833
848586|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200712110203
848851|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200712110636
849160|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200712111048
849338|pass|200712|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200712111202
850973|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|cygwin-thread-multi-64int|5.10.0|cygwin|1.5.24(0.15642)|200712112231
852048|pass|200712|paranoid@sarpedon.site5.com ((Mike Bissett))|Data-Compare|0.17|x86_64-linux-gnu|5.8.8|linux|2.6.15.7|200712121210
852788|pass|200712|chris@bingosnet.co.uk|Devel-CheckOS|1.42|cygwin-thread-multi-64int|5.10.0|cygwin|1.5.24(0.15642)|200712121443
852849|pass|200712|chris@bingosnet.co.uk|Devel-CheckOS|1.42|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200712120434
854890|pass|200712|chris@bingosnet.co.uk|Devel-CheckOS|1.42|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200712131119
855361|pass|200712|rlane10@wi.rr.com|Devel-CheckOS|1.42|darwin-thread-multi-2level|5.8.8|darwin|8.9.0|200712130738
857132|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.0|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200712141556
857136|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.0|i686-linux|5.10.0|linux|2.6.22-1-k7|200712141557
857139|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.0|i686-linux-thread-multi|5.10.0|linux|2.6.22-1-k7|200712141558
857142|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.0|i686-linux-64int|5.9.5|linux|2.6.22-1-k7|200712141559
857143|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.0|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200712141600
857146|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.0|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200712141601
857148|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.0|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200712141602
857151|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.0|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200712141604
857197|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200712141517
857215|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|darwin-thread-multi-2level|5.8.1|darwin|8.10.0|200712141525
857223|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|darwin-thread-multi-2level|5.8.2|darwin|8.10.0|200712141532
857243|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|darwin-thread-multi-2level|5.8.3|darwin|8.10.0|200712141540
857270|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|darwin-thread-multi-2level|5.8.4|darwin|8.10.0|200712141547
857285|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|darwin-thread-multi-2level|5.8.5|darwin|8.10.0|200712141555
857307|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|darwin-thread-multi-2level|5.8.6|darwin|8.10.0|200712141603
857326|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|darwin-thread-multi-2level|5.8.7|darwin|8.10.0|200712141611
857348|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|1.99_01|darwin-thread-multi-2level|5.8.8|darwin|8.10.0|200712141620
857425|pass|200712|srezic@cpan.org|CPAN-FindDependencies|2.0|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200712141822
857789|pass|200712|srezic@cpan.org|CPAN-FindDependencies|2.0|amd64-freebsd|5.10.0 patch 32559|freebsd|6.2-release|200712142121
857887|pass|200712|icerider70@gmail.com ("ICERIDER")|CPAN-FindDependencies|2.0|i486-linux-gnu|5.8.8|linux|2.6.22-3-amd64|200712141449
857999|pass|200712|srezic@cpan.org|CPAN-FindDependencies|2.0|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200712142200
858010|fail|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200712142141
858011|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200712142142
858012|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200712142144
858013|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200712142145
858014|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200712142146
858016|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200712142147
858018|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200712142149
858020|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200712142150
858175|pass|200712|srezic@cpan.org|CPAN-FindDependencies|2.0|amd64-freebsd-thread-multi|5.10.0 patch 32367|freebsd|6.2-release|200712150009
859571|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200712152036
860492|pass|200712|overlordq@gmail.com|Net-Random|2.0|i486-linux-gnu|5.8.8|linux|2.6.22.10|200712160142
860548|pass|200712|overlordq@gmail.com|Data-Compare|0.17|i486-linux-gnu|5.8.8|linux|2.6.22.10|200712160211
861014|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200712161129
861340|pass|200712|imacat@mail.imacat.idv.tw|CPAN-FindDependencies|2.0|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200712170123
862554|pass|200712|andy@hexten.net|Number-Phone|1.58|darwin-thread-multi-2level|5.10.0 patch 32604|darwin|9.1.0|200712170017
862555|pass|200712|andy@hexten.net|Class-CanBeA|1.2|darwin-thread-multi-2level|5.10.0 patch 32604|darwin|9.1.0|200712170017
862556|pass|200712|andy@hexten.net|Data-Compare|0.17|darwin-thread-multi-2level|5.10.0 patch 32604|darwin|9.1.0|200712170017
862557|pass|200712|andy@hexten.net|Data-Transactional|1.0|darwin-thread-multi-2level|5.10.0 patch 32604|darwin|9.1.0|200712170017
862558|pass|200712|andy@hexten.net|Net-Random|2.0|darwin-thread-multi-2level|5.10.0 patch 32604|darwin|9.1.0|200712170017
862559|pass|200712|andy@hexten.net|Number-Phone-UK-DetailedLocations|1.3|darwin-thread-multi-2level|5.10.0 patch 32604|darwin|9.1.0|200712170018
862561|pass|200712|andy@hexten.net|CPAN-FindDependencies|2.0|darwin-thread-multi-2level|5.10.0 patch 32604|darwin|9.1.0|200712170020
862562|pass|200712|andy@hexten.net|Devel-CheckLib|0.3|darwin-thread-multi-2level|5.10.0 patch 32604|darwin|9.1.0|200712170020
862563|pass|200712|andy@hexten.net|Games-Dice-Advanced|1.1|darwin-thread-multi-2level|5.10.0 patch 32604|darwin|9.1.0|200712170021
862763|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|darwin-thread-multi|5.6.2|darwin|8.10.0|200712170404
863091|pass|200712|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|CPAN-FindDependencies|2.0|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200712170901
863578|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200712170416
864697|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200712180430
865057|pass|200712|cpan@pqpq.de|CPAN-FindDependencies|2.0|amd64-freebsd|5.10.0 patch 32559|freebsd|6.2-release|200712180921
865873|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200712181359
866599|pass|200712|salvacpan@gmail.com|Data-Compare|0.17|sparc64-netbsd|5.10.0|netbsd|4.0_rc5|200712181435
866762|pass|200712|andy@hexten.net|Devel-CheckLib|0.3|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200712181902
866770|pass|200712|andy@hexten.net|Devel-CheckLib|0.3|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200712181803
866873|pass|200712|andy@hexten.net|Class-CanBeA|1.2|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200712181751
866889|pass|200712|andy@hexten.net|Games-Dice-Advanced|1.1|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200712181917
867078|pass|200712|andy@hexten.net|Class-CanBeA|1.2|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200712181849
867192|pass|200712|andy@hexten.net|Number-Phone|1.58|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200712181931
867193|pass|200712|andy@hexten.net|Number-Phone-UK-DetailedLocations|1.3|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200712181931
867611|pass|200712|andy@hexten.net|Games-Dice-Advanced|1.1|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200712181816
867648|pass|200712|andy@hexten.net|Devel-CheckOS|1.42|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200712182105
867785|pass|200712|andy@hexten.net|Net-Random|2.0|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200712182123
868027|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|i386-pc-solaris2.11-thread-multi-64int|5.10.0|solaris|2.11|200712182138
868129|pass|200712|andy@hexten.net|CPAN-FindDependencies|2.0|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200712182225
868221|pass|200712|andy@hexten.net|Data-Compare|0.17|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200712182234
868223|pass|200712|andy@hexten.net|Data-Transactional|1.0|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200712182235
868857|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|i386-pc-solaris2.11-thread-multi-64int|5.10.0|solaris|2.11|200712190230
870729|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200712191657
872660|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200712200418
874234|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux|5.11.0|linux|2.6.22-1-k7|200712202153
874402|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux|5.11.0|linux|2.6.22-1-k7|200712202233
874775|pass|200712|bricas@cpan.org ("Brian Cassidy")|Devel-CheckOS|1.42|i486-linux-gnu|5.8.8|linux|2.6.15.7|200712201446
875337|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux|5.11.0|linux|2.6.22-1-k7|200712210125
875730|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200712210230
877394|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux|5.11.0|linux|2.6.22-1-k7|200712210733
877531|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.3|i686-linux|5.11.0|linux|2.6.22-1-k7|200712211038
878404|pass|200712|bricas@cpan.org ("Brian Cassidy")|Devel-CheckLib|0.3|i486-linux-gnu|5.8.8|linux|2.6.15.7|200712210939
878801|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200712211947
879522|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-64int|5.11.0|linux|2.6.22-1-k7|200712212314
879947|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0|linux|2.6.22-1-k7|200712220025
880232|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|cygwin-thread-multi-64int|5.10.0|cygwin|1.5.24(0.15642)|200712220044
880595|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-64int|5.11.0|linux|2.6.22-1-k7|200712220224
880734|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.3|i686-linux-64int|5.11.0|linux|2.6.22-1-k7|200712220340
880776|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0|linux|2.6.22-1-k7|200712220304
884030|pass|200712|david@cantrell.org.uk|Bryar|2.8_02|i686-linux|5.6.2|linux|2.4.27-3-686|200712222041
885997|pass|200712|stro@cpan.org|CPAN-FindDependencies|2.0|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200712230141
888988|pass|200712|srezic@cpan.org|Data-Compare|0.17|amd64-freebsd|5.10.0|freebsd|6.2-release|200712241030
889201|pass|200712|andy@hexten.net|XML-Tiny|1.11|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200712241127
890820|pass|200712|cpan@pqpq.de|Devel-CheckOS|1.42|amd64-freebsd|5.10.0 patch 32559|freebsd|6.2-release|200712251539
891034|pass|200712|rlane10@wi.rr.com|Devel-CheckOS|1.42|darwin-2level|5.10.0|darwin|8.11.0|200712250509
892422|pass|200712|chris@bingosnet.co.uk|Devel-CheckOS|1.42|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200712260259
894567|fail|200712|david@cantrell.org.uk|Bryar|2.8_02|darwin-thread-multi-2level|5.10.0 patch 32468|darwin|8.10.1|200712270027
894603|pass|200712|srezic@cpan.org|Data-Compare|0.17|amd64-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200712270134
894628|pass|200712|rlane10@wi.rr.com|XML-Tiny|1.11|darwin-2level|5.10.0|darwin|8.11.0|200712261305
894709|pass|200712|icerider70@gmail.com ("ICERIDER")|Devel-CheckOS|1.42|i386-linux|5.8.8|linux|2.6.18-53.el5|200712270448
895423|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int|5.11.0|linux|2.6.22-1-k7|200712271029
895613|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-thread-multi-64int|5.11.0|linux|2.6.22-1-k7|200712271144
896014|pass|200712|claco@chrislaco.com ("Christopher H. Laco")|Data-Compare|0.17|darwin-thread-multi-2level|5.8.6|darwin|8.0|200712270048
896291|pass|200712|icerider70@gmail.com ("ICERIDER")|Data-Compare|0.17|i386-linux|5.8.8|linux|2.6.18-53.el5|200712270727
896293|pass|200712|root@alysande.altrion.org ((System Administrator))|Devel-CheckOS|1.42|darwin-2level|5.10.0|darwin|8.11.1|200712271305
896347|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi-64int|5.11.0|linux|2.6.22-1-k7|200712271437
896767|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.3|i686-linux-thread-multi-64int|5.11.0|linux|2.6.22-1-k7|200712271414
897406|pass|200712|david@cantrell.org.uk|XML-Tiny|1.11|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200712271820
897407|pass|200712|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|0.17|i686-linux-thread-multi-64int|5.11.0|linux|2.6.22-1-k7|200712271021
897430|pass|200712|david@cantrell.org.uk|Data-Compare|0.17|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200712271826
897445|pass|200712|david@cantrell.org.uk|XML-Tiny|1.11|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200712271817
897550|pass|200712|david@cantrell.org.uk|Net-Random|2.0|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200712271856
897601|pass|200712|david@cantrell.org.uk|Net-Random|2.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200712271853
897769|pass|200712|david@cantrell.org.uk|Net-Random|2.0|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200712271354
897863|pass|200712|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200712271233
897915|pass|200712|david@cantrell.org.uk|Net-Random|2.0|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200712272028
897993|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200712271258
898157|pass|200712|david@cantrell.org.uk|XML-Tiny|1.11|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200712272144
898228|pass|200712|david@cantrell.org.uk|Net-Random|2.0|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200712272152
898271|pass|200712|david@cantrell.org.uk|Data-Compare|0.17|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200712272143
899962|pass|200712|david@cantrell.org.uk|XML-Tiny|1.11|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200712281636
900858|pass|200712|imacat@mail.imacat.idv.tw|Data-Compare|0.17|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200712290438
903334|pass|200712|rlane10@wi.rr.com|Tie-STDOUT|1.03|darwin-2level|5.10.0|darwin|8.11.0|200712300053
904773|pass|200712|srezic@cpan.org|Statistics-ChiSquare|0.5|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200712302026
904802|pass|200712|srezic@cpan.org|Statistics-SerialCorrelation|1.1|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200712302034
905218|pass|200712|rlane10@wi.rr.com|Devel-CheckOS|1.42|i686-linux|5.10.0|linux|2.6.23.1luckyseven|200712301444
906166|pass|200712|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-pc-solaris2.11-thread-multi-64int|5.10.0|solaris|2.11|200712310730
906374|pass|200712|chris@bingosnet.co.uk|Devel-CheckOS|1.42|cygwin-thread-multi-64int|5.10.0|cygwin|1.5.24(0.15642)|200712310854
906487|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|cygwin-thread-multi-64int|5.10.0|cygwin|1.5.24(0.15642)|200712311007
906696|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200712311151
906698|na|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|darwin-thread-multi-2level|5.8.1|darwin|8.10.0|200712311154
906700|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|darwin-thread-multi-2level|5.8.2|darwin|8.10.0|200712311157
906701|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|darwin-thread-multi-2level|5.8.3|darwin|8.10.0|200712311159
906705|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|darwin-thread-multi-2level|5.8.4|darwin|8.10.0|200712311202
906707|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|darwin-thread-multi-2level|5.8.5|darwin|8.10.0|200712311204
906712|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|darwin-thread-multi-2level|5.8.6|darwin|8.10.0|200712311206
906715|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|darwin-thread-multi-2level|5.8.7|darwin|8.10.0|200712311208
906717|pass|200712|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|darwin-thread-multi-2level|5.8.8|darwin|8.10.0|200712311210
907980|pass|200712|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200712310840
908298|pass|200712|chris@bingosnet.co.uk|Data-Compare|0.17|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200712310949
908891|pass|200712|cpantester@zombie.org.uk|Data-Compare|0.17|x86_64-linux-ld|5.11.0 patch 32787|linux|2.6.22-3-amd64|200712311732
910528|pass|200712|cpantester@zombie.org.uk|Data-Compare|0.17|x86_64-linux-ld|5.11.0 patch 32790|linux|2.6.22-3-amd64|200712312036
911392|pass|200801|chris@bingosnet.co.uk|Data-Compare|0.17|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200801011056
911683|pass|200801|cpantester@zombie.org.uk|Data-Compare|0.17|x86_64-linux-ld|5.11.0 patch 32790|linux|2.6.22-3-amd64|200801011209
912997|pass|200801|cpantester@hotmail.co.uk|Data-Compare|0.17|x86_64-linux-thread-multi-ld|5.11.0 patch 32792|linux|2.6.22-3-amd64|200801012023
914399|pass|200801|cpantester@hotmail.co.uk|Data-Compare|0.17|x86_64-linux-thread-multi-ld|5.11.0 patch 32792|linux|2.6.22-3-amd64|200801020106
914536|pass|200801|andy@hexten.net|Data-Hexdumper|1.3|i686-linux-thread-multi|5.10.0|linux|2.6.18-5-686|200801021551
914904|pass|200801|andy@hexten.net|Data-Hexdumper|1.3|i686-linux-thread-multi|5.10.0|linux|2.6.20-16-server|200801021710
915075|pass|200801|andy@hexten.net|Data-Hexdumper|1.3|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200801021749
915506|pass|200801|andy@hexten.net|Statistics-ChiSquare|0.5|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200801022013
915791|pass|200801|d@0x64.nl|XML-Tiny|1.11|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200801021048
916082|pass|200801|cpantester@hotmail.co.uk|Data-Compare|0.17|x86_64-linux-thread-multi-ld|5.11.0 patch 32805|linux|2.6.22-3-amd64|200801022238
917482|pass|200801|chris@bingosnet.co.uk|Data-Compare|0.17|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200801030803
918102|pass|200801|cpantester@hotmail.co.uk|Data-Compare|0.17|x86_64-linux-thread-multi-ld|5.11.0 patch 32800|linux|2.6.22-3-amd64|200801021356
918614|pass|200801|Jost.Krieger@ruhr-uni-bochum.de|Acme-Pony|1.1.2|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200801031055
919916|pass|200801|imacat@mail.imacat.idv.tw|Number-Phone|1.58|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200801040155
920490|pass|200801|d@0x64.nl|Data-Compare|0.17|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200801032352
921985|pass|200801|cpantester@hotmail.co.uk|Data-Compare|0.17|x86_64-linux-thread-multi-ld|5.11.0 patch 32809|linux|2.6.22-3-amd64|200801041016
923011|pass|200801|cpantester@hotmail.co.uk|Data-Compare|0.17|x86_64-linux-thread-multi-ld|5.11.0 patch 32824|linux|2.6.22-3-amd64|200801041358
923177|pass|200801|bob@randomness.org.uk ("bob walker")|Data-Compare|0.17|i86pc-solaris|5.10.0|solaris|2.10|200801041601
923700|pass|200801|cpantester@hotmail.co.uk|Data-Compare|0.17|x86_64-linux-thread-multi-ld|5.11.0 patch 32832|linux|2.6.22-3-amd64|200801041845
923985|pass|200801|chris@bingosnet.co.uk|Data-Compare|0.17|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200801042006
924386|pass|200801|david@cantrell.org.uk|Data-Compare|0.17|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200801042156
924870|pass|200801|david@cantrell.org.uk|Data-Compare|0.17|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200801050035
926854|pass|200801|chris@bingosnet.co.uk|Data-Compare|0.17|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200801052014
927106|pass|200801|david@cantrell.org.uk|Data-Compare|0.17|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801051404
927629|pass|200801|chris@bingosnet.co.uk|Data-Compare|0.17|cygwin-thread-multi-64int|5.11.0 patch 32837|cygwin|1.5.24(0.15642)|200801060155
930316|pass|200801|chris@bingosnet.co.uk|Data-Compare|0.17|cygwin-thread-multi-64int|5.11.0 patch 32864|cygwin|1.5.24(0.15642)|200801070135
931226|pass|200801|chris@bingosnet.co.uk|Data-Compare|0.17|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200801071157
931657|pass|200801|kamensky@cpan.org ("Moshe Kamensky")|Data-Compare|0.17|sun4-solaris|5.10.0|solaris|2.8|200801070928
931974|pass|200801|chris@bingosnet.co.uk|Data-Compare|0.17|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200801071713
932623|pass|200801|chris@bingosnet.co.uk|Data-Compare|0.17|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200801072025
934822|pass|200801|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Compare|0.17|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200801081334
938822|pass|200801|chris@bingosnet.co.uk|Data-Compare|0.17|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200801091702
939607|pass|200801|chris@bingosnet.co.uk|Data-Compare|0.17|alpha-linux-thread-multi|5.11.0 patch 32908|linux|2.6.18-5-alpha-generic|200801092044
939664|pass|200801|david@cantrell.org.uk|Bryar|2.6|i686-linux|5.6.2|linux|2.4.27-3-686|200801092100
939750|fail|200801|david@cantrell.org.uk|Bryar|3.0|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200801092123
940465|pass|200801|bob@randomness.org.uk ("bob walker")|Data-Compare|0.17|i86pc-solaris|5.8.8|solaris|2.10|200801100207
946793|pass|200801|chris@bingosnet.co.uk|Data-Compare|0.17|cygwin-thread-multi-64int|5.11.0 patch 32953|cygwin|1.5.24(0.15642)|200801120913
947178|pass|200801|cpantester@hotmail.co.uk|Data-Compare|0.17|x86_64-linux-thread-multi-ld|5.11.0 patch 32966|linux|2.6.22-3-amd64|200801121142
953318|fail|200801|rjbs@cpan.org|Bryar|3.0|darwin-2level|5.10.0|darwin|9.1.0|200801141030
955700|pass|200801|mmusgrove@cpan.org ("Matthew Musgrove")|Data-Compare|0.17|i686-linux-64int-ld|5.8.8|linux|2.6.22-14-generic|200801150051
955762|pass|200801|mmusgrove@cpan.org ("Matthew Musgrove")|Data-Compare|0.17|i686-linux-64int-ld|5.10.0|linux|2.6.22-14-generic|200801150101
956019|pass|200801|robertl@apnic.net ("Robert Loomans")|Data-Compare|0.17|darwin-thread-multi-2level|5.8.8|darwin|8.11.1|200801151755
958423|pass|200801|bob@randomness.org.uk ("bob walker")|Data-Compare|0.17|OpenBSD.i386-openbsd|5.8.8|openbsd|4.2|200801152111
958437|pass|200801|bob@randomness.org.uk ("bob walker")|Data-Compare|0.17|OpenBSD.i386-openbsd|5.10.0|openbsd|4.2|200801152114
958862|pass|200801|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux|5.11.0|linux|2.6.22-1-k7|200801160016
958863|pass|200801|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200801160016
958866|pass|200801|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200801160016
958869|pass|200801|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200801160017
958871|pass|200801|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200801160017
958873|pass|200801|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200801160018
958875|pass|200801|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200801160019
959222|pass|200801|bob@randomness.org.uk ("bob walker")|Data-Compare|1.18|i86pc-solaris|5.8.8|solaris|2.10|200801160124
959225|pass|200801|bob@randomness.org.uk ("bob walker")|Data-Compare|1.18|i86pc-solaris|5.10.0|solaris|2.10|200801160125
959422|pass|200801|icerider70@gmail.com ("ICERIDER")|Data-Compare|1.18|i386-linux|5.8.8|linux|2.6.18-53.el5|200801160210
960005|pass|200801|mmusgrove@cpan.org ("Matthew Musgrove")|CPAN-FindDependencies|1.99_01|i686-linux-64int-ld|5.8.8|linux|2.6.22-14-generic|200801160419
960006|pass|200801|mmusgrove@cpan.org ("Matthew Musgrove")|Devel-CheckOS|1.42|i686-linux-64int-ld|5.8.8|linux|2.6.22-14-generic|200801160419
960026|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200801160430
960030|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|darwin-thread-multi|5.6.2|darwin|8.10.0|200801160433
960035|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|darwin-thread-multi-2level|5.8.8|darwin|8.10.0|200801160439
960318|pass|200801|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Data-Compare|1.18|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200801161514
960472|pass|200801|Jost.Krieger+perl@rub.de|Data-Compare|1.18|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200801160920
960531|pass|200801|d@0x64.nl (DEVOGON)|Data-Compare|1.18|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200801160948
961467|pass|200801|david@cantrell.org.uk|Data-Compare|1.18|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200801161415
961485|pass|200801|david@cantrell.org.uk|Data-Compare|1.18|i686-linux|5.8.8|linux|2.4.27-3-686|200801161417
961502|pass|200801|david@cantrell.org.uk|Data-Compare|1.18|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200801161347
961506|pass|200801|david@cantrell.org.uk|Data-Compare|1.18|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801160820
961509|pass|200801|david@cantrell.org.uk|Data-Compare|1.18|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200801161420
961510|pass|200801|david@cantrell.org.uk|Data-Compare|1.18|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200801161420
961520|pass|200801|david@cantrell.org.uk|Data-Compare|1.18|i686-linux|5.6.2|linux|2.4.27-3-686|200801161421
961562|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|alpha-linux-thread-multi|5.11.0 patch 32981|linux|2.6.18-5-alpha-generic|200801161431
961963|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200801161638
962412|pass|200801|cpantester@hotmail.co.uk|Data-Compare|1.18|x86_64-linux-thread-multi-ld|5.11.0 patch 32982|linux|2.6.22-3-amd64|200801161929
962482|pass|200801|srezic@cpan.org|Data-Compare|1.18|amd64-freebsd|5.10.0|freebsd|6.2-release|200801162046
963003|pass|200801|srezic@cpan.org|Data-Compare|1.18|amd64-freebsd|5.6.2|freebsd|6.2-release|200801162135
963156|pass|200801|g.grigelionis@computer.org|Data-Compare|1.18|i586-linux-thread-multi|5.8.8|linux|2.6.16|200801162212
963203|pass|200801|srezic@cpan.org|Data-Compare|1.18|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200801162221
963745|pass|200801|bob@randomness.org.uk ("bob walker")|Data-Compare|1.18|OpenBSD.i386-openbsd|5.8.8|openbsd|4.2|200801162358
963770|pass|200801|bob@randomness.org.uk ("bob walker")|Data-Compare|1.18|OpenBSD.i386-openbsd|5.10.0|openbsd|4.2|200801170006
964012|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200801170031
964037|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200801170036
964040|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200801170039
964071|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200801170059
964084|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200801170102
964087|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200801170105
964091|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200801170108
964106|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200801170112
964125|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200801170115
964127|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200801170118
964635|pass|200801|d@0x64.nl (DEVOGON)|Data-Compare|1.18|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200801170756
964967|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200801170952
965354|pass|200801|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Compare|1.18|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200801171203
966001|pass|200801|imacat@mail.imacat.idv.tw|Data-Compare|1.18|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200801180044
967324|pass|200801|salve.nilsen@met.no ("Salve J Nilsen")|Data-Compare|1.18|i386-linux|5.8.8|linux|2.6.9-34.elsmp|200801171735
967365|pass|200801|kamensky@cpan.org ("Moshe Kamensky")|Data-Compare|1.18|sun4-solaris|5.10.0|solaris|2.8|200801171332
968007|pass|200801|overlordq@gmail.com|Data-Compare|1.18|i486-linux-gnu|5.8.8|linux|2.6.22.10|200801180317
968952|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200801181450
969190|pass|200801|d@0x64.nl (DEVOGON)|Data-Compare|1.18|i686-linux|5.10.0|linux|2.6.22-3-486|200801181718
969205|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200801181623
969783|pass|200801|dillman@ria.buffalo.edu|Data-Compare|1.18|sun4-solaris|5.8.8|solaris|2.9|200801181347
970950|pass|200801|mmusgrove@cpan.org ("Matthew Musgrove")|Devel-CheckOS|1.42|i686-linux-64int-ld|5.10.0|linux|2.6.22-14-generic|200801190046
972266|pass|200801|srezic@cpan.org|Data-Compare|1.18|amd64-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200801191057
972674|pass|200801|d@0x64.nl (DEVOGON)|Devel-CheckOS|1.42|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200801191419
972689|pass|200801|d@0x64.nl (DEVOGON)|Devel-CheckOS|1.42|i686-linux|5.10.0|linux|2.6.22-3-486|200801191423
972692|pass|200801|d@0x64.nl (DEVOGON)|Devel-CheckLib|0.3|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200801191423
972693|pass|200801|d@0x64.nl (DEVOGON)|Devel-CheckLib|0.3|i686-linux|5.10.0|linux|2.6.22-3-486|200801191425
974401|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|cygwin-thread-multi-64int|5.11.0 patch 33004|cygwin|1.5.24(0.15642)|200801192233
974723|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200801200019
974764|pass|200801|david@cantrell.org.uk|Class-CanBeA|1.2|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801200048
974766|pass|200801|david@cantrell.org.uk|Data-Hexdumper|1.0.1|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801200049
974769|pass|200801|david@cantrell.org.uk|Data-Hexdumper|1.2|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801200050
974786|fail|200801|david@cantrell.org.uk|Net-Random|2.0|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801200104
974787|pass|200801|david@cantrell.org.uk|Number-Phone|1.58|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801200105
974789|pass|200801|david@cantrell.org.uk|Number-Phone-UK-DetailedLocations|1.3|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801200106
974791|pass|200801|david@cantrell.org.uk|Statistics-ChiSquare|0.5|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801200109
974793|pass|200801|david@cantrell.org.uk|Statistics-SerialCorrelation|1.1|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801200109
974795|pass|200801|david@cantrell.org.uk|Sub-WrapPackages|1.2|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801200110
974797|pass|200801|david@cantrell.org.uk|Tie-Hash-Longest|1.1|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801200111
974799|pass|200801|david@cantrell.org.uk|Tie-Hash-Rank|1.0.1|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801200111
974800|pass|200801|david@cantrell.org.uk|Tie-STDOUT|1.02|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801200111
974803|pass|200801|david@cantrell.org.uk|Tie-Scalar-Decay|1.1.1|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801200112
974806|pass|200801|david@cantrell.org.uk|Tie-Scalar-Decay|1.1|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801200113
974809|pass|200801|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801200113
974811|pass|200801|david@cantrell.org.uk|XML-Tiny|1.06|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801200114
975159|pass|200801|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.18|i686-linux-64int-ld|5.10.0|linux|2.6.22-14-386|200801200355
975983|pass|200801|andy@hexten.net|Data-Compare|1.18|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200801201242
976661|pass|200801|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux|5.8.8 patch 33008|linux|2.6.22-1-k7|200801201822
977539|pass|200801|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux|5.8.8 patch 33008|linux|2.6.22-1-k7|200801202155
977631|pass|200801|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux|5.8.8 patch 33008|linux|2.6.22-1-k7|200801202104
977717|pass|200801|david@cantrell.org.uk|NestedMap|1.0|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801202208
977792|fail|200801|david@cantrell.org.uk|Bryar|3.0|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801202224
977962|pass|200801|david@cantrell.org.uk|Acme-Scurvy-Whoreson-BilgeRat|1.1|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801202248
978266|unknown|200801|david@cantrell.org.uk|Acme-Licence|1.0|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801202350
978376|pass|200801|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.3|i686-linux|5.8.8 patch 33008|linux|2.6.22-1-k7|200801210111
978612|unknown|200801|david@cantrell.org.uk|Acme-Pony|1.1.2|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801210109
978852|fail|200801|david@cantrell.org.uk|Data-Transactional|0.1|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801210149
978863|pass|200801|david@cantrell.org.uk|File-Find-Rule-Permissions|1.2|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801210151
978893|pass|200801|david@cantrell.org.uk|File-Overwrite|1.0|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801210158
978896|pass|200801|david@cantrell.org.uk|Games-Dice-Advanced|1.1|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801210159
978966|pass|200801|david@cantrell.org.uk|Palm-TreoPhoneCallDB|1.1|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801210217
978970|pass|200801|david@cantrell.org.uk|Palm-TreoPhoneCallDB|1|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801210219
979198|pass|200801|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux|5.8.8 patch 33008|linux|2.6.22-1-k7|200801210425
979494|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200801210515
980641|fail|200801|d@0x64.nl (DEVOGON)|Bryar|3.0|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200801211628
980947|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|alpha-linux-thread-multi|5.11.0 patch 33022|linux|2.6.18-5-alpha-generic|200801211712
982905|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200801220618
983254|pass|200801|fayland@cpan.org|Data-Compare|1.18|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200801220922
983260|pass|200801|rlane10@wi.rr.com|Data-Compare|1.18|darwin-2level-ld|5.10.0|darwin|8.11.0|200801220144
987518|pass|200801|andy@hexten.net|Data-Compare|1.18|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200801231403
987884|fail|200801|david@cantrell.org.uk|Bryar|2.6|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200801231711
988738|pass|200801|david@cantrell.org.uk|Acme-Scurvy-Whoreson-BilgeRat|1.1|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200801240003
989519|pass|200801|callen@kpi-consulting.net|Devel-CheckOS|1.42|i386-linux|5.8.8|linux|2.6.9-42.0.3.elsmp|200801231502
989759|pass|200801|stro@cpan.org|Data-Compare|1.18|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200801240511
994236|pass|200801|rjbs@cpan.org|Data-Compare|1.18|darwin-2level|5.10.0|darwin|9.1.0|200801261005
994237|pass|200801|rjbs@cpan.org|Data-Compare|1.18|darwin-2level|5.10.0|darwin|9.1.0|200801261005
996329|pass|200801|david@cantrell.org.uk|CPAN-FindDependencies|2.0|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200801271620
998612|pass|200801|bob@randomness.org.uk ("bob walker")|Devel-CheckOS|1.42|i86pc-solaris|5.8.8|solaris|2.10|200801281301
998621|pass|200801|bob@randomness.org.uk ("bob walker")|Devel-CheckOS|1.42|i86pc-solaris|5.10.0|solaris|2.10|200801281302
999761|pass|200801|chris@bingosnet.co.uk|Devel-CheckOS|1.42|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200801281923
1000713|pass|200801|bob@randomness.org.uk ("bob walker")|Devel-CheckOS|1.42|OpenBSD.i386-openbsd|5.8.8|openbsd|4.2|200801282355
1000716|pass|200801|bob@randomness.org.uk ("bob walker")|Devel-CheckOS|1.42|OpenBSD.i386-openbsd|5.10.0|openbsd|4.2|200801282358
1000819|pass|200801|g.grigelionis@computer.org|Devel-CheckOS|1.42|i386-linux|5.8.0|linux|2.6.9-22.18.bz155725.elsmp|200801282113
1000850|pass|200801|g.grigelionis@computer.org|Devel-CheckOS|1.42|x86_64-linux-thread-multi|5.8.3|linux|2.6.4|200801282117
1001686|pass|200801|david@cantrell.org.uk|Devel-CheckOS|1.42|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801290757
1001838|pass|200801|david@cantrell.org.uk|Devel-CheckOS|1.42|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200801291312
1001936|pass|200801|david@cantrell.org.uk|Devel-CheckOS|1.42|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200801291540
1002105|pass|200801|david@cantrell.org.uk|Devel-CheckOS|1.42|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200801291616
1002250|pass|200801|chris@bingosnet.co.uk|Devel-CheckOS|1.42|alpha-linux-thread-multi|5.11.0 patch 33082|linux|2.6.18-5-alpha-generic|200801291819
1002345|pass|200801|david@cantrell.org.uk|Devel-CheckOS|1.42|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200801291841
1002699|pass|200801|cpantester@hotmail.co.uk|Devel-CheckOS|1.42|x86_64-linux-thread-multi-ld|5.11.0 patch 33109|linux|2.6.22-3-amd64|200801292057
1002733|pass|200801|chris@bingosnet.co.uk|Data-Compare|1.18|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200801292109
1006145|pass|200801|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-netbsd-thread-multi-64int|5.11.0 patch 33125|netbsd|4.0|200801310014
1006342|pass|200801|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200801310442
1008067|pass|200801|david@cantrell.org.uk|XML-Tiny|1.11|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200801311506
1008614|pass|200802|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200802010308
1008765|pass|200802|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200802010622
1009502|pass|200802|ferreira@cpan.org|Devel-CheckOS|1.42|i486-linux-gnu|5.8.8|linux|2.6.15.7|200802011257
1010498|pass|200802|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200802020418
1010848|pass|200802|chris@bingosnet.co.uk|Devel-CheckOS|1.42|cygwin-thread-multi-64int|5.11.0 patch 33163|cygwin|1.5.24(0.15642)|200802021012
1010888|pass|200802|feldt@nhn.ou.edu|Data-Compare|1.18|sun4-solaris-thread-multi|5.8.8|solaris|2.8|200802011342
1011511|pass|200802|andy@hexten.net|Devel-CheckOS|1.42|i486-linux-gnu|5.8.8|linux|2.6.15.7|200802021419
1012282|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-64int|5.8.8|linux|2.6.14|200802022222
1012807|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|Class-CanBeA|1.2|i686-linux-64int|5.8.8|linux|2.6.14|200802030414
1013908|pass|200802|rjbs@cpan.org|Data-Compare|1.18|darwin-2level|5.10.0|darwin|9.1.0|200802031115
1014995|pass|200802|claco@chrislaco.com ("Christopher H. Laco")|Data-Compare|1.18|darwin-thread-multi-2level|5.8.6|darwin|8.0|200802031702
1016124|pass|200802|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i686-linux-thread-multi-64int-ld|5.11.0 patch 33228|linux|2.6.16.38|200802041359
1017394|pass|200802|rlane10@wi.rr.com|Data-Compare|1.18|i686-linux|5.10.0|linux|2.6.23.1luckyseven|200802041337
1017716|pass|200802|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200802050423
1021750|fail|200802|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Devel-CheckLib|0.3|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200802062240
1027183|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.11.0|linux|2.6.22-1-k7|200802091719
1030279|pass|200802|autarch@urth.org|Devel-CheckOS|1.42|x86_64-linux-gnu|5.8.8|linux|2.6.15.7|200802101104
1031268|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.8.8 patch 33243|linux|2.6.22-1-k7|200802110552
1031357|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-64int|5.8.8 patch 33243|linux|2.6.22-1-k7|200802110628
1033621|pass|200802|autarch@urth.org|Data-Compare|1.18|x86_64-linux-gnu|5.8.8|linux|2.6.15.7|200802111240
1036645|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.58|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200802131211
1036650|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.58|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200802131213
1036653|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.58|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200802131215
1036660|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.58|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200802131217
1036665|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.58|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200802131220
1036669|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.58|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200802131223
1038238|pass|200802|d@0x64.nl (DEVOGON)|Number-Phone|1.58|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200802140759
1038319|pass|200802|d@0x64.nl (DEVOGON)|Number-Phone|1.58|i686-linux|5.10.0|linux|2.6.22-3-486|200802140931
1038655|pass|200802|david@cantrell.org.uk|Number-Phone|1.58|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200802141118
1038698|pass|200802|david@cantrell.org.uk|Number-Phone|1.58|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200802141225
1038987|pass|200802|david@cantrell.org.uk|Number-Phone|1.58|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200802141532
1040103|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.0|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200802150401
1040104|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.0|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200802150401
1040105|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.0|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200802150401
1040106|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.0|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200802150402
1040107|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.0|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200802150402
1040108|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.0|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200802150402
1040520|pass|200802|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Number-Phone|1.58|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200802151127
1040656|pass|200802|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|CPU-Emulator-Memory|1.0|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200802151141
1040805|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|i386-netbsd-thread-multi-64int|5.11.0 patch 33314|netbsd|4.0|200802151235
1040850|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.0|i686-linux|5.6.2|linux|2.4.27-3-686|200802151310
1040931|pass|200802|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Devel-CheckOS|1.42|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200802151346
1041070|pass|200802|cpantester@hotmail.co.uk|Number-Phone|1.58|x86_64-linux-thread-multi-ld|5.11.0 patch 33317|linux|2.6.22-3-amd64|200802151430
1041385|pass|200802|cpantester@hotmail.co.uk|CPU-Emulator-Memory|1.0|x86_64-linux-thread-multi-ld|5.11.0 patch 33317|linux|2.6.22-3-amd64|200802151500
1041451|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.0|i686-linux|5.8.8|linux|2.4.27-3-686|200802151656
1041529|pass|200802|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-netbsd-thread-multi-64int|5.11.0 patch 33314|netbsd|4.0|200802151718
1041778|pass|200802|chris@bingosnet.co.uk|Data-Compare|1.18|i386-netbsd-thread-multi-64int|5.11.0 patch 33314|netbsd|4.0|200802151914
1041836|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.0|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200802151957
1041864|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200802152026
1041899|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.0|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200802152026
1041986|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.0|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200802152119
1042830|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200802161045
1043635|pass|200802|chris@bingosnet.co.uk|Data-Compare|1.18|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200802161908
1043680|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200802162012
1043681|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200802162012
1043682|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200802162012
1043683|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200802162012
1043684|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200802162013
1043685|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200802162013
1043686|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200802162013
1043687|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200802162014
1043688|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200802162014
1043689|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200802162014
1043725|pass|200802|imacat@mail.imacat.idv.tw|CPU-Emulator-Memory|1.0|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200802170429
1044143|pass|200802|g.grigelionis@computer.org|Devel-CheckOS|1.42|i586-linux-thread-multi|5.8.5|linux|2.6.8.1|200802162101
1044165|pass|200802|g.grigelionis@computer.org|Data-Compare|1.18|i586-linux-thread-multi|5.8.5|linux|2.6.8.1|200802162129
1044281|pass|200802|chris@bingosnet.co.uk|Data-Compare|1.18|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200802170216
1044955|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200802171216
1045272|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|cygwin-thread-multi-64int|5.11.0 patch 33318|cygwin|1.5.24(0.15642)|200802171534
1045402|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.0|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200802171038
1045403|pass|200802|chris@bingosnet.co.uk|Number-Phone|1.58|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200802171639
1045878|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200802171913
1045909|pass|200802|cpantester@hotmail.co.uk|Devel-CheckOS|1.42|x86_64-linux-thread-multi-ld|5.11.0 patch 33325|linux|2.6.22-3-amd64|200802171932
1046019|pass|200802|cpantester@hotmail.co.uk|Data-Compare|1.18|x86_64-linux-thread-multi-ld|5.11.0 patch 33325|linux|2.6.22-3-amd64|200802171959
1046334|pass|200802|chris@bingosnet.co.uk|Devel-CheckOS|1.42|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200802172212
1046421|pass|200802|chris@bingosnet.co.uk|Devel-CheckOS|1.42|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200802172247
1046511|pass|200802|chris@bingosnet.co.uk|Number-Phone|1.58|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200802180017
1046517|pass|200802|chris@bingosnet.co.uk|Number-Phone|1.58|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200802180019
1046529|pass|200802|chris@bingosnet.co.uk|Number-Phone|1.58|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200802180021
1046827|pass|200802|chris@bingosnet.co.uk|Devel-CheckOS|1.42|cygwin-thread-multi-64int|5.11.0 patch 33318|cygwin|1.5.24(0.15642)|200802180336
1047355|pass|200802|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|CPU-Emulator-Memory|1.0|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200802180921
1047665|pass|200802|chris@bingosnet.co.uk|Number-Phone|1.58|alpha-linux-thread-multi|5.11.0 patch 33325|linux|2.6.18-5-alpha-generic|200802181229
1047818|pass|200802|chris@bingosnet.co.uk|Number-Phone|1.58|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200802181324
1048072|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|alpha-linux-thread-multi|5.11.0 patch 33325|linux|2.6.18-5-alpha-generic|200802181442
1048201|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200802181524
1049215|pass|200802|d@0x64.nl (DEVOGON)|CPU-Emulator-Memory|1.0|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200802182238
1049645|pass|200802|chris@bingosnet.co.uk|Data-Compare|1.18|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200802190111
1049667|pass|200802|chris@bingosnet.co.uk|Data-Compare|1.18|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200802190131
1050454|pass|200802|chris@bingosnet.co.uk|Devel-CheckOS|1.42|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200802191305
1050470|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200802191323
1050471|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200802191323
1050472|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200802191324
1050914|pass|200802|chris@bingosnet.co.uk|Devel-CheckOS|1.42|alpha-linux-thread-multi|5.11.0 patch 33325|linux|2.6.18-5-alpha-generic|200802191607
1051021|pass|200802|chris@bingosnet.co.uk|Number-Phone|1.58|i686-linux-thread-multi-64int|5.11.0 patch 33334|linux|2.6.16.38|200802191639
1051219|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|i686-linux-thread-multi-64int|5.11.0 patch 33334|linux|2.6.16.38|200802191819
1051837|pass|200802|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i686-linux-thread-multi-64int|5.11.0 patch 33334|linux|2.6.16.38|200802192329
1052526|pass|200802|chris@bingosnet.co.uk|Number-Phone|1.58|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200802201039
1053541|pass|200802|chris@bingosnet.co.uk|Number-Phone|1.58|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200802201212
1053595|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200802201220
1053936|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200802201357
1054546|pass|200802|d@0x64.nl (DEVOGON)|CPU-Emulator-Memory|1.0|i686-linux|5.10.0|linux|2.6.22-3-486|200802201934
1057443|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200802211259
1059347|pass|200802|d@0x64.nl ("DEVOGON")|Data-Compare|1.18|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200802220038
1060654|pass|200802|stro@cpan.org|Class-CanBeA|1.2|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200802211610
1061638|pass|200802|stro@cpan.org|Data-Hexdumper|1.3|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200802220802
1061838|pass|200802|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200802221641
1061925|pass|200802|stro@cpan.org|Devel-AssertLib|0.1|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200802221016
1061941|pass|200802|stro@cpan.org|Devel-CheckLib|0.2|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200802221018
1062177|pass|200802|d@0x64.nl ("DEVOGON")|XML-DoubleEncodedEntities|1.0|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200802221926
1062313|pass|200802|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200802221916
1062433|pass|200802|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|amd64-freebsd|5.10.0|freebsd|6.2-release|200802222043
1062487|unknown|200802|imacat@mail.imacat.idv.tw|Acme-Licence|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230357
1062489|unknown|200802|imacat@mail.imacat.idv.tw|Acme-Pony|1.1.2|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230358
1062491|pass|200802|imacat@mail.imacat.idv.tw|Acme-Scurvy-Whoreson-BilgeRat|1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230358
1062503|fail|200802|imacat@mail.imacat.idv.tw|Bryar|3.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230401
1062507|pass|200802|imacat@mail.imacat.idv.tw|CPAN-FindDependencies|2.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230402
1062510|pass|200802|imacat@mail.imacat.idv.tw|CPU-Emulator-Memory|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230403
1062512|pass|200802|imacat@mail.imacat.idv.tw|Class-CanBeA|1.2|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230403
1062516|pass|200802|imacat@mail.imacat.idv.tw|Data-Compare|1.18|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230404
1062517|pass|200802|imacat@mail.imacat.idv.tw|Data-Hexdumper|1.3|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230404
1062521|pass|200802|imacat@mail.imacat.idv.tw|Data-Transactional|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230405
1062523|pass|200802|imacat@mail.imacat.idv.tw|Devel-CheckLib|0.3|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230405
1062526|fail|200802|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.42|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230406
1062529|pass|200802|imacat@mail.imacat.idv.tw|File-Overwrite|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230407
1062532|pass|200802|imacat@mail.imacat.idv.tw|Games-Dice-Advanced|1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230407
1062534|pass|200802|imacat@mail.imacat.idv.tw|NestedMap|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230408
1062537|pass|200802|imacat@mail.imacat.idv.tw|Net-Random|2.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230409
1062544|pass|200802|imacat@mail.imacat.idv.tw|Number-Phone-UK-DetailedLocations|1.3|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230412
1062554|pass|200802|imacat@mail.imacat.idv.tw|Palm-TreoPhoneCallDB|1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230413
1062556|pass|200802|imacat@mail.imacat.idv.tw|Statistics-ChiSquare|0.5|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230414
1062561|pass|200802|imacat@mail.imacat.idv.tw|Statistics-SerialCorrelation|1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230414
1062567|pass|200802|imacat@mail.imacat.idv.tw|Sub-WrapPackages|1.2|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230415
1062571|pass|200802|imacat@mail.imacat.idv.tw|Tie-Hash-Longest|1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230415
1062573|pass|200802|imacat@mail.imacat.idv.tw|Tie-Hash-Rank|1.0.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230416
1062576|pass|200802|imacat@mail.imacat.idv.tw|Tie-STDOUT|1.03|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230416
1062581|pass|200802|imacat@mail.imacat.idv.tw|Tie-Scalar-Decay|1.1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230417
1062582|pass|200802|imacat@mail.imacat.idv.tw|XML-DoubleEncodedEntities|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230417
1062583|pass|200802|imacat@mail.imacat.idv.tw|XML-Tiny|1.11|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802230418
1062858|pass|200802|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200802222106
1062932|pass|200802|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|amd64-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200802222236
1062969|pass|200802|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200802222147
1063197|pass|200802|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200802222304
1063404|pass|200802|cpantester@hotmail.co.uk|XML-DoubleEncodedEntities|1.0|x86_64-linux-thread-multi-ld|5.11.0 patch 33346|linux|2.6.22-3-amd64|200802221902
1063577|pass|200802|stro@cpan.org|Games-Dice-Advanced|1.1|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200802230059
1064682|pass|200802|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200802230840
1064703|pass|200802|d@0x64.nl (DEVOGON)|XML-DoubleEncodedEntities|1.0|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200802230944
1066771|pass|200802|srezic@cpan.org|Tie-Hash-Longest|1.1|amd64-freebsd|5.10.0|freebsd|6.2-release|200802232046
1066779|pass|200802|srezic@cpan.org|Tie-Hash-Rank|1.0.1|amd64-freebsd|5.10.0|freebsd|6.2-release|200802232046
1067204|pass|200802|srezic@cpan.org|Tie-Scalar-Decay|1.1.1|amd64-freebsd|5.10.0|freebsd|6.2-release|200802232147
1067751|pass|200802|srezic@cpan.org|Tie-STDOUT|1.03|amd64-freebsd|5.10.0|freebsd|6.2-release|200802232144
1069310|pass|200802|srezic@cpan.org|Data-Hexdumper|1.3|amd64-freebsd|5.10.0|freebsd|6.2-release|200802241047
1070548|pass|200802|srezic@cpan.org|Data-Transactional|1.0|amd64-freebsd|5.10.0|freebsd|6.2-release|200802242001
1070873|pass|200802|srezic@cpan.org|Data-Transactional|1.0|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200802242125
1071241|pass|200802|srezic@cpan.org|Data-Transactional|1.0|amd64-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200802242226
1071744|pass|200802|srezic@cpan.org|CPU-Emulator-Memory|1.0|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200802250102
1071782|pass|200802|srezic@cpan.org|Data-Hexdumper|1.3|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200802250116
1071902|pass|200802|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200802250138
1072310|pass|200802|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200802250340
1073849|pass|200802|chris@bingosnet.co.uk|Data-Compare|1.18|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200802251738
1074148|pass|200802|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|cygwin-thread-multi-64int|5.10.0|cygwin|1.5.24(0.15642)|200802251900
1074970|fail|200802|srezic@cpan.org|CPU-Emulator-Memory|1.0|amd64-freebsd|5.5.5|freebsd|6.2-release|200802252312
1075214|fail|200802|srezic@cpan.org|Number-Phone|1.58|amd64-freebsd|5.5.5|freebsd|6.2-release|200802252348
1075256|pass|200802|srezic@cpan.org|Number-Phone|1.58|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200802260012
1076381|pass|200802|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|alpha-linux-thread-multi|5.11.0 patch 33365|linux|2.6.18-5-alpha-generic|200802260931
1076900|pass|200802|stro@cpan.org|XML-Tiny|1.11|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200802260921
1077232|pass|200802|andy@hexten.net|Data-Hexdumper|1.3|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200802261643
1077815|pass|200802|chris@bingosnet.co.uk|Devel-CheckOS|1.42|alpha-linux-thread-multi|5.11.0 patch 33365|linux|2.6.18-5-alpha-generic|200802261908
1078107|pass|200802|srezic@cpan.org|Number-Phone|1.58|amd64-freebsd|5.10.0|freebsd|6.2-release|200802262125
1078409|pass|200802|srezic@cpan.org|CPU-Emulator-Memory|1.0|amd64-freebsd|5.10.0|freebsd|6.2-release|200802262304
1078638|pass|200802|andy@hexten.net|Data-Hexdumper|1.3|i386-netbsd|5.8.8|netbsd|3.1|200802262241
1078761|pass|200802|srezic@cpan.org|Number-Phone|1.58|amd64-freebsd|5.6.2|freebsd|6.2-release|200802262356
1079212|pass|200802|stro@cpan.org|NestedMap|1.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200802270116
1079566|pass|200802|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.42|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200802271221
1079567|pass|200802|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.42|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802271222
1079627|fail|200802|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.42|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200802271247
1081377|pass|200802|stro@cpan.org|Number-Phone|1.58|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200802271210
1082191|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.581|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200802272252
1082196|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.581|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200802272253
1082201|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.581|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200802272254
1082205|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.581|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200802272254
1082208|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.581|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200802272256
1082209|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.581|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200802272256
1082741|pass|200802|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200802280301
1082847|pass|200802|stro@cpan.org|Statistics-ChiSquare|0.5|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200802280359
1083055|pass|200802|stro@cpan.org|Statistics-SerialCorrelation|1.1|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200802280146
1083328|pass|200802|d@0x64.nl (DEVOGON)|Number-Phone|1.581|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200802280857
1083450|pass|200802|srezic@cpan.org|Number-Phone|1.581|amd64-freebsd|5.6.2|freebsd|6.2-release|200802281005
1083451|pass|200802|srezic@cpan.org|Number-Phone|1.581|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200802281005
1083456|na|200802|srezic@cpan.org|Number-Phone|1.581|amd64-freebsd|5.5.5|freebsd|6.2-release|200802280849
1083536|pass|200802|srezic@cpan.org|Number-Phone|1.581|amd64-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200802281005
1083539|pass|200802|srezic@cpan.org|Number-Phone|1.581|amd64-freebsd|5.10.0|freebsd|6.2-release|200802281005
1083913|pass|200802|david@cantrell.org.uk|Number-Phone|1.581|i686-linux|5.8.8|linux|2.4.27-3-686|200802281253
1083955|pass|200802|chris@bingosnet.co.uk|Number-Phone|1.581|i386-pc-solaris2.11-thread-multi-64int|5.10.0|solaris|2.11|200802281338
1083992|pass|200802|david@cantrell.org.uk|Number-Phone|1.581|i686-linux|5.6.2|linux|2.4.27-3-686|200802281357
1084043|pass|200802|david@cantrell.org.uk|Number-Phone|1.581|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200802281426
1084397|pass|200802|david@cantrell.org.uk|Number-Phone|1.581|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200802281624
1084412|pass|200802|david@cantrell.org.uk|Number-Phone|1.581|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200802281612
1084423|pass|200802|david@cantrell.org.uk|Number-Phone|1.581|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200802281043
1084809|pass|200802|d@0x64.nl|Number-Phone|1.581|i686-linux|5.10.0|linux|2.6.22-3-486|200802282044
1085311|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200802282220
1085315|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200802282220
1085318|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200802282221
1085326|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200802282222
1085331|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200802282222
1085334|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200802282223
1085752|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.001|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200802290020
1085753|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.001|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200802290020
1085754|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.001|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200802290021
1085755|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.001|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200802290021
1085756|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.001|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200802290022
1085758|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.001|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200802290022
1085783|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200802290024
1085784|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200802290024
1085785|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200802290025
1085786|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1001|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200802290025
1085810|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1001|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200802290024
1085811|pass|200802|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1001|i686-linux-64int|5.8.8 patch 32025|linux|2.6.22-1-k7|200802290024
1085818|pass|200802|stro@cpan.org|Tie-Hash-Longest|1.1|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200802290133
1085828|pass|200802|stro@cpan.org|Tie-Hash-Rank|1.0.1|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200802290135
1085940|pass|200802|stro@cpan.org|Tie-Scalar-Decay|1.1.1|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200802290003
1086158|pass|200802|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Number-Phone|1.581|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200802290102
1086808|pass|200802|imacat@mail.imacat.idv.tw|CPU-Emulator-Memory|1.1001|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802291357
1086809|pass|200802|imacat@mail.imacat.idv.tw|Number-Phone|1.581|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200802291358
1086909|pass|200802|d@0x64.nl (DEVOGON)|CPU-Emulator-Memory|1.1001|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200802290725
1086987|pass|200802|stro@cpan.org|XML-DoubleEncodedEntities|1.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200802290139
1087243|pass|200802|srezic@cpan.org|Devel-CheckOS|1.42|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200802290837
1087254|na|200802|srezic@cpan.org|CPU-Emulator-Memory|1.1001|amd64-freebsd|5.5.5|freebsd|6.2-release|200802290841
1087257|pass|200802|srezic@cpan.org|CPU-Emulator-Memory|1.1001|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200802290841
1087392|pass|200802|imacat@mail.imacat.idv.tw|Number-Phone|1.581|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200802291728
1087462|pass|200802|srezic@cpan.org|Devel-CheckOS|1.42|amd64-freebsd|5.10.0|freebsd|6.2-release|200802290958
1087479|pass|200802|srezic@cpan.org|CPU-Emulator-Memory|1.1001|amd64-freebsd|5.10.0|freebsd|6.2-release|200802291017
1087738|pass|200802|srezic@cpan.org|CPAN-FindDependencies|2.0|amd64-freebsd|5.10.0|freebsd|6.2-release|200802291223
1087808|pass|200802|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|CPU-Emulator-Memory|1.1001|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200802291206
1087820|pass|200802|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Number-Phone|1.581|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200802291209
1088420|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.001|i686-linux|5.8.8|linux|2.4.27-3-686|200802291542
1088421|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.1|i686-linux|5.8.8|linux|2.4.27-3-686|200802291542
1088423|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.1001|i686-linux|5.8.8|linux|2.4.27-3-686|200802291542
1088438|fail|200802|srezic@cpan.org|Bryar|3.0|amd64-freebsd|5.10.0|freebsd|6.2-release|200802291520
1088488|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.001|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200802291646
1088490|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.1|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200802291647
1088491|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.1001|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200802291647
1088496|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.001|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200802291626
1088497|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.001|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200802291651
1088498|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.1|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200802291628
1088499|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.1|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200802291652
1088501|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200802291652
1088502|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.001|i686-linux|5.6.2|linux|2.4.27-3-686|200802291653
1088503|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.1|i686-linux|5.6.2|linux|2.4.27-3-686|200802291653
1088504|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.1001|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200802291629
1088505|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.1001|i686-linux|5.6.2|linux|2.4.27-3-686|200802291654
1088757|pass|200802|srezic@cpan.org|Devel-CheckOS|1.42|amd64-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200802291930
1089198|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200802292052
1089200|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200802292052
1089202|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200802292053
1089203|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200802292053
1089205|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200802292054
1089207|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200802292054
1089209|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200802292054
1089210|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200802292055
1089212|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200802292055
1089213|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200802292056
1089296|pass|200802|srezic@cpan.org|CPU-Emulator-Memory|1.1001|amd64-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200802292222
1089383|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200802292310
1089384|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200802292311
1089386|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200802292311
1089387|pass|200802|chris@bingosnet.co.uk|Number-Phone|1.581|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200802292312
1089556|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200802292338
1089559|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200802292339
1089561|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200802292339
1089562|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200802292339
1089563|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200802292340
1089565|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200802292340
1089566|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200802292340
1089568|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200802292341
1089569|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200802292341
1089571|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200802292342
1089572|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200802292342
1089573|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200802292342
1089574|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200802292343
1089576|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200802292343
1089578|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200802292344
1089580|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200802292344
1089581|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200802292344
1089583|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200802292345
1089586|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200802292345
1089588|pass|200802|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200802292345
1089617|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.001|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200802292355
1089621|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.1|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200802292356
1089625|pass|200802|david@cantrell.org.uk|CPU-Emulator-Memory|1.1001|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200802292356
1089803|pass|200803|icerider70@gmail.com ("ICERIDER")|Number-Phone|1.581|i386-linux|5.8.8|linux|2.6.18-53.el5|200803010151
1089978|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|i686-cygwin-thread-multi-64int|5.11.0 patch 33400|cygwin|1.5.24(0.15642)|200803010318
1089979|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|i686-cygwin-thread-multi-64int|5.11.0 patch 33400|cygwin|1.5.24(0.15642)|200803010319
1089983|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i686-cygwin-thread-multi-64int|5.11.0 patch 33400|cygwin|1.5.24(0.15642)|200803010319
1091103|pass|200803|srezic@cpan.org|CPU-Emulator-Memory|1.1001|amd64-freebsd|5.6.2|freebsd|6.2-release|200803011302
1091197|pass|200803|d@0x64.nl (DEVOGON)|CPU-Emulator-Memory|1.1001|i686-linux|5.10.0|linux|2.6.22-3-486|200803011427
1091212|pass|200803|d@0x64.nl (DEVOGON)|Number-Phone|1.581|i686-linux|5.10.0|linux|2.6.22-3-486|200803011443
1091352|pass|200803|icerider70@gmail.com ("ICERIDER")|CPU-Emulator-Memory|1.0|i386-linux|5.8.8|linux|2.6.18-53.el5|200803010851
1091742|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200803011746
1091891|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i686-cygwin-thread-multi-64int|5.11.0 patch 33400|cygwin|1.5.24(0.15642)|200803011756
1092328|fail|200803|andy@hexten.net|Devel-CheckOS|1.42|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200803012024
1092354|pass|200803|imacat@mail.imacat.idv.tw|CPU-Emulator-Memory|1.1|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200803020435
1092383|pass|200803|imacat@mail.imacat.idv.tw|CPU-Emulator-Memory|1.001|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200803020446
1092386|pass|200803|imacat@mail.imacat.idv.tw|CPU-Emulator-Memory|1.1001|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200803020446
1092659|pass|200803|srezic@cpan.org|Palm-TreoPhoneCallDB|1.1|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200803012108
1092959|pass|200803|srezic@cpan.org|Devel-CheckLib|0.3|amd64-freebsd|5.10.0|freebsd|6.2-release|200803020039
1093139|pass|200803|srezic@cpan.org|XML-Tiny|1.11|amd64-freebsd|5.10.0|freebsd|6.2-release|200803020102
1094820|pass|200803|srezic@cpan.org|File-Find-Rule-Permissions|1.3|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200803021432
1094966|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 33408|openbsd|4.2|200803021600
1094968|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 33408|openbsd|4.2|200803021600
1094970|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 33408|openbsd|4.2|200803021601
1094978|pass|200803|chris@bingosnet.co.uk|Number-Phone|1.581|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 33408|openbsd|4.2|200803021604
1095292|pass|200803|d@0x64.nl (DEVOGON)|Devel-CheckOS|1.42|i686-linux-thread-multi|5.11.0 patch 33261|linux|2.6.22-3-486|200803021652
1095759|pass|200803|d@0x64.nl (DEVOGON)|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi|5.11.0 patch 33261|linux|2.6.22-3-486|200803021909
1095788|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200803021917
1095789|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200803021917
1095791|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200803021918
1095803|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200803021918
1095808|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200803021919
1095809|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200803021919
1095811|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200803021920
1095813|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200803021920
1095815|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200803021921
1095821|pass|200803|chris@bingosnet.co.uk|Number-Phone|1.581|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200803021922
1095822|pass|200803|chris@bingosnet.co.uk|Number-Phone|1.581|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200803021922
1095823|pass|200803|chris@bingosnet.co.uk|Number-Phone|1.581|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200803021923
1096373|pass|200803|srezic@cpan.org|Data-Hexdumper|1.3|amd64-freebsd|5.6.2|freebsd|6.2-release|200803022143
1096416|pass|200803|srezic@cpan.org|Tie-STDOUT|1.03|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200803022230
1096935|pass|200803|david@cantrell.org.uk|CPU-Emulator-Memory|1.001|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200803021712
1096946|pass|200803|david@cantrell.org.uk|CPU-Emulator-Memory|1.1|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200803021713
1096963|pass|200803|david@cantrell.org.uk|CPU-Emulator-Memory|1.1001|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200803021714
1096964|pass|200803|icerider70@gmail.com ("ICERIDER")|CPU-Emulator-Memory|1.1001|i386-linux|5.8.8|linux|2.6.18-53.el5|200803021714
1097054|pass|200803|srezic@cpan.org|Palm-TreoPhoneCallDB|1.1|amd64-freebsd|5.10.0|freebsd|6.2-release|200803030000
1097084|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.42|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 33408|openbsd|4.2|200803030214
1098112|pass|200803|stro@cpan.org|Number-Phone|1.581|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200803031135
1098687|pass|200803|d@0x64.nl ("DEVOGON")|CPU-Emulator-Memory|1.1001|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200803031242
1099024|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|alpha-linux-thread-multi|5.11.0 patch 33411|linux|2.6.18-5-alpha-generic|200803031447
1099026|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|alpha-linux-thread-multi|5.11.0 patch 33411|linux|2.6.18-5-alpha-generic|200803031448
1099029|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|alpha-linux-thread-multi|5.11.0 patch 33411|linux|2.6.18-5-alpha-generic|200803031448
1099031|pass|200803|chris@bingosnet.co.uk|Number-Phone|1.581|alpha-linux-thread-multi|5.11.0 patch 33411|linux|2.6.18-5-alpha-generic|200803031450
1099628|pass|200803|stro@cpan.org|CPAN-FindDependencies|2.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200803032027
1099664|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200803040107
1099666|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200803040108
1099669|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200803040108
1099674|pass|200803|chris@bingosnet.co.uk|Number-Phone|1.581|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200803040111
1100247|pass|200803|srezic@cpan.org|Data-Transactional|1.0|amd64-freebsd|5.6.2|freebsd|6.2-release|200803032215
1100248|na|200803|srezic@cpan.org|Tie-STDOUT|1.03|amd64-freebsd|5.6.2|freebsd|6.2-release|200803032215
1100533|pass|200803|stro@cpan.org|Data-Compare|0.17|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200803040008
1100537|pass|200803|stro@cpan.org|Data-Compare|1.18|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200803040009
1100802|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux|5.11.0 patch 33423|linux|2.6.22-1-k7|200803032326
1100928|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.3|i686-linux|5.11.0 patch 33423|linux|2.6.22-1-k7|200803040020
1101250|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux|5.11.0 patch 33423|linux|2.6.22-1-k7|200803040142
1101756|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux|5.11.0 patch 33423|linux|2.6.22-1-k7|200803040434
1101874|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.42|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200803041208
1102093|pass|200803|stro@cpan.org|Data-Transactional|1.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200803040643
1103433|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux|5.11.0 patch 33423|linux|2.6.22-1-k7|200803040929
1103711|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-thread-multi|5.8.8 patch 33430|linux|2.6.22-1-k7|200803041016
1104229|pass|200803|Jost.Krieger@ruhr-uni-bochum.de|Devel-CheckOS|1.42|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200803041047
1104296|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-thread-multi|5.8.8 patch 33430|linux|2.6.22-1-k7|200803041207
1104629|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.3|i686-linux-thread-multi|5.8.8 patch 33430|linux|2.6.22-1-k7|200803041241
1104915|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.42|alpha-linux-thread-multi|5.11.0 patch 33411|linux|2.6.18-5-alpha-generic|200803041323
1105211|pass|200803|stro@cpan.org|Devel-CheckOS|1.42|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200803041613
1105438|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.8.8|linux|2.6.14|200803041602
1105779|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi|5.8.8 patch 33430|linux|2.6.22-1-k7|200803041713
1105891|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-thread-multi|5.8.8 patch 33430|linux|2.6.22-1-k7|200803041708
1105955|pass|200803|andy@hexten.net|Number-Phone|1.581|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200803041654
1107340|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-thread-multi-64int|5.10.0 patch 33412|linux|2.6.22-1-k7|200803042222
1107362|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux-thread-multi-64int|5.10.0 patch 33412|linux|2.6.22-1-k7|200803042255
1107366|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200803042256
1107381|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux-thread-multi|5.8.8 patch 33430|linux|2.6.22-1-k7|200803042257
1107382|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux-64int|5.8.8|linux|2.6.14|200803042257
1107384|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200803042259
1107386|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200803042259
1107387|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200803042300
1107408|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux-64int|5.8.8 patch 33243|linux|2.6.22-1-k7|200803042257
1107423|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-thread-multi-64int|5.10.0 patch 33412|linux|2.6.22-1-k7|200803042251
1107614|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi-64int|5.10.0 patch 33412|linux|2.6.22-1-k7|200803042331
1108058|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200803042331
1108061|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200803042332
1108065|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200803042333
1108080|pass|200803|chris@bingosnet.co.uk|Number-Phone|1.581|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200803042335
1109224|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int|5.10.0 patch 33412|linux|2.6.22-1-k7|200803050548
1109254|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200803050457
1109259|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200803050458
1109263|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200803050459
1109269|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200803050500
1109275|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200803050501
1109279|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200803050501
1109282|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200803050502
1109284|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200803050503
1109288|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200803050504
1109292|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200803050504
1109373|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux|5.11.0 patch 33423|linux|2.6.22-1-k7|200803050617
1109863|pass|200803|srezic@cpan.org|Devel-CheckLib|0.4|amd64-freebsd|5.6.2|freebsd|6.2-release|200803050722
1109864|pass|200803|srezic@cpan.org|Devel-CheckLib|0.4|amd64-freebsd|5.10.0|freebsd|6.2-release|200803050722
1109984|pass|200803|srezic@cpan.org|Devel-CheckLib|0.4|amd64-freebsd|5.5.5|freebsd|6.2-release|200803050836
1110587|pass|200803|imacat@mail.imacat.idv.tw|Devel-CheckLib|0.4|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200803051857
1110726|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|alpha-linux-thread-multi|5.11.0 patch 33411|linux|2.6.18-5-alpha-generic|200803051124
1110843|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|alpha-linux-thread-multi|5.11.0 patch 33411|linux|2.6.18-5-alpha-generic|200803051203
1111089|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200803051404
1111091|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200803051405
1111095|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200803051407
1111102|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200803051408
1111109|pass|200803|chris@bingosnet.co.uk|Number-Phone|1.581|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200803051410
1111240|pass|200803|andy@hexten.net|Devel-CheckLib|0.4|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200803051449
1111292|pass|200803|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Devel-CheckLib|0.4|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200803051454
1111491|pass|200803|andy@hexten.net|Number-Phone|1.581|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200803051556
1111498|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200803051558
1111809|pass|200803|david@cantrell.org.uk|Devel-CheckLib|0.4|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200803051707
1111850|pass|200803|david@cantrell.org.uk|Devel-CheckLib|0.4|i686-linux|5.6.2|linux|2.4.27-3-686|200803051716
1111851|pass|200803|david@cantrell.org.uk|Devel-CheckLib|0.4|i686-linux|5.8.8|linux|2.4.27-3-686|200803051717
1112005|pass|200803|david@cantrell.org.uk|Devel-CheckLib|0.4|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200803051747
1112021|pass|200803|srezic@cpan.org|File-Find-Rule-Permissions|1.3|amd64-freebsd|5.10.0|freebsd|6.2-release|200803051839
1112103|pass|200803|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|CPU-Emulator-Memory|1.1001|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200803051758
1112105|pass|200803|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Devel-CheckLib|0.4|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200803051759
1112985|pass|200803|srezic@cpan.org|Devel-CheckLib|0.4|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200803052123
1113234|pass|200803|d@0x64.nl ("DEVOGON")|Number-Phone|1.581|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200803052211
1113289|pass|200803|d@0x64.nl (DEVOGON)|Devel-CheckLib|0.4|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200803052224
1113959|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux|5.8.8 patch 33436|linux|2.6.22-1-k7|200803060059
1114475|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux|5.8.8 patch 33436|linux|2.6.22-1-k7|200803060313
1115303|pass|200803|stro@cpan.org|Net-Random|2.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200803060814
1115390|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-64int|5.10.0 patch 33443|linux|2.6.22-1-k7|200803060747
1115424|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 33443|linux|2.6.22-1-k7|200803060744
1117012|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.10.0 patch 33443|linux|2.6.22-1-k7|200803061219
1117025|pass|200803|stro@cpan.org|Number-Phone|1.581|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200803061327
1117073|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.001|i386-netbsd-thread-multi-64int|5.11.0 patch 33445|netbsd|4.0|200803061137
1117075|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1|i386-netbsd-thread-multi-64int|5.11.0 patch 33445|netbsd|4.0|200803061137
1117077|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i386-netbsd-thread-multi-64int|5.11.0 patch 33445|netbsd|4.0|200803061138
1117079|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|i386-netbsd-thread-multi-64int|5.11.0 patch 33445|netbsd|4.0|200803061138
1117086|pass|200803|stro@cpan.org|Number-Phone-UK-DetailedLocations|1.3|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200803061340
1117231|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|i386-netbsd-thread-multi-64int|5.11.0 patch 33445|netbsd|4.0|200803061205
1117687|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200803061318
1117693|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200803061319
1117694|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200803061320
1118024|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.42|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200803061412
1118753|pass|200803|d@0x64.nl ("DEVOGON")|Devel-CheckLib|0.4|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200803061721
1118838|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 33443|linux|2.6.22-1-k7|200803061739
1119041|pass|200803|d@0x64.nl ("DEVOGON")|CPAN-FindDependencies|2.0|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200803061805
1119501|pass|200803|imacat@mail.imacat.idv.tw|Devel-CheckLib|0.4|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200803070159
1119566|pass|200803|stro@cpan.org|Palm-TreoPhoneCallDB|1.1|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200803062004
1119718|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux-64int|5.10.0 patch 33443|linux|2.6.22-1-k7|200803061924
1119800|pass|200803|andy@hexten.net|Devel-CheckLib|0.4|i686-linux-thread-multi|5.10.0|linux|2.6.20-16-server|200803061839
1119955|pass|200803|andy@hexten.net|Data-Compare|1.18|i686-linux-thread-multi|5.10.0|linux|2.6.20-16-server|200803061932
1120096|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i386-netbsd-thread-multi-64int|5.11.0 patch 33445|netbsd|4.0|200803062000
1120629|pass|200803|srezic@cpan.org|Devel-CheckOS|1.42|amd64-freebsd|5.6.2|freebsd|6.2-release|200803062319
1120958|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-thread-multi|5.11.0 patch 33450|linux|2.6.22-1-k7|200803070001
1121130|pass|200803|david@cantrell.org.uk|Devel-CheckLib|0.4|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200803062341
1121862|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi|5.11.0 patch 33450|linux|2.6.22-1-k7|200803070301
1122380|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux-thread-multi|5.11.0 patch 33450|linux|2.6.22-1-k7|200803070500
1122639|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-thread-multi|5.11.0 patch 33450|linux|2.6.22-1-k7|200803070021
1122686|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-thread-multi|5.11.0 patch 33450|linux|2.6.22-1-k7|200803070547
1123451|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200803070919
1123847|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux-thread-multi-64int|5.8.8 patch 33449|linux|2.6.22-1-k7|200803071224
1123868|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-thread-multi-64int|5.8.8 patch 33449|linux|2.6.22-1-k7|200803071229
1124313|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-thread-multi-64int|5.8.8 patch 33449|linux|2.6.22-1-k7|200803071514
1124659|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi-64int|5.8.8 patch 33449|linux|2.6.22-1-k7|200803071714
1124710|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int|5.8.8 patch 33449|linux|2.6.22-1-k7|200803071711
1125157|pass|200803|d@0x64.nl|Devel-CheckLib|0.4|i686-linux|5.10.0|linux|2.6.22-3-486|200803071854
1125298|pass|200803|d@0x64.nl|Data-Compare|1.18|i686-linux|5.10.0|linux|2.6.22-3-486|200803071916
1126853|fail|200803|d@0x64.nl ("DEVOGON")|Bryar|3.0|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200803072259
1126885|pass|200803|david@cantrell.org.uk|Devel-CheckLib|0.4|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200803071606
1126910|pass|200803|d@0x64.nl ("DEVOGON")|Devel-CheckOS|1.42|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200803072306
1126958|pass|200803|david@cantrell.org.uk|Devel-CheckLib|0.4|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200803072218
1127166|pass|200803|srezic@cpan.org|Devel-CheckLib|0.4|amd64-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200803080007
1127202|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|i686-cygwin-thread-multi-64int|5.11.0 patch 33450|cygwin|1.5.24(0.15642)|200803072324
1127604|pass|200803|d@0x64.nl|CPU-Emulator-Memory|1.1001|i686-linux|5.10.0|linux|2.6.22-3-486|200803080142
1127792|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|i686-cygwin-thread-multi-64int|5.11.0 patch 33450|cygwin|1.5.24(0.15642)|200803080054
1128559|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200803080351
1128897|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux|5.11.0 patch 33451|linux|2.6.22-1-k7|200803080630
1129275|pass|200803|d@0x64.nl (DEVOGON)|Devel-CheckLib|0.4|i686-linux|5.10.0|linux|2.6.22-3-486|200803080819
1129286|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.0|darwin-2level|5.8.8|darwin|8.10.1|200803080814
1129634|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux|5.11.0 patch 33451|linux|2.6.22-1-k7|200803080853
1129685|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux|5.11.0 patch 33451|linux|2.6.22-1-k7|200803080905
1130112|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux|5.11.0 patch 33451|linux|2.6.22-1-k7|200803080921
1131122|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i686-cygwin-thread-multi-64int|5.11.0 patch 33450|cygwin|1.5.24(0.15642)|200803081156
1131719|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux|5.11.0 patch 33451|linux|2.6.22-1-k7|200803081236
1132402|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux|5.11.0 patch 33452|linux|2.6.22-1-k7|200803081702
1132651|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux|5.11.0 patch 33452|linux|2.6.22-1-k7|200803081738
1132947|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux|5.11.0 patch 33452|linux|2.6.22-1-k7|200803081626
1133084|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|darwin-2level|5.8.8|darwin|8.10.1|200803081604
1133252|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux|5.11.0 patch 33452|linux|2.6.22-1-k7|200803081858
1133258|pass|200803|srezic@cpan.org|File-Find-Rule-Permissions|1.3|amd64-freebsd|5.6.2|freebsd|6.2-release|200803081629
1133307|pass|200803|srezic@cpan.org|Devel-CheckOS|1.42|amd64-freebsd-thread-multi|5.8.8|freebsd|6.2-release|200803081632
1134224|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux|5.11.0 patch 33452|linux|2.6.22-1-k7|200803082210
1134295|pass|200803|d@0x64.nl|XML-DoubleEncodedEntities|1.0|i686-linux|5.10.0|linux|2.6.22-3-486|200803082233
1134317|pass|200803|stro@cpan.org|Sub-WrapPackages|1.2|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200803082331
1135446|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200803090046
1135548|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200803090105
1136200|pass|200803|srezic@cpan.org|Net-Random|2.0|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200803090407
1137541|pass|200803|d@0x64.nl ("DEVOGON")|XML-Tiny|1.11|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200803091228
1137905|pass|200803|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.4|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200803090836
1137986|pass|200803|srezic@cpan.org|Data-Compare|1.18|amd64-freebsd-thread-multi|5.8.8|freebsd|6.2-release|200803091335
1138137|pass|200803|srezic@cpan.org|Devel-CheckLib|0.4|amd64-freebsd-thread-multi|5.8.8|freebsd|6.2-release|200803091353
1138692|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux-64int|5.8.8 patch 33454|linux|2.6.22-1-k7|200803091548
1139225|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-64int|5.8.8 patch 33454|linux|2.6.22-1-k7|200803091727
1139601|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.8.8 patch 33454|linux|2.6.22-1-k7|200803091824
1139874|pass|200803|srezic@cpan.org|Number-Phone-UK-DetailedLocations|1.3|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200803091845
1140020|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|cygwin-thread-multi-64int|5.10.0|cygwin|1.5.24(0.15642)|200803091835
1140224|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|cygwin-thread-multi-64int|5.10.0|cygwin|1.5.24(0.15642)|200803091919
1140377|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200803091942
1140430|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 33454|linux|2.6.22-1-k7|200803092053
1140490|pass|200803|srezic@cpan.org|Palm-TreoPhoneCallDB|1.1|amd64-freebsd|5.6.2|freebsd|6.2-release|200803092059
1140822|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-64int|5.8.8 patch 33454|linux|2.6.22-1-k7|200803092158
1141239|pass|200803|d@0x64.nl (DEVOGON)|Devel-CheckLib|0.4|i686-linux-thread-multi|5.11.0 patch 33261|linux|2.6.22-3-486|200803092254
1141354|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 33454|linux|2.6.22-1-k7|200803100000
1141393|pass|200803|d@0x64.nl (DEVOGON)|Data-Compare|1.18|i686-linux-thread-multi|5.11.0 patch 33261|linux|2.6.22-3-486|200803100018
1141448|pass|200803|andy@hexten.net|Devel-CheckLib|0.4|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200803092324
1142886|pass|200803|daxim@cpan.org|XML-DoubleEncodedEntities|1.0|x86_64-linux|5.8.8|linux|2.6.20-gentoo-r8|200803092241
1143297|pass|200803|andy@hexten.net|Devel-CheckLib|0.4|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200803101156
1143388|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-thread-multi|5.11.0 patch 33457|linux|2.6.22-1-k7|200803101336
1144603|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|alpha-linux-thread-multi|5.11.0 patch 33456|linux|2.6.18-5-alpha-generic|200803101704
1144632|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-thread-multi|5.11.0 patch 33457|linux|2.6.22-1-k7|200803101810
1144753|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|alpha-linux-thread-multi|5.11.0 patch 33456|linux|2.6.18-5-alpha-generic|200803101728
1145110|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-thread-multi|5.11.0 patch 33457|linux|2.6.22-1-k7|200803101835
1146007|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-thread-multi|5.11.0 patch 33457|linux|2.6.22-1-k7|200803102130
1146153|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux-thread-multi|5.11.0 patch 33457|linux|2.6.22-1-k7|200803102210
1146201|pass|200803|schwern@pobox.com|Devel-CheckLib|0.4|darwin-thread-multi-2level|5.8.8|darwin|8.8.1|200803101116
1146230|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi|5.11.0 patch 33457|linux|2.6.22-1-k7|200803102214
1147736|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|i686-linux-thread-multi-64int|5.11.0 patch 33467|linux|2.6.16.38|200803110257
1147779|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|i686-linux-thread-multi-64int|5.11.0 patch 33467|linux|2.6.16.38|200803110307
1147933|fail|200803|imacat@mail.imacat.idv.tw|Palm-SMS|0.02|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200803111159
1148083|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|darwin-2level|5.8.8|darwin|8.10.1|200803110610
1148832|pass|200803|stro@cpan.org|CPU-Emulator-Memory|1.0|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200803111132
1148834|pass|200803|stro@cpan.org|CPU-Emulator-Memory|1.001|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200803111133
1148837|pass|200803|stro@cpan.org|CPU-Emulator-Memory|1.1|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200803111133
1148839|pass|200803|stro@cpan.org|CPU-Emulator-Memory|1.1001|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200803111133
1148988|pass|200803|stro@cpan.org|CPU-Emulator-Memory|1.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200803111231
1148993|pass|200803|stro@cpan.org|CPU-Emulator-Memory|1.001|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200803111232
1148994|pass|200803|stro@cpan.org|CPU-Emulator-Memory|1.1|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200803111233
1148995|pass|200803|stro@cpan.org|CPU-Emulator-Memory|1.1001|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200803111234
1149737|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.42|i686-linux-thread-multi-64int|5.11.0 patch 33467|linux|2.6.16.38|200803111346
1150508|pass|200803|d@0x64.nl ("DEVOGON")|File-Find-Rule-Permissions|1.3|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200803111731
1151369|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.42|alpha-linux-thread-multi|5.11.0 patch 33456|linux|2.6.18-5-alpha-generic|200803111923
1151818|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-thread-multi|5.11.0 patch 33470|linux|2.6.22-1-k7|200803112101
1152039|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-thread-multi|5.11.0 patch 33470|linux|2.6.22-1-k7|200803112202
1152397|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux-thread-multi|5.11.0 patch 33470|linux|2.6.22-1-k7|200803112317
1152971|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-thread-multi|5.11.0 patch 33470|linux|2.6.22-1-k7|200803120059
1153370|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-thread-multi|5.11.0 patch 33470|linux|2.6.22-1-k7|200803120240
1153517|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi|5.11.0 patch 33470|linux|2.6.22-1-k7|200803120254
1153896|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.4|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200803120341
1154487|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-thread-multi-64int|5.11.0 patch 33485|linux|2.6.22-1-k7|200803120848
1155279|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi-64int|5.11.0 patch 33485|linux|2.6.22-1-k7|200803121135
1155343|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-thread-multi-64int|5.11.0 patch 33485|linux|2.6.22-1-k7|200803121113
1155416|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int|5.11.0 patch 33485|linux|2.6.22-1-k7|200803121030
1155581|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-thread-multi-64int|5.11.0 patch 33485|linux|2.6.22-1-k7|200803121207
1156349|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux-thread-multi-64int|5.11.0 patch 33485|linux|2.6.22-1-k7|200803121248
1157352|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 33443|linux|2.6.22-1-k7|200803122103
1157355|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-thread-multi-64int|5.10.0 patch 33412|linux|2.6.22-1-k7|200803122104
1157358|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200803122104
1157363|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-thread-multi|5.8.8 patch 33430|linux|2.6.22-1-k7|200803122105
1157366|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 33243|linux|2.6.22-1-k7|200803122105
1157369|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200803122106
1157374|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8|linux|2.6.14|200803122107
1157385|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200803122107
1157390|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200803122107
1157394|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200803122108
1157434|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.10.0 patch 33443|linux|2.6.22-1-k7|200803122119
1157436|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-thread-multi-64int|5.10.0 patch 33412|linux|2.6.22-1-k7|200803122120
1157438|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200803122120
1157439|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-thread-multi|5.8.8 patch 33430|linux|2.6.22-1-k7|200803122121
1157441|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.8.8 patch 33243|linux|2.6.22-1-k7|200803122121
1157443|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200803122122
1157447|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.8.8|linux|2.6.14|200803122122
1157449|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200803122123
1157450|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200803122124
1157454|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200803122125
1157638|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|darwin-2level|5.8.8|darwin|8.10.1|200803122209
1157643|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|darwin-2level|5.8.8|darwin|8.10.1|200803122205
1158128|pass|200803|d@0x64.nl (DEVOGON)|Devel-CheckLib|0.5|i686-linux-thread-multi|5.11.0 patch 33261|linux|2.6.22-3-486|200803122231
1158271|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1001|i686-linux-64int|5.10.0 patch 33443|linux|2.6.22-1-k7|200803122351
1158283|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-64int|5.10.0 patch 33412|linux|2.6.22-1-k7|200803122353
1158286|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|0.9|i686-linux-thread-multi-64int|5.10.0 patch 33412|linux|2.6.22-1-k7|200803122354
1158289|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|0.9|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200803122355
1158291|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi|5.8.8 patch 33430|linux|2.6.22-1-k7|200803122355
1158292|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|0.9|i686-linux-thread-multi|5.8.8 patch 33430|linux|2.6.22-1-k7|200803122356
1158310|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|0.9|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200803122359
1158312|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1001|i686-linux-64int|5.8.8|linux|2.6.14|200803130000
1158313|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|0.9|i686-linux-64int|5.8.8|linux|2.6.14|200803130001
1158316|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|0.9|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200803130003
1158400|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|0.9|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200803130004
1158411|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1001|i686-linux-64int|5.8.8 patch 33243|linux|2.6.22-1-k7|200803122357
1158412|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1001|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200803122359
1158422|pass|200803|g.grigelionis@computer.org|Devel-CheckOS|1.43|i586-linux-thread-multi|5.8.8|linux|2.6.16|200803122325
1158434|pass|200803|g.grigelionis@computer.org|Devel-CheckOS|1.43|i386-linux|5.8.0|linux|2.6.9-22.18.bz155725.elsmp|200803130003
1158441|pass|200803|g.grigelionis@computer.org|Devel-CheckOS|1.43|x86_64-linux-thread-multi|5.8.3|linux|2.6.4|200803130011
1158519|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|0.9|i686-linux-64int|5.8.8 patch 33243|linux|2.6.22-1-k7|200803122357
1158521|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|0.9|i686-linux-64int|5.10.0 patch 33443|linux|2.6.22-1-k7|200803122352
1158548|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.42|i686-linux-thread-multi-64int|5.8.8 patch 33497|linux|2.6.22-1-k7|200803122351
1158832|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-thread-multi-64int|5.8.8 patch 33497|linux|2.6.22-1-k7|200803130127
1159673|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.4|i686-linux-thread-multi-64int|5.8.8 patch 33497|linux|2.6.22-1-k7|200803130437
1160079|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi-64int|5.8.8 patch 33497|linux|2.6.22-1-k7|200803130539
1160098|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int|5.8.8 patch 33497|linux|2.6.22-1-k7|200803130545
1160108|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-thread-multi-64int|5.8.8 patch 33497|linux|2.6.22-1-k7|200803130546
1160637|pass|200803|hohokus@gmail.com|Devel-CheckOS|1.43|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200803121942
1160646|pass|200803|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|CPU-Emulator-Memory|1.1001|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200803130956
1160656|pass|200803|artur+cpan@niif.spb.su|Devel-CheckOS|1.43|i386-linux|5.8.8|linux|2.6.18-53.1.4.el5.bz358661.1|200803130833
1160961|pass|200803|david@cantrell.org.uk|CPU-Emulator-Z80|0.9|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200803130957
1160963|pass|200803|david@cantrell.org.uk|Devel-CheckLib|0.5|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200803130957
1160965|pass|200803|d@0x64.nl|CPU-Emulator-Z80|0.9|i686-linux|5.10.0|linux|2.6.22-3-486|200803131104
1160988|pass|200803|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Devel-CheckOS|1.43|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200803130949
1160990|pass|200803|d@0x64.nl|Devel-CheckOS|1.43|i686-linux|5.10.0|linux|2.6.22-3-486|200803131109
1160994|pass|200803|d@0x64.nl|Devel-CheckLib|0.5|i686-linux|5.10.0|linux|2.6.22-3-486|200803131111
1161055|pass|200803|david@cantrell.org.uk|Devel-CheckOS|1.43|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200803131026
1161078|pass|200803|david@cantrell.org.uk|CPU-Emulator-Z80|0.9|i686-linux|5.8.8|linux|2.4.27-3-686|200803131035
1161086|pass|200803|david@cantrell.org.uk|Devel-CheckLib|0.5|i686-linux|5.8.8|linux|2.4.27-3-686|200803131036
1161091|pass|200803|david@cantrell.org.uk|CPU-Emulator-Z80|0.9|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200803131041
1161093|pass|200803|david@cantrell.org.uk|CPU-Emulator-Z80|0.9|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200803131039
1161099|pass|200803|david@cantrell.org.uk|Devel-CheckLib|0.5|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200803131044
1161521|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200803131308
1161673|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux|5.10.0 patch 33501|linux|2.6.22-1-k7|200803131453
1161703|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux|5.10.0 patch 33501|linux|2.6.22-1-k7|200803131500
1161788|pass|200803|david@cantrell.org.uk|Devel-CheckLib|0.5|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200803131414
1161794|pass|200803|david@cantrell.org.uk|Devel-CheckOS|1.43|i686-linux|5.8.8|linux|2.4.27-3-686|200803131415
1161800|pass|200803|david@cantrell.org.uk|Devel-CheckOS|1.43|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200803131418
1161845|pass|200803|david@cantrell.org.uk|Devel-CheckOS|1.43|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200803131424
1161859|pass|200803|david@cantrell.org.uk|Devel-CheckLib|0.5|i686-linux|5.6.2|linux|2.4.27-3-686|200803131427
1161889|pass|200803|david@cantrell.org.uk|Devel-CheckOS|1.43|i686-linux|5.6.2|linux|2.4.27-3-686|200803131432
1161990|pass|200803|d@0x64.nl (DEVOGON)|Devel-CheckLib|0.5|i686-linux|5.10.0|linux|2.6.22-3-486|200803131558
1162040|fail|200803|d@0x64.nl (DEVOGON)|Devel-CheckOS|1.43|i686-linux|5.10.0|linux|2.6.22-3-486|200803131611
1162060|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200803131508
1162067|pass|200803|d@0x64.nl|Devel-CheckOS|1.43|i686-linux|5.10.0|linux|2.6.22-3-486|200803131616
1162156|pass|200803|justinr@pollackassociates.com|Devel-CheckOS|1.43|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200803130840
1162359|pass|200803|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200803131154
1162362|pass|200803|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Z80|0.9|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200803131155
1162405|pass|200803|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200803131204
1162419|pass|200803|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200803131205
1162522|pass|200803|d@0x64.nl (DEVOGON)|Devel-CheckOS|1.43|i686-linux-thread-multi|5.11.0 patch 33261|linux|2.6.22-3-486|200803131630
1162536|pass|200803|rlane10@wi.rr.com|Devel-CheckOS|1.43|darwin-2level-ld|5.10.0|darwin|8.11.0|200803130855
1163299|pass|200803|david@cantrell.org.uk|CPU-Emulator-Z80|0.9|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200803131849
1163308|pass|200803|david@cantrell.org.uk|Devel-CheckLib|0.5|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200803131851
1163319|pass|200803|david@cantrell.org.uk|Devel-CheckOS|1.43|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200803131852
1163412|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux|5.10.0 patch 33501|linux|2.6.22-1-k7|200803132004
1163421|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux|5.10.0 patch 33501|linux|2.6.22-1-k7|200803132011
1163434|pass|200803|d@0x64.nl (DEVOGON)|Number-Phone|1.581|i686-linux-thread-multi|5.11.0 patch 33261|linux|2.6.22-3-486|200803131922
1163766|pass|200803|srezic@cpan.org|Devel-CheckOS|1.43|amd64-freebsd|5.10.0|freebsd|6.2-release|200803132120
1163769|pass|200803|srezic@cpan.org|Devel-CheckLib|0.5|amd64-freebsd|5.10.0|freebsd|6.2-release|200803132120
1163771|pass|200803|srezic@cpan.org|Devel-CheckLib|0.5|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200803132121
1163772|pass|200803|srezic@cpan.org|Devel-CheckOS|1.43|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200803132121
1163818|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux|5.10.0 patch 33501|linux|2.6.22-1-k7|200803132126
1164045|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux|5.10.0 patch 33501|linux|2.6.22-1-k7|200803132031
1164132|pass|200803|srezic@cpan.org|CPU-Emulator-Z80|0.9|amd64-freebsd|5.8.8|freebsd|6.2-prerelease|200803132151
1164362|pass|200803|srezic@cpan.org|Devel-CheckOS|1.43|amd64-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200803132246
1164411|pass|200803|srezic@cpan.org|CPU-Emulator-Z80|0.9|amd64-freebsd-thread-multi|5.8.8|freebsd|6.2-release|200803132303
1164413|pass|200803|srezic@cpan.org|CPU-Emulator-Memory|1.1001|amd64-freebsd-thread-multi|5.8.8|freebsd|6.2-release|200803132302
1164476|pass|200803|srezic@cpan.org|Devel-CheckLib|0.5|amd64-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200803132247
1164479|pass|200803|srezic@cpan.org|CPU-Emulator-Z80|0.9|amd64-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200803132250
1164492|pass|200803|srezic@cpan.org|Devel-CheckOS|1.43|amd64-freebsd-thread-multi|5.8.8|freebsd|6.2-release|200803132301
1164500|pass|200803|srezic@cpan.org|Devel-CheckLib|0.5|amd64-freebsd-thread-multi|5.8.8|freebsd|6.2-release|200803132301
1164748|pass|200803|srezic@cpan.org|CPAN-FindDependencies|2.0|amd64-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200803140009
1164865|pass|200803|david@cantrell.org.uk|CPU-Emulator-Z80|0.9|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200803131853
1164866|pass|200803|david@cantrell.org.uk|Devel-CheckLib|0.5|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200803131854
1164886|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200803140006
1164887|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200803140007
1164888|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200803140008
1164889|pass|200803|david@cantrell.org.uk|Devel-CheckOS|1.43|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200803140009
1164896|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200803140013
1164899|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200803140014
1164902|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200803140015
1164995|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Z80|0.9|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200803140114
1165001|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Z80|0.9|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200803140118
1165072|pass|200803|imacat@mail.imacat.idv.tw|CPU-Emulator-Z80|0.9|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200803141048
1165073|pass|200803|imacat@mail.imacat.idv.tw|Devel-CheckLib|0.5|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200803141048
1165074|pass|200803|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.43|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200803141048
1165344|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200803140603
1165350|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Z80|0.9|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200803140604
1165353|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200803140605
1165354|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200803140606
1165541|pass|200803|imacat@mail.imacat.idv.tw|Devel-CheckLib|0.5|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200803141620
1165549|pass|200803|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.43|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200803141622
1165579|pass|200803|imacat@mail.imacat.idv.tw|CPU-Emulator-Z80|0.9|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200803141634
1165909|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200803141047
1166648|pass|200803|dillman@ria.buffalo.edu|Devel-CheckOS|1.43|sun4-solaris|5.10.0|solaris|2.9|200803141303
1167659|pass|200803|pjacklam@online.no|Devel-CheckOS|1.43|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200803142333
1167878|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200803150009
1167910|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Z80|0.9|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200803150015
1167915|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200803150016
1167924|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200803150018
1168085|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33523|linux|2.6.22-1-k7|200803150144
1168411|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33523|linux|2.6.22-1-k7|200803150255
1168612|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|cygwin-thread-multi-64int|5.11.0 patch 33523|cygwin|1.5.24(0.15642)|200803150315
1168614|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Z80|0.9|cygwin-thread-multi-64int|5.11.0 patch 33523|cygwin|1.5.24(0.15642)|200803150316
1168616|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|cygwin-thread-multi-64int|5.11.0 patch 33523|cygwin|1.5.24(0.15642)|200803150318
1168618|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|cygwin-thread-multi-64int|5.11.0 patch 33523|cygwin|1.5.24(0.15642)|200803150320
1168671|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200803150345
1168674|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200803150347
1168676|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200803150349
1168679|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200803150350
1168682|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200803150352
1168773|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200803150425
1168780|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200803150426
1168783|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200803150428
1168787|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200803150429
1168791|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200803150431
1168843|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200803150452
1168844|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200803150453
1168847|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200803150455
1168848|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200803150456
1168850|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200803150457
1168852|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200803150459
1168855|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200803150500
1168857|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200803150501
1168862|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200803150502
1168863|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200803150504
1169140|pass|200803|srezic@cpan.org|Devel-CheckOS|1.43|amd64-freebsd|5.6.2|freebsd|6.2-release|200803150824
1169142|pass|200803|srezic@cpan.org|Devel-CheckLib|0.5|amd64-freebsd|5.6.2|freebsd|6.2-release|200803150824
1170491|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|cygwin-thread-multi-64int|5.11.0 patch 33523|cygwin|1.5.24(0.15642)|200803151654
1171035|pass|200803|d@0x64.nl (DEVOGON)|Devel-CheckLib|0.5|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200803152243
1171036|pass|200803|d@0x64.nl (DEVOGON)|Devel-CheckOS|1.43|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200803152243
1171715|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-thread-multi|5.8.8 patch 33536|linux|2.6.22-1-k7|200803160058
1172433|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200803160609
1172435|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Z80|0.9|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200803160609
1172436|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200803160610
1172438|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200803160611
1172467|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200803160656
1172895|pass|200803|srezic@cpan.org|CPAN-FindDependencies|2.0|amd64-freebsd|5.8.8|freebsd|6.3-stable|200803161138
1173125|pass|200803|srezic@cpan.org|Data-Compare|1.18|amd64-freebsd|5.8.8|freebsd|6.3-stable|200803161212
1173183|pass|200803|srezic@cpan.org|Devel-CheckOS|1.43|amd64-freebsd|5.8.8|freebsd|6.3-stable|200803161216
1173817|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-thread-multi|5.8.8 patch 33536|linux|2.6.22-1-k7|200803161246
1174405|pass|200803|g.grigelionis@computer.org|Devel-CheckOS|1.43|i586-linux-thread-multi|5.8.5|linux|2.6.8.1|200803151032
1174545|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-thread-multi|5.8.8 patch 33536|linux|2.6.22-1-k7|200803161404
1174714|pass|200803|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|CPU-Emulator-Z80|0.9|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200803161337
1174721|pass|200803|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Devel-CheckLib|0.5|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200803161338
1174724|pass|200803|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Devel-CheckOS|1.43|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200803161339
1174757|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi|5.8.8 patch 33536|linux|2.6.22-1-k7|200803161451
1176590|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-thread-multi|5.8.8 patch 33536|linux|2.6.22-1-k7|200803161921
1176723|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-thread-multi|5.8.8 patch 33536|linux|2.6.22-1-k7|200803161932
1176968|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|cygwin-thread-multi-64int|5.10.0|cygwin|1.5.24(0.15642)|200803161952
1176971|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Z80|0.9|cygwin-thread-multi-64int|5.10.0|cygwin|1.5.24(0.15642)|200803161954
1176974|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|cygwin-thread-multi-64int|5.10.0|cygwin|1.5.24(0.15642)|200803161956
1176979|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|cygwin-thread-multi-64int|5.10.0|cygwin|1.5.24(0.15642)|200803161958
1177423|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200803162007
1177430|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200803162008
1177436|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200803162011
1177445|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Z80|0.9|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200803162014
1178306|pass|200803|d@0x64.nl ("DEVOGON")|CPU-Emulator-Z80|0.9|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200803170730
1178312|pass|200803|d@0x64.nl ("DEVOGON")|Devel-CheckOS|1.43|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200803170734
1178317|pass|200803|d@0x64.nl ("DEVOGON")|Devel-CheckLib|0.5|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200803170735
1179297|pass|200803|jrockway@cpan.org|Data-Compare|1.18|x86_64-linux|5.10.0|linux|2.6.22-3-amd64|200803170556
1179476|pass|200803|stro@cpan.org|Devel-CheckOS|1.43|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200803171456
1179553|pass|200803|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.581|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200803170934
1180042|pass|200803|stro@cpan.org|Devel-CheckOS|1.43|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200803171718
1180067|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|alpha-linux-thread-multi|5.11.0 patch 33539|linux|2.6.18-5-alpha-generic|200803171523
1180073|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Z80|0.9|alpha-linux-thread-multi|5.11.0 patch 33539|linux|2.6.18-5-alpha-generic|200803171524
1180076|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|alpha-linux-thread-multi|5.11.0 patch 33539|linux|2.6.18-5-alpha-generic|200803171525
1180083|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|alpha-linux-thread-multi|5.11.0 patch 33539|linux|2.6.18-5-alpha-generic|200803171526
1181294|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-thread-multi-64int|5.10.0 patch 33535|linux|2.6.22-1-k7|200803172316
1181486|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-thread-multi-64int|5.10.0 patch 33535|linux|2.6.22-1-k7|200803172255
1182212|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int|5.10.0 patch 33535|linux|2.6.22-1-k7|200803180154
1182266|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi-64int|5.10.0 patch 33535|linux|2.6.22-1-k7|200803180159
1182400|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-thread-multi-64int|5.10.0 patch 33535|linux|2.6.22-1-k7|200803180219
1182464|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-thread-multi-64int|5.10.0 patch 33535|linux|2.6.22-1-k7|200803180238
1183117|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200803180413
1183120|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Z80|0.9|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200803180414
1183122|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200803180415
1183125|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200803180416
1183818|pass|200803|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|CPU-Emulator-Z80|0.9|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200803180856
1183820|pass|200803|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Devel-CheckLib|0.5|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200803180857
1183823|pass|200803|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Devel-CheckOS|1.43|sun4-solaris-thread-multi|5.8.8|solaris|2.9|200803180858
1184296|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200803181113
1184598|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|alpha-linux-thread-multi|5.11.0 patch 33539|linux|2.6.18-5-alpha-generic|200803181322
1185299|pass|200803|srezic@cpan.org|CPU-Emulator-Memory|1.1001|amd64-freebsd|5.8.8|freebsd|6.3-stable|200803181634
1185301|pass|200803|srezic@cpan.org|CPU-Emulator-Z80|0.9|amd64-freebsd|5.8.8|freebsd|6.3-stable|200803181635
1185302|pass|200803|srezic@cpan.org|Devel-CheckLib|0.5|amd64-freebsd|5.8.8|freebsd|6.3-stable|200803181634
1185643|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200803181710
1185646|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Z80|0.9|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200803181711
1185669|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200803181713
1185676|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200803181714
1186256|pass|200803|srezic@cpan.org|Net-Random|2.0|amd64-freebsd|5.6.2|freebsd|6.2-release|200803182000
1186749|pass|200803|srezic@cpan.org|Number-Phone|1.581|amd64-freebsd-thread-multi|5.8.8|freebsd|6.2-release|200803182152
1187699|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200803190210
1189947|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200803191518
1189953|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Z80|0.9|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200803191519
1189956|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200803191520
1190402|pass|200803|chris@bingosnet.co.uk|CPU-Emulator-Z80|0.9|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200803191634
1190403|pass|200803|chris@bingosnet.co.uk|Devel-CheckLib|0.5|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200803191635
1190405|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200803191635
1190469|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux|5.11.0 patch 33542|linux|2.6.22-1-k7|200803191916
1190666|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux|5.11.0 patch 33542|linux|2.6.22-1-k7|200803191844
1191004|fail|200803|srezic@cpan.org|Bryar|3.0|amd64-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200803192109
1191974|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux|5.11.0 patch 33542|linux|2.6.22-1-k7|200803200158
1192362|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux|5.11.0 patch 33542|linux|2.6.22-1-k7|200803200342
1192598|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200803200110
1192731|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux|5.11.0 patch 33542|linux|2.6.22-1-k7|200803200339
1193708|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux|5.11.0 patch 33542|linux|2.6.22-1-k7|200803200740
1193709|pass|200803|imacat@mail.imacat.idv.tw|Acme-Licence|1.0|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200803201440
1193710|pass|200803|imacat@mail.imacat.idv.tw|Acme-Pony|1.1.2|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200803201440
1193712|pass|200803|imacat@mail.imacat.idv.tw|Acme-Scurvy-Whoreson-BilgeRat|1.1|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200803201440
1193714|pass|200803|imacat@mail.imacat.idv.tw|NestedMap|1.0|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200803201440
1193716|pass|200803|imacat@mail.imacat.idv.tw|Statistics-ChiSquare|0.5|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200803201441
1193717|pass|200803|imacat@mail.imacat.idv.tw|Statistics-SerialCorrelation|1.1|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200803201441
1193719|pass|200803|imacat@mail.imacat.idv.tw|Tie-Hash-Longest|1.1|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200803201441
1193720|pass|200803|imacat@mail.imacat.idv.tw|Tie-Hash-Rank|1.0.1|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200803201441
1193725|pass|200803|imacat@mail.imacat.idv.tw|Tie-Scalar-Decay|1.1.1|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200803201442
1197084|pass|200803|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Compare|1.18|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200803202243
1197372|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|i386-freebsd-64int|5.11.0 patch 33543|freebsd|7.0-release|200803201903
1198104|pass|200803|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|CPU-Emulator-Memory|1.1001|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200803210453
1198112|pass|200803|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|CPU-Emulator-Z80|0.9|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200803210455
1198152|pass|200803|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Devel-CheckOS|1.43|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200803210504
1198166|pass|200803|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Devel-CheckLib|0.5|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200803210507
1198952|pass|200803|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|amd64-freebsd-thread-multi|5.8.8|freebsd|6.2-release|200803210013
1198983|pass|200803|d@0x64.nl ("DEVOGON")|Data-Transactional|1.0|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200803210932
1198998|pass|200803|d@0x64.nl ("DEVOGON")|Tie-STDOUT|1.03|x86_64-linux|5.10.0|linux|2.6.23.1-slh64-smp-32|200803210934
1200386|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|cygwin-thread-multi-64int|5.11.0 patch 33543|cygwin|1.5.24(0.15642)|200803211349
1201926|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux|5.11.0 patch 33547|linux|2.6.22-1-k7|200803212318
1202128|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux|5.11.0 patch 33547|linux|2.6.22-1-k7|200803212309
1203286|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|i86pc-solaris-thread-multi-64int|5.11.0 patch 33544|solaris|2.11|200803221120
1204426|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200803222136
1204805|pass|200803|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200803230315
1205235|fail|200803|imacat@mail.imacat.idv.tw|Palm-SMS|0.02|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200803231512
1205373|pass|200803|imacat@mail.imacat.idv.tw|Bryar|3.0|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200803231644
1207445|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|i386-pc-solaris2.11-thread-multi-64int|5.10.0|solaris|2.11|200803240109
1208722|pass|200803|pjacklam@online.no|Statistics-ChiSquare|0.5|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200803240946
1209414|pass|200803|pjacklam@online.no|Statistics-SerialCorrelation|1.1|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200803241042
1211329|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux|5.11.0 patch 33551|linux|2.6.22-1-k7|200803242029
1211522|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux|5.11.0 patch 33551|linux|2.6.22-1-k7|200803242028
1211793|pass|200803|srezic@cpan.org|Number-Phone-UK-DetailedLocations|1.3|amd64-freebsd|5.6.2|freebsd|6.2-release|200803242136
1211844|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux|5.11.0 patch 33551|linux|2.6.22-1-k7|200803241943
1211972|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux|5.11.0 patch 33551|linux|2.6.22-1-k7|200803242157
1213162|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux|5.11.0 patch 33551|linux|2.6.22-1-k7|200803250221
1213482|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux|5.11.0 patch 33551|linux|2.6.22-1-k7|200803250005
1214140|pass|200803|victorg@hacktux.com|Devel-CheckOS|1.43|i686-linux|5.10.0|linux|2.6.18-53.1.13.el5|200803250023
1219608|pass|200803|mthurn@cpan.org|Devel-CheckOS|1.43|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200803260359
1222775|pass|200803|srezic@cpan.org|Devel-CheckOS|1.43|i386-freebsd|5.6.2|freebsd|6.1-release|200803270221
1223051|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 33580|linux|2.6.22-1-k7|200803270427
1223335|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33580|linux|2.6.22-1-k7|200803270607
1223339|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 33580|linux|2.6.22-1-k7|200803270608
1223390|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-64int|5.11.0 patch 33580|linux|2.6.22-1-k7|200803270618
1223398|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.11.0 patch 33580|linux|2.6.22-1-k7|200803270615
1224707|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33580|linux|2.6.22-1-k7|200803271100
1224842|pass|200803|imacat@mail.imacat.idv.tw|File-Find-Rule-Permissions|1.3|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200803271844
1225443|pass|200803|andy@hexten.net|Devel-CheckLib|0.5|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200803271533
1227809|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.8.8 patch 33576|linux|2.6.22-1-k7|200803281302
1227848|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.8.8 patch 33576|linux|2.6.22-1-k7|200803281322
1228566|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 33576|linux|2.6.22-1-k7|200803281513
1228671|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 33576|linux|2.6.22-1-k7|200803281726
1228876|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200803281722
1228967|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200803281737
1229101|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-64int|5.8.8 patch 33576|linux|2.6.22-1-k7|200803281854
1229520|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 33576|linux|2.6.22-1-k7|200803282040
1230271|pass|200803|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.18|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200803290030
1231970|pass|200803|srezic@cpan.org|Devel-CheckOS|1.43|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200803291342
1232260|pass|200803|srezic@cpan.org|Devel-CheckOS|1.43|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200803291414
1232395|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.11.0 patch 33584|netbsd|4.0|200803291444
1232715|pass|200803|srezic@cpan.org|Data-Compare|1.18|i386-freebsd|5.10.0|freebsd|6.1-release|200803291728
1232969|pass|200803|srezic@cpan.org|Data-Compare|1.18|i386-freebsd|5.6.2|freebsd|6.1-release|200803291533
1233051|pass|200803|chris@bingosnet.co.uk|Data-Compare|1.18|i386-netbsd-thread-multi-64int|5.11.0 patch 33584|netbsd|4.0|200803291737
1233578|pass|200803|icerider70@gmail.com ("ICERIDER")|Devel-CheckLib|0.4|i386-linux|5.8.8|linux|2.6.18-53.el5|200803291440
1233672|pass|200803|srezic@cpan.org|Data-Compare|1.18|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200803292004
1234280|pass|200803|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Number-Phone|1.581|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200803292355
1234283|fail|200803|david@cantrell.org.uk|Devel-CheckOS|1.43|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200803292256
1234304|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|cygwin-thread-multi-64int|5.11.0 patch 33592|cygwin|1.5.24(0.15642)|200803292311
1235061|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 33243|linux|2.6.22-1-k7|200803301032
1235065|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200803301033
1235068|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8|linux|2.6.14|200803301034
1235070|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200803301035
1235073|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200803301036
1235078|pass|200803|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.6.2|linux|2.6.16-2-k7|200803301037
1236117|pass|200803|chris@bingosnet.co.uk|XML-Tiny|1.11|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200803301809
1236572|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200803301949
1236770|pass|200803|d@0x64.nl (DEVOGON)|Devel-CheckOS|1.43|i686-linux|5.10.0|linux|2.6.22-3-486|200803302143
1236800|pass|200803|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|XML-DoubleEncodedEntities|1.0|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200803302152
1237579|pass|200803|srezic@cpan.org|Devel-CheckOS|1.43|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200803302354
1237843|pass|200803|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200803310046
1238161|pass|200803|icerider70@gmail.com ("ICERIDER")|Devel-CheckLib|0.5|i386-linux|5.8.8|linux|2.6.18-53.el5|200803310322
1238163|pass|200803|icerider70@gmail.com ("ICERIDER")|Devel-CheckOS|1.43|i386-linux|5.8.8|linux|2.6.18-53.el5|200803310324
1238202|pass|200803|icerider70@gmail.com ("ICERIDER")|CPU-Emulator-Z80|0.9|i386-linux|5.8.8|linux|2.6.18-53.el5|200803310336
1239152|pass|200803|d@0x64.nl|XML-Tiny|1.11|i686-linux|5.10.0|linux|2.6.22-3-486|200803311156
1239517|pass|200803|flemming@ipk-gatersleben.de|Data-Compare|1.18|x86_64-linux|5.8.5|linux|2.6.9-55.0.9.elsmp|200803311253
1240050|pass|200803|flemming@ipk-gatersleben.de|File-Find-Rule-Permissions|1.3|x86_64-linux|5.8.5|linux|2.6.9-55.0.9.elsmp|200803311452
1240150|pass|200803|flemming@ipk-gatersleben.de|Devel-CheckOS|1.43|x86_64-linux|5.8.5|linux|2.6.9-55.0.9.elsmp|200803311452
1240236|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200803311522
1240361|pass|200803|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|XML-Tiny|1.11|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200803311646
1240809|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|alpha-linux-thread-multi|5.11.0 patch 33609|linux|2.6.18-5-alpha-generic|200803311748
1241129|pass|200803|d@0x64.nl (DEVOGON)|XML-Tiny|1.11|i686-linux-thread-multi|5.11.0 patch 33261|linux|2.6.22-3-486|200803311839
1241188|pass|200803|chris@bingosnet.co.uk|XML-Tiny|1.11|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200803311943
1241491|pass|200803|srezic@cpan.org|CPU-Emulator-Memory|1.1001|i386-freebsd|5.6.2|freebsd|6.1-release|200803312138
1241502|pass|200803|srezic@cpan.org|Devel-CheckLib|0.5|i386-freebsd|5.6.2|freebsd|6.1-release|200803312143
1241637|pass|200803|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.11.0 patch 33609|netbsd|3.1|200803312125
1242439|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.11.0 patch 33609|netbsd|3.1|200804010218
1242549|pass|200804|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200804010215
1242883|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 33620|linux|2.6.22-1-k7|200804010613
1243016|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 33620|linux|2.6.22-1-k7|200804010702
1243595|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.10.0 patch 33620|linux|2.6.22-1-k7|200804010745
1244664|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-64int|5.10.0 patch 33620|linux|2.6.22-1-k7|200804011204
1245283|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 33620|linux|2.6.22-1-k7|200804011238
1245469|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.10.0 patch 33620|linux|2.6.22-1-k7|200804010443
1245671|pass|200804|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int|5.11.0 patch 33623|linux|2.6.16.38|200804011411
1246102|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|alpha-linux-thread-multi|5.11.0 patch 33609|linux|2.6.18-5-alpha-generic|200804011728
1246605|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|i686-linux-thread-multi-64int|5.11.0 patch 33623|linux|2.6.16.38|200804011834
1247239|pass|200804|srezic@cpan.org|Data-Compare|1.18|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200804020018
1249436|fail|200804|dagolden@cpan.org (DAGOLDEN)|Bryar|3.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200804020616
1250046|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200804021400
1250303|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200804021720
1250327|pass|200804|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200804021640
1250845|pass|200804|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200804021958
1250931|pass|200804|srezic@cpan.org|Devel-CheckLib|0.5|i386-freebsd|5.5.5|freebsd|6.1-release|200804022228
1251046|pass|200804|g.grigelionis@computer.org|Devel-CheckOS|1.43|x86_64-linux|5.8.8|linux|2.6.5-7.276-smp|200804022200
1251060|na|200804|srezic@cpan.org|CPU-Emulator-Memory|1.1001|i386-freebsd|5.5.5|freebsd|6.1-release|200804022228
1251663|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200804030054
1252351|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-thread-multi-64int|5.10.0 patch 33628|linux|2.6.22-1-k7|200804030743
1252495|pass|200804|srezic@cpan.org|XML-Tiny|1.11|i386-freebsd|5.6.2|freebsd|6.1-release|200804030804
1252565|pass|200804|srezic@cpan.org|XML-Tiny|1.11|i386-freebsd|5.10.0|freebsd|6.1-release|200804030807
1252690|pass|200804|srezic@cpan.org|XML-Tiny|1.11|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200804030846
1252944|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-thread-multi-64int|5.10.0 patch 33628|linux|2.6.22-1-k7|200804031013
1253193|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-thread-multi-64int|5.10.0 patch 33628|linux|2.6.22-1-k7|200804031110
1253816|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 33632|openbsd|4.1|200804031219
1254074|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-thread-multi-64int|5.10.0 patch 33628|linux|2.6.22-1-k7|200804031419
1254640|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi-64int|5.10.0 patch 33628|linux|2.6.22-1-k7|200804031557
1254832|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int|5.10.0 patch 33628|linux|2.6.22-1-k7|200804031639
1256353|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 33632|openbsd|4.1|200804032241
1256641|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-thread-multi-64int|5.8.8 patch 33638|linux|2.6.22-1-k7|200804040035
1256699|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-thread-multi-64int|5.8.8 patch 33638|linux|2.6.22-1-k7|200804040046
1258103|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-thread-multi-64int|5.8.8 patch 33638|linux|2.6.22-1-k7|200804040823
1258262|pass|200804|srezic@cpan.org|XML-Tiny|1.11|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200804040844
1258588|pass|200804|srezic@cpan.org|XML-Tiny|1.11|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200804040903
1258998|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int|5.8.8 patch 33638|linux|2.6.22-1-k7|200804041115
1259325|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-thread-multi-64int|5.8.8 patch 33638|linux|2.6.22-1-k7|200804041211
1259700|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi-64int|5.8.8 patch 33638|linux|2.6.22-1-k7|200804041104
1260650|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|cygwin-thread-multi-64int|5.11.0 patch 33643|cygwin|1.5.24(0.15642)|200804041859
1261597|pass|200804|srezic@cpan.org|Devel-CheckOS|1.43|i386-freebsd|5.10.0|freebsd|6.1-release|200804050039
1261932|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux|5.11.0 patch 33646|linux|2.6.22-1-k7|200804050156
1262025|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux|5.11.0 patch 33646|linux|2.6.22-1-k7|200804050243
1262253|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux|5.11.0 patch 33646|linux|2.6.22-1-k7|200804050337
1262830|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux|5.11.0 patch 33646|linux|2.6.22-1-k7|200804050707
1262965|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux|5.11.0 patch 33646|linux|2.6.22-1-k7|200804050734
1263296|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux|5.11.0 patch 33646|linux|2.6.22-1-k7|200804050833
1264058|pass|200804|srezic@cpan.org|Devel-CheckLib|0.5|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200804051046
1264059|pass|200804|srezic@cpan.org|CPU-Emulator-Memory|1.1001|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200804051047
1264192|pass|200804|srezic@cpan.org|CPU-Emulator-Z80|0.9|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200804051048
1264387|pass|200804|srezic@cpan.org|Number-Phone|1.581|i386-freebsd|5.6.2|freebsd|6.1-release|200804051254
1264466|pass|200804|srezic@cpan.org|Devel-CheckLib|0.5|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200804051243
1264548|pass|200804|srezic@cpan.org|CPU-Emulator-Memory|1.1001|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200804051244
1264550|pass|200804|srezic@cpan.org|CPU-Emulator-Z80|0.9|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200804051245
1264740|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200804051349
1265774|pass|200804|srezic@cpan.org|Devel-CheckLib|0.5|i386-freebsd|5.10.0|freebsd|6.1-release|200804051816
1265782|pass|200804|srezic@cpan.org|CPU-Emulator-Z80|0.9|i386-freebsd|5.10.0|freebsd|6.1-release|200804051819
1265898|pass|200804|srezic@cpan.org|CPU-Emulator-Memory|1.1001|i386-freebsd|5.10.0|freebsd|6.1-release|200804051818
1266599|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux|5.11.0 patch 33648|linux|2.6.22-1-k7|200804052116
1267915|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux|5.11.0 patch 33648|linux|2.6.22-1-k7|200804060102
1267993|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux|5.11.0 patch 33648|linux|2.6.22-1-k7|200804060048
1268179|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux|5.11.0 patch 33648|linux|2.6.22-1-k7|200804060113
1268941|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux|5.11.0 patch 33648|linux|2.6.22-1-k7|200804060231
1269251|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux|5.11.0 patch 33648|linux|2.6.22-1-k7|200804060207
1270069|pass|200804|srezic@cpan.org|CPU-Emulator-Memory|1.1001|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200804061049
1270100|pass|200804|srezic@cpan.org|CPU-Emulator-Z80|0.9|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200804061054
1270807|pass|200804|srezic@cpan.org|Number-Phone|1.581|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200804061538
1271214|pass|200804|gabor@pti.co.il ("Gabor Szabo")|Data-Compare|1.18|i486-linux-gnu|5.8.8|linux|2.6.15.7|200804062023
1272262|pass|200804|daxim@cpan.org|XML-Tiny|1.11|x86_64-linux|5.8.8|linux|2.6.20-gentoo-r8|200804062307
1272703|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200804070033
1273196|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.10.0 patch 33642|linux|2.6.22-1-k7|200804070512
1273374|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.10.0 patch 33642|linux|2.6.22-1-k7|200804070526
1273441|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-64int|5.10.0 patch 33642|linux|2.6.22-1-k7|200804070552
1273961|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 33642|linux|2.6.22-1-k7|200804070811
1274437|pass|200804|srezic@cpan.org|Number-Phone|1.581|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200804070943
1274783|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 33642|linux|2.6.22-1-k7|200804071031
1275261|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|i686-linux-thread-multi-64int|5.11.0 patch 33655|linux|2.6.21.5-smp|200804071002
1275273|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 33642|linux|2.6.22-1-k7|200804071154
1275419|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|alpha-linux-thread-multi|5.11.0 patch 33655|linux|2.6.18-5-alpha-generic|200804071127
1275742|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|i386-netbsd-thread-multi-64int|5.11.0 patch 33655|netbsd|3.1|200804071205
1275956|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|darwin-thread-multi-64int-2level|5.11.0 patch 33655|darwin|8.10.0|200804071235
1276355|pass|200804|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200804071621
1276975|pass|200804|srezic@cpan.org|Bryar|3.0|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200804071919
1280289|pass|200804|srezic@cpan.org|Bryar|3.0|i386-freebsd|5.6.2|freebsd|6.1-release|200804080705
1282762|pass|200804|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|CPAN-FindDependencies|2.0|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200804081911
1285009|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 33659|linux|2.6.22-1-k7|200804090442
1285671|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33659|linux|2.6.22-1-k7|200804090643
1285857|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-64int|5.11.0 patch 33659|linux|2.6.22-1-k7|200804090734
1286432|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 33659|linux|2.6.22-1-k7|200804090756
1286479|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33659|linux|2.6.22-1-k7|200804090922
1286504|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.11.0 patch 33659|linux|2.6.22-1-k7|200804090219
1286579|pass|200804|srezic@cpan.org|Devel-CheckLib|0.5|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200804090930
1287189|pass|200804|rafl@debian.org|Data-Compare|1.18|i686-linux-multi|5.10.0|linux|2.6.24-rc5|200804091259
1287547|pass|200804|srezic@cpan.org|Number-Phone|1.581|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200804091419
1288483|pass|200804|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200804091852
1289736|pass|200804|srezic@cpan.org|Number-Phone|1.581|i386-freebsd|5.10.0|freebsd|6.1-release|200804100214
1291176|pass|200804|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd|5.10.0|freebsd|6.1-release|200804101229
1292857|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|darwin-thread-multi-64int-2level|5.11.0 patch 33659|darwin|8.10.0|200804102014
1292969|pass|200804|srezic@cpan.org|Data-Compare|1.18|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200804102153
1294270|pass|200804|srezic@cpan.org|File-Find-Rule-Permissions|1.3|i386-freebsd|5.6.2|freebsd|6.1-release|200804110417
1295251|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|i86pc-solaris-thread-multi-64int|5.11.0 patch 33659|solaris|2.11|200804111006
1296395|pass|200804|srezic@cpan.org|CPAN-FindDependencies|2.0|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200804111548
1296444|fail|200804|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Bryar|3.0|sun4-solaris-thread-multi-64int|5.10.0|solaris|2.9|200804111559
1298893|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|cygwin-thread-multi-64int|5.11.0 patch 33669|cygwin|1.5.24(0.15642)|200804112348
1299784|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-thread-multi-64int|5.11.0 patch 33669|linux|2.6.22-1-k7|200804120744
1299806|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-thread-multi-64int|5.11.0 patch 33669|linux|2.6.22-1-k7|200804120736
1300813|fail|200804|srezic@cpan.org|Bryar|3.0|i386-freebsd|5.10.0|freebsd|6.1-release|200804121203
1302083|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-thread-multi-64int|5.11.0 patch 33669|linux|2.6.22-1-k7|200804121611
1302530|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int|5.11.0 patch 33669|linux|2.6.22-1-k7|200804121655
1302619|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-thread-multi-64int|5.11.0 patch 33669|linux|2.6.22-1-k7|200804121648
1302734|pass|200804|andy@hexten.net|Devel-CheckLib|0.5|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200804121616
1302878|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi-64int|5.11.0 patch 33669|linux|2.6.22-1-k7|200804121725
1303747|pass|200804|bobtfish@bobtfish.net|Data-Compare|1.18|darwin-thread-multi-2level|5.8.6|darwin|8.0|200804121656
1304958|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 33670|openbsd|4.2|200804122311
1306578|pass|200804|srezic@cpan.org|Net-Random|2.0|i386-freebsd|5.6.2|freebsd|6.1-release|200804131000
1307348|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200804131229
1308479|pass|200804|srezic@cpan.org|Net-Random|2.0|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200804131759
1308506|pass|200804|srezic@cpan.org|Net-Random|2.0|i386-freebsd|5.10.0|freebsd|6.1-release|200804131809
1309852|pass|200804|srezic@cpan.org|Palm-TreoPhoneCallDB|1.1|i386-freebsd|5.6.2|freebsd|6.1-release|200804132352
1309901|pass|200804|srezic@cpan.org|Palm-TreoPhoneCallDB|1.1|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200804140004
1309990|pass|200804|srezic@cpan.org|Palm-TreoPhoneCallDB|1.1|i386-freebsd|5.10.0|freebsd|6.1-release|200804140024
1310065|pass|200804|srezic@cpan.org|Palm-TreoPhoneCallDB|1.1|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200804140032
1310264|pass|200804|srezic@cpan.org|Palm-TreoPhoneCallDB|1.1|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200804140112
1312796|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|alpha-linux-thread-multi|5.11.0 patch 33673|linux|2.6.18-5-alpha-generic|200804141247
1314443|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|i386-netbsd-thread-multi-64int|5.11.0 patch 33673|netbsd|3.1|200804142104
1314651|pass|200804|srezic@cpan.org|Games-Dice-Advanced|1.1|i386-freebsd|5.10.0|freebsd|6.1-release|200804142240
1316284|pass|200804|srezic@cpan.org|File-Find-Rule-Permissions|1.3|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200804150951
1318154|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|darwin-thread-multi-64int-2level|5.11.0 patch 33684|darwin|8.10.0|200804151533
1318181|pass|200804|srezic@cpan.org|Data-Compare|1.18|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804151633
1319178|pass|200804|srezic@cpan.org|Devel-CheckOS|1.43|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804152010
1319424|pass|200804|srezic@cpan.org|Tie-STDOUT|1.03|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200804152133
1319427|pass|200804|srezic@cpan.org|Data-Transactional|1.0|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200804152135
1320037|pass|200804|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804160006
1320202|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 33695|linux|2.6.22-1-k7|200804160045
1320688|pass|200804|srezic@cpan.org|CPU-Emulator-Z80|0.9|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804160154
1320854|pass|200804|srezic@cpan.org|CPU-Emulator-Memory|1.1001|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804160153
1321240|pass|200804|srezic@cpan.org|File-Find-Rule-Permissions|1.3|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200804160306
1321718|pass|200804|bobtfish@bobtfish.net|Data-Compare|1.18|i486-linux-gnu|5.8.8|linux|2.6.22-3-amd64|200804160036
1321958|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.11.0 patch 33695|linux|2.6.22-1-k7|200804160426
1322246|pass|200804|srezic@cpan.org|Devel-CheckLib|0.5|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804160545
1322346|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33695|linux|2.6.22-1-k7|200804160609
1322726|pass|200804|srezic@cpan.org|XML-Tiny|1.11|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804160707
1322791|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33695|linux|2.6.22-1-k7|200804160722
1323725|pass|200804|srezic@cpan.org|Bryar|3.0|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804160935
1323925|pass|200804|srezic@cpan.org|CPAN-FindDependencies|2.0|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804160955
1324139|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 33695|linux|2.6.22-1-k7|200804160937
1324238|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-64int|5.11.0 patch 33695|linux|2.6.22-1-k7|200804161023
1324470|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200804161007
1324479|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200804161008
1324893|pass|200804|srezic@cpan.org|File-Find-Rule-Permissions|1.3|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804161208
1325309|pass|200804|daxim@cpan.org|Devel-CheckOS|1.43|x86_64-linux|5.8.8|linux|2.6.20-gentoo-r8|200804161332
1329851|pass|200804|srezic@cpan.org|Number-Phone|1.581|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804170219
1330384|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200804170602
1330587|pass|200804|srezic@cpan.org|Bryar|3.0|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200804170816
1330869|pass|200804|srezic@cpan.org|Data-Hexdumper|1.3|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804170940
1330877|pass|200804|srezic@cpan.org|Data-Transactional|1.0|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804170941
1331563|pass|200804|srezic@cpan.org|Palm-TreoPhoneCallDB|1.1|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804171159
1331870|pass|200804|srezic@cpan.org|Tie-STDOUT|1.03|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804171256
1333040|pass|200804|srezic@cpan.org|XML-Tiny|1.11|i386-freebsd|5.5.5|freebsd|6.1-release|200804171656
1333594|pass|200804|srezic@cpan.org|Number-Phone-UK-DetailedLocations|1.3|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804171929
1333630|pass|200804|srezic@cpan.org|Net-Random|2.0|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804171941
1334678|pass|200804|srezic@cpan.org|Sub-WrapPackages|1.2|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804172336
1334892|pass|200804|chris@bingosnet.co.uk|Number-Phone|1.581|darwin-thread-multi-64int-2level|5.11.0 patch 33684|darwin|8.10.0|200804172324
1334893|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200804172324
1336996|pass|200804|stro@cpan.org|Devel-CheckOS|1.43|i686-linux|5.10.0|linux|2.4.27-2-386|200804181300
1337168|pass|200804|srezic@cpan.org|Games-Dice-Advanced|1.1|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804181149
1338088|pass|200804|stro@cpan.org|Devel-CheckOS|1.43|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200804181535
1338833|pass|200804|srezic@cpan.org|File-Overwrite|1.0|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804181627
1340146|pass|200804|srezic@cpan.org|Palm-SMS|0.02|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804182058
1341403|pass|200804|srezic@cpan.org|Class-CanBeA|1.2|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804190052
1342227|pass|200804|srezic@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804190341
1343054|pass|200804|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.11.0 patch 33713|netbsd|3.1|200804191000
1343247|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.11.0 patch 33713|netbsd|3.1|200804191113
1343738|pass|200804|chris@bingosnet.co.uk|Devel-CheckOS|1.43|cygwin-thread-multi-64int|5.11.0 patch 33713|cygwin|1.5.24(0.15642)|200804191330
1344079|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|cygwin-thread-multi-64int|5.11.0 patch 33713|cygwin|1.5.24(0.15642)|200804191507
1344109|pass|200804|stro@cpan.org|XML-Tiny|1.11|i686-linux|5.8.8|linux|2.4.27-2-386|200804191416
1344425|pass|200804|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Devel-CheckOS|1.43|sun4-solaris|5.10.0|solaris|2.9|200804191717
1344978|pass|200804|stro@cpan.org|Devel-CheckOS|1.43|i686-linux|5.8.8|linux|2.4.27-2-386|200804191700
1345067|pass|200804|stro@cpan.org|Devel-CheckOS|1.43|i386-freebsd-64int|5.8.8|freebsd|7.0-release|200804192108
1346102|pass|200804|srezic@cpan.org|Statistics-SerialCorrelation|1.1|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804192337
1346403|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|darwin-thread-multi-64int-2level|5.11.0 patch 33713|darwin|8.10.0|200804192340
1346430|unknown|200804|srezic@cpan.org|Acme-Licence|1.0|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804200042
1346569|pass|200804|stro@cpan.org|XML-Tiny|1.11|i686-linux|5.10.0|linux|2.4.27-2-386|200804200011
1346627|unknown|200804|srezic@cpan.org|Acme-Pony|1.1.2|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804200043
1347609|pass|200804|cpansmoker@renormalist.net ("Steffen Schwigon")|XML-Tiny|1.11|i686-linux|5.10.0|linux|2.6.18-5-k7|200804200050
1348140|pass|200804|cpansmoker@renormalist.net ("Steffen Schwigon")|Devel-CheckOS|1.43|i686-linux|5.10.0|linux|2.6.18-5-k7|200804200841
1348289|pass|200804|cpansmoker@renormalist.net ("Steffen Schwigon")|Data-Compare|1.18|i686-linux|5.10.0|linux|2.6.18-5-k7|200804200911
1348942|pass|200804|srezic@cpan.org|Statistics-ChiSquare|0.5|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804201142
1349045|pass|200804|srezic@cpan.org|Tie-Hash-Longest|1.1|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804201117
1349578|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 33714|linux|2.6.22-1-k7|200804201309
1349660|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33714|linux|2.6.22-1-k7|200804201322
1349824|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.11.0 patch 33714|linux|2.6.22-1-k7|200804201341
1349960|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 33714|linux|2.6.22-1-k7|200804201306
1350142|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33714|linux|2.6.22-1-k7|200804201239
1350369|pass|200804|srezic@cpan.org|NestedMap|1.0|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804201434
1350385|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|darwin-2level|5.10.0|darwin|8.11.1|200804201202
1351314|pass|200804|artur+cpan@niif.spb.su|XML-Tiny|1.11|i386-linux|5.8.8|linux|2.6.18-53.1.4.el5.bz358661.1|200804201138
1352831|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|darwin-2level|5.10.0|darwin|8.11.1|200804201815
1352844|pass|200804|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int|5.11.0 patch 33714|linux|2.6.16.38|200804202018
1352857|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-64int|5.11.0 patch 33714|linux|2.6.22-1-k7|200804201809
1353180|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|i686-linux-thread-multi-64int|5.11.0 patch 33714|linux|2.6.16.38|200804202058
1354151|pass|200804|stro@cpan.org|XML-Tiny|1.11|armv5tejl-linux|5.8.8|linux|2.6.17.14|200804210414
1355985|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 33714|openbsd|4.1|200804211355
1356080|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|alpha-linux-thread-multi|5.11.0 patch 33716|linux|2.6.18-5-alpha-generic|200804211407
1357339|pass|200804|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Compare|1.18|sun4-solaris|5.10.0|solaris|2.9|200804211952
1357838|pass|200804|stro@cpan.org|Data-Compare|1.18|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200804212200
1359285|unknown|200804|stro@cpan.org|Acme-Licence|1.0|i686-linux|5.8.8|linux|2.4.27-2-386|200804220029
1360100|unknown|200804|stro@cpan.org|Acme-Pony|1.1.2|i686-linux|5.8.8|linux|2.4.27-2-386|200804220442
1360220|pass|200804|stro@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux|5.8.8|linux|2.4.27-2-386|200804220549
1360275|pass|200804|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 33714|openbsd|4.1|200804220557
1360430|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 33714|openbsd|4.1|200804220712
1360762|pass|200804|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|XML-Tiny|1.11|sun4-solaris|5.10.0|solaris|2.9|200804221035
1361828|unknown|200804|stro@cpan.org|Acme-Licence|1.0|armv5tejl-linux|5.8.8|linux|2.6.17.14|200804221657
1361841|unknown|200804|stro@cpan.org|Acme-Licence|1.0|i686-linux|5.10.0|linux|2.4.27-2-386|200804221541
1362745|pass|200804|srezic@cpan.org|Tie-Scalar-Decay|1.1.1|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804221953
1362863|pass|200804|srezic@cpan.org|Tie-Hash-Rank|1.0.1|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200804221951
1362883|unknown|200804|stro@cpan.org|Acme-Pony|1.1.2|i686-linux|5.10.0|linux|2.4.27-2-386|200804222000
1362913|unknown|200804|stro@cpan.org|Acme-Licence|1.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200804222148
1365292|pass|200804|mgardner@usatech.com|Data-Compare|1.18|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200804221340
1365401|pass|200804|stro@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux|5.10.0|linux|2.4.27-2-386|200804231029
1365419|unknown|200804|stro@cpan.org|Acme-Pony|1.1.2|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200804231207
1365571|pass|200804|stro@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200804231245
1365589|pass|200804|rafl@debian.org|XML-Tiny|1.11|i686-linux-multi|5.10.0|linux|2.6.24-rc5|200804230011
1365735|pass|200804|mgardner@usatech.com|Devel-CheckOS|1.43|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200804221350
1366323|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|cygwin-thread-multi-64int|5.10.0|cygwin|1.5.24(0.15642)|200804231630
1366439|unknown|200804|stro@cpan.org|Acme-Pony|1.1.2|armv5tejl-linux|5.8.8|linux|2.6.17.14|200804231843
1366955|pass|200804|chris@bingosnet.co.uk|Devel-CheckOS|1.43|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200804231902
1367132|pass|200804|stro@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|armv5tejl-linux|5.8.8|linux|2.6.17.14|200804232203
1368275|pass|200804|chris@bingosnet.co.uk|Acme-Licence|1.0|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240105
1368277|pass|200804|chris@bingosnet.co.uk|Acme-Pony|1.1.2|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240106
1368280|pass|200804|chris@bingosnet.co.uk|Acme-Scurvy-Whoreson-BilgeRat|1.1|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240107
1368324|pass|200804|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240122
1368325|pass|200804|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240123
1368334|pass|200804|chris@bingosnet.co.uk|CPU-Emulator-Z80|0.9|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240126
1368335|pass|200804|chris@bingosnet.co.uk|Class-CanBeA|1.2|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240126
1368340|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240127
1368341|pass|200804|chris@bingosnet.co.uk|Data-Hexdumper|1.3|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240128
1368343|pass|200804|chris@bingosnet.co.uk|Data-Transactional|1.0|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240129
1368347|pass|200804|chris@bingosnet.co.uk|Devel-CheckLib|0.5|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240130
1368352|pass|200804|chris@bingosnet.co.uk|Devel-CheckOS|1.43|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240131
1368354|pass|200804|chris@bingosnet.co.uk|File-Find-Rule-Permissions|1.3|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240132
1368358|pass|200804|chris@bingosnet.co.uk|File-Overwrite|1.0|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240132
1368360|pass|200804|chris@bingosnet.co.uk|Games-Dice-Advanced|1.1|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240133
1368362|pass|200804|chris@bingosnet.co.uk|NestedMap|1.0|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240133
1368371|pass|200804|chris@bingosnet.co.uk|Net-Random|2.0|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240138
1368372|pass|200804|chris@bingosnet.co.uk|Number-Phone|1.581|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240139
1368373|pass|200804|chris@bingosnet.co.uk|Number-Phone-UK-DetailedLocations|1.3|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240141
1368375|pass|200804|chris@bingosnet.co.uk|Statistics-ChiSquare|0.5|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240142
1368376|pass|200804|chris@bingosnet.co.uk|Statistics-SerialCorrelation|1.1|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240142
1368377|pass|200804|chris@bingosnet.co.uk|Sub-WrapPackages|1.2|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240143
1368378|pass|200804|chris@bingosnet.co.uk|Tie-Hash-Longest|1.1|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240143
1368379|pass|200804|chris@bingosnet.co.uk|Tie-Hash-Rank|1.0.1|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240143
1368380|pass|200804|chris@bingosnet.co.uk|Tie-STDOUT|1.03|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240144
1368381|pass|200804|chris@bingosnet.co.uk|Tie-Scalar-Decay|1.1.1|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240144
1368383|pass|200804|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240145
1368386|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|darwin-thread-multi-64int-2level|5.11.0 patch 33724|darwin|8.10.0|200804240145
1371563|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux|5.10.0 patch 33735|linux|2.6.22-1-k7|200804250734
1372822|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux|5.10.0 patch 33735|linux|2.6.22-1-k7|200804251054
1373335|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux|5.10.0 patch 33735|linux|2.6.22-1-k7|200804250926
1373398|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|i386-freebsd-64int|5.11.0 patch 33741|freebsd|7.0-release|200804251510
1373795|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200804251645
1375050|pass|200804|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-64int|5.11.0 patch 33741|freebsd|7.0-release|200804260104
1375153|pass|200804|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-freebsd-64int|5.11.0 patch 33741|freebsd|7.0-release|200804260154
1375157|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux|5.10.0 patch 33735|linux|2.6.22-1-k7|200804260235
1375177|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux|5.10.0 patch 33735|linux|2.6.22-1-k7|200804260242
1375466|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux|5.10.0 patch 33735|linux|2.6.22-1-k7|200804260413
1377678|pass|200804|chris@bingosnet.co.uk|Data-Compare|1.18|darwin-thread-multi-64int-2level|5.11.0 patch 33741|darwin|8.10.0|200804262114
1380106|pass|200804|alexchorny@gmail.com|Data-Compare|1.18|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200804271820
1380947|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-thread-multi|5.11.0 patch 33760|linux|2.6.22-1-k7|200804281120
1382101|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-thread-multi|5.11.0 patch 33760|linux|2.6.22-1-k7|200804281840
1382290|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-thread-multi|5.11.0 patch 33760|linux|2.6.22-1-k7|200804281946
1384214|pass|200804|daxim@cpan.org|CPU-Emulator-Z80|0.9|x86_64-linux|5.8.8|linux|2.6.20-gentoo-r8|200804290214
1384220|pass|200804|daxim@cpan.org|Devel-CheckLib|0.5|x86_64-linux|5.8.8|linux|2.6.20-gentoo-r8|200804290215
1384268|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-thread-multi|5.11.0 patch 33760|linux|2.6.22-1-k7|200804290043
1385201|pass|200804|daxim@cpan.org|Bryar|3.0|x86_64-linux|5.8.8|linux|2.6.20-gentoo-r8|200804290723
1385859|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-thread-multi|5.11.0 patch 33760|linux|2.6.22-1-k7|200804290843
1386203|pass|200804|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi|5.11.0 patch 33760|linux|2.6.22-1-k7|200804291034
1386295|pass|200804|tester@tcool.org ("ishigaki")|Data-Compare|1.18|i386-freebsd-64int|5.8.8|freebsd|5.4-release-p16|200804291212
1391895|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.3|darwin-thread-multi-64int-2level|5.11.0 patch 33741|darwin|8.10.0|200805011812
1392094|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux|5.10.0 patch 33755|linux|2.6.22-1-k7|200805011924
1392295|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux|5.10.0 patch 33755|linux|2.6.22-1-k7|200805012217
1392648|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux|5.10.0 patch 33755|linux|2.6.22-1-k7|200805012357
1392682|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux|5.10.0 patch 33755|linux|2.6.22-1-k7|200805012352
1392765|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux|5.10.0 patch 33755|linux|2.6.22-1-k7|200805012331
1393191|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux|5.10.0 patch 33755|linux|2.6.22-1-k7|200805020601
1394719|na|200805|srezic@cpan.org|Tie-STDOUT|1.03|i386-freebsd|5.5.5|freebsd|6.1-release|200805021325
1396129|na|200805|srezic@cpan.org|Number-Phone|1.581|i386-freebsd|5.5.5|freebsd|6.1-release|200805021916
1396253|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-64int|5.11.0 patch 33776|linux|2.6.22-1-k7|200805021945
1396966|na|200805|srezic@cpan.org|Bryar|3.0|i386-freebsd|5.5.5|freebsd|6.1-release|200805022257
1397502|pass|200805|gam3-cpanp@gam3.net|Devel-CheckOS|1.43|i486-linux-gnu|5.8.8|linux|2.6.22.10|200805020917
1397513|pass|200805|ewhipple@kubuntu-1.americanstaffing.org ((ewhipple))|Devel-CheckOS|1.43|i686-linux|5.10.0|linux|2.6.24-16-generic|200805021038
1397923|na|200805|srezic@cpan.org|Data-Transactional|1.0|i386-freebsd|5.5.5|freebsd|6.1-release|200805030108
1398049|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 33776|linux|2.6.22-1-k7|200805030108
1398157|pass|200805|srezic@cpan.org|Data-Hexdumper|1.3|i386-freebsd|5.5.5|freebsd|6.1-release|200805030059
1398349|pass|200805|jrockway@cpan.org|Devel-CheckOS|1.43|x86_64-linux|5.10.0|linux|2.6.22-3-amd64|200805021352
1398975|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.11.0 patch 33776|linux|2.6.22-1-k7|200805030441
1401073|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-64int|5.8.8 patch 33736|linux|2.6.22-1-k7|200805031752
1401597|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33776|linux|2.6.22-1-k7|200805031236
1402427|pass|200805|d@0x64.nl ("DEVOGON")|Devel-CheckLib|0.5|x86_64-linux-gnu|5.10.0|linux|2.6.24-1-amd64|200805032247
1402547|fail|200805|chris@bingosnet.co.uk|Palm-SMS|0.02|darwin-thread-multi-64int-2level|5.11.0 patch 33782|darwin|8.10.0|200805032158
1402664|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 33736|linux|2.6.22-1-k7|200805032304
1403160|pass|200805|d@0x64.nl ("DEVOGON")|XML-Tiny|1.11|x86_64-linux-gnu|5.10.0|linux|2.6.24-1-amd64|200805040108
1403333|pass|200805|d@0x64.nl ("DEVOGON")|Devel-CheckOS|1.43|x86_64-linux-gnu|5.10.0|linux|2.6.24-1-amd64|200805040137
1403502|pass|200805|d@0x64.nl ("DEVOGON")|Data-Compare|1.18|x86_64-linux-gnu|5.10.0|linux|2.6.24-1-amd64|200805040200
1404074|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 33736|linux|2.6.22-1-k7|200805040138
1404127|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 33736|linux|2.6.22-1-k7|200805040133
1404862|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.18|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 33782|openbsd|4.2|200805041041
1406078|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.8.8 patch 33736|linux|2.6.22-1-k7|200805041339
1406538|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.8.8 patch 33736|linux|2.6.22-1-k7|200805041535
1407046|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33776|linux|2.6.22-1-k7|200805041750
1407061|pass|200805|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200805041858
1407291|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 33776|linux|2.6.22-1-k7|200805041929
1408339|pass|200805|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 33782|openbsd|4.2|200805050643
1409006|pass|200805|chris@bingosnet.co.uk|Devel-CheckOS|1.43|darwin-thread-multi-64int-2level|5.11.0 patch 33783|darwin|8.10.0|200805051201
1410617|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.18|i386-netbsd-thread-multi-64int|5.11.0 patch 33788|netbsd|4.0|200805052144
1410773|pass|200805|frank.wiegand@gmail.com ("Frank Wiegand")|Devel-CheckOS|1.43|i686-linux|5.11.0 patch 33788|linux|2.6.24-17-generic|200805052323
1413440|pass|200805|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.11.0 patch 33788|netbsd|4.0|200805061317
1414429|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 33788|linux|2.6.22-1-k7|200805061700
1414480|fail|200805|chris@bingosnet.co.uk|Bryar|3.0|darwin-thread-multi-64int-2level|5.11.0 patch 33783|darwin|8.10.0|200805061751
1416098|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-64int|5.11.0 patch 33788|linux|2.6.22-1-k7|200805070052
1417178|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 33788|linux|2.6.22-1-k7|200805070124
1417798|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33788|linux|2.6.22-1-k7|200805070924
1418235|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33788|linux|2.6.22-1-k7|200805071058
1418752|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.11.0 patch 33788|linux|2.6.22-1-k7|200805071326
1419366|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.18|darwin-thread-multi-64int-2level|5.11.0 patch 33783|darwin|8.10.0|200805071751
1420893|pass|200805|srezic@cpan.org|CPAN-FindDependencies|2.0|i386-freebsd|5.10.0|freebsd|6.1-release|200805080836
1422492|pass|200805|artur+cpan@niif.spb.su (ARTO)|Data-Compare|1.18|i386-linux|5.10.0|linux|2.6.18-53.1.6.el5xen|200805081517
1423103|pass|200805|d@0x64.nl|Data-Compare|1.18|i486-linux-gnu|5.8.8|linux|2.6.22.10|200805081919
1423254|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.18|i86pc-solaris-thread-multi-64int|5.11.0 patch 33789|solaris|2.10|200805082054
1424466|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.3|darwin-thread-multi-64int-2level|5.11.0 patch 33783|darwin|8.10.0|200805090547
1425874|pass|200805|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200805090915
1426736|pass|200805|jj@jonallen.info ("JJ")|Data-Compare|1.18|darwin-2level|5.8.1|darwin|7.9.0|200805091542
1427856|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux|5.10.0 patch 33787|linux|2.6.22-1-k7|200805092020
1428749|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux|5.10.0 patch 33787|linux|2.6.22-1-k7|200805092230
1428861|pass|200805|bob@randomness.org.uk ("bob walker")|Devel-CheckOS|1.43|OpenBSD.i386-openbsd|5.8.8|openbsd|4.2|200805092240
1428880|pass|200805|bob@randomness.org.uk ("bob walker")|Devel-CheckOS|1.43|OpenBSD.i386-openbsd|5.10.0|openbsd|4.2|200805092244
1429209|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.18|darwin-thread-multi-64int-2level|5.11.0 patch 33797|darwin|8.10.0|200805092351
1430158|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.3|darwin-thread-multi-64int-2level|5.11.0 patch 33797|darwin|8.10.0|200805100357
1431052|pass|200805|frank.wiegand@gmail.com ("Frank Wiegand")|Devel-CheckOS|1.43|i686-linux|5.11.0 patch 33797|linux|2.6.24-17-generic|200805100805
1431653|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33798|linux|2.6.22-1-k7|200805101021
1432538|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 33798|linux|2.6.22-1-k7|200805101402
1432694|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33798|linux|2.6.22-1-k7|200805101402
1433056|pass|200805|jj@jonallen.info ("JJ")|Devel-CheckOS|1.43|darwin-2level|5.8.1|darwin|7.9.0|200805101404
1433313|pass|200805|icerider70@gmail.com ("ICERIDER")|XML-Tiny|1.11|i386-linux|5.8.8|linux|2.6.18-53.el5|200805100903
1433348|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.18|cygwin-thread-multi-64int|5.11.0 patch 33796|cygwin|1.5.24(0.15642)|200805101508
1433636|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-64int|5.11.0 patch 33798|linux|2.6.22-1-k7|200805101544
1434358|pass|200805|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200805101412
1434661|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200805101847
1435099|pass|200805|frank.wiegand@gmail.com ("Frank Wiegand")|Data-Compare|1.18|i686-linux|5.11.0 patch 33797|linux|2.6.24-17-generic|200805102320
1436231|pass|200805|jj@jonallen.info ("JJ")|XML-Tiny|1.11|darwin-2level|5.8.1|darwin|7.9.0|200805110257
1436370|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33811|linux|2.6.22-1-k7|200805110336
1437768|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-64int|5.11.0 patch 33811|linux|2.6.22-1-k7|200805110214
1438065|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33811|linux|2.6.22-1-k7|200805110149
1438346|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 33811|linux|2.6.22-1-k7|200805111359
1438484|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 33811|linux|2.6.22-1-k7|200805111438
1438874|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.18|i686-linux-thread-multi-64int|5.11.0 patch 33816|linux|2.6.21.5-smp|200805111621
1438928|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.11.0 patch 33811|linux|2.6.22-1-k7|200805111638
1439819|pass|200805|srezic@cpan.org|Data-Transactional|1.0|i386-freebsd|5.6.2|freebsd|6.1-release|200805112200
1439820|na|200805|srezic@cpan.org|Tie-STDOUT|1.03|i386-freebsd|5.6.2|freebsd|6.1-release|200805112201
1441505|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 33798|linux|2.6.22-1-k7|200805120600
1441873|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.11.0 patch 33798|linux|2.6.22-1-k7|200805120751
1443828|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux|5.10.0 patch 33787|linux|2.6.22-1-k7|200805121541
1444415|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux|5.10.0 patch 33787|linux|2.6.22-1-k7|200805121738
1444798|pass|200805|srezic@cpan.org|Data-Compare|1.19|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200805122042
1445168|pass|200805|srezic@cpan.org|Data-Hexdumper|1.3|i386-freebsd|5.6.2|freebsd|6.1-release|200805122155
1445749|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.18|i686-linux-64int|5.10.0 patch 33823|linux|2.6.22-1-k7|200805122232
1445774|pass|200805|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Data-Compare|1.19|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200805130718
1446070|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.10.0 patch 33823|linux|2.6.22-1-k7|200805130005
1446250|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200805130109
1446252|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.8.8 patch 33243|linux|2.6.22-1-k7|200805130110
1446254|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200805130111
1446260|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.8.8|linux|2.6.14|200805130112
1446262|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-thread-multi|5.8.8 patch 33430|linux|2.6.22-1-k7|200805130113
1446263|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200805130113
1446267|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200805130114
1446271|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-thread-multi-64int|5.10.0 patch 33628|linux|2.6.22-1-k7|200805130115
1446273|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200805130116
1446274|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200805130216
1446276|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200805130218
1446279|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200805130220
1446294|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.10.0 patch 33443|linux|2.6.22-1-k7|200805130127
1446649|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200805130443
1446652|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200805130443
1446655|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200805130444
1446656|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200805130444
1446659|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200805130445
1446665|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200805130445
1446669|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200805130446
1446671|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200805130447
1446674|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200805130447
1446676|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200805130448
1447210|pass|200805|d@0x64.nl ("DEVOGON")|Data-Compare|1.19|x86_64-linux-gnu|5.10.0|linux|2.6.18-6-xen-amd64|200805130904
1447307|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 33823|linux|2.6.22-1-k7|200805130810
1447348|pass|200805|david@cantrell.org.uk|Data-Compare|1.19|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200805130716
1447365|pass|200805|david@cantrell.org.uk|Data-Compare|1.19|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200805130219
1447405|pass|200805|rafl@debian.org|Data-Compare|1.19|i686-linux-multi|5.10.0|linux|2.6.24-rc5|200805130713
1447428|pass|200805|david@cantrell.org.uk|Data-Compare|1.19|i686-linux|5.8.8|linux|2.4.27-3-686|200805130827
1447433|pass|200805|david@cantrell.org.uk|Data-Compare|1.19|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200805130829
1447622|pass|200805|artur+cpan@niif.spb.su (ARTO)|Data-Compare|1.19|i386-linux|5.10.0|linux|2.6.18-53.1.6.el5xen|200805130901
1447705|pass|200805|jj@jonallen.info ("JJ")|Data-Compare|1.19|darwin-2level|5.8.1|darwin|7.9.0|200805130933
1447804|pass|200805|imacat@mail.imacat.idv.tw|Data-Compare|1.19|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200805131651
1447805|pass|200805|imacat@mail.imacat.idv.tw|Data-Compare|1.19|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200805131651
1447883|pass|200805|david@cantrell.org.uk|Data-Compare|1.19|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200805131008
1447892|pass|200805|david@cantrell.org.uk|Data-Compare|1.19|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200805130735
1448176|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200805131111
1448508|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.18|i386-freebsd-64int|5.11.0 patch 33822|freebsd|7.0-release|200805131027
1448683|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200805131253
1448737|pass|200805|david@cantrell.org.uk|Data-Compare|1.19|i686-linux|5.6.2|linux|2.4.27-3-686|200805131306
1448903|pass|200805|hohokus@gmail.com|Data-Compare|1.19|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200805130857
1449105|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|alpha-linux-thread-multi|5.11.0 patch 33822|linux|2.6.18-5-alpha-generic|200805131449
1449658|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 33823|linux|2.6.22-1-k7|200805131025
1449817|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|darwin-thread-multi-64int-2level|5.11.0 patch 33822|darwin|8.10.0|200805131730
1449960|pass|200805|jj@jonallen.info ("JJ")|CPU-Emulator-Memory|1.1001|darwin-2level|5.8.1|darwin|7.9.0|200805131811
1449999|pass|200805|jj@jonallen.info ("JJ")|Devel-CheckLib|0.5|darwin-2level|5.8.1|darwin|7.9.0|200805131821
1450090|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200805131934
1450191|pass|200805|d@0x64.nl (DEVOGON)|Data-Compare|1.19|i686-linux-thread-multi|5.11.0 patch 33261|linux|2.6.22-3-486|200805131835
1450235|pass|200805|g.grigelionis@computer.org|Data-Compare|1.19|i586-linux-thread-multi|5.8.8|linux|2.6.16|200805131958
1450615|pass|200805|srezic@cpan.org|Data-Compare|1.19|i386-freebsd|5.10.0|freebsd|6.1-release|200805132205
1451204|pass|200805|srezic@cpan.org|Data-Compare|1.19|i386-freebsd|5.6.2|freebsd|6.1-release|200805140013
1451206|pass|200805|srezic@cpan.org|Data-Compare|1.19|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200805140015
1451390|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 33822|openbsd|4.2|200805132353
1451490|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200805140016
1451562|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-64int|5.10.0 patch 33823|linux|2.6.22-1-k7|200805140029
1451906|pass|200805|rlane10@wi.rr.com|Data-Compare|1.19|darwin-2level-ld|5.10.0|darwin|8.11.0|200805131335
1452642|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 33824|linux|2.6.22-1-k7|200805140623
1453081|pass|200805|srezic@cpan.org|Data-Compare|1.19|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200805140836
1453184|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 33824|linux|2.6.22-1-k7|200805140900
1453262|pass|200805|jj@jonallen.info ("JJ")|Number-Phone|1.581|darwin-2level|5.8.1|darwin|7.9.0|200805140826
1453544|pass|200805|claco@chrislaco.com ("Christopher H. Laco")|Data-Compare|1.19|darwin-thread-multi-2level|5.8.6|darwin|8.0|200805140110
1453574|pass|200805|artur+cpan@niif.spb.su (ARTO)|Devel-CheckOS|1.43|i386-linux|5.10.0|linux|2.6.18-53.1.6.el5xen|200805141023
1454231|pass|200805|chris@bingosnet.co.uk|Devel-CheckOS|1.43|darwin-thread-multi-64int-2level|5.11.0 patch 33822|darwin|8.10.0|200805141259
1454795|pass|200805|stro@cpan.org|Data-Compare|1.19|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200805141727
1455003|pass|200805|stro@cpan.org|Data-Compare|1.19|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200805141817
1455200|pass|200805|jj@jonallen.info ("JJ")|XML-DoubleEncodedEntities|1.0|darwin-2level|5.8.1|darwin|7.9.0|200805141701
1455314|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33824|linux|2.6.22-1-k7|200805141105
1455526|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 33824|linux|2.6.22-1-k7|200805141019
1456155|pass|200805|d@0x64.nl ("DEVOGON")|Data-Compare|1.19|i686-linux|5.10.0|linux|2.6.22-3-486|200805142209
1456214|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200805142209
1456357|pass|200805|srezic@cpan.org|Data-Compare|1.19|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200805142248
1456779|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200805142347
1456839|pass|200805|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.19|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200805141907
1457182|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-64int|5.11.0 patch 33824|linux|2.6.22-1-k7|200805150105
1458610|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33824|linux|2.6.22-1-k7|200805150835
1459230|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200805151214
1460527|pass|200805|stro@cpan.org|Devel-CheckOS|1.43|armv5tejl-linux|5.8.8|linux|2.6.17.14|200805151846
1460623|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200805151656
1460959|pass|200805|dillman@ria.buffalo.edu|Data-Compare|1.19|sun4-solaris|5.10.0|solaris|2.9|200805150915
1461199|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 33823|linux|2.6.22-1-k7|200805151803
1461793|pass|200805|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805151626
1461874|pass|200805|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.19|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805151640
1462583|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.8.8 patch 33819|linux|2.6.22-1-k7|200805160026
1462813|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.8.8 patch 33819|linux|2.6.22-1-k7|200805160218
1463177|pass|200805|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200805160229
1463182|pass|200805|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Z80|0.9|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200805160229
1463203|pass|200805|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200805160233
1463474|fail|200805|chris@bingosnet.co.uk|Bryar|3.0|darwin-thread-multi-64int-2level|5.11.0 patch 33824|darwin|8.10.0|200805160418
1463606|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|darwin-thread-multi-64int-2level|5.11.0 patch 33824|darwin|8.10.0|200805160448
1464426|pass|200805|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805160243
1464429|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200805160143
1465183|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 33819|linux|2.6.22-1-k7|200805161049
1465966|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 33832|linux|2.6.22-1-k7|200805161339
1466142|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 33832|linux|2.6.22-1-k7|200805161335
1466168|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.11.0 patch 33831|netbsd|4.0|200805160737
1466808|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33832|linux|2.6.22-1-k7|200805161613
1466853|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33832|linux|2.6.22-1-k7|200805161715
1467732|pass|200805|david@cantrell.org.uk|CPAN-FindDependencies|2.0|IP35-irix|5.8.8|irix|6.5|200805162203
1467738|pass|200805|david@cantrell.org.uk|CPU-Emulator-Memory|1.1001|IP35-irix|5.8.8|irix|6.5|200805162204
1467769|pass|200805|david@cantrell.org.uk|Class-CanBeA|1.2|IP35-irix|5.8.8|irix|6.5|200805162209
1467784|pass|200805|david@cantrell.org.uk|Data-Compare|1.19|IP35-irix|5.8.8|irix|6.5|200805162211
1467790|pass|200805|david@cantrell.org.uk|Data-Hexdumper|1.3|IP35-irix|5.8.8|irix|6.5|200805162212
1467792|pass|200805|david@cantrell.org.uk|Data-Transactional|1.0|IP35-irix|5.8.8|irix|6.5|200805162213
1467796|pass|200805|david@cantrell.org.uk|Devel-CheckLib|0.5|IP35-irix|5.8.8|irix|6.5|200805162213
1467804|pass|200805|david@cantrell.org.uk|Devel-CheckOS|1.43|IP35-irix|5.8.8|irix|6.5|200805162214
1467811|pass|200805|david@cantrell.org.uk|File-Find-Rule-Permissions|1.3|IP35-irix|5.8.8|irix|6.5|200805162215
1467815|pass|200805|david@cantrell.org.uk|File-Overwrite|1.0|IP35-irix|5.8.8|irix|6.5|200805162215
1467819|pass|200805|david@cantrell.org.uk|Games-Dice-Advanced|1.1|IP35-irix|5.8.8|irix|6.5|200805162216
1467823|pass|200805|david@cantrell.org.uk|NestedMap|1.0|IP35-irix|5.8.8|irix|6.5|200805162216
1467841|pass|200805|david@cantrell.org.uk|Net-Random|2.0|IP35-irix|5.8.8|irix|6.5|200805162219
1467846|pass|200805|david@cantrell.org.uk|Number-Phone|1.581|IP35-irix|5.8.8|irix|6.5|200805162220
1467856|pass|200805|david@cantrell.org.uk|Number-Phone-UK-DetailedLocations|1.3|IP35-irix|5.8.8|irix|6.5|200805162221
1467869|pass|200805|chris@bingosnet.co.uk|XML-Tiny|1.11|darwin-thread-multi-64int-2level|5.11.0 patch 33824|darwin|8.10.0|200805162123
1467879|pass|200805|david@cantrell.org.uk|Palm-TreoPhoneCallDB|1.1|IP35-irix|5.8.8|irix|6.5|200805162225
1467884|pass|200805|david@cantrell.org.uk|Statistics-ChiSquare|0.5|IP35-irix|5.8.8|irix|6.5|200805162225
1467885|pass|200805|david@cantrell.org.uk|Statistics-SerialCorrelation|1.1|IP35-irix|5.8.8|irix|6.5|200805162226
1467893|pass|200805|david@cantrell.org.uk|Sub-WrapPackages|1.2|IP35-irix|5.8.8|irix|6.5|200805162227
1467896|pass|200805|david@cantrell.org.uk|Tie-Hash-Longest|1.1|IP35-irix|5.8.8|irix|6.5|200805162227
1467900|pass|200805|david@cantrell.org.uk|Tie-Hash-Rank|1.0.1|IP35-irix|5.8.8|irix|6.5|200805162228
1467903|pass|200805|david@cantrell.org.uk|Tie-STDOUT|1.03|IP35-irix|5.8.8|irix|6.5|200805162228
1467906|pass|200805|david@cantrell.org.uk|Tie-Scalar-Decay|1.1.1|IP35-irix|5.8.8|irix|6.5|200805162229
1467908|pass|200805|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|IP35-irix|5.8.8|irix|6.5|200805162229
1467909|pass|200805|david@cantrell.org.uk|XML-Tiny|1.11|IP35-irix|5.8.8|irix|6.5|200805162230
1468355|pass|200805|srezic@cpan.org|CPAN-FindDependencies|2.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200805170020
1468495|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|cygwin-thread-multi-64int|5.11.0 patch 33836|cygwin|1.5.25(0.15642)|200805162356
1468894|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-64int|5.11.0 patch 33832|linux|2.6.22-1-k7|200805170117
1469173|pass|200805|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.581|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200805170251
1469324|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 33832|linux|2.6.22-1-k7|200805170336
1469658|pass|200805|jj@jonallen.info ("JJ")|Bryar|3.0|darwin-2level|5.8.1|darwin|7.9.0|200805170606
1470250|pass|200805|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200805170326
1470540|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200805170334
1471030|pass|200805|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.11.0 patch 33831|netbsd|4.0|200805170512
1473276|pass|200805|chris@bingosnet.co.uk|Devel-CheckOS|1.43|cygwin-thread-multi-64int|5.11.0 patch 33836|cygwin|1.5.25(0.15642)|200805171845
1474419|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.3|i686-linux-64int|5.8.8 patch 33819|linux|2.6.22-1-k7|200805172158
1474499|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 33819|linux|2.6.22-1-k7|200805172209
1474669|pass|200805|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805171833
1474670|pass|200805|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Z80|0.9|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805171833
1474676|pass|200805|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805171834
1475128|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 33819|linux|2.6.22-1-k7|200805172348
1475216|pass|200805|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.581|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805180005
1475628|pass|200805|g.grigelionis@computer.org|Data-Compare|1.19|i586-linux-thread-multi|5.8.5|linux|2.6.8.1|200805172247
1475926|pass|200805|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805180241
1476902|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200805180007
1477233|pass|200805|jj@jonallen.info ("JJ")|File-Find-Rule-Permissions|1.3|darwin-2level|5.8.1|darwin|7.9.0|200805180904
1479601|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33845|linux|2.6.22-1-k7|200805181231
1480382|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200805182014
1480561|pass|200805|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Data-Hexdumper|1.4|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200805190443
1480879|pass|200805|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.0|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805181630
1480882|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33845|linux|2.6.22-1-k7|200805180852
1481334|fail|200805|dagolden@cpan.org (DAGOLDEN)|Bryar|3.0|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805181746
1481748|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 33845|linux|2.6.22-1-k7|200805182258
1482086|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 33845|linux|2.6.22-1-k7|200805190039
1482184|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200805190202
1482186|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200805190202
1482187|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200805190202
1482189|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200805190202
1482190|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200805190203
1482191|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200805190203
1482193|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200805190203
1482194|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200805190204
1482196|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200805190204
1482197|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200805190204
1482284|pass|200805|stro@cpan.org|Data-Hexdumper|1.4|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200805190430
1482543|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200805190422
1482545|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200805190423
1482546|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200805190425
1482735|pass|200805|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805190027
1482909|pass|200805|jj@jonallen.info ("JJ")|Data-Transactional|1.0|darwin-2level|5.8.1|darwin|7.9.0|200805190618
1482957|pass|200805|jj@jonallen.info ("JJ")|Tie-STDOUT|1.03|darwin-2level|5.8.1|darwin|7.9.0|200805190640
1483249|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i686-linux-thread-multi-64int-ld|5.11.0 patch 33858|linux|2.6.16.38|200805190900
1483386|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0 patch 33443|linux|2.6.22-1-k7|200805191049
1483387|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-thread-multi-64int|5.10.0 patch 33412|linux|2.6.22-1-k7|200805191050
1483454|pass|200805|imacat@mail.imacat.idv.tw|Data-Hexdumper|1.4|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200805191703
1483455|pass|200805|imacat@mail.imacat.idv.tw|Data-Hexdumper|1.4|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200805191703
1483524|pass|200805|david@cantrell.org.uk|Data-Hexdumper|1.4|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200805191018
1483536|pass|200805|jj@jonallen.info ("JJ")|Data-Hexdumper|1.4|darwin-2level|5.8.1|darwin|7.9.0|200805191021
1483579|pass|200805|david@cantrell.org.uk|Data-Hexdumper|1.4|i686-linux|5.8.8|linux|2.4.27-3-686|200805191026
1483593|pass|200805|david@cantrell.org.uk|Data-Hexdumper|1.4|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200805191028
1483622|pass|200805|david@cantrell.org.uk|Data-Hexdumper|1.4|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200805190431
1483630|pass|200805|david@cantrell.org.uk|Data-Hexdumper|1.4|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200805190718
1483690|pass|200805|david@cantrell.org.uk|Data-Hexdumper|1.4|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200805190944
1483703|pass|200805|david@cantrell.org.uk|Data-Hexdumper|1.4|IP35-irix|5.8.8|irix|6.5|200805191145
1484127|pass|200805|david@cantrell.org.uk|Data-Hexdumper|1.4|i686-linux|5.6.2|linux|2.4.27-3-686|200805191221
1484190|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|alpha-linux-thread-multi|5.11.0 patch 33858|linux|2.6.18-5-alpha-generic|200805191236
1484331|pass|200805|stro@cpan.org|Data-Hexdumper|1.4|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200805191506
1484424|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.11.0 patch 33858|linux|2.6.16.38|200805191331
1485485|pass|200805|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|1.3|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805191154
1486143|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 33858|linux|2.6.22-1-k7|200805191312
1486218|pass|200805|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200805191411
1486232|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|alpha-linux-thread-multi|5.11.0 patch 33858|linux|2.6.18-5-alpha-generic|200805191914
1486326|pass|200805|jj@jonallen.info ("JJ")|Palm-TreoPhoneCallDB|1.1|darwin-2level|5.8.1|darwin|7.9.0|200805191942
1486432|pass|200805|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.0|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805191502
1486442|pass|200805|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805191503
1486855|pass|200805|dagolden@cpan.org (DAGOLDEN)|Palm-TreoPhoneCallDB|1.1|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805191602
1487259|pass|200805|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.11.0 patch 33858|linux|2.6.16.38|200805192153
1487680|pass|200805|srezic@cpan.org|Data-Hexdumper|1.4|i386-freebsd|5.6.2|freebsd|6.1-release|200805200038
1488161|pass|200805|feldt@nhn.ou.edu|Data-Compare|1.19|sun4-solaris-thread-multi|5.8.8|solaris|2.8|200805191428
1489311|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33858|linux|2.6.22-1-k7|200805200546
1489810|pass|200805|srezic@cpan.org|Data-Hexdumper|1.4|i386-freebsd|5.10.0|freebsd|6.1-release|200805200901
1489931|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 33858|linux|2.6.22-1-k7|200805200241
1490380|pass|200805|srezic@cpan.org|Data-Hexdumper|1.4|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200805201058
1490505|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33858|linux|2.6.22-1-k7|200805200147
1490736|pass|200805|d@0x64.nl ("DEVOGON")|Data-Hexdumper|1.4|x86_64-linux-gnu|5.10.0|linux|2.6.18-6-xen-amd64|200805201311
1491016|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 33871|openbsd|4.1|200805201248
1491047|pass|200805|dagolden@cpan.org (DAGOLDEN)|Net-Random|2.0|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805200752
1491605|pass|200805|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805200852
1492814|pass|200805|srezic@cpan.org|Data-Hexdumper|1.4|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200805201725
1493168|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200805201705
1493397|pass|200805|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805201231
1493723|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 33858|linux|2.6.22-1-k7|200805201734
1493871|pass|200805|chris@bingosnet.co.uk|Devel-CheckOS|1.43|alpha-linux-thread-multi|5.11.0 patch 33858|linux|2.6.18-5-alpha-generic|200805201917
1494038|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 33871|openbsd|4.1|200805202004
1494083|pass|200805|srezic@cpan.org|Data-Hexdumper|1.4|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200805202115
1494478|pass|200805|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805201620
1494758|pass|200805|srezic@cpan.org|Data-Hexdumper|1.4|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200805202316
1494873|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 33887|linux|2.6.22-1-k7|200805202306
1495109|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 33887|linux|2.6.22-1-k7|200805202251
1495311|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 33887|linux|2.6.22-1-k7|200805210027
1495328|pass|200805|chris@bingosnet.co.uk|Acme-Scurvy-Whoreson-BilgeRat|1.1|darwin-thread-multi-64int-2level|5.11.0 patch 33852|darwin|8.10.0|200805210132
1495828|pass|200805|jj@jonallen.info ("JJ")|Net-Random|2.0|darwin-2level|5.8.1|darwin|7.9.0|200805210434
1496297|pass|200805|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 33871|openbsd|4.1|200805210724
1497266|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200805211319
1497286|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200805211325
1497634|pass|200805|jj@jonallen.info ("JJ")|Number-Phone-UK-DetailedLocations|1.3|darwin-2level|5.8.1|darwin|7.9.0|200805211454
1497883|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33887|linux|2.6.22-1-k7|200805210933
1497924|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33887|linux|2.6.22-1-k7|200805210936
1498842|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200805212052
1499156|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 33887|linux|2.6.22-1-k7|200805212132
1499608|pass|200805|d@0x64.nl (DEVOGON)|Data-Hexdumper|1.4|i686-linux-thread-multi|5.11.0 patch 33261|linux|2.6.22-3-486|200805220051
1500298|pass|200805|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.0|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805220037
1500367|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200805220152
1500524|pass|200805|dagolden@cpan.org (DAGOLDEN)|Class-CanBeA|1.2|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805220119
1501526|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1001|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200805220657
1501834|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200805220845
1502059|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200805220843
1502062|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 33243|linux|2.6.22-1-k7|200805220844
1502068|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200805220845
1502069|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8|linux|2.6.14|200805220845
1502073|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-thread-multi|5.8.8 patch 33430|linux|2.6.22-1-k7|200805220847
1502087|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200805220851
1502088|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200805220852
1504694|pass|200805|d@0x64.nl ("DEVOGON")|Devel-CheckOS|1.43|i486-linux-gnu|5.10.0|linux|2.6.24.4|200805222319
1504918|pass|200805|chris@bingosnet.co.uk|Devel-CheckOS|1.43|darwin-thread-multi-64int-2level|5.11.0 patch 33852|darwin|8.10.0|200805222259
1505260|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|darwin-thread-multi-64int-2level|5.11.0 patch 33852|darwin|8.10.0|200805230048
1505550|fail|200805|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.02|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805230140
1505637|pass|200805|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805230230
1506120|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200805230724
1506208|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200805230759
1506315|pass|200805|jj@jonallen.info ("JJ")|Games-Dice-Advanced|1.1|darwin-2level|5.8.1|darwin|7.9.0|200805230737
1506856|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 33911|linux|2.6.22-1-k7|200805231023
1506900|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200805230934
1507112|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 33911|linux|2.6.22-1-k7|200805230941
1507301|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 33911|linux|2.6.22-1-k7|200805231205
1507645|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33911|linux|2.6.22-1-k7|200805231316
1508364|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33911|linux|2.6.22-1-k7|200805231302
1508500|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200805231410
1509961|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 33911|linux|2.6.22-1-k7|200805231127
1510117|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200805231809
1510229|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|cygwin-thread-multi-64int|5.10.0|cygwin|1.5.24(0.15642)|200805231850
1510416|pass|200805|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200805231459
1510860|pass|200805|jj@jonallen.info ("JJ")|Sub-WrapPackages|1.2|darwin-2level|5.8.1|darwin|7.9.0|200805232145
1511003|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200805232101
1512245|pass|200805|dagolden@cpan.org (DAGOLDEN)|Statistics-SerialCorrelation|1.1|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805240206
1512408|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|cygwin-thread-multi-64int|5.10.0|cygwin|1.5.24(0.15642)|200805240348
1513036|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 33916|linux|2.6.22-1-k7|200805240848
1513111|pass|200805|srezic@cpan.org|File-Find-Rule-Permissions|1.3|i386-freebsd|5.10.0|freebsd|6.1-release|200805240835
1513937|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 33916|linux|2.6.22-1-k7|200805241100
1515510|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200805241515
1515514|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33916|linux|2.6.22-1-k7|200805241615
1515633|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33916|linux|2.6.22-1-k7|200805241635
1515743|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 33916|linux|2.6.22-1-k7|200805241652
1516065|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200805241728
1516784|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200805241904
1516931|fail|200805|srezic@cpan.org|Bryar|3.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200805241940
1517129|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200805241305
1517199|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.581|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200805241040
1517200|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 33916|linux|2.6.22-1-k7|200805241039
1517791|pass|200805|dagolden@cpan.org (DAGOLDEN)|Statistics-ChiSquare|0.5|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805241538
1517792|pass|200805|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Longest|1.1|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805241538
1518444|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200805250045
1519128|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|0.9|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200805250431
1519161|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.0|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200805250425
1519188|fail|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Bryar|3.0|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200805250422
1520092|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200805251403
1520295|pass|200805|dagolden@cpan.org (DAGOLDEN)|NestedMap|1.0|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805251044
1520358|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200805251605
1520824|pass|200805|srezic@cpan.org|Data-Transactional|1.0|i386-freebsd|5.10.0|freebsd|6.1-release|200805251949
1520825|pass|200805|srezic@cpan.org|Tie-STDOUT|1.03|i386-freebsd|5.10.0|freebsd|6.1-release|200805251949
1521312|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi|5.11.0 patch 33919|linux|2.6.22-1-k7|200805252129
1521493|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-thread-multi|5.11.0 patch 33919|linux|2.6.22-1-k7|200805252213
1521967|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-thread-multi|5.11.0 patch 33919|linux|2.6.22-1-k7|200805252315
1522401|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-thread-multi|5.11.0 patch 33919|linux|2.6.22-1-k7|200805260053
1522787|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200805260024
1523008|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-thread-multi|5.11.0 patch 33919|linux|2.6.22-1-k7|200805260301
1523621|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-thread-multi|5.11.0 patch 33919|linux|2.6.22-1-k7|200805260529
1523732|pass|200805|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.10.0 patch 33921|linux|2.4.26|200805260503
1525227|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|darwin-thread-multi-64int-2level|5.10.0 patch 33921|darwin|8.10.0|200805261723
1525775|pass|200805|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.10.0 patch 33921|linux|2.4.26|200805262215
1526175|unknown|200805|dagolden@cpan.org (DAGOLDEN)|Acme-Licence|1.0|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805262000
1526260|pass|200805|stro@cpan.org|Data-Compare|1.19|i686-linux|5.8.8|linux|2.4.27-2-386|200805270227
1527261|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.8.8 patch 33935|linux|2.6.22-1-k7|200805270739
1527373|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.8.8 patch 33935|linux|2.6.22-1-k7|200805270839
1528611|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 33935|linux|2.6.22-1-k7|200805271252
1528936|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 33935|linux|2.6.22-1-k7|200805271348
1529854|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 33935|linux|2.6.22-1-k7|200805271519
1529942|pass|200805|jj@jonallen.info ("JJ")|Devel-CheckOS|1.43|darwin-2level|5.8.2|darwin|7.9.0|200805271516
1531701|unknown|200805|dagolden@cpan.org (DAGOLDEN)|Acme-Pony|1.1.2|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805271600
1531753|pass|200805|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Rank|1.0.1|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805271604
1531855|pass|200805|dagolden@cpan.org (DAGOLDEN)|Tie-Scalar-Decay|1.1.1|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805271618
1532012|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 33935|linux|2.6.22-1-k7|200805272231
1533859|pass|200805|jj@jonallen.info ("JJ")|Data-Hexdumper|1.4|darwin-2level|5.8.2|darwin|7.9.0|200805280504
1534264|pass|200805|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200805280623
1534518|pass|200805|jj@jonallen.info ("JJ")|Data-Compare|1.19|darwin-2level|5.8.2|darwin|7.9.0|200805280702
1534830|pass|200805|chris@bingosnet.co.uk|Number-Phone|1.6|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200805280835
1534833|pass|200805|chris@bingosnet.co.uk|Number-Phone|1.6|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200805280836
1534837|pass|200805|chris@bingosnet.co.uk|Number-Phone|1.6|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200805280837
1534968|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.6|i686-linux-64int|5.10.0 patch 33443|linux|2.6.22-1-k7|200805281011
1534970|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.6|i686-linux-thread-multi-64int|5.10.0 patch 33628|linux|2.6.22-1-k7|200805281011
1534971|pass|200805|chris@bingosnet.co.uk|Number-Phone|1.581|darwin-thread-multi-64int-2level|5.10.0 patch 33921|darwin|8.10.0|200805280911
1534974|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.6|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200805281011
1534976|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.6|i686-linux-thread-multi|5.8.8 patch 33430|linux|2.6.22-1-k7|200805281012
1534978|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.6|i686-linux-64int|5.8.8 patch 33243|linux|2.6.22-1-k7|200805281012
1534979|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.6|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200805281013
1534981|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.6|i686-linux-64int|5.8.8|linux|2.6.14|200805281013
1534982|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.6|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200805281013
1534986|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.6|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200805281014
1534989|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.6|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200805281015
1535216|pass|200805|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Number-Phone|1.6|sun4-solaris|5.10.0|solaris|2.9|200805281141
1535363|pass|200805|imacat@mail.imacat.idv.tw|Number-Phone|1.6|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200805281807
1535406|pass|200805|imacat@mail.imacat.idv.tw|Number-Phone|1.6|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200805281814
1535658|pass|200805|david@cantrell.org.uk|Number-Phone|1.6|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200805281052
1535682|pass|200805|david@cantrell.org.uk|Number-Phone|1.6|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200805281157
1535806|pass|200805|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805280721
1536198|pass|200805|stro@cpan.org|Number-Phone|1.6|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200805281549
1536545|pass|200805|david@cantrell.org.uk|Number-Phone|1.6|i686-linux|5.6.2|linux|2.4.27-3-686|200805281657
1537012|pass|200805|srezic@cpan.org|Number-Phone|1.6|i386-freebsd|5.10.0|freebsd|6.1-release|200805281511
1537078|pass|200805|chris@bingosnet.co.uk|Number-Phone|1.6|i686-linux-thread-multi-64int-ld|5.10.0 patch 33942|linux|2.4.26|200805281715
1537130|pass|200805|david@cantrell.org.uk|Number-Phone|1.6|IP35-irix|5.8.8|irix|6.5|200805281838
1537263|pass|200805|david@cantrell.org.uk|Number-Phone|1.6|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200805280739
1537595|pass|200805|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805281352
1537697|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 33939|linux|2.6.22-1-k7|200805281949
1537706|pass|200805|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Hexdumper|1.4|sun4-solaris|5.10.0|solaris|2.9|200805282003
1537736|pass|200805|srezic@cpan.org|Number-Phone|1.6|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200805281901
1537752|pass|200805|david@cantrell.org.uk|Number-Phone|1.6|i686-linux|5.8.8|linux|2.4.27-3-686|200805282041
1537844|pass|200805|srezic@cpan.org|Data-Hexdumper|1.4|i386-freebsd|5.5.5|freebsd|6.1-release|200805281852
1537953|pass|200805|david@cantrell.org.uk|Number-Phone|1.6|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200805281321
1538003|pass|200805|david@cantrell.org.uk|Number-Phone|1.6|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200805282101
1538119|pass|200805|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805281438
1538401|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 33939|linux|2.6.22-1-k7|200805282009
1539141|pass|200805|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805281600
1539258|pass|200805|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.19|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805281618
1539381|pass|200805|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Compare|1.19|sun4-solaris|5.10.0|solaris|2.9|200805282235
1540914|pass|200805|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805281955
1541207|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 33939|linux|2.6.22-1-k7|200805290225
1541712|pass|200805|stro@cpan.org|Number-Phone|1.6|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200805290428
1541900|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33939|linux|2.6.22-1-k7|200805290306
1542288|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 33939|linux|2.6.22-1-k7|200805290451
1542404|pass|200805|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.10.0 patch 33942|linux|2.4.26|200805290404
1542411|pass|200805|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200805290305
1542913|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33939|linux|2.6.22-1-k7|200805290508
1543246|pass|200805|jj@jonallen.info ("JJ")|Devel-CheckOS|1.43|darwin-2level|5.8.3|darwin|7.9.0|200805290544
1543714|pass|200805|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805290151
1543716|pass|200805|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Z80|0.9|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805290151
1543723|pass|200805|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805290152
1544485|pass|200805|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805290403
1545530|pass|200805|frank.wiegand@gmail.com ("Frank Wiegand")|Number-Phone|1.581|i686-linux|5.11.0 patch 33939|linux|2.6.24-17-generic|200805291306
1546019|pass|200805|jj@jonallen.info ("JJ")|Number-Phone|1.6|darwin-2level|5.8.3|darwin|7.9.0|200805291421
1546378|fail|200805|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.0|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805291059
1546701|pass|200805|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Number-Phone|1.6|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200805300056
1546823|fail|200805|dagolden@cpan.org (DAGOLDEN)|Bryar|3.0|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805291223
1547548|pass|200805|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|1.3|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805291502
1547613|pass|200805|jj@jonallen.info ("JJ")|Number-Phone|1.6|darwin-2level|5.8.2|darwin|7.9.0|200805292012
1547844|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.6|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200805292232
1547908|pass|200805|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.0|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805291631
1547923|pass|200805|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805291632
1548260|pass|200805|dagolden@cpan.org (DAGOLDEN)|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805291807
1549334|pass|200805|dagolden@cpan.org (DAGOLDEN)|Net-Random|2.0|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805300138
1549677|pass|200805|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805300301
1550907|pass|200805|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805300326
1551105|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 33949|linux|2.6.22-1-k7|200805301000
1551379|pass|200805|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805300436
1551554|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 33949|linux|2.6.22-1-k7|200805301024
1552861|pass|200805|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.0|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805300915
1553080|pass|200805|dagolden@cpan.org (DAGOLDEN)|Class-CanBeA|1.2|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805301002
1553233|pass|200805|jj@jonallen.info ("JJ")|Data-Hexdumper|1.4|darwin-2level|5.8.3|darwin|7.9.0|200805301529
1553452|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200805301607
1553827|pass|200805|chris@bingosnet.co.uk|Number-Phone|1.6|i686-linux-thread-multi-64int-ld|5.10.0 patch 33947|linux|2.4.26|200805301707
1554311|pass|200805|jj@jonallen.info ("JJ")|Data-Compare|1.19|darwin-2level|5.8.3|darwin|7.9.0|200805301832
1555790|pass|200805|chris@bingosnet.co.uk|Number-Phone|1.6|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200805302248
1556369|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33949|linux|2.6.22-1-k7|200805310153
1556382|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 33949|linux|2.6.22-1-k7|200805310149
1557706|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33949|linux|2.6.22-1-k7|200805310526
1558052|pass|200805|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.10.0 patch 33947|linux|2.4.26|200805310524
1558204|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 33949|linux|2.6.22-1-k7|200805310629
1558565|pass|200805|jj@jonallen.info ("JJ")|XML-Tiny|1.11|darwin-2level|5.8.2|darwin|7.9.0|200805310703
1559072|na|200805|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|1.3|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200805310424
1559315|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200805311047
1559566|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200805311146
1559768|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200805311229
1560142|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200805311240
1560523|fail|200805|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.02|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805310810
1560668|pass|200805|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805310821
1561580|pass|200805|chris@bingosnet.co.uk|Data-Compare|1.19|darwin-thread-multi-64int-2level|5.10.0 patch 33947|darwin|8.10.0|200805311505
1561739|pass|200805|chris@bingosnet.co.uk|Number-Phone|1.6|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200805311427
1563597|pass|200805|dagolden@cpan.org (DAGOLDEN)|Statistics-SerialCorrelation|1.1|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805311346
1563631|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200805311949
1563925|pass|200805|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200805312016
1564012|pass|200805|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Longest|1.1|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805311430
1564013|pass|200805|dagolden@cpan.org (DAGOLDEN)|Statistics-ChiSquare|0.5|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805311431
1565202|pass|200805|dagolden@cpan.org (DAGOLDEN)|NestedMap|1.0|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805311633
1565287|pass|200805|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200805311647
1566031|pass|200805|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200805311827
1566359|unknown|200805|dagolden@cpan.org (DAGOLDEN)|Acme-Licence|1.0|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200805311901
1567194|pass|200806|icerider70@gmail.com ("ICERIDER")|Data-Compare|1.19|i386-linux|5.8.8|linux|2.6.18-53.el5|200806010053
1567352|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806010110
1567593|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.19|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806010135
1567652|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Pony|1.1.2|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200806010141
1567710|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Rank|1.0.1|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200806010147
1567753|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Scalar-Decay|1.1.1|i686-linux-thread-multi-64int-ld|5.10.0 patch 33823|linux|2.6.24-16-generic|200806010153
1569849|pass|200806|chris@bingosnet.co.uk|Devel-CheckOS|1.43|cygwin-thread-multi-64int|5.10.0 patch 33947|cygwin|1.5.25(0.15642)|200806011153
1570315|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806010845
1570728|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806010925
1570773|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806010933
1571548|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806011106
1571625|pass|200806|chris@bingosnet.co.uk|Number-Phone|1.6|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200806011615
1571654|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.19|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806011119
1572524|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200806011401
1572574|fail|200806|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200806011421
1574018|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806011810
1574265|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806011854
1574269|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Z80|0.9|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806011854
1574295|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806011857
1574313|pass|200806|chris@bingosnet.co.uk|Number-Phone|1.6|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200806020000
1575798|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806020329
1575812|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806020331
1575813|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Z80|0.9|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806020331
1575819|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806020332
1576599|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806020112
1578359|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 33986|linux|2.6.22-1-k7|200806021115
1578579|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 33986|linux|2.6.22-1-k7|200806021128
1578637|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.0|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806020544
1578699|pass|200806|chris@bingosnet.co.uk|Number-Phone|1.6|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200806021049
1578853|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.0|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806020604
1578906|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 33986|linux|2.6.22-1-k7|200806021204
1579398|fail|200806|dagolden@cpan.org (DAGOLDEN)|Bryar|3.0|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806020705
1579434|pass|200806|dagolden@cpan.org (DAGOLDEN)|Palm-TreoPhoneCallDB|1.1|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200806020709
1580012|pass|200806|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200806021223
1581408|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|1.3|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806021328
1581689|pass|200806|chris@bingosnet.co.uk|Number-Phone|1.6|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200806021910
1581836|pass|200806|jj@jonallen.info ("JJ")|XML-Tiny|1.11|darwin-2level|5.8.3|darwin|7.9.0|200806021935
1581849|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.0|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806021441
1581858|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806021442
1582149|pass|200806|flemming@ipk-gatersleben.de|Data-Compare|1.19|x86_64-linux|5.8.5|linux|2.6.9-55.0.9.elsmp|200806021423
1582419|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 33986|linux|2.6.22-1-k7|200806022203
1582436|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|1.3|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806021604
1582778|pass|200806|dagolden@cpan.org (DAGOLDEN)|Palm-TreoPhoneCallDB|1.1|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806021631
1583949|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.0|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806021819
1583985|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806021821
1584228|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 33986|linux|2.6.22-1-k7|200806030046
1584692|pass|200806|chris@bingosnet.co.uk|Number-Phone|1.6|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200806030026
1584702|pass|200806|dagolden@cpan.org (DAGOLDEN)|Net-Random|2.0|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806021928
1584989|pass|200806|dagolden@cpan.org (DAGOLDEN)|Palm-TreoPhoneCallDB|1.1|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806021949
1585271|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 33986|linux|2.6.22-1-k7|200806030216
1585349|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806030023
1587211|pass|200806|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806030010
1587592|pass|200806|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806030118
1589241|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.0|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806030433
1589570|pass|200806|dagolden@cpan.org (DAGOLDEN)|Class-CanBeA|1.2|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806030510
1590004|pass|200806|dagolden@cpan.org (DAGOLDEN)|Net-Random|2.0|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806030604
1590985|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806030830
1591364|pass|200806|icerider70@gmail.com ("ICERIDER")|Data-Hexdumper|1.4|i386-linux|5.8.8|linux|2.6.18-53.el5|200806030838
1591564|pass|200806|chris@bingosnet.co.uk|Number-Phone|1.6|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200806031308
1592681|pass|200806|chris@bingosnet.co.uk|Number-Phone|1.6|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200806031827
1594222|fail|200806|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.02|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806031907
1594268|pass|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806031916
1596106|pass|200806|chris@bingosnet.co.uk|Number-Phone|1.6|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200806040506
1596136|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-SerialCorrelation|1.1|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806040103
1596418|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Longest|1.1|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806040139
1596420|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-ChiSquare|0.5|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806040140
1597040|pass|200806|dagolden@cpan.org (DAGOLDEN)|NestedMap|1.0|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806040324
1598207|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Licence|1.0|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806040538
1599062|pass|200806|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806040715
1599765|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Pony|1.1.2|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806040824
1599815|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Rank|1.0.1|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806040829
1599859|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Scalar-Decay|1.1.1|i686-linux-64int-ld|5.10.0|linux|2.6.24-16-generic|200806040835
1600101|pass|200806|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806040852
1602119|pass|200806|chris@bingosnet.co.uk|Number-Phone|1.6|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200806041622
1602224|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200806041822
1602252|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200806041824
1602709|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806041318
1602894|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200806041850
1603182|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806041406
1604234|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.0|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806041543
1604457|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806041611
1604702|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.19|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806041631
1604828|pass|200806|dagolden@cpan.org (DAGOLDEN)|Class-CanBeA|1.2|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806041642
1605376|fail|200806|jj@jonallen.info ("JJ")|CPAN-FindDependencies|2.0|darwin-2level|5.8.1|darwin|7.9.0|200806042232
1606004|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200806050018
1606511|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200806050051
1606666|pass|200806|icerider70@gmail.com ("ICERIDER")|Number-Phone|1.6|i386-linux|5.8.8|linux|2.6.18-53.el5|200806041827
1607012|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200806050157
1607495|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806050045
1609203|pass|200806|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|CPU-Emulator-Memory|1.1001|sun4-solaris|5.10.0|solaris|2.9|200806050719
1609209|pass|200806|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|CPU-Emulator-Z80|0.9|sun4-solaris|5.10.0|solaris|2.9|200806050720
1609274|pass|200806|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Devel-CheckLib|0.5|sun4-solaris|5.10.0|solaris|2.9|200806050731
1609546|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806050239
1609548|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Z80|0.9|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806050239
1609553|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806050240
1610500|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806050511
1610872|pass|200806|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200806051117
1612263|pass|200806|rurban@x-ray.at (Reini Urban)|Data-Compare|1.19|cygwin-thread-multi-64int|5.10.0|cygwin|1.5.25(0.15642)|200806051131
1612459|pass|200806|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200806051454
1613348|fail|200806|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.02|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806051119
1613350|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806051119
1613481|pass|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806051131
1614296|fail|200806|dagolden@cpan.org (DAGOLDEN)|Bryar|3.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806051253
1615657|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|1.3|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806051522
1616492|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806051650
1616507|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806051652
1616957|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-SerialCorrelation|1.1|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806051741
1617027|pass|200806|dagolden@cpan.org (DAGOLDEN)|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806051749
1617452|pass|200806|david@cantrell.org.uk|XML-Tiny|1.11|IP35-irix-64int|5.8.8|irix|6.5|200806060027
1617506|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Longest|1.1|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806051831
1617508|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-ChiSquare|0.5|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806051832
1617612|pass|200806|david@cantrell.org.uk|CPU-Emulator-Memory|1.1001|IP35-irix-64int|5.8.8|irix|6.5|200806060044
1617620|pass|200806|david@cantrell.org.uk|CPU-Emulator-Z80|0.9|IP35-irix-64int|5.8.8|irix|6.5|200806060044
1618807|pass|200806|dagolden@cpan.org (DAGOLDEN)|Net-Random|2.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806060122
1619151|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200806060405
1619228|pass|200806|dagolden@cpan.org (DAGOLDEN)|NestedMap|1.0|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806060216
1619306|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806060225
1620600|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Licence|1.0|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806060107
1622038|pass|200806|stro@cpan.org|CPU-Emulator-Memory|1.1001|i686-linux|5.8.8|linux|2.4.27-2-386|200806060117
1622052|pass|200806|stro@cpan.org|CPU-Emulator-Z80|0.9|i686-linux|5.8.8|linux|2.4.27-2-386|200806060118
1622201|pass|200806|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200806061127
1623332|pass|200806|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806060916
1623843|pass|200806|andy@hexten.net|Data-Hexdumper|1.4|darwin-thread-multi-2level|5.10.0|darwin|9.2.2|200806061504
1624056|pass|200806|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806061023
1624257|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Pony|1.1.2|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806061039
1624325|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Rank|1.0.1|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806061044
1624400|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Scalar-Decay|1.1.1|i686-linux-64int-ld|5.8.8 patch 33935|linux|2.6.24-16-generic|200806061052
1625162|pass|200806|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200806061703
1626149|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806061413
1626487|pass|200806|dagolden@cpan.org (DAGOLDEN)|Class-CanBeA|1.2|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806061453
1627601|pass|200806|chris@bingosnet.co.uk|Number-Phone|1.6|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200806062213
1628001|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806061748
1629230|pass|200806|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|XML-DoubleEncodedEntities|1.0|sun4-solaris|5.10.0|solaris|2.9|200806070226
1629294|fail|200806|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.02|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806070036
1629367|pass|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806070044
1629841|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806070146
1630504|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806070320
1631286|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-SerialCorrelation|1.1|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806070101
1631562|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Longest|1.1|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806070138
1631563|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-ChiSquare|0.5|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806070138
1632128|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806070307
1632247|pass|200806|dagolden@cpan.org (DAGOLDEN)|NestedMap|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806070335
1632360|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.19|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806070354
1633163|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Licence|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806070545
1634343|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Pony|1.1.2|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806070823
1634390|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Rank|1.0.1|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806070827
1634430|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Scalar-Decay|1.1.1|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200806070833
1635103|pass|200806|andy@hexten.net|Data-Compare|1.19|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200806071437
1635106|pass|200806|andy@hexten.net|Data-Hexdumper|1.4|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200806071437
1635216|pass|200806|andy@hexten.net|Data-Compare|1.19|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200806071445
1635218|pass|200806|andy@hexten.net|Data-Hexdumper|1.4|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200806071445
1635354|pass|200806|andy@hexten.net|Data-Hexdumper|1.4|i686-linux-thread-multi|5.10.0|linux|2.6.20-16-server|200806071455
1635515|pass|200806|andy@hexten.net|Number-Phone|1.6|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200806071505
1635990|pass|200806|andy@hexten.net|Number-Phone|1.6|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200806071524
1636361|pass|200806|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-freebsd-64int|5.11.0 patch 34005|freebsd|7.0-release|200806071453
1636805|pass|200806|stro@cpan.org|Data-Hexdumper|1.4|i686-linux|5.8.8|linux|2.4.27-2-386|200806071730
1637399|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806071448
1637953|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806071623
1638005|pass|200806|stro@cpan.org|Data-Transactional|1.0|i686-linux|5.8.8|linux|2.4.27-2-386|200806072033
1638178|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806071700
1639376|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806080352
1639451|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.19|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806080007
1640367|pass|200806|jj@jonallen.info ("JJ")|Devel-CheckLib|0.5|darwin-2level|5.8.2|darwin|7.9.0|200806080749
1641848|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806080705
1641850|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Z80|0.9|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806080705
1641862|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806080707
1642539|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806080853
1642540|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Z80|0.9|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806080854
1642550|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806080855
1643221|pass|200806|jj@jonallen.info ("JJ")|CPU-Emulator-Memory|1.1001|darwin-2level|5.8.3|darwin|7.9.0|200806081536
1643232|pass|200806|jj@jonallen.info ("JJ")|CPU-Emulator-Z80|0.9|darwin-2level|5.8.3|darwin|7.9.0|200806081537
1643318|pass|200806|jj@jonallen.info ("JJ")|Devel-CheckLib|0.5|darwin-2level|5.8.3|darwin|7.9.0|200806081554
1643424|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806081116
1644483|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806081431
1645296|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.0|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806081757
1646787|pass|200806|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-64int|5.11.0 patch 34005|freebsd|7.0-release|200806090359
1646850|pass|200806|chris@bingosnet.co.uk|Devel-CheckOS|1.43|darwin-thread-multi-64int-2level|5.10.0 patch 33976|darwin|8.10.0|200806090948
1647408|pass|200806|jj@jonallen.info ("JJ")|CPU-Emulator-Memory|1.1001|darwin-2level|5.8.2|darwin|7.9.0|200806091145
1647420|pass|200806|chris@bingosnet.co.uk|XML-Tiny|1.11|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200806091147
1647683|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.0|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806090759
1648404|pass|200806|dagolden@cpan.org (DAGOLDEN)|Bryar|3.0|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806090946
1648608|pass|200806|chris@bingosnet.co.uk|Number-Phone|1.6|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200806091520
1649407|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|1.3|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806091208
1649840|pass|200806|dagolden@cpan.org (DAGOLDEN)|Bryar|3.0|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806091325
1650133|pass|200806|chris@bingosnet.co.uk|Number-Phone|1.6|i386-freebsd-64int|5.11.0 patch 34005|freebsd|7.0-release|200806091233
1650326|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.0|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806091553
1650336|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806091554
1651712|pass|200806|dagolden@cpan.org (DAGOLDEN)|Net-Random|2.0|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806091940
1652060|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806100044
1653013|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|1.3|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806100317
1653402|pass|200806|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806100019
1653717|pass|200806|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806100117
1654962|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.0|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806100426
1655173|pass|200806|dagolden@cpan.org (DAGOLDEN)|Class-CanBeA|1.2|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806100502
1655194|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.0|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806100505
1655216|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806100508
1655523|pass|200806|jj@jonallen.info ("JJ")|File-Overwrite|1.0|darwin-2level|5.8.1|darwin|7.9.0|200806101102
1655746|pass|200806|jj@jonallen.info ("JJ")|XML-DoubleEncodedEntities|1.0|darwin-2level|5.8.2|darwin|7.9.0|200806101129
1657381|fail|200806|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.02|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806101014
1657448|pass|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806101021
1658997|pass|200806|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-freebsd-64int|5.11.0 patch 34005|freebsd|7.0-release|200806100709
1659175|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-SerialCorrelation|1.1|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806101422
1659193|pass|200806|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|CPAN-FindDependencies|2.0|sun4-solaris|5.10.0|solaris|2.9|200806102025
1659368|pass|200806|chris@bingosnet.co.uk|XML-Tiny|1.11|alpha-linux-thread-multi|5.11.0 patch 34031|linux|2.6.18-5-alpha-generic|200806101942
1659428|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Longest|1.1|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806101451
1659430|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-ChiSquare|0.5|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806101451
1660151|pass|200806|dagolden@cpan.org (DAGOLDEN)|NestedMap|1.0|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806101620
1660908|pass|200806|dagolden@cpan.org (DAGOLDEN)|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806101750
1661054|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Licence|1.0|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806101804
1661124|pass|200806|chris@bingosnet.co.uk|Data-Compare|1.19|i386-freebsd-64int|5.11.0 patch 34005|freebsd|7.0-release|200806102322
1662159|pass|200806|jj@jonallen.info ("JJ")|XML-DoubleEncodedEntities|1.0|darwin-2level|5.8.3|darwin|7.9.0|200806110209
1662363|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Pony|1.1.2|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806110130
1662410|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Rank|1.0.1|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806110134
1662449|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Scalar-Decay|1.1.1|i686-linux-64int-ld|5.8.8|linux|2.6.24-16-generic|200806110138
1664439|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806110439
1664738|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806110545
1665737|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806110746
1665964|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806110814
1666351|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806110922
1666368|pass|200806|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200806111417
1666437|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.19|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806110933
1667846|pass|200806|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806111412
1668478|pass|200806|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806111638
1669283|pass|200806|andy@hexten.net|Data-Hexdumper|1.4|i686-linux-thread-multi|5.10.0|linux|2.6.18-8.el5|200806120142
1669424|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.8.8 patch 34037|linux|2.6.22-1-k7|200806120259
1669539|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806120117
1669540|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Z80|0.9|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806120117
1669547|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806120118
1669897|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 34037|linux|2.6.22-1-k7|200806120423
1670596|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806120003
1671876|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.0|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806120241
1672276|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 34037|linux|2.6.22-1-k7|200806120933
1672311|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806120336
1672452|fail|200806|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Bryar|3.0|sun4-solaris|5.10.0|solaris|2.9|200806120955
1672725|pass|200806|dagolden@cpan.org (DAGOLDEN)|Bryar|3.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806120443
1673498|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|1.3|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806120646
1673962|pass|200806|dagolden@cpan.org (DAGOLDEN)|Class-CanBeA|1.2|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806120744
1674081|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806120801
1674093|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806120802
1674468|pass|200806|dagolden@cpan.org (DAGOLDEN)|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806120851
1675737|pass|200806|dagolden@cpan.org (DAGOLDEN)|Net-Random|2.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806121133
1676188|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806121226
1676852|pass|200806|kamensky@cpan.org ("Moshe Kamensky")|Data-Compare|1.19|sun4-solaris|5.10.0|solaris|2.8|200806121006
1677239|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.8.8 patch 34037|linux|2.6.22-1-k7|200806121957
1677784|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 34037|linux|2.6.22-1-k7|200806122038
1678451|pass|200806|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806121559
1679107|pass|200806|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806121659
1680250|fail|200806|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.02|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806121900
1680311|pass|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806121912
1680879|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806130037
1681135|pass|200806|dagolden@cpan.org (DAGOLDEN)|Class-CanBeA|1.2|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806130111
1682379|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 34037|linux|2.6.22-1-k7|200806130553
1684521|fail|200806|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.02|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806130829
1684598|pass|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806130837
1685330|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-SerialCorrelation|1.1|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806130950
1686169|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Longest|1.1|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806131105
1686172|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-ChiSquare|0.5|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806131105
1686674|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-SerialCorrelation|1.1|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806131150
1686957|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Longest|1.1|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806131217
1686959|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-ChiSquare|0.5|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806131218
1687989|pass|200806|dagolden@cpan.org (DAGOLDEN)|NestedMap|1.0|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806131500
1688343|pass|200806|dagolden@cpan.org (DAGOLDEN)|NestedMap|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806131528
1688392|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 34034|linux|2.6.22-1-k7|200806132133
1688860|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 34034|linux|2.6.22-1-k7|200806132202
1689558|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1001|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200806132254
1689563|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|1.0|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200806132255
1689568|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|1.0|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200806132255
1689573|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Memory|1.1001|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200806132256
1689579|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|1.0|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200806132257
1689623|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|1.0|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200806132300
1689644|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|1.0|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200806132302
1689666|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|1.0|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200806132304
1689729|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|1.0|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200806132257
1689733|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|1.0|i686-linux-64int|5.8.8|linux|2.6.14|200806132259
1689826|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Licence|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806131713
1690007|pass|200806|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200806131927
1690730|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.10.0 patch 34034|linux|2.6.22-1-k7|200806132226
1690743|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.10.0 patch 34034|linux|2.6.22-1-k7|200806132221
1690801|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Licence|1.0|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806131827
1691351|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Pony|1.1.2|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806131909
1691414|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Rank|1.0.1|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806131913
1691484|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Scalar-Decay|1.1.1|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-16-generic|200806131917
1691805|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 34034|linux|2.6.22-1-k7|200806140110
1693311|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|CPU-Emulator-Z80|1.0|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200806140324
1693934|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Pony|1.1.2|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806140325
1693991|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Rank|1.0.1|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806140331
1694046|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Scalar-Decay|1.1.1|i686-linux-thread-multi-64int-ld|5.8.8 patch 33819|linux|2.6.24-16-generic|200806140339
1694082|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806140343
1694324|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|i686-linux-64int-ld|5.11.0 patch 34053|linux|2.6.24-16-generic|200806140001
1694392|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0 patch 34034|linux|2.6.22-1-k7|200806140606
1694436|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200806140510
1694616|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux-64int-ld|5.11.0 patch 34053|linux|2.6.24-16-generic|200806140025
1696842|pass|200806|david@cantrell.org.uk|CPU-Emulator-Z80|1.0|i686-linux|5.8.8|linux|2.4.27-3-686|200806141100
1696860|pass|200806|david@cantrell.org.uk|CPU-Emulator-Z80|1.0|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200806141107
1696874|pass|200806|david@cantrell.org.uk|CPU-Emulator-Z80|1.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200806141101
1696950|pass|200806|david@cantrell.org.uk|CPU-Emulator-Z80|1.0|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200806141132
1696955|pass|200806|david@cantrell.org.uk|CPU-Emulator-Z80|1.0|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200806140533
1696965|pass|200806|david@cantrell.org.uk|CPU-Emulator-Z80|1.0|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200806141035
1697410|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux-64int-ld|5.11.0 patch 34053|linux|2.6.24-16-generic|200806140818
1697529|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux-64int-ld|5.11.0 patch 34053|linux|2.6.24-16-generic|200806140837
1697947|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux-64int-ld|5.11.0 patch 34053|linux|2.6.24-16-generic|200806140939
1698033|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.19|i686-linux-64int-ld|5.11.0 patch 34053|linux|2.6.24-16-generic|200806140950
1698059|pass|200806|jj@jonallen.info ("JJ")|CPU-Emulator-Z80|1.0|darwin-2level|5.8.3|darwin|7.9.0|200806141453
1698529|pass|200806|imacat@mail.imacat.idv.tw|CPU-Emulator-Z80|1.0|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200806142300
1698538|pass|200806|imacat@mail.imacat.idv.tw|CPU-Emulator-Z80|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200806142301
1699335|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-64int-ld|5.11.0 patch 34053|linux|2.6.21.5-smp|200806141722
1699342|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|i686-linux-thread-multi-64int-ld|5.11.0 patch 34053|linux|2.6.21.5-smp|200806141723
1699778|pass|200806|stro@cpan.org|XML-Tiny|1.11|i386-freebsd-64int|5.8.8|freebsd|7.0-release|200806142055
1700297|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux-64int-ld|5.11.0 patch 34053|linux|2.6.24-16-generic|200806141629
1700782|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux-64int-ld|5.11.0 patch 34053|linux|2.6.24-16-generic|200806141752
1701090|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200806150036
1702144|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.0|i686-linux-64int-ld|5.11.0 patch 34053|linux|2.6.24-16-generic|200806150218
1702831|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34053|linux|2.6.22-1-k7|200806151122
1702859|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34053|linux|2.6.22-1-k7|200806151142
1702893|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 34053|linux|2.6.22-1-k7|200806151145
1703095|pass|200806|chris@bingosnet.co.uk|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200806151047
1703430|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34053|linux|2.6.22-1-k7|200806151344
1703470|pass|200806|chris@bingosnet.co.uk|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.4.26|200806151250
1703635|pass|200806|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|File-Find-Rule-Permissions|1.3|sun4-solaris|5.10.0|solaris|2.9|200806151434
1704329|fail|200806|dagolden@cpan.org (DAGOLDEN)|Bryar|3.0|i686-linux-64int-ld|5.11.0 patch 34053|linux|2.6.24-16-generic|200806151039
1704932|fail|200806|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200806151653
1704995|fail|200806|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200806151702
1705126|fail|200806|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200806151716
1705195|fail|200806|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200806151722
1705271|fail|200806|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200806151727
1705279|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|1.3|i686-linux-64int-ld|5.11.0 patch 34053|linux|2.6.24-16-generic|200806151230
1705323|fail|200806|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200806151733
1706395|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34053|linux|2.6.22-1-k7|200806152106
1706425|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34053|linux|2.6.22-1-k7|200806152058
1706452|pass|200806|chris@bingosnet.co.uk|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.11.0 patch 34053|linux|2.6.21.5-smp|200806151921
1707633|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200806160103
1707653|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.0|i686-linux-64int-ld|5.11.0 patch 34053|linux|2.6.24-16-generic|200806160101
1707663|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux-64int-ld|5.11.0 patch 34053|linux|2.6.24-16-generic|200806160102
1707973|pass|200806|dagolden@cpan.org (DAGOLDEN)|Palm-TreoPhoneCallDB|1.1|i686-linux-64int-ld|5.11.0 patch 34053|linux|2.6.24-16-generic|200806160144
1708315|pass|200806|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|CPU-Emulator-Z80|1.0|sun4-solaris|5.10.0|solaris|2.9|200806160502
1708637|pass|200806|chris@bingosnet.co.uk|XML-Tiny|1.11|darwin-thread-multi-64int-2level|5.10.0 patch 34034|darwin|8.10.0|200806160805
1708672|pass|200806|andy@hexten.net|Data-Compare|1.19|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200806160903
1708673|pass|200806|andy@hexten.net|Data-Hexdumper|1.4|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200806160905
1709088|pass|200806|icerider70@gmail.com ("ICERIDER")|CPU-Emulator-Z80|1.0|i386-linux|5.8.8|linux|2.6.18-53.el5|200806160648
1709801|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806161222
1709934|pass|200806|chris@bingosnet.co.uk|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200806161651
1710635|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806161430
1710712|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806161446
1711100|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Transactional|1.02|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200806162138
1711102|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Transactional|1.02|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200806162139
1711104|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Transactional|1.02|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200806162139
1711107|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Transactional|1.02|i686-linux-64int|5.8.8|linux|2.6.14|200806162139
1711109|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Transactional|1.02|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200806162139
1711111|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Transactional|1.02|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200806162140
1711114|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Transactional|1.02|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200806162140
1711130|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806161542
1711213|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.19|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806161554
1711344|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Transactional|1.02|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200806162139
1711788|pass|200806|jj@jonallen.info ("JJ")|Number-Phone|1.6|darwin-2level|5.8.1|darwin|7.9.0|200806162225
1712009|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200806162253
1712014|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200806162255
1712052|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200806162257
1712140|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200806162312
1712173|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200806162314
1712180|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200806162316
1713291|pass|200806|icerider70@gmail.com ("ICERIDER")|Data-Transactional|1.02|i386-linux|5.8.8|linux|2.6.18-53.el5|200806170216
1713400|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.02|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806170325
1713550|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806170343
1713618|pass|200806|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Data-Transactional|1.02|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200806171259
1713683|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806170407
1713736|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806170019
1713937|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.4.26|200806170606
1713951|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.4.26|200806170608
1714719|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.0|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806170416
1714825|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200806170905
1714858|pass|200806|imacat@mail.imacat.idv.tw|Data-Transactional|1.02|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200806171643
1714961|pass|200806|imacat@mail.imacat.idv.tw|Data-Transactional|1.02|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200806171700
1714995|pass|200806|stro@cpan.org|Data-Transactional|1.01|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200806171208
1714999|pass|200806|stro@cpan.org|Data-Transactional|1.02|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200806171208
1715113|fail|200806|dagolden@cpan.org (DAGOLDEN)|Bryar|3.0|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806170534
1715183|pass|200806|david@cantrell.org.uk|Data-Transactional|1.01|i686-linux|5.8.8|linux|2.4.27-3-686|200806171047
1715190|pass|200806|david@cantrell.org.uk|Data-Transactional|1.01|i686-linux|5.6.2|linux|2.4.27-3-686|200806171048
1715208|pass|200806|david@cantrell.org.uk|Data-Transactional|1.02|i686-linux|5.8.8|linux|2.4.27-3-686|200806171047
1715210|pass|200806|david@cantrell.org.uk|Data-Transactional|1.02|i686-linux|5.6.2|linux|2.4.27-3-686|200806171048
1715252|pass|200806|jj@jonallen.info ("JJ")|Data-Transactional|1.02|darwin-2level|5.8.2|darwin|7.9.0|200806171056
1715318|pass|200806|david@cantrell.org.uk|Data-Transactional|1.01|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200806171105
1715320|pass|200806|david@cantrell.org.uk|Data-Transactional|1.02|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200806171105
1715342|pass|200806|david@cantrell.org.uk|Data-Transactional|1.01|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200806170509
1715353|pass|200806|david@cantrell.org.uk|Data-Transactional|1.02|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200806170510
1715365|pass|200806|david@cantrell.org.uk|Data-Transactional|1.01|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200806171106
1715367|pass|200806|david@cantrell.org.uk|Data-Transactional|1.02|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200806171107
1715383|pass|200806|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Transactional|1.02|sun4-solaris|5.10.0|solaris|2.9|200806171214
1715448|pass|200806|david@cantrell.org.uk|Data-Transactional|1.01|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200806171059
1715465|pass|200806|david@cantrell.org.uk|Data-Transactional|1.02|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200806171101
1715871|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|1.3|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806170756
1716180|pass|200806|david@cantrell.org.uk|Data-Compare|1.19|IP35-irix-64int|5.8.8|irix|6.5|200806171455
1716184|pass|200806|david@cantrell.org.uk|Data-Transactional|1.01|IP35-irix-64int|5.8.8|irix|6.5|200806171455
1716186|pass|200806|david@cantrell.org.uk|Data-Transactional|1.02|IP35-irix-64int|5.8.8|irix|6.5|200806171455
1716276|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806170912
1716633|pass|200806|david@cantrell.org.uk|Data-Transactional|1.01|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200806171400
1716636|pass|200806|david@cantrell.org.uk|Data-Transactional|1.02|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200806171400
1716688|pass|200806|dagolden@cpan.org (DAGOLDEN)|Palm-TreoPhoneCallDB|1.1|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806171006
1716761|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200806170909
1716770|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200806170910
1716945|pass|200806|chris@bingosnet.co.uk|XML-Tiny|1.11|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200806171504
1717130|pass|200806|robertl@apnic.net ("Robert Loomans")|CPAN-FindDependencies|2.0|darwin-2level|5.8.8|darwin|9.2.2|200806171530
1717223|pass|200806|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Tie-STDOUT|1.03|sun4-solaris|5.10.0|solaris|2.9|200806171701
1717464|pass|200806|srezic@cpan.org|CPU-Emulator-Z80|1.0|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200806171732
1718386|pass|200806|dagolden@cpan.org (DAGOLDEN)|Net-Random|2.0|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806171319
1718665|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.8.8 patch 34066|linux|2.6.22-1-k7|200806171931
1719000|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.8.8 patch 34066|linux|2.6.22-1-k7|200806171958
1719112|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806171420
1719207|pass|200806|srezic@cpan.org|Data-Transactional|1.02|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200806172032
1719292|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200806171946
1719316|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200806171948
1719348|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200806171950
1719382|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200806171952
1719408|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200806171954
1719445|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200806171956
1719472|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200806171958
1719487|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200806172000
1719502|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200806172001
1719510|pass|200806|srezic@cpan.org|Data-Transactional|1.02|i386-freebsd|5.6.2|freebsd|6.1-release|200806172058
1719534|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200806172004
1719991|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200806172058
1720006|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200806172059
1720028|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200806172101
1720046|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 34066|linux|2.6.22-1-k7|200806172202
1720049|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200806172103
1720057|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200806172104
1720065|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200806172105
1720087|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200806172107
1720104|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200806172108
1720126|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200806172110
1720153|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200806172111
1720412|pass|200806|srezic@cpan.org|Data-Transactional|1.02|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200806172156
1720585|na|200806|srezic@cpan.org|Data-Transactional|1.02|i386-freebsd|5.5.5|freebsd|6.1-release|200806172255
1720855|pass|200806|david@cantrell.org.uk|Devel-CheckOS|1.43|IP35-irix-64int|5.8.8|irix|6.5|200806172325
1721186|pass|200806|srezic@cpan.org|Data-Transactional|1.02|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200806180001
1721360|pass|200806|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806171817
1721943|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 34066|linux|2.6.22-1-k7|200806172057
1722051|pass|200806|srezic@cpan.org|Data-Transactional|1.02|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200806180106
1722175|pass|200806|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806171926
1722205|pass|200806|srezic@cpan.org|Data-Transactional|1.02|i386-freebsd|5.10.0|freebsd|6.1-release|200806180128
1722423|pass|200806|chris@bingosnet.co.uk|XML-Tiny|1.11|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200806180029
1722727|pass|200806|chris@bingosnet.co.uk|Data-Compare|1.19|darwin-thread-multi-64int-2level|5.10.0 patch 34065|darwin|8.10.0|200806180144
1722737|pass|200806|srezic@cpan.org|CPU-Emulator-Z80|1.0|i386-freebsd|5.10.0|freebsd|6.1-release|200806180244
1723607|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.0|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806180258
1723848|pass|200806|srezic@cpan.org|CPU-Emulator-Z80|1.0|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200806180539
1723888|pass|200806|dagolden@cpan.org (DAGOLDEN)|Class-CanBeA|1.2|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806180348
1725628|pass|200806|srezic@cpan.org|Number-Phone|1.6|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200806180837
1726053|pass|200806|srezic@cpan.org|CPU-Emulator-Z80|1.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200806180944
1726116|na|200806|srezic@cpan.org|Number-Phone|1.6|i386-freebsd|5.5.5|freebsd|6.1-release|200806180953
1726167|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 34066|linux|2.6.22-1-k7|200806180942
1726246|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 34066|linux|2.6.22-1-k7|200806180959
1726885|fail|200806|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.02|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806180509
1727011|pass|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806180517
1727998|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200806181118
1728032|pass|200806|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200806180257
1728632|pass|200806|chris@bingosnet.co.uk|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.4.26|200806181255
1729448|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-SerialCorrelation|1.1|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806180919
1729848|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-ChiSquare|0.5|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806180951
1729855|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Longest|1.1|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806180950
1730104|pass|200806|srezic@cpan.org|Number-Phone|1.6|i386-freebsd|5.6.2|freebsd|6.1-release|200806181532
1730602|pass|200806|stro@cpan.org|File-Find-Rule-Permissions|1.3|i686-linux|5.8.8|linux|2.4.27-2-386|200806181452
1730977|pass|200806|dagolden@cpan.org (DAGOLDEN)|NestedMap|1.0|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806181129
1731015|pass|200806|stro@cpan.org|File-Overwrite|1.0|i686-linux|5.8.8|linux|2.4.27-2-386|200806181535
1732000|pass|200806|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Palm-TreoPhoneCallDB|1.1|sun4-solaris|5.10.0|solaris|2.9|200806181923
1732178|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Licence|1.0|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806181340
1733402|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Pony|1.1.2|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806181558
1733445|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Rank|1.0.1|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806181602
1733518|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Scalar-Decay|1.1.1|i686-linux-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806181606
1735264|pass|200806|srezic@cpan.org|CPU-Emulator-Z80|1.0|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200806190000
1735338|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.19|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806181844
1735339|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.02|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806181844
1736559|pass|200806|stro@cpan.org|Data-Transactional|1.02|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200806190450
1736819|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806190232
1737043|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i386-netbsd-thread-multi-64int|5.11.0 patch 34073|netbsd|4.0|200806190418
1737055|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|i386-netbsd-thread-multi-64int|5.11.0 patch 34073|netbsd|4.0|200806190419
1737077|pass|200806|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.11.0 patch 34073|netbsd|4.0|200806190421
1737080|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i386-netbsd-thread-multi-64int|5.11.0 patch 34073|netbsd|4.0|200806190421
1737088|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i386-netbsd-thread-multi-64int|5.11.0 patch 34073|netbsd|4.0|200806190422
1737193|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806190352
1737404|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806190037
1737459|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806190048
1737510|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806190053
1737535|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.19|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806190058
1738004|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806190237
1738286|pass|200806|chris@bingosnet.co.uk|Number-Phone|1.6|darwin-thread-multi-64int-2level|5.10.0 patch 34065|darwin|8.10.0|200806190841
1739632|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806190843
1739702|pass|200806|srezic@cpan.org|Number-Phone|1.6|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200806191453
1740244|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806191013
1740729|pass|200806|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.11.0 patch 34073|netbsd|4.0|200806191653
1741329|pass|200806|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.11.0 patch 34073|netbsd|4.0|200806191749
1741735|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806191338
1741954|pass|200806|srezic@cpan.org|Tie-STDOUT|1.03|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200806192004
1742423|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200806192013
1742430|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200806192014
1742448|pass|200806|chris@bingosnet.co.uk|Data-Compare|1.19|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200806192017
1742449|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200806192017
1742455|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200806192019
1742894|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806191719
1743652|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806191852
1744687|pass|200806|srezic@cpan.org|Number-Phone-UK-DetailedLocations|1.3|i386-freebsd|5.6.2|freebsd|6.1-release|200806200318
1744728|pass|200806|dagolden@cpan.org (DAGOLDEN)|Bryar|3.0|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806200133
1745499|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200806200427
1745508|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200806200428
1745580|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|1.3|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806200338
1745643|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200806200444
1745655|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200806200446
1745714|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806200355
1746263|na|200806|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806200113
1746585|pass|200806|dagolden@cpan.org (DAGOLDEN)|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806200154
1747063|pass|200806|srezic@cpan.org|Number-Phone-UK-DetailedLocations|1.3|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200806200848
1747364|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34074|linux|2.6.22-1-k7|200806200933
1747482|pass|200806|srezic@cpan.org|Number-Phone-UK-DetailedLocations|1.3|i386-freebsd|5.10.0|freebsd|6.1-release|200806200954
1748185|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34074|linux|2.6.22-1-k7|200806201012
1748835|pass|200806|srezic@cpan.org|Net-Random|2.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200806201319
1749601|pass|200806|dagolden@cpan.org (DAGOLDEN)|Net-Random|2.0|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806200926
1749788|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200806201454
1749808|pass|200806|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200806201455
1749834|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200806201500
1749849|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200806201502
1749881|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806201013
1750206|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|1.3|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806201137
1750249|pass|200806|chris@bingosnet.co.uk|XML-Tiny|1.11|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200806201605
1750719|na|200806|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806201458
1751000|pass|200806|chris@bingosnet.co.uk|Data-Compare|1.19|i686-linux-thread-multi-64int-ld|5.10.0 patch 34065|linux|2.4.26|200806202127
1752441|pass|200806|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200806210053
1752510|pass|200806|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806210004
1752784|pass|200806|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806210050
1753897|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806210326
1753972|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.0|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806210338
1754219|pass|200806|dagolden@cpan.org (DAGOLDEN)|Class-CanBeA|1.2|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806210029
1755355|pass|200806|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806210532
1755663|pass|200806|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806210653
1757045|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.0|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806211146
1757339|na|200806|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.02|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806211228
1757376|pass|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806211233
1757418|pass|200806|dagolden@cpan.org (DAGOLDEN)|Class-CanBeA|1.2|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806211240
1758182|pass|200806|chris@bingosnet.co.uk|Data-Compare|1.19|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34075|openbsd|4.3|200806211908
1758185|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34075|openbsd|4.3|200806211908
1758189|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34075|openbsd|4.3|200806211910
1759401|na|200806|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.02|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806220030
1759456|pass|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806220039
1759671|pass|200806|rjbs@cpan.org|Data-Compare|1.19|darwin-2level|5.10.0|darwin|9.1.0|200806220149
1759781|pass|200806|chris@bingosnet.co.uk|Data-Compare|1.19|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200806220346
1759782|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200806220347
1759787|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200806220348
1760117|pass|200806|chris@bingosnet.co.uk|Data-Compare|1.19|i86pc-solaris-thread-multi-64int|5.11.0 patch 34076|solaris|2.10|200806220459
1760119|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i86pc-solaris-thread-multi-64int|5.11.0 patch 34076|solaris|2.10|200806220459
1760130|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i86pc-solaris-thread-multi-64int|5.11.0 patch 34076|solaris|2.10|200806220500
1761272|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-SerialCorrelation|1.1|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806220437
1761507|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Longest|1.1|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806220540
1761508|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-ChiSquare|0.5|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806220540
1762012|pass|200806|dagolden@cpan.org (DAGOLDEN)|NestedMap|1.0|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806220748
1762046|pass|200806|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34075|openbsd|4.3|200806221116
1762754|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Licence|1.0|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806221051
1763690|pass|200806|andy@hexten.net|Data-Hexdumper|1.4|i686-linux-thread-multi|5.10.0|linux|2.6.18-5-686|200806221802
1764682|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-SerialCorrelation|1.1|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806221424
1765012|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Pony|1.1.2|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806221441
1765096|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Rank|1.0.1|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806221446
1765124|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Longest|1.1|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806221448
1765127|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-ChiSquare|0.5|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806221448
1765208|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Scalar-Decay|1.1.1|i686-linux-64int-ld|5.6.2|linux|2.6.24-16-generic|200806221453
1765564|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 34074|linux|2.6.22-1-k7|200806222048
1765798|pass|200806|andy@hexten.net|Data-Transactional|1.02|i686-linux-thread-multi|5.10.0|linux|2.6.22-14-generic|200806222033
1766324|pass|200806|dagolden@cpan.org (DAGOLDEN)|NestedMap|1.0|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806221605
1766480|pass|200806|andy@hexten.net|Data-Transactional|1.02|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200806222113
1767765|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Licence|1.0|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806221755
1767873|pass|200806|stro@cpan.org|Data-Hexdumper|1.4|i686-linux|5.10.0|linux|2.4.27-2-386|200806230109
1768010|pass|200806|stro@cpan.org|Data-Compare|1.19|i686-linux|5.10.0|linux|2.4.27-2-386|200806230128
1768606|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Pony|1.1.2|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806221943
1768644|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Rank|1.0.1|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806221946
1768673|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Scalar-Decay|1.1.1|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200806221950
1769631|pass|200806|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.11.0 patch 34076|solaris|2.10|200806230341
1770114|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34074|linux|2.6.22-1-k7|200806230642
1770299|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34074|linux|2.6.22-1-k7|200806230724
1770586|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806230204
1770609|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.19|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806230207
1770610|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.02|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806230207
1770857|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806230233
1770858|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Z80|1.0|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806230233
1770886|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34074|linux|2.6.22-1-k7|200806230728
1770959|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806230246
1771753|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806230450
1772060|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806230550
1772065|pass|200806|stro@cpan.org|Games-Dice-Advanced|1.1|i686-linux|5.8.8|linux|2.4.27-2-386|200806231141
1772583|pass|200806|srezic@cpan.org|Devel-CheckLib|0.5|i486-linux-gnu|5.8.8|linux|2.6.22.10|200806231259
1774401|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806231217
1775143|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806231356
1775988|pass|200806|chris@bingosnet.co.uk|Data-Compare|1.19|i86pc-solaris-thread-multi-64int|5.6.2|solaris|2.10|200806232121
1775991|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i86pc-solaris-thread-multi-64int|5.6.2|solaris|2.10|200806232121
1776015|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i86pc-solaris-thread-multi-64int|5.6.2|solaris|2.10|200806232123
1776860|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.0|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806231816
1777468|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200806240003
1777470|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200806240005
1777653|pass|200806|chris@bingosnet.co.uk|Data-Compare|1.19|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200806240214
1777657|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200806240214
1777673|pass|200806|chris@bingosnet.co.uk|Data-Compare|1.19|i686-linux|5.11.0 patch 34080|linux|2.6.16.38|200806240116
1777675|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.01|i686-linux|5.11.0 patch 34080|linux|2.6.16.38|200806240116
1777677|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200806240217
1777685|pass|200806|chris@bingosnet.co.uk|Data-Transactional|1.02|i686-linux|5.11.0 patch 34080|linux|2.6.16.38|200806240118
1777743|fail|200806|dagolden@cpan.org (DAGOLDEN)|Bryar|3.0|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806240127
1778008|pass|200806|srezic@cpan.org|Games-Dice-Advanced|1.1|i386-freebsd|5.6.2|freebsd|6.1-release|200806240417
1778545|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|1.3|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806240331
1779184|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806240047
1779573|pass|200806|dagolden@cpan.org (DAGOLDEN)|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806240137
1780615|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806240409
1780753|pass|200806|dagolden@cpan.org (DAGOLDEN)|Net-Random|2.0|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806240429
1781189|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806240524
1782392|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806240811
1782638|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806240847
1782872|pass|200806|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806240926
1783244|pass|200806|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806241025
1783498|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806241059
1785150|fail|200806|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.02|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806241944
1785597|pass|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806250244
1786349|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806250134
1786732|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.19|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806250710
1786739|pass|200806|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Net-Random|2.0|sun4-solaris|5.10.0|solaris|2.9|200806250908
1786870|pass|200806|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux|5.11.0 patch 34080|linux|2.6.16.38|200806250724
1787188|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34085|linux|2.6.22-1-k7|200806250740
1787558|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 34085|linux|2.6.22-1-k7|200806250647
1787890|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806250520
1788301|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Longest|1.1|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806250554
1788304|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-ChiSquare|0.5|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806250554
1789082|pass|200806|chris@bingosnet.co.uk|Data-Compare|1.19|darwin-thread-multi-64int-2level|5.10.0 patch 34065|darwin|8.10.0|200806251123
1789201|pass|200806|dagolden@cpan.org (DAGOLDEN)|NestedMap|1.0|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806250741
1790185|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Licence|1.0|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806250949
1790585|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|DBIx-Class-SingletonRows|0.1|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200806251642
1790586|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|DBIx-Class-SingletonRows|0.1|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200806251642
1790589|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|DBIx-Class-SingletonRows|0.1|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200806251642
1790595|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|DBIx-Class-SingletonRows|0.1|i686-linux-64int|5.8.8|linux|2.6.14|200806251642
1790599|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|DBIx-Class-SingletonRows|0.1|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200806251643
1790852|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|DBIx-Class-SingletonRows|0.1|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200806251642
1790854|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|DBIx-Class-SingletonRows|0.1|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200806251643
1790858|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|DBIx-Class-SingletonRows|0.1|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200806251643
1791331|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Pony|1.1.2|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806251204
1791378|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Rank|1.0.1|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806251207
1791409|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Scalar-Decay|1.1.1|i686-linux-thread-multi-64int-ld|5.10.0 patch 34034|linux|2.6.24-16-generic|200806251211
1793688|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34085|linux|2.6.22-1-k7|200806252241
1793705|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34085|linux|2.6.22-1-k7|200806252246
1793794|fail|200806|srezic@cpan.org|DBIx-Class-SingletonRows|0.1|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200806252323
1795338|pass|200806|danl@metrex.net|Data-Compare|1.19|i586-linux-thread-multi|5.10.0|linux|2.6.25|200806260023
1795508|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 34088|linux|2.6.22-1-k7|200806260948
1795580|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|1.3|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806260400
1795706|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34088|linux|2.6.22-1-k7|200806261019
1796391|pass|200806|stro@cpan.org|Data-Compare|1.19|i386-freebsd-64int|5.8.8|freebsd|7.0-release|200806260455
1796581|pass|200806|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd|5.11.0 patch 34085|openbsd|4.1|200806260621
1796835|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806260800
1797399|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34088|linux|2.6.22-1-k7|200806261514
1797468|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34088|linux|2.6.22-1-k7|200806261456
1797741|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34088|linux|2.6.22-1-k7|200806261530
1798310|fail|200806|david@cantrell.org.uk|DBIx-Class-SingletonRows|0.1|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200806261650
1798878|fail|200806|david@cantrell.org.uk|DBIx-Class-SingletonRows|0.1|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200806261706
1799261|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|DBIx-Class-SingletonRows|0.11|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200806261934
1799264|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|DBIx-Class-SingletonRows|0.11|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200806261934
1799266|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|DBIx-Class-SingletonRows|0.11|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200806261934
1799293|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34088|linux|2.6.22-1-k7|200806261937
1799302|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|DBIx-Class-SingletonRows|0.11|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200806261938
1799396|pass|200806|david@cantrell.org.uk|DBIx-Class-SingletonRows|0.1|i686-linux|5.8.8|linux|2.4.27-3-686|200806261703
1799427|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|DBIx-Class-SingletonRows|0.11|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200806261935
1799462|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|DBIx-Class-SingletonRows|0.11|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200806261936
1799463|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|DBIx-Class-SingletonRows|0.11|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200806261937
1799469|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|DBIx-Class-SingletonRows|0.11|i686-linux-64int|5.8.8|linux|2.6.14|200806261936
1800691|pass|200806|dagolden@cpan.org (DAGOLDEN)|DBIx-Class-SingletonRows|0.11|i686-linux|5.10.0|linux|2.6.24-16-generic|200806261801
1801186|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux|5.10.0|linux|2.6.24-16-generic|200806261850
1801665|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806270028
1802752|pass|200806|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|DBIx-Class-SingletonRows|0.1|sun4-solaris|5.10.0|solaris|2.9|200806270638
1802983|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.19|i686-linux|5.10.0|linux|2.6.24-16-generic|200806270136
1802984|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.02|i686-linux|5.10.0|linux|2.6.24-16-generic|200806270136
1803070|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|darwin-2level|5.8.8|darwin|8.10.1|200806270744
1803075|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|darwin-2level|5.8.8|darwin|8.10.1|200806270744
1803651|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|i686-linux|5.10.0|linux|2.6.24-16-generic|200806270316
1803652|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Z80|1.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200806270316
1803824|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux|5.10.0|linux|2.6.24-16-generic|200806270355
1804043|pass|200806|srezic@cpan.org|DBIx-Class-SingletonRows|0.11|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200806270937
1804246|pass|200806|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806270522
1804400|pass|200806|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|DBIx-Class-SingletonRows|0.11|sun4-solaris|5.10.0|solaris|2.9|200806271202
1804487|fail|200806|david@cantrell.org.uk|DBIx-Class-SingletonRows|0.1|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200806271132
1804540|pass|200806|david@cantrell.org.uk|DBIx-Class-SingletonRows|0.1|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200806270546
1804665|pass|200806|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Number-Phone-UK-DetailedLocations|1.3|sun4-solaris|5.10.0|solaris|2.9|200806271315
1804813|pass|200806|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806270753
1805948|pass|200806|david@cantrell.org.uk|DBIx-Class-SingletonRows|0.1|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200806271358
1806046|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux|5.10.0|linux|2.6.24-16-generic|200806271150
1806122|pass|200806|david@cantrell.org.uk|DBIx-Class-SingletonRows|0.11|i686-linux|5.8.8|linux|2.4.27-3-686|200806271432
1806936|pass|200806|david@cantrell.org.uk|DBIx-Class-SingletonRows|0.1|IP35-irix-64int|5.8.8|irix|6.5|200806272105
1806945|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux|5.10.0|linux|2.6.24-16-generic|200806271506
1807304|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.0|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806271548
1807966|pass|200806|dagolden@cpan.org (DAGOLDEN)|Class-CanBeA|1.2|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806271729
1808105|pass|200806|stro@cpan.org|Devel-CheckOS|1.43|armv5tejl-linux|5.10.0|linux|2.6.17.14|200806280102
1809253|fail|200806|icerider70@gmail.com ("ICERIDER")|DBIx-Class-SingletonRows|0.11|i386-linux|5.8.8|linux|2.6.18-53.el5|200806280206
1810869|fail|200806|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.02|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806280426
1810950|pass|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806280434
1813714|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-SerialCorrelation|1.1|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806281155
1813849|pass|200806|david@cantrell.org.uk|DBIx-Class-SingletonRows|0.11|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200806281343
1814361|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Longest|1.1|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806281300
1814366|pass|200806|dagolden@cpan.org (DAGOLDEN)|Statistics-ChiSquare|0.5|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806281300
1814639|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34090|linux|2.6.22-1-k7|200806281931
1815011|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34090|linux|2.6.22-1-k7|200806282015
1815274|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34090|linux|2.6.22-1-k7|200806282048
1815862|pass|200806|dagolden@cpan.org (DAGOLDEN)|NestedMap|1.0|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806281540
1815886|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 34090|linux|2.6.22-1-k7|200806282010
1815954|pass|200806|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux|5.10.0|linux|2.6.24-16-generic|200806281547
1815995|pass|200806|david@cantrell.org.uk|DBIx-Class-SingletonRows|0.11|IP35-irix-64int|5.8.8|irix|6.5|200806282149
1817322|pass|200806|stro@cpan.org|Palm-SMS|0.02|i686-linux|5.8.8|linux|2.4.27-2-386|200806290122
1817742|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Licence|1.0|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806281911
1817860|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34090|linux|2.6.22-1-k7|200806290032
1818177|pass|200806|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34090|linux|2.6.22-1-k7|200806290200
1818908|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200806290106
1820182|unknown|200806|dagolden@cpan.org (DAGOLDEN)|Acme-Pony|1.1.2|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806290329
1820329|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Rank|1.0.1|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806290007
1820383|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-Scalar-Decay|1.1.1|i686-linux-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806290014
1820882|pass|200806|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Games-Dice-Advanced|1.1|sun4-solaris|5.10.0|solaris|2.9|200806290715
1821606|pass|200806|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-ld|5.11.0 patch 34090|linux|2.6.21.5-smp|200806290644
1822116|pass|200806|andy@hexten.net|Data-Compare|1.19|darwin-thread-multi-2level|5.10.0|darwin|9.2.2|200806290951
1822309|pass|200806|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200806290530
1822949|fail|200806|dagolden@cpan.org (DAGOLDEN)|Bryar|3.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200806290755
1823789|pass|200806|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|1.3|i686-linux|5.10.0|linux|2.6.24-16-generic|200806291145
1824422|pass|200806|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Sub-WrapPackages|1.2|sun4-solaris|5.10.0|solaris|2.9|200806291954
1824425|pass|200806|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux|5.10.0|linux|2.6.24-16-generic|200806291355
1824977|pass|200806|dagolden@cpan.org (DAGOLDEN)|Palm-TreoPhoneCallDB|1.1|i686-linux|5.10.0|linux|2.6.24-16-generic|200806291536
1826691|pass|200806|andy@hexten.net|Devel-CheckLib|0.5|darwin-thread-multi-2level|5.10.0|darwin|9.2.2|200806300051
1826968|pass|200806|dagolden@cpan.org (DAGOLDEN)|Net-Random|2.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200806300042
1827379|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux|5.10.0|linux|2.6.24-16-generic|200806300225
1828898|pass|200806|stro@cpan.org|XML-Tiny|1.11|armv5tejl-linux|5.10.0|linux|2.6.17.14|200806301547
1829462|pass|200806|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806301028
1830918|pass|200806|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux-thread-multi-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806301443
1830919|pass|200806|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux|5.10.0|linux|2.6.24-16-generic|200806301443
1831419|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806301631
1831425|pass|200806|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux|5.10.0|linux|2.6.24-16-generic|200806301631
1831539|pass|200806|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.19|i686-linux-thread-multi-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200806301648
1832513|pass|200806|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-64int|5.11.0 patch 34093|openbsd|4.2|200806302346
1833685|pass|200807|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200807010141
1833952|pass|200807|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200807010218
1834202|pass|200807|stro@cpan.org|Class-CanBeA|1.2|i386-freebsd-64int|5.8.8|freebsd|7.0-release|200807011305
1834458|pass|200807|dagolden@cpan.org (DAGOLDEN)|Class-CanBeA|1.2|i686-linux|5.10.0|linux|2.6.24-16-generic|200807010357
1836068|pass|200807|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux-thread-multi-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200807010526
1837205|fail|200807|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.02|i686-linux|5.10.0|linux|2.6.24-16-generic|200807010935
1837251|pass|200807|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux|5.10.0|linux|2.6.24-16-generic|200807010950
1839694|pass|200807|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200807011613
1839907|pass|200807|dagolden@cpan.org (DAGOLDEN)|Statistics-SerialCorrelation|1.1|i686-linux|5.10.0|linux|2.6.24-16-generic|200807011646
1840333|pass|200807|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Longest|1.1|i686-linux|5.10.0|linux|2.6.24-16-generic|200807011737
1840334|pass|200807|dagolden@cpan.org (DAGOLDEN)|Statistics-ChiSquare|0.5|i686-linux|5.10.0|linux|2.6.24-16-generic|200807011737
1840914|pass|200807|stro@cpan.org|Devel-CheckLib|0.5|os2|5.10.0|os2|2.45|200807020331
1842188|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd|5.11.0 patch 34094|netbsd|4.0|200807021131
1842724|pass|200807|stro@cpan.org|XML-DoubleEncodedEntities|1.0|armv5tejl-linux|5.10.0|linux|2.6.17.14|200807021706
1842760|pass|200807|stro@cpan.org|XML-DoubleEncodedEntities|1.0|armv5tejl-linux|5.8.8|linux|2.6.17.14|200807021715
1842991|pass|200807|stro@cpan.org|Devel-CheckOS|1.43|os2|5.10.0|os2|2.45|200807021803
1843586|pass|200807|andy@hexten.net|Data-Compare|1.19|darwin-thread-multi-2level|5.10.0|darwin|9.2.2|200807022017
1844679|pass|200807|stro@cpan.org|CPAN-FindDependencies|2.0|i386-freebsd-64int|5.8.8|freebsd|7.0-release|200807030836
1845117|pass|200807|stro@cpan.org|CPU-Emulator-Memory|1.1001|i386-freebsd-64int|5.8.8|freebsd|7.0-release|200807031253
1845123|pass|200807|stro@cpan.org|CPU-Emulator-Z80|0.9|i386-freebsd-64int|5.8.8|freebsd|7.0-release|200807031258
1845746|pass|200807|dagolden@cpan.org (DAGOLDEN)|NestedMap|1.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200807030249
1847050|pass|200807|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Rank|1.0.1|i686-linux|5.10.0|linux|2.6.24-16-generic|200807030911
1847135|pass|200807|dagolden@cpan.org (DAGOLDEN)|Tie-Scalar-Decay|1.1.1|i686-linux|5.10.0|linux|2.6.24-16-generic|200807030919
1847182|unknown|200807|dagolden@cpan.org (DAGOLDEN)|Acme-Licence|1.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200807031854
1847204|pass|200807|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|File-Overwrite|1.0|sun4-solaris|5.10.0|solaris|2.9|200807032050
1849403|unknown|200807|dagolden@cpan.org (DAGOLDEN)|Acme-Pony|1.1.2|i686-linux|5.10.0|linux|2.6.24-16-generic|200807032210
1849983|pass|200807|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux-thread-multi-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200807031927
1850429|pass|200807|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux-thread-multi-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200807040048
1851555|pass|200807|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Class-CanBeA|1.2|sun4-solaris|5.10.0|solaris|2.9|200807040710
1852008|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|darwin-2level|5.10.0|darwin|8.11.1|200807040854
1852380|pass|200807|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux-thread-multi-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200807040421
1852837|pass|200807|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int-ld|5.8.8 patch 34066|linux|2.6.24-16-generic|200807040634
1857100|pass|200807|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200807050239
1857594|pass|200807|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.19|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200807050403
1857595|pass|200807|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.02|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200807050403
1857796|pass|200807|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200807050437
1858026|pass|200807|srezic@cpan.org|Sub-WrapPackages|1.2|i386-freebsd|5.6.2|freebsd|6.1-release|200807051120
1858459|pass|200807|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200807050641
1858671|pass|200807|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200807050727
1860147|pass|200807|stro@cpan.org|Data-Transactional|1.0|i386-freebsd-64int|5.8.8|freebsd|7.0-release|200807051639
1864049|pass|200807|sanko@cpan.org|Devel-CheckOS|1.43|MSWin32-x86-multi-thread|5.11.0 patch 34101|MSWin32|5.1|200807051408
1865845|pass|200807|postmaster@TOMXP.spectra-direct.com|Data-Compare|1.19|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200807071710
1870236|pass|200807|mgardner@usatech.com|Data-Compare|1.19|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200807081438
1870237|pass|200807|sanko@cpan.org|Devel-CheckOS|1.43|MSWin32-x86-multi-thread|5.11.0 patch 34107|MSWin32|5.1|200807080816
1870763|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 34112|linux|2.6.22-1-k7|200807090026
1871046|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34112|linux|2.6.22-1-k7|200807090005
1871077|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34112|linux|2.6.22-1-k7|200807090000
1871298|pass|200807|bob@randomness.org.uk ("bob walker")|Data-Compare|1.19|OpenBSD.i386-openbsd|5.8.8|openbsd|4.2|200807090112
1871307|pass|200807|bob@randomness.org.uk ("bob walker")|Data-Compare|1.19|OpenBSD.i386-openbsd|5.10.0|openbsd|4.2|200807090114
1871711|fail|200807|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Palm-SMS|0.02|sun4-solaris|5.10.0|solaris|2.9|200807090504
1871761|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34112|linux|2.6.22-1-k7|200807090518
1871953|pass|200807|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Acme-Scurvy-Whoreson-BilgeRat|1.1|sun4-solaris|5.10.0|solaris|2.9|200807090630
1874988|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34112|linux|2.6.22-1-k7|200807100101
1875310|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34112|linux|2.6.22-1-k7|200807100240
1877308|pass|200807|chris@bingosnet.co.uk|Data-Compare|1.19|OpenBSD.i386-openbsd-64int|5.11.0 patch 34114|openbsd|4.1|200807101527
1879484|pass|200807|srezic@cpan.org|Palm-Treo680MessagesDB|1.0|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200807110816
1879866|pass|200807|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Palm-Treo680MessagesDB|1.0|sun4-solaris|5.10.0|solaris|2.9|200807111104
1880013|pass|200807|imacat@mail.imacat.idv.tw|Palm-Treo680MessagesDB|1.0|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200807111811
1880015|pass|200807|srezic@cpan.org|Palm-Treo680MessagesDB|1.0|i386-freebsd|5.10.0|freebsd|6.1-release|200807111209
1880023|pass|200807|imacat@mail.imacat.idv.tw|Palm-Treo680MessagesDB|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200807111815
1880186|pass|200807|srezic@cpan.org|Palm-Treo680MessagesDB|1.0|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200807111238
1880257|pass|200807|srezic@cpan.org|Palm-Treo680MessagesDB|1.0|i386-freebsd|5.6.2|freebsd|6.1-release|200807111333
1880431|pass|200807|srezic@cpan.org|Palm-Treo680MessagesDB|1.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200807111434
1880469|pass|200807|chris@bingosnet.co.uk|Data-Compare|1.19|OpenBSD.i386-openbsd-thread-multi|5.11.0 patch 34117|openbsd|4.3|200807111214
1880550|pass|200807|srezic@cpan.org|Palm-Treo680MessagesDB|1.0|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200807111538
1880567|pass|200807|david@cantrell.org.uk|Palm-Treo680MessagesDB|1.0|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200807110851
1880653|pass|200807|david@cantrell.org.uk|Data-Hexdumper|1.4|IP35-irix-64int|5.8.8|irix|6.5|200807111613
1880661|pass|200807|david@cantrell.org.uk|Palm-Treo680MessagesDB|1.0|IP35-irix-64int|5.8.8|irix|6.5|200807111614
1880688|pass|200807|david@cantrell.org.uk|Palm-Treo680MessagesDB|1.0|i686-linux|5.8.8|linux|2.4.27-3-686|200807110844
1880690|pass|200807|david@cantrell.org.uk|Palm-Treo680MessagesDB|1.0|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200807110844
1880696|pass|200807|david@cantrell.org.uk|Palm-Treo680MessagesDB|1.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200807111521
1880888|pass|200807|david@cantrell.org.uk|Palm-Treo680MessagesDB|1.0|i686-linux|5.6.2|linux|2.4.27-3-686|200807110924
1880913|pass|200807|david@cantrell.org.uk|Palm-Treo680MessagesDB|1.0|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200807111604
1881263|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200807111611
1881373|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-freebsd|5.11.0 patch 34120|freebsd|6.2-release|200807111700
1881381|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i386-freebsd|5.11.0 patch 34120|freebsd|6.2-release|200807111706
1881526|pass|200807|david@cantrell.org.uk|Palm-Treo680MessagesDB|1.0|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200807111839
1881589|pass|200807|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Statistics-SerialCorrelation|1.1|sun4-solaris|5.10.0|solaris|2.9|200807112102
1881597|pass|200807|chris@bingosnet.co.uk|Data-Compare|1.19|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200807112010
1881631|pass|200807|chris@bingosnet.co.uk|Data-Compare|1.19|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200807112030
1881979|pass|200807|pjacklam@online.no|Devel-CheckOS|1.43|cygwin-thread-multi-64int|5.10.0|cygwin|1.5.25(0.15642)|200807120016
1882976|pass|200807|chris@bingosnet.co.uk|Data-Compare|1.19|i386-freebsd|5.11.0 patch 34120|freebsd|6.2-release|200807120643
1883227|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200807121146
1883249|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200807121153
1883279|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200807121206
1883513|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200807121315
1883722|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i86pc-solaris|5.11.0 patch 34131|solaris|2.10|200807121410
1883742|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i86pc-solaris|5.11.0 patch 34131|solaris|2.10|200807121416
1884017|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34134|linux|2.6.22-1-k7|200807121625
1884092|pass|200807|sanko@cpan.org|Data-Hexdumper|1.4|MSWin32-x86-multi-thread|5.11.0 patch 34116|MSWin32|5.1|200807120352
1884116|fail|200807|sanko@cpan.org|Palm-Treo680MessagesDB|1.0|MSWin32-x86-multi-thread|5.11.0 patch 34116|MSWin32|5.1|200807120358
1884331|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 34134|linux|2.6.22-1-k7|200807121639
1884411|pass|200807|chris@bingosnet.co.uk|Data-Compare|1.19|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200807121621
1884991|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34134|linux|2.6.22-1-k7|200807121957
1885642|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34134|linux|2.6.22-1-k7|200807122151
1886018|pass|200807|chris@bingosnet.co.uk|Data-Compare|1.19|i86pc-solaris|5.11.0 patch 34131|solaris|2.10|200807122246
1886250|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34134|linux|2.6.22-1-k7|200807130030
1886715|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34134|linux|2.6.22-1-k7|200807130209
1887496|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200807130705
1887713|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i86pc-solaris-thread-multi-64int|5.6.2|solaris|2.10|200807131014
1887720|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i86pc-solaris-thread-multi-64int|5.6.2|solaris|2.10|200807131017
1888851|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i686-linux-thread-multi-ld|5.11.0 patch 34136|linux|2.6.21.5-smp|200807131751
1888875|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i686-linux-thread-multi-ld|5.11.0 patch 34136|linux|2.6.21.5-smp|200807131803
1889291|pass|200807|chris@bingosnet.co.uk|Data-Compare|1.19|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200807132039
1889768|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200807140010
1889801|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200807140028
1889814|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200807140047
1889836|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200807140059
1889837|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200807140101
1889858|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200807140115
1890161|pass|200807|chris@bingosnet.co.uk|Data-Compare|1.19|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200807140411
1891058|pass|200807|chris@bingosnet.co.uk|Data-Compare|1.19|i686-linux-thread-multi-ld|5.11.0 patch 34136|linux|2.6.21.5-smp|200807141140
1891574|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200807141638
1891600|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200807141652
1891730|pass|200807|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Tie-Hash-Longest|1.1|sun4-solaris|5.10.0|solaris|2.9|200807141846
1891733|pass|200807|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Statistics-ChiSquare|0.5|sun4-solaris|5.10.0|solaris|2.9|200807141847
1891774|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-netbsd-64int|5.11.0 patch 34134|netbsd|4.0|200807141758
1891813|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i386-netbsd-64int|5.11.0 patch 34134|netbsd|4.0|200807141811
1893246|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200807150513
1893480|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|OpenBSD.i386-openbsd-thread-multi|5.11.0 patch 34134|openbsd|4.2|200807150707
1893723|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200807150928
1893919|pass|200807|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200807151108
1894000|pass|200807|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-64int|5.11.0 patch 34134|netbsd|4.0|200807151145
1894908|pass|200807|srezic@cpan.org|Sub-WrapPackages|1.2|i386-freebsd|5.10.0|freebsd|6.1-release|200807151806
1897024|pass|200807|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|NestedMap|1.0|sun4-solaris|5.10.0|solaris|2.9|200807160638
1897245|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200807160506
1897251|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-freebsd-64int|5.11.0 patch 34144|freebsd|7.0-release|200807160507
1897276|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200807160516
1897283|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i386-freebsd-64int|5.11.0 patch 34144|freebsd|7.0-release|200807160517
1897439|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34144|linux|2.6.22-1-k7|200807160837
1897771|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200807160715
1897774|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-64int|5.11.0 patch 34144|freebsd|7.0-release|200807160715
1897831|pass|200807|chris@bingosnet.co.uk|Data-Compare|1.19|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200807160906
1898169|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 34144|linux|2.6.22-1-k7|200807161054
1899447|pass|200807|chris@bingosnet.co.uk|Data-Compare|1.19|OpenBSD.i386-openbsd-thread-multi|5.11.0 patch 34134|openbsd|4.2|200807161430
1899642|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34144|linux|2.6.22-1-k7|200807161619
1900007|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34144|linux|2.6.22-1-k7|200807161741
1900014|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34144|linux|2.6.22-1-k7|200807161731
1901024|pass|200807|chris@bingosnet.co.uk|Data-Compare|1.19|i386-freebsd-64int|5.11.0 patch 34144|freebsd|7.0-release|200807161945
1901200|pass|200807|chris@bingosnet.co.uk|Data-Compare|1.19|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200807162036
1901212|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200807162227
1901431|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200807162323
1901474|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200807162328
1901477|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34144|linux|2.6.22-1-k7|200807170035
1901489|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200807162333
1901499|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200807162334
1901510|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200807162339
1901513|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200807162341
1901520|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200807162342
1901535|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200807162347
1901566|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200807162352
1901576|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200807162358
1902813|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i686-linux-thread-multi-64int|5.10.0|linux|2.4.26|200807170739
1902859|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i686-linux-thread-multi-64int|5.10.0|linux|2.4.26|200807170747
1903090|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int|5.10.0|linux|2.4.26|200807170853
1904723|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i686-linux-thread-multi-ld|5.11.0 patch 34150|linux|2.6.24.5-smp|200807171449
1904735|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i686-linux-thread-multi-ld|5.11.0 patch 34150|linux|2.6.24.5-smp|200807171453
1904820|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-netbsd-64int|5.11.0 patch 34150|netbsd|4.0|200807171506
1904869|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|i386-netbsd-64int|5.11.0 patch 34150|netbsd|4.0|200807171523
1904936|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-ld|5.11.0 patch 34150|linux|2.6.24.5-smp|200807171529
1905990|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200807180620
1907801|unknown|200807|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Acme-Licence|1.0|sun4-solaris|5.10.0|solaris|2.9|200807181658
1909274|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200807182328
1909836|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-64int|5.11.0 patch 34151|freebsd|7.0-release|200807190141
1909974|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200807190317
1910043|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34151|openbsd|4.3|200807190545
1911641|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34151|linux|2.6.22-1-k7|200807191421
1912343|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 34151|linux|2.6.22-1-k7|200807191741
1914094|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34151|linux|2.6.22-1-k7|200807192239
1915717|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34151|linux|2.6.22-1-k7|200807200325
1916521|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34151|linux|2.6.22-1-k7|200807200940
1916732|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34151|linux|2.6.22-1-k7|200807201023
1916950|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.11.0 patch 34151|solaris|2.10|200807201014
1917372|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200807201134
1917468|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200807201154
1918191|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200807201617
1918224|unknown|200807|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Acme-Pony|1.1.2|sun4-solaris|5.10.0|solaris|2.9|200807201729
1918463|pass|200807|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Tie-Hash-Rank|1.0.1|sun4-solaris|5.10.0|solaris|2.9|200807201834
1918665|pass|200807|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Tie-Scalar-Decay|1.1.1|sun4-solaris|5.10.0|solaris|2.9|200807201949
1918745|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.6.2|solaris|2.10|200807201910
1918958|pass|200807|chris@bingosnet.co.uk|Acme-Licence|1.0|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807202018
1918963|pass|200807|chris@bingosnet.co.uk|Acme-Pony|1.1.2|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807202019
1918968|pass|200807|chris@bingosnet.co.uk|Acme-Scurvy-Whoreson-BilgeRat|1.1|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807202020
1919527|pass|200807|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210026
1919530|pass|200807|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210028
1919544|pass|200807|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210034
1919550|pass|200807|chris@bingosnet.co.uk|Class-CanBeA|1.2|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210037
1919562|pass|200807|chris@bingosnet.co.uk|Data-Compare|1.19|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210043
1919563|pass|200807|chris@bingosnet.co.uk|Data-Hexdumper|1.4|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210044
1919567|pass|200807|chris@bingosnet.co.uk|Data-Transactional|1.02|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210047
1919574|pass|200807|chris@bingosnet.co.uk|Devel-CheckLib|0.5|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210049
1919576|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210051
1919615|pass|200807|chris@bingosnet.co.uk|File-Find-Rule-Permissions|1.3|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210107
1919619|pass|200807|chris@bingosnet.co.uk|File-Overwrite|1.0|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210109
1919621|pass|200807|chris@bingosnet.co.uk|Games-Dice-Advanced|1.1|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210110
1919622|pass|200807|chris@bingosnet.co.uk|NestedMap|1.0|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210111
1919632|pass|200807|chris@bingosnet.co.uk|Net-Random|2.0|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210118
1919642|pass|200807|chris@bingosnet.co.uk|Number-Phone|1.6|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210120
1919662|pass|200807|chris@bingosnet.co.uk|Number-Phone-UK-DetailedLocations|1.3|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210132
1919679|pass|200807|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210141
1919693|pass|200807|chris@bingosnet.co.uk|Palm-TreoPhoneCallDB|1.1|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210154
1919695|pass|200807|chris@bingosnet.co.uk|Statistics-ChiSquare|0.5|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210155
1919696|pass|200807|chris@bingosnet.co.uk|Statistics-SerialCorrelation|1.1|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210155
1919699|pass|200807|chris@bingosnet.co.uk|Sub-WrapPackages|1.2|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210156
1919701|pass|200807|chris@bingosnet.co.uk|Tie-Hash-Longest|1.1|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210157
1919703|pass|200807|chris@bingosnet.co.uk|Tie-Hash-Rank|1.0.1|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210158
1919707|pass|200807|chris@bingosnet.co.uk|Tie-STDOUT|1.03|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210159
1919712|pass|200807|chris@bingosnet.co.uk|Tie-Scalar-Decay|1.1.1|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210200
1919717|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210202
1919719|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|darwin-thread-multi-64int-2level|5.10.0 patch 34095|darwin|8.10.0|200807210201
1920793|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200807210841
1920887|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.11.0 patch 34151|solaris|2.10|200807210931
1921536|pass|200807|andy@hexten.net|Data-Compare|1.19|i86pc-solaris-thread-multi|5.10.0|solaris|2.11|200807211405
1922183|fail|200807|stro@cpan.org|Bryar|3.0|armv5tejl-linux|5.10.0|linux|2.6.17.14|200807211946
1924105|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-ld|5.11.0 patch 34151|linux|2.6.16.38|200807220155
1924198|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200807220244
1924342|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200807220346
1926834|pass|200807|stro@cpan.org|Data-Hexdumper|1.4|armv5tejl-linux|5.10.0|linux|2.6.17.14|200807221251
1926928|pass|200807|stro@cpan.org|Data-Compare|1.19|armv5tejl-linux|5.10.0|linux|2.6.17.14|200807221609
1927030|pass|200807|stro@cpan.org|Data-Hexdumper|1.4|armv5tejl-linux|5.8.8|linux|2.6.17.14|200807221523
1927087|pass|200807|stro@cpan.org|Data-Compare|1.19|armv5tejl-linux|5.8.8|linux|2.6.17.14|200807221639
1927327|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34153|linux|2.6.22-1-k7|200807222103
1927555|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34153|linux|2.6.22-1-k7|200807222147
1929231|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200807230207
1929554|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34153|linux|2.6.22-1-k7|200807230416
1929713|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 34153|linux|2.6.22-1-k7|200807230450
1930126|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.11.0 patch 34153|netbsd|4.0|200807230525
1930619|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200807230704
1930745|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-64int|5.11.0 patch 34153|openbsd|4.1|200807230724
1930792|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200807230733
1931384|pass|200807|stro@cpan.org|Tie-Hash-Longest|1.1|i686-linux|5.8.8|linux|2.4.27-2-386|200807230824
1932393|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34153|linux|2.6.22-1-k7|200807231418
1932727|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34153|linux|2.6.22-1-k7|200807231536
1938533|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200807242216
1938537|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200807242218
1938542|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200807242218
1938547|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-64int|5.8.8|linux|2.6.14|200807242219
1939381|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-freebsd-thread-multi|5.11.0 patch 34156|freebsd|6.2-release|200807242148
1939657|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200807250026
1939754|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200807250038
1939827|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200807250047
1939856|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200807250056
1939913|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200807250107
1939928|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200807250109
1939959|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200807250115
1939961|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200807250116
1940011|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200807250126
1940530|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200807250139
1940570|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200807250150
1940772|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200807250514
1940799|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-freebsd-64int|5.11.0 patch 34156|freebsd|7.0-release|200807250125
1941869|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200807251244
1941928|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200807251315
1942156|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.11.0 patch 34156|linux|2.6.21.5-smp|200807251421
1943494|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sort-MultipleFields|0.001_01|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200807252350
1943498|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sort-MultipleFields|0.001_01|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200807260004
1943503|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sort-MultipleFields|0.001_01|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200807260004
1943506|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sort-MultipleFields|0.001_01|i686-linux-64int|5.8.8|linux|2.6.14|200807260005
1943508|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sort-MultipleFields|0.001_01|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200807260005
1943509|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sort-MultipleFields|0.001_01|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200807260006
1943511|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sort-MultipleFields|0.001_01|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200807260006
1943532|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sort-MultipleFields|0.001_01|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200807260004
1943533|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sort-MultipleFields|0.001_01|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200807260003
1943639|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34156|linux|2.6.22-1-k7|200807260040
1944678|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 34156|linux|2.6.22-1-k7|200807260447
1944791|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34156|linux|2.6.22-1-k7|200807260424
1945089|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.11.0 patch 34156|linux|2.6.21.5-smp|200807260451
1945112|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200807260601
1945113|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200807260602
1945116|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200807260603
1945635|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.11.0 patch 34156|linux|2.6.21.5-smp|200807260730
1945982|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200807260851
1946153|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200807260936
1946430|pass|200807|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|IP35-irix-64int|5.8.8|irix|6.5|200807261226
1946495|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200807261047
1946586|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200807261115
1946745|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34156|linux|2.6.22-1-k7|200807261603
1946879|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34156|linux|2.6.22-1-k7|200807261633
1947584|pass|200807|david@cantrell.org.uk|Sort-MultipleFields|0.001_01|i686-linux|5.6.2|linux|2.4.27-3-686|200807261910
1947597|pass|200807|david@cantrell.org.uk|Sort-MultipleFields|0.001_01|i686-linux|5.8.8|linux|2.4.27-3-686|200807261915
1947656|pass|200807|david@cantrell.org.uk|Sort-MultipleFields|0.001_01|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200807261930
1947672|pass|200807|david@cantrell.org.uk|Sort-MultipleFields|0.001_01|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200807261935
1947857|pass|200807|david@cantrell.org.uk|Sort-MultipleFields|0.001_01|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200807261455
1947861|pass|200807|david@cantrell.org.uk|Sort-MultipleFields|0.001_01|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200807261956
1948026|pass|200807|david@cantrell.org.uk|Sort-MultipleFields|0.001_01|IP35-irix-64int|5.8.8|irix|6.5|200807262244
1948071|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34156|linux|2.6.22-1-k7|200807262300
1948183|pass|200807|david@cantrell.org.uk|Sort-MultipleFields|0.001_01|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200807262320
1949069|pass|200807|stro@cpan.org|Net-Random|2.0|i686-linux|5.10.0|linux|2.4.27-2-386|200807270954
1950303|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-dragonfly-thread-multi-64int|5.11.0 patch 34156|dragonfly|2.0.0-release|200807271731
1950420|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-dragonfly-thread-multi-64int|5.10.0|dragonfly|2.0.0-release|200807271749
1950511|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200807271712
1950562|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-dragonfly-thread-multi-64int|5.11.0 patch 34156|dragonfly|2.0.0-release|200807271826
1950606|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|OpenBSD.i386-openbsd-thread-multi|5.11.0 patch 34160|openbsd|4.2|200807271738
1950656|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-dragonfly-thread-multi-64int|5.10.0|dragonfly|2.0.0-release|200807271846
1950912|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200807271901
1950950|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200807271910
1950981|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi|5.11.0 patch 34160|openbsd|4.2|200807271922
1951060|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200807271959
1952306|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-dragonfly-thread-multi-64int|5.8.8|dragonfly|2.0.0-release|200807280602
1952445|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-dragonfly-thread-multi-64int|5.8.8|dragonfly|2.0.0-release|200807280711
1952552|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200807280659
1952593|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200807280707
1952613|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200807280712
1952631|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200807280716
1952647|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200807280721
1952670|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200807280726
1952689|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200807280730
1953802|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|OpenBSD.i386-openbsd-thread-multi|5.11.0 patch 34160|openbsd|4.2|200807281343
1953922|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200807281437
1953930|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200807281439
1954513|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200807281812
1954596|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200807281904
1954667|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-dragonfly-thread-multi-64int|5.11.0 patch 34156|dragonfly|2.0.0-release|200807282035
1955416|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200807282252
1955426|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200807282255
1955429|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sort-MultipleFields|1.0|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200807282358
1955430|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sort-MultipleFields|1.0|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200807282359
1955431|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sort-MultipleFields|1.0|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200807282359
1955432|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sort-MultipleFields|1.0|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200807282359
1955435|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sort-MultipleFields|1.0|i686-linux-64int|5.8.8|linux|2.6.14|200807282359
1955436|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200807282257
1955437|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sort-MultipleFields|1.0|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200807290000
1955439|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sort-MultipleFields|1.0|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200807290000
1955441|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sort-MultipleFields|1.0|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200807290000
1955445|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200807282259
1955448|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200807282301
1955452|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200807282303
1955457|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200807282305
1955474|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200807282307
1955483|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sort-MultipleFields|1.0|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200807290007
1955880|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-dragonfly-thread-multi-64int|5.11.0 patch 34156|dragonfly|2.0.0-release|200807290246
1955968|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-dragonfly-thread-multi-64int|5.10.0|dragonfly|2.0.0-release|200807290338
1955990|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200807290252
1955999|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200807290256
1956268|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200807290547
1956270|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200807290547
1956271|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200807290548
1956531|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-dragonfly-thread-multi-64int|5.8.8|dragonfly|2.0.0-release|200807290813
1956557|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 34160|linux|2.6.22-1-k7|200807290823
1956909|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34160|linux|2.6.22-1-k7|200807290951
1956961|pass|200807|david@cantrell.org.uk|Sort-MultipleFields|1.0|i686-linux|5.8.8|linux|2.4.27-3-686|200807290901
1957223|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34160|linux|2.6.22-1-k7|200807291103
1957283|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-netbsd-thread-multi-64int|5.11.0 patch 34160|netbsd|4.0|200807291013
1957288|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-netbsd-thread-multi-64int|5.11.0 patch 34160|netbsd|4.0|200807291014
1957376|pass|200807|imacat@mail.imacat.idv.tw|Sort-MultipleFields|1.0|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200807291735
1957406|pass|200807|david@cantrell.org.uk|Sort-MultipleFields|1.0|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200807291045
1957407|pass|200807|david@cantrell.org.uk|Sort-MultipleFields|1.0|i686-linux|5.6.2|linux|2.4.27-3-686|200807291045
1957413|pass|200807|david@cantrell.org.uk|Sort-MultipleFields|1.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200807291009
1957470|pass|200807|david@cantrell.org.uk|Sort-MultipleFields|1.0|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200807290505
1957471|pass|200807|david@cantrell.org.uk|Sort-MultipleFields|1.0|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200807291005
1957476|pass|200807|imacat@mail.imacat.idv.tw|Sort-MultipleFields|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200807291808
1957513|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.11.0 patch 34160|netbsd|4.0|200807291115
1957650|pass|200807|david@cantrell.org.uk|Sort-MultipleFields|1.0|IP35-irix-64int|5.8.8|irix|6.5|200807291312
1957671|pass|200807|david@cantrell.org.uk|Sort-MultipleFields|1.0|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200807291216
1958030|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-dragonfly-thread-multi-64int|5.10.0 patch 34161|dragonfly|2.0.0-release|200807291444
1958035|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-dragonfly-thread-multi-64int|5.10.0 patch 34161|dragonfly|2.0.0-release|200807291445
1958203|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-dragonfly-thread-multi-64int|5.10.0 patch 34161|dragonfly|2.0.0-release|200807291541
1958905|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-dragonfly-thread-multi-64int|5.8.8 patch 34096|dragonfly|2.0.0-release|200807291925
1958907|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-dragonfly-thread-multi-64int|5.8.8 patch 34096|dragonfly|2.0.0-release|200807291926
1958966|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-dragonfly-thread-multi-64int|5.8.8 patch 34096|dragonfly|2.0.0-release|200807291953
1959723|pass|200807|andy@hexten.net|Data-Hexdumper|1.4|i686-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200807300117
1959756|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200807300048
1959797|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200807300109
1959916|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200807300151
1959917|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200807300151
1959918|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200807300152
1959919|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200807300152
1959921|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200807300153
1959922|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200807300153
1959923|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200807300153
1959924|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200807300154
1959925|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200807300154
1959926|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200807300154
1960114|pass|200807|jeff@zeroclue.com|Acme-Scurvy-Whoreson-BilgeRat|1.1|darwin-2level|5.10.0|darwin|9.2.0|200807300151
1960233|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34160|linux|2.6.22-1-k7|200807300451
1960358|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.11.0 patch 34160|netbsd|4.0|200807300435
1960665|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200807300600
1960702|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200807300611
1960894|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.11.0 patch 34160|netbsd|4.0|200807300708
1961785|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200807301059
1961787|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200807301059
1962019|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34160|linux|2.6.22-1-k7|200807301409
1962097|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-dragonfly-thread-multi-64int|5.8.8 patch 34096|dragonfly|2.0.0-release|200807301357
1962234|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-freebsd-64int|5.11.0 patch 34162|freebsd|7.0-release|200807301233
1962240|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-freebsd-64int|5.11.0 patch 34162|freebsd|7.0-release|200807301234
1962428|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-64int|5.11.0 patch 34162|freebsd|7.0-release|200807301314
1962664|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-dragonfly-thread-multi-64int|5.8.8 patch 34096|dragonfly|2.0.0-release|200807301544
1962973|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-dragonfly-thread-multi-64int|5.10.0 patch 34161|dragonfly|2.0.0-release|200807301640
1963162|pass|200807|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34160|linux|2.6.22-1-k7|200807301740
1963239|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i386-freebsd-64int|5.10.0 patch 34161|freebsd|7.0-release|200807301538
1963241|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-freebsd-64int|5.10.0 patch 34161|freebsd|7.0-release|200807301539
1963368|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-dragonfly-thread-multi-64int|5.10.0 patch 34161|dragonfly|2.0.0-release|200807301800
1963405|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-64int|5.10.0 patch 34161|freebsd|7.0-release|200807301617
1964343|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200807302232
1964347|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200807302232
1964707|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34164|openbsd|4.3|200807310048
1964708|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34164|openbsd|4.3|200807310049
1964821|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34164|openbsd|4.3|200807310133
1965204|pass|200807|stro@cpan.org|Sort-MultipleFields|1.0|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200807310506
1965814|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-freebsd-64int|5.11.0 patch 34162|freebsd|7.0-release|200807310624
1966158|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200807310826
1966586|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200807311051
1966756|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-freebsd-64int|5.10.0 patch 34161|freebsd|7.0-release|200807311155
1966845|pass|200807|stro@cpan.org|Number-Phone|1.6|i686-linux|5.10.0|linux|2.4.27-2-386|200807311319
1966860|pass|200807|stro@cpan.org|Number-Phone-UK-DetailedLocations|1.3|i686-linux|5.10.0|linux|2.4.27-2-386|200807311323
1967293|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34164|openbsd|4.3|200807311603
1967574|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34164|openbsd|4.3|200807311835
1967656|pass|200807|chris@bingosnet.co.uk|XML-Tiny|1.11|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200807311918
1967811|pass|200807|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.11.0 patch 34166|solaris|2.10|200807312059
1967829|pass|200807|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200807312046
1968131|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i86pc-solaris-thread-multi-64int|5.11.0 patch 34166|solaris|2.10|200807312348
1968132|pass|200807|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i86pc-solaris-thread-multi-64int|5.11.0 patch 34166|solaris|2.10|200807312348
1968342|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200808010101
1968348|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200808010102
1968695|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.10.0 patch 34161|linux|2.6.22-1-k7|200808010330
1968845|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200808010255
1968847|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200808010255
1968849|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200808010256
1968851|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200808010256
1968853|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200808010256
1968854|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200808010257
1968856|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200808010257
1968858|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200808010257
1968859|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200808010258
1968861|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200808010258
1969122|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.10.0 patch 34161|linux|2.6.22-1-k7|200808010531
1969492|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 34161|linux|2.6.22-1-k7|200808010701
1969982|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 34161|linux|2.6.22-1-k7|200808010854
1971018|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.11.0 patch 34169|solaris|2.10|200808011258
1971041|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i86pc-solaris-thread-multi-64int|5.6.2|solaris|2.10|200808011303
1971045|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i86pc-solaris-thread-multi-64int|5.6.2|solaris|2.10|200808011305
1971441|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200808011451
1971445|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200808011452
1971643|pass|200808|chris@bingosnet.co.uk|XML-Tiny|1.11|i86pc-solaris-thread-multi-64int|5.11.0 patch 34166|solaris|2.10|200808011530
1972120|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|0.001_01|i86pc-solaris-thread-multi-64int|5.11.0 patch 34169|solaris|2.10|200808011756
1972121|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i86pc-solaris-thread-multi-64int|5.11.0 patch 34169|solaris|2.10|200808011757
1972217|pass|200808|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i86pc-solaris-thread-multi-64int|5.11.0 patch 34166|solaris|2.10|200808011840
1972481|pass|200808|chris@bingosnet.co.uk|XML-Tiny|1.11|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200808011953
1972895|pass|200808|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200808012154
1973809|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 34161|linux|2.6.22-1-k7|200808020555
1974358|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0 patch 34161|linux|2.6.22-1-k7|200808020821
1975026|pass|200808|chris@bingosnet.co.uk|XML-Tiny|1.11|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200808021053
1975119|pass|200808|srezic@cpan.org|Sort-MultipleFields|1.0|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200808021219
1975137|pass|200808|chris@bingosnet.co.uk|XML-Tiny|1.11|i86pc-solaris-thread-multi-64int|5.11.0 patch 34169|solaris|2.10|200808021124
1975297|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-ld|5.11.0 patch 34169|linux|2.6.16.38|200808021204
1976192|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200808021702
1976242|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i686-linux-thread-multi-ld|5.11.0 patch 34169|linux|2.6.16.38|200808021716
1976243|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200808021717
1976693|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200808022003
1976706|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-64int|5.11.0 patch 34169|netbsd|4.0|200808022006
1976985|pass|200808|conrad@fringehead.org|XML-Tiny|1.11|darwin-2level|5.10.0|darwin|9.3.0|200808021334
1976994|pass|200808|srezic@cpan.org|Sort-MultipleFields|1.0|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200808022237
1977069|pass|200808|stro@cpan.org|Palm-SMS|0.02|i686-linux|5.10.0|linux|2.4.27-2-386|200808022054
1977074|pass|200808|stro@cpan.org|Palm-TreoPhoneCallDB|1.1|i686-linux|5.10.0|linux|2.4.27-2-386|200808022055
1977127|pass|200808|srezic@cpan.org|Sort-MultipleFields|1.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200808022314
1977128|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-netbsd-64int|5.11.0 patch 34169|netbsd|4.0|200808022212
1977200|pass|200808|conrad@fringehead.org|XML-Tiny|1.11|i686-linux-thread-multi|5.8.8|linux|2.6.15.1|200808021437
1977216|pass|200808|srezic@cpan.org|Sort-MultipleFields|1.0|i386-freebsd|5.10.0|freebsd|6.1-release|200808022336
1978598|pass|200808|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-64int|5.11.0 patch 34169|netbsd|4.0|200808030831
1978664|pass|200808|chris@bingosnet.co.uk|XML-Tiny|1.11|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200808030849
1978893|pass|200808|chris@bingosnet.co.uk|XML-Tiny|1.11|i686-linux-thread-multi-ld|5.11.0 patch 34169|linux|2.6.16.38|200808031003
1978994|pass|200808|srezic@cpan.org|Sort-MultipleFields|1.0|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200808031131
1979426|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200808031308
1979701|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200808031444
1980396|pass|200808|dpavlin@rot13.org ((Dobrica Pavlinusic))|Data-Compare|1.19|i486-linux-gnu|5.10.0|linux|2.6.25.10|200808032133
1980946|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-thread-multi-64int|5.11.0 patch 34169|freebsd|7.0-release|200808032245
1981115|na|200808|srezic@cpan.org|Palm-Treo680MessagesDB|1.0|i386-freebsd|5.5.5|freebsd|6.1-release|200808040140
1981178|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200808040110
1981369|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200808040136
1981852|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34169|openbsd|4.1|200808040345
1981982|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34169|openbsd|4.1|200808040454
1981986|pass|200808|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|i386-freebsd-thread-multi-64int|5.11.0 patch 34169|freebsd|7.0-release|200808040324
1982322|pass|200808|srezic@cpan.org|Sort-MultipleFields|1.0|i386-freebsd|5.6.2|freebsd|6.1-release|200808040829
1985957|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200808050827
1985958|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200808050827
1986173|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-thread-multi-64int|5.11.0 patch 34170|freebsd|6.2-release|200808050950
1986927|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-19-generic|200808051559
1987031|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.11.0 patch 34170|linux|2.6.24-19-generic|200808051640
1988273|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.10.0 patch 34161|linux|2.6.24-19-generic|200808052200
1990402|pass|200808|yanick@cpan.org ("Yanick Champoux")|Data-Compare|1.19|i686-linux|5.10.0|linux|2.6.22-gentoo-r2enkidu|200808060050
1993045|pass|200808|srezic@cpan.org|Tie-Hash-Rank|1.0.1|i386-freebsd|5.5.5|freebsd|6.1-release|200808061711
1993209|pass|200808|srezic@cpan.org|Tie-Scalar-Decay|1.1.1|i386-freebsd|5.5.5|freebsd|6.1-release|200808061710
1994194|pass|200808|srezic@cpan.org|Tie-Scalar-Decay|1.1.1|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200808062152
1994372|pass|200808|srezic@cpan.org|Tie-Hash-Rank|1.0.1|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200808062154
1994469|unknown|200808|srezic@cpan.org|Acme-Pony|1.1.2|i386-freebsd|5.5.5|freebsd|6.1-release|200808062231
1994636|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.11.0 patch 34174|linux|2.6.21.5-smp|200808062134
1996505|pass|200808|srezic@cpan.org|Tie-Scalar-Decay|1.1.1|i386-freebsd|5.6.2|freebsd|6.1-release|200808070918
1996526|pass|200808|srezic@cpan.org|Tie-Hash-Rank|1.0.1|i386-freebsd|5.6.2|freebsd|6.1-release|200808070921
1996771|unknown|200808|srezic@cpan.org|Acme-Pony|1.1.2|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200808071017
1997654|unknown|200808|srezic@cpan.org|Acme-Pony|1.1.2|i386-freebsd|5.6.2|freebsd|6.1-release|200808071253
1999527|unknown|200808|srezic@cpan.org|Acme-Licence|1.0|i386-freebsd|5.5.5|freebsd|6.1-release|200808071845
2000199|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-dragonfly-thread-multi-64int|5.11.0 patch 34175|dragonfly|2.0.0-release|200808072155
2000255|pass|200808|srezic@cpan.org|Tie-Hash-Longest|1.1|i386-freebsd|5.5.5|freebsd|6.1-release|200808072227
2001810|pass|200808|srezic@cpan.org|Statistics-ChiSquare|0.5|i386-freebsd|5.5.5|freebsd|6.1-release|200808080804
2001827|unknown|200808|srezic@cpan.org|Acme-Licence|1.0|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200808080830
2002243|unknown|200808|srezic@cpan.org|Acme-Licence|1.0|i386-freebsd|5.6.2|freebsd|6.1-release|200808080953
2002433|pass|200808|vpit@cpan.org|Devel-CheckLib|0.5|sun4-solaris-thread-multi-64|5.10.0|solaris|2.7|200808080127
2002615|pass|200808|srezic@cpan.org|NestedMap|1.0|i386-freebsd|5.5.5|freebsd|6.1-release|200808081055
2002850|pass|200808|vpit@cpan.org|Devel-CheckOS|1.43|sun4-solaris-thread-multi-64|5.10.0|solaris|2.7|200808080125
2003523|pass|200808|srezic@cpan.org|Tie-Hash-Longest|1.1|i386-freebsd|5.6.2|freebsd|6.1-release|200808081350
2003686|pass|200808|srezic@cpan.org|Statistics-ChiSquare|0.5|i386-freebsd|5.6.2|freebsd|6.1-release|200808081425
2004412|pass|200808|srezic@cpan.org|Tie-Hash-Longest|1.1|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200808081652
2004545|pass|200808|srezic@cpan.org|NestedMap|1.0|i386-freebsd|5.6.2|freebsd|6.1-release|200808081722
2004701|pass|200808|srezic@cpan.org|Statistics-ChiSquare|0.5|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200808081737
2005947|pass|200808|srezic@cpan.org|NestedMap|1.0|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200808082105
2006352|pass|200808|srezic@cpan.org|Statistics-SerialCorrelation|1.1|i386-freebsd|5.5.5|freebsd|6.1-release|200808082202
2007141|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-dragonfly-thread-multi-64int|5.10.0 patch 34161|dragonfly|2.0.0-release|200808090233
2008766|pass|200808|srezic@cpan.org|Statistics-SerialCorrelation|1.1|i386-freebsd|5.6.2|freebsd|6.1-release|200808091214
2009821|pass|200808|srezic@cpan.org|Statistics-SerialCorrelation|1.1|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200808091353
2010349|pass|200808|stro@cpan.org|Sort-MultipleFields|1.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200808091937
2011846|pass|200808|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Devel-CheckOS|1.43|i86pc-solaris|5.10.0|solaris|2.11|200808100230
2012066|pass|200808|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Sort-MultipleFields|1.0|i86pc-solaris|5.10.0|solaris|2.11|200808100338
2012237|pass|200808|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|XML-Tiny|1.11|i86pc-solaris|5.10.0|solaris|2.11|200808100415
2012452|pass|200808|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|XML-DoubleEncodedEntities|1.0|i86pc-solaris|5.10.0|solaris|2.11|200808100701
2014850|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.11.0 patch 34192|netbsd|4.0|200808110207
2016490|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34197|linux|2.6.22-1-k7|200808111356
2016902|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34197|linux|2.6.22-1-k7|200808111555
2017203|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 34197|linux|2.6.22-1-k7|200808111703
2017416|pass|200808|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Hexdumper|1.4|i86pc-solaris|5.10.0|solaris|2.11|200808111825
2017451|pass|200808|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Compare|1.19|i86pc-solaris|5.10.0|solaris|2.11|200808111834
2018259|pass|200808|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|DBIx-Class-SingletonRows|0.11|i86pc-solaris|5.10.0|solaris|2.11|200808120046
2018882|pass|200808|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Data-Transactional|1.02|i86pc-solaris|5.10.0|solaris|2.11|200808120433
2019240|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-thread-multi-64int|5.11.0 patch 34201|freebsd|7.0-release|200808120403
2019367|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-ld|5.11.0 patch 34197|linux|2.6.24.5-smp|200808120651
2020015|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-thread-multi-64int|5.10.0 patch 34161|freebsd|7.0-release|200808120710
2021100|pass|200808|rick@jrmyers.net ("Rick Myers")|Devel-CheckOS|1.43|i686-linux|5.10.0|linux|2.6.24.3|200808120312
2021633|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34197|linux|2.6.22-1-k7|200808121726
2021647|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34197|linux|2.6.22-1-k7|200808121727
2023871|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34197|linux|2.6.22-1-k7|200808131503
2027286|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.11.0 patch 34203|solaris|2.10|200808141452
2028942|pass|200808|brooks-cpan-tester@deseret.com|Devel-CheckOS|1.43|x86_64-linux|5.8.8|linux|2.6.9-023stab044.4-smp|200808141358
2029012|pass|200808|rick@jrmyers.net ("Rick Myers")|Data-Hexdumper|1.4|i686-linux|5.8.8|linux|2.6.24.3|200808150140
2029368|pass|200808|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Devel-CheckOS|1.43|i86pc-solaris-64int|5.10.0|solaris|2.11|200808151003
2029752|pass|200808|stro@cpan.org|Statistics-ChiSquare|0.5|i686-linux|5.10.0|linux|2.4.27-2-386|200808151016
2029819|pass|200808|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|Sort-MultipleFields|1.0|i86pc-solaris-64int|5.10.0|solaris|2.11|200808151231
2029928|pass|200808|stro@cpan.org|Statistics-SerialCorrelation|1.1|i686-linux|5.10.0|linux|2.4.27-2-386|200808151101
2029960|pass|200808|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|XML-Tiny|1.11|i86pc-solaris-64int|5.10.0|solaris|2.11|200808151305
2030081|pass|200808|stro@cpan.org|Sub-WrapPackages|1.2|i686-linux|5.10.0|linux|2.4.27-2-386|200808151133
2030248|pass|200808|Jost.Krieger+smokeback@ruhr-uni-bochum.de ("Josts Smokehouse")|XML-DoubleEncodedEntities|1.0|i86pc-solaris-64int|5.10.0|solaris|2.11|200808151443
2032178|pass|200808|rick@jrmyers.net ("Rick Myers")|Devel-CheckOS|1.43|i686-linux|5.8.8|linux|2.6.24.3|200808150705
2032940|pass|200808|srezic@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|i386-freebsd|5.10.0|freebsd|6.1-release|200808160849
2035176|pass|200808|srezic@cpan.org|Sub-WrapPackages|1.2|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200808161754
2035227|pass|200808|rick@jrmyers.net ("Rick Myers")|Number-Phone|1.6|i686-linux|5.8.8|linux|2.6.24.3|200808161251
2035519|pass|200808|JOST@cpan.org ("Josts Smokehouse")|Data-Hexdumper|1.4|i86pc-solaris-64int|5.10.0|solaris|2.11|200808161324
2035534|pass|200808|JOST@cpan.org ("Josts Smokehouse")|Data-Compare|1.19|i86pc-solaris-64int|5.10.0|solaris|2.11|200808161341
2037202|pass|200808|srezic@cpan.org|Class-CanBeA|1.2|i386-freebsd|5.10.0|freebsd|6.1-release|200808170827
2037266|pass|200808|srezic@cpan.org|Sub-WrapPackages|1.2|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200808170900
2037620|pass|200808|srezic@cpan.org|Sub-WrapPackages|1.2|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200808171001
2038600|pass|200808|srezic@cpan.org|File-Overwrite|1.0|i386-freebsd|5.10.0|freebsd|6.1-release|200808171540
2038755|pass|200808|srezic@cpan.org|Palm-SMS|0.02|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200808171541
2038973|pass|200808|rafl@debian.org|Devel-CheckLib|0.5|i686-linux-multi|5.10.0|linux|2.6.24-rc5|200808171641
2039107|pass|200808|rafl@debian.org|Devel-CheckOS|1.43|i686-linux-multi|5.10.0|linux|2.6.24-rc5|200808171644
2040114|pass|200808|srezic@cpan.org|Palm-SMS|0.02|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200808172114
2040460|pass|200808|srezic@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200808172321
2041167|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux|5.11.0 patch 34204|linux|2.6.16.38|200808180203
2041799|pass|200808|srezic@cpan.org|Palm-SMS|0.02|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200808180838
2042345|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 34204|linux|2.6.22-1-k7|200808181112
2042373|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34204|linux|2.6.22-1-k7|200808181115
2042962|pass|200808|rick@jrmyers.net ("Rick Myers")|Bryar|3.0|i686-linux|5.8.8|linux|2.6.24.3|200808180651
2043275|pass|200808|rick@jrmyers.net ("Rick Myers")|Data-Compare|1.19|i686-linux|5.8.8|linux|2.6.24.3|200808180740
2043444|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34204|linux|2.6.22-1-k7|200808181402
2044098|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34204|linux|2.6.22-1-k7|200808181544
2044616|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34204|linux|2.6.22-1-k7|200808181713
2045471|pass|200808|srezic@cpan.org|Number-Phone|1.6|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200808181711
2045989|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34204|linux|2.6.22-1-k7|200808181920
2046283|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0 patch 34161|openbsd|4.3|200808181934
2046651|pass|200808|srezic@cpan.org|Palm-SMS|0.02|i386-freebsd|5.10.0|freebsd|6.1-release|200808181958
2047415|pass|200808|srezic@cpan.org|Class-CanBeA|1.2|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200808182212
2048317|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200808190533
2048368|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200808190548
2048381|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200808190550
2048399|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200808190553
2048411|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200808190557
2048450|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200808190614
2048486|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200808190632
2048568|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200808190645
2048588|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200808190658
2048980|pass|200808|JOST@cpan.org ("Josts Smokehouse")|Data-Transactional|1.02|i86pc-solaris-64int|5.10.0|solaris|2.11|200808182153
2049353|pass|200808|JOST@cpan.org ("Josts Smokehouse")|Number-Phone|1.6|i86pc-solaris-64int|5.10.0|solaris|2.11|200808190300
2049726|pass|200808|rafl@debian.org|Devel-CheckOS|1.43|x86_64-linux|5.10.0|linux|2.6.18-6-xen-amd64|200808182146
2051551|pass|200808|rafl@debian.org|Devel-CheckOS|1.43|x86_64-linux|5.10.0|linux|2.6.18-6-xen-amd64|200808191418
2051862|pass|200808|rick@jrmyers.net ("Rick Myers")|Number-Phone|1.6|i686-linux|5.6.2|linux|2.6.24.3|200808190925
2052618|pass|200808|srezic@cpan.org|File-Overwrite|1.0|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200808191826
2053168|pass|200808|srezic@cpan.org|Games-Dice-Advanced|1.1|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200808191826
2053301|pass|200808|srezic@cpan.org|File-Overwrite|1.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200808191826
2053463|pass|200808|srezic@cpan.org|Games-Dice-Advanced|1.1|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200808191826
2054987|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.19|i86pc-solaris-thread-multi-64int|5.6.2|solaris|2.10|200808192316
2055631|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.19|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200808200539
2055923|pass|200808|srezic@cpan.org|File-Overwrite|1.0|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200808200749
2055983|pass|200808|srezic@cpan.org|Games-Dice-Advanced|1.1|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200808200749
2057195|pass|200808|shlomif@cpan.org|Devel-CheckOS|1.43|i686-linux|5.10.0|linux|2.6.26.2-desktop-2mnb|200808192108
2057250|pass|200808|rick@jrmyers.net ("Rick Myers")|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux|5.8.8|linux|2.6.24.3|200808200655
2057810|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.10.0 patch 34161|solaris|2.10|200808201411
2059653|pass|200808|srezic@cpan.org|Class-CanBeA|1.2|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200808202236
2060673|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34208|linux|2.6.22-1-k7|200808210423
2060880|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.19|i686-linux-64int|5.11.0 patch 34208|linux|2.6.22-1-k7|200808210513
2061033|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.19|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200808210107
2061156|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.19|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200808210137
2061484|pass|200808|srezic@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200808210739
2061621|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34208|linux|2.6.22-1-k7|200808210810
2063986|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34208|linux|2.6.22-1-k7|200808211357
2064102|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34208|linux|2.6.22-1-k7|200808211516
2064510|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.20|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200808211743
2064513|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.20|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200808211743
2064515|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.20|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200808211743
2064516|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.20|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200808211744
2064523|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.20|i686-linux-64int|5.8.8|linux|2.6.14|200808211744
2064530|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.20|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200808211746
2064533|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.20|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200808211746
2064554|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.10.0 patch 34161|netbsd|4.0|200808211645
2064567|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.20|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200808211745
2064936|pass|200808|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Data-Compare|1.20|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200808220323
2064961|pass|200808|srezic@cpan.org|Data-Compare|1.20|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200808212029
2065204|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34208|linux|2.6.22-1-k7|200808212130
2065411|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.8.8 patch 34096|netbsd|4.0|200808212048
2066247|pass|200808|david@cantrell.org.uk|Data-Compare|1.20|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200808220134
2066292|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200808220251
2066294|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200808220252
2066295|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200808220253
2066296|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200808220253
2066298|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200808220254
2066300|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200808220255
2066302|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200808220255
2066304|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200808220256
2066306|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200808220257
2066308|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200808220257
2066367|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.11.0 patch 34212|linux|2.6.21.5-smp|200808220245
2067198|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.20|i686-linux-64int|5.11.0 patch 34208|linux|2.6.22-1-k7|200808220857
2067222|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.19|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200808220819
2067427|pass|200808|imacat@mail.imacat.idv.tw|Data-Compare|1.20|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200808221703
2067458|pass|200808|imacat@mail.imacat.idv.tw|Data-Compare|1.20|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200808221717
2068324|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-netbsd-thread-multi-64int|5.10.0 patch 34161|netbsd|4.0|200808221626
2068382|pass|200808|stro@cpan.org|Data-Compare|1.20|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200808221842
2068467|pass|200808|david@cantrell.org.uk|Data-Compare|1.20|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200808221701
2068507|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200808221719
2068520|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-netbsd-64int|5.11.0 patch 34169|netbsd|4.0|200808221726
2068840|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200808221733
2068887|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200808221740
2068905|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-netbsd-thread-multi-64int|5.8.8 patch 34096|netbsd|4.0|200808221834
2068971|pass|200808|g.grigelionis@computer.org|Data-Compare|1.20|i586-linux-thread-multi|5.8.5|linux|2.6.8.1|200808222001
2068972|pass|200808|g.grigelionis@computer.org|Data-Compare|1.20|i586-linux-thread-multi|5.8.8|linux|2.6.16|200808222002
2069329|pass|200808|srezic@cpan.org|Data-Compare|1.20|i386-freebsd|5.10.0|freebsd|6.1-release|200808222136
2069741|pass|200808|shlomif@cpan.org|Data-Compare|1.20|i686-linux|5.10.0|linux|2.6.26.2-desktop-2mnb|200808221504
2069771|pass|200808|artur+cpan@niif.spb.su (ARTO)|Data-Compare|1.20|i386-linux-thread-multi|5.10.0|linux|2.6.18-92.1.6.el5|200808222145
2070163|pass|200808|david@cantrell.org.uk|Data-Compare|1.20|i686-linux|5.6.2|linux|2.4.27-3-686|200808230033
2070440|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i686-linux-thread-multi-64int-ld|5.11.0 patch 34212|linux|2.6.21.5-smp|200808230235
2070919|pass|200808|rick@jrmyers.net ("Rick Myers")|XML-Tiny|1.11|i686-linux|5.10.0|linux|2.6.24.3|200808230317
2071133|pass|200808|rick@jrmyers.net ("Rick Myers")|Data-Hexdumper|1.4|i686-linux|5.10.0|linux|2.6.24.3|200808230555
2071338|pass|200808|claco@chrislaco.com|Data-Compare|1.20|darwin-thread-multi-2level|5.8.8|darwin|9.0|200808230018
2071425|pass|200808|rick@jrmyers.net ("Rick Myers")|XML-DoubleEncodedEntities|1.0|i686-linux|5.10.0|linux|2.6.24.3|200808230709
2071692|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200808231339
2072061|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200808231723
2072209|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34218|linux|2.6.22-1-k7|200808231936
2072470|pass|200808|rafl@debian.org|Data-Compare|1.20|i686-linux-multi|5.10.0|linux|2.6.24-rc5|200808232020
2072533|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200808231829
2073155|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.20|i686-linux-64int|5.11.0 patch 34218|linux|2.6.22-1-k7|200808232148
2073186|fail|200808|srezic@cpan.org|Data-Compare|1.20|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200808232151
2073566|pass|200808|srezic@cpan.org|NestedMap|1.0|i386-freebsd|5.10.0|freebsd|6.1-release|200808232225
2073732|pass|200808|srezic@cpan.org|Data-Compare|1.20|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200808232123
2074123|pass|200808|srezic@cpan.org|Class-CanBeA|1.2|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200808232123
2074801|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-freebsd-thread-multi-64int|5.11.0 patch 34218|freebsd|7.0-release|200808232205
2074805|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-thread-multi-64int|5.11.0 patch 34218|freebsd|7.0-release|200808232206
2075907|pass|200808|rick@jrmyers.net ("Rick Myers")|Devel-CheckOS|1.43|i686-linux|5.6.2|linux|2.6.24.3|200808240047
2075993|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34218|linux|2.6.22-1-k7|200808240708
2076432|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34218|linux|2.6.22-1-k7|200808240834
2076708|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34218|linux|2.6.22-1-k7|200808240918
2076997|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34218|linux|2.6.22-1-k7|200808240951
2077246|pass|200808|srezic@cpan.org|Data-Compare|1.20|i386-freebsd|5.6.2|freebsd|6.1-release|200808241010
2078672|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|OpenBSD.i386-openbsd-64int|5.11.0 patch 34218|openbsd|4.2|200808241519
2078905|pass|200808|srezic@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200808241606
2079549|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-19-generic|200808242014
2080095|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i686-linux-thread-multi-ld|5.11.0 patch 34225|linux|2.6.24-19-generic|200808242229
2081096|pass|200808|srezic@cpan.org|Data-Compare|1.20|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200808250450
2081180|pass|200808|rick@jrmyers.net|Devel-CheckOS|1.43|i686-linux|5.10.0|linux|2.6.24.3|200808250311
2082190|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i686-linux-64int-ld|5.10.0 patch 34161|linux|2.6.24-19-generic|200808251225
2082734|pass|200808|hohokus@gmail.com|Data-Compare|1.20|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200808250932
2082785|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200808251445
2082915|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200808251519
2083908|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-netbsd-thread-multi-64int|5.11.0 patch 34225|netbsd|4.0|200808251845
2084703|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i686-linux-64int-ld|5.8.8 patch 34096|linux|2.6.24-19-generic|200808260127
2085617|pass|200808|srezic@cpan.org|Statistics-SerialCorrelation|1.1|i386-freebsd|5.10.0|freebsd|6.1-release|200808260842
2085777|pass|200808|srezic@cpan.org|Statistics-ChiSquare|0.5|i386-freebsd|5.10.0|freebsd|6.1-release|200808260842
2086330|pass|200808|stro@cpan.org|Data-Compare|1.20|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200808261335
2086870|pass|200808|rick@jrmyers.net ("Rick Myers")|XML-Tiny|1.11|i686-linux|5.8.8|linux|2.6.24.3|200808260813
2086986|pass|200808|rick@jrmyers.net ("Rick Myers")|XML-Tiny|1.11|i686-linux|5.6.2|linux|2.6.24.3|200808260836
2087422|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34226|linux|2.6.22-1-k7|200808261521
2087755|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34226|linux|2.6.22-1-k7|200808261456
2087975|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.20|i686-linux-64int|5.11.0 patch 34226|linux|2.6.22-1-k7|200808261643
2089096|pass|200808|rick@jrmyers.net ("Rick Myers")|Data-Compare|1.20|i686-linux|5.10.0|linux|2.6.24.3|200808261322
2089169|pass|200808|rick@jrmyers.net ("Rick Myers")|XML-DoubleEncodedEntities|1.0|i686-linux|5.8.8|linux|2.6.24.3|200808261335
2089572|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34226|linux|2.6.22-1-k7|200808262033
2089756|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-dragonfly-thread-multi-64int|5.8.8|dragonfly|2.0.0-release|200808262039
2089787|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-dragonfly-thread-multi-64int|5.10.0|dragonfly|2.0.0-release|200808262045
2089835|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34226|linux|2.6.22-1-k7|200808262040
2090135|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34226|linux|2.6.22-1-k7|200808262145
2090474|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.200_500|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200808262312
2090475|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.200_500|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200808262313
2090477|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.200_500|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200808262313
2090479|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.200_500|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200808262314
2090489|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.200_500|i686-linux-64int|5.8.8|linux|2.6.14|200808262315
2090503|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.200_500|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200808262318
2090706|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.200_500|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200808262316
2090707|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.200_500|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200808262317
2090725|pass|200808|david@cantrell.org.uk|Data-Compare|1.20|i686-linux|5.8.8|linux|2.4.27-3-686|200808262257
2090771|pass|200808|srezic@cpan.org|Data-Compare|1.200_500|i386-freebsd|5.10.0|freebsd|6.1-release|200808270000
2092079|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200808270619
2092080|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200808270619
2092082|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200808270620
2092083|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200808270621
2092085|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200808270621
2092086|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200808270622
2092089|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200808270623
2092092|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200808270624
2092094|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200808270625
2092097|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200808270625
2092290|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.11.0 patch 34225|netbsd|4.0|200808270722
2093120|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200808271221
2093398|pass|200808|david@cantrell.org.uk|Data-Compare|1.20|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200808270738
2093507|pass|200808|david@cantrell.org.uk|Data-Compare|1.20|IP35-irix-64int|5.8.8|irix|6.5|200808271500
2093624|pass|200808|david@cantrell.org.uk|Data-Compare|1.20|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200808271232
2094114|pass|200808|cpan@herlpacker.co.uk|Data-Compare|1.200_500|i486-linux-gnu|5.8.8|linux|2.6.15.7|200808271419
2094165|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-dragonfly-thread-multi-64int|5.10.0 patch 34161|dragonfly|2.0.0-release|200808271658
2094167|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i386-dragonfly-thread-multi-64int|5.10.0 patch 34161|dragonfly|2.0.0-release|200808271700
2094507|fail|200808|rick@jrmyers.net ("Rick Myers")|Bryar|3.0|i686-linux|5.10.0|linux|2.6.24.3|200808270846
2094847|pass|200808|stro@cpan.org|Devel-CheckLib|0.5|armv5tejl-linux|5.10.0|linux|2.6.17.14|200808272011
2094886|pass|200808|stro@cpan.org|Devel-CheckLib|0.5|armv5tejl-linux|5.8.8|linux|2.6.17.14|200808272014
2094976|pass|200808|srezic@cpan.org|Data-Compare|1.200_500|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200808271919
2094995|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-dragonfly-64int|5.11.0 patch 34233|dragonfly|2.0.0-release|200808271901
2095008|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i386-dragonfly-64int|5.11.0 patch 34233|dragonfly|2.0.0-release|200808271902
2095144|pass|200808|rick@jrmyers.net ("Rick Myers")|CPAN-FindDependencies|2.0|i686-linux|5.8.8|linux|2.6.24.3|200808271337
2095165|pass|200808|rick@jrmyers.net ("Rick Myers")|CPAN-FindDependencies|2.0|i686-linux|5.10.0|linux|2.6.24.3|200808271338
2095284|pass|200808|rick@jrmyers.net ("Rick Myers")|Data-Compare|1.20|i686-linux|5.6.2|linux|2.6.24.3|200808271348
2095553|fail|200808|rick@jrmyers.net ("Rick Myers")|CPAN-FindDependencies|2.0|i686-linux|5.6.2|linux|2.6.24.3|200808271432
2095708|pass|200808|rick@jrmyers.net ("Rick Myers")|CPU-Emulator-Memory|1.1001|i686-linux|5.8.8|linux|2.6.24.3|200808271510
2095711|pass|200808|rick@jrmyers.net ("Rick Myers")|CPU-Emulator-Z80|1.0|i686-linux|5.8.8|linux|2.6.24.3|200808271512
2095822|pass|200808|rick@jrmyers.net ("Rick Myers")|CPU-Emulator-Memory|1.1001|i686-linux|5.10.0|linux|2.6.24.3|200808271549
2095833|pass|200808|rick@jrmyers.net ("Rick Myers")|CPU-Emulator-Z80|1.0|i686-linux|5.10.0|linux|2.6.24.3|200808271550
2096531|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-dragonfly-thread-multi-64int|5.8.8 patch 34096|dragonfly|2.0.0-release|200808280030
2096537|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i386-dragonfly-thread-multi-64int|5.8.8 patch 34096|dragonfly|2.0.0-release|200808280032
2096958|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-freebsd-thread-multi-64int|5.10.0 patch 34161|freebsd|7.0-release|200808280031
2096961|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i386-freebsd-thread-multi-64int|5.10.0 patch 34161|freebsd|7.0-release|200808280033
2097113|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-freebsd-thread-multi|5.11.0 patch 34233|freebsd|7.0-release|200808280141
2097115|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i386-freebsd-thread-multi|5.11.0 patch 34233|freebsd|7.0-release|200808280143
2097915|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-dragonfly-thread-multi-64int|5.10.0 patch 34161|dragonfly|2.0.0-release|200808280949
2098421|pass|200808|cpan@herlpacker.co.uk|Data-Compare|1.20|i486-linux-gnu|5.8.8|linux|2.6.15.7|200808280347
2098456|pass|200808|cpan@herlpacker.co.uk|Devel-CheckOS|1.43|i486-linux-gnu|5.8.8|linux|2.6.15.7|200808281035
2098735|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-dragonfly-64int|5.11.0 patch 34233|dragonfly|2.0.0-release|200808281333
2098779|pass|200808|stro@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux|5.10.0|linux|2.4.27-2-386|200808281020
2099111|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200808281117
2099969|pass|200808|stro@cpan.org|Data-Compare|1.20|i686-linux|5.10.0|linux|2.4.27-2-386|200808281411
2100251|pass|200808|david@cantrell.org.uk|Data-Compare|1.200_500|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200808281754
2100283|pass|200808|stro@cpan.org|Data-Compare|1.20|i686-linux|5.8.8|linux|2.4.27-2-386|200808281956
2100457|pass|200808|david@cantrell.org.uk|Data-Compare|1.200_500|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200808281721
2101020|pass|200808|srezic@cpan.org|Tie-Hash-Longest|1.1|i386-freebsd|5.10.0|freebsd|6.1-release|200808282129
2101100|unknown|200808|srezic@cpan.org|Acme-Pony|1.1.2|i386-freebsd|5.10.0|freebsd|6.1-release|200808282130
2101199|pass|200808|srezic@cpan.org|Data-Compare|1.200_500|i386-freebsd|5.6.2|freebsd|6.1-release|200808282135
2101531|unknown|200808|srezic@cpan.org|Acme-Licence|1.0|i386-freebsd|5.10.0|freebsd|6.1-release|200808282130
2102011|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-dragonfly-thread-multi-64int|5.8.8 patch 34096|dragonfly|2.0.0-release|200808282328
2102284|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Class-DBI-ClassGenerator|1.0|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200808290030
2102290|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Class-DBI-ClassGenerator|1.0|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200808290031
2102308|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Class-DBI-ClassGenerator|1.0|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200808290034
2102312|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Class-DBI-ClassGenerator|1.0|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200808290034
2102320|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Class-DBI-ClassGenerator|1.0|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200808290031
2102324|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Class-DBI-ClassGenerator|1.0|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200808290032
2102325|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Class-DBI-ClassGenerator|1.0|i686-linux-64int|5.8.8|linux|2.6.14|200808290033
2102326|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Class-DBI-ClassGenerator|1.0|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200808290033
2102373|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-thread-multi|5.11.0 patch 34233|freebsd|7.0-release|200808282034
2102398|pass|200808|stro@cpan.org|Number-Phone|1.6|i686-linux|5.8.8|linux|2.4.27-2-386|200808290126
2102647|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-thread-multi-64int|5.10.0 patch 34161|freebsd|7.0-release|200808290017
2102935|pass|200808|david@cantrell.org.uk|Data-Compare|1.20|i686-linux-thread-multi|5.10.0|linux|2.4.27-3-686|200808290055
2103035|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34234|linux|2.6.22-1-k7|200808290211
2103229|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34234|linux|2.6.22-1-k7|200808290227
2103434|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.20|i686-linux-64int|5.11.0 patch 34234|linux|2.6.22-1-k7|200808290312
2103568|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34234|linux|2.6.22-1-k7|200808290348
2104091|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34234|linux|2.6.22-1-k7|200808290554
2105206|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200808290814
2105209|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200808290815
2105219|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|darwin-thread-multi-64int-2level|5.8.6|darwin|8.10.0|200808290816
2105234|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200808290816
2105247|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200808290817
2105252|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200808290818
2105258|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|darwin-thread-multi-64int-2level|5.8.6|darwin|8.10.0|200808290819
2105265|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200808290820
2105938|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200808290951
2106681|pass|200808|srezic@cpan.org|Data-Compare|1.200_500|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200808291334
2106692|fail|200808|chris@bingosnet.co.uk|Class-DBI-ClassGenerator|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200808291239
2107403|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34234|linux|2.6.22-1-k7|200808291506
2108246|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200808291712
2108287|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200808291727
2108604|fail|200808|srezic@cpan.org|Class-DBI-ClassGenerator|1.0|i386-freebsd|5.6.2|freebsd|6.1-release|200808292011
2108719|fail|200808|srezic@cpan.org|Class-DBI-ClassGenerator|1.0|i386-freebsd|5.10.0|freebsd|6.1-release|200808292040
2108765|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i86pc-solaris-thread-multi-64int|5.11.0 patch 34234|solaris|2.10|200808291956
2108996|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200808292103
2109051|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0 patch 34161|openbsd|4.3|200808292112
2109463|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34234|openbsd|4.3|200808292339
2110749|fail|200808|srezic@cpan.org|Class-DBI-ClassGenerator|1.0|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200808300810
2110879|fail|200808|srezic@cpan.org|Class-DBI-ClassGenerator|1.0|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200808300841
2110994|pass|200808|rick@jrmyers.net ("Rick Myers")|Games-Dice-Advanced|1.1|i686-linux|5.6.2|linux|2.6.24.3|200808300306
2111004|fail|200808|srezic@cpan.org|Class-DBI-ClassGenerator|1.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200808300904
2111005|fail|200808|srezic@cpan.org|Class-DBI-ClassGenerator|1.0|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200808300904
2112339|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200808301515
2112373|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200808301528
2112873|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i86pc-solaris-thread-multi-64int|5.11.0 patch 34234|solaris|2.10|200808301829
2113254|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0 patch 34161|openbsd|4.3|200808302103
2113274|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.11.0 patch 34234|solaris|2.10|200808302114
2113325|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200808302139
2113593|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0 patch 34161|openbsd|4.3|200808310000
2113928|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34234|openbsd|4.3|200808310238
2114152|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34234|openbsd|4.3|200808310515
2114170|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|darwin-thread-multi-64int-2level|5.8.6|darwin|8.10.0|200808310534
2114229|pass|200808|chris@bingosnet.co.uk|Devel-CheckOS|1.43|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200808310606
2115011|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34237|linux|2.6.22-1-k7|200808311239
2115094|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200808311155
2115120|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200808311202
2116106|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200808311420
2116113|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200808311422
2116364|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200808311502
2116382|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200808311504
2116417|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.20|i686-linux-64int|5.11.0 patch 34237|linux|2.6.22-1-k7|200808311548
2116847|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34237|linux|2.6.22-1-k7|200808311627
2117292|pass|200808|rick@jrmyers.net ("Rick Myers")|Number-Phone|1.6|i686-linux|5.10.0|linux|2.6.24.3|200808311248
2117351|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34237|linux|2.6.22-1-k7|200808311854
2118245|pass|200808|rick@jrmyers.net ("Rick Myers")|Palm-TreoPhoneCallDB|1.1|i686-linux|5.10.0|linux|2.6.24.3|200808311437
2118585|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34237|linux|2.6.22-1-k7|200808312124
2118957|pass|200808|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34237|linux|2.6.22-1-k7|200808312218
2119384|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.200_500|i686-linux-thread-multi-64int-ld|5.11.0 patch 34237|linux|2.6.16.38|200808312231
2119435|pass|200808|chris@bingosnet.co.uk|Data-Compare|1.20|i686-linux-thread-multi-64int-ld|5.11.0 patch 34237|linux|2.6.16.38|200808312239
2119735|pass|200808|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200808312337
2119793|pass|200809|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.11.0 patch 34237|linux|2.6.16.38|200809010033
2120156|pass|200809|stro@cpan.org|Class-DBI-ClassGenerator|1.0|i686-linux|5.10.0|linux|2.4.27-2-386|200809010537
2120442|pass|200809|stro@cpan.org|Data-Compare|1.200_500|i686-linux|5.10.0|linux|2.4.27-2-386|200809010647
2120514|pass|200809|stro@cpan.org|Class-DBI-ClassGenerator|1.0|i686-linux|5.8.8|linux|2.4.27-2-386|200809010655
2120517|pass|200809|rick@jrmyers.net ("Rick Myers")|Class-DBI-ClassGenerator|1.0|i686-linux|5.10.0|linux|2.6.24.3|200809010029
2120778|pass|200809|stro@cpan.org|Data-Compare|1.200_500|i686-linux|5.8.8|linux|2.4.27-2-386|200809010855
2121277|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.200_500|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200809011108
2121300|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.20|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200809011114
2121647|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.200_500|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200809011256
2121692|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.20|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200809011302
2122006|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.200_500|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200809011452
2122014|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.20|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200809011454
2122158|pass|200809|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200809011552
2122165|pass|200809|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|darwin-thread-multi-64int-2level|5.8.6|darwin|8.10.0|200809011555
2122170|pass|200809|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200809011558
2122414|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200809011833
2122416|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200809011833
2122420|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200809011833
2122421|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200809011834
2122424|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.8|linux|2.6.14|200809011834
2122425|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200809011834
2122429|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-thread-multi-64int|5.8.6|linux|2.6.18-4-k7|200809011835
2122432|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-thread-multi-64int|5.8.5|linux|2.6.18-4-k7|200809011835
2122703|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.200_500|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200809011911
2122729|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.20|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200809011920
2122923|pass|200809|srezic@cpan.org|Data-Compare|1.21|i386-freebsd|5.10.0|freebsd|6.1-release|200809012154
2123235|pass|200809|srezic@cpan.org|Data-Compare|1.21|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200809012347
2123274|pass|200809|srezic@cpan.org|Data-Compare|1.21|i386-freebsd|5.6.2|freebsd|6.1-release|200809012355
2123275|pass|200809|srezic@cpan.org|Data-Compare|1.21|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200809012355
2123451|pass|200809|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Data-Compare|1.21|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200809020814
2123737|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200809020145
2123739|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200809020146
2123741|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200809020147
2123742|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200809020147
2123744|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200809020148
2123745|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200809020149
2123747|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200809020149
2123748|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200809020150
2123749|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200809020151
2123752|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200809020151
2124180|pass|200809|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.1|200809020557
2124282|pass|200809|srezic@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200809020753
2124283|pass|200809|srezic@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200809020751
2124285|pass|200809|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200809020656
2124458|fail|200809|david@cantrell.org.uk|Class-DBI-ClassGenerator|1.0|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200809020816
2124573|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200809020845
2124579|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200809020846
2124587|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|darwin-thread-multi-64int-2level|5.8.6|darwin|8.10.0|200809020846
2124592|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200809020847
2124706|fail|200809|david@cantrell.org.uk|Class-DBI-ClassGenerator|1.0|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200809020822
2124735|pass|200809|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200809020928
2124737|pass|200809|david@cantrell.org.uk|Data-Compare|1.21|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200809020828
2124773|pass|200809|imacat@mail.imacat.idv.tw|Data-Compare|1.21|x86_64-linux-thread-multi-ld|5.8.8|linux|2.6.22.10|200809021637
2124774|pass|200809|imacat@mail.imacat.idv.tw|Data-Compare|1.21|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200809021637
2125023|pass|200809|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi|5.8.8|openbsd|4.1|200809021104
2125137|pass|200809|rafl@debian.org|Data-Compare|1.21|i686-linux-multi|5.10.0|linux|2.6.24-rc5|200809021255
2125465|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.200_500|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34238|openbsd|4.1|200809021411
2125471|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34238|openbsd|4.1|200809021412
2125812|pass|200809|hohokus@gmail.com|Data-Compare|1.21|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200809020949
2126196|pass|200809|artur+cpan@niif.spb.su (ARTO)|Data-Compare|1.21|i386-linux-thread-multi|5.10.0|linux|2.6.18-92.1.6.el5|200809012146
2126563|pass|200809|rick@jrmyers.net ("Rick Myers")|Palm-TreoPhoneCallDB|1.1|i686-linux|5.8.8|linux|2.6.24.3|200809021204
2126630|pass|200809|rick@jrmyers.net ("Rick Myers")|Class-DBI-ClassGenerator|1.0|i686-linux|5.8.8|linux|2.6.24.3|200809021214
2126647|pass|200809|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200809021715
2126719|pass|200809|david@cantrell.org.uk|Data-Compare|1.200_500|i686-linux|5.6.2|linux|2.4.27-3-686|200809021742
2126957|pass|200809|stro@cpan.org|Data-Compare|1.21|i686-linux|5.10.0|linux|2.4.27-2-386|200809022101
2127042|pass|200809|david@cantrell.org.uk|Data-Compare|1.21|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200809021603
2127349|pass|200809|david@cantrell.org.uk|Data-Compare|1.21|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200809021425
2127399|pass|200809|g.grigelionis@computer.org|Data-Compare|1.21|i586-linux-thread-multi|5.8.8|linux|2.6.16|200809022139
2127698|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34243|linux|2.6.22-1-k7|200809022243
2127723|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34243|linux|2.6.22-1-k7|200809022254
2127739|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i86pc-solaris-thread-multi-64int|5.10.0 patch 34161|solaris|2.10|200809022153
2128059|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34243|linux|2.6.22-1-k7|200809030011
2128162|pass|200809|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.10.0 patch 34161|solaris|2.10|200809022328
2128314|pass|200809|kamensky@cpan.org ("Moshe Kamensky")|Data-Compare|1.21|sun4-solaris|5.10.0|solaris|2.8|200809021035
2128711|pass|200809|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34238|openbsd|4.1|200809030050
2128775|pass|200809|david@cantrell.org.uk|Data-Compare|1.21|i686-linux|5.8.8|linux|2.4.27-3-686|200809030101
2129258|pass|200809|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200809030255
2129836|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34243|linux|2.6.22-1-k7|200809030817
2130314|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34243|linux|2.6.22-1-k7|200809030928
2130745|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34243|linux|2.6.22-1-k7|200809031006
2131867|pass|200809|JOST@cpan.org ("Josts Smokehouse")|Data-Compare|1.21|i86pc-solaris-64int|5.10.0|solaris|2.11|200809031242
2132073|pass|200809|JOST@cpan.org ("Josts Smokehouse")|Class-DBI-ClassGenerator|1.0|i86pc-solaris-64int|5.10.0|solaris|2.11|200809031339
2132570|pass|200809|vpit@cpan.org|Data-Compare|1.200_500|x86_64-linux-thread-multi|5.10.0|linux|2.6.24.2-fuuka.profvince.com|200809031533
2132572|pass|200809|vpit@cpan.org|Data-Compare|1.21|x86_64-linux-thread-multi|5.10.0|linux|2.6.24.2-fuuka.profvince.com|200809031534
2132621|pass|200809|vpit@cpan.org|Devel-CheckOS|1.43|x86_64-linux-thread-multi|5.10.0|linux|2.6.24.2-fuuka.profvince.com|200809031543
2132704|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200809031335
2132760|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200809031347
2132788|pass|200809|david@cantrell.org.uk|Data-Compare|1.200_500|i686-linux|5.8.8|linux|2.4.27-3-686|200809031526
2132967|pass|200809|david@cantrell.org.uk|Data-Compare|1.200_500|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200809031015
2133120|fail|200809|david@cantrell.org.uk|Class-DBI-ClassGenerator|1.0|IP35-irix-64int|5.8.8|irix|6.5|200809031755
2133127|pass|200809|david@cantrell.org.uk|Data-Compare|1.200_500|IP35-irix-64int|5.8.8|irix|6.5|200809031756
2134058|fail|200809|srezic@cpan.org|Class-DBI-ClassGenerator|1.01|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200809032134
2134276|fail|200809|david@cantrell.org.uk|Class-DBI-ClassGenerator|1.0|i686-linux|5.8.8|linux|2.4.27-3-686|200809032122
2134304|pass|200809|david@cantrell.org.uk|Data-Compare|1.21|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200809032125
2134327|fail|200809|srezic@cpan.org|Class-DBI-ClassGenerator|1.01|i386-freebsd|5.10.0|freebsd|6.1-release|200809032232
2134426|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Class-DBI-ClassGenerator|1.01|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200809032244
2134428|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Class-DBI-ClassGenerator|1.01|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200809032244
2134429|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Class-DBI-ClassGenerator|1.01|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200809032245
2134431|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Class-DBI-ClassGenerator|1.01|i686-linux-64int|5.8.8|linux|2.6.14|200809032245
2134432|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Class-DBI-ClassGenerator|1.01|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200809032245
2134542|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Class-DBI-ClassGenerator|1.01|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200809032244
2134677|fail|200809|srezic@cpan.org|Class-DBI-ClassGenerator|1.01|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200809032345
2135029|pass|200809|vpit@cpan.org|XML-DoubleEncodedEntities|1.0|x86_64-linux-thread-multi|5.10.0|linux|2.6.24.2-fuuka.profvince.com|200809040049
2135133|pass|200809|david@cantrell.org.uk|Data-Compare|1.21|i686-linux|5.6.2|linux|2.4.27-3-686|200809040028
2136726|fail|200809|david@cantrell.org.uk|Class-DBI-ClassGenerator|1.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200809040443
2136741|pass|200809|david@cantrell.org.uk|Data-Compare|1.200_500|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200809040445
2136884|pass|200809|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200809040528
2136914|pass|200809|chris@bingosnet.co.uk|Class-DBI-ClassGenerator|1.01|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200809040912
2136974|pass|200809|srezic@cpan.org|Class-DBI-ClassGenerator|1.01|i386-freebsd|5.6.2|freebsd|6.1-release|200809041033
2137123|pass|200809|JOST@cpan.org ("Josts Smokehouse")|Class-DBI-ClassGenerator|1.01|i86pc-solaris-64int|5.10.0|solaris|2.11|200809041109
2137189|pass|200809|claco@chrislaco.com|Data-Compare|1.21|darwin-thread-multi-2level|5.8.8|darwin|9.0|200809031956
2137306|pass|200809|srezic@cpan.org|Class-DBI-ClassGenerator|1.01|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200809041141
2137485|pass|200809|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200809040720
2137803|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200809041316
2138046|pass|200809|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i86pc-solaris-thread-multi-64int|5.10.0 patch 34161|solaris|2.10|200809041517
2138691|pass|200809|rick@jrmyers.net ("Rick Myers")|Data-Compare|1.21|i686-linux|5.10.0|linux|2.6.24.3|200809041233
2138785|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200809041704
2139060|pass|200809|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckLib|0.5|i86pc-solaris-64int|5.10.0|solaris|2.11|200809042023
2140153|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 34266|linux|2.6.22-1-k7|200809042353
2140156|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.10.0 patch 34266|linux|2.6.22-1-k7|200809042354
2140181|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.10.0 patch 34266|linux|2.6.22-1-k7|200809050030
2140954|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.21.5-smp|200809050032
2140978|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 34266|linux|2.6.22-1-k7|200809050207
2141182|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200809050112
2142272|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0 patch 34266|linux|2.6.22-1-k7|200809050642
2142646|pass|200809|JOST@cpan.org ("Josts Smokehouse")|CPAN-FindDependencies|2.0|i86pc-solaris-64int|5.10.0|solaris|2.11|200809050740
2143390|fail|200809|JOST@cpan.org ("Josts Smokehouse")|Bryar|3.0|i86pc-solaris-64int|5.10.0|solaris|2.11|200809050959
2143413|pass|200809|g.grigelionis@computer.org|Data-Compare|1.21|i586-linux-thread-multi|5.8.5|linux|2.6.8.1|200809051001
2143584|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 34266|linux|2.6.22-1-k7|200809050954
2144940|pass|200809|JOST@cpan.org ("Josts Smokehouse")|File-Find-Rule-Permissions|1.3|i86pc-solaris-64int|5.10.0|solaris|2.11|200809051418
2145002|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200809051336
2145208|pass|200809|jrockway@cpan.org|Data-Compare|1.21|x86_64-linux|5.10.0|linux|2.6.22-3-amd64|200809050656
2145547|pass|200809|david@cantrell.org.uk|Class-DBI-ClassGenerator|1.01|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200809051514
2145666|pass|200809|rick@jrmyers.net ("Rick Myers")|Palm-SMS|0.02|i686-linux|5.10.0|linux|2.6.24.3|200809051039
2146109|pass|200809|rick@jrmyers.net ("Rick Myers")|Data-Transactional|1.02|i686-linux|5.8.8|linux|2.6.24.3|200809051232
2146110|pass|200809|rick@jrmyers.net ("Rick Myers")|Data-Compare|1.21|i686-linux|5.8.8|linux|2.6.24.3|200809051232
2146406|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200809051828
2146859|pass|200809|andy@hexten.net|Data-Compare|1.21|darwin-thread-multi-2level|5.10.0|darwin|9.2.2|200809052042
2146903|pass|200809|andy@hexten.net|Data-Compare|1.21|darwin-thread-multi-2level|5.10.0|darwin|9.2.2|200809052044
2146936|pass|200809|shlomif@cpan.org|Data-Compare|1.21|i686-linux|5.10.0|linux|2.6.26.2-desktop-2mnb|200809052248
2146953|pass|200809|david@cantrell.org.uk|Data-Compare|1.21|IP35-irix-64int|5.8.8|irix|6.5|200809052153
2147039|pass|200809|andy@hexten.net|Data-Compare|1.21|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200809052103
2147815|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.8.8 patch 34279|linux|2.6.22-1-k7|200809060038
2148192|pass|200809|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200809060001
2148403|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 34279|linux|2.6.22-1-k7|200809060245
2148500|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 34279|linux|2.6.22-1-k7|200809060246
2148806|pass|200809|rjbs@cpan.org|Data-Compare|1.21|darwin-2level|5.10.0|darwin|9.1.0|200809060200
2148848|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.8 patch 34279|linux|2.6.22-1-k7|200809060205
2148887|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.6.2|netbsd|4.0|200809060310
2149478|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 34279|linux|2.6.22-1-k7|200809060620
2150047|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 34279|linux|2.6.22-1-k7|200809060919
2150562|pass|200809|dillman@ria.buffalo.edu|Data-Compare|1.21|sun4-solaris|5.10.0|solaris|2.9|200809051516
2151414|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i686-linux-thread-multi-ld|5.11.0 patch 34282|linux|2.6.21.5-smp|200809061758
2151492|pass|200809|rick@jrmyers.net ("Rick Myers")|Palm-Treo680MessagesDB|1.0|i686-linux|5.10.0|linux|2.6.24.3|200809061318
2152473|fail|200809|rick@jrmyers.net ("Rick Myers")|Net-Random|2.0|i686-linux|5.10.0|linux|2.6.24.3|200809061749
2152821|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 34301|linux|2.6.22-1-k7|200809062350
2152856|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.8.8 patch 34301|linux|2.6.22-1-k7|200809070133
2153132|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 34301|linux|2.6.22-1-k7|200809070233
2154234|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.8 patch 34301|linux|2.6.22-1-k7|200809070700
2154322|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 34301|linux|2.6.22-1-k7|200809070719
2154565|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 34301|linux|2.6.22-1-k7|200809070831
2155585|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-Hash-Rank|1.0.1|i686-linux|5.8.8|linux|2.6.24.3|200809070542
2155658|pass|200809|rick@jrmyers.net ("Rick Myers")|Data-Transactional|1.02|i686-linux|5.10.0|linux|2.6.24.3|200809070555
2156076|pass|200809|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-ld|5.11.0 patch 34282|linux|2.6.21.5-smp|200809071257
2156378|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-Hash-Rank|1.0.1|i686-linux|5.10.0|linux|2.6.24.3|200809070934
2157510|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.8 patch 34304|linux|2.6.22-1-k7|200809071950
2158290|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.8.8 patch 34304|linux|2.6.22-1-k7|200809072204
2158758|pass|200809|david@cantrell.org.uk|Class-DBI-ClassGenerator|1.01|IP35-irix-64int|5.8.8|irix|6.5|200809072310
2158801|pass|200809|david@cantrell.org.uk|Class-DBI-ClassGenerator|1.01|i686-linux|5.8.8|linux|2.4.27-3-686|200809072214
2159044|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200809072251
2159070|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200809072254
2159366|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-Hash-Longest|1.1|i686-linux|5.10.0|linux|2.6.24.3|200809071842
2159738|pass|200809|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200809080100
2159772|pass|200809|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200809080116
2160285|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 34304|linux|2.6.22-1-k7|200809080652
2160582|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 34304|linux|2.6.22-1-k7|200809080745
2161534|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 34304|linux|2.6.22-1-k7|200809081021
2161756|pass|200809|JOST@cpan.org ("Josts Smokehouse")|Tie-STDOUT|1.03|i86pc-solaris-64int|5.10.0|solaris|2.11|200809081110
2162236|pass|200809|JOST@cpan.org ("Josts Smokehouse")|Palm-TreoPhoneCallDB|1.1|i86pc-solaris-64int|5.10.0|solaris|2.11|200809081327
2162663|pass|200809|rick@jrmyers.net ("Rick Myers")|NestedMap|1.0|i686-linux|5.10.0|linux|2.6.24.3|200809080915
2164470|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.10.0 patch 34305|netbsd|4.0|200809081903
2164719|pass|200809|dwheeler@cpan.org|Data-Compare|1.21|darwin-2level|5.10.0|darwin|9.4.0|200809081854
2164829|pass|200809|JOST@cpan.org ("Josts Smokehouse")|Net-Random|2.0|i86pc-solaris-64int|5.10.0|solaris|2.11|200809082106
2165191|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-Hash-Longest|1.1|i686-linux|5.8.8|linux|2.6.24.3|200809081551
2165321|pass|200809|srezic@cpan.org|Data-Compare|1.21|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809082130
2166664|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 34304|linux|2.6.22-1-k7|200809090041
2166850|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200809090011
2166888|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200809090017
2166963|pass|200809|JOST@cpan.org ("Josts Smokehouse")|Number-Phone-UK-DetailedLocations|1.3|i86pc-solaris-64int|5.10.0|solaris|2.11|200809090135
2168102|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34322|linux|2.6.22-1-k7|200809090453
2168515|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.8 patch 34313|netbsd|4.0|200809090446
2168549|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34322|linux|2.6.22-1-k7|200809090530
2168917|pass|200809|rick@jrmyers.net ("Rick Myers")|Palm-SMS|0.02|i686-linux|5.8.8|linux|2.6.24.3|200809090101
2169493|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34322|linux|2.6.22-1-k7|200809090845
2169887|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34322|linux|2.6.22-1-k7|200809091001
2170196|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-STDOUT|1.03|i686-linux|5.10.0|linux|2.6.24.3|200809090442
2170228|pass|200809|rick@jrmyers.net ("Rick Myers")|Games-Dice-Advanced|1.1|i686-linux|5.10.0|linux|2.6.24.3|200809090445
2170535|pass|200809|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809091035
2170688|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34322|linux|2.6.22-1-k7|200809091146
2171000|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34322|linux|2.6.22-1-k7|200809091224
2171365|pass|200809|rick@jrmyers.net ("Rick Myers")|File-Overwrite|1.0|i686-linux|5.10.0|linux|2.6.24.3|200809090717
2172415|pass|200809|rick@jrmyers.net ("Rick Myers")|Sub-WrapPackages|1.2|i686-linux|5.10.0|linux|2.6.24.3|200809090948
2173132|pass|200809|david@cantrell.org.uk|Class-DBI-ClassGenerator|1.01|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200809091118
2173799|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.10.0 patch 34305|linux|2.6.22-1-k7|200809091905
2173893|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.10.0 patch 34305|linux|2.6.22-1-k7|200809091923
2174004|pass|200809|JOST@cpan.org ("Josts Smokehouse")|Games-Dice-Advanced|1.1|i86pc-solaris-64int|5.10.0|solaris|2.11|200809091941
2174587|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 34305|linux|2.6.22-1-k7|200809092124
2174928|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 34305|linux|2.6.22-1-k7|200809092215
2175725|pass|200809|JOST@cpan.org ("Josts Smokehouse")|Sub-WrapPackages|1.2|i86pc-solaris-64int|5.10.0|solaris|2.11|200809100019
2176322|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0 patch 34305|linux|2.6.22-1-k7|200809100330
2177303|pass|200809|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.10.0 patch 34305|netbsd|4.0|200809100536
2177586|pass|200809|rick@jrmyers.net ("Rick Myers")|Palm-Treo680MessagesDB|1.0|i686-linux|5.8.8|linux|2.6.24.3|200809100233
2178345|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 34305|linux|2.6.22-1-k7|200809100908
2179057|pass|200809|srezic@cpan.org|Devel-CheckLib|0.5|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809100846
2179468|pass|200809|srezic@cpan.org|Devel-CheckOS|1.43|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809100846
2179568|pass|200809|srezic@cpan.org|CPU-Emulator-Z80|1.0|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809100845
2179569|pass|200809|srezic@cpan.org|CPU-Emulator-Memory|1.1001|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809100845
2179797|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-freebsd-64int|5.10.0|freebsd|7.0-release|200809100822
2179849|pass|200809|srezic@cpan.org|Sort-MultipleFields|1.0|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809100847
2179889|pass|200809|srezic@cpan.org|DBIx-Class-SingletonRows|0.11|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809100845
2179938|pass|200809|srezic@cpan.org|Data-Hexdumper|1.4|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809100845
2180325|pass|200809|srezic@cpan.org|XML-Tiny|1.11|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809100848
2180393|pass|200809|srezic@cpan.org|Data-Transactional|1.02|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809100845
2180813|pass|200809|srezic@cpan.org|Number-Phone|1.6|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809101259
2180933|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.10.0 patch 34325|freebsd|7.0-release|200809101058
2182471|pass|200809|JOST@cpan.org ("Josts Smokehouse")|File-Overwrite|1.0|i86pc-solaris-64int|5.10.0|solaris|2.11|200809101709
2182493|pass|200809|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-netbsd-thread-multi-64int|5.8.8 patch 34313|netbsd|4.0|200809101525
2182526|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.8.8 patch 34327|freebsd|7.0-release|200809101506
2183605|pass|200809|JOST@cpan.org ("Josts Smokehouse")|Class-CanBeA|1.2|i86pc-solaris-64int|5.10.0|solaris|2.11|200809102018
2183883|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809101945
2184050|pass|200809|rick@jrmyers.net ("Rick Myers")|Statistics-SerialCorrelation|1.1|i686-linux|5.10.0|linux|2.6.24.3|200809101520
2184500|pass|200809|srezic@cpan.org|Bryar|3.0|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809102241
2184805|pass|200809|srezic@cpan.org|CPAN-FindDependencies|2.0|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809102322
2185551|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|i686-linux-64int-ld|5.10.0|linux|2.6.24.5-smp|200809110110
2186886|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 34327|linux|2.6.22-1-k7|200809110758
2186951|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.8.8 patch 34327|linux|2.6.22-1-k7|200809110813
2187759|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.8 patch 34327|linux|2.6.22-1-k7|200809110955
2188217|pass|200809|srezic@cpan.org|Palm-TreoPhoneCallDB|1.1|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809111052
2188457|pass|200809|srezic@cpan.org|Number-Phone-UK-DetailedLocations|1.3|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809111110
2188515|pass|200809|srezic@cpan.org|Net-Random|2.0|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809111117
2189291|fail|200809|JOST@cpan.org ("Josts Smokehouse")|Palm-SMS|0.02|i86pc-solaris-64int|5.10.0|solaris|2.11|200809111240
2189385|pass|200809|JOST@cpan.org ("Josts Smokehouse")|Acme-Scurvy-Whoreson-BilgeRat|1.1|i86pc-solaris-64int|5.10.0|solaris|2.11|200809111252
2189538|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 34327|linux|2.6.22-1-k7|200809111311
2189654|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 34327|linux|2.6.22-1-k7|200809111325
2190278|pass|200809|srezic@cpan.org|File-Find-Rule-Permissions|1.3|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809111426
2191574|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 34327|linux|2.6.22-1-k7|200809111659
2192929|pass|200809|rick@jrmyers.net ("Rick Myers")|Net-Random|2.0|i686-linux|5.8.8|linux|2.6.24.3|200809111332
2193628|pass|200809|andy@hexten.net|Data-Compare|1.21|i86pc-solaris-thread-multi|5.10.0|solaris|2.11|200809120152
2194022|pass|200809|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-thread-multi-64int|5.10.0 patch 34325|freebsd|7.0-release|200809111951
2194874|pass|200809|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-freebsd-thread-multi-64int|5.8.8 patch 34327|freebsd|7.0-release|200809112155
2195859|pass|200809|chris@bingosnet.co.uk|Devel-CheckOS|1.43|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809120405
2195893|pass|200809|JOST@cpan.org ("Josts Smokehouse")|Statistics-SerialCorrelation|1.1|i86pc-solaris-64int|5.10.0|solaris|2.11|200809120514
2196189|pass|200809|rick@jrmyers.net ("Rick Myers")|NestedMap|1.0|i686-linux|5.8.8|linux|2.6.24.3|200809120114
2196789|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200809120852
2197199|pass|200809|JOST@cpan.org ("Josts Smokehouse")|Tie-Hash-Longest|1.1|i86pc-solaris-64int|5.10.0|solaris|2.11|200809121015
2197203|pass|200809|JOST@cpan.org ("Josts Smokehouse")|Statistics-ChiSquare|0.5|i86pc-solaris-64int|5.10.0|solaris|2.11|200809121015
2197209|pass|200809|rick@jrmyers.net ("Rick Myers")|Sort-MultipleFields|1.0|i686-linux|5.10.0|linux|2.6.24.3|200809120417
2197970|pass|200809|srezic@cpan.org|Sub-WrapPackages|1.2|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809121217
2200967|pass|200809|srezic@cpan.org|Palm-SMS|0.02|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809121821
2201560|pass|200809|JOST@cpan.org ("Josts Smokehouse")|NestedMap|1.0|i86pc-solaris-64int|5.10.0|solaris|2.11|200809122005
2202046|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34354|linux|2.6.22-1-k7|200809122121
2202341|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-STDOUT|1.03|i686-linux|5.8.8|linux|2.6.24.3|200809121558
2202370|pass|200809|rick@jrmyers.net ("Rick Myers")|Games-Dice-Advanced|1.1|i686-linux|5.8.8|linux|2.6.24.3|200809121602
2203007|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34354|linux|2.6.22-1-k7|200809122334
2203270|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34354|linux|2.6.22-1-k7|200809122156
2203548|pass|200809|rick@jrmyers.net ("Rick Myers")|File-Overwrite|1.0|i686-linux|5.8.8|linux|2.6.24.3|200809121901
2204171|pass|200809|rick@jrmyers.net ("Rick Myers")|Sub-WrapPackages|1.2|i686-linux|5.8.8|linux|2.6.24.3|200809130040
2206477|pass|200809|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24.5-smp|200809130456
2206563|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34354|linux|2.6.22-1-k7|200809130623
2206854|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|darwin-2level|5.8.8|darwin|8.10.1|200809130639
2207038|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34354|linux|2.6.22-1-k7|200809130738
2207162|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34354|linux|2.6.22-1-k7|200809130749
2207391|unknown|200809|JOST@cpan.org ("Josts Smokehouse")|Acme-Licence|1.0|i86pc-solaris-64int|5.10.0|solaris|2.11|200809130817
2209250|pass|200809|srezic@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809131320
2209305|pass|200809|bingos@cpan.org|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.10.0 patch 34342|linux|2.6.24.5-smp|200809131218
2209944|pass|200809|bingos@cpan.org|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.8.8 patch 34347|linux|2.6.24.5-smp|200809131408
2210024|pass|200809|rick@jrmyers.net ("Rick Myers")|Number-Phone-UK-DetailedLocations|1.3|i686-linux|5.10.0|linux|2.6.24.3|200809130931
2210032|pass|200809|rick@jrmyers.net ("Rick Myers")|Statistics-SerialCorrelation|1.1|i686-linux|5.8.8|linux|2.6.24.3|200809130932
2211035|pass|200809|srezic@cpan.org|Class-CanBeA|1.2|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809131806
2211723|pass|200809|rick@jrmyers.net ("Rick Myers")|Class-CanBeA|1.2|i686-linux|5.10.0|linux|2.6.24.3|200809131419
2213262|pass|200809|rick@jrmyers.net ("Rick Myers")|Sort-MultipleFields|1.0|i686-linux|5.8.8|linux|2.6.24.3|200809131828
2214291|unknown|200809|JOST@cpan.org ("Josts Smokehouse")|Acme-Pony|1.1.2|i86pc-solaris-64int|5.10.0|solaris|2.11|200809140253
2214362|pass|200809|JOST@cpan.org ("Josts Smokehouse")|Tie-Hash-Rank|1.0.1|i86pc-solaris-64int|5.10.0|solaris|2.11|200809140301
2214565|pass|200809|JOST@cpan.org ("Josts Smokehouse")|Tie-Scalar-Decay|1.1.1|i86pc-solaris-64int|5.10.0|solaris|2.11|200809140355
2216453|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 34342|linux|2.6.22-1-k7|200809140830
2216455|pass|200809|bingos@cpan.org|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.8.8 patch 34347|solaris|2.10|200809140729
2216472|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0 patch 34342|linux|2.6.22-1-k7|200809140834
2216530|pass|200809|rick@jrmyers.net ("Rick Myers")|Number-Phone-UK-DetailedLocations|1.3|i686-linux|5.8.8|linux|2.6.24.3|200809140241
2216571|pass|200809|bingos@cpan.org|Devel-CheckOS|1.43|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200809140745
2217634|pass|200809|rick@jrmyers.net ("Rick Myers")|Class-CanBeA|1.2|i686-linux|5.8.8|linux|2.6.24.3|200809140716
2217915|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-Scalar-Decay|1.1.1|i686-linux|5.10.0|linux|2.6.24.3|200809140815
2218095|pass|200809|rick@jrmyers.net ("Rick Myers")|Statistics-ChiSquare|0.5|i686-linux|5.10.0|linux|2.6.24.3|200809140844
2218653|pass|200809|rick@jrmyers.net ("Rick Myers")|File-Find-Rule-Permissions|1.3|i686-linux|5.10.0|linux|2.6.24.3|200809141107
2220303|pass|200809|srezic@cpan.org|NestedMap|1.0|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809142146
2220467|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.8.8 patch 34360|linux|2.6.22-1-k7|200809142250
2220769|pass|200809|srezic@cpan.org|Statistics-ChiSquare|0.5|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809142343
2221043|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 34360|linux|2.6.22-1-k7|200809150019
2221441|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 34360|linux|2.6.22-1-k7|200809150114
2221518|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.8 patch 34360|linux|2.6.22-1-k7|200809150145
2222888|pass|200809|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200809150543
2222968|pass|200809|bingos@cpan.org|Devel-CheckOS|1.43|i386-freebsd-thread-multi-64int|5.10.0 patch 34342|freebsd|7.0-release|200809150451
2223014|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 34360|linux|2.6.22-1-k7|200809150650
2223018|pass|200809|rick@jrmyers.net ("Rick Myers")|Class-DBI-ClassGenerator|1.01|i686-linux|5.10.0|linux|2.6.24.3|200809150058
2223485|pass|200809|srezic@cpan.org|Tie-Hash-Longest|1.1|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809150710
2223644|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 34360|linux|2.6.22-1-k7|200809150759
2226111|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-Scalar-Decay|1.1.1|i686-linux|5.8.8|linux|2.6.24.3|200809151119
2226205|pass|200809|rick@jrmyers.net ("Rick Myers")|Statistics-ChiSquare|0.5|i686-linux|5.8.8|linux|2.6.24.3|200809151148
2226565|pass|200809|rick@jrmyers.net ("Rick Myers")|File-Find-Rule-Permissions|1.3|i686-linux|5.8.8|linux|2.6.24.3|200809151239
2228780|pass|200809|bingos@cpan.org|Devel-CheckOS|1.43|i386-freebsd-thread-multi-64int|5.8.8 patch 34360|freebsd|7.0-release|200809160415
2229376|pass|200809|feldt@nhn.ou.edu|Data-Compare|1.21|sun4-solaris-thread-multi|5.8.8|solaris|2.8|200809151343
2232416|pass|200809|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.10.0 patch 34372|linux|2.6.24-19-generic|200809170320
2233753|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34378|linux|2.6.22-1-k7|200809171121
2234064|pass|200809|dmdk1@comcast.net|Devel-CheckOS|1.43|i486-linux-gnu|5.8.8|linux|2.6.15.7|200809170640
2235521|pass|200809|stro@cpan.org|Class-DBI-ClassGenerator|1.0|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200809171951
2235523|pass|200809|stro@cpan.org|Class-DBI-ClassGenerator|1.01|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200809171952
2236401|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34378|linux|2.6.22-1-k7|200809172122
2236480|pass|200809|bingos@cpan.org|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.8.8 patch 34376|linux|2.6.24-19-generic|200809172047
2236751|pass|200809|stro@cpan.org|Data-Compare|1.21|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200809172332
2236778|pass|200809|stro@cpan.org|Data-Compare|1.21|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200809172334
2237726|pass|200809|bingos@cpan.org|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.11.0 patch 34378|linux|2.6.24-19-generic|200809172353
2237796|pass|200809|rick@jrmyers.net ("Rick Myers")|Data-Compare|1.20|i686-linux|5.8.8|linux|2.6.24.3|200809171902
2238524|pass|200809|bingos@cpan.org|Devel-CheckOS|1.43|i386-freebsd-thread-multi-64int|5.8.8 patch 34376|freebsd|6.2-release|200809180053
2238607|pass|200809|rick@jrmyers.net ("Rick Myers")|Class-DBI-ClassGenerator|1.01|i686-linux|5.8.8|linux|2.6.24.3|200809180105
2238659|pass|200809|chris@bingosnet.co.uk|Acme-Licence|1.0|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809180212
2238665|pass|200809|chris@bingosnet.co.uk|Acme-Pony|1.1.2|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809180212
2238669|pass|200809|chris@bingosnet.co.uk|Acme-Scurvy-Whoreson-BilgeRat|1.1|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809180213
2238701|pass|200809|chris@bingosnet.co.uk|Bryar|3.0|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809180218
2238722|pass|200809|chris@bingosnet.co.uk|CPAN-FindDependencies|2.0|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809180221
2238727|pass|200809|chris@bingosnet.co.uk|CPU-Emulator-Memory|1.1001|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809180222
2238745|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34378|linux|2.6.22-1-k7|200809180325
2238766|pass|200809|chris@bingosnet.co.uk|CPU-Emulator-Z80|1.0|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809180228
2238776|pass|200809|chris@bingosnet.co.uk|Class-CanBeA|1.2|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809180229
2238813|pass|200809|chris@bingosnet.co.uk|Class-DBI-ClassGenerator|1.01|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809180234
2238901|pass|200809|chris@bingosnet.co.uk|DBIx-Class-SingletonRows|0.11|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809180245
2238908|pass|200809|chris@bingosnet.co.uk|Data-Hexdumper|1.4|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809180245
2238916|pass|200809|chris@bingosnet.co.uk|Data-Transactional|1.02|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809180246
2238924|pass|200809|chris@bingosnet.co.uk|Devel-CheckLib|0.5|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809180247
2238937|pass|200809|chris@bingosnet.co.uk|File-Find-Rule-Permissions|1.3|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809180248
2238941|pass|200809|chris@bingosnet.co.uk|File-Overwrite|1.0|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809180249
2238945|pass|200809|chris@bingosnet.co.uk|Games-Dice-Advanced|1.1|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809180249
2238948|pass|200809|chris@bingosnet.co.uk|NestedMap|1.0|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809180250
2238956|pass|200809|chris@bingosnet.co.uk|Net-Random|2.0|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809180252
2238963|pass|200809|chris@bingosnet.co.uk|Number-Phone|1.6|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809180253
2238976|pass|200809|chris@bingosnet.co.uk|Number-Phone-UK-DetailedLocations|1.3|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809180255
2238977|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34378|linux|2.6.22-1-k7|200809180355
2239002|pass|200809|chris@bingosnet.co.uk|Palm-Treo680MessagesDB|1.0|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809180258
2239018|pass|200809|chris@bingosnet.co.uk|Palm-TreoPhoneCallDB|1.1|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809180300
2239023|pass|200809|chris@bingosnet.co.uk|Sort-MultipleFields|1.0|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809180301
2239033|pass|200809|chris@bingosnet.co.uk|Statistics-ChiSquare|0.5|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809180302
2239042|pass|200809|chris@bingosnet.co.uk|Statistics-SerialCorrelation|1.1|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809180303
2239047|pass|200809|chris@bingosnet.co.uk|Sub-WrapPackages|1.2|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809180304
2239048|pass|200809|chris@bingosnet.co.uk|Tie-Hash-Longest|1.1|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809180304
2239051|pass|200809|chris@bingosnet.co.uk|Tie-Hash-Rank|1.0.1|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809180305
2239057|pass|200809|chris@bingosnet.co.uk|Tie-STDOUT|1.03|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809180306
2239060|pass|200809|chris@bingosnet.co.uk|Tie-Scalar-Decay|1.1.1|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809180307
2239066|pass|200809|chris@bingosnet.co.uk|XML-DoubleEncodedEntities|1.0|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809180308
2239072|pass|200809|chris@bingosnet.co.uk|XML-Tiny|1.11|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809180309
2239564|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34378|linux|2.6.22-1-k7|200809180543
2240143|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34378|linux|2.6.22-1-k7|200809180743
2240621|pass|200809|rlane10@wi.rr.com|Data-Compare|1.21|i386-linux-thread-multi|5.10.0|linux|2.6.18-92.1.6.el5xen|200809171734
2242839|pass|200809|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200809182205
2244075|pass|200809|chris@bingosnet.co.uk|Devel-CheckOS|1.43|i386-dragonfly-thread-multi-64int|5.8.8|dragonfly|2.0.0-release|200809190651
2245958|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 34376|linux|2.6.22-1-k7|200809191226
2246060|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 34376|linux|2.6.22-1-k7|200809191305
2246291|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 34376|linux|2.6.22-1-k7|200809191259
2246429|pass|200809|rlane10@wi.rr.com|Data-Compare|1.21|darwin-2level-ld|5.10.0|darwin|8.11.0|200809190145
2246637|pass|200809|stro@cpan.org|Data-Compare|1.21|i686-linux|5.8.8|linux|2.4.27-2-386|200809191143
2246666|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.8.8 patch 34376|linux|2.6.22-1-k7|200809191147
2246685|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.8 patch 34376|linux|2.6.22-1-k7|200809191137
2247058|pass|200809|rick@jrmyers.net ("Rick Myers")|Devel-CheckOS|1.43|i686-linux|5.11.0 patch 34379|linux|2.6.24.3|200809190942
2247620|pass|200809|bingos@cpan.org|Devel-CheckOS|1.43|i686-linux-thread-multi-64int-ld|5.11.0 patch 34379|linux|2.6.21.5-smp|200809191726
2247631|pass|200809|bingos@cpan.org|Devel-CheckOS|1.43|i386-dragonfly-thread-multi-64int|5.8.8 patch 34376|dragonfly|2.0.0-release|200809191814
2248904|pass|200809|srezic@cpan.org|File-Overwrite|1.0|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809192112
2248907|pass|200809|srezic@cpan.org|Games-Dice-Advanced|1.1|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809192112
2250153|pass|200809|srezic@cpan.org|Statistics-SerialCorrelation|1.1|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809192113
2250570|pass|200809|srezic@cpan.org|Palm-Treo680MessagesDB|1.0|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809192112
2250695|pass|200809|srezic@cpan.org|Tie-STDOUT|1.03|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809192113
2250965|pass|200809|vpit@cpan.org|Devel-CheckOS|1.43|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200809192322
2252554|pass|200809|bingos@cpan.org|Devel-CheckOS|1.43|i386-dragonfly-thread-multi-64int|5.10.0 patch 34372|dragonfly|2.0.0-release|200809200237
2253129|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 34376|linux|2.6.22-1-k7|200809200458
2254479|pass|200809|rick@jrmyers.net ("Rick Myers")|Devel-CheckLib|0.5|i686-linux|5.8.8|linux|2.6.24.3|200809200851
2257805|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34384|linux|2.6.22-1-k7|200809210901
2258126|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34384|linux|2.6.22-1-k7|200809210954
2258972|pass|200809|rick@jrmyers.net ("Rick Myers")|Devel-CheckOS|1.43|i686-linux|5.10.0|linux|2.6.26.5|200809210654
2259096|pass|200809|rick@jrmyers.net ("Rick Myers")|Devel-CheckOS|1.43|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809210730
2259896|pass|200809|shlomif@cpan.org|Devel-CheckOS|1.43|i686-linux|5.10.0|linux|2.6.27-desktop-0.rc6.3.2mnb|200809211534
2259900|pass|200809|rick@jrmyers.net ("Rick Myers")|Devel-CheckOS|1.43|i686-linux|5.8.8|linux|2.6.26.5|200809210835
2259935|pass|200809|vpit@cpan.org|Data-Compare|1.21|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200809211442
2261013|fail|200809|vpit@cpan.org|Bryar|3.0|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200809211725
2261749|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34384|linux|2.6.22-1-k7|200809211920
2262089|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34384|linux|2.6.22-1-k7|200809211948
2263304|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34384|linux|2.6.22-1-k7|200809212219
2264023|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34384|linux|2.6.22-1-k7|200809220007
2267038|pass|200809|stro@cpan.org|Class-DBI-ClassGenerator|1.01|i686-linux|5.10.0|linux|2.4.27-2-386|200809220120
2270334|pass|200809|andy@hexten.net|Data-Hexdumper|1.4|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200809230309
2272202|pass|200809|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.21|i686-linux|5.10.0|linux|2.6.24-16-generic|200809230108
2272391|pass|200809|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200809230118
2273987|pass|200809|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux|5.10.0|linux|2.6.24-16-generic|200809230243
2274657|pass|200809|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux|5.10.0|linux|2.6.24-16-generic|200809230315
2275372|pass|200809|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux|5.10.0|linux|2.6.24-16-generic|200809230356
2276210|pass|200809|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux|5.10.0|linux|2.6.24-16-generic|200809230438
2276822|pass|200809|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200809230510
2277905|fail|200809|rick@jrmyers.net ("Rick Myers")|Bryar|3.0|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809230605
2278342|pass|200809|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.21|i686-linux|5.10.0|linux|2.6.24-16-generic|200809230630
2278630|pass|200809|andy@hexten.net|Devel-CheckLib|0.5|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200809231241
2278891|pass|200809|rick@jrmyers.net ("Rick Myers")|Data-Compare|1.21|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809230655
2279583|fail|200809|rick@jrmyers.net ("Rick Myers")|Bryar|3.0|i686-linux|5.10.0|linux|2.6.26.5|200809230813
2279896|pass|200809|rick@jrmyers.net ("Rick Myers")|Data-Compare|1.21|i686-linux|5.10.0|linux|2.6.26.5|200809230902
2282157|pass|200809|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200809231319
2282765|pass|200809|dagolden@cpan.org (DAGOLDEN)|Class-DBI-ClassGenerator|1.01|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200809231409
2282820|pass|200809|rick@jrmyers.net ("Rick Myers")|Bryar|3.0|i686-linux|5.8.8|linux|2.6.26.5|200809231412
2282898|pass|200809|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.21|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200809231419
2283100|pass|200809|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200809231436
2283878|pass|200809|rick@jrmyers.net ("Rick Myers")|Data-Hexdumper|1.4|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809231648
2284464|pass|200809|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200809231822
2284488|pass|200809|dagolden@cpan.org (DAGOLDEN)|Class-DBI-ClassGenerator|1.01|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200809231825
2284690|pass|200809|stro@cpan.org|Class-DBI-ClassGenerator|1.01|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200809240157
2285740|pass|200809|rick@jrmyers.net ("Rick Myers")|Data-Compare|1.21|i686-linux|5.8.8|linux|2.6.26.5|200809240143
2286365|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.8 patch 34412|linux|2.6.22-1-k7|200809240446
2286699|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.8.8 patch 34412|linux|2.6.22-1-k7|200809240451
2288619|pass|200809|jeff@zeroclue.com|Data-Compare|1.21|darwin-2level|5.10.0|darwin|9.2.0|200809240350
2291076|pass|200809|rick@jrmyers.net ("Rick Myers")|Data-Hexdumper|1.4|i686-linux|5.8.8|linux|2.6.26.5|200809241242
2291763|pass|200809|rick@jrmyers.net ("Rick Myers")|Data-Hexdumper|1.4|i686-linux|5.10.0|linux|2.6.26.5|200809241416
2292092|pass|200809|rick@jrmyers.net ("Rick Myers")|Palm-SMS|0.02|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809241509
2294629|pass|200809|dagolden@cpan.org (DAGOLDEN)|Sort-MultipleFields|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200809250030
2296227|pass|200809|dagolden@cpan.org (DAGOLDEN)|Palm-Treo680MessagesDB|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200809250025
2296959|pass|200809|dagolden@cpan.org (DAGOLDEN)|DBIx-Class-SingletonRows|0.11|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200809250204
2297496|pass|200809|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.02|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200809250318
2297597|pass|200809|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Z80|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200809250331
2302018|pass|200809|rick@jrmyers.net ("Rick Myers")|Palm-SMS|0.02|i686-linux|5.8.8|linux|2.6.26.5|200809251429
2302099|pass|200809|rick@jrmyers.net ("Rick Myers")|Palm-SMS|0.02|i686-linux|5.10.0|linux|2.6.26.5|200809251442
2309012|pass|200809|mgardner@usatech.com|Data-Compare|1.21|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200809261128
2309548|pass|200809|gabor@pti.co.il ("Gabor Szabo")|Devel-CheckOS|1.43|i486-linux-gnu|5.8.8|linux|2.6.15.7|200809261932
2312149|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-Hash-Rank|1.0.1|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270000
2312313|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-Hash-Longest|1.1|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270037
2312446|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-Scalar-Decay|1.1.1|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270100
2312499|pass|200809|rick@jrmyers.net ("Rick Myers")|CPU-Emulator-Memory|1.1001|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270110
2312505|pass|200809|rick@jrmyers.net ("Rick Myers")|CPU-Emulator-Z80|1.0|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270111
2312544|pass|200809|rick@jrmyers.net ("Rick Myers")|Statistics-ChiSquare|0.5|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270119
2312597|pass|200809|rick@jrmyers.net ("Rick Myers")|NestedMap|1.0|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270128
2312628|pass|200809|rick@jrmyers.net ("Rick Myers")|Data-Transactional|1.02|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270134
2312795|pass|200809|rick@jrmyers.net ("Rick Myers")|File-Find-Rule-Permissions|1.3|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270201
2312978|pass|200809|rick@jrmyers.net ("Rick Myers")|Class-DBI-ClassGenerator|1.0|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270224
2313151|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 34412|linux|2.6.22-1-k7|200809270448
2313310|pass|200809|rick@jrmyers.net ("Rick Myers")|Number-Phone|1.6|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270310
2313316|pass|200809|rick@jrmyers.net ("Rick Myers")|Devel-CheckLib|0.5|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270310
2313639|fail|200809|rick@jrmyers.net ("Rick Myers")|Data-Compare|1.20|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270349
2313864|pass|200809|rick@jrmyers.net ("Rick Myers")|Number-Phone-UK-DetailedLocations|1.3|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270025
2313873|pass|200809|rick@jrmyers.net ("Rick Myers")|Class-CanBeA|1.2|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270027
2313923|pass|200809|rick@jrmyers.net ("Rick Myers")|Statistics-SerialCorrelation|1.1|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270027
2314486|pass|200809|rick@jrmyers.net ("Rick Myers")|Sub-WrapPackages|1.2|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270235
2314715|pass|200809|rick@jrmyers.net ("Rick Myers")|Sort-MultipleFields|1.0|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270323
2314721|pass|200809|rick@jrmyers.net ("Rick Myers")|Net-Random|2.0|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270324
2314737|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-STDOUT|1.03|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270327
2314821|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-Hash-Rank|1.0.1|i686-linux|5.8.8|linux|2.6.26.5|200809270350
2314935|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-Hash-Longest|1.1|i686-linux|5.8.8|linux|2.6.26.5|200809270418
2315002|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-Scalar-Decay|1.1.1|i686-linux|5.8.8|linux|2.6.26.5|200809270429
2315044|pass|200809|rick@jrmyers.net ("Rick Myers")|CPU-Emulator-Memory|1.1001|i686-linux|5.8.8|linux|2.6.26.5|200809270437
2315047|pass|200809|rick@jrmyers.net ("Rick Myers")|CPU-Emulator-Z80|1.0|i686-linux|5.8.8|linux|2.6.26.5|200809270438
2315100|pass|200809|rick@jrmyers.net ("Rick Myers")|Statistics-ChiSquare|0.5|i686-linux|5.8.8|linux|2.6.26.5|200809270445
2315158|pass|200809|rick@jrmyers.net ("Rick Myers")|NestedMap|1.0|i686-linux|5.8.8|linux|2.6.26.5|200809270455
2315184|pass|200809|rick@jrmyers.net ("Rick Myers")|Data-Transactional|1.02|i686-linux|5.8.8|linux|2.6.26.5|200809270459
2315312|pass|200809|rick@jrmyers.net ("Rick Myers")|File-Find-Rule-Permissions|1.3|i686-linux|5.8.8|linux|2.6.26.5|200809270525
2315331|pass|200809|rick@jrmyers.net ("Rick Myers")|Games-Dice-Advanced|1.1|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270527
2315436|pass|200809|rick@jrmyers.net ("Rick Myers")|Class-DBI-ClassGenerator|1.0|i686-linux|5.8.8|linux|2.6.26.5|200809270540
2315811|pass|200809|rick@jrmyers.net ("Rick Myers")|Number-Phone|1.6|i686-linux|5.8.8|linux|2.6.26.5|200809270632
2315816|pass|200809|rick@jrmyers.net ("Rick Myers")|Devel-CheckLib|0.5|i686-linux|5.8.8|linux|2.6.26.5|200809270633
2315825|pass|200809|rick@jrmyers.net ("Rick Myers")|Class-DBI-ClassGenerator|1.01|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270633
2315835|pass|200809|rick@jrmyers.net ("Rick Myers")|File-Overwrite|1.0|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270635
2316116|fail|200809|rick@jrmyers.net ("Rick Myers")|Data-Compare|1.20|i686-linux|5.8.8|linux|2.6.26.5|200809270712
2316271|pass|200809|eshank@drwholdings.com|Data-Compare|1.21|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200809261140
2316346|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200809271246
2316400|pass|200809|rick@jrmyers.net ("Rick Myers")|Number-Phone-UK-DetailedLocations|1.3|i686-linux|5.8.8|linux|2.6.26.5|200809270757
2316405|pass|200809|rick@jrmyers.net ("Rick Myers")|Statistics-SerialCorrelation|1.1|i686-linux|5.8.8|linux|2.6.26.5|200809270758
2316406|pass|200809|rick@jrmyers.net ("Rick Myers")|Class-CanBeA|1.2|i686-linux|5.8.8|linux|2.6.26.5|200809270758
2316478|pass|200809|rick@jrmyers.net ("Rick Myers")|Sub-WrapPackages|1.2|i686-linux|5.8.8|linux|2.6.26.5|200809270808
2316644|pass|200809|rick@jrmyers.net ("Rick Myers")|Sort-MultipleFields|1.0|i686-linux|5.8.8|linux|2.6.26.5|200809270838
2316652|pass|200809|rick@jrmyers.net ("Rick Myers")|Net-Random|2.0|i686-linux|5.8.8|linux|2.6.26.5|200809270840
2316677|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-STDOUT|1.03|i686-linux|5.8.8|linux|2.6.26.5|200809270844
2316816|pass|200809|rick@jrmyers.net ("Rick Myers")|Games-Dice-Advanced|1.1|i686-linux|5.8.8|linux|2.6.26.5|200809270913
2317346|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-Hash-Rank|1.0.1|i686-linux|5.10.0|linux|2.6.26.5|200809271226
2317418|pass|200809|rick@jrmyers.net ("Rick Myers")|Class-DBI-ClassGenerator|1.01|i686-linux|5.8.8|linux|2.6.26.5|200809271241
2317421|pass|200809|rick@jrmyers.net ("Rick Myers")|File-Overwrite|1.0|i686-linux|5.8.8|linux|2.6.26.5|200809271242
2317491|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-Hash-Longest|1.1|i686-linux|5.10.0|linux|2.6.26.5|200809271256
2317583|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-Scalar-Decay|1.1.1|i686-linux|5.10.0|linux|2.6.26.5|200809271305
2317661|pass|200809|rick@jrmyers.net ("Rick Myers")|CPU-Emulator-Memory|1.1001|i686-linux|5.10.0|linux|2.6.26.5|200809271316
2317671|pass|200809|rick@jrmyers.net ("Rick Myers")|CPU-Emulator-Z80|1.0|i686-linux|5.10.0|linux|2.6.26.5|200809271317
2317715|pass|200809|rick@jrmyers.net ("Rick Myers")|Statistics-ChiSquare|0.5|i686-linux|5.10.0|linux|2.6.26.5|200809271322
2317727|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 34412|linux|2.6.22-1-k7|200809271923
2317778|pass|200809|rick@jrmyers.net ("Rick Myers")|NestedMap|1.0|i686-linux|5.10.0|linux|2.6.26.5|200809271330
2317821|pass|200809|rick@jrmyers.net ("Rick Myers")|Data-Transactional|1.02|i686-linux|5.10.0|linux|2.6.26.5|200809271335
2317935|pass|200809|rick@jrmyers.net ("Rick Myers")|File-Find-Rule-Permissions|1.3|i686-linux|5.10.0|linux|2.6.26.5|200809271348
2318062|pass|200809|rick@jrmyers.net ("Rick Myers")|Class-DBI-ClassGenerator|1.0|i686-linux|5.10.0|linux|2.6.26.5|200809271359
2318391|pass|200809|rick@jrmyers.net ("Rick Myers")|Number-Phone|1.6|i686-linux|5.10.0|linux|2.6.26.5|200809271435
2318395|pass|200809|rick@jrmyers.net ("Rick Myers")|Devel-CheckLib|0.5|i686-linux|5.10.0|linux|2.6.26.5|200809271436
2318736|fail|200809|rick@jrmyers.net ("Rick Myers")|Data-Compare|1.20|i686-linux|5.10.0|linux|2.6.26.5|200809271505
2318831|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 34412|linux|2.6.22-1-k7|200809272029
2318871|pass|200809|rick@jrmyers.net ("Rick Myers")|Number-Phone-UK-DetailedLocations|1.3|i686-linux|5.10.0|linux|2.6.26.5|200809271519
2318877|pass|200809|rick@jrmyers.net ("Rick Myers")|Statistics-SerialCorrelation|1.1|i686-linux|5.10.0|linux|2.6.26.5|200809271520
2318880|pass|200809|rick@jrmyers.net ("Rick Myers")|Class-CanBeA|1.2|i686-linux|5.10.0|linux|2.6.26.5|200809271520
2318903|pass|200809|rick@jrmyers.net ("Rick Myers")|Sub-WrapPackages|1.2|i686-linux|5.10.0|linux|2.6.26.5|200809271524
2319004|pass|200809|rick@jrmyers.net ("Rick Myers")|Sort-MultipleFields|1.0|i686-linux|5.10.0|linux|2.6.26.5|200809271550
2319010|pass|200809|rick@jrmyers.net ("Rick Myers")|Net-Random|2.0|i686-linux|5.10.0|linux|2.6.26.5|200809271551
2319035|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-STDOUT|1.03|i686-linux|5.10.0|linux|2.6.26.5|200809271557
2319157|pass|200809|rick@jrmyers.net ("Rick Myers")|Games-Dice-Advanced|1.1|i686-linux|5.10.0|linux|2.6.26.5|200809271616
2319483|pass|200809|rick@jrmyers.net ("Rick Myers")|Class-DBI-ClassGenerator|1.01|i686-linux|5.10.0|linux|2.6.26.5|200809271708
2319498|pass|200809|rick@jrmyers.net ("Rick Myers")|File-Overwrite|1.0|i686-linux|5.10.0|linux|2.6.26.5|200809271713
2322337|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 34412|linux|2.6.22-1-k7|200809280900
2323841|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34435|linux|2.6.22-1-k7|200809281313
2324323|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34435|linux|2.6.22-1-k7|200809281350
2325540|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34435|linux|2.6.22-1-k7|200809281607
2325883|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34435|linux|2.6.22-1-k7|200809281411
2326598|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34435|linux|2.6.22-1-k7|200809281754
2327831|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34435|linux|2.6.22-1-k7|200809282012
2335378|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.44|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200809291729
2335380|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.44|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200809291729
2335382|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.44|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200809291729
2335386|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.44|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200809291730
2335389|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.44|i686-linux-64int|5.8.8|linux|2.6.14|200809291730
2335392|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.44|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200809291730
2336854|fail|200809|chris@bingosnet.co.uk|Palm-SMS|0.02|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809291916
2337033|fail|200809|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Devel-CheckOS|1.44|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200809300340
2337154|pass|200809|srezic@cpan.org|Devel-CheckOS|1.44|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200809292051
2337323|fail|200809|rick@jrmyers.net ("Rick Myers")|Bryar|3.0|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200809291512
2337451|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.10.0 patch 34437|linux|2.6.22-1-k7|200809292130
2337521|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 34437|linux|2.6.22-1-k7|200809292123
2337627|pass|200809|rick@jrmyers.net ("Rick Myers")|Data-Compare|1.21|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200809291552
2337779|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0 patch 34437|linux|2.6.22-1-k7|200809292158
2339049|pass|200809|rick@jrmyers.net ("Rick Myers")|Devel-CheckOS|1.43|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200809291830
2339181|pass|200809|chris@bingosnet.co.uk|Devel-CheckOS|1.44|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200809292343
2339185|pass|200809|chris@bingosnet.co.uk|Devel-CheckOS|1.44|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200809292344
2339198|pass|200809|chris@bingosnet.co.uk|Devel-CheckOS|1.44|darwin-thread-multi-64int-2level|5.8.6|darwin|8.10.0|200809292346
2339210|pass|200809|chris@bingosnet.co.uk|Devel-CheckOS|1.44|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200809292348
2339578|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.10.0 patch 34437|linux|2.6.22-1-k7|200809300028
2339645|pass|200809|flemming@ipk-gatersleben.de|Data-Compare|1.21|x86_64-linux|5.8.5|linux|2.6.9-55.0.9.elsmp|200809291527
2339791|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 34437|linux|2.6.22-1-k7|200809300132
2339931|pass|200809|rick@jrmyers.net ("Rick Myers")|Data-Hexdumper|1.4|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200809300003
2340018|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 34437|linux|2.6.22-1-k7|200809300215
2342112|pass|200809|srezic@cpan.org|Devel-CheckOS|1.44|i386-freebsd|5.10.0|freebsd|6.1-release|200809300836
2342658|pass|200809|rick@jrmyers.net ("Rick Myers")|Devel-CheckOS|1.44|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200809300359
2342758|pass|200809|chris@bingosnet.co.uk|XML-Tiny|1.11|darwin-thread-multi-64int-2level|5.8.6|darwin|8.10.0|200809300923
2342820|pass|200809|chris@bingosnet.co.uk|XML-Tiny|1.11|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200809300936
2342845|pass|200809|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckOS|1.44|i86pc-solaris-64int|5.10.0|solaris|2.11|200809301041
2343727|pass|200809|david@cantrell.org.uk|Devel-CheckOS|1.44|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200809301219
2343785|pass|200809|rick@jrmyers.net ("Rick Myers")|Palm-SMS|0.02|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200809300738
2343916|pass|200809|david@cantrell.org.uk|Devel-CheckOS|1.44|IP35-irix-64int|5.8.8|irix|6.5|200809301357
2343943|pass|200809|david@cantrell.org.uk|Devel-CheckOS|1.44|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200809301300
2343970|pass|200809|david@cantrell.org.uk|Devel-CheckOS|1.44|i686-linux|5.8.8|linux|2.4.27-3-686|200809301306
2344168|fail|200809|hohokus@gmail.com|Devel-CheckOS|1.44|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200809300838
2344321|pass|200809|david@cantrell.org.uk|Devel-CheckOS|1.44|i686-linux|5.6.2|linux|2.4.27-3-686|200809301358
2344441|pass|200809|martin.evans@easysoft.com|XML-Tiny|1.11|i686-linux-ld|5.10.0|linux|2.6.24-19-generic|200809301419
2344936|pass|200809|david@cantrell.org.uk|Devel-CheckOS|1.44|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200809300930
2345254|pass|200809|david@cantrell.org.uk|Devel-CheckOS|1.44|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200809301522
2345303|pass|200809|martin.evans@easysoft.com|Devel-CheckOS|1.44|i686-linux-ld|5.10.0|linux|2.6.24-19-generic|200809301631
2345328|pass|200809|bingos@cpan.org|Devel-CheckOS|1.44|i386-netbsd-thread-multi-64int|5.8.8 patch 34439|netbsd|4.0|200809301628
2345369|pass|200809|srezic@cpan.org|Devel-CheckOS|1.44|i386-freebsd|5.6.2|freebsd|6.1-release|200809301737
2345442|pass|200809|g.grigelionis@computer.org|Devel-CheckOS|1.44|x86_64-linux|5.8.8|linux|2.6.5-7.276-smp|200809301745
2345463|pass|200809|g.grigelionis@computer.org|Devel-CheckOS|1.44|x86_64-linux-thread-multi|5.8.3|linux|2.6.4|200809301747
2345489|pass|200809|g.grigelionis@computer.org|Devel-CheckOS|1.44|i386-linux|5.8.0|linux|2.6.9-22.18.bz155725.elsmp|200809301750
2345644|pass|200809|g.grigelionis@computer.org|Devel-CheckOS|1.44|i586-linux-thread-multi|5.8.5|linux|2.6.8.1|200809301820
2345731|pass|200809|g.grigelionis@computer.org|Devel-CheckOS|1.44|i586-linux-thread-multi|5.8.8|linux|2.6.16|200809301848
2345835|pass|200809|bingos@cpan.org|Devel-CheckOS|1.44|i386-netbsd-thread-multi-64int|5.10.0 patch 34437|netbsd|4.0|200809301800
2345902|pass|200809|vpit@cpan.org|Devel-CheckOS|1.44|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200809301921
2346083|pass|200809|rick@jrmyers.net ("Rick Myers")|Devel-CheckOS|1.44|i686-linux|5.10.0|linux|2.6.26.5|200809301352
2346495|pass|200809|srezic@cpan.org|Devel-CheckOS|1.44|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200809302053
2346541|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.44|i686-linux-64int|5.10.0 patch 34437|linux|2.6.22-1-k7|200809302059
2346554|pass|200809|bingos@cpan.org|Devel-CheckOS|1.44|i386-netbsd-thread-multi-64int|5.11.0 patch 34444|netbsd|4.0|200809301957
2347621|pass|200809|bingos@cpan.org|Devel-CheckOS|1.44|i386-freebsd-thread-multi-64int|5.11.0 patch 34448|freebsd|6.2-release|200809301612
2348573|pass|200809|artur+cpan@niif.spb.su (ARTO)|Devel-CheckOS|1.44|i386-linux-thread-multi|5.10.0|linux|2.6.18-92.1.6.el5xen|200809300833
2350694|pass|200810|rick@jrmyers.net ("Rick Myers")|Devel-CheckOS|1.44|i686-linux|5.8.8|linux|2.6.26.5|200810010205
2351580|pass|200810|rick@jrmyers.net ("Rick Myers")|Tie-Hash-Rank|1.0.1|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010457
2351637|pass|200810|rick@jrmyers.net ("Rick Myers")|Tie-Hash-Longest|1.1|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010509
2351789|pass|200810|rick@jrmyers.net ("Rick Myers")|Tie-Scalar-Decay|1.1.1|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010542
2351821|pass|200810|rick@jrmyers.net ("Rick Myers")|CPU-Emulator-Memory|1.1001|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010547
2351822|pass|200810|rick@jrmyers.net ("Rick Myers")|CPU-Emulator-Z80|1.0|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010547
2351852|pass|200810|rick@jrmyers.net ("Rick Myers")|Palm-TreoPhoneCallDB|1.1|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010553
2351853|pass|200810|rick@jrmyers.net ("Rick Myers")|Statistics-ChiSquare|0.5|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010553
2351900|pass|200810|rick@jrmyers.net ("Rick Myers")|NestedMap|1.0|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010600
2351912|pass|200810|rick@jrmyers.net ("Rick Myers")|Data-Transactional|1.02|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010603
2351970|pass|200810|rick@jrmyers.net ("Rick Myers")|File-Find-Rule-Permissions|1.3|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010615
2352031|pass|200810|rick@jrmyers.net ("Rick Myers")|Class-DBI-ClassGenerator|1.0|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010625
2352092|pass|200810|rick@jrmyers.net ("Rick Myers")|Number-Phone|1.6|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010644
2352098|pass|200810|rick@jrmyers.net ("Rick Myers")|Devel-CheckLib|0.5|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010644
2352196|fail|200810|rick@jrmyers.net ("Rick Myers")|Data-Compare|1.20|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010701
2352283|pass|200810|rick@jrmyers.net ("Rick Myers")|Number-Phone-UK-DetailedLocations|1.3|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010718
2352285|pass|200810|rick@jrmyers.net ("Rick Myers")|Statistics-SerialCorrelation|1.1|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010718
2352286|pass|200810|rick@jrmyers.net ("Rick Myers")|Class-CanBeA|1.2|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010718
2352415|pass|200810|rick@jrmyers.net ("Rick Myers")|Sub-WrapPackages|1.2|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010758
2352501|pass|200810|rick@jrmyers.net ("Rick Myers")|Sort-MultipleFields|1.0|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010816
2352502|pass|200810|rick@jrmyers.net ("Rick Myers")|Net-Random|2.0|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010817
2352512|pass|200810|rick@jrmyers.net ("Rick Myers")|Tie-STDOUT|1.03|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010820
2352544|pass|200810|rick@jrmyers.net ("Rick Myers")|Games-Dice-Advanced|1.1|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010829
2352706|pass|200810|srezic@cpan.org|Devel-CheckOS|1.44|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200810011515
2352827|pass|200810|rick@jrmyers.net ("Rick Myers")|Class-DBI-ClassGenerator|1.01|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010943
2352829|pass|200810|rick@jrmyers.net ("Rick Myers")|File-Overwrite|1.0|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010943
2354265|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200810011736
2355010|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200810012054
2355042|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200810012104
2355309|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-freebsd-thread-multi-64int|5.11.0 patch 34451|freebsd|7.0-release|200810012041
2357351|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200810020504
2357354|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200810020505
2357359|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200810020506
2357363|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200810020507
2357367|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200810020507
2357371|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200810020508
2357380|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200810020509
2357386|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200810020510
2357391|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200810020510
2358373|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 34450|linux|2.6.22-1-k7|200810020824
2358649|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 34450|linux|2.6.22-1-k7|200810020842
2358794|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.44|i686-linux-64int|5.8.8 patch 34450|linux|2.6.22-1-k7|200810020757
2359161|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.21.5-smp|200810020905
2359337|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.8 patch 34450|linux|2.6.22-1-k7|200810021013
2361236|pass|200810|rlane10@wi.rr.com|Devel-CheckOS|1.44|i386-linux-thread-multi|5.10.0|linux|2.6.18-92.1.6.el5xen|200810020617
2363266|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 34450|linux|2.6.22-1-k7|200810022037
2363552|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-freebsd-thread-multi-64int|5.10.0 patch 34445|freebsd|7.0-release|200810021809
2363838|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-freebsd-thread-multi-64int|5.8.8 patch 34450|freebsd|7.0-release|200810021830
2364436|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 34450|linux|2.6.22-1-k7|200810022223
2365777|pass|200810|rlane10@wi.rr.com|Devel-CheckOS|1.44|darwin-2level-ld|5.10.0|darwin|8.11.0|200810021301
2366104|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200810030138
2366361|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200810030315
2366661|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200810030522
2366665|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|darwin-thread-multi-64int-2level|5.8.6|darwin|8.10.0|200810030523
2366669|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200810030525
2372448|pass|200810|rick@jrmyers.net ("Rick Myers")|Data-Compare|1.21|i686-linux|5.8.7|linux|2.6.26.5|200810031624
2373001|pass|200810|rick@jrmyers.net ("Rick Myers")|Bryar|3.0|i686-linux|5.8.7|linux|2.6.26.5|200810031758
2373652|pass|200810|rick@jrmyers.net ("Rick Myers")|Data-Hexdumper|1.4|i686-linux|5.8.7|linux|2.6.26.5|200810040007
2373731|pass|200810|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.10.0 patch 34445|freebsd|7.0-release|200810040024
2373780|pass|200810|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.8.8 patch 34450|freebsd|7.0-release|200810040031
2374313|pass|200810|rick@jrmyers.net ("Rick Myers")|Devel-CheckOS|1.44|i686-linux|5.8.7|linux|2.6.26.5|200810040228
2374876|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34456|openbsd|4.2|200810040217
2378123|pass|200810|rick@jrmyers.net ("Rick Myers")|Data-Compare|1.21|i686-linux|5.8.6|linux|2.6.26.5|200810040907
2379530|pass|200810|rick@jrmyers.net ("Rick Myers")|Bryar|3.0|i686-linux|5.8.6|linux|2.6.26.5|200810041330
2380687|pass|200810|rick@jrmyers.net ("Rick Myers")|Data-Hexdumper|1.4|i686-linux|5.8.6|linux|2.6.26.5|200810041718
2381463|pass|200810|rick@jrmyers.net ("Rick Myers")|Data-Compare|1.21|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810041930
2382914|pass|200810|rick@jrmyers.net ("Rick Myers")|Palm-SMS|0.02|i686-linux|5.8.6|linux|2.6.26.5|200810060430
2384752|pass|200810|rick@jrmyers.net ("Rick Myers")|Palm-SMS|0.02|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810061018
2386783|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200810062050
2386863|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200810062112
2387692|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.44|i686-linux-64int|5.8.8 patch 34468|linux|2.6.22-1-k7|200810062332
2387762|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 34468|linux|2.6.22-1-k7|200810062317
2388111|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.8 patch 34468|linux|2.6.22-1-k7|200810070108
2389245|pass|200810|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.44|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200810071033
2389583|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 34468|linux|2.6.22-1-k7|200810070515
2390588|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 34468|linux|2.6.22-1-k7|200810070830
2392379|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 34468|linux|2.6.22-1-k7|200810071232
2392460|pass|200810|rick@jrmyers.net ("Rick Myers")|Number-Phone|1.6|i686-linux|5.8.7|linux|2.6.26.5|200810070640
2392462|pass|200810|rick@jrmyers.net ("Rick Myers")|Number-Phone-UK-DetailedLocations|1.3|i686-linux|5.8.7|linux|2.6.26.5|200810070641
2392480|pass|200810|rick@jrmyers.net ("Rick Myers")|Statistics-SerialCorrelation|1.1|i686-linux|5.8.7|linux|2.6.26.5|200810070643
2392485|pass|200810|rick@jrmyers.net ("Rick Myers")|Class-CanBeA|1.2|i686-linux|5.8.7|linux|2.6.26.5|200810070643
2392700|pass|200810|rick@jrmyers.net ("Rick Myers")|Tie-Hash-Rank|1.0.1|i686-linux|5.8.7|linux|2.6.26.5|200810070706
2393059|pass|200810|rick@jrmyers.net ("Rick Myers")|Sub-WrapPackages|1.2|i686-linux|5.8.7|linux|2.6.26.5|200810070757
2393415|pass|200810|rick@jrmyers.net ("Rick Myers")|Tie-Hash-Longest|1.1|i686-linux|5.8.7|linux|2.6.26.5|200810070845
2394310|pass|200810|rick@jrmyers.net ("Rick Myers")|Tie-Scalar-Decay|1.1.1|i686-linux|5.8.7|linux|2.6.26.5|200810071113
2394327|pass|200810|rick@jrmyers.net ("Rick Myers")|Sort-MultipleFields|1.0|i686-linux|5.8.7|linux|2.6.26.5|200810071115
2394350|pass|200810|rick@jrmyers.net ("Rick Myers")|Net-Random|2.0|i686-linux|5.8.7|linux|2.6.26.5|200810071118
2394559|pass|200810|rick@jrmyers.net ("Rick Myers")|Tie-STDOUT|1.03|i686-linux|5.8.7|linux|2.6.26.5|200810071146
2394604|pass|200810|rick@jrmyers.net ("Rick Myers")|CPU-Emulator-Memory|1.1001|i686-linux|5.8.7|linux|2.6.26.5|200810071150
2394616|pass|200810|rick@jrmyers.net ("Rick Myers")|CPU-Emulator-Z80|1.0|i686-linux|5.8.7|linux|2.6.26.5|200810071151
2395006|pass|200810|rick@jrmyers.net ("Rick Myers")|Statistics-ChiSquare|0.5|i686-linux|5.8.7|linux|2.6.26.5|200810071238
2395145|pass|200810|rick@jrmyers.net ("Rick Myers")|Games-Dice-Advanced|1.1|i686-linux|5.8.7|linux|2.6.26.5|200810071250
2395438|pass|200810|rick@jrmyers.net ("Rick Myers")|NestedMap|1.0|i686-linux|5.8.7|linux|2.6.26.5|200810071326
2395613|pass|200810|bingos@cpan.org|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-19-generic|200810071843
2395781|pass|200810|rick@jrmyers.net ("Rick Myers")|Data-Transactional|1.02|i686-linux|5.8.7|linux|2.6.26.5|200810071408
2396275|pass|200810|rick@jrmyers.net ("Rick Myers")|File-Find-Rule-Permissions|1.3|i686-linux|5.8.7|linux|2.6.26.5|200810071516
2397251|pass|200810|rick@jrmyers.net ("Rick Myers")|Class-DBI-ClassGenerator|1.01|i686-linux|5.8.7|linux|2.6.26.5|200810071712
2397252|pass|200810|rick@jrmyers.net ("Rick Myers")|Number-Phone|1.6|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810071712
2397253|pass|200810|rick@jrmyers.net ("Rick Myers")|Number-Phone-UK-DetailedLocations|1.3|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810071712
2397323|pass|200810|rick@jrmyers.net ("Rick Myers")|File-Overwrite|1.0|i686-linux|5.8.7|linux|2.6.26.5|200810071721
2397374|pass|200810|rick@jrmyers.net ("Rick Myers")|Statistics-SerialCorrelation|1.1|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810071726
2397382|pass|200810|rick@jrmyers.net ("Rick Myers")|Class-CanBeA|1.2|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810071728
2397824|pass|200810|rick@jrmyers.net ("Rick Myers")|Devel-CheckLib|0.5|i686-linux|5.8.7|linux|2.6.26.5|200810071837
2397861|pass|200810|rick@jrmyers.net ("Rick Myers")|Tie-Hash-Rank|1.0.1|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810071841
2398034|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-19-generic|200810080002
2398537|pass|200810|rick@jrmyers.net ("Rick Myers")|Sub-WrapPackages|1.2|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810080002
2398624|pass|200810|bingos@cpan.org|Acme-Licence|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-19-generic|200810080112
2398627|pass|200810|bingos@cpan.org|Acme-Licence|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-19-generic|200810080112
2399199|pass|200810|bingos@cpan.org|Acme-Pony|1.1.2|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-19-generic|200810080220
2399200|pass|200810|bingos@cpan.org|Acme-Pony|1.1.2|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-19-generic|200810080220
2399215|pass|200810|rick@jrmyers.net ("Rick Myers")|Tie-Hash-Longest|1.1|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810080123
2399310|pass|200810|bingos@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-19-generic|200810080240
2399311|pass|200810|bingos@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-19-generic|200810080240
2399487|pass|200810|rick@jrmyers.net ("Rick Myers")|Number-Phone|1.6|i686-linux|5.8.6|linux|2.6.26.5|200810080203
2399488|pass|200810|rick@jrmyers.net ("Rick Myers")|Number-Phone-UK-DetailedLocations|1.3|i686-linux|5.8.6|linux|2.6.26.5|200810080204
2399545|pass|200810|rick@jrmyers.net ("Rick Myers")|Statistics-SerialCorrelation|1.1|i686-linux|5.8.6|linux|2.6.26.5|200810080216
2399550|pass|200810|rick@jrmyers.net ("Rick Myers")|Class-CanBeA|1.2|i686-linux|5.8.6|linux|2.6.26.5|200810080217
2399570|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-19-generic|200810080323
2400188|pass|200810|rick@jrmyers.net ("Rick Myers")|Tie-Scalar-Decay|1.1.1|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810080354
2400201|pass|200810|rick@jrmyers.net ("Rick Myers")|Sort-MultipleFields|1.0|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810080355
2400205|pass|200810|rick@jrmyers.net ("Rick Myers")|Net-Random|2.0|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810080357
2400344|pass|200810|rick@jrmyers.net ("Rick Myers")|Tie-STDOUT|1.03|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810080020
2400375|pass|200810|rick@jrmyers.net ("Rick Myers")|CPU-Emulator-Memory|1.1001|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810080024
2400381|pass|200810|rick@jrmyers.net ("Rick Myers")|CPU-Emulator-Z80|1.0|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810080025
2400672|pass|200810|rick@jrmyers.net ("Rick Myers")|Tie-Hash-Rank|1.0.1|i686-linux|5.8.6|linux|2.6.26.5|200810080103
2400758|pass|200810|rick@jrmyers.net ("Rick Myers")|Statistics-ChiSquare|0.5|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810080117
2400962|pass|200810|rick@jrmyers.net ("Rick Myers")|Games-Dice-Advanced|1.1|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810080145
2401241|pass|200810|rick@jrmyers.net ("Rick Myers")|NestedMap|1.0|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810080229
2401256|pass|200810|rick@jrmyers.net ("Rick Myers")|Sub-WrapPackages|1.2|i686-linux|5.8.6|linux|2.6.26.5|200810080232
2401425|pass|200810|rick@jrmyers.net ("Rick Myers")|Data-Transactional|1.02|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810080304
2401687|pass|200810|rick@jrmyers.net ("Rick Myers")|Tie-Hash-Longest|1.1|i686-linux|5.8.6|linux|2.6.26.5|200810080346
2403309|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.44|i686-linux-64int|5.10.0 patch 34465|linux|2.6.22-1-k7|200810081519
2403986|pass|200810|rick@jrmyers.net ("Rick Myers")|Tie-Scalar-Decay|1.1.1|i686-linux|5.8.6|linux|2.6.26.5|200810081127
2404020|pass|200810|rick@jrmyers.net ("Rick Myers")|Sort-MultipleFields|1.0|i686-linux|5.8.6|linux|2.6.26.5|200810081130
2404034|pass|200810|rick@jrmyers.net ("Rick Myers")|Net-Random|2.0|i686-linux|5.8.6|linux|2.6.26.5|200810081132
2404177|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0 patch 34465|linux|2.6.22-1-k7|200810081739
2404347|pass|200810|rick@jrmyers.net ("Rick Myers")|Tie-STDOUT|1.03|i686-linux|5.8.6|linux|2.6.26.5|200810081205
2404417|pass|200810|rick@jrmyers.net ("Rick Myers")|CPU-Emulator-Memory|1.1001|i686-linux|5.8.6|linux|2.6.26.5|200810081212
2404428|pass|200810|rick@jrmyers.net ("Rick Myers")|CPU-Emulator-Z80|1.0|i686-linux|5.8.6|linux|2.6.26.5|200810081213
2404578|pass|200810|bingos@cpan.org|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-19-generic|200810081730
2404816|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 34465|linux|2.6.22-1-k7|200810081854
2404994|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.10.0 patch 34465|linux|2.6.22-1-k7|200810081912
2405186|pass|200810|bingos@cpan.org|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-19-generic|200810081836
2405256|pass|200810|rick@jrmyers.net ("Rick Myers")|Statistics-ChiSquare|0.5|i686-linux|5.8.6|linux|2.6.26.5|200810081345
2405326|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 34465|linux|2.6.22-1-k7|200810081953
2405457|pass|200810|bingos@cpan.org|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-19-generic|200810081909
2405496|pass|200810|rick@jrmyers.net ("Rick Myers")|Games-Dice-Advanced|1.1|i686-linux|5.8.6|linux|2.6.26.5|200810081413
2405826|pass|200810|rick@jrmyers.net ("Rick Myers")|NestedMap|1.0|i686-linux|5.8.6|linux|2.6.26.5|200810081450
2406205|pass|200810|rick@jrmyers.net ("Rick Myers")|Data-Transactional|1.02|i686-linux|5.8.6|linux|2.6.26.5|200810081530
2406514|pass|200810|rick@jrmyers.net ("Rick Myers")|File-Find-Rule-Permissions|1.3|i686-linux|5.8.6|linux|2.6.26.5|200810081629
2406680|pass|200810|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-19-generic|200810082151
2406981|pass|200810|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-19-generic|200810082229
2407181|pass|200810|rick@jrmyers.net ("Rick Myers")|Class-DBI-ClassGenerator|1.01|i686-linux|5.8.6|linux|2.6.26.5|200810081801
2407241|pass|200810|rick@jrmyers.net ("Rick Myers")|File-Overwrite|1.0|i686-linux|5.8.6|linux|2.6.26.5|200810081808
2407740|pass|200810|rick@jrmyers.net ("Rick Myers")|Devel-CheckLib|0.5|i686-linux|5.8.6|linux|2.6.26.5|200810081940
2408656|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 34465|linux|2.6.22-1-k7|200810090900
2408741|pass|200810|rick@jrmyers.net ("Rick Myers")|File-Find-Rule-Permissions|1.3|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810090314
2410831|pass|200810|rick@jrmyers.net ("Rick Myers")|Devel-CheckOS|1.44|i686-linux|5.10.0|linux|2.6.26.6|200810091027
2411206|pass|200810|rick@jrmyers.net ("Rick Myers")|Devel-CheckOS|1.44|i686-linux|5.8.8|linux|2.6.26.6|200810091126
2411451|pass|200810|rick@jrmyers.net ("Rick Myers")|Devel-CheckOS|1.44|i686-linux|5.8.7|linux|2.6.26.6|200810091200
2411847|pass|200810|rick@jrmyers.net ("Rick Myers")|Devel-CheckOS|1.44|i686-linux|5.8.6|linux|2.6.26.6|200810091249
2412109|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200810091157
2415526|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200810100234
2415612|pass|200810|rafl@debian.org|Devel-CheckOS|1.44|i686-linux-multi|5.10.0|linux|2.6.24-rc5|200810100449
2416834|pass|200810|srezic@cpan.org|Devel-CheckOS|1.44|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200810101125
2418723|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200810102248
2418784|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.26.3-29.fc9.i686|200810102304
2419263|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.26.3-29.fc9.i686|200810110123
2421100|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200810111405
2422180|pass|200810|bobtfish@bobtfish.net|XML-Tiny|1.11|darwin-thread-multi-2level|5.8.6|darwin|8.0|200810112343
2423591|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200810121816
2423624|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200810121825
2426648|pass|200810|andy@hexten.net|Devel-CheckOS|1.44|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200810131955
2428930|pass|200810|srezic@cpan.org|Number-Phone|1.6|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810140103
2429590|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200810140051
2430265|pass|200810|srezic@cpan.org|CPU-Emulator-Memory|1.1001|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810140047
2430274|pass|200810|srezic@cpan.org|Data-Compare|1.21|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810140048
2430283|pass|200810|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810140049
2430375|pass|200810|srezic@cpan.org|CPU-Emulator-Memory|1.1001|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810140054
2430749|pass|200810|srezic@cpan.org|Palm-Treo680MessagesDB|1.0|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810140113
2431812|pass|200810|srezic@cpan.org|Number-Phone|1.6|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810140236
2432263|pass|200810|srezic@cpan.org|Palm-Treo680MessagesDB|1.0|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810140315
2433004|pass|200810|srezic@cpan.org|Data-Transactional|1.02|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810140422
2433208|pass|200810|srezic@cpan.org|Sort-MultipleFields|1.0|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810140451
2433436|pass|200810|srezic@cpan.org|XML-Tiny|1.11|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810140518
2433775|pass|200810|srezic@cpan.org|Data-Hexdumper|1.4|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810140552
2433845|pass|200810|srezic@cpan.org|Data-Transactional|1.02|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810140601
2434066|pass|200810|srezic@cpan.org|Devel-CheckOS|1.44|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810140623
2434099|pass|200810|srezic@cpan.org|Sort-MultipleFields|1.0|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810140626
2434685|pass|200810|srezic@cpan.org|CPU-Emulator-Z80|1.0|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810140745
2434782|pass|200810|srezic@cpan.org|XML-Tiny|1.11|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810140000
2435083|pass|200810|srezic@cpan.org|Data-Compare|1.21|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810140831
2435324|pass|200810|srezic@cpan.org|Devel-CheckLib|0.5|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810140906
2435512|pass|200810|srezic@cpan.org|Devel-CheckLib|0.5|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810140948
2435778|pass|200810|srezic@cpan.org|Data-Hexdumper|1.4|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810141032
2435795|pass|200810|srezic@cpan.org|Devel-CheckOS|1.44|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810141034
2437103|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0.1|200810141734
2437120|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0.1|200810141738
2438337|pass|200810|srezic@cpan.org|Tie-STDOUT|1.03|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810150855
2438472|pass|200810|srezic@cpan.org|Tie-STDOUT|1.03|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810150851
2438808|pass|200810|srezic@cpan.org|File-Find-Rule-Permissions|1.3|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810150854
2439360|pass|200810|srezic@cpan.org|CPAN-FindDependencies|2.0|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810151026
2439526|pass|200810|srezic@cpan.org|File-Find-Rule-Permissions|1.3|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810150851
2439551|pass|200810|srezic@cpan.org|Bryar|3.0|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810151054
2439931|pass|200810|srezic@cpan.org|Bryar|3.0|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810151111
2440340|pass|200810|srezic@cpan.org|CPAN-FindDependencies|2.0|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810151129
2441130|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34482|linux|2.6.22-1-k7|200810151701
2441253|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34482|linux|2.6.22-1-k7|200810151730
2441288|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.44|i686-linux-64int|5.11.0 patch 34482|linux|2.6.22-1-k7|200810151737
2441374|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34482|linux|2.6.22-1-k7|200810151755
2441749|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34482|linux|2.6.22-1-k7|200810151724
2442108|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200810151925
2442170|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200810151943
2442624|pass|200810|srezic@cpan.org|Palm-TreoPhoneCallDB|1.1|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810152056
2442929|pass|200810|srezic@cpan.org|Palm-TreoPhoneCallDB|1.1|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810152045
2443542|pass|200810|srezic@cpan.org|Net-Random|2.0|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810152301
2443632|pass|200810|srezic@cpan.org|Net-Random|2.0|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810152300
2445048|fail|200810|mthurn@cpan.org|Devel-CheckOS|1.44|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200810160055
2446197|fail|200810|alexchorny@gmail.com|Devel-CheckOS|1.44|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200810160657
2448136|pass|200810|srezic@cpan.org|Number-Phone-UK-DetailedLocations|1.3|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810162113
2449165|pass|200810|andy@hexten.net|Data-Hexdumper|1.4|darwin-thread-multi-2level|5.10.0|darwin|9.2.2|200810170121
2451647|pass|200810|srezic@cpan.org|Games-Dice-Advanced|1.1|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810172339
2451763|pass|200810|srezic@cpan.org|Sub-WrapPackages|1.2|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810172340
2453135|pass|200810|srezic@cpan.org|Games-Dice-Advanced|1.1|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810180826
2453261|pass|200810|srezic@cpan.org|Sub-WrapPackages|1.2|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810180826
2454735|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200810181713
2454809|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200810181733
2457101|pass|200810|stro@cpan.org|Devel-CheckOS|1.44|i686-linux|5.8.8|linux|2.4.27-2-386|200810190906
2457513|pass|200810|srezic@cpan.org|Class-CanBeA|1.2|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810191252
2457553|pass|200810|srezic@cpan.org|File-Overwrite|1.0|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810191252
2458962|pass|200810|srezic@cpan.org|File-Overwrite|1.0|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810191305
2459015|pass|200810|srezic@cpan.org|Class-CanBeA|1.2|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810191305
2463982|pass|200810|srezic@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810210000
2464503|pass|200810|srezic@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810210004
2464860|fail|200810|srezic@cpan.org|Palm-SMS|0.02|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810210001
2468226|pass|200810|davidp@preshweb.co.uk ("David Precious")|Acme-Scurvy-Whoreson-BilgeRat|1.1|i486-linux-gnu|5.10.0|linux|2.6.26.1|200810211951
2469182|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.44|i686-linux-64int|5.11.0 patch 34555|linux|2.6.22-1-k7|200810220802
2469824|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34555|linux|2.6.22-1-k7|200810220953
2470404|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34555|linux|2.6.22-1-k7|200810221158
2470612|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34555|linux|2.6.22-1-k7|200810221236
2470971|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34555|linux|2.6.22-1-k7|200810221344
2473021|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.45|i686-linux-64int|5.10.0 patch 34437|linux|2.6.22-1-k7|200810222203
2473031|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.45|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200810222204
2473033|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.45|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200810222205
2473060|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.45|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200810222159
2473064|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.45|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200810222203
2473065|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.45|i686-linux-64int|5.8.8 patch 34468|linux|2.6.22-1-k7|200810222204
2473143|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34555|linux|2.6.22-1-k7|200810222219
2473302|pass|200810|g.grigelionis@computer.org|Devel-CheckOS|1.45|x86_64-linux|5.8.8|linux|2.6.5-7.276-smp|200810222304
2473339|pass|200810|g.grigelionis@computer.org|Devel-CheckOS|1.45|x86_64-linux-thread-multi|5.8.3|linux|2.6.4|200810222312
2473532|pass|200810|g.grigelionis@computer.org|Devel-CheckOS|1.45|i386-linux|5.8.0|linux|2.6.9-22.18.bz155725.elsmp|200810222320
2474135|pass|200810|srezic@cpan.org|Statistics-SerialCorrelation|1.1|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810222317
2474805|pass|200810|srezic@cpan.org|Statistics-ChiSquare|0.5|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810222317
2475248|fail|200810|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Devel-CheckOS|1.45|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200810230753
2475451|pass|200810|srezic@cpan.org|Statistics-SerialCorrelation|1.1|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810230042
2475879|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|i386-dragonfly-thread-multi-64int|5.10.0|dragonfly|2.0.0-release|200810230118
2476257|pass|200810|bingos@cpan.org|Data-Compare|1.21|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200810230102
2476350|pass|200810|bingos@cpan.org|Data-Compare|1.21|darwin-thread-multi-64int-2level|5.8.6|darwin|8.10.0|200810230106
2476362|pass|200810|srezic@cpan.org|Tie-Hash-Longest|1.1|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810230007
2476412|pass|200810|bingos@cpan.org|Data-Compare|1.21|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200810230109
2476585|pass|200810|srezic@cpan.org|Statistics-ChiSquare|0.5|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810230042
2476637|pass|200810|srezic@cpan.org|NestedMap|1.0|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810230040
2476647|pass|200810|srezic@cpan.org|Tie-Hash-Longest|1.1|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810230042
2476744|pass|200810|srezic@cpan.org|NestedMap|1.0|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810230030
2476833|pass|200810|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200810230207
2476865|pass|200810|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200810230218
2476895|pass|200810|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200810230222
2477132|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200810230449
2477135|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|darwin-thread-multi-64int-2level|5.8.6|darwin|8.10.0|200810230450
2477139|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200810230451
2477330|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200810230626
2477361|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200810230635
2477363|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200810230636
2477713|pass|200810|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.45|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200810231631
2477945|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.45|i686-linux-64int|5.8.8 patch 34560|linux|2.6.22-1-k7|200810231041
2478227|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.8 patch 34560|linux|2.6.22-1-k7|200810231107
2478379|pass|200810|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckOS|1.45|i86pc-solaris-64int|5.10.0|solaris|2.11|200810231248
2478472|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200810231132
2478478|pass|200810|david@cantrell.org.uk|Devel-CheckOS|1.45|i686-linux|5.6.2|linux|2.4.27-3-686|200810231222
2478516|pass|200810|srezic@cpan.org|Devel-CheckOS|1.45|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200810231244
2478663|pass|200810|srezic@cpan.org|Devel-CheckOS|1.45|i386-freebsd|5.10.0|freebsd|6.1-release|200810231414
2478685|pass|200810|srezic@cpan.org|Devel-CheckOS|1.45|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200810231416
2478847|fail|200810|hohokus@gmail.com|Devel-CheckOS|1.45|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200810230941
2479494|pass|200810|david@cantrell.org.uk|Devel-CheckOS|1.45|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200810231726
2479548|pass|200810|david@cantrell.org.uk|Devel-CheckOS|1.45|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200810231842
2480107|pass|200810|srezic@cpan.org|Devel-CheckOS|1.45|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810232057
2480933|pass|200810|srezic@cpan.org|Data-Compare|1.21|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810232151
2480989|pass|200810|srezic@cpan.org|Devel-CheckOS|1.44|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810232057
2481074|pass|200810|david@cantrell.org.uk|Devel-CheckOS|1.45|i686-linux|5.8.8|linux|2.4.27-3-686|200810232130
2481082|pass|200810|srezic@cpan.org|Devel-CheckOS|1.45|i386-freebsd|5.6.2|freebsd|6.1-release|200810232229
2481358|pass|200810|srezic@cpan.org|XML-Tiny|1.11|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810232100
2481577|pass|200810|david@cantrell.org.uk|Devel-CheckOS|1.45|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200810232239
2481785|pass|200810|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810240003
2482494|pass|200810|srezic@cpan.org|Sort-MultipleFields|1.0|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810240123
2482565|pass|200810|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200810240001
2483115|pass|200810|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200810240215
2483233|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 34560|linux|2.6.22-1-k7|200810240655
2484095|pass|200810|srezic@cpan.org|Palm-Treo680MessagesDB|1.0|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810240824
2484114|pass|200810|srezic@cpan.org|Data-Hexdumper|1.4|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810240822
2484151|pass|200810|srezic@cpan.org|CPU-Emulator-Memory|1.1001|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810240822
2484233|pass|200810|srezic@cpan.org|Data-Transactional|1.02|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810240822
2484461|pass|200810|srezic@cpan.org|CPU-Emulator-Z80|1.0|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810240822
2484591|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 34560|linux|2.6.22-1-k7|200810240947
2485089|pass|200810|srezic@cpan.org|Number-Phone|1.6|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810240958
2485503|pass|200810|david@cantrell.org.uk|Devel-CheckOS|1.45|IP35-irix-64int|5.8.8|irix|6.5|200810241215
2486019|pass|200810|g.grigelionis@computer.org|Devel-CheckOS|1.45|i586-linux-thread-multi|5.8.8|linux|2.6.16|200810241428
2486095|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 34560|linux|2.6.22-1-k7|200810241442
2486113|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 34560|linux|2.6.22-1-k7|200810241439
2487352|pass|200810|david@cantrell.org.uk|Devel-CheckOS|1.45|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200810241400
2488319|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.45|i686-linux-64int|5.10.0 patch 34563|linux|2.6.22-1-k7|200810250250
2488744|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 34563|linux|2.6.22-1-k7|200810250758
2490042|pass|200810|bingos@cpan.org|Data-Compare|1.21|MirBSD.i386-mirbsd-thread-multi-64int|5.8.8|mirbsd|10|200810251236
2490325|pass|200810|srezic@cpan.org|CPAN-FindDependencies|2.0|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810251455
2490399|pass|200810|srezic@cpan.org|Devel-CheckLib|0.5|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810251455
2490625|pass|200810|srezic@cpan.org|Bryar|3.0|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810251454
2491173|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.10.0 patch 34563|linux|2.6.22-1-k7|200810251546
2492058|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 34563|linux|2.6.22-1-k7|200810251638
2492134|pass|200810|stro@cpan.org|Devel-CheckOS|1.45|i686-linux|5.8.8|linux|2.4.27-2-386|200810251741
2492206|pass|200810|srezic@cpan.org|File-Find-Rule-Permissions|1.3|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810251540
2492486|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200810251609
2492844|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 34563|linux|2.6.22-1-k7|200810251819
2492857|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200810251722
2494614|pass|200810|srezic@cpan.org|Number-Phone-UK-DetailedLocations|1.3|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810252226
2494749|pass|200810|srezic@cpan.org|Games-Dice-Advanced|1.1|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810252223
2495081|pass|200810|srezic@cpan.org|Net-Random|2.0|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810252225
2495509|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|MirBSD.i386-mirbsd-thread-multi-64int|5.10.0|mirbsd|10|200810252129
2495552|pass|200810|srezic@cpan.org|Palm-TreoPhoneCallDB|1.1|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810252226
2496066|pass|200810|srezic@cpan.org|Sub-WrapPackages|1.2|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810252228
2496163|pass|200810|srezic@cpan.org|Tie-STDOUT|1.03|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810252229
2496323|pass|200810|srezic@cpan.org|Class-CanBeA|1.2|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810260009
2496562|pass|200810|srezic@cpan.org|File-Overwrite|1.0|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810260009
2496914|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0 patch 34563|linux|2.6.22-1-k7|200810260107
2497613|pass|200810|bingos@cpan.org|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200810260509
2497640|pass|200810|bingos@cpan.org|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200810260550
2498986|pass|200810|rafl@debian.org|Devel-CheckOS|1.45|i686-linux-multi|5.10.0|linux|2.6.24-rc5|200810261508
2499251|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200810261651
2499275|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200810261639
2499297|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200810261714
2499318|pass|200810|g.grigelionis@computer.org|Devel-CheckOS|1.45|i586-linux-thread-multi|5.8.5|linux|2.6.8.1|200810261821
2499468|pass|200810|srezic@cpan.org|Devel-CheckOS|1.45|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810261939
2500305|pass|200810|srezic@cpan.org|Palm-SMS|0.02|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810262156
2500744|pass|200810|srezic@cpan.org|Tie-Hash-Longest|1.1|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810262157
2501284|pass|200810|srezic@cpan.org|NestedMap|1.0|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810262302
2501355|pass|200810|srezic@cpan.org|Statistics-ChiSquare|0.5|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810262157
2501608|pass|200810|srezic@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810262151
2501723|pass|200810|srezic@cpan.org|Statistics-SerialCorrelation|1.1|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810262157
2502687|pass|200810|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200810270422
2502805|pass|200810|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200810270652
2502865|pass|200810|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200810270725
2503585|pass|200810|autarch@urth.org|Devel-CheckOS|1.45|x86_64-linux-gnu|5.10.0|linux|2.6.24-15-server|200810270459
2504009|pass|200810|stro@cpan.org|Devel-CheckOS|1.45|i686-linux|5.10.0|linux|2.4.27-2-386|200810270430
2504185|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|i386-freebsd-thread-multi-64int|5.6.2|freebsd|7.0-release|200810271210
2504471|pass|200810|JOST@cpan.org ("Josts Smokehouse")|Data-Compare|1.21|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200810271551
2504937|pass|200810|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckOS|1.45|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200810271734
2505236|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200810271551
2505240|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200810271740
2505308|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200810271749
2505629|pass|200810|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.6.2|freebsd|7.0-release|200810271709
2505988|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.46|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200810272139
2506062|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.46|i686-linux-64int|5.10.0 patch 34437|linux|2.6.22-1-k7|200810272155
2506063|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.46|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200810272156
2506064|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.46|i686-linux-64int|5.8.8 patch 34468|linux|2.6.22-1-k7|200810272157
2506067|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.46|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200810272158
2506068|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.46|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200810272158
2506737|pass|200810|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Devel-CheckOS|1.46|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200810280933
2507019|pass|200810|JOST@cpan.org ("Josts Smokehouse")|XML-Tiny|1.11|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200810280325
2507269|pass|200810|gabor@pti.co.il ("Gabor Szabo")|Devel-CheckOS|1.45|i486-linux-gnu|5.8.8|linux|2.6.15.7|200810280542
2507534|pass|200810|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200810280300
2507692|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200810280606
2507696|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|darwin-thread-multi-64int-2level|5.8.6|darwin|8.10.0|200810280608
2507698|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200810280609
2508250|pass|200810|srezic@cpan.org|Tie-Hash-Rank|1.0.1|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810280801
2508475|pass|200810|srezic@cpan.org|Tie-Scalar-Decay|1.1.1|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810280801
2508788|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200810280809
2508796|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200810280818
2508800|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200810280819
2508952|pass|200810|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.46|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200810281653
2509020|pass|200810|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckOS|1.46|i86pc-solaris-64int|5.10.0|solaris|2.11|200810281028
2509286|pass|200810|bingos@cpan.org|Data-Compare|1.21|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200810281129
2509343|pass|200810|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckOS|1.46|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200810281250
2509570|pass|200810|bingos@cpan.org|Data-Compare|1.21|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200810281247
2509878|pass|200810|JOST@cpan.org ("Josts Smokehouse")|Class-DBI-ClassGenerator|1.01|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200810281524
2509990|pass|200810|JOST@cpan.org ("Josts Smokehouse")|XML-DoubleEncodedEntities|1.0|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200810281554
2510613|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200810281821
2510615|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200810281822
2510625|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200810281825
2510628|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200810281827
2510832|pass|200810|g.grigelionis@computer.org|Devel-CheckOS|1.46|i586-linux-thread-multi|5.8.8|linux|2.6.16|200810282005
2510919|pass|200810|g.grigelionis@computer.org|Devel-CheckOS|1.46|x86_64-linux|5.8.8|linux|2.6.5-7.276-smp|200810282017
2510931|pass|200810|g.grigelionis@computer.org|Devel-CheckOS|1.46|x86_64-linux-thread-multi|5.8.3|linux|2.6.4|200810282020
2511490|pass|200810|JOST@cpan.org ("Josts Smokehouse")|Sort-MultipleFields|1.0|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200810282313
2511798|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34613|linux|2.6.22-1-k7|200810282356
2512110|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.46|i686-linux-64int|5.11.0 patch 34613|linux|2.6.22-1-k7|200810290105
2512344|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34613|linux|2.6.22-1-k7|200810290145
2512788|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200810290214
2512798|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200810290216
2513012|pass|200810|JOST@cpan.org ("Josts Smokehouse")|Data-Hexdumper|1.4|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200810290401
2513016|pass|200810|JOST@cpan.org ("Josts Smokehouse")|Palm-Treo680MessagesDB|1.0|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200810290401
2513133|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34613|linux|2.6.22-1-k7|200810290409
2513793|pass|200810|bingos@cpan.org|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200810290519
2514498|pass|200810|bingos@cpan.org|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200810290725
2514504|pass|200810|JOST@cpan.org ("Josts Smokehouse")|Data-Transactional|1.02|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200810290834
2514752|pass|200810|JOST@cpan.org ("Josts Smokehouse")|CPU-Emulator-Memory|1.1001|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200810290912
2514753|pass|200810|JOST@cpan.org ("Josts Smokehouse")|CPU-Emulator-Z80|1.0|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200810290912
2514837|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34613|linux|2.6.22-1-k7|200810290901
2515288|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34613|linux|2.6.22-1-k7|200810291044
2516057|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 34633|linux|2.6.22-1-k7|200810291518
2516371|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.46|i686-linux-64int|5.8.8 patch 34633|linux|2.6.22-1-k7|200810291556
2516446|pass|200810|bingos@cpan.org|Data-Compare|1.21|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200810291504
2516478|pass|200810|JOST@cpan.org ("Josts Smokehouse")|Number-Phone|1.6|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200810291616
2516753|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 34633|linux|2.6.22-1-k7|200810291637
2516933|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24.5-smp|200810291659
2516935|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24.5-smp|200810291700
2517027|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.8 patch 34633|linux|2.6.22-1-k7|200810291722
2517459|pass|200810|david@cantrell.org.uk|Devel-CheckOS|1.46|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200810291723
2517605|pass|200810|david@cantrell.org.uk|Devel-CheckOS|1.46|i686-linux|5.8.8|linux|2.4.27-3-686|200810291747
2517662|pass|200810|david@cantrell.org.uk|Devel-CheckOS|1.46|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200810291800
2517664|pass|200810|david@cantrell.org.uk|Devel-CheckOS|1.46|IP35-irix-64int|5.8.8|irix|6.5|200810291900
2517712|pass|200810|david@cantrell.org.uk|Devel-CheckOS|1.46|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200810291315
2517761|pass|200810|david@cantrell.org.uk|Devel-CheckOS|1.46|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200810291837
2517906|pass|200810|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24.5-smp|200810292043
2518027|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24.5-smp|200810292138
2518031|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24.5-smp|200810292138
2518078|pass|200810|david@cantrell.org.uk|Devel-CheckOS|1.46|i686-linux|5.6.2|linux|2.4.27-3-686|200810292107
2518110|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24.5-smp|200810292217
2518115|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24.5-smp|200810292218
2518295|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-64int|5.10.0 patch 34437|linux|2.6.22-1-k7|200810292333
2518304|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-64int|5.8.8 patch 34468|linux|2.6.22-1-k7|200810292335
2518625|pass|200810|srezic@cpan.org|Devel-CheckOS|1.46|i386-freebsd|5.10.0|freebsd|6.1-release|200810300137
2519361|pass|200810|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200810300654
2519368|pass|200810|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|darwin-thread-multi-64int-2level|5.8.6|darwin|8.10.0|200810300657
2519379|pass|200810|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200810300700
2519407|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 34633|linux|2.6.22-1-k7|200810300804
2519534|pass|200810|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24.5-smp|200810300822
2520307|pass|200810|srezic@cpan.org|Devel-CheckOS|1.46|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200810301052
2520628|pass|200810|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24.5-smp|200810301125
2521123|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|i386-freebsd-thread-multi-64int|5.6.2|freebsd|7.0-release|200810301043
2521267|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 34633|linux|2.6.22-1-k7|200810301300
2521584|pass|200810|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200810301248
2521744|pass|200810|rafl@debian.org|Devel-CheckOS|1.46|i686-linux-multi|5.10.0|linux|2.6.24-rc5|200810301406
2521755|pass|200810|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200810301308
2521791|pass|200810|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200810301311
2522809|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200810301439
2522998|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200810301500
2524061|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 34662|linux|2.6.22-1-k7|200810302041
2524206|pass|200810|alexchorny@gmail.com|Devel-CheckOS|1.46|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200810301605
2524431|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.26.3-29.fc9.i686|200810302038
2524729|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.26.3-29.fc9.i686|200810302110
2524941|pass|200810|alexchorny@gmail.com|Devel-CheckOS|1.46|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200810301734
2525121|pass|200810|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckLib|0.5|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200810302318
2525689|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.46|i686-linux-64int|5.8.8 patch 34662|linux|2.6.22-1-k7|200810310106
2526519|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.8 patch 34662|linux|2.6.22-1-k7|200810310305
2528098|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 34662|linux|2.6.22-1-k7|200810310740
2528209|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 34662|linux|2.6.22-1-k7|200810310756
2528306|pass|200810|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200810310543
2528355|pass|200810|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.26.3-29.fc9.i686|200810310720
2528663|pass|200810|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200810310637
2529022|pass|200810|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200810310743
2529027|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 34662|linux|2.6.22-1-k7|200810311016
2529118|pass|200810|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.26.3-29.fc9.i686|200810310945
2529333|pass|200810|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200810310847
2529508|pass|200810|JOST@cpan.org ("Josts Smokehouse")|CPAN-FindDependencies|2.0|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200810311157
2529526|pass|200810|g.grigelionis@computer.org|Devel-CheckOS|1.46|i386-linux|5.8.0|linux|2.6.9-22.18.bz155725.elsmp|200810311155
2529589|pass|200810|frank.wiegand@gmail.com ("Frank Wiegand")|Number-Phone|1.6|i686-linux|5.11.0 patch 34669|linux|2.6.24-21-generic|200810311207
2529657|pass|200810|frank.wiegand@gmail.com ("Frank Wiegand")|Devel-CheckOS|1.45|i686-linux|5.11.0 patch 34669|linux|2.6.24-21-generic|200810311221
2529744|pass|200810|frank.wiegand@gmail.com ("Frank Wiegand")|XML-Tiny|1.11|i686-linux|5.11.0 patch 34669|linux|2.6.24-21-generic|200810311244
2529946|pass|200810|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.26.3-29.fc9.i686|200810311309
2529997|pass|200810|artur+cpan@niif.spb.su (ARTO)|Devel-CheckOS|1.46|i386-linux-thread-multi|5.10.0|linux|2.6.18-92.1.10.el5|200810311256
2530130|pass|200810|g.grigelionis@computer.org|Devel-CheckOS|1.46|i586-linux-thread-multi|5.8.5|linux|2.6.8.1|200810311510
2530145|pass|200810|artur+cpan@niif.spb.su (ARTO)|Devel-CheckOS|1.45|i386-linux-thread-multi|5.10.0|linux|2.6.18-92.1.10.el5|200810311042
2530332|pass|200810|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.26.3-29.fc9.i686|200810311512
2530558|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200810311714
2530598|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200810311724
2531155|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-19-generic|200810312117
2531165|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-19-generic|200810312125
2532014|pass|200811|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200811010701
2532160|pass|200811|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200811010744
2532466|pass|200811|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200811010839
2532502|pass|200811|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200811010844
2532782|pass|200811|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200811010932
2532885|pass|200811|srezic@cpan.org|Devel-CheckOS|1.46|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200811011018
2534035|pass|200811|srezic@cpan.org|Devel-CheckOS|1.46|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200811011812
2534107|pass|200811|bingos@cpan.org|Devel-CheckOS|1.46|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200811011558
2534123|pass|200811|srezic@cpan.org|Devel-CheckOS|1.46|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200811011854
2534583|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34693|linux|2.6.22-1-k7|200811012020
2535111|pass|200811|srezic@cpan.org|Devel-CheckOS|1.46|i386-freebsd|5.6.2|freebsd|6.1-release|200811012242
2535874|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.46|i686-linux-64int|5.11.0 patch 34693|linux|2.6.22-1-k7|200811020919
2536236|pass|200811|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0.1|200811021008
2536237|pass|200811|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0.1|200811021009
2537140|pass|200811|bingos@cpan.org|Devel-CheckOS|1.46|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0.1|200811021430
2537248|pass|200811|bingos@cpan.org|Devel-CheckOS|1.46|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0.1|200811021456
2538157|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34693|linux|2.6.22-1-k7|200811021941
2538385|pass|200811|srezic@cpan.org|Class-DBI-ClassGenerator|1.01|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200811021942
2539180|pass|200811|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200811030147
2539183|pass|200811|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200811030150
2539727|pass|200811|srezic@cpan.org|DBIx-Class-SingletonRows|0.11|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200811030815
2541079|pass|200811|schwern@pobox.com|Devel-CheckOS|1.46|darwin-thread-multi-2level|5.10.0|darwin|8.11.1|200811030129
2541577|pass|200811|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0.1|200811031621
2542036|pass|200811|vpit@cpan.org|Devel-CheckOS|1.46|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200811031927
2542522|pass|200811|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0.1|200811032027
2542678|pass|200811|rlane10@wi.rr.com|Devel-CheckOS|1.46|i386-linux-thread-multi|5.10.0|linux|2.6.18-92.1.10.el5|200811031045
2542819|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34693|linux|2.6.22-1-k7|200811032155
2542832|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34693|linux|2.6.22-1-k7|200811032156
2543230|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34693|linux|2.6.22-1-k7|200811032210
2543298|pass|200811|srezic@cpan.org|Devel-CheckOS|1.46|i486-linux-gnu|5.8.8|linux|2.6.22.10|200811032250
2543970|pass|200811|vpit@cpan.org|XML-DoubleEncodedEntities|1.0|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200811040106
2544048|pass|200811|JOST@cpan.org ("Josts Smokehouse")|File-Find-Rule-Permissions|1.3|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811040119
2544168|pass|200811|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200811040106
2544179|pass|200811|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200811040109
2544575|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Tie-STDOUT|1.03|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811040415
2544934|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.46|i686-linux-64int|5.10.0 patch 34707|linux|2.6.22-1-k7|200811040531
2544949|pass|200811|bingos@cpan.org|Devel-CheckOS|1.46|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200811040433
2544952|pass|200811|bingos@cpan.org|Devel-CheckOS|1.46|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200811040434
2545268|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.10.0 patch 34707|linux|2.6.22-1-k7|200811040626
2545541|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 34707|linux|2.6.22-1-k7|200811040747
2545569|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0 patch 34707|linux|2.6.22-1-k7|200811040753
2546290|pass|200811|srezic@cpan.org|Devel-CheckOS|1.46|sun4-solaris-64int|5.8.4|solaris|2.10|200811041057
2546913|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 34707|linux|2.6.22-1-k7|200811041319
2546981|pass|200811|altblue@gmail.com ("Marius Feraru")|File-Find-Rule-Permissions|1.3|x86_64-linux|5.11.0 patch 34595|linux|2.6.18-8.1.14.el5|200811040044
2547071|pass|200811|bingos@cpan.org|Devel-CheckOS|1.46|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200811041235
2547087|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 34707|linux|2.6.22-1-k7|200811041339
2548496|pass|200811|bingos@cpan.org|Devel-CheckOS|1.46|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200811041751
2548813|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Number-Phone-UK-DetailedLocations|1.3|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811042025
2550270|pass|200811|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200811050048
2550539|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.46|i686-linux-64int|5.8.8 patch 34732|linux|2.6.22-1-k7|200811050439
2551320|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.8 patch 34732|linux|2.6.22-1-k7|200811050829
2552614|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 34732|linux|2.6.22-1-k7|200811051252
2552717|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 34732|linux|2.6.22-1-k7|200811051322
2552720|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 34732|linux|2.6.22-1-k7|200811051322
2552776|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 34732|linux|2.6.22-1-k7|200811051334
2553083|fail|200811|cbrown@mail ((root))|Devel-CheckOS|1.46|i386-linux|5.8.8|linux|2.6.18-53.1.14.el5xen|200811041414
2553764|pass|200811|rlane10@wi.rr.com|Devel-CheckOS|1.46|darwin-2level-ld|5.10.0|darwin|8.11.0|200811041145
2554662|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 34736|linux|2.6.22-1-k7|200811051937
2555016|pass|200811|bingos@cpan.org|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200811051933
2555052|pass|200811|bingos@cpan.org|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200811051938
2555055|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Bryar|3.0|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811052048
2555227|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.46|i686-linux-64int|5.8.8 patch 34736|linux|2.6.22-1-k7|200811052117
2555567|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 34736|linux|2.6.22-1-k7|200811052217
2555720|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.8 patch 34736|linux|2.6.22-1-k7|200811052249
2556200|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.49_01|i686-linux-64int|5.10.0 patch 34437|linux|2.6.22-1-k7|200811052353
2556214|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.49_01|i686-linux-64int|5.8.8 patch 34468|linux|2.6.22-1-k7|200811052355
2556218|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.49_01|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200811052355
2556227|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.49_01|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200811052357
2556243|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.49_01|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200811052354
2556397|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Games-Dice-Advanced|1.1|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811060121
2556513|pass|200811|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Devel-CheckOS|1.49_01|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200811061104
2556889|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 34736|linux|2.6.22-1-k7|200811060446
2557660|pass|200811|bingos@cpan.org|Devel-CheckOS|1.46|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200811060652
2557834|pass|200811|bingos@cpan.org|Devel-CheckOS|1.46|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200811060723
2557935|pass|200811|bingos@cpan.org|Devel-CheckOS|1.46|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200811060740
2557975|pass|200811|bingos@cpan.org|Devel-CheckOS|1.46|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200811060755
2558649|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 34736|linux|2.6.22-1-k7|200811061111
2558755|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckOS|1.49_01|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811061133
2558802|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckOS|1.49_01|i86pc-solaris-64int|5.10.0|solaris|2.11|200811061140
2560242|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Sub-WrapPackages|1.2|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811061914
2560840|pass|200811|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200811062110
2560842|pass|200811|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200811062111
2561302|pass|200811|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200811062317
2561375|pass|200811|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.6.2|freebsd|7.0-release|200811062340
2561391|pass|200811|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200811062353
2561849|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|i386-freebsd-thread-multi-64int|5.6.2|freebsd|7.0-release|200811070206
2562116|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200811070355
2562127|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200811070401
2562184|pass|200811|JOST@cpan.org ("Josts Smokehouse")|File-Overwrite|1.0|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811070616
2562584|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Class-CanBeA|1.2|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811070944
2563372|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.46|i686-linux-64int|5.11.0 patch 34757|linux|2.6.22-1-k7|200811071258
2563389|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34757|linux|2.6.22-1-k7|200811071301
2563414|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24.5-smp|200811071300
2563590|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200811071224
2563595|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200811071225
2564255|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34757|linux|2.6.22-1-k7|200811071441
2564363|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24.5-smp|200811071500
2564732|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34757|linux|2.6.22-1-k7|200811071535
2565220|pass|200811|vpit@cpan.org|Devel-CheckOS|1.49_01|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200811071653
2565716|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34757|linux|2.6.22-1-k7|200811071809
2567524|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34757|linux|2.6.22-1-k7|200811072307
2567855|fail|200811|JOST@cpan.org ("Josts Smokehouse")|Palm-SMS|0.02|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811080017
2567904|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Acme-Scurvy-Whoreson-BilgeRat|1.1|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811080031
2569847|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24.5-smp|200811081210
2570655|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Statistics-SerialCorrelation|1.1|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811081538
2570711|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200811081443
2570723|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200811081452
2571920|pass|200811|alexchorny@gmail.com|Sub-WrapPackages|1.2|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200811081546
2571947|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34778|linux|2.6.22-1-k7|200811082138
2572453|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.46|i686-linux-64int|5.11.0 patch 34778|linux|2.6.22-1-k7|200811082341
2573394|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34778|linux|2.6.22-1-k7|200811090239
2573577|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34778|linux|2.6.22-1-k7|200811090313
2574056|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34778|linux|2.6.22-1-k7|200811090553
2574416|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34778|linux|2.6.22-1-k7|200811090653
2575216|pass|200811|bingos@cpan.org|Devel-CheckOS|1.46|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200811091043
2575228|pass|200811|bingos@cpan.org|Devel-CheckOS|1.46|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200811091048
2575313|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.46|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200811081751
2575535|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200811082110
2575964|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200811090137
2576097|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.46|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200811090239
2576216|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux|5.8.8|linux|2.6.18-92.el5|200811090327
2576354|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.el5|200811090437
2576423|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.46|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.el5|200811090541
2577302|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.46|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200811082145
2577534|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.46|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200811090310
2577669|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.46|s390x-linux|5.8.8|linux|2.6.18-92.el5|200811090432
2577802|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.49_01|s390x-linux|5.10.0|linux|2.6.18-92.el5|200811091331
2578105|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.49_01|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200811091341
2578208|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux|5.10.0|linux|2.6.18-92.el5|200811090209
2578531|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200811090232
2579270|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.46|s390x-linux|5.10.0|linux|2.6.18-92.el5|200811090245
2579695|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200811091603
2579703|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|darwin-thread-multi-64int-2level|5.8.6|darwin|8.10.0|200811091604
2579710|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200811091605
2579751|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.49_01|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200811091712
2580238|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.49_01|s390x-linux|5.8.8|linux|2.6.18-92.el5|200811092001
2580961|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.26.3-29.fc9.i686|200811092241
2580967|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.26.3-29.fc9.i686|200811092243
2582050|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34785|linux|2.6.22-1-k7|200811100723
2582078|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.46|i686-linux-64int|5.11.0 patch 34785|linux|2.6.22-1-k7|200811100726
2582326|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.49_01|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.el5|200811100809
2582385|pass|200811|gabor@pti.co.il ("Gabor Szabo")|Data-Compare|1.21|i486-linux-gnu|5.10.0|linux|2.6.24-15-server|200811100919
2583005|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.49_01|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200811100952
2583698|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34785|linux|2.6.22-1-k7|200811101153
2583840|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34785|linux|2.6.22-1-k7|200811101204
2584335|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200811101218
2585044|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200811101523
2585532|pass|200811|mw487.yahoo.com@gmail.com|Data-Compare|1.21|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200811101037
2585862|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Tie-Hash-Longest|1.1|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811101759
2585863|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Statistics-ChiSquare|0.5|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811101759
2586407|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux|5.10.0|linux|2.6.18-92.el5|200811102044
2587312|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux|5.8.8|linux|2.6.18-92.el5|200811110028
2587440|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34785|linux|2.6.22-1-k7|200811110042
2587527|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34785|linux|2.6.22-1-k7|200811110103
2587638|pass|200811|david@cantrell.org.uk|Devel-CheckOS|1.49_01|i686-linux|5.8.8|linux|2.4.27-3-686|200811110029
2587640|pass|200811|david@cantrell.org.uk|Devel-CheckOS|1.49_01|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200811110031
2587641|pass|200811|david@cantrell.org.uk|Devel-CheckOS|1.49_01|IP35-irix-64int|5.8.8|irix|6.5|200811110131
2587663|pass|200811|david@cantrell.org.uk|Devel-CheckOS|1.49_01|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200811110042
2587691|pass|200811|JOST@cpan.org ("Josts Smokehouse")|NestedMap|1.0|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811110154
2587697|pass|200811|david@cantrell.org.uk|Devel-CheckOS|1.49_01|i686-linux|5.6.2|linux|2.4.27-3-686|200811110046
2587984|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.46|i686-linux-64int|5.8.9 patch 34813|linux|2.6.22-1-k7|200811110308
2589161|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.9 patch 34813|linux|2.6.22-1-k7|200811110650
2589185|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.9 patch 34813|linux|2.6.22-1-k7|200811110713
2589391|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.46|s390x-linux|5.6.2|linux|2.6.18-92.1.17.el5|200811110737
2590240|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.9 patch 34813|linux|2.6.22-1-k7|200811110919
2590531|pass|200811|david@cantrell.org.uk|Devel-CheckOS|1.49_01|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200811110903
2591312|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.9 patch 34813|linux|2.6.22-1-k7|200811111117
2591868|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200811111124
2591883|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200811111127
2593003|pass|200811|david@cantrell.org.uk|Devel-CheckOS|1.49_01|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200811110859
2593091|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.9 patch 34813|linux|2.6.22-1-k7|200811111247
2593304|unknown|200811|JOST@cpan.org ("Josts Smokehouse")|Acme-Licence|1.0|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811111642
2593570|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.4|200811111625
2593601|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200811111628
2593627|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.4|200811111630
2593732|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200811111649
2594983|pass|200811|gabor@pti.co.il ("Gabor Szabo")|Devel-CheckOS|1.46|i486-linux-gnu|5.10.0|linux|2.6.24-15-server|200811112313
2595600|pass|200811|mw487.yahoo.com@gmail.com|Devel-CheckOS|1.46|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200811111753
2595945|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200811120055
2595946|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.10.0 patch 34437|linux|2.6.22-1-k7|200811120056
2595947|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200811120056
2595949|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.8.8 patch 34468|linux|2.6.22-1-k7|200811120056
2595950|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200811120056
2595952|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200811120057
2596294|pass|200811|bingos@cpan.org|Devel-CheckOS|1.46|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.4|200811120059
2596797|pass|200811|bingos@cpan.org|Devel-CheckOS|1.46|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.4|200811120242
2597159|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200811120500
2597576|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.46|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200811120650
2597637|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200811120708
2597801|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200811120642
2597814|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200811120644
2597818|pass|200811|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Devel-CheckOS|1.50|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200811121544
2597982|pass|200811|srezic@cpan.org|Data-Compare|1.21|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811120755
2597996|pass|200811|srezic@cpan.org|Devel-CheckOS|1.46|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811120755
2598013|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.46|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200811120757
2598086|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux|5.6.2|linux|2.6.18-92.1.17.el5|200811120801
2598113|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.17.el5|200811120803
2598372|pass|200811|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811120809
2598466|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux|5.10.0|linux|2.6.18-92.el5|200811120819
2598531|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200811120822
2598573|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux|5.8.8|linux|2.6.18-92.el5|200811120825
2599401|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200811121000
2599568|pass|200811|srezic@cpan.org|Devel-CheckOS|1.46|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200811120954
2600248|unknown|200811|JOST@cpan.org ("Josts Smokehouse")|Acme-Pony|1.1.2|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811121057
2600283|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Tie-Hash-Rank|1.0.1|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811121101
2600480|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Tie-Scalar-Decay|1.1.1|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811121135
2600709|pass|200811|srezic@cpan.org|XML-Tiny|1.11|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811121131
2601466|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckOS|1.50|i86pc-solaris-64int|5.10.0|solaris|2.11|200811121400
2601594|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckOS|1.50|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811121411
2601746|pass|200811|srezic@cpan.org|Sort-MultipleFields|1.0|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811121403
2601943|pass|200811|hohokus@gmail.com|Devel-CheckOS|1.50|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200811120834
2602130|pass|200811|srezic@cpan.org|Palm-Treo680MessagesDB|1.0|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811121457
2602468|pass|200811|srezic@cpan.org|Data-Hexdumper|1.4|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811121457
2603473|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24.5-smp|200811121908
2603481|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200811121719
2603489|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200811121720
2603493|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24.5-smp|200811121914
2603691|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200811121920
2603694|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|darwin-thread-multi-64int-2level|5.8.6|darwin|8.10.0|200811121922
2603695|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200811121923
2604162|pass|200811|g.grigelionis@computer.org|Devel-CheckOS|1.50|i586-linux-thread-multi|5.8.8|linux|2.6.16|200811122335
2604312|pass|200811|srezic@cpan.org|CPU-Emulator-Memory|1.1001|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811130001
2604455|pass|200811|srezic@cpan.org|Data-Transactional|1.02|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811130001
2604575|pass|200811|srezic@cpan.org|CPU-Emulator-Z80|1.0|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811130001
2604699|pass|200811|srezic@cpan.org|Number-Phone|1.6|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811130026
2606310|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.49_01|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200811130757
2606331|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.49_01|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200811130802
2606456|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200811130758
2606908|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200811130906
2606926|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200811130910
2607802|pass|200811|srezic@cpan.org|Devel-CheckLib|0.5|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811131049
2608821|pass|200811|srezic@cpan.org|CPAN-FindDependencies|2.0|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811131306
2609436|pass|200811|srezic@cpan.org|Bryar|3.0|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811131443
2609496|pass|200811|srezic@cpan.org|File-Find-Rule-Permissions|1.3|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811131443
2609715|pass|200811|srezic@cpan.org|Tie-STDOUT|1.03|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811131532
2609908|pass|200811|bingos@cpan.org|Data-Compare|1.21|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200811131500
2610048|pass|200811|bingos@cpan.org|Data-Compare|1.21|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200811131522
2610198|pass|200811|robbiebow@gmail.com|CPAN-FindDependencies|2.0|i486-linux-gnu|5.10.0|linux|2.6.26.1|200811122351
2610256|pass|200811|srezic@cpan.org|Palm-TreoPhoneCallDB|1.1|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811131615
2610415|pass|200811|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.50|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200811140034
2610452|pass|200811|srezic@cpan.org|Number-Phone-UK-DetailedLocations|1.3|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811131727
2610645|pass|200811|srezic@cpan.org|Net-Random|2.0|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811131727
2611193|pass|200811|g.grigelionis@computer.org|Devel-CheckOS|1.50|x86_64-linux|5.8.8|linux|2.6.5-7.276-smp|200811131835
2611223|pass|200811|g.grigelionis@computer.org|Devel-CheckOS|1.50|x86_64-linux-thread-multi|5.8.3|linux|2.6.4|200811131840
2611233|pass|200811|g.grigelionis@computer.org|Devel-CheckOS|1.50|i586-linux-thread-multi|5.8.5|linux|2.6.8.1|200811131921
2611260|pass|200811|vpit@cpan.org|Devel-CheckOS|1.50|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200811131847
2611359|pass|200811|g.grigelionis@computer.org|Devel-CheckOS|1.50|i386-linux|5.8.0|linux|2.6.9-22.18.bz155725.elsmp|200811131852
2611588|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200811131811
2611689|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200811131825
2612452|pass|200811|srezic@cpan.org|Games-Dice-Advanced|1.1|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811132052
2612701|pass|200811|srezic@cpan.org|Sub-WrapPackages|1.2|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811132107
2613115|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200811132141
2614137|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.10.0 patch 34793|linux|2.6.22-1-k7|200811132239
2615275|pass|200811|srezic@cpan.org|Class-CanBeA|1.2|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811132348
2615288|pass|200811|srezic@cpan.org|File-Overwrite|1.0|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811132348
2616015|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200811140029
2616757|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.10.0 patch 34793|linux|2.6.22-1-k7|200811140150
2617296|pass|200811|mthurn@cpan.org|Devel-CheckOS|1.50|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200811140257
2617329|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 34793|linux|2.6.22-1-k7|200811140403
2617636|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 34793|linux|2.6.22-1-k7|200811140518
2618156|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200811140733
2618435|pass|200811|srezic@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811140748
2618695|pass|200811|srezic@cpan.org|Palm-SMS|0.02|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811140750
2619006|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 34793|linux|2.6.22-1-k7|200811140834
2620460|pass|200811|srezic@cpan.org|Statistics-SerialCorrelation|1.1|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811141004
2620572|pass|200811|srezic@cpan.org|Palm-SMS|0.02|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200811140748
2620625|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0 patch 34793|linux|2.6.22-1-k7|200811141025
2621323|pass|200811|srezic@cpan.org|Tie-Hash-Longest|1.1|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811141123
2621490|pass|200811|srezic@cpan.org|Statistics-ChiSquare|0.5|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811141123
2621677|pass|200811|david@cantrell.org.uk|Devel-CheckOS|1.50|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200811140501
2621832|pass|200811|srezic@cpan.org|NestedMap|1.0|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811141209
2621900|pass|200811|srezic@cpan.org|Data-Compare|1.21|i686-linux|5.8.9|linux|2.6.18-4-486|200811141203
2622725|pass|200811|mw487.yahoo.com@gmail.com|Devel-CheckOS|1.50|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200811140858
2623012|pass|200811|david@cantrell.org.uk|Devel-CheckOS|1.50|i686-linux|5.6.2|linux|2.4.27-3-686|200811141453
2623102|pass|200811|david@cantrell.org.uk|Devel-CheckOS|1.50|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200811141519
2623103|pass|200811|david@cantrell.org.uk|Devel-CheckOS|1.50|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200811141519
2623107|pass|200811|david@cantrell.org.uk|Devel-CheckOS|1.50|i686-linux|5.8.8|linux|2.4.27-3-686|200811141519
2624397|pass|200811|david@cantrell.org.uk|Devel-CheckOS|1.50|IP35-irix-64int|5.8.8|irix|6.5|200811141813
2624447|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200811141816
2625842|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34836|linux|2.6.22-1-k7|200811142302
2626311|pass|200811|srezic@cpan.org|Tie-Scalar-Decay|1.1.1|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200811150006
2626493|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.11.0 patch 34836|linux|2.6.22-1-k7|200811142318
2626630|pass|200811|srezic@cpan.org|Tie-Hash-Rank|1.0.1|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811150007
2626667|pass|200811|srezic@cpan.org|Tie-Scalar-Decay|1.1.1|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811150007
2626762|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34836|linux|2.6.22-1-k7|200811150024
2626763|pass|200811|david@cantrell.org.uk|Devel-CheckOS|1.50|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200811142331
2627138|pass|200811|srezic@cpan.org|Tie-Hash-Rank|1.0.1|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200811150006
2627681|pass|200811|bingos@cpan.org|Data-Compare|1.21|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200811150124
2628382|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200811150359
2628475|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34836|linux|2.6.22-1-k7|200811150517
2628657|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200811150456
2628664|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200811150458
2628823|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34836|linux|2.6.22-1-k7|200811150634
2629266|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34836|linux|2.6.22-1-k7|200811150819
2632873|pass|200811|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-19-generic|200811161135
2632879|pass|200811|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-19-generic|200811161137
2633154|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-19-generic|200811161249
2633214|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-19-generic|200811161305
2633552|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34839|linux|2.6.22-1-k7|200811161440
2634255|pass|200811|rlane10@wi.rr.com|Devel-CheckOS|1.50|i386-linux-thread-multi|5.10.0|linux|2.6.18-92.1.10.el5|200811151624
2634399|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.11.0 patch 34839|linux|2.6.22-1-k7|200811161651
2634650|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200811161637
2634691|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200811161641
2635457|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34839|linux|2.6.22-1-k7|200811161927
2636444|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34839|linux|2.6.22-1-k7|200811162117
2637023|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34839|linux|2.6.22-1-k7|200811162226
2638283|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34839|linux|2.6.22-1-k7|200811170451
2639731|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.6.2|freebsd|7.0-release|200811171306
2640048|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200811171609
2641399|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.9 patch 34869|linux|2.6.22-1-k7|200811172327
2641438|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200811172217
2641746|pass|200811|srezic@cpan.org|Devel-CheckOS|1.50|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200811180027
2641757|pass|200811|srezic@cpan.org|Devel-CheckOS|1.50|i386-freebsd|5.10.0|freebsd|6.1-release|200811180029
2642172|pass|200811|srezic@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200811180125
2642177|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.8.9 patch 34869|linux|2.6.22-1-k7|200811180126
2642350|pass|200811|srezic@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200811180144
2642369|pass|200811|srezic@cpan.org|Devel-CheckOS|1.50|i386-freebsd|5.6.2|freebsd|6.1-release|200811180117
2642985|pass|200811|bingos@cpan.org|Data-Compare|1.21|i386-dragonfly-thread-multi-64int|5.10.0|dragonfly|2.0.1-release|200811180237
2643177|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.9 patch 34869|linux|2.6.22-1-k7|200811180351
2643414|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.9 patch 34869|linux|2.6.22-1-k7|200811180445
2643566|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.9 patch 34869|linux|2.6.22-1-k7|200811180515
2643594|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.9 patch 34869|linux|2.6.22-1-k7|200811180512
2643742|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-dragonfly-thread-multi-64int|5.10.0|dragonfly|2.0.1-release|200811180453
2644577|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.el5|200811180914
2644766|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200811180946
2645722|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.el5|200811181254
2645781|pass|200811|bingos@cpan.org|Data-Compare|1.21|i386-dragonfly-thread-multi-64int|5.8.8|dragonfly|2.0.1-release|200811181218
2645990|pass|200811|JOST@cpan.org ("Josts Smokehouse")|XML-DoubleEncodedEntities|1.0|i86pc-solaris-64int|5.8.9|solaris|2.11|200811181441
2646137|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckOS|1.50|i86pc-solaris-64int|5.8.9|solaris|2.11|200811181505
2646760|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Data-Compare|1.21|i86pc-solaris-64int|5.8.9|solaris|2.11|200811181722
2647154|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-dragonfly-thread-multi-64int|5.8.8|dragonfly|2.0.1-release|200811181754
2647682|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200811182002
2649448|pass|200811|JOST@cpan.org ("Josts Smokehouse")|XML-Tiny|1.11|i86pc-solaris-64int|5.8.9|solaris|2.11|200811190447
2650706|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Class-DBI-ClassGenerator|1.01|i86pc-solaris-64int|5.8.9|solaris|2.11|200811190859
2651864|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200811191400
2652026|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Sort-MultipleFields|1.0|i86pc-solaris-64int|5.8.9|solaris|2.11|200811191553
2652487|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Data-Hexdumper|1.4|i86pc-solaris-64int|5.8.9|solaris|2.11|200811191829
2652489|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Palm-Treo680MessagesDB|1.0|i86pc-solaris-64int|5.8.9|solaris|2.11|200811191830
2652993|pass|200811|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i386-dragonfly-thread-multi-64int|5.8.8|dragonfly|2.0.1-release|200811191942
2653123|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Data-Transactional|1.02|i86pc-solaris-64int|5.8.9|solaris|2.11|200811192129
2653317|pass|200811|JOST@cpan.org ("Josts Smokehouse")|CPU-Emulator-Memory|1.1001|i86pc-solaris-64int|5.8.9|solaris|2.11|200811192205
2653319|pass|200811|JOST@cpan.org ("Josts Smokehouse")|CPU-Emulator-Z80|1.0|i86pc-solaris-64int|5.8.9|solaris|2.11|200811192205
2653778|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.11.0 patch 34883|linux|2.6.22-1-k7|200811192255
2654011|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.11|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200811192320
2654013|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Number-Phone|1.6|i86pc-solaris-64int|5.8.9|solaris|2.11|200811192332
2654271|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34883|linux|2.6.22-1-k7|200811200003
2654660|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34883|linux|2.6.22-1-k7|200811200044
2655089|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34883|linux|2.6.22-1-k7|200811200228
2655915|pass|200811|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200811200359
2656392|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34883|linux|2.6.22-1-k7|200811200620
2657124|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckLib|0.5|i86pc-solaris-64int|5.8.9|solaris|2.11|200811200832
2657474|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34883|linux|2.6.22-1-k7|200811200932
2658270|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-dragonfly-thread-multi-64int|5.8.9|dragonfly|2.0.1-release|200811201152
2658653|pass|200811|bingos@cpan.org|Data-Compare|1.21|i386-dragonfly-thread-multi-64int|5.8.9|dragonfly|2.0.1-release|200811201322
2658696|pass|200811|stro@cpan.org|Data-Compare|1.21|armv5tejl-linux|5.8.8|linux|2.6.17.14|200811201730
2658954|pass|200811|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200811201415
2659533|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.10.0 patch 34893|linux|2.6.22-1-k7|200811201758
2659844|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.11|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200811201838
2659938|pass|200811|JOST@cpan.org ("Josts Smokehouse")|CPAN-FindDependencies|2.0|i86pc-solaris-64int|5.8.9|solaris|2.11|200811201900
2660119|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.11|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200811201917
2660519|fail|200811|JOST@cpan.org ("Josts Smokehouse")|Bryar|3.0|i86pc-solaris-64int|5.8.9|solaris|2.11|200811202023
2660902|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.10.0 patch 34893|linux|2.6.22-1-k7|200811202057
2660920|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0 patch 34893|linux|2.6.22-1-k7|200811202100
2661386|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200811202109
2662093|pass|200811|JOST@cpan.org ("Josts Smokehouse")|File-Find-Rule-Permissions|1.3|i86pc-solaris-64int|5.8.9|solaris|2.11|200811210002
2662766|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Tie-STDOUT|1.03|i86pc-solaris-64int|5.8.9|solaris|2.11|200811210158
2663127|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Palm-TreoPhoneCallDB|1.1|i86pc-solaris-64int|5.8.9|solaris|2.11|200811210309
2664231|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 34893|linux|2.6.22-1-k7|200811210935
2664535|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 34893|linux|2.6.22-1-k7|200811211039
2664721|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Number-Phone-UK-DetailedLocations|1.3|i86pc-solaris-64int|5.8.9|solaris|2.11|200811211129
2664815|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 34893|linux|2.6.22-1-k7|200811211137
2666546|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200811212250
2666633|pass|200811|srezic@cpan.org|Devel-CheckOS|1.50|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811212337
2667070|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.8.9 patch 34899|linux|2.6.22-1-k7|200811220320
2667220|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.9 patch 34899|linux|2.6.22-1-k7|200811220401
2667663|pass|200811|stro@cpan.org|Data-Compare|1.21|armv5tejl-linux|5.10.0|linux|2.6.17.14|200811220848
2667763|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.9 patch 34899|linux|2.6.22-1-k7|200811220614
2668317|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.9 patch 34899|linux|2.6.22-1-k7|200811220920
2669079|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200811221129
2669945|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.9 patch 34899|linux|2.6.22-1-k7|200811221537
2669975|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.9 patch 34899|linux|2.6.22-1-k7|200811221546
2672840|pass|200811|jeff@zeroclue.com|Devel-CheckOS|1.50|i686-linux|5.10.0|linux|2.6.18.8|200811230307
2672888|pass|200811|rafl@debian.org|Devel-CheckOS|1.50|x86_64-linux-multi|5.11.0|linux|2.6.26-1-xen-amd64|200811230413
2673381|pass|200811|rafl@debian.org|XML-DoubleEncodedEntities|1.0|x86_64-linux-multi|5.11.0|linux|2.6.26-1-xen-amd64|200811230651
2673626|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200811230505
2674615|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200811230922
2675217|pass|200811|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200811231348
2676161|pass|200811|rafl@debian.org|Data-Compare|1.21|x86_64-linux-multi|5.11.0|linux|2.6.26-1-xen-amd64|200811232125
2678038|pass|200811|rafl@debian.org|Class-DBI-ClassGenerator|1.01|x86_64-linux-multi|5.11.0|linux|2.6.26-1-xen-amd64|200811240747
2678887|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200811240554
2679219|pass|200811|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200811240814
2680013|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Games-Dice-Advanced|1.1|i86pc-solaris-64int|5.8.9|solaris|2.11|200811241933
2680361|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Sub-WrapPackages|1.2|i86pc-solaris-64int|5.8.9|solaris|2.11|200811242131
2682122|pass|200811|JOST@cpan.org ("Josts Smokehouse")|File-Overwrite|1.0|i86pc-solaris-64int|5.8.9|solaris|2.11|200811250602
2682515|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34904|linux|2.6.22-1-k7|200811250743
2682752|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Class-CanBeA|1.2|i86pc-solaris-64int|5.8.9|solaris|2.11|200811250844
2683184|pass|200811|stro@cpan.org|Devel-CheckOS|1.50|armv5tejl-linux|5.10.0|linux|2.6.17.14|200811251247
2684229|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.9|freebsd|6.2-release|200811251034
2684539|pass|200811|stsc@refcnt.org ("Steven Schubiger")|Devel-CheckOS|1.50|i686-linux|5.10.0|linux|2.6.18-4-686|200811250930
2685559|pass|200811|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200811250431
2686529|fail|200811|JOST@cpan.org ("Josts Smokehouse")|Palm-SMS|0.02|i86pc-solaris-64int|5.8.9|solaris|2.11|200811252007
2686586|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Acme-Scurvy-Whoreson-BilgeRat|1.1|i86pc-solaris-64int|5.8.9|solaris|2.11|200811252018
2687313|pass|200811|stsc@refcnt.org ("Steven Schubiger")|XML-DoubleEncodedEntities|1.0|i686-linux|5.10.0|linux|2.6.18-4-686|200811252254
2687773|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.11.0 patch 34904|linux|2.6.22-1-k7|200811260025
2690322|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Statistics-SerialCorrelation|1.1|i86pc-solaris-64int|5.8.9|solaris|2.11|200811260930
2690890|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34904|linux|2.6.22-1-k7|200811261142
2691240|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34904|linux|2.6.22-1-k7|200811261244
2691249|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200811261146
2691266|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200811261148
2691526|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34904|linux|2.6.22-1-k7|200811261302
2691558|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0.1|200811261249
2691588|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0.1|200811261259
2692351|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34904|linux|2.6.22-1-k7|200811261736
2694091|pass|200811|stsc@refcnt.org ("Steven Schubiger")|Data-Compare|1.21|i686-linux|5.10.0|linux|2.6.18-4-686|200811270500
2694990|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34943|linux|2.6.22-1-k7|200811271010
2695035|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34943|linux|2.6.22-1-k7|200811271013
2695139|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.11.0 patch 34943|linux|2.6.22-1-k7|200811271036
2696271|pass|200811|rafl@debian.org|Devel-CheckOS|1.50|i686-linux-multi|5.10.0|linux|2.6.24-rc5|200811271420
2696614|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34943|linux|2.6.22-1-k7|200811271526
2696650|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34943|linux|2.6.22-1-k7|200811271534
2697611|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34943|linux|2.6.22-1-k7|200811271846
2698909|pass|200811|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200811272231
2699083|pass|200811|srezic@cpan.org|Devel-CheckOS|1.50|amd64-freebsd|5.8.9|freebsd|7.0-release|200811272254
2699400|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0.1|200811272249
2700276|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.11.0 patch 34948|linux|2.6.22-1-k7|200811280245
2701374|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34948|linux|2.6.22-1-k7|200811280622
2702042|pass|200811|srezic@cpan.org|Data-Compare|1.21|amd64-freebsd|5.8.9|freebsd|7.0-release|200811280745
2703484|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34948|linux|2.6.22-1-k7|200811281247
2703503|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34948|linux|2.6.22-1-k7|200811281252
2704176|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34948|linux|2.6.22-1-k7|200811281304
2704505|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34948|linux|2.6.22-1-k7|200811281414
2704593|fail|200811|ackack1024@gmail.com|Bryar|3.0|x86_64-linux-gnu|5.10.0|linux|2.6.22-3-vserver-amd64|200811280034
2705095|pass|200811|srezic@cpan.org|Sort-MultipleFields|1.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200811281600
2705307|pass|200811|srezic@cpan.org|Data-Hexdumper|1.4|amd64-freebsd|5.8.9|freebsd|7.0-release|200811281656
2705560|pass|200811|srezic@cpan.org|Palm-Treo680MessagesDB|1.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200811281702
2706019|pass|200811|srezic@cpan.org|CPU-Emulator-Memory|1.1001|amd64-freebsd|5.8.9|freebsd|7.0-release|200811281903
2706021|pass|200811|srezic@cpan.org|CPU-Emulator-Z80|1.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200811281903
2706119|pass|200811|srezic@cpan.org|Data-Transactional|1.02|amd64-freebsd|5.8.9|freebsd|7.0-release|200811281903
2706439|pass|200811|srezic@cpan.org|Number-Phone|1.6|amd64-freebsd|5.8.9|freebsd|7.0-release|200811282003
2707483|pass|200811|srezic@cpan.org|Devel-CheckLib|0.5|amd64-freebsd|5.8.9|freebsd|7.0-release|200811290019
2707819|pass|200811|srezic@cpan.org|CPAN-FindDependencies|2.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200811290036
2707901|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.10.0 patch 34916|linux|2.6.22-1-k7|200811290043
2708650|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.10.0 patch 34916|linux|2.6.22-1-k7|200811290213
2708838|pass|200811|srezic@cpan.org|Bryar|3.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200811290207
2709107|pass|200811|srezic@cpan.org|Games-Dice-Advanced|1.1|amd64-freebsd|5.8.9|freebsd|7.0-release|200811290314
2709271|pass|200811|srezic@cpan.org|Sub-WrapPackages|1.2|amd64-freebsd|5.8.9|freebsd|7.0-release|200811290315
2709871|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.11|s390x-linux|5.10.0|linux|2.6.18-92.el5|200811290459
2710189|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 34916|linux|2.6.22-1-k7|200811290641
2710754|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 34916|linux|2.6.22-1-k7|200811290915
2710866|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 34916|linux|2.6.22-1-k7|200811290937
2712234|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0 patch 34916|linux|2.6.22-1-k7|200811291301
2712827|pass|200811|srezic@cpan.org|Class-CanBeA|1.2|amd64-freebsd|5.8.9|freebsd|7.0-release|200811291334
2713044|pass|200811|srezic@cpan.org|File-Overwrite|1.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200811291335
2714072|pass|200811|srezic@cpan.org|Palm-SMS|0.02|amd64-freebsd|5.8.9|freebsd|7.0-release|200811291337
2714520|pass|200811|srezic@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|amd64-freebsd|5.8.9|freebsd|7.0-release|200811291334
2716111|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200811300040
2716120|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200811300042
2717845|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200811300924
2718220|pass|200811|"Rick Myers <rick@jrmyers.net>"|Data-Compare|1.21|i686-linux|5.10.0|linux|2.6.24.3|200811300444
2719984|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.9 patch 34928|linux|2.6.22-1-k7|200811301552
2720540|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.8.9 patch 34928|linux|2.6.22-1-k7|200811301659
2720556|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.9 patch 34928|linux|2.6.22-1-k7|200811301658
2722147|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Class-DBI-ClassGenerator|1.01|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200811302019
2723860|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.9 patch 34928|linux|2.6.22-1-k7|200812010046
2724097|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200812010137
2724251|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.9 patch 34928|linux|2.6.22-1-k7|200812010203
2724627|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.9 patch 34928|linux|2.6.22-1-k7|200812010311
2726411|pass|200812|salvacpan@gmail.com ("Salvador Fandino")|XML-DoubleEncodedEntities|1.0|sparc64-netbsd|5.10.0|netbsd|4.0_rc5|200812011403
2726822|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200812011210
2729332|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200812020300
2730254|pass|200812|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200812020607
2730352|pass|200812|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200812020629
2730753|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.9 patch 34974|linux|2.6.22-1-k7|200812020858
2731076|pass|200812|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200812020307
2731451|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.8.9 patch 34974|linux|2.6.22-1-k7|200812021111
2731466|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.9 patch 34974|linux|2.6.22-1-k7|200812021113
2732934|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.9 patch 34974|linux|2.6.22-1-k7|200812021428
2732946|fail|200812|rivy@cpan.org ("Roy Ivy III")|Devel-CheckLib|0.5|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200812020341
2732948|fail|200812|rivy@cpan.org ("Roy Ivy III")|Devel-CheckLib|0.5|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200812020454
2733597|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.9 patch 34974|linux|2.6.22-1-k7|200812021605
2734063|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.9 patch 34974|linux|2.6.22-1-k7|200812021726
2735721|pass|200812|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.3|200812022258
2736160|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Class-DBI-ClassGenerator|1.01|s390x-linux|5.10.0|linux|2.6.18-92.el5|200812030416
2736235|pass|200812|stsc@refcnt.org ("Steven Schubiger")|XML-Tiny|1.11|i686-linux|5.10.0|linux|2.6.18-4-686|200812030547
2736945|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34986|linux|2.6.22-1-k7|200812030843
2736963|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34986|linux|2.6.22-1-k7|200812030850
2736968|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.11.0 patch 34986|linux|2.6.22-1-k7|200812030852
2737577|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34986|linux|2.6.22-1-k7|200812031025
2739030|pass|200812|stro@cpan.org|Devel-CheckOS|1.50|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200812031531
2739588|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34986|linux|2.6.22-1-k7|200812031553
2739594|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34986|linux|2.6.22-1-k7|200812031554
2741814|pass|200812|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0|200812032155
2743024|pass|200812|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.4|200812040607
2743095|pass|200812|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.4|200812040643
2743368|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.10.0 patch 35005|linux|2.6.22-1-k7|200812040926
2743415|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.10.0 patch 35005|linux|2.6.22-1-k7|200812040920
2744247|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0 patch 35005|linux|2.6.22-1-k7|200812041138
2745522|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 35005|linux|2.6.22-1-k7|200812041617
2746236|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.11|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200812041816
2746682|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 35005|linux|2.6.22-1-k7|200812041910
2747434|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 35005|linux|2.6.22-1-k7|200812042145
2747547|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.11|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.el5|200812042206
2748728|pass|200812|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.4|200812050103
2748988|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 35013|linux|2.6.22-1-k7|200812050338
2749116|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.11.0 patch 35013|linux|2.6.22-1-k7|200812050401
2749471|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 35013|linux|2.6.22-1-k7|200812050437
2749841|pass|200812|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200812050514
2751713|pass|200812|salvacpan@gmail.com ("Salvador Fandino")|Devel-CheckOS|1.50|sparc64-netbsd|5.10.0|netbsd|4.0_rc5|200812051112
2752288|pass|200812|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200812051129
2753021|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 35013|linux|2.6.22-1-k7|200812051438
2753040|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 35013|linux|2.6.22-1-k7|200812051434
2753936|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 35013|linux|2.6.22-1-k7|200812051658
2754877|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Class-DBI-ClassGenerator|1.01|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200812052029
2755272|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.11|s390x-linux|5.6.2|linux|2.6.18-92.1.17.el5|200812052216
2756192|pass|200812|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24.5-smp|200812060213
2757104|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.17.el5|200812060801
2757570|pass|200812|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200812060938
2757875|pass|200812|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24.5-smp|200812061144
2760109|pass|200812|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i86pc-solaris-thread-multi-64int|5.8.9|solaris|2.10|200812061806
2760549|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.10.0 patch 35041|linux|2.6.22-1-k7|200812062053
2760698|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.10.0 patch 35041|linux|2.6.22-1-k7|200812062121
2761654|pass|200812|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812062319
2762696|pass|200812|salvacpan@gmail.com ("Salvador Fandino")|Data-Compare|1.21|sparc64-netbsd|5.10.0|netbsd|4.0_rc5|200812070205
2763041|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0 patch 35041|linux|2.6.22-1-k7|200812070419
2763250|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 35041|linux|2.6.22-1-k7|200812070354
2764240|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 35041|linux|2.6.22-1-k7|200812070834
2765084|pass|200812|srezic@cpan.org|Data-Compare|1.21|amd64-freebsd|5.8.9|freebsd|7.0-release|200812071006
2766527|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 35041|linux|2.6.22-1-k7|200812071127
2767043|pass|200812|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200812071009
2767155|pass|200812|srezic@cpan.org|Devel-CheckOS|1.50|amd64-freebsd|5.8.9|freebsd|7.0-release|200812071006
2768263|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux|5.6.2|linux|2.6.18-92.1.17.el5|200812071746
2768458|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200812071756
2770111|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200812072059
2770543|pass|200812|srezic@cpan.org|XML-Tiny|1.11|amd64-freebsd|5.8.9|freebsd|7.0-release|200812072153
2772224|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200812080152
2772792|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200812080334
2773286|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200812080534
2773440|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200812080645
2774364|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.11|s390x-linux|5.8.8|linux|2.6.18-92.el5|200812080935
2775728|pass|200812|srezic@cpan.org|Sort-MultipleFields|1.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200812081416
2776027|pass|200812|srezic@cpan.org|Data-Hexdumper|1.4|amd64-freebsd|5.8.9|freebsd|7.0-release|200812081449
2776080|pass|200812|srezic@cpan.org|Palm-Treo680MessagesDB|1.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200812081449
2776152|pass|200812|srezic@cpan.org|DBIx-Class-SingletonRows|0.11|amd64-freebsd|5.8.9|freebsd|7.0-release|200812081527
2776594|pass|200812|srezic@cpan.org|Data-Transactional|1.02|amd64-freebsd|5.8.9|freebsd|7.0-release|200812081622
2776837|pass|200812|srezic@cpan.org|CPU-Emulator-Z80|1.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200812081622
2776950|pass|200812|srezic@cpan.org|CPU-Emulator-Memory|1.1001|amd64-freebsd|5.8.9|freebsd|7.0-release|200812081622
2777481|pass|200812|srezic@cpan.org|Number-Phone|1.6|amd64-freebsd|5.8.9|freebsd|7.0-release|200812081700
2779448|pass|200812|srezic@cpan.org|Devel-CheckLib|0.5|amd64-freebsd|5.8.9|freebsd|7.0-release|200812082312
2782107|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 35055|linux|2.6.22-1-k7|200812090949
2782725|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.11.0 patch 35055|linux|2.6.22-1-k7|200812091105
2782913|pass|200812|srezic@cpan.org|Bryar|3.0|amd64-freebsd|5.8.8|freebsd|7.0-release|200812091107
2782968|pass|200812|srezic@cpan.org|CPAN-FindDependencies|2.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200812091102
2782974|pass|200812|srezic@cpan.org|CPAN-FindDependencies|2.0|amd64-freebsd|5.8.8|freebsd|7.0-release|200812091123
2782996|pass|200812|srezic@cpan.org|CPU-Emulator-Z80|1.0|amd64-freebsd|5.8.8|freebsd|7.0-release|200812091124
2783051|pass|200812|srezic@cpan.org|Data-Compare|1.21|amd64-freebsd|5.8.8|freebsd|7.0-release|200812091126
2783242|pass|200812|srezic@cpan.org|Data-Hexdumper|1.4|amd64-freebsd|5.8.8|freebsd|7.0-release|200812091127
2784434|pass|200812|srezic@cpan.org|Bryar|3.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200812091209
2784544|pass|200812|srezic@cpan.org|CPU-Emulator-Memory|1.1001|amd64-freebsd|5.8.8|freebsd|7.0-release|200812091124
2785638|pass|200812|srezic@cpan.org|Sort-MultipleFields|1.0|amd64-freebsd|5.8.8|freebsd|7.0-release|200812091256
2785705|pass|200812|srezic@cpan.org|Palm-Treo680MessagesDB|1.0|amd64-freebsd|5.8.8|freebsd|7.0-release|200812091234
2786578|pass|200812|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|amd64-freebsd|5.8.8|freebsd|7.0-release|200812091332
2787491|pass|200812|srezic@cpan.org|File-Find-Rule-Permissions|1.3|amd64-freebsd|5.8.9|freebsd|7.0-release|200812091414
2787559|pass|200812|srezic@cpan.org|Palm-TreoPhoneCallDB|1.1|amd64-freebsd|5.8.9|freebsd|7.0-release|200812091415
2788027|pass|200812|srezic@cpan.org|Tie-STDOUT|1.03|amd64-freebsd|5.8.9|freebsd|7.0-release|200812091415
2788306|pass|200812|srezic@cpan.org|Devel-CheckLib|0.5|amd64-freebsd|5.8.8|freebsd|7.0-release|200812091137
2789178|pass|200812|srezic@cpan.org|Number-Phone|1.6|amd64-freebsd|5.8.8|freebsd|7.0-release|200812091232
2789378|pass|200812|srezic@cpan.org|Devel-CheckOS|1.50|amd64-freebsd|5.8.8|freebsd|7.0-release|200812091137
2789569|pass|200812|srezic@cpan.org|File-Find-Rule-Permissions|1.3|amd64-freebsd|5.8.8|freebsd|7.0-release|200812091145
2790751|pass|200812|srezic@cpan.org|Tie-STDOUT|1.03|amd64-freebsd|5.8.8|freebsd|7.0-release|200812091637
2791311|pass|200812|srezic@cpan.org|Data-Transactional|1.02|amd64-freebsd|5.8.8|freebsd|7.0-release|200812091130
2791699|pass|200812|srezic@cpan.org|Palm-TreoPhoneCallDB|1.1|amd64-freebsd|5.8.8|freebsd|7.0-release|200812091636
2791702|pass|200812|srezic@cpan.org|DBIx-Class-SingletonRows|0.11|amd64-freebsd|5.8.8|freebsd|7.0-release|200812091135
2793718|pass|200812|srezic@cpan.org|Net-Random|2.0|amd64-freebsd|5.8.8|freebsd|7.0-release|200812092201
2794124|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.11|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200812092252
2794647|pass|200812|srezic@cpan.org|Number-Phone-UK-DetailedLocations|1.3|amd64-freebsd|5.8.9|freebsd|7.0-release|200812092124
2795233|pass|200812|srezic@cpan.org|Number-Phone-UK-DetailedLocations|1.3|amd64-freebsd|5.8.8|freebsd|7.0-release|200812100006
2795306|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 35055|linux|2.6.22-1-k7|200812100101
2795329|pass|200812|srezic@cpan.org|Net-Random|2.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200812100013
2796282|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 35055|linux|2.6.22-1-k7|200812100345
2796290|pass|200812|srezic@cpan.org|XML-Tiny|1.11|amd64-freebsd|5.8.8|freebsd|7.0-release|200812100246
2796907|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 35055|linux|2.6.22-1-k7|200812100621
2798039|pass|200812|srezic@cpan.org|Games-Dice-Advanced|1.1|amd64-freebsd|5.8.9|freebsd|7.0-release|200812100855
2798287|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 35055|linux|2.6.22-1-k7|200812101018
2798388|pass|200812|srezic@cpan.org|Games-Dice-Advanced|1.1|amd64-freebsd|5.8.8|freebsd|7.0-release|200812101005
2800622|pass|200812|srezic@cpan.org|File-Overwrite|1.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200812101524
2800852|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i86pc-solaris-thread-multi-64int|5.8.9|solaris|2.10|200812101452
2800876|pass|200812|srezic@cpan.org|File-Overwrite|1.0|amd64-freebsd|5.8.8|freebsd|7.0-release|200812101524
2801403|pass|200812|fayland@gmail.com ("Fayland Lam")|Data-Compare|1.21|i386-linux|5.8.6|linux|2.6.9-34.elsmp|200812100817
2801494|pass|200812|srezic@cpan.org|Class-CanBeA|1.2|amd64-freebsd|5.8.8|freebsd|7.0-release|200812101628
2801623|pass|200812|stsc@refcnt.org ("Steven Schubiger")|Sort-MultipleFields|1.0|i686-linux|5.10.0|linux|2.6.18-4-686|200812101728
2802464|pass|200812|srezic@cpan.org|Class-CanBeA|1.2|amd64-freebsd|5.8.9|freebsd|7.0-release|200812101824
2804353|pass|200812|srezic@cpan.org|Palm-SMS|0.02|amd64-freebsd|5.8.9|freebsd|7.0-release|200812102246
2804681|pass|200812|srezic@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|amd64-freebsd|5.8.9|freebsd|7.0-release|200812102312
2804813|pass|200812|srezic@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|amd64-freebsd|5.8.8|freebsd|7.0-release|200812102313
2805174|pass|200812|srezic@cpan.org|Palm-SMS|0.02|amd64-freebsd|5.8.8|freebsd|7.0-release|200812102313
2805789|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.9|freebsd|6.2-release|200812110053
2806541|pass|200812|srezic@cpan.org|Devel-CheckOS|1.50|amd64-freebsd|5.10.0|freebsd|7.0-release|200812110753
2808229|pass|200812|srezic@cpan.org|Statistics-SerialCorrelation|1.1|amd64-freebsd|5.8.8|freebsd|7.0-release|200812111228
2808263|pass|200812|srezic@cpan.org|Statistics-SerialCorrelation|1.1|amd64-freebsd|5.8.9|freebsd|7.0-release|200812111228
2809562|pass|200812|srezic@cpan.org|Devel-CheckOS|1.50|amd64-freebsd-thread-multi|5.10.0|freebsd|7.0-release|200812111431
2809848|pass|200812|srezic@cpan.org|Tie-Hash-Longest|1.1|amd64-freebsd|5.8.8|freebsd|7.0-release|200812111619
2809851|pass|200812|srezic@cpan.org|Statistics-ChiSquare|0.5|amd64-freebsd|5.8.8|freebsd|7.0-release|200812111619
2809905|pass|200812|srezic@cpan.org|Statistics-ChiSquare|0.5|amd64-freebsd|5.8.9|freebsd|7.0-release|200812111619
2810112|pass|200812|srezic@cpan.org|Tie-Hash-Longest|1.1|amd64-freebsd|5.8.9|freebsd|7.0-release|200812111619
2811405|pass|200812|srezic@cpan.org|Devel-CheckOS|1.50|amd64-freebsd|5.6.2|freebsd|7.0-release|200812112151
2811883|pass|200812|srezic@cpan.org|NestedMap|1.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200812112229
2811921|pass|200812|srezic@cpan.org|NestedMap|1.0|amd64-freebsd|5.8.8|freebsd|7.0-release|200812112230
2812369|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-dragonfly-thread-multi-64int|5.8.8|dragonfly|2.0.0-release|200812112308
2812566|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-dragonfly-thread-multi-64int|5.10.0|dragonfly|2.0.0-release|200812112335
2813036|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200812120105
2813038|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200812120105
2814278|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int|5.10.0|linux|2.4.26|200812120728
2814386|unknown|200812|srezic@cpan.org|Acme-Licence|1.0|amd64-freebsd|5.8.8|freebsd|7.0-release|200812120840
2814813|unknown|200812|srezic@cpan.org|Acme-Licence|1.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200812120848
2817042|pass|200812|stsc@refcnt.org ("Steven Schubiger")|Data-Hexdumper|1.4|i686-linux|5.10.0|linux|2.6.18-4-686|200812121620
2817047|pass|200812|stsc@refcnt.org ("Steven Schubiger")|Palm-Treo680MessagesDB|1.0|i686-linux|5.10.0|linux|2.6.18-4-686|200812121621
2817048|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.10.0 patch 35072|linux|2.6.22-1-k7|200812121621
2817254|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.10.0 patch 35072|linux|2.6.22-1-k7|200812121634
2817255|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 35072|linux|2.6.22-1-k7|200812121637
2818742|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 35072|linux|2.6.22-1-k7|200812121927
2819533|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-dragonfly-thread-multi-64int|5.8.9|dragonfly|2.0.0-release|200812122018
2820611|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.2|200812130038
2820804|pass|200812|andy@hexten.net|Devel-CheckOS|1.50|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200812130117
2821002|pass|200812|srezic@cpan.org|Tie-Hash-Rank|1.0.1|amd64-freebsd|5.8.9|freebsd|7.0-release|200812130246
2821016|pass|200812|srezic@cpan.org|Tie-Hash-Rank|1.0.1|amd64-freebsd|5.8.8|freebsd|7.0-release|200812130246
2821110|pass|200812|srezic@cpan.org|Tie-Scalar-Decay|1.1.1|amd64-freebsd|5.8.9|freebsd|7.0-release|200812130257
2821122|pass|200812|srezic@cpan.org|Tie-Scalar-Decay|1.1.1|amd64-freebsd|5.8.8|freebsd|7.0-release|200812130257
2821229|unknown|200812|srezic@cpan.org|Acme-Pony|1.1.2|amd64-freebsd|5.8.9|freebsd|7.0-release|200812130246
2821240|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200812130223
2821344|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|darwin-thread-multi-64int|5.6.2|darwin|8.11.0|200812130233
2821357|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200812130239
2821406|unknown|200812|srezic@cpan.org|Acme-Pony|1.1.2|amd64-freebsd|5.8.8|freebsd|7.0-release|200812130246
2821425|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200812130252
2822282|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.11|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.17.el5|200812130920
2823527|fail|200812|srezic@cpan.org|Class-DBI-ClassGenerator|1.01|amd64-freebsd|5.8.9|freebsd|7.0-release|200812131152
2823660|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 35072|linux|2.6.22-1-k7|200812131155
2824656|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0 patch 35072|linux|2.6.22-1-k7|200812131413
2828016|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.21.5-smp|200812140459
2828047|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.11|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200812140608
2829387|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-dragonfly-thread-multi-64int|5.8.8|dragonfly|2.0.1-release|200812141129
2829396|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-dragonfly-thread-multi-64int|5.10.0|dragonfly|2.0.1-release|200812141130
2830115|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.11.0 patch 35088|linux|2.6.22-1-k7|200812141423
2830148|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 35088|linux|2.6.22-1-k7|200812141427
2830281|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|darwin-2level|5.10.0|darwin|8.11.1|200812141408
2830300|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|darwin-2level|5.10.0|darwin|8.11.1|200812141411
2831034|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 35088|linux|2.6.22-1-k7|200812141557
2831500|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 35088|linux|2.6.22-1-k7|200812141707
2832569|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 35088|linux|2.6.22-1-k7|200812141947
2832650|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200812141901
2832930|pass|200812|stsc@refcnt.org ("Steven Schubiger")|Data-Transactional|1.02|i686-linux|5.10.0|linux|2.6.18-4-686|200812142100
2833867|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 35088|linux|2.6.22-1-k7|200812142334
2835370|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-dragonfly-thread-multi-64int|5.8.9|dragonfly|2.0.1-release|200812150522
2836735|pass|200812|stsc@refcnt.org ("Steven Schubiger")|CPU-Emulator-Memory|1.1001|i686-linux|5.10.0|linux|2.6.18-4-686|200812151501
2836740|pass|200812|stsc@refcnt.org ("Steven Schubiger")|CPU-Emulator-Z80|1.0|i686-linux|5.10.0|linux|2.6.18-4-686|200812151502
2839840|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.6.2|freebsd|7.0-release|200812160329
2843801|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200812161611
2844746|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200812161906
2845107|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200812161942
2845633|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200812162048
2845878|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200812162120
2845909|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200812162124
2846388|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200812162214
2846671|pass|200812|stro@cpan.org|DBIx-Class-SingletonRows|0.11|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200812162354
2847277|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200812162242
2847950|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0|200812170001
2849222|pass|200812|carlos.arenas@gmail.com ("Carlos Arenas")|Devel-CheckOS|1.50|i686-linux-thread-multi|5.10.0|linux|2.6.25-arch|200812161530
2849275|pass|200812|wolfman.ncsu2000@gmail.com|Devel-CheckOS|1.50|darwin-2level|5.10.0|darwin|9.5.0|200812161800
2849289|pass|200812|wolfman.ncsu2000@gmail.com|Data-Compare|1.21|darwin-2level|5.10.0|darwin|9.5.0|200812161813
2849440|pass|200812|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.50|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812170143
2851400|pass|200812|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812170447
2853227|pass|200812|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.21|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812170252
2853285|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux-thread-multi|5.8.9|linux|2.6.16.60-0.31-default|200812170858
2854457|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux-thread-multi|5.8.9|linux|2.6.18-92.1.18.el5|200812171144
2854842|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200812171258
2854913|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200812171312
2855399|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux|5.8.9|linux|2.6.18-92.1.18.el5|200812171405
2856215|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200812171459
2856409|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200812171511
2856657|pass|200812|stsc@refcnt.org ("Steven Schubiger")|Number-Phone|1.6|i686-linux|5.10.0|linux|2.6.18-4-686|200812171543
2857148|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux|5.8.9|linux|2.6.16.60-0.31-default|200812171628
2857770|pass|200812|rhaen@cpan.org (Ulrich Habel)|Devel-CheckOS|1.50|i386-netbsd|5.10.0|netbsd|5.0_beta|200812171618
2857853|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200812171615
2858912|pass|200812|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812171346
2859596|pass|200812|dagolden@cpan.org (DAGOLDEN)|Class-DBI-ClassGenerator|1.01|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812171524
2861126|pass|200812|dagolden@cpan.org (DAGOLDEN)|Sort-MultipleFields|1.0|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812180001
2861439|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Sort-MultipleFields|1.0|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200812180152
2861591|pass|200812|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812180115
2861592|pass|200812|dagolden@cpan.org (DAGOLDEN)|Palm-Treo680MessagesDB|1.0|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812180115
2861966|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.6.2|freebsd|7.0-release|200812180150
2862222|pass|200812|dagolden@cpan.org (DAGOLDEN)|DBIx-Class-SingletonRows|0.11|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812180223
2862505|pass|200812|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.02|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812180258
2862575|pass|200812|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812180311
2862576|pass|200812|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Z80|1.0|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812180311
2862849|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Sort-MultipleFields|1.0|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200812180450
2862970|pass|200812|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812180409
2863975|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Sort-MultipleFields|1.0|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200812180758
2864299|pass|200812|bingos@cpan.org|Data-Compare|1.21|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200812180750
2864423|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200812180724
2864465|pass|200812|bingos@cpan.org|Data-Compare|1.21|darwin-thread-multi-64int|5.6.2|darwin|8.11.0|200812180809
2864474|pass|200812|bingos@cpan.org|Data-Compare|1.21|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200812180809
2864717|pass|200812|bingos@cpan.org|Data-Compare|1.21|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200812180844
2864805|pass|200812|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812180358
2865836|pass|200812|vutral@gmail.com|Data-Compare|1.21|x86_64-linux-gnu|5.8.8|linux|2.6.26-1-vserver-amd64|200812180200
2866826|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200812181205
2867138|pass|200812|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.0|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812180825
2867688|pass|200812|dagolden@cpan.org (DAGOLDEN)|Bryar|3.0|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812180954
2867800|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0.1|200812181506
2868546|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux|5.8.9|linux|2.6.18-92.1.18.el5|200812181749
2868648|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux-thread-multi|5.8.9|linux|2.6.18-92.1.18.el5|200812181801
2868649|pass|200812|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|1.3|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812181203
2868906|pass|200812|srezic@cpan.org|Data-Compare|1.21|amd64-freebsd-thread-multi|5.10.0|freebsd|7.0-release|200812181830
2869146|pass|200812|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812181305
2869551|pass|200812|dagolden@cpan.org (DAGOLDEN)|Palm-TreoPhoneCallDB|1.1|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812181348
2870198|pass|200812|imacat@mail.imacat.idv.tw|Data-Compare|1.21|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812190343
2870238|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200812182045
2870415|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200812182102
2870523|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200812182112
2870612|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux|5.8.9|linux|2.6.16.60-0.31-default|200812182118
2870726|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200812182127
2870795|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux|5.8.9|linux|2.6.18-92.1.18.el5|200812182132
2870944|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux-thread-multi|5.8.9|linux|2.6.18-92.1.18.el5|200812182149
2871153|pass|200812|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200812182009
2871313|pass|200812|dagolden@cpan.org (DAGOLDEN)|Net-Random|2.0|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812181620
2871351|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux-thread-multi|5.8.9|linux|2.6.16.60-0.31-default|200812182222
2871924|pass|200812|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812181711
1883728|pass|200807|mw487.yahoo.com@gmail.com|Data-Hexdumper|1.4|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200807120916
1896279|pass|200807|mw487.yahoo.com@gmail.com|Devel-CheckOS|1.43|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200807151949
1973625|pass|200808|mw487.yahoo.com@gmail.com|XML-Tiny|1.11|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200808020125
2498600|pass|200810|mw487.yahoo.com@gmail.com|Devel-CheckOS|1.44|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200810260600
2315664|pass|200809|jeff@zeroclue.com|Data-Compare|1.21|darwin-2level|5.10.0|darwin|9.2.0|200809261459
2315682|pass|200809|jeff@zeroclue.com|XML-DoubleEncodedEntities|1.0|darwin-2level|5.10.0|darwin|9.2.0|200809261539
2316178|pass|200809|jeff@zeroclue.com|Devel-CheckOS|1.43|darwin-2level|5.10.0|darwin|9.2.0|200809261645
2348696|fail|200809|jeff@zeroclue.com|Class-DBI-ClassGenerator|1.0|darwin-2level|5.10.0|darwin|9.2.0|200809300310
2637111|pass|200811|jeff@zeroclue.com|Data-Compare|1.21|darwin-2level|5.10.0|darwin|9.2.0|200811161337
2637310|pass|200811|jeff@zeroclue.com|Devel-CheckOS|1.50|darwin-2level|5.10.0|darwin|9.2.0|200811161412
2637720|pass|200811|jeff@zeroclue.com|Devel-CheckOS|1.49_01|darwin-2level|5.10.0|darwin|9.2.0|200811161554
2873201|pass|200812|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0.1|200812190000
2873297|pass|200812|srezic@cpan.org|Data-Compare|1.21|amd64-freebsd|5.6.2|freebsd|7.0-release|200812190105
2873610|pass|200812|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.50|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812190833
2874399|pass|200812|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812190150
2874665|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200812190220
2874953|pass|200812|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812190259
2876360|pass|200812|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.0|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812190134
2876685|pass|200812|dagolden@cpan.org (DAGOLDEN)|Class-CanBeA|1.2|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812190216
2877419|pass|200812|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200812190843
2879660|fail|200812|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.02|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812190819
2879718|pass|200812|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812190830
2880250|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux|5.8.9|linux|2.6.16.60-0.31-default|200812191538
2881650|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0.1|200812191723
2881720|pass|200812|dagolden@cpan.org (DAGOLDEN)|Statistics-SerialCorrelation|1.1|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812191233
2882111|pass|200812|dagolden@cpan.org (DAGOLDEN)|Statistics-ChiSquare|0.5|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812191310
2882117|pass|200812|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Longest|1.1|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812191310
2882982|pass|200812|dagolden@cpan.org (DAGOLDEN)|NestedMap|1.0|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812191448
2883908|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200812192235
2883928|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200812192238
2884187|pass|200812|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0.1|200812192209
2885389|unknown|200812|dagolden@cpan.org (DAGOLDEN)|Acme-Licence|1.0|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812200009
2885630|pass|200812|rhaen@cpan.org (Ulrich Habel)|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi|5.10.0|netbsd|5.0_beta|200812200028
2886280|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200812200127
2886887|unknown|200812|dagolden@cpan.org (DAGOLDEN)|Acme-Pony|1.1.2|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812200250
2886959|pass|200812|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Rank|1.0.1|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812200256
2887031|pass|200812|dagolden@cpan.org (DAGOLDEN)|Tie-Scalar-Decay|1.1.1|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812200305
2887113|pass|200812|imacat@mail.imacat.idv.tw|Number-Phone|1.6|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812201113
2887212|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200812200322
2891001|pass|200812|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812200928
2891047|pass|200812|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812200931
2892048|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200812201605
2892266|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200812201627
2892522|pass|200812|bingos@cpan.org|Data-Compare|1.21|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200812201714
2893169|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200812201808
2893502|pass|200812|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.6.2|netbsd|4.0|200812201849
2894325|pass|200812|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812201534
2895545|pass|200812|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200812202304
2895875|pass|200812|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200812202348
2896124|pass|200812|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200812210021
2897833|pass|200812|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812210048
2898774|pass|200812|dagolden@cpan.org (DAGOLDEN)|Class-DBI-ClassGenerator|1.01|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812210302
2900393|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200812211106
2900660|pass|200812|dagolden@cpan.org (DAGOLDEN)|Sort-MultipleFields|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812210640
2900706|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.16.38|200812211142
2901362|pass|200812|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812210758
2901366|pass|200812|dagolden@cpan.org (DAGOLDEN)|Palm-Treo680MessagesDB|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812210758
2902320|pass|200812|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200812211533
2902335|pass|200812|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.02|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812211037
2902449|pass|200812|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812211052
2902457|pass|200812|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Z80|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812211052
2902634|pass|200812|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.16.38|200812211616
2903009|pass|200812|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812211211
2904781|pass|200812|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812211756
2905324|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0|200812220006
2906678|pass|200812|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812220428
2906994|pass|200812|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0|200812220523
2907217|pass|200812|dagolden@cpan.org (DAGOLDEN)|Bryar|3.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812220121
2908320|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|1.4|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200812220925
2908326|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.0|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200812220926
2908670|pass|200812|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|1.3|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812220416
2909264|pass|200812|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812220542
2909708|pass|200812|dagolden@cpan.org (DAGOLDEN)|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812220645
2910034|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200812221231
2910076|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200812221239
2911775|pass|200812|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200812221903
2911789|pass|200812|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200812221905
2912851|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Transactional|1.02|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200812222326
2913208|pass|200812|dagolden@cpan.org (DAGOLDEN)|Net-Random|2.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812221817
2913520|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Memory|1.1001|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200812230056
2913521|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Z80|1.0|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200812230057
2913601|pass|200812|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812230007
2913921|pass|200812|rhaen@cpan.org (Ulrich Habel)|Data-Compare|1.21|i386-netbsd-thread-multi|5.10.0|netbsd|5.0_beta|200812230056
2914905|pass|200812|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812230350
2915226|pass|200812|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812230458
2915479|pass|200812|bingos@cpan.org|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200812230552
2915507|pass|200812|bingos@cpan.org|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200812230557
2916888|pass|200812|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812230351
2917138|pass|200812|dagolden@cpan.org (DAGOLDEN)|Class-CanBeA|1.2|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812230432
2918558|unknown|200812|imacat@mail.imacat.idv.tw|Acme-Licence|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232145
2918561|unknown|200812|imacat@mail.imacat.idv.tw|Acme-Pony|1.1.2|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232145
2918563|pass|200812|imacat@mail.imacat.idv.tw|Acme-Scurvy-Whoreson-BilgeRat|1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232146
2918583|fail|200812|imacat@mail.imacat.idv.tw|Bryar|3.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232149
2918586|pass|200812|imacat@mail.imacat.idv.tw|CPAN-FindDependencies|2.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232150
2918589|pass|200812|imacat@mail.imacat.idv.tw|CPU-Emulator-Memory|1.1001|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232150
2918601|pass|200812|imacat@mail.imacat.idv.tw|CPU-Emulator-Z80|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232151
2918602|pass|200812|imacat@mail.imacat.idv.tw|Class-CanBeA|1.2|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232152
2918628|pass|200812|imacat@mail.imacat.idv.tw|Data-Hexdumper|1.4|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232157
2918631|pass|200812|imacat@mail.imacat.idv.tw|Data-Transactional|1.02|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232158
2918634|pass|200812|imacat@mail.imacat.idv.tw|Devel-CheckLib|0.5|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232158
2918638|pass|200812|imacat@mail.imacat.idv.tw|File-Find-Rule-Permissions|1.3|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232159
2918643|pass|200812|imacat@mail.imacat.idv.tw|File-Overwrite|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232159
2918646|pass|200812|imacat@mail.imacat.idv.tw|Games-Dice-Advanced|1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232159
2918650|pass|200812|imacat@mail.imacat.idv.tw|NestedMap|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232200
2918661|pass|200812|imacat@mail.imacat.idv.tw|Net-Random|2.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232201
2918668|pass|200812|imacat@mail.imacat.idv.tw|Number-Phone-UK-DetailedLocations|1.3|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232202
2918680|pass|200812|imacat@mail.imacat.idv.tw|Palm-Treo680MessagesDB|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232204
2918697|pass|200812|imacat@mail.imacat.idv.tw|Palm-TreoPhoneCallDB|1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232206
2918699|pass|200812|imacat@mail.imacat.idv.tw|Sort-MultipleFields|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232206
2918700|pass|200812|imacat@mail.imacat.idv.tw|Statistics-ChiSquare|0.5|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232206
2918704|pass|200812|imacat@mail.imacat.idv.tw|Statistics-SerialCorrelation|1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232207
2918710|pass|200812|imacat@mail.imacat.idv.tw|Sub-WrapPackages|1.2|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232207
2918712|pass|200812|imacat@mail.imacat.idv.tw|Tie-Hash-Longest|1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232207
2918716|pass|200812|imacat@mail.imacat.idv.tw|Tie-Hash-Rank|1.0.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232208
2918718|pass|200812|imacat@mail.imacat.idv.tw|Tie-STDOUT|1.03|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232208
2918720|pass|200812|imacat@mail.imacat.idv.tw|Tie-Scalar-Decay|1.1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232209
2918721|pass|200812|imacat@mail.imacat.idv.tw|XML-DoubleEncodedEntities|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232209
2918722|pass|200812|imacat@mail.imacat.idv.tw|XML-Tiny|1.11|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232209
2919291|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.9|freebsd|6.2-release|200812231502
2919546|fail|200812|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.02|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812231042
2919615|pass|200812|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812231050
2921115|pass|200812|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.8.9|freebsd|6.2-release|200812231853
2921664|pass|200812|dagolden@cpan.org (DAGOLDEN)|Statistics-SerialCorrelation|1.1|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812231501
2921955|pass|200812|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Longest|1.1|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812231537
2921957|pass|200812|dagolden@cpan.org (DAGOLDEN)|Statistics-ChiSquare|0.5|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812231537
2922125|pass|200812|bingos@cpan.org|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.3|200812232106
2922744|pass|200812|dagolden@cpan.org (DAGOLDEN)|NestedMap|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812231759
2923002|pass|200812|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int|5.10.0|linux|2.4.26|200812232325
2923611|unknown|200812|dagolden@cpan.org (DAGOLDEN)|Acme-Licence|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812240212
2924466|unknown|200812|dagolden@cpan.org (DAGOLDEN)|Acme-Pony|1.1.2|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812240452
2924504|pass|200812|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Rank|1.0.1|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812240457
2924535|pass|200812|dagolden@cpan.org (DAGOLDEN)|Tie-Scalar-Decay|1.1.1|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812240004
2926641|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.3|200812240957
2926853|pass|200812|chris@bingosnet.co.uk|Data-Compare|1.21|i686-linux-thread-multi-64int|5.6.2|linux|2.4.26|200812241109
2928116|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200812241554
2929091|pass|200812|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200812241937
2929486|pass|200812|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200812242118
2929897|pass|200812|bingos@cpan.org|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.4|200812250031
2929898|pass|200812|bingos@cpan.org|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.4|200812250031
2930317|pass|200812|yanick@cpan.org ("Yanick Champoux")|Devel-CheckOS|1.50|i686-linux-thread-multi|5.10.0|linux|2.6.22-gentoo-r2enkidu|200812241237
2933025|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.4|200812251348
2933026|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.4|200812251348
2933659|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200812252043
2938494|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.4|200812262114
2939739|pass|200812|davidrw@cpan.org|XML-DoubleEncodedEntities|1.0|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200812270346
2939983|pass|200812|stsc@refcnt.org ("Steven Schubiger")|Devel-CheckLib|0.5|i686-linux|5.10.0|linux|2.6.18-4-686|200812270958
2940826|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.11|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200812271408
2943274|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0|linux|2.6.22-1-k7|200812272317
2943341|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.11.0|linux|2.6.22-1-k7|200812272329
2947073|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0|linux|2.6.22-1-k7|200812280846
2947328|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0|linux|2.6.22-1-k7|200812280915
2948935|pass|200812|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812281405
2949187|pass|200812|bingos@cpan.org|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.2|200812281409
2949939|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Sort-MultipleFields|1.0|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200812281750
2950457|pass|200812|bingos@cpan.org|Number-Phone|1.6|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812282054
2951440|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.11|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200812290139
2953740|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|1.4|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200812291511
2953742|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.0|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200812291512
2953993|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|1.4|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200812291627
2953998|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.0|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200812291628
2955398|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.6|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200812292207
2956271|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.11|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200812300130
2956422|pass|200812|bingos@cpan.org|Acme-Licence|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812300215
2956428|pass|200812|bingos@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812300215
2956429|pass|200812|bingos@cpan.org|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812300216
2956431|pass|200812|bingos@cpan.org|Class-DBI-ClassGenerator|1.01|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812300218
2956432|pass|200812|bingos@cpan.org|CPU-Emulator-Z80|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812300217
2956452|pass|200812|bingos@cpan.org|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812300223
2956453|pass|200812|bingos@cpan.org|Tie-STDOUT|1.03|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812300223
2956454|pass|200812|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812300224
2956456|pass|200812|bingos@cpan.org|Statistics-SerialCorrelation|1.1|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812300223
2956457|pass|200812|bingos@cpan.org|Net-Random|2.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812300221
2956461|pass|200812|bingos@cpan.org|Net-Random|2.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812300221
2956463|pass|200812|bingos@cpan.org|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812300222
2956465|pass|200812|bingos@cpan.org|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812300218
2956474|pass|200812|bingos@cpan.org|Sort-MultipleFields|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812300222
2956487|pass|200812|bingos@cpan.org|Number-Phone-UK-DetailedLocations|1.3|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812300221
2956494|pass|200812|bingos@cpan.org|Statistics-ChiSquare|0.5|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812300223
2956495|pass|200812|bingos@cpan.org|Data-Transactional|1.02|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812300219
2956497|pass|200812|bingos@cpan.org|File-Find-Rule-Permissions|1.3|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812300219
2957197|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|DBIx-Class-SingletonRows|0.11|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200812300611
2962681|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Transactional|1.02|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200812311512
2963147|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Memory|1.1001|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200812311739
2963177|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Z80|1.0|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200812311740
2963351|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Transactional|1.02|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200812311746
2963683|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Sort-MultipleFields|1.0|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200812311840
2964004|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Memory|1.1001|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200812312003
2964005|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Z80|1.0|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200812312004
2964154|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Sort-MultipleFields|1.0|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200812312109
2964386|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Sort-MultipleFields|1.0|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200812312216
2965522|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.6|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200901010245
2966094|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.6|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200901010440
2970431|pass|200812|schwern@pobox.com|Data-Compare|