#!/usr/bin/perl
#Copyright 2007-2008, Bastian Angerstein.  All rights reserved.  This program is free
#software; you can redistribute it and/or modify it under the same terms as
#PERL itself.

package Net::Ping::Network;

# infrastructure requirements
require 5.008_008;	

use strict;

use threads;
use threads::shared;
use Thread::Queue;
use Exporter;

use Config;
$Config{useithreads} or die "Recompile Perl with threads to run this program.";

use Net::Ping::External qw(ping);

our $VERSION = '1.55';
our @EXPORT = qw(new doping calchosts listAllHost results);
our @ISA = qw (Exporter);
our %REGISTRY;
our $verbose = 1;

my $DataQueue = Thread::Queue->new; # a shared Queue Object
my %results : shared;     # a shared $hash

sub AUTOLOAD {
   our $AUTOLOAD;
   (my $method = $AUTOLOAD) =~ s/.*:://s;
}


sub new {

    # this is the constructor of net::ping::networks.
    my $class = shift; # read the Name of our Class
    my $net = undef;   # initialize a var for our net
    my $mask = undef;  # initialize a var for our mask
    my @hostlist = ();  # initialize an array to contain a list of user given hosts

    if (ref $_[0]) {   # if we where called with a ref, we expect this to be an ref of array containing ips
        @hostlist = @{$_[0]}; # a user specified list of all hosts to ping given as reference to an array
    } else { # if we dont get a ref, we expect regular usage with a given netwrok and a mask
        $net = shift;
        $mask = shift;
    }

    my $timeout = shift; #expect an optional timeout in seconds
    $timeout = defined($timeout)?$timeout:3;   # no timeout specified? default to 3
    my $retries = shift;
    $retries = defined($retries)?$retries:3;  # no retries specified? default to 3
    my $threads = shift;
    $threads = defined($threads)?$threads:10; #no amount of threads specified? default to 10

    my ($self) = {       # Building our Objecthash
        NET => $net, # Base Adress
        MASK => $mask,    # Netmask
        TIMEOUT => $timeout, #Max. Timeout ins seconds
        RETRIES => $retries, #Max. Retries
        TC => $threads, #Max. Threads
        TJ => 0, #Joinable Threads
        TR => 0, #Running Threads
        #VERBOSE => 0, # Debugging
        HOSTS => 0,   # Number of Hosts
        SUMOFHOSTS => 0,  # Sum of all Hosts
        RESULTS => 0,
        CONF_PING => \&conf_ping, # A Code-Ref need for threading
    };

    if ( @hostlist ){ #if we received a list of hosts from the user
      @{ $self->{ 'HOSTLIST' } } = @hostlist;
    } else {
      @{ $self->{ 'HOSTLIST' } } = ();  
    }
    $self->{ 'ALLHOSTS' } = (); # for a autogenerated the list of all hosts to ping

    $REGISTRY{$self} = $self;
    bless ($self, $class);
    return ($self);
}

################################################################################
#sub verbose ($$) { # Only a poor Debugging Sub
#  my ($self) = shift;
#  my @output = shift;
#  print @output if ( $self->{'VERBOSE'} );
#}

sub verbose ($) { # Only a poor Debugging Sub
  my @output = shift;
  print @output if ( $verbose );
}
################################################################################
sub setHosts{ # Hand a List of Hosts by Yourself.
  my ($self) = shift;
  print @_;
  @{$self->{'HOSTLIST'}} = @_;

  return ($self);
}

################################################################################
sub calchosts { # Berechnet anhand der Maske die Anzahl der Mglichen Hosts in einem Netz.
    #Die Broadcastadress ist kein mglicher Hosts.
    #Die Netzbasisadresse wird ebenso entfernt.
   my ($self) = shift;
   my $lmask;  #get the mask

   if ( ref ($self) ) {
     if ( ${ $self->{'HOSTLIST'} }[0]  ) {  # if there is a userdefined list of hosts, return the amount of hosts found
        return  scalar ( @{ $self->{'HOSTLIST'} } );
     }
     $lmask = $self->{MASK};
   } elsif ($self) {
      if ($self >= 0 && $self <= 32) { 
        $lmask = $self;
      } else {
        die "No useable netmask found: $self is not a netmask.\n";
      }
   } else {
      print STDERR "A parameter is missing.";
   }

    my $bits = 32 - $lmask; # Calculate the amount of bits in the host section of the mask
    my $pO2 = (2 ** $bits) -2; # substract net and broadcast address
    if ($pO2 < 1) {
        $pO2=1;
    }
    if (ref $self ){
      $self->{'HOSTS'} = $pO2;
    } else {
      return $pO2;
    }
}

################################################################################
sub listAllHost { # List all possible host of a net or all host received from user.
	#  expects a network address and a mask
	# or expects that net::ping::networks has received a list of hosts
    my ($self) = shift;
    
    my $net = undef; #Net like 127.0.0.0
    my $mask = undef; #Mask like 24.

    if ( ref ($self) ) {
      if ( ${ $self->{'HOSTLIST'} }[0]  ) {
          return wantarray ? @{ $self->{'HOSTLIST'} }: join(" ",@{ $self->{'HOSTLIST'} });
      }
      $mask = $self->{'MASK'};
      $net = $self->{'NET'};
   } else {
      $net = $self;
      $mask = shift;
   }
    die "Missing parameters listAllHost" unless ( defined $net && defined $mask);

    my @allHosts; # Array fr die Liste aller Hosts
    my @net_p = split(/\./, $net ); # Zerlegung der Dottet-Decimal IP

    my $sumOfHosts = calchosts( $mask ); #Ermittle die Anzahl der mglichen Hosts

    if ( ref ($self) ) {
      $self->{'SUMOFHOSTS'} = $sumOfHosts;
    }

    my $i = 1; #Counter/Itterator
    while ($i <= $sumOfHosts ) { # Solange wie Counter kleiner Anzahl der Hosts ist
        $net_p[3]++;            # Inkrementiere letzten Abschnitt der IP
        if ($net_p[3] > 255){   # Wenn der letzte Abschnitt nun eine hhreren Wert hat als 255
            $net_p[2]++;        # Inkrementiere den vorletzten Abschitt.
            $net_p[3] = 0;      # und setze den vierten Abschnitt auf 0
        }
        if ($net_p[2] > 255){   # Wenn der dritte Abschnitt nun grer ist als 255
            $net_p[1]++;        # inkrementiere den zweiten Abschnitten
            $net_p[2] = 0;      # und setze den dritte Abschnitt auf 0
        }
        if ($net_p[1] > 255){   # Wenn der zweite Abschnitt...
            $net_p[0]++;
            $net_p[1] = 0;
        }
        if ($net_p[0] > 255){   # Wenn der erste Abschnitt grer als 255 ist
            die "Out of IP-Range"; # Sterbe und gebe Out of IP-Range.
        }
        my $ip = join(".",@net_p); #fge die Abschnitte zu einem String zusammen.
        push (@allHosts,$ip);   # Sammle alle Strings
        $i++;                   #inkrementiere Counter
    } #while

    if ( ref ($self) ) {
      $self->{'ALLHOSTS'} = @allHosts;
    }

    return wantarray ? @allHosts : join(" ",@allHosts); #if wantarray 1 then @
                                                   #if wantarray 0 dann $
}

################################################################################
sub conf_ping {
    # Thread-Sub which does the pinging
    my ($self) = shift;
    verbose ( $self . " thread\n" );
    my $thr = threads->self; #Der thread selbst
    my $tid = $thr->tid; # Die ID des Threads

    verbose "$tid has started.\n"; # Thread-ID Status mit.

    while ( my $host = $DataQueue->dequeue_nb ) { # nonblocking dequeuen of an address.
        verbose( "$tid is working.\n" ); #Debugging
        if( ping ( host => "$host",  count => $self->{RETRIES}, timeout => $self->{TIMEOUT} )){ # Den Host pingen
            verbose ("$host is alive.\n");
            $results{$host}  = 1;              # Good
        } else {
            verbose ( "$host is unreachable!\n" );
            $results{$host}  = 0;           #Bad
        }
         $thr->yield;                          # Be gentle
    }
    verbose ("$tid is done.\n");
}

################################################################################

sub doping {

    my ($self) = shift;
    %results = ();
    verbose ( @{ $self->{ 'HOSTLIST' } } );

    if ( @{ $self->{ 'HOSTLIST' } } ){ # If User provides a List of Hosts
         $DataQueue->enqueue ( @{ $self->{ 'HOSTLIST' } } );
    } else {
      $DataQueue->enqueue ( listAllHost($self->{'NET'}, $self->{'MASK'}) ); # Build and Enqueue a list of hosts to ping.
    }
    verbose ( "Main: StartingUp" . $self->{'TC'} . "Threads.\n" );
    for (my $i=0; $i < $self->{'TC'}; $i++){
      $self->{ $i } = 0;
      $self->{ $i } = threads->new({'context' => 'list'}, $self->{CONF_PING}, $self); ##############
      select(undef, undef, undef, 0.05);   # take a napp
      if ($self->{ $i }->error) {
        print "Main: Error:" . $self->{ $i }->error . "\n";
      }
         verbose ("Main: $i Threads have been initialized.\n");
    }
    verbose ( "Main: StartUp-Sequence of" . $self->{'TC'} . "Threads completed.\n");

    while ( threads->list(threads::running) or threads->list(threads::joinable ) ) {
         my @joinable = threads->list(threads::joinable); #Check for finished Threads
         $self->{'TJ'} = scalar (@joinable);                     #Get Amount of Finished Threads
         $self->{'TR'} = threads->list(threads::running);        #Check for running Threads
         verbose ( "Main: Queued Items = " . $DataQueue->pending . ".\nJoinable Threads = " . $self->{'TJ'} . " Running Threads = " . $self->{'TR'} . ".\n"); #Give a Process Status
         foreach my $t (@joinable) {
            $t->join;
         }
         select(undef, undef, undef, 0.05);     # be gentle
    }
    verbose ( %results );
    $self->{RESULTS} = \%results;
}

################################################################################
sub results {
    my ($self) = shift;
    return $self->{RESULTS};
}
1;

=head1 NAME
Net::Ping::Network  - A modul to ICMP-request nodes in networks very fast

=head1 SYNOPSIS

Import Net-Ping-Network and use the original Interface.
Simply give a network address and a mask to the constructor new().

    use Net::Ping::Network;
    my $net = Net::Ping::Network->new("127.0.0.0", 29);


Optionally the timeout in seconds (3), the amount of retries (3) and
the number of threads utilized (10) can be specified.

    my $net = Net::Ping::Network->new("127.0.0.0", 29, $timeout, $retries, $threads);


To ping the hosts in the network use the doping() methode of your Net::Ping::Network methode.
When Net::Ping::Network is done, you can get the results as hashref using the methode results().


    $net->doping();
    $results = $net->results();


The hashkeys are the ips the value is 1 for reachable, 0 for unreachable.


The hash is not sorted in anyway, to sort a hash is useless.
If you need sorted results try this:

1. get the Keys from the retruned hashref (ips).

    my @unsorted_keys = keys %$results;

2. using a sort over the packed data. This is much fast then sort by every field.

    my @keys = sort { # sort list of ips accending
     pack('C4' => $a =~ /(\d+)\.(\d+)\.(\d+)\.(\d+)/)
     cmp pack('C4' => $b =~ /(\d+)\.(\d+)\.(\d+)\.(\d+)/) } @unsorted_keys;


    foreach my $key ( @keys ) {
        print "$key" . " is ";
        if ( $$h{"$key"} ) {
          print  "alive.\n";
        } else {
          print "unreachable!\n";
        }
    }

A list of all hosts to ping, can be gathered form the methode listAllHost()

    my  @all = $net->listAllHost();
    my $list = $net->listAllHost();

In list context listAllHost returns an array containing all hosts to ping.
In scalar context this moethode returns a whitespace separeted string.

If you need the number of Host for a given netmask use
  my $x = $net->calchosts(22);
or
  my $y = calchost(22);

calchosts() calculates the max. number of host in a network with the given mask.
The broadcast address is not a possible host, the network base address ist not a possible host.



=head2 DESCRIPTION


The existing ping moduls are slow and can only handle one ping at a time.
Net::Ping::Networks (Net::Ping::Multi) can handle even large ip ranges and entire networks.
Depending of your computing power and memory you can scan a class c network in less then 5 seconds.

Threads are utilised to boost performace. Threads feel a still a little bit beta today.

=head2 Methodes

=over 1

=item C<new()>

creates a new Net::Ping::Network instance. Needs a network base address and netmask or an array of ips to ping.
If a network base address and a mask is supplied, Net::Ping::Networks will build a List of all host-ips in the net
automaticaly.

C<< $n = Net::Ping::Network->new("127.0.0.0", 29, [$timeout, $retries, $threads]); >>


=item C<listAllHost()>

depending on the context it returns a list containig all possible Hosts the network or a whitespace seperated string.


=item C<doping()>

executes the configured ping utilising the given parameters.
As lower the amount auf pings per threads is, as faster the methode will return.

=item C<calchosts()>

Calculates the amount of possible hosts for a Netmask, value between 0 and 32 is expected.
Network-Address and Broadcast is removed, but a /32 has 1 Address. 

=item C<results()>

Returns a Hashref of the Results. Keys are IPs, the Values are returncodes (0 for bad or 1 for ok).


=back

=head1 COPYRIGHT

Copyright 2007-2008, Bastian Angerstein.  All rights reserved.  This program is free
software; you can redistribute it and/or modify it under the same terms as
PERL itself.

=head1 AVAILABILITY

=head1 CAVEATS

Threads are cpu and memory intensive and feel still beta. Have an Eye on memory leaks.
Net::Ping::Networks is a quick and dirty but easy to read and understand implementation.
Documentation is in the Code.

Also it "could" lead into trouble to use a multithreaded modul in a multithreaded environment.

=head1 AUTHOR

Bastian Angerstein - L<http://cul.de/>

=head1 SEE ALSO

L<net::ping>, L<net::ping::external>

=cut

