=head1 NAME

Data::ID::Maildir - generate maildir-style unique IDs

=head1 SYNOPSIS

	use Data::ID::Maildir qw(maildir_id);

	$id = maildir_id;

=head1 DESCRIPTION

This module exports one function, C<maildir_id>, which generates
maildir-style unique IDs.

=cut

package Data::ID::Maildir;

use warnings;
use strict;

use Exporter;
use Sys::Hostname qw(hostname);
use Time::HiRes 1.00 qw(gettimeofday);

our $VERSION = "0.001";

our @ISA = qw(Exporter);

our @EXPORT_OK = qw(maildir_id);

=head1 FUNCTIONS

=over

=item maildir_id

=item maildir_id(HOSTNAME)

Generates a maildir-style unique ID.  The ID is based on the time,
process ID, and hostname; it is guaranteed to be unique among IDs
generated by this algorithm provided the hostname is unique among hosts
using this algorithm.  The hostname may be specified to the function if
it is necessary to override it.

=cut

sub maildir_id(;$) {
	my($hostname) = @_;
	$hostname = hostname unless defined $hostname;
	my($sec, $usec) = gettimeofday;
	my($new_sec, $new_usec);
	do {
		($new_sec, $new_usec) = gettimeofday;
	} while($new_sec == $sec && $new_usec == $usec);
	$sec.".M".$usec."P".$$.".".$hostname;
}

=back

=head1 BUGS

Can theoretically generate duplicate message IDs during a leap second.

=head1 SEE ALSO

L<Data::ID::Exim>,
L<UUID>,
L<Win32::Genguid>,
L<http://cr.yp.to/proto/maildir.html>

=head1 AUTHOR

Andrew Main (Zefram) <zefram@fysh.org>

=head1 COPYRIGHT

Copyright (C) 2004 Andrew Main (Zefram) <zefram@fysh.org>

This module is free software; you can redistribute it and/or modify it
under the same terms as Perl itself.

=cut

1;
