package My::Module::Recommend;

use strict;
use warnings;

use Carp;
use Config;

my ( $is_5_010, $is_5_012 );

eval {
    require 5.012;
    $is_5_012 = $is_5_010 = 1;
} or eval {
    require 5.010;
    $is_5_010 = 1;
};

my %misbehaving_os = map { $_ => 1 } qw{ MSWin32 cygwin };

my @optionals = (
    [ 'Astro::SIMBAD::Client'	=> <<'EOD' ],
      This module is required for the 'lookup' subcommand of the
      Astro::App::Satpass2 sky() method, but is otherwise unused by this
      package. If you do not intend to use this functionality,
      Astro::SIMBAD::Client is not needed.
EOD
    [ 'Astro::SpaceTrack'	=> <<'EOD' ],
      This module is required for the Astro::App::Satpass2 st() method,
      but is otherwise unused by this package. If you do not intend to
      use this functionality, Astro::SpaceTrack is not needed.
EOD
    [ 'Date::Manip'		=> <<'EOD' .
      This module is not required, but the alternative to installing it
      is to specify times in ISO 8601 format.  See 'SPECIFYING TIMES' in
      the 'Astro::App::Satpass2' documentation for the details.
EOD
	( $is_5_010 ? '' : <<'EOD' ) ],

      Unfortunately, the current Date::Manip requires Perl 5.10. Since
      you are running an earlier Perl, you can try installing Date-Manip
      5.54, which is the most recent version that does _not_ require
      Perl 5.10. This version of Date::Manip does not understand summer
      time (a.k.a. daylight saving time).
EOD
    [ [ qw{ DateTime DateTime::TimeZone } ] => <<'EOD' ],
      These modules are used to format times, and provide full time zone
      support. If they are not installed, POSIX::strftime() will be
      used, and you may find that you can not display correct local
      times for zones other than your system's default zone.
EOD
    [ 'Geo::Coder::OSM'		=> <<'EOD' ],
      This module is required for the Astro::App::Satpass2 geocode()
      method. If you do not intend to use this functionality, this
      module is not needed.
EOD
    [ 'Geo::WebService::Elevation::USGS'	=> <<'EOD' ],
      This module is required for the Astro::App::Satpass2 height()
      method, but is otherwise unused by this package. If you do not
      intend to use this functionality, Geo::WebService::Elevation::USGS
      is not needed.
EOD
    [ [ qw{ LWP::UserAgent LWP::Protocol URI } ] => <<'EOD' ],
      These modules are required if you want to use URLs in the init(),
      load(), or source() methods. If you do not intend to use URLs
      there, you do not need these packages. All three packages are
      requirements for most of the other Internet-access functionality,
      so you may get them implicitly if you install some of the other
      optional modules.
EOD
	$is_5_012 ? () : [ 'Time::y2038' => <<'EOD' .
      This module is not required, but if installed allows you to do
      computations for times outside the usual range of system epoch to
      system epoch + 0x7FFFFFFF seconds.
EOD
	( $misbehaving_os{$^O} ? <<"EOD" : '' ) .

      Unfortunately, Time::y2038 has been known to misbehave when
      running under $^O, so you may be better off just accepting the
      restricted time range.
EOD
	( ( $Config{use64bitint} || $Config{use64bitall} ) ? <<'EOD' : '' )

      Since your Perl appears to support 64-bit integers, you may well
      not need Time::y2038 to do computations for times outside the
      so-called 'usual range.' Time::y2038 will be used, though, if it
      is available.
EOD
    ],
);

sub make_optional_modules_tests {
    eval {
	require Test::Without::Module;
	1;
    } or return;
    my $dir = 'xt/author/optionals';
    -d $dir
	or mkdir $dir
	or die "Can not create $dir: $!\n";
    opendir my $dh, 't'
	or die "Can not access t/: $!\n";
    while ( readdir $dh ) {
	m/ \A [.] /smx
	    and next;
	m/ [.] t \z /smx
	    or next;
	my $fn = "$dir/$_";
	-e $fn
	    and next;
	print "Creating $fn\n";
	open my $fh, '>:encoding(utf-8)', $fn
	    or die "Can not create $fn: $!\n";
	print { $fh } <<"EOD";
package main;

use strict;
use warnings;

use Test::More 0.88;

use lib qw{ inc };

use My::Module::Recommend;

BEGIN {
    eval {
	require Test::Without::Module;
	Test::Without::Module->import(
	    My::Module::Recommend->optionals() );
	1;
    } or plan skip_all => 'Test::Without::Module not available';
}

do 't/$_';

1;

__END__

# ex: set textwidth=72 :
EOD
    }
    closedir $dh;

    return $dir;
}

sub optionals {
    my @rslt;
    foreach my $spec ( @optionals ) {
	my $module = $spec->[0];
	if ( 'ARRAY' eq ref $module ) {
	    push @rslt, @{ $module };
	} elsif ( ref $module ) {
	    confess 'Module spec may not be a ', ref $module, ' reference';
	} else {
	    push @rslt, $module;
	}
    }
    return @rslt;
}

sub recommend {
    my $need_some;
    SPEC_LOOP:
    foreach my $spec ( @optionals ) {
	my ( $module, $message ) = @{ $spec };
	foreach my $mod ( ref $module ? @{ $module } : $module ) {
	    eval "require $mod; 1"
		and next SPEC_LOOP;
	}
	$need_some++
	    or warn <<'EOD';

The following optional modules were not found:
EOD
	warn format_module_line( $module ), $message;
    }
    $need_some
	and warn <<'EOD';

It is not necessary to install these now. If you decide to install them
later, this software will make use of them when it finds them.

EOD

    return;
}

sub format_module_line {
    my ( $module ) = @_;
    ref $module
	or return "\n    * $module is not installed.\n";
    return "\n    * None of " . join( ', ', @{ $module } ) .
	" is installed.\n";
}

1;

__END__

=head1 NAME

Astro::Coord::ECI::Recommend - Recommend modules to install. 

=head1 SYNOPSIS

 use lib qw{ inc };
 use Astro::Coord::ECI::Recommend;
 Astro::Coord::ECI::Recommend->recommend();

=head1 DETAILS

This package generates the recommendations for optional modules. It is
intended to be called by the build system. The build system's own
mechanism is not used because we find its output on the Draconian side.

=head1 METHODS

This class supports the following public methods:

=head2 make_optional_modules_tests

 My::Module::Recommend->make_optional_modules_tests()

This static method creates the optional module tests. These are stub
files in F<xt/author/optionals/> that use C<Test::Without::Module> to
hide all the optional modules and then invoke the normal tests in F<t/>.
The aim of these tests is to ensure that we get no test failures if the
optional modules are missing.

This method is idempotent; that is, it only creates the directory and
the individual stub files if they are missing.

On success this method returns the name of the optional tests directory.
If C<Test::Without::Module> can not be loaded this method returns
nothing. If the directory or any file can not be created, an exception
is thrown.

=head2 optionals

 say for My::Module::Recommend->optionals();

This static method simply returns the names of the optional modules.

=head2 recommend

 Astro::Coord::ECI::Recommend->recommend();

This static method examines the current Perl to see which optional
modules are installed. If any are not installed, a message is printed to
standard out explaining the benefits to be gained from installing the
module, and any possible problems with installing it.

=head1 SUPPORT

Support is by the author. Please file bug reports at
L<http://rt.cpan.org>, or in electronic mail to the author.

=head1 AUTHOR

Thomas R. Wyant, III F<wyant at cpan dot org>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2010-2016 by Thomas R. Wyant, III

This program is free software; you can redistribute it and/or modify it
under the same terms as Perl 5.10.0. For more details, see the full text
of the licenses in the directory LICENSES.

This program is distributed in the hope that it will be useful, but
without any warranty; without even the implied warranty of
merchantability or fitness for a particular purpose.

=cut

# ex: set textwidth=72 :
