package Firewall::PaloAlto;
use Moose;

use 5.006;
use strict;
use warnings;

use XML::Simple;
use LWP::UserAgent;
use Carp;
use Data::Dumper qw(Dumper);
use Modern::Perl;
use Params::Validate qw(:all);


=head1 NAME

Firewall::PaloAlto - Interact with a Palo Alto firewall's API through Perl.

=head1 VERSION

Version 0.03

=cut

our $VERSION = '0.03';

=head1 SYNOPSIS

The Firewall::PaloAlto module provides interfaces into the XML API of a Palo Alto firewall.

    use Firewall::PaloAlto;

    my $fw = Firewall::PaloAlto->new(host => 'pa.local', username => 'admin', password => 'admin');
    $fw->connect();

    #Add a new virtual system
    $fw->address('set', vsys_id => 6, 'display-name' => "Script_Tenant");

    #Add a virtual router to the chassis
    $fw->virtual_router('set', vr_name => 'NEW_VR', interface => [ 'ae1.65', 'ae1.66' ]);

    #Add a new address - if the vsys is not specified it defaults to vsys1.
    #This works for devices such as the VM series, which only have a vsys1. 
    $fw->address('set', name => 'Google_DNS', ip-netmask => '8.8.8.8/32');

    #Get the configuration for the newly created address:
    my $address_config = $fw->address('get', name => 'Google_DNS');

    #Delete the newly created address
    $fw->address('delete', name => 'Google_DNS);

A list of functions that can be exported.  You can delete this section
if you don't export anything, such as for a purely object-oriented module.

=cut

=head1 CLASS METHODS

=head2 new(%parameters)

The constructor generates a new object to interact with a specific firewall.

The host, username and password parameters are mandatory.
If not specified, SSL is used, but it can be disabled using the argument ssl => 0

Detailed debugging can be turned on using the debu => 1 argument. It is off by default.

    my $palo = Firewall::PaloAlto->new(host => 'paloalto.local', username => 'admin', password => 'admin', ssl => 0, debug => 1);

B<Parameters>

=over 

=item * 
host - the hostname or IP of the firewall to connect to.

=item *
username - a username to connect to the firewall.

=item *
password - a password to connect to the firewall.

=item *
ssl (optional, default: 1) - use SSL to connect to the firewall. 

=item *
debug (optional, default: 0) - print debugging messages.

=back

=cut


has host => ( is => 'ro',
                isa => 'Str',
                required => '1',
);

has username => ( is => 'ro',
                    isa => 'Str',
                    required => '1',
);

has password => ( is => 'ro',
                    isa => 'Str',
                    required => '1',
);

has api_key => ( is => 'rw', isa => 'Str', init_arg => undef, writer => '_api_key');

has ssl => ( is => 'ro', isa => 'Bool', default => 1 );
has base_uri => (   is => 'ro', 
                    isa => 'Str', 
                    lazy => '1',
                    init_arg => undef,
                    default => sub { return 'http' . ($_[0]->ssl ? 's' : '') . '://' . $_[0]->host . '/api/?'; } 

);

has 'debug' => ( is => 'ro', isa => 'Bool', required => 0, default => 0 );

sub _create_requester {
    my $self = shift;
    my %args = @_;

    my $request =  $self->base_uri. 'key=' . $self->api_key;

    for my $key (keys %args) {
        #Add the argument on to the command line
        $request .= "&$key=$args{$key}";
    }


    return sub {
        my (%request_args) = @_;
        my $user_agent = LWP::UserAgent->new;

        $self->_debug_print("API Request action", $args{action});

        for my $key (keys %request_args) {
            $self->_debug_print("API Request String", "$key=$request_args{$key}");
            $request .= "&$key=$request_args{$key}";
        }

        my $http_request = HTTP::Request->new(GET => $request);
        my $http_response = $user_agent->request($http_request);

        return $self->_check_response_and_parse($http_response);
    }
}

sub _check_response_and_parse {
    my $self = shift;
    my $http_response = shift;
   
    #We locally redefine croak so we can get some nice red printing around it.
    no warnings 'redefine';
    local *original_croak = \&croak;
    local *croak = sub {  original_croak("\e[31m".$_[0]."\e[0m"); };
    
    #Check the http response message - croak if its not successful
    croak "*[HTTP Request] Failed: ".$http_response->content if !$http_response->is_success;
    
    my $palo_response = XMLin($http_response->content, KeyAttr => 'name');
    
    #If the response doesn't contain a code (i.e. the keygen request doesn't), then we check the status string.
    #If its successful, we assign our own 'custom code' as a workaround
    if (!defined $palo_response->{code} && $palo_response->{status} eq "success") {
        $palo_response->{code} = 1023;
    }
    
    #Check the response from the Palo Alto:
    my $code_info = $self->_get_response_info($palo_response->{code});
    my $api_error = $code_info->{parse}->($palo_response);

    carp "*[API Request] $code_info->{string}: $api_error" and return 0 if $code_info->{is_error};
    
    $self->_debug_print("API Response", $palo_response->{status});
    
    return $palo_response;
}

sub _get_response_info {
    my ($self, $code) = @_;

    my $response_codes = {
                            400 => {is_error => 1, string => "Bad Request", 
                                parse => sub { "Error 400" } },
                            403 => { is_error => 1, string => "Forbidden", 
                                parse => sub { $_[0]->{result}->{msg}; } 
                            },
                            1 => { is_error => 1, string => "Unknown Command", 
                                parse => sub { "Error 1" } },
                            2 => { is_error => 1, string => "Internal Error (2)", 
                                parse => sub { "Error 2" } },
                            3 => { is_error => 1, string => "Internal Error (3)", 
                                parse => sub { "Error 3" } },
                            4 => { is_error => 1, string => "Internal Error (4)", 
                                parse => sub { "Error 4" } },
                            5 => { is_error => 1, string => "Internal Error (4)", 
                                parse => sub { "Error 5" } },
                            6 => { is_error => 1, string => "Bad XPath", 
                                parse => sub { "Error 6" } },
                            7 => { is_error => 1, string => "Object not present", 
                                parse => sub { "Error 7" } },
                            8 => { is_error => 1, string => "Object not unique", 
                                parse => sub { "Error 8" } },
                            9 => { is_error => 1, string => "Internal Error (9)", 
                                parse => sub { "Error 9" } },
                            10 => { is_error => 1, string => "Reference count not zero", 
                                parse => sub { "Error 10" } },
                            11 => { is_error => 1, string => "Internal Error (11)", 
                                parse => sub { "Error 11" } },
                            12 => { is_error => 1, string => "Invalid Object", 
                                parse => sub { ref($_[0]->{msg}->{line}) eq 'ARRAY' ? join(" / ", @{ $_[0]->{msg}->{line} }) : $_[0]->{msg}->{line} } 
                                },
                            13 => { is_error => 1, string => "Operation Failed", 
                                parse => sub { $_[0]->{msg}->{line} } },
                            14 => { is_error => 1, string => "Operation Not Possible", 
                                parse => sub { "Error 14" } },
                            15 => { is_error => 1, string => "Operation Denied", 
                                parse => sub { "Error 15" } },
                            16 => { is_error => 1, string => "Unauthorized", 
                                parse => sub { "Error 16" } },
                            17 => { is_error => 1, string => "Invalid Command", 
                                parse => sub { "Error 16" } },
                            18 => { is_error => 1, string => "Malformed XML", 
                                parse => sub { $_[0]->{msg}->{line} } },
                            19 => { is_error => 0, string => "Get Request Successful", 
                                parse => sub{} },
                            20 => { is_error => 0, string => "Set Request Successful", 
                                parse => sub{} },
                            21 => { is_error => 1, string => "Internal Error (21)", 
                                parse => sub { "Error 21" } },
                            22 => { is_error => 1, string => "Session Timed Out", 
                                parse => sub { "Error 22" } },
                            #Custom code for keygen success
                            1023 => { is_error => 0, string => "KeyGen Successful", parse => sub {} },
                        };

    #Return the hash, or 'undef' if the code doesn't exist.
    return $response_codes->{$code}; 
}

sub _generate_elements {
    my $self = shift;
    my %element_hash = @_;
    my $element_string = "";

    for my $key (keys %element_hash) {
        $element_string .= "<$key>";

        #If our hash points to an array reference, we iterate through the array and add member.
        #This creates <key><member>a</member><member>b</member></key>
        if (ref($element_hash{$key}) eq "ARRAY") {
            for my $member (@{ $element_hash{$key} }) {
                   $element_string .= "<member>$member</member>";
            }
        }
        #If we're pointing to another hash, we recurse down, as the structure will be the same.
        #This allows us to do <key><another-key>data</another-key></key>
        elsif (ref($element_hash{$key}) eq "HASH") {
            $element_string .= $self->_generate_elements(%{ $element_hash{$key} });
        }
        #Otherwise its just a normal <key>value</key>
        else {
            $element_string .= "$element_hash{$key}";
        }

        $element_string .= "</$key>";
    }

    return $element_string;
}

sub _debug_print {
    my $self = shift;
    my ($category, $debug_string, $colourise_sub) = @_;
    my $string_colour = "\e[0;36m"; 
    my $string_norm = "\e[0m";

    if (!$self->debug()) {
        return 0;
    }

    #We pass code in $colorise_sub - if it evaluates to true, we print the category in green
    #If its false, we print in red. If its not defined, we print in orange.
    if (defined $colourise_sub) { 
        $string_colour = $colourise_sub->() ? "\e[32m" : "\e[31m";
    }

    say "*[".$string_colour.$category.$string_norm."] $debug_string";
}

=head1 OBJECT METHODS 

=head2 connect()

The connect functions connects to the Palo Alto, validates and saves the API key.
It has no parameters

 $pa->connect();

=cut

sub connect {
    my $self = shift; 

    my $user_agent = LWP::UserAgent->new(ssl_opts => { verify_hostname => 0 });
    my $key_request = HTTP::Request->new(GET => $self->base_uri.'type=keygen&user='.$self->username.'&password='.$self->password);

    $self->_debug_print("Key Request", "");

    my $key_response = $user_agent->request($key_request);

    my $pa_response = $self->_check_response_and_parse($key_response);

    if ($pa_response->{status} eq 'success') {
        $self->_api_key($pa_response->{result}->{key});
        return 1;
    }
    else {
        return 0;
    }   
}


=head2 commit(%parameters)

The commit function commits the current candidate configuration to the Palo Alto firewall.

    $pa_firewall->commit(vsys_id => 10, poll => 10);

B<Parameters>

=over 

=item * 
vsys_id (optional) - if supplied, performs a partial commit on the vsys specified. If not provided, performs a full commit on the device.

=item *
poll (optional, default: 5 seconds) - defines the interval (seconds) that the method will poll the device to get the current status of the commit job.

=back

=cut

sub commit {
    my $self = shift;
    my %args = @_;
    my $requester = $self->_create_requester(type => 'commit');
    my $operate = $self->_create_requester(type => 'op');
    my $cmd = "<commit></commit>";

    #If a poll interval is not defined, we default to 5 seconds.
    $args{poll} //= 5;

    if (defined $args{vsys_id}) {
        $cmd = "<commit><partial><vsys><member>vsys$args{vsys_id}</member></vsys></partial></commit>";
    }

    my $response = $requester->(cmd => $cmd);

    my $job_result;

    do {
        $job_result = $operate->(cmd => "<show><jobs><id>$response->{result}->{job}</id></jobs></show>");
        $self->_debug_print((caller(0))[3], "JobID: $response->{result}->{job}, Status: $job_result->{result}->{job}->{result}, Progress: $job_result->{result}->{job}->{progress}",
                            sub { $job_result->{result}->{job}->{result} ne "FAILED" });

        sleep($args{poll});
    } while ($job_result->{result}->{job}->{result} eq 'PEND');
}


=head2 vsys($action, %parameters)

The vsys function creates a new virtual system.

 $pa->vsys('set', vsys_id => 4, display-name => "Tenant 4");

B<Parameters>

=over

=item * $action - perform an action: ['get' | 'set' | 'delete']

=item * vsys_id - the ID of the virtual system to perform the action on.

=item * display-name - sets the display name of the virtual system.

=back

=cut

sub vsys {
    my $self = shift;
    my $action = shift;
    my (%args) = @_;

    #Validate our parameters.
    validate(
        @_, {
            vsys_id => 1,
            "display-name" => 1
        }
    );

    my $vsys_id = delete $args{vsys_id};
    my $requester = $self->_create_requester(type => 'config', action => $action);

    my $elements = $self->_generate_elements(%args);

    return $requester->(xpath => "/config/devices/entry[\@name='localhost.localdomain']/vsys/entry[\@name='vsys$vsys_id']", element => $elements);
}

=head2 vsys_import($action, %parameters);

The vsys_import function imports interfaces and virtual-routers into a virtual system. 

    $pa_firewall->vsys_import('set', vsys_id => 4, interface => [ ethernet1/1, ethernet1/2 ], virtual-router => [ default ]);

B<Parameters>

=over

=item * $action - perform an action: ['get' | 'set' | 'delete']

=item * vsys_id - the ID of the virtual system to perform the action on.

=item * interface - an anonymous array of one or more interfaces to add to the virtual system.

=item * virtual-router - an anonymous array of one or more virtual routers to add to the virtual system.

=back

=cut

sub vsys_import {
    my $self = shift;
    my $action = shift;
    my (%args) = @_;

    #validate(
    #    @_, {
    #        vsys_id => 1,
    #        interface => 0;
    #        "virtual-router" => 0,
    #    }
    #);

    my $vsys_id = delete @args{vsys_id};

    my $requester = $self->_create_requester(type => 'config', action => $action);
    my $elements = $self->_generate_elements(%args);

    #Add the interface or virtual router to a vsys
    return $requester->(xpath => "/config/devices/entry[\@name='localhost.localdomain']/vsys/entry[\@name='vsys$vsys_id']/import/network", element => $elements);
}

=head2 l3_subinterface($action, %parameters)

This function creates a new layer 3 subinterface underneath a parent interface.

    $pa->l3_subinterface('set', parent => 'ethernet1/1', tag => 5, description => 'Tenant x untrust interface');

B<Parameters>

=over


=item * $action - perform an action: ['get' | 'set' | 'delete']

=item * parent - the parent interface of the new subinterface

=item * tag - the VLAN tag to use on the sub-interface. This is also used as the logical sub-interface identifier.

=item * description - a description to add to the sub-interface.

=back

=cut

sub l3_subinterface {
    my $self = shift;
    my ($action, %args) = @_;

    my $parent_interface = delete @args{'parent'};

    my $requester = $self->_create_requester(type => 'config', action => $action);
    my $elements = $self->_generate_elements(%args);

    #Create the sub-interface
    return $requester->(xpath => "/config/devices/entry[\@name='localhost.localdomain']/network/interface/aggregate-ethernet/entry[\@name='$parent_interface']/layer3/units/entry[\@name='$parent_interface.$args{tag}']",
                        element => $elements);

}


=head2 virtual_router($action, %parameters)

B<Parameters>

=over


=item * $action - perform an action: ['get' | 'set' | 'delete']

=item * vr_name - the name of the virtual router to perform the action on.

=item * interface - an anonymous array of one or more interfaces to add to the virtual router.

=back

=cut

sub virtual_router {
    my $self = shift;
    my ($action, %args) = @_;

    my $vr_name = delete @args{'vr_name'};

    my $requester = $self->_create_requester(type => 'config', action => $action);
    my $elements = $self->_generate_elements(%args);
    
    #Create the virtual router
    return $requester->(xpath => "/config/devices/entry[\@name='localhost.localdomain']/network/virtual-router/entry[\@name='$vr_name']",
                        element => $elements);     
}

=head2 zone($action, %parameters)

B<Parameters>

=over

=item * $action - perform an action: ['get' | 'set' | 'delete']

=item * vsys_id - the virtual system ID to which the zone is/should be a member of.

=item * zone - the name of the zone to create/update/delete.

=item * layer3 - an anonymous array of one or more interfaces to add to the zone.

=back

=cut

sub zone {
    my $self = shift;
    my ($action, %args) = @_;
    my $requester = $self->_create_requester(type => 'config', action => $action);

    my ($vsys_id, $zone) = delete @args{'vsys_id', 'zone'};
    $vsys_id //= 1;

    my $elements = $self->_generate_elements(%args);

    return $requester->(xpath => "/config/devices/entry[\@name='localhost.localdomain']/vsys/entry[\@name='vsys$vsys_id']/zone/entry[\@name='$zone']/network",
                        element => $elements);
}

=head2 ipv4_static_route($action, %parameters)

B<Parameters>

=over

=item * $action - perform an action: ['get' | 'set' | 'delete']

=item * vr_name - the name of the virtual router in which the static route resides.

=item * route_name - the name of the route to perform the action on.

=item * destination - the IPv4 destination of the route (IP/prefix)

=item * nexthop - an anonymous hash specifying the next hop

=item * ip-address - the next hop IP address

=item * interface - the next hop interface

=back

=cut

sub ipv4_static_route {
    my $self = shift;
    my ($action, %args) = @_;
    my $requester = $self->_create_requester(type => 'config', action => $action);

    my ($vr_name, $route_name) = delete @args{'vr_name', 'route_name'};
    $route_name = defined $route_name ? "\@name='$route_name'" : "'*'";

    my $elements = $self->_generate_elements(%args);

    return $requester->(xpath => "/config/devices/entry[\@name='localhost.localdomain']/network/virtual-router/entry[\@name='$vr_name']/routing-table/ip/static-route/entry[$route_name]",
                        element => $elements);
}


=head2 ipv6_static_route($action, %parameters)

B<Parameters>

=over


=item * $action - perform an action: ['get' | 'set' | 'delete']

=item * vr_name - the name of the virtual router in which the static route resides.

=item * route_name - the name of the route to perform the action on.

=item * destination - the IPv6 destination of the route (IP/prefix)

=item * nexthop - an anonymous hash specifying the next hop

=item * ipv6-address - the next hop IPv6 address

=item * interface - the next hop interface

=back

=cut

sub ipv6_static_route {
    my $self = shift;
    my ($action, %args) = @_;
    my $requester = $self->_create_requester(type => 'config', action => $action);

    my ($vr_name, $route_name) = delete @args{'vr_name', 'route_name'};
    $route_name = defined $route_name ? "\@name='$route_name'" : "'*'";

    my $elements = $self->_generate_elements(%args);

    return $requester->(xpath => "/config/devices/entry[\@name='localhost.localdomain']/network/virtual-router/entry[\@name='$vr_name']/routing-table/ipv6/static-route/entry[$route_name]",
                        element => $elements);
}

=head2 address($action, %parameters)

B<Parameters>

=over

=item * vsys_id - the vsys ID in which the resides/shall reside.

=item * name - the name of the address.

=item * ip-netmask - the IP/netmask combination which defines the address.

=item * ip-range - an IP address range (format: 'IPstart-IPend')

=back

=cut

sub address {
    my $self = shift;
    my ($action, %args) = @_;
    my $requester = $self->_create_requester(type => 'config', action => $action);

    #Keys to be extracted and deleted because they aren't part of the element
    #delete returns the values that were deleted.
    my ($vsys, $address) = delete @args{'vsys_id', 'name'};

    #If the vsys if not defined, we default to vsys1
    $vsys //= "vsys1";

    $address = defined $address ? "\@name='$address'" : "'*'";

    my $elements = $self->_generate_elements(%args);

    return $requester->(xpath => "/config/devices/entry[\@name='localhost.localdomain']/vsys/entry[\@name=\'$vsys\']/address/entry[$address]", element => $elements);
}


=head2 address_group($action, %parameters)

B<Parameters>

=over


=item * $action - perform an action: ['get' | 'set' | 'delete']

=item * vsys_id - the vsys ID in which the address group resides/shall reside.A

=item * name - the name of the address group.

=item * member - an anonymous array of one or more addresses. These can be either address entries created with address(), or explicit IP/netmasks (e.g. 9.8.8.8/32)

=back

=cut

sub address_group {
    my $self = shift;
    my ($action, %args) = @_;
    my $requester = $self->_create_requester(type => 'config', action => $action);

    #Keys to be extracted and deleted because they aren't part of the element
    #delete returns the values that were deleted.
    my ($vsys, $name) = delete @args{'vsys_id', 'name'};

    #If the vsys if not defined, we default to vsys1
    $vsys //= "vsys1";

    $name = defined $name ? "\@name='$name'" : "'*'";

    my $elements = $self->_generate_elements(%args);

    return $requester->(xpath => "/config/devices/entry/vsys/entry[\@name=\'$vsys\']/address-group/entry[$name]", element => $elements);
}



=head2 security_rule($action, %parameters)

B<Parameters>

=over


=item * $action - perform an action: ['get' | 'set' | 'delete']

=item * vsys_id - the virtual system ID of the vsys in which the security rule resides/will reside. Defaults to 1 if not supplied.

=item * name - the name of the rule.

=item * from - the source zone, defaults to 'any' if not supplied.

=item * to - the destination zone, defaults to 'any' if not supplied.

=item * source - an anonymous array of source addresses - can be addresses, address groups or explicit IP/netmask entries. Defaults to 'any' if not supplied.

=item * destination - an anonymous array of destination addresses - can be addresses, address groups or explicit IP/netmask entries. Defaults to 'any' if not supplied.

=item * service - an anonymous array of one or more services. Defaults to 'application-default' if not supplied.

=item * appplication - an anonymous array of one or more Palo Alto applications. Defaults to 'any' if not supplied.

=item * source-user - an anonymous array of one or more Palo Alto source user mappings. Defaults to 'any' if not supplied.

=item * hip-profile - an anonymous array of Host Information Profiles. defaults to 'any' if not supplied.

=item * action - an action for the rule, either 'allow', 'deny' or 'drop'. Defaults to 'allow' if not supplied.

=back

=cut


sub security_rule {
    my $self = shift;
    my ($action, %args) = @_;
    my $requester = $self->_create_requester(type => 'config', action => $action);

    #If the name isn't defined, default to all rules
    my $rule_name = defined $args{name} ? "\@name='$args{name}'" : "'*'";
    delete $args{name};
    
    #If the vsys if not defined, we default to vsys1
    my $vsys = $args{vsys_id} // "vsys1";
    delete $args{vsys_id};

    #If any of the following items aren't defined, we default to an anon array of type 'any'
    my @default_any = qw(to from source destination application source-user hip-profiles);
    for my $key (@default_any) {
        $args{$key} //= ['any'];
    }    

    #If the service isn't defined, we default to 'application-default'
    $args{service} //= ['application-default'];

    #If the action isn't defined, we defailt to 'allow'
    $args{action} //= 'allow';

    my $elements = $self->_generate_elements(%args);

    return $requester->(xpath => "/config/devices/entry/vsys/entry[\@name=\'$vsys\']/rulebase/security/rules/entry[$rule_name]", element => $elements);
}

=head1 AUTHOR

Greg Foletta, C<< <greg at foletta.org> >>

=head1 BUGS

Please report any bugs or feature requests to C<bug-firewall-paloalto at rt.cpan.org>, or through
the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=Firewall-PaloAlto>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.



=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Firewall::PaloAlto


You can also look for information at:

=over 4

=item * RT: CPAN's request tracker (report bugs here)

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=Firewall-PaloAlto>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/Firewall-PaloAlto>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/Firewall-PaloAlto>

=item * Search CPAN

L<http://search.cpan.org/dist/Firewall-PaloAlto/>

=back


=head1 ACKNOWLEDGEMENTS


=head1 LICENSE AND COPYRIGHT

Copyright 2015 Greg Foletta.

This program is free software; you can redistribute it and/or modify it
under the terms of the the Artistic License (2.0). You may obtain a
copy of the full license at:

L<http://www.perlfoundation.org/artistic_license_2_0>

Any use, modification, and distribution of the Standard or Modified
Versions is governed by this Artistic License. By using, modifying or
distributing the Package, you accept this license. Do not use, modify,
or distribute the Package, if you do not accept this license.

If your Modified Version has been derived from a Modified Version made
by someone other than you, you are nevertheless required to ensure that
your Modified Version complies with the requirements of this license.

This license does not grant you the right to use any trademark, service
mark, tradename, or logo of the Copyright Holder.

This license includes the non-exclusive, worldwide, free-of-charge
patent license to make, have made, use, offer to sell, sell, import and
otherwise transfer the Package with respect to any patent claims
licensable by the Copyright Holder that are necessarily infringed by the
Package. If you institute patent litigation (including a cross-claim or
counterclaim) against any party alleging that the Package constitutes
direct or contributory patent infringement, then this Artistic License
to you shall terminate on the date that such litigation is filed.

Disclaimer of Warranty: THE PACKAGE IS PROVIDED BY THE COPYRIGHT HOLDER
AND CONTRIBUTORS "AS IS' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.
THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
PURPOSE, OR NON-INFRINGEMENT ARE DISCLAIMED TO THE EXTENT PERMITTED BY
YOUR LOCAL LAW. UNLESS REQUIRED BY LAW, NO COPYRIGHT HOLDER OR
CONTRIBUTOR WILL BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, OR
CONSEQUENTIAL DAMAGES ARISING IN ANY WAY OUT OF THE USE OF THE PACKAGE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


=cut

1; # End of Firewall::PaloAlto
