package Firewall::PaloAlto;
use Moose;

use 5.006;
use strict;
use warnings;

use XML::Simple;
use LWP::UserAgent;
use Carp;
use Data::Dumper qw(Dumper);
use Modern::Perl;
use Params::Validate qw(:all);


=head1 NAME

Firewall::PaloAlto - Interact with a Palo Alto firewall's API through Perl.

=head1 VERSION

Version 0.01

=cut

our $VERSION = '0.01';


=head1 SYNOPSIS

Quick summary of what the module does.

Perhaps a little code snippet.

    use Firewall::PaloAlto;

    my $foo = Firewall::PaloAlto->new();
    ...

=head1 EXPORT

A list of functions that can be exported.  You can delete this section
if you don't export anything, such as for a purely object-oriented module.

=cut


=head1 SUBROUTINES/METHODS
=cut

has host => ( is => 'ro',
                isa => 'Str',
                required => '1',
);

has username => ( is => 'ro',
                    isa => 'Str',
                    required => '1',
);

has password => ( is => 'ro',
                    isa => 'Str',
                    required => '1',
);

has api_key => ( is => 'rw', isa => 'Str', init_arg => undef, writer => '_api_key');

has ssl => ( is => 'ro', isa => 'Bool', default => 1 );
has base_uri => (   is => 'ro', 
                    isa => 'Str', 
                    lazy => '1',
                    init_arg => undef,
                    default => sub { return 'http' . ($_[0]->ssl ? 's' : '') . '://' . $_[0]->host . '/api/?'; } 

);

has 'debug' => ( is => 'ro', isa => 'Bool', required => 0, default => 0 );

sub _create_requester {
    my $self = shift;
    my %args = @_;

    my $request =  $self->base_uri. 'key=' . $self->api_key;

    for my $key (keys %args) {
        #Add the argument on to the command line
        $request .= "&$key=$args{$key}";
    }


    return sub {
        my (%request_args) = @_;
        my $user_agent = LWP::UserAgent->new;

        for my $key (keys %request_args) {
            $request .= "&$key=$request_args{$key}";
        }

        $self->_debug_print((caller(1))[3], "Sending HTTP Request");

        my $http_request = HTTP::Request->new(GET => $request);
        my $http_response = $user_agent->request($http_request);

        return $self->_check_response_and_parse($http_response);
    }
}

sub _check_response_and_parse {
    my $self = shift;
    my $http_response = shift;

    my $response_codes = {
                            400 => {is_error => 1, string => "Bad Request", parse => sub { "Error 400" } },
                            403 => { is_error => 1, string => "Forbidden", parse => sub { $_[0]->{result}->{msg}; } },
                            1 => { is_error => 1, string => "Unknown Command", parse => sub { "Error 1" } },
                            2 => { is_error => 1, string => "Internal Error (2)", parse => sub { "Error 2" } },
                            3 => { is_error => 1, string => "Internal Error (3)", parse => sub { "Error 3" } },
                            4 => { is_error => 1, string => "Internal Error (4)", parse => sub { "Error 4" } },
                            5 => { is_error => 1, string => "Internal Error (4)", parse => sub { "Error 5" } },
                            6 => { is_error => 1, string => "Bad XPath", parse => sub { "Error 6" } },
                            7 => { is_error => 1, string => "Object not present", parse => sub { "Error 7" } },
                            8 => { is_error => 1, string => "Object not unique", parse => sub { "Error 8" } },
                            9 => { is_error => 1, string => "Internal Error (9)", parse => sub { "Error 9" } },
                            10 => { is_error => 1, string => "Reference count not zero", parse => sub { "Error 10" } },
                            11 => { is_error => 1, string => "Internal Error (11)", parse => sub { "Error 11" } },
                            12 => { is_error => 1, string => "Invalid Object", parse => sub { join(", ", @{$_[0]->{msg}->{line}}); } },
                            13 => { is_error => 1, string => "Operation Failed", parse => sub { "Error 13" } },
                            14 => { is_error => 1, string => "Operation Not Possible", parse => sub { "Error 14" } },
                            15 => { is_error => 1, string => "Operation Denied", parse => sub { "Error 15" } },
                            16 => { is_error => 1, string => "Unauthorized", parse => sub { "Error 16" } },
                            17 => { is_error => 1, string => "Invalid Command", parse => sub { "Error 16" } },
                            18 => { is_error => 1, string => "Malformed XML", parse => sub { $_[0]->{msg}->{line} } },
                            19 => { is_error => 0, string => "Get Request Successful", parse => sub{} },
                            20 => { is_error => 0, string => "Set Request Successful", parse => sub{} },
                            21 => { is_error => 1, string => "Internal Error (21)", parse => sub { "Error 21" } },
                            22 => { is_error => 1, string => "Session Timed Out", parse => sub { "Error 22" } },
                            #Custom code for keygen success
                            1023 => { is_error => 0, string => "KeyGen Successful", parse => sub {} },
                        };
   
    #We locally redefine croak so we can get some nice red printing around it.
    no warnings 'redefine';
    local *original_croak = \&croak;
    local *croak = sub {  original_croak("\e[31m".$_[0]."\e[0m"); };
    
    #Check the http response message - croak if its not successful
    croak "*[HTTP Request] Failed" if !$http_response->is_success;
    
    my $palo_response = XMLin($http_response->content, KeyAttr => 'name');
    
    #If the response doesn't contain a code (i.e. the keygen request doesn't), then we check the status string.
    #If its successful, we assign our own 'custom code' as a workaround
    if (!defined $palo_response->{code} && $palo_response->{status} eq "success") {
        $palo_response->{code} = 1023;
    }
    
    #Check the response from the Palo Alto:
    my $code = $response_codes->{$palo_response->{code}};
    my $api_error = $code->{parse}->($palo_response);
    croak "*[API Request] $code->{string}: $api_error" if $code->{is_error};
    
    $self->_debug_print((caller(1))[3], "API Request $palo_response->{status}");
    
    
    return $palo_response;
}

sub _generate_elements {
    my $self = shift;
    my %element_hash = @_;
    my $element_string = "";

    for my $key (keys %element_hash) {
        $element_string .= "<$key>";

        #If our hash points to an array reference, we iterate through the array and add member.
        #This creates <key><member>a</member><member>b</member></key>
        if (ref($element_hash{$key}) eq "ARRAY") {
            for my $member (@{ $element_hash{$key} }) {
                   $element_string .= "<member>$member</member>";
            }
        }
        #If we're pointing to another hash, we recurse down, as the structure will be the same.
        #This allows us to do <key><another-key>data</another-key></key>
        elsif (ref($element_hash{$key}) eq "HASH") {
            $element_string .= $self->_generate_elements(%{ $element_hash{$key} });
        }
        #Otherwise its just a normal <key>value</key>
        else {
            $element_string .= "$element_hash{$key}";
        }

        $element_string .= "</$key>";
    }

    return $element_string;
}

sub _debug_print {
    my $self = shift;
    my ($category, $debug_string, $colourise_sub) = @_;
    my $string_colour = "\e[0;36m"; 
    my $string_norm = "\e[0m";

    if (!$self->debug()) {
        return 0;
    }

    #We pass code in $colorise_sub - if it evaluates to true, we print the category in green
    #If its false, we print in red. If its not defined, we print in orange.
    if (defined $colourise_sub) { 
        $string_colour = $colourise_sub->() ? "\e[32m" : "\e[31m";
    }

    say "*[".$string_colour.$category.$string_norm."] $debug_string";
}

=head1 OBJECT METHODS 

=head2 commit([vsys_id => $num], [poll => $seconds])

The commit function commits the current candidate configuration to the Palo Alto firewall.

The 'vsys_id' argument takes the virtual system ID (which ranges from [1 - 225]). If it is defined, a partial commit is performed on that vsys. i
If no vsys is defined, the firewall performs a complete commit.

The 'poll' argument takes an integer number of seconds. The script will check on the status of the commit job and return information every poll interval.
If poll is not specified, the poll interval defaults to 5 seconds.

    $pa_firewall->commit(vsys_id => 10, poll => 10);

=cut

sub commit {
    my $self = shift;
    my %args = @_;
    my $requester = $self->_create_requester(type => 'commit');
    my $operate = $self->_create_requester(type => 'op');
    my $cmd = "<commit></commit>";

    #If a poll interval is not defined, we default to 5 seconds.
    $args{poll} //= 5;

    if (defined $args{vsys_id}) {
        $cmd = "<commit><partial><vsys><member>vsys$args{vsys_id}</member></vsys></partial></commit>";
    }

    my $response = $requester->(cmd => $cmd);

    my $job_result;

    do {
        $job_result = $operate->(cmd => "<show><jobs><id>$response->{result}->{job}</id></jobs></show>");
        $self->_debug_print((caller(0))[3], "JobID: $response->{result}->{job}, Status: $job_result->{result}->{job}->{result}, Progress: $job_result->{result}->{job}->{progress}",
                            sub { $job_result->{result}->{job}->{result} ne "FAILED" });

        sleep($args{poll});
    } while ($job_result->{result}->{job}->{result} eq 'PEND');
}

=head2 connect(host => $hostname, username => $user, password => $pass, [ssl => 0|1], [debug => 0|1])

This function connects to the Palo Alto with the credentials specified.

Host, username and password are mandatory.
If not specified SSL is used, but it can be disabled using the argument ssl => 0

Detailed debugging can be turned on using the debu => 1 argument. It is off by default.

=cut

sub connect {
    my $self = shift; 

    my $user_agent = LWP::UserAgent->new(ssl_opts => { verify_hostname => 0 });
    my $key_request = HTTP::Request->new(GET => $self->base_uri.'type=keygen&user='.$self->username.'&password='.$self->password);
    my $key_response = $user_agent->request($key_request);

    my $pa_response = $self->_check_response_and_parse($key_response);

    if ($pa_response->{status} eq 'success') {
        $self->_api_key($pa_response->{result}->{key});
        return 1;
    }
    else {
        return 0;
    }   
}

=head2 vsys($action, vsys_id => $id, display-name => $name)
=cut

sub vsys {
    my $self = shift;
    my $action = shift;
    my (%args) = @_;

    #Validate our parameters.
    validate(
        @_, {
            vsys_id => 1,
            "display-name" => 1
        }
    );

    my $vsys_id = delete $args{vsys_id};
    my $requester = $self->_create_requester(type => 'config', action => $action);

    my $elements = $self->_generate_elements(%args);

    return $requester->(xpath => "/config/devices/entry[\@name='localhost.localdomain']/vsys/entry[\@name='vsys$vsys_id']", element => $elements);
}

=head2 vsys_import($action, vsys_id => $id, interface => [ $int1, $int2 ], virtual-router => [ $vr1, $vr2 ])
=cut

sub vsys_import {
    my $self = shift;
    my $action = shift;
    my (%args) = @_;

    #validate(
    #    @_, {
    #        vsys_id => 1,
    #        interface => 0;
    #        "virtual-router" => 0,
    #    }
    #);

    my $vsys_id = delete @args{vsys_id};

    my $requester = $self->_create_requester(type => 'config', action => $action);
    my $elements = $self->_generate_elements(%args);

    #Add the interface or virtual router to a vsys
    return $requester->(xpath => "/config/devices/entry[\@name='localhost.localdomain']/vsys/entry[\@name='vsys$vsys_id']/import/network", element => $elements);
}

=head2 l3_subinterface($action, parent => $interface, tag => $vlan, comment => $description)
=cut

sub l3_subinterface {
    my $self = shift;
    my ($action, %args) = @_;

    my $parent_interface = delete @args{'parent'};

    my $requester = $self->_create_requester(type => 'config', action => $action);
    my $elements = $self->_generate_elements(%args);

    #Create the sub-interface
    return $requester->(xpath => "/config/devices/entry[\@name='localhost.localdomain']/network/interface/aggregate-ethernet/entry[\@name='$parent_interface']/layer3/units/entry[\@name='$parent_interface.$args{tag}']",
                        element => $elements);

}


=head2 virtual_router($action, vr_name => $name, interface => [ $int1, $int2 ])
=cut

sub virtual_router {
    my $self = shift;
    my ($action, %args) = @_;

    my $vr_name = delete @args{'vr_name'};

    my $requester = $self->_create_requester(type => 'config', action => $action);
    my $elements = $self->_generate_elements(%args);
    
    #Create the virtual router
    return $requester->(xpath => "/config/devices/entry[\@name='localhost.localdomain']/network/virtual-router/entry[\@name='$vr_name']",
                        element => $elements);     
}

=head2 zone($action, vsys_id => $id, zone => $name, layer3 => [ $int1, $int2 ])
=cut

sub zone {
    my $self = shift;
    my ($action, %args) = @_;
    my $requester = $self->_create_requester(type => 'config', action => $action);

    my ($vsys_id, $zone) = delete @args{'vsys_id', 'zone'};
    $vsys_id //= 1;

    my $elements = $self->_generate_elements(%args);

    return $requester->(xpath => "/config/devices/entry[\@name='localhost.localdomain']/vsys/entry[\@name='vsys$vsys_id']/zone/entry[\@name='$zone']/network",
                        element => $elements);
}

=head2 ipv4_static_route($action, vr_name => $vr_name, route_name => $rt_name, desintation => $route, nexthop => { ip-address => $ip | interface => $egress_int })
=cut

sub ipv4_static_route {
    my $self = shift;
    my ($action, %args) = @_;
    my $requester = $self->_create_requester(type => 'config', action => $action);

    my ($vr_name, $route_name) = delete @args{'vr_name', 'route_name'};
    $route_name = defined $route_name ? "\@name='$route_name'" : "'*'";

    my $elements = $self->_generate_elements(%args);

    return $requester->(xpath => "/config/devices/entry[\@name='localhost.localdomain']/network/virtual-router/entry[\@name='$vr_name']/routing-table/ip/static-route/entry[$route_name]",
                        element => $elements);
}


=head2 ipv6_static_route($action, vr_name => $vr_name, route_name => $rt_name, desintation => $route, nexthop => { ipv6-address => $ip | interface => $egress_int })
=cut

sub ipv6_static_route {
    my $self = shift;
    my ($action, %args) = @_;
    my $requester = $self->_create_requester(type => 'config', action => $action);

    my ($vr_name, $route_name) = delete @args{'vr_name', 'route_name'};
    $route_name = defined $route_name ? "\@name='$route_name'" : "'*'";

    my $elements = $self->_generate_elements(%args);

    return $requester->(xpath => "/config/devices/entry[\@name='localhost.localdomain']/network/virtual-router/entry[\@name='$vr_name']/routing-table/ipv6/static-route/entry[$route_name]",
                        element => $elements);
}

=head2 address($action, vsys_id => $id, name => $address_name, ip-netmask => $ipnet | ip-range => $range)
=cut

sub address {
    my $self = shift;
    my ($action, %args) = @_;
    my $requester = $self->_create_requester(type => 'config', action => $action);

    #Keys to be extracted and deleted because they aren't part of the element
    #delete returns the values that were deleted.
    my ($vsys, $address) = delete @args{'vsys_id', 'name'};

    #If the vsys if not defined, we default to vsys1
    $vsys //= "vsys1";

    $address = defined $address ? "\@name='$address'" : "'*'";

    my $elements = $self->_generate_elements(%args);

    return $requester->(xpath => "/config/devices/entry[\@name='localhost.localdomain']/vsys/entry[\@name=\'$vsys\']/address/entry[$address]/static", element => $elements);
}


=head2 address_group($action, vsys_id => $id, name => $group_name, member => [ $addr1, $addr2 ])
=cut

sub address_group {
    my $self = shift;
    my ($action, %args) = @_;
    my $requester = $self->_create_requester(type => 'config', action => $action);

    #Keys to be extracted and deleted because they aren't part of the element
    #delete returns the values that were deleted.
    my ($vsys, $name) = delete @args{'vsys_id', 'name'};

    #If the vsys if not defined, we default to vsys1
    $vsys //= "vsys1";

    $name = defined $name ? "\@name='$name'" : "'*'";

    my $elements = $self->_generate_elements(%args);

    return $requester->(xpath => "/config/devices/entry/vsys/entry[\@name=\'$vsys\']/address-group/entry[$name]", element => $elements);
}

=head2 security_rule($action, vsys_id => $id, name => $rulename, [from => $zone_frm, to => $zone_to, source => [ $addr1, ... ], $destination => [ $addr2, ... ], application => [ $app1, ... ], source-user => [ $user1, ... ], hip-profiles => [ $profile1, ... ], action => 'allow'|'deny')
=cut

sub security_rule {
    my $self = shift;
    my ($action, %args) = @_;
    my $requester = $self->_create_requester(type => 'config', action => $action);

    #If the name isn't defined, default to all rules
    my $rule_name = defined $args{name} ? "\@name='$args{name}'" : "'*'";
    delete $args{name};
    
    #If the vsys if not defined, we default to vsys1
    my $vsys = $args{vsys_id} // "vsys1";
    delete $args{vsys_id};

    #If any of the following items aren't defined, we default to an anon array of type 'any'
    my @default_any = qw(to from source destination application source-user hip-profiles);
    for my $key (@default_any) {
        $args{$key} //= ['any'];
    }    

    #If the service isn't defined, we default to 'application-default'
    $args{service} //= ['application-default'];

    #If the action isn't defined, we defailt to 'allow'
    $args{action} //= 'allow';

    my $elements = $self->_generate_elements(%args);

    return $requester->(xpath => "/config/devices/entry/vsys/entry[\@name=\'$vsys\']/rulebase/security/rules/entry[$rule_name]", element => $elements);
}

=head1 AUTHOR

Greg Foletta, C<< <greg at foletta.org> >>

=head1 BUGS

Please report any bugs or feature requests to C<bug-firewall-paloalto at rt.cpan.org>, or through
the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=Firewall-PaloAlto>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.



=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Firewall::PaloAlto


You can also look for information at:

=over 4

=item * RT: CPAN's request tracker (report bugs here)

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=Firewall-PaloAlto>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/Firewall-PaloAlto>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/Firewall-PaloAlto>

=item * Search CPAN

L<http://search.cpan.org/dist/Firewall-PaloAlto/>

=back


=head1 ACKNOWLEDGEMENTS


=head1 LICENSE AND COPYRIGHT

Copyright 2015 Greg Foletta.

This program is free software; you can redistribute it and/or modify it
under the terms of the the Artistic License (2.0). You may obtain a
copy of the full license at:

L<http://www.perlfoundation.org/artistic_license_2_0>

Any use, modification, and distribution of the Standard or Modified
Versions is governed by this Artistic License. By using, modifying or
distributing the Package, you accept this license. Do not use, modify,
or distribute the Package, if you do not accept this license.

If your Modified Version has been derived from a Modified Version made
by someone other than you, you are nevertheless required to ensure that
your Modified Version complies with the requirements of this license.

This license does not grant you the right to use any trademark, service
mark, tradename, or logo of the Copyright Holder.

This license includes the non-exclusive, worldwide, free-of-charge
patent license to make, have made, use, offer to sell, sell, import and
otherwise transfer the Package with respect to any patent claims
licensable by the Copyright Holder that are necessarily infringed by the
Package. If you institute patent litigation (including a cross-claim or
counterclaim) against any party alleging that the Package constitutes
direct or contributory patent infringement, then this Artistic License
to you shall terminate on the date that such litigation is filed.

Disclaimer of Warranty: THE PACKAGE IS PROVIDED BY THE COPYRIGHT HOLDER
AND CONTRIBUTORS "AS IS' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.
THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
PURPOSE, OR NON-INFRINGEMENT ARE DISCLAIMED TO THE EXTENT PERMITTED BY
YOUR LOCAL LAW. UNLESS REQUIRED BY LAW, NO COPYRIGHT HOLDER OR
CONTRIBUTOR WILL BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, OR
CONSEQUENTIAL DAMAGES ARISING IN ANY WAY OUT OF THE USE OF THE PACKAGE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


=cut

1; # End of Firewall::PaloAlto
