package Device::PaloAlto::Firewall::Test;

use 5.006;
use strict;
use warnings;

our $VERSION = '0.05'; # VERSION - generated by DZP::OurPkgVersion

use Moose;
use Modern::Perl;
use Carp;
use List::Util qw( any first );
#use List::MoreUtils qw( uniq );
#use Array::Utils qw{ array_minus };
use Params::Validate qw( :all );

use Data::Dumper;

=head1 NAME

Device::PaloAlto::Firewall::Test- Run a suite of tests against Palo Alto firewalls.

=head1 VERSION

version 0.05

=cut

=head1 SYNOPSIS

This module contains a set of methods that run tests against an Palo Alto firewall.
The functions take arguments and return 1 or 0 depending on the current runtime state of the firewall.

These methods should be used in conjunction with the B<ok()> function provided by B<Test::More>.

    use Device::PaloAlto::Firewall
    use Test::More;

=cut

has 'firewall'      => ( is => 'ro', isa => 'Device::PaloAlto::Firewall', default => sub { });

=head1 SUBROUTINES

=head2 System Tests

These methods test system related aspects of the firewalls.

=head3 environmentals

Returns 1 if there are no environmental alarms. These are platform dependent, but generally consist of fantray and fans, power supplies and power, and temperature. If there are B<any> alarms, returns 0.

VMs don't have any environmental information. In this instance the test will succeed, but a warning is generated.

    ok( $test->environmentals(), "No environmental alarms" );

=cut

sub environmentals {
    my $self = shift;

    my $environ = $self->firewall->environmentals();

    # VMs don't have any environmental info and return an empty hash.
    if (!%{ $environ }) {
        carp "No environmentals - is this a VM? Returning success";
        return 1;
    }

    for my $property (values %{ $environ }) {
        for my $slot (values %{ $property }) {
            return 0 if any { lc $_->{alarm} ne 'false' } @{ $slot };
        }
    }

    return 1;
}




=head2 Network Tests

These methods test network related functions of the firewalls.

=head3 interfaces_up

C<interfaces_up> takes an ARRAYREF that contains interface match criteria. Returns 0 if B<any> of the interfaces matched are down.
Internally the sub uses a case insensitive regex to create an array of interfaces that match all of the match criteria.
Consider the following values of the 'interfaces' parameter:

    ok( $fw_test->interfaces_up(interfaces => ['ethernet1/1', 'ethernet./(2|3)']), "Interfaces are up" );

=over

=item *
[ ] - will warn that the ARRAYREF is empty, however the sub will return 1 as no interfaces matches are 'down'.

=item *
['ethrnt1/1'] - a typo or any criteria that causes no interfaces to be matched will warn, however the sub will return 1 as no interfaces matched are 'down'.

=item *

['ethrnt1/1', 'ethernet1/2'] - if 'ethrnt1/1' matches no interfaces, and 'ethernet1/2' does, the return value will depend on whether 'ethernet1/2' is 'up' or 'down'.

=back

=cut

sub interfaces_up {
    my $self = shift;
    my %args = validate(@_, 
        {   
            interfaces => { type => ARRAYREF },
        }   
    );

    if (!@{ $args{interfaces} }) {
        carp "Warning: no interfaces specified - test returns true";
        return 1;
    }

    my @testable_interfaces = $self->_get_and_filter_interfaces( $args{interfaces} );
    return 0 if grep { $_->{state} eq 'down' } @testable_interfaces;

    return 1;
}

=head3 interfaces_duplex


C<interfaces_duplex> takes an ARRAYREF of interface match criteria. The match criteria can contain regex.
See C<interfaces_up> for some of the nuances around the matching.

It returns 1 if B<all> of the interfaces are in a full duplex state. If any are not, it returns 0.
If the device is a VM, physical interface state cannot be determined. The function will emit a warning, however
it will still return a successful test.

    ok( 
        $fw_test->interfaces_duplex(
            interfaces => ['ethernet1/1', 'ethernet./(2|3)']
        ), "Interfaces are running full duplex"
     );

=cut

sub interfaces_duplex {
    my $self = shift;
    my %args = validate(@_, 
        {   
            interfaces => { type => ARRAYREF },
        }   
    );

    if (!@{ $args{interfaces} }) {
            carp "Warning: no interfaces specified - test returns true";
            return 1;
    }

    # Get the interfaces - we only care about ones that are in the up state
    my @testable_interfaces = $self->_get_and_filter_interfaces( $args{interfaces} );

    return 0 if grep { _half_duplex_search($_) } @testable_interfaces;

    return 1;

}


# _half_duplex_search( $interface_structure_ref )
#
# Takes a "hw" interace array member returned from a firewall
# Returns 0 if the interface is:
#   * Not up
#   * A probable virtual machine interface (also warns)
#   * Is in full duplex mode
# Returns 1 for everything else. Most likely 'duplex' == 'half', but could be 'duplex' == '[n/a]'

sub _half_duplex_search {
    my $interface_ref = shift;

    return 0 if $interface_ref->{state} ne 'up';
    
    if ($interface_ref->{duplex} eq 'auto') {
        carp "Warning: detected 'auto' duplex, probable VM? Test may still succeed";
        return 0;
    }

    return 0 if $interface_ref->{duplex} eq 'full'; 

    return 1;
}


    

# _get_and_filter_interfaces( $self, $interface_filter_arrayref )
#
# Utility function which retrieves the response from the firewall, and 
# returns the interfaces specified in the filter.
 
sub _get_and_filter_interfaces {
    my $self = shift;
    my $interface_filters_ref  = shift; # Array of interface filters
    my @complete_filtered_interfaces;

    my $fw_response = $self->firewall->interfaces();

    for my $interface_filter (@{ $interface_filters_ref }) {
        my $int_filter_regex = qr{$interface_filter}i; 

        my @filtered_interfaces = grep { $_->{name} =~ m{$int_filter_regex} } @{ $fw_response->{hw}->{entry} };
        # Warn if our search matched no interfaces. However the following grep won't fail;
        carp "Warning: '$interface_filter' matched no interfaces. Test may still succeed" if !@filtered_interfaces;

        push @complete_filtered_interfaces, @filtered_interfaces;
    }
    return @complete_filtered_interfaces;
}



=head3 routes_exist 

Takes an ARRAYREF of routes and searches for these routes in the virtual router specified by C<vrouter>.
If B<all> of the exact routes are present in the routing table it returns 1. If B<any> exact routes are not present, it 
returns 0. 

C<routes> is mandatory. C<vrouter> is optional, and is set to 'default' if not specified.
An empty ARRAYREF will emit a warning but will still return 1.

    ok( 
        $fw_test->routes_exist(
            vrouter => 'virt_router_a',
            routes => ['192.0.2.0/30', '192.0.2.128/25']
        ), "All expected routes are present in 'virt_router_a'"
    );


=cut
sub routes_exist {
    my $self = shift;
    my %args = validate(@_,
        {
            routes      => { type => ARRAYREF },
            vrouter     => { default => 'default', type => SCALAR | UNDEF },
        }
    );

    if (!@{ $args{routes} }) {
        carp "Empty routes ARRAYREF specified - test will still return true";
        return 1;
    }

    my $route_search_ref = delete $args{routes};

    my $routing_table = $self->firewall->routing_table(%args);

    for my $route (@{ $route_search_ref }) {
        if (!grep { $route eq $_->{destination} } @{ $routing_table }) {
            return 0;
        }
    }

    return 1;
}


=head3 bgp_peers_up
    
Returns 1 if B<all> of the BGP peers specified in the C<peer_ips> are established. Returns 0 if any of the peers are not in the established state.

C<vrouter> specifies the virtual router that the BGP peers are configured under. If not supplied, the vrouter 'default' will be used. 

    ok( 
        $fw_test->bgp_peers_ip(
            vrouter => 'virt_router_a',
            peer_ips => ['192.0.2.1', '192.0.2.20']
        ), "BGP peerings for 'virt_router-a' are up"
    );



=cut

sub bgp_peers_up {
    my $self = shift;
    my %args = validate(@_,
        {
            peer_ips    => { type => ARRAYREF },
            vrouter     => { default => 'default', type => SCALAR | UNDEF },
        }
    );

    my $peer_ip_search_ref = delete $args{peer_ips};

    my $bgp_peers = $self->firewall->bgp_peers(%args);

    my @up_peers = grep { $_->{status} eq 'Established' } @{ $bgp_peers };

    # Iterate through the peer IPs passed to us and determine whether they're up.
    # If the peer is up, 'peer-address' is host:port, so we split and match against 
    # the first array member
    for my $peer_search (@{ $peer_ip_search_ref }) {
        if (!grep { $peer_search eq (split(':', $_->{'peer-address'}))[0] } @up_peers ) {
                return 0;
        }
    }

    return 1;
}


=head3 ntp_synchronised 

Returns 0 if the firewall is not synchronised with an NTP peer. Returns 1 if the firewall is synchronised with B<at least> one NTP peer.

    ok( $fw_test->ntp_synchronised(), "Firewall is synchronised with at least one NTP server" );

=cut

sub ntp_synchronised {
    my $self = shift;

    my $ntp_response = $self->firewall->ntp();

    return 0 if $ntp_response->{synched} eq 'LOCAL';

    return 1;
}

=head3 ntp_reachable

Returns 1 if all of the configured NTP servers are reachable. Returns 0 if any of the configured NTP servers are not reachable.

    ok ( $fw_test->ntp_reachable(), "Firewall can reach all of its NTP servers" );

=cut

sub ntp_reachable {
    my $self = shift;

    my $ntp_response = $self->firewall->ntp();

    delete $ntp_response->{synched};

    return 0 if !keys %{ $ntp_response }; # No peers configured.

    return 0 if any { $ntp_response->{$_}->{reachable} ne 'yes' } keys %{ $ntp_response }; # Any of the servers are not reachable.

    return 1;
}

=head2 High Availability Tests

These methods test aspects of the high availability function of the firewalls.

=head3 ha_enabled

Returns 1 if HA is enabled on the devices. Returns if HA is not enabled.

    ok( $test->ha_enabled(), "HA is enabled on the firewall" );

=cut

sub ha_enabled {
    my $self = shift;

    my $ha_response = $self->firewall->high_availability();

    return $self->_check_ha_enabled($ha_response->{enabled});

    return 0;
}

=head3 ha_state

Returns 1 if the firewall is in the same state as the C<state> parameter passed to the function. Returns 0 if it is not, or if HA is not enabled on the device.

    ok( $test->ha_state(state => 'active'), "Firewall is in the active HA state" );

The SCALAR string passed must be either 'active' or 'passive', however it is case insensitive.

=cut

sub ha_state {
    my $self = shift;
    my %args = validate(@_,
        {
            state => { type => SCALAR, regex => qr{active|passive}i }
        }
    );

    my $ha_response = $self->firewall->high_availability();

    # Check if HA is running
    return 0 if !$self->_check_ha_enabled($ha_response->{enabled});

    return 1 if (lc $args{state} eq lc $ha_response->{group}->{'local-info'}->{state});

    return 0;
}

=head3 ha_version 

Returns 1 if the app, threat, antivirus, PAN-OS and GlobalProtect versions match between the HA peers. Returns 0 if any one of these do not match, or HA is not enabled on the device.

    ok( $test->ha_version(), "HA peers have matching versions" );

=cut

sub ha_version {
    my $self = shift;

	# These are the keys from the returned hash that all need to eq 'Match'
    my @version_match_keys = qw{ url-compat threat-compat av-compat gpclient-compat build-compat vpnclient-compat app-compat };

    my $ha_response = $self->firewall->high_availability();

    # Check if HA is running
    return 0 if !$self->_check_ha_enabled($ha_response->{enabled});

	return 0 if any { lc $_ ne 'match' } @{ $ha_response->{group}->{'local-info'} }{ @version_match_keys };


    return 1;
}


=head3 ha_peer_up

Returns 1 if the peer firewall is considerd 'up', and that the HA1, heartbeat backup and HA2 connections are 'up'. Returns 0 if any one of these conditions is not 'up'.

    ok( $test->ha_peer_up(), "HA peer is up" );

=cut

sub ha_peer_up {
    my $self = shift;

    my @ha_interface_keys = qw{ conn-mgmt conn-ha1 conn-ha2 };

    my $ha_response = $self->firewall->high_availability();

    # Check if HA is running
    return 0 if !$self->_check_ha_enabled($ha_response->{enabled});

    my $peer_info = $ha_response->{group}->{'peer-info'};

    # Return 0 if the peer isn't considered 'up', or any of the HA interfaces aren't considered 'up'
    return 0 if (lc $peer_info->{'conn-status'} ne 'up') and (any { lc $_->{'conn-status'} ne 'up' } @{ $peer_info }{ @ha_interface_keys });

    return 1;
}

=head3 ha_config_sync

Returns 1 if the configuration has been successfully synchronised between the devices. Returns 0 if the configuration has not been synchronised, if config synchronisation is not enabled, or if HA is not enabled.

=cut

sub ha_config_sync {
    my $self = shift;

    my $ha_response = $self->firewall->high_availability();

    # Check if HA is running
    return 0 if !$self->_check_ha_enabled($ha_response->{enabled});

    return 0 if (lc $ha_response->{group}->{'running-sync-enabled'} ne 'yes') and (lc $ha_response->{group}->{'running-sync'} ne 'synchronized');

    return 1;
}




# This functon is used in all of the ha_* subs to check if HA is enabled before doing any further checks.
sub _check_ha_enabled {
    my $self = shift;
    my $enabled = shift;

    return 0 if !defined $enabled;
    return 0 if $enabled ne 'yes';

    return 1;

}


=head2 Firewall Tests

=head3 ip_user_mapping 

Takes a C<domain> and an ARRAYREF of C<users> as arguments. Returns 1 if there is a valid IP mapping for all of these users within that domain. 

If no C<domain> is specified then the users are matched for any domain. If no C<domain> or C<users> are specified then it returns 1 if there is B<any> user to IP mapping, and 0 if there are none.

=cut

sub ip_user_mapping {
    my $self = shift;
    my %args = validate(@_,
        {
            domain  => { type => SCALAR, optional => 1 },
            users   => { type => ARRAYREF, optional => 1 },
        }
    );

    my @user_mappings = @{ $self->firewall->ip_user_mapping() };

    # If domain is supplied, we only want to look at users matching that domain.
    if (defined $args{domain}) {
        @user_mappings = grep { lc $_->{domain} eq lc $args{domain} } @user_mappings;
    }

    # If no users are specified, we just need to see a single user to IP mapping
    # But we're still only looking within the domain if specified.
    if (!defined $args{users}) {
        return 0 if !grep { lc $_->{type} ne 'unknown' } @user_mappings;
    }

    # Return 0 if our user isn't in the user to IP mappings
    for my $user (@{ $args{users} }) {
        return 0 if !grep { $user eq $_->{user} } @user_mappings;
    }
   
    return 1; # All users have an entry.
}


=head3 userid_server_monitor

Takes an ARRAYREF of C<servers> returns 1 if all of the servers are connnected. Returns 0 if B<any> of the servers are not connected. Each server must be specified as their fully qualified domain name, e.g. 'ad01.domain.int'.

If no C<servers> argument is given, returns 1 if B<all> of the servers configured are connected, and returns 0 of B<any> of the servers are not connected.

=cut

sub userid_server_monitor {
	my $self = shift;
    my %args = validate(@_,
        {
            servers => { type => ARRAYREF, optional => 1 },
        }
    );

	my $server_monitor = $self->firewall->userid_server_monitor();

    # No servers is considered a failure
    return 0 if !@{ $server_monitor };

    # If no server arg is specified, any server that's not 'connected' is a failure
    if (!defined $args{servers}) {
        return if any { lc $_->{connected} ne 'connected' } @{ $server_monitor };
    }

    for my $server (@{ $args{servers} }) {
        my $server_state = first { $server eq $_->{name} } @{ $server_monitor };
        return 0 if (!$server_state or lc $server_state->{connected} ne 'connected')
    }

    return 1;
}


=head1 AUTHOR

Greg Foletta, C<< <greg at foletta.org> >>

=head1 BUGS

Please report any bugs or feature requests to C<bug-device-firewall-paloalto-test at rt.cpan.org>, or through
the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=Device-PaloAlto-Firewall-Test>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.




=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Device::PaloAlto::Firewall::Test


You can also look for information at:

=over 4

=item * RT: CPAN's request tracker (report bugs here)

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=Device-PaloAlto-Firewall-Test>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/Device-PaloAlto-Firewall-Test>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/Device-PaloAlto-Firewall-Test>

=item * Search CPAN

L<http://search.cpan.org/dist/Device-PaloAlto-Firewall-Test/>

=back


=head1 ACKNOWLEDGEMENTS


=head1 LICENSE AND COPYRIGHT

Copyright 2016 Greg Foletta.

This program is free software; you can redistribute it and/or modify it
under the terms of the the Artistic License (2.0). You may obtain a
copy of the full license at:

L<http://www.perlfoundation.org/artistic_license_2_0>

Any use, modification, and distribution of the Standard or Modified
Versions is governed by this Artistic License. By using, modifying or
distributing the Package, you accept this license. Do not use, modify,
or distribute the Package, if you do not accept this license.

If your Modified Version has been derived from a Modified Version made
by someone other than you, you are nevertheless required to ensure that
your Modified Version complies with the requirements of this license.

This license does not grant you the right to use any trademark, service
mark, tradename, or logo of the Copyright Holder.

This license includes the non-exclusive, worldwide, free-of-charge
patent license to make, have made, use, offer to sell, sell, import and
otherwise transfer the Package with respect to any patent claims
licensable by the Copyright Holder that are necessarily infringed by the
Package. If you institute patent litigation (including a cross-claim or
counterclaim) against any party alleging that the Package constitutes
direct or contributory patent infringement, then this Artistic License
to you shall terminate on the date that such litigation is filed.

Disclaimer of Warranty: THE PACKAGE IS PROVIDED BY THE COPYRIGHT HOLDER
AND CONTRIBUTORS "AS IS' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.
THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
PURPOSE, OR NON-INFRINGEMENT ARE DISCLAIMED TO THE EXTENT PERMITTED BY
YOUR LOCAL LAW. UNLESS REQUIRED BY LAW, NO COPYRIGHT HOLDER OR
CONTRIBUTOR WILL BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, OR
CONSEQUENTIAL DAMAGES ARISING IN ANY WAY OUT OF THE USE OF THE PACKAGE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


=cut

1; # End of Device::PaloAlto::Firewall::Test
