use utf8;
package CPAN::Testers::Schema::Result::Release;
our $VERSION = '0.013';
# ABSTRACT: Collected test report stats about a single CPAN release

#pod =head1 SYNOPSIS
#pod
#pod     my $release = $schema->resultset( 'Release' )->find({
#pod         dist => 'My-Dist',
#pod         version => '1.001',
#pod     });
#pod
#pod     say sprintf "My dist has %d pass and %d fail reports",
#pod         $release->pass, $release->fail;
#pod
#pod     $schema->resultset( 'Release' )
#pod         ->search({ dist => 'My-Dist', version => '1.001' })
#pod         ->update({ pass => \'pass+1' }); # increment PASSes
#pod
#pod =head1 DESCRIPTION
#pod
#pod This table contains a collected summary of release data suitable for
#pod quick views of sets of distributions and versions.
#pod
#pod This table's data is generated by L<Labyrinth::Plugin::CPAN::Release>.
#pod
#pod =head1 SEE ALSO
#pod
#pod L<DBIx::Class::Row>, L<CPAN::Testers::Schema>
#pod
#pod =cut

use CPAN::Testers::Schema::Base 'Result';
table 'release_summary';

#pod =attr dist
#pod
#pod The name of the distribution.
#pod
#pod =cut

column dist => {
    data_type => 'varchar',
    is_nullable => 0,
};

#pod =attr version
#pod
#pod The version of the distribution.
#pod
#pod =cut

column version => {
    data_type => 'varchar',
    is_nullable => 0,
};

#pod =attr id
#pod
#pod The ID of the latest report for this release from the `cpanstats` table.
#pod See L<CPAN::Testers::Schema::Result::Stats>.
#pod
#pod =cut

column id => {
    data_type => 'int',
    is_nullable => 0,
};

#pod =attr guid
#pod
#pod The GUID of the latest report for this release from the `cpanstats`
#pod table. See L<CPAN::Testers::Schema::Result::Stats>.
#pod
#pod =cut

column guid => {
    data_type => 'char',
    size => 36,
    is_nullable => 0,
};

#pod =attr oncpan
#pod
#pod The installability of this release: C<1> if the release is on CPAN. C<2>
#pod if the release has been deleted from CPAN and is only on BackPAN.
#pod
#pod =cut

column oncpan => {
    data_type => 'int',
    is_nullable => 0,
};

#pod =attr distmat
#pod
#pod The maturity of this release. C<1> if the release is stable and
#pod ostensibly indexed by CPAN. C<2> if the release is a developer release,
#pod unindexed by CPAN.
#pod
#pod =cut

column distmat => {
    data_type => 'int',
    is_nullable => 0,
};

#pod =attr perlmat
#pod
#pod The maturity of the Perl these reports were sent by: C<1> if the Perl is
#pod a stable release. C<2> if the Perl is a developer release.
#pod
#pod =cut

column perlmat => {
    data_type => 'int',
    is_nullable => 0,
};

#pod =attr patched
#pod
#pod The patch status of the Perl that sent the report. C<2> if the Perl reports
#pod being patched, C<1> otherwise.
#pod
#pod =cut

column patched => {
    data_type => 'int',
    is_nullable => 0,
};

#pod =attr pass
#pod
#pod The number of C<PASS> results for this release.
#pod
#pod =cut

column pass => {
    data_type => 'int',
    is_nullable => 0,
};

#pod =attr fail
#pod
#pod The number of C<FAIL> results for this release.
#pod
#pod =cut

column fail => {
    data_type => 'int',
    is_nullable => 0,
};

#pod =attr na
#pod
#pod The number of C<NA> results for this release.
#pod
#pod =cut

column na => {
    data_type => 'int',
    is_nullable => 0,
};

#pod =attr unknown
#pod
#pod The number of C<UNKNOWN> results for this release.
#pod
#pod =cut

column unknown => {
    data_type => 'int',
    is_nullable => 0,
};

#pod =attr uploadid
#pod
#pod The ID of this upload from the `uploads` table.
#pod
#pod =cut

column uploadid => {
    data_type => 'int',
    extra       => { unsigned => 1 },
    is_nullable => 0,
};

#pod =method upload
#pod
#pod Get the related row from the `uploads` table. See
#pod L<CPAN::Testers::Schema::Result::Upload>.
#pod
#pod =cut

belongs_to upload => 'CPAN::Testers::Schema::Result::Upload' => 'uploadid';

#pod =method report
#pod
#pod Get the related row from the `cpanstats` table. See
#pod L<CPAN::Testers::Schema::Result::Stats>.
#pod
#pod This report is the latest report, by date, that went in to this release
#pod summary. The date field in the report can be used to determine when this
#pod release was last updated.
#pod
#pod =cut

belongs_to report => 'CPAN::Testers::Schema::Result::Stats', {
    'foreign.guid' => 'self.guid',
};

1;

__END__

=pod

=head1 NAME

CPAN::Testers::Schema::Result::Release - Collected test report stats about a single CPAN release

=head1 VERSION

version 0.013

=head1 SYNOPSIS

    my $release = $schema->resultset( 'Release' )->find({
        dist => 'My-Dist',
        version => '1.001',
    });

    say sprintf "My dist has %d pass and %d fail reports",
        $release->pass, $release->fail;

    $schema->resultset( 'Release' )
        ->search({ dist => 'My-Dist', version => '1.001' })
        ->update({ pass => \'pass+1' }); # increment PASSes

=head1 DESCRIPTION

This table contains a collected summary of release data suitable for
quick views of sets of distributions and versions.

This table's data is generated by L<Labyrinth::Plugin::CPAN::Release>.

=head1 ATTRIBUTES

=head2 dist

The name of the distribution.

=head2 version

The version of the distribution.

=head2 id

The ID of the latest report for this release from the `cpanstats` table.
See L<CPAN::Testers::Schema::Result::Stats>.

=head2 guid

The GUID of the latest report for this release from the `cpanstats`
table. See L<CPAN::Testers::Schema::Result::Stats>.

=head2 oncpan

The installability of this release: C<1> if the release is on CPAN. C<2>
if the release has been deleted from CPAN and is only on BackPAN.

=head2 distmat

The maturity of this release. C<1> if the release is stable and
ostensibly indexed by CPAN. C<2> if the release is a developer release,
unindexed by CPAN.

=head2 perlmat

The maturity of the Perl these reports were sent by: C<1> if the Perl is
a stable release. C<2> if the Perl is a developer release.

=head2 patched

The patch status of the Perl that sent the report. C<2> if the Perl reports
being patched, C<1> otherwise.

=head2 pass

The number of C<PASS> results for this release.

=head2 fail

The number of C<FAIL> results for this release.

=head2 na

The number of C<NA> results for this release.

=head2 unknown

The number of C<UNKNOWN> results for this release.

=head2 uploadid

The ID of this upload from the `uploads` table.

=head1 METHODS

=head2 upload

Get the related row from the `uploads` table. See
L<CPAN::Testers::Schema::Result::Upload>.

=head2 report

Get the related row from the `cpanstats` table. See
L<CPAN::Testers::Schema::Result::Stats>.

This report is the latest report, by date, that went in to this release
summary. The date field in the report can be used to determine when this
release was last updated.

=head1 SEE ALSO

L<DBIx::Class::Row>, L<CPAN::Testers::Schema>

=head1 AUTHORS

=over 4

=item *

Oriol Soriano <oriolsoriano@gmail.com>

=item *

Doug Bell <preaction@cpan.org>

=back

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by Oriol Soriano, Doug Bell.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
