#  You may distribute under the terms of either the GNU General Public License
#  or the Artistic License (the same terms as Perl itself)
#
#  (C) Paul Evans, 2013-2017 -- leonerd@leonerd.org.uk

package Devel::MAT::Tool::Inrefs;

use strict;
use warnings;

our $VERSION = '0.28';

use List::Util qw( pairs );

my %STRENGTH_TO_IDX = (
   strong   => 0,
   weak     => 1,
   indirect => 2,
   inferred => 3,
);

=head1 NAME

C<Devel::MAT::Tool::Inrefs> - annotate which SVs are referred to by others

=head1 DESCRIPTION

This C<Devel::MAT> tool annotates each SV with back-references from other SVs
that refer to it. It follows the C<outrefs> method of every heap SV and
annotates the referred SVs with back-references pointing back to the SVs that
refer to them.

=cut

sub new
{
   my $class = shift;
   my ( $pmat, %args ) = @_;

   $class->patch_inrefs( $pmat->dumpfile, progress => $args{progress} );

   return $class;
}

sub patch_inrefs
{
   my $self = shift;
   my ( $df, %args ) = @_;

   my $progress = $args{progress};

   # Performance optimisation since we don't care about the 'name' of any of
   #   these outrefs
   no warnings 'redefine';
   local *Devel::MAT::Cmd::format_value = sub { "" };
   local *Devel::MAT::Cmd::format_note  = sub { "" };

   my $heap_total = scalar $df->heap;
   my $count = 0;
   foreach my $sv ( $df->heap ) {
      foreach my $ref ( $sv->outrefs ) {
         my $refsv = $ref->sv;
         push @{ $refsv->{tool_inrefs}[ $STRENGTH_TO_IDX{ $ref->strength } ] }, $sv->addr if !$refsv->immortal;
      }

      $count++;
      $progress->( sprintf "Patching refs in %d of %d (%.2f%%)",
         $count, $heap_total, 100*$count / $heap_total ) if $progress and ($count % 10000) == 0
   }

   $progress->() if $progress;
}

=head1 SV METHODS

This tool adds the following SV methods.

=head2 inrefs

   @refs = $sv->inrefs

Returns a list of Reference objects for each of the SVs that refer to this
one. This is formed by the inverse mapping along the SV graph from C<outrefs>.

=head2 inrefs_strong

=head2 inrefs_weak

=head2 inrefs_direct

=head2 inrefs_indirect

=head2 inrefs_inferred

   @refs = $sv->inrefs_strong

   @refs = $sv->inrefs_weak

   @refs = $sv->inrefs_direct

   @refs = $sv->inrefs_indirect

   @refs = $sv->inrefs_inferred

Returns lists of Reference objects filtered by type, analogous to the various
C<outrefs_*> methods.

=cut

sub Devel::MAT::SV::_inrefs
{
   my $self = shift;
   my ( $just_svs, @strengths ) = @_;

   $self->{tool_inrefs} ||= [];

   my $df = $self->df;
   my @inrefs;
   foreach my $strength ( @strengths ) {
      my %seen;
      foreach my $addr ( @{ $self->{tool_inrefs}[ $STRENGTH_TO_IDX{$strength} ] // [] } ) {
         my $sv = $df->sv_at( $addr );

         if( $just_svs ) {
            push @inrefs, $sv;
         }
         else {
            $seen{$addr}++ and next;

            push @inrefs, Devel::MAT::SV::Reference( $_->name, $_->strength, $sv )
               for grep { $_->strength eq $strength and $_->sv == $self } $sv->outrefs;
         }
      }
   }

   foreach ( pairs $df->_roots ) {
      my ( $name, $sv ) = @$_;
      push @inrefs, $just_svs ? undef : Devel::MAT::SV::Reference( $name, strong => undef )
         if defined $sv and $sv == $self;
   }

   foreach my $addr ( @{ $df->{stack_at} } ) { # TODO
      next unless $addr == $self->addr;

      push @inrefs, $just_svs ? undef :
         Devel::MAT::SV::Reference( "a value on the stack", strong => undef );
   }

   return @inrefs;
}

sub Devel::MAT::SV::inrefs          { shift->_inrefs( !wantarray, qw( strong weak indirect inferred )) }

sub Devel::MAT::SV::inrefs_strong   { shift->_inrefs( !wantarray, qw( strong      )) }
sub Devel::MAT::SV::inrefs_weak     { shift->_inrefs( !wantarray, qw( weak        )) }
sub Devel::MAT::SV::inrefs_direct   { shift->_inrefs( !wantarray, qw( strong weak )) }
sub Devel::MAT::SV::inrefs_indirect { shift->_inrefs( !wantarray, qw( indirect    )) }
sub Devel::MAT::SV::inrefs_inferred { shift->_inrefs( !wantarray, qw( inferred    )) }

=head1 AUTHOR

Paul Evans <leonerd@leonerd.org.uk>

=cut

0x55AA;
