package Bencher::Scenario::CBlocks::Numeric;

our $DATE = '2016-12-10'; # DATE
our $VERSION = '0.001'; # VERSION

use 5.010001;
use strict;
use warnings;

our $scenario = {
    summary => 'Benchmark numeric performance of C::Blocks',
    description => <<'_',

Each code generates random number (the `perl` participant using pure-perl code.

_
    precision => 6,
    participants => [
        {
            name => 'perl',
            code_template => <<'_',
my $a = 698769069;
my ($x, $y, $z, $c) = (123456789, 362436000, 521288629, 7654321);
my $rand;
for (1 .. <N>) {
    my $t;
    $x = 69069*$x+12345;
    $y ^= ($y<<13); $y ^= ($y>>17); $y ^= ($y<<5);
    $t = $a*$z+$c; $c = ($t>>32);
    $z = $t;
    $rand = $x+$y+$z;
}
return $rand;
_
        },
        {
            name => 'C::Blocks',
            code_template => <<'_',
use C::Blocks;
use C::Blocks::Types qw(uint);
clex {
    /* Note: y must never be set to zero;
     * z and c must not be simultaneously zero */
    unsigned int x = 123456789,y = 362436000,
        z = 521288629,c = 7654321; /* State variables */

    unsigned int KISS() {
        unsigned long long t, a = 698769069ULL;
        x = 69069*x+12345;
        y ^= (y<<13); y ^= (y>>17); y ^= (y<<5);
        t = a*z+c; c = (t>>32);
        return x+y+(z=t);
    }
}

my uint $to_return = 0;
cblock {
    for (int i = 0; i < <N>; i++) $to_return = KISS();
}
return $to_return;

_
        },
    ],

    datasets => [
        {args=>{N=>int(10**1)}},
        {args=>{N=>int(10**1.5)}},
        {args=>{N=>int(10**2)}},
        {args=>{N=>int(10**2.5)}},
        {args=>{N=>int(10**3)}},
        {args=>{N=>int(10**3.51)}},
        {args=>{N=>int(10**4)}},
        {args=>{N=>int(10**4.5)}},
        {args=>{N=>int(10**5)}},
        {args=>{N=>int(10**5.5)}},
    ],
};

1;
# ABSTRACT: Benchmark numeric performance of C::Blocks

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::CBlocks::Numeric - Benchmark numeric performance of C::Blocks

=head1 VERSION

This document describes version 0.001 of Bencher::Scenario::CBlocks::Numeric (from Perl distribution Bencher-Scenarios-CBlocks), released on 2016-12-10.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m CBlocks::Numeric

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 DESCRIPTION

Each code generates random number (the C<perl> participant using pure-perl code.


Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 BENCHMARK PARTICIPANTS

=over

=item * perl (perl_code)

Code template:

 my $a = 698769069;
 my ($x, $y, $z, $c) = (123456789, 362436000, 521288629, 7654321);
 my $rand;
 for (1 .. <N>) {
     my $t;
     $x = 69069*$x+12345;
     $y ^= ($y<<13); $y ^= ($y>>17); $y ^= ($y<<5);
     $t = $a*$z+$c; $c = ($t>>32);
     $z = $t;
     $rand = $x+$y+$z;
 }
 return $rand;




=item * C::Blocks (perl_code)

Code template:

 use C::Blocks;
 use C::Blocks::Types qw(uint);
 clex {
     /* Note: y must never be set to zero;
      * z and c must not be simultaneously zero */
     unsigned int x = 123456789,y = 362436000,
         z = 521288629,c = 7654321; /* State variables */
 
     unsigned int KISS() {
         unsigned long long t, a = 698769069ULL;
         x = 69069*x+12345;
         y ^= (y<<13); y ^= (y>>17); y ^= (y<<5);
         t = a*z+c; c = (t>>32);
         return x+y+(z=t);
     }
 }
 
 my uint $to_return = 0;
 cblock {
     for (int i = 0; i < <N>; i++) $to_return = KISS();
 }
 return $to_return;
 




=back

=head1 BENCHMARK DATASETS

=over

=item * 10

=item * 31

=item * 100

=item * 316

=item * 1000

=item * 3235

=item * 10000

=item * 31622

=item * 100000

=item * 316227

=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.24.0 >>, CPU: I<< Intel(R) Core(TM) i7-4770 CPU @ 3.40GHz (4 cores) >>, OS: I<< GNU/Linux Debian version 8.5 >>, OS kernel: I<< Linux version 3.16.0-4-amd64 >>.

Benchmark with default options (C<< bencher -m CBlocks::Numeric >>):

 #table1#
 +-------------+---------+------------+---------------+------------+-----------+---------+
 | participant | dataset | rate (/s)  |   time (ms)   | vs_slowest |  errors   | samples |
 +-------------+---------+------------+---------------+------------+-----------+---------+
 | C::Blocks   | 316227  |       8.6  | 120           |      1     |   0.00012 |       6 |
 | perl        | 316227  |       9.29 | 108           |      1.08  | 2.7e-05   |       6 |
 | C::Blocks   | 100000  |      27.3  |  36.6         |      3.17  | 1.3e-05   |       6 |
 | perl        | 100000  |      29    |  35           |      3.4   |   0.00016 |       6 |
 | perl        | 31622   |      91    |  11           |     11     | 8.6e-05   |       6 |
 | C::Blocks   | 10000   |     239.2  |   4.18        |     27.76  | 3.4e-07   |       8 |
 | perl        | 10000   |     297.4  |   3.362       |     34.51  | 2.9e-07   |       6 |
 | C::Blocks   | 3235    |     775    |   1.29        |     89.9   |   9e-07   |      10 |
 | perl        | 3235    |     921    |   1.09        |    107     | 3.9e-07   |       6 |
 | C::Blocks   | 1000    |    2786.35 |   0.358893    |    323.34  |   0       |       6 |
 | perl        | 1000    |    2994.21 |   0.333978    |    347.461 |   0       |       6 |
 | C::Blocks   | 31622   |    4370.72 |   0.228795    |    507.197 |   0       |       6 |
 | C::Blocks   | 316     |    8778.29 |   0.113917    |   1018.67  |   0       |       6 |
 | perl        | 316     |    9426.91 |   0.106079    |   1093.94  |   0       |       8 |
 | C::Blocks   | 100     |   26500    |   0.0378      |   3070     | 2.3e-08   |       7 |
 | perl        | 100     |   28619.5  |   0.0349412   |   3321.13  |   0       |       6 |
 | perl        | 31      |   94007    |   0.0106375   |  10909     |   0       |       6 |
 | C::Blocks   | 10      |  239790    |   0.00417031  |  27826.3   |   0       |       6 |
 | perl        | 10      |  284515    |   0.00351475  |  33016.3   |   0       |       6 |
 | C::Blocks   | 31      | 3552810    |   0.000281467 | 412283     |   0       |       7 |
 +-------------+---------+------------+---------------+------------+-----------+---------+

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenarios-CBlocks>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenarios-CBlocks>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenarios-CBlocks>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 SEE ALSO

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
